/*
 * Decompiled with CFR 0.152.
 */
package org.exoplatform.addon.ethereum.wallet.migration;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.addon.ethereum.wallet.migration.DeprecatedEthereumWalletService;
import org.exoplatform.addon.ethereum.wallet.migration.EthereumTransactionDecoder;
import org.exoplatform.addon.ethereum.wallet.model.GlobalSettings;
import org.exoplatform.addon.ethereum.wallet.model.TransactionDetail;
import org.exoplatform.addon.ethereum.wallet.model.Wallet;
import org.exoplatform.addon.ethereum.wallet.service.EthereumWalletAccountService;
import org.exoplatform.addon.ethereum.wallet.service.EthereumWalletService;
import org.exoplatform.addon.ethereum.wallet.storage.TransactionStorage;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.container.ExoContainer;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.container.component.RequestLifeCycle;
import org.exoplatform.portal.config.UserACL;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.picocontainer.Startable;

public class MigrationService
implements Startable {
    private static final Log LOG = ExoLogger.getLogger(MigrationService.class);
    private static final int GLOBAL_DATA_VERSION = 4;
    private DeprecatedEthereumWalletService deprecatedEthereumWalletService;
    private EthereumWalletService ethereumWalletService;
    private EthereumWalletAccountService accountService;
    private TransactionStorage transactionStorage;
    private EthereumTransactionDecoder transactionDecoder;
    private UserACL userACL;
    private ExoContainer container;
    private ExecutorService migrationExecutorService;
    private Set<String> transactionHashes = new HashSet<String>();

    public MigrationService(ExoContainer container, EthereumWalletService ethereumWalletService, EthereumWalletAccountService accountService, TransactionStorage transactionStorage, EthereumTransactionDecoder transactionDecoder, UserACL userACL) {
        this.userACL = userACL;
        this.container = container;
        this.ethereumWalletService = ethereumWalletService;
        this.transactionDecoder = transactionDecoder;
        this.transactionStorage = transactionStorage;
        this.accountService = accountService;
    }

    public void start() {
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("Ethereum-migration-%d").build();
        this.migrationExecutorService = Executors.newSingleThreadExecutor(namedThreadFactory);
        this.migrationExecutorService.execute(() -> {
            GlobalSettings settings = this.getEthereumWalletService().getSettings();
            Integer dataVersion = settings.getDataVersion();
            if (dataVersion != null && dataVersion >= 4) {
                return;
            }
            Long networkId = settings.getDefaultNetworkId();
            ExoContainerContext.setCurrentContainer((ExoContainer)this.container);
            String superUser = this.userACL.getSuperUser();
            Set<Wallet> listUserWallets = null;
            int migratedWallets = 0;
            boolean hasWalletMigrationErrors = false;
            LOG.info((Object)"Check list of users wallets to migrate");
            RequestLifeCycle.begin((ExoContainer)this.container);
            try {
                listUserWallets = this.getDeprecatedWalletService().listUserWallets();
                LOG.info("{} users wallets to migrate", new Object[]{listUserWallets.size()});
                migratedWallets = this.migrateWallets(listUserWallets, superUser);
                LOG.info("{}/{} users wallets are migrated", new Object[]{migratedWallets, listUserWallets.size()});
                hasWalletMigrationErrors = migratedWallets != listUserWallets.size();
            }
            catch (Exception e) {
                LOG.error((Object)"Error while migrating users wallets", (Throwable)e);
                hasWalletMigrationErrors = true;
            }
            finally {
                RequestLifeCycle.end();
            }
            Set<Wallet> listSpaceWallets = null;
            LOG.info((Object)"Check list of space wallets to migrate");
            RequestLifeCycle.begin((ExoContainer)this.container);
            try {
                listSpaceWallets = this.getDeprecatedWalletService().listSpacesWallets();
                LOG.info("{} spaces wallets to migrate", new Object[]{listSpaceWallets.size()});
                migratedWallets = this.migrateWallets(listSpaceWallets, superUser);
                LOG.info("{}/{} spaces wallets are migrated", new Object[]{migratedWallets, listSpaceWallets.size()});
                hasWalletMigrationErrors |= migratedWallets != listSpaceWallets.size();
            }
            catch (Exception e) {
                LOG.error((Object)"Error while migrating spaces wallets", (Throwable)e);
                hasWalletMigrationErrors = true;
            }
            finally {
                RequestLifeCycle.end();
            }
            LOG.info((Object)"Migrate user wallet transactions");
            RequestLifeCycle.begin((ExoContainer)this.container);
            try {
                hasWalletMigrationErrors |= this.migrateTransactions(listUserWallets, networkId);
                LOG.info((Object)"Users wallets transactions migrated successfully");
            }
            catch (Exception e) {
                LOG.error((Object)"Error while migrating user wallet transactions", (Throwable)e);
                hasWalletMigrationErrors = true;
            }
            finally {
                RequestLifeCycle.end();
            }
            LOG.info((Object)"Migrate space wallets transactions");
            RequestLifeCycle.begin((ExoContainer)this.container);
            try {
                hasWalletMigrationErrors |= this.migrateTransactions(listSpaceWallets, networkId);
                LOG.info((Object)"Spaces wallets transactions migrated successfully");
            }
            catch (Exception e) {
                LOG.error((Object)"Error while migrating space wallet transactions", (Throwable)e);
                hasWalletMigrationErrors = true;
            }
            finally {
                RequestLifeCycle.end();
            }
            String principalContractAddress = settings.getDefaultPrincipalAccount();
            if (StringUtils.isNotBlank((CharSequence)principalContractAddress)) {
                RequestLifeCycle.begin((ExoContainer)this.container);
                try {
                    hasWalletMigrationErrors |= this.migrateContractTransactions(principalContractAddress, networkId);
                }
                catch (Exception e) {
                    LOG.error((Object)"Error while migrating contract transactions", (Throwable)e);
                    hasWalletMigrationErrors = true;
                }
                finally {
                    RequestLifeCycle.end();
                }
            }
            RequestLifeCycle.begin((ExoContainer)this.container);
            try {
                hasWalletMigrationErrors |= this.retrieveTransactionDetailsFromBlockchain();
            }
            catch (Exception e) {
                LOG.error((Object)"Error while migrating contract transactions", (Throwable)e);
                hasWalletMigrationErrors = true;
            }
            finally {
                RequestLifeCycle.end();
            }
            if (hasWalletMigrationErrors) {
                LOG.info((Object)"Wallet and Transactions migration seems to have some errors, it will be reattempted again next startup");
            } else {
                LOG.info((Object)"Wallet and Transactions migration finished susccessfully");
                RequestLifeCycle.begin((ExoContainer)this.container);
                try {
                    this.ethereumWalletService.saveSettings(settings, 4);
                }
                finally {
                    RequestLifeCycle.end();
                }
            }
        });
    }

    private boolean retrieveTransactionDetailsFromBlockchain() {
        int computedTransactions = 0;
        int errorComputedTransactions = 0;
        int transactionsCount = this.transactionHashes.size();
        LOG.debug("Compute {} saved transactions details from blockchain", new Object[]{transactionsCount});
        for (String hash : this.transactionHashes) {
            TransactionDetail transactionDetail = this.transactionStorage.getTransactionByHash(hash, false);
            try {
                this.transactionDecoder.computeTransactionDetail(transactionDetail);
                this.transactionStorage.saveTransactionDetail(transactionDetail);
            }
            catch (InterruptedException e) {
                try {
                    this.transactionDecoder.computeTransactionDetail(transactionDetail);
                    this.transactionStorage.saveTransactionDetail(transactionDetail);
                }
                catch (InterruptedException e1) {
                    ++errorComputedTransactions;
                    LOG.warn((Object)("Can't compute transaction details with hash: " + transactionDetail.getHash()));
                }
            }
            if (++computedTransactions % 100 != 0) continue;
            LOG.info("{}/{} transactions has been computed from blockchain, errors = {}", new Object[]{computedTransactions, transactionsCount, errorComputedTransactions});
        }
        LOG.info("{} transactions has been computed from blockchain with treatment errors = {}", new Object[]{computedTransactions, errorComputedTransactions});
        return errorComputedTransactions > 0;
    }

    private boolean migrateContractTransactions(String principalContractAddress, Long networkId) {
        List<TransactionDetail> contractTransactions = this.getDeprecatedWalletService().getAccountTransactions(networkId, principalContractAddress);
        LOG.info("Migrate contract {} transactions, count = {}", new Object[]{principalContractAddress, contractTransactions == null ? 0 : contractTransactions.size()});
        boolean migrationHasErrors = false;
        for (TransactionDetail transactionDetail : contractTransactions) {
            String hash = transactionDetail.getHash();
            LOG.debug("Migrate contract transaction {}", new Object[]{hash});
            try {
                transactionDetail.setContractAddress(principalContractAddress);
                this.migrateTransaction(principalContractAddress, transactionDetail, networkId);
            }
            catch (Exception e) {
                migrationHasErrors = true;
                LOG.error("Error while migrating principal contract transaction {}", new Object[]{hash, e});
            }
        }
        return migrationHasErrors;
    }

    private boolean migrateTransactions(Set<Wallet> listWallets, Long networkId) {
        boolean migrationHasErrors = false;
        for (Wallet wallet : listWallets) {
            try {
                List<TransactionDetail> accountTransactions = this.getDeprecatedWalletService().getAccountTransactions(networkId, wallet.getAddress());
                LOG.info("    -- migrate wallet {} transactions of {} {}", new Object[]{accountTransactions.size(), wallet.getType(), wallet.getId()});
                for (TransactionDetail transaction : accountTransactions) {
                    try {
                        this.migrateTransaction(wallet.getAddress(), transaction, networkId);
                    }
                    catch (Exception e) {
                        LOG.warn("Error while migrating address {} transaction {}", new Object[]{wallet.getAddress(), transaction.getHash(), e});
                        migrationHasErrors = true;
                    }
                }
            }
            catch (Exception e) {
                LOG.warn("Error while migrating {} {} transactions", new Object[]{wallet.getType(), wallet.getId(), e});
                migrationHasErrors = true;
            }
        }
        return migrationHasErrors;
    }

    private void migrateTransaction(String address, TransactionDetail transaction, long networkId) {
        String hash = transaction.getHash();
        this.transactionHashes.add(transaction.getHash().toLowerCase());
        TransactionDetail migratedTransaction = this.getTransactionStorage().getTransactionByHash(hash, false);
        if (migratedTransaction == null) {
            transaction.setNetworkId(networkId);
            transaction.setFrom(address);
            this.getTransactionStorage().saveTransactionDetail(transaction);
        } else {
            migratedTransaction.setNetworkId(networkId);
            if (transaction.getTo() == null) {
                transaction.setTo(address);
            }
            if (StringUtils.isBlank((CharSequence)migratedTransaction.getLabel())) {
                migratedTransaction.setLabel(transaction.getLabel());
            }
            if (StringUtils.isBlank((CharSequence)migratedTransaction.getMessage())) {
                migratedTransaction.setMessage(transaction.getMessage());
            }
            this.getTransactionStorage().saveTransactionDetail(migratedTransaction);
        }
    }

    private int migrateWallets(Set<Wallet> listWallets, String superUser) {
        int migratedWallets = 0;
        for (Wallet wallet : listWallets) {
            if (StringUtils.isBlank((CharSequence)wallet.getAddress())) {
                LOG.warn("Wallet address of {} / {} wasn't found, skip", new Object[]{wallet.getType(), wallet.getId()});
                continue;
            }
            try {
                LOG.debug("    -- migrate wallet of {} {}", new Object[]{wallet.getType(), wallet.getId()});
                this.getAccountService().saveWallet(wallet, superUser, false);
                ++migratedWallets;
            }
            catch (Exception e) {
                LOG.warn("Error while migrating wallet {}", new Object[]{wallet, e});
            }
        }
        return migratedWallets;
    }

    public void stop() {
        if (this.migrationExecutorService != null) {
            this.migrationExecutorService.shutdownNow();
        }
    }

    public EthereumWalletService getEthereumWalletService() {
        if (this.ethereumWalletService == null) {
            this.ethereumWalletService = (EthereumWalletService)CommonsUtils.getService(EthereumWalletService.class);
        }
        return this.ethereumWalletService;
    }

    public EthereumWalletAccountService getAccountService() {
        if (this.accountService == null) {
            this.accountService = (EthereumWalletAccountService)CommonsUtils.getService(EthereumWalletAccountService.class);
        }
        return this.accountService;
    }

    public TransactionStorage getTransactionStorage() {
        if (this.transactionStorage == null) {
            this.transactionStorage = (TransactionStorage)CommonsUtils.getService(TransactionStorage.class);
        }
        return this.transactionStorage;
    }

    public DeprecatedEthereumWalletService getDeprecatedWalletService() {
        if (this.deprecatedEthereumWalletService == null) {
            this.deprecatedEthereumWalletService = (DeprecatedEthereumWalletService)CommonsUtils.getService(DeprecatedEthereumWalletService.class);
        }
        return this.deprecatedEthereumWalletService;
    }
}

