package org.exoplatform.writeinldap.services;

import org.exoplatform.commons.api.settings.ExoFeatureService;
import org.exoplatform.services.listener.ListenerService;
import org.exoplatform.services.organization.externalstore.IDMExternalStoreImportService;
import org.exoplatform.services.organization.externalstore.IDMExternalStoreService;
import org.exoplatform.services.organization.externalstore.model.IDMEntityType;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.organization.User;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.identity.model.Profile;
import org.exoplatform.social.core.identity.provider.OrganizationIdentityProvider;
import org.exoplatform.social.core.manager.IdentityManager;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.List;

public class ActiveDirectoryService {
  private static final Log LOG = ExoLogger.getLogger(ActiveDirectoryService.class);
  
  private static final String LDAP_SERVICE_NAME = "ldapService";
  
  private static final String LDAP_TYPE = "ldap";
  private static final String AD_TYPE = "ad";

  private static final int UF_ACCOUNTDISABLE = 0x2;


  private static final String USER_ACCOUNT_CONTROL = "userAccountControl";
  
  private String ldapUrl;
  
  private String ldapType;
  private String ldapPrincipal;
  
  private String ldapCredentials;
  
  private String userBaseDn;
  
  private String userFilter;
  private String userAttributeUserName;
  private String userAttributeFirstName;
  private String userAttributeEmail;
  private String userAttributeLastName;
  private String userAttributePassword;
  private List<String> defaultLdapGroups;
  private boolean isFeatureEnabled;
  
  private IDMExternalStoreImportService externalStoreImportService;
  private IdentityManager identityManager;

  private final ListenerService listenerService;
  
  private Hashtable env;

  private String userAccountControl="userAccountControl";

  public ActiveDirectoryService(InitParams initParams, ExoFeatureService exoFeatureService,
                                IDMExternalStoreImportService externalStoreImportService, ListenerService listenerService,
                                IdentityManager identityManager) {
  
    isFeatureEnabled=exoFeatureService.isActiveFeature("write-in-ldap");
    this.externalStoreImportService=externalStoreImportService;
    this.listenerService=listenerService;
    if (initParams.getValueParam("ldapType") != null &&
      !initParams.getValueParam("ldapType").getValue().isEmpty()) {
      ldapType=initParams.getValueParam("ldapType").getValue();
    } else {
      if (System.getProperty("exo.ldap.type")!=null && !System.getProperty("exo.ldap.type").isEmpty()) {
        ldapType=System.getProperty("exo.ldap.type");
      } else {
        isFeatureEnabled = false;
      }
    }
    
    if (initParams.getValueParam("ldapCredentials") != null &&
        !initParams.getValueParam("ldapCredentials").getValue().isEmpty()) {
      this.ldapCredentials=initParams.getValueParam("ldapCredentials").getValue();
    } else {
      if (System.getProperty("exo.ldap.admin.password")!=null && !System.getProperty("exo.ldap.admin.password").isEmpty()) {
        this.ldapCredentials=System.getProperty("exo.ldap.admin.password");
      } else {
        this.ldapCredentials="";
      }
    }
  
    if (initParams.getValueParam("ldapPrincipal") != null &&
        !initParams.getValueParam("ldapPrincipal").getValue().isEmpty()) {
      this.ldapPrincipal=initParams.getValueParam("ldapPrincipal").getValue();
    } else {
      if (System.getProperty("exo.ldap.admin.dn")!=null && !System.getProperty("exo.ldap.admin.dn").isEmpty()) {
        this.ldapPrincipal=System.getProperty("exo.ldap.admin.dn");
      } else {
        this.ldapPrincipal="cn=admin";
      }
    }
  
    if (initParams.getValueParam("ldapUrl") != null &&
        !initParams.getValueParam("ldapUrl").getValue().isEmpty()) {
      this.ldapUrl=initParams.getValueParam("ldapUrl").getValue();
    } else {
      if (System.getProperty("exo.ldap.url")!=null && !System.getProperty("exo.ldap.url").isEmpty()) {
        this.ldapUrl=System.getProperty("exo.ldap.url");
      } else {
        this.ldapUrl="http://localhost:389";
      }
    }
  
    if (initParams.getValueParam("userBaseDn") != null &&
        !initParams.getValueParam("userBaseDn").getValue().isEmpty()) {
      this.userBaseDn=initParams.getValueParam("userBaseDn").getValue();
    } else {
      if (System.getProperty("exo.ldap.users.base.dn")!=null && !System.getProperty("exo.ldap.users.base.dn").isEmpty()) {
        userBaseDn=System.getProperty("exo.ldap.users.base.dn");
      } else {
        this.userBaseDn="ou=users,dc=company,dc=org";
      }
    }
  
    if (initParams.getValueParam("userAttributeUserName") != null &&
        !initParams.getValueParam("userAttributeUserName").getValue().isEmpty()) {
      this.userAttributeUserName =initParams.getValueParam("userAttributeUserName").getValue();
    } else {
      if (System.getProperty("exo.ldap.users.id.attributeName")!=null && !System.getProperty("exo.ldap.users.id.attributeName").isEmpty()) {
        this.userAttributeUserName=System.getProperty("exo.ldap.users.id.attributeName");
      } else {
        this.userAttributeUserName = this.ldapType.equals(LDAP_TYPE) ? "uid" : "samAccountName";
      }
    }
    if (initParams.getValueParam("userFilter") != null &&
        !initParams.getValueParam("userFilter").getValue().isEmpty()) {
      this.userFilter=initParams.getValueParam("userFilter").getValue();
    } else {
      if (System.getProperty("exo.ldap.users.filter")!=null && !System.getProperty("exo.ldap.users.filter").isEmpty()) {
        this.userFilter=System.getProperty("exo.ldap.users.filter");
      } else {
        this.userFilter = null;
      }
    }
    if (initParams.getValueParam("userAttributeFirstName") != null &&
        !initParams.getValueParam("userAttributeFirstName").getValue().isEmpty()) {
      this.userAttributeFirstName=initParams.getValueParam("userAttributeFirstName").getValue();
    } else {
      if (System.getProperty("exo.ldap.users.attributes.firstName.mapping")!=null && !System.getProperty("exo.ldap.users.attributes.firstName.mapping").isEmpty()) {
        this.userAttributeFirstName=System.getProperty("exo.ldap.users.attributes.firstName.mapping");
      } else {
        this.userAttributeFirstName = this.ldapType.equals(LDAP_TYPE) ? "cn" : "givenName";
      }
    }
    if (initParams.getValueParam("userAttributeLastName") != null &&
        !initParams.getValueParam("userAttributeLastName").getValue().isEmpty()) {
      this.userAttributeLastName=initParams.getValueParam("userAttributeLastName").getValue();
    } else {
      if (System.getProperty("exo.ldap.users.attributes.lastName.mapping")!=null && !System.getProperty("exo.ldap.users.attributes.lastName.mapping").isEmpty()) {
        this.userAttributeLastName=System.getProperty("exo.ldap.users.attributes.lastName.mapping");
      } else {
        this.userAttributeLastName="sn";
      }
    }
    if (initParams.getValueParam("userAttributeEmail") != null &&
        !initParams.getValueParam("userAttributeEmail").getValue().isEmpty()) {
      this.userAttributeEmail=initParams.getValueParam("userAttributeEmail").getValue();
    } else {
      if (System.getProperty("exo.ldap.users.attributes.email.mapping")!=null && !System.getProperty("exo.ldap.users.attributes.email.mapping").isEmpty()) {
        this.userAttributeEmail=System.getProperty("exo.ldap.users.attributes.email.mapping");
      } else {
        this.userAttributeEmail="mail";
      }
    }
    if (initParams.getValueParam("userAttributePassword") != null &&
        !initParams.getValueParam("userAttributePassword").getValue().isEmpty()) {
      this.userAttributePassword=initParams.getValueParam("userAttributePassword").getValue();
    } else {
      if (System.getProperty("exo.ldap.users.password.attributeName")!=null && !System.getProperty("exo.ldap.users.password.attributeName").isEmpty()) {
        this.userAttributePassword=System.getProperty("exo.ldap.users.password.attributeName");
      } else {
        this.userAttributePassword = this.ldapType.equals(LDAP_TYPE) ? "userPassword" : "unicodePwd";
      }
    }

    defaultLdapGroups=new ArrayList<>();
    if (initParams.getValueParam("defaultLdapGroups")!=null &&
        !initParams.getValueParam("defaultLdapGroups").getValue().isEmpty()) {
      defaultLdapGroups.addAll(Arrays.asList(initParams.getValueParam("defaultLdapGroups").getValue().split("/")));
    }
    
    env = new Hashtable();
    env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env.put(Context.PROVIDER_URL, ldapUrl);
    env.put("com.sun.jndi.ldap.connect.pool", "true");
  
    env.put(Context.SECURITY_AUTHENTICATION, "simple");
    env.put(Context.SECURITY_PRINCIPAL, ldapPrincipal);
    env.put(Context.SECURITY_CREDENTIALS, ldapCredentials);

    this.identityManager=identityManager;
    
    
    
  }
  
  public boolean isFeatureEnabled() {
    return isFeatureEnabled;
  }

  public boolean mustBeUpdated(User user, boolean isNew) {
    LOG.debug("Check if user {} must be updated",user.getUserName());
    if (isNew) {
      LOG.debug("User {} is new, do not push it in LDAP",user.getUserName());
      return false;
    }
    SearchResult adUser=null;
    try {
      adUser = searchUser(user);
    } catch (NamingException e) {
      LOG.error("Unable to connect LDAP",e);
      return false;
    }
    if (adUser == null) {
      LOG.debug("User is not present in LDAP, need to add it");
      Identity identity =identityManager.getOrCreateIdentity(OrganizationIdentityProvider.NAME, user.getUserName());
      if (identity!=null
          && identity.getProfile()!=null
          && Boolean.parseBoolean((String)identity.getProfile().getProperty(Profile.EXTERNAL))) {
        //user is external, return false, to not push him in AD
        LOG.debug("User is external, don't put him in LDAP");

        return false;
      }
      if (user.getPassword()==null) {
        LOG.debug("User password is unknown, unable to add it in LDAP");
        return false;
      } else {
        LOG.debug("User password known, add it in LDAP");

        return true; //need to create it
      }
    } else {
  
      //user is modified if one of his main property is modified
      boolean result = false;
      try {
        result = (user.getFirstName()!=null && !user.getFirstName().equals(adUser.getAttributes().get(userAttributeFirstName).get())) ||
                (user.getLastName()!=null && !user.getLastName().equals(adUser.getAttributes().get(userAttributeLastName).get())) ||
                (user.getEmail()!=null && !user.getEmail().equals(adUser.getAttributes().get(userAttributeEmail).get()));
      } catch (NamingException e) {
        LOG.error("Error when reading attributes for user {}"+adUser);
        return false;
      }
      LOG.debug("User is present in AD, needUpdate={}",result);
      return result;
    }
    
  }
  
  private SearchResult searchUser(User user) throws NamingException {
    long startTime = System.currentTimeMillis();
    InitialDirContext ldapContext=null;
    SearchResult result=null;
    SearchControls searchControls;
    String searchQuery="";
    try {
      ldapContext = new InitialDirContext(env);
      searchControls = new SearchControls();
      searchControls.setSearchScope(SearchControls.SUBTREE_SCOPE);
      searchQuery = "("+this.userAttributeUserName+"="+user.getUserName()+")";
      NamingEnumeration<SearchResult> searchResult = ldapContext.search(userBaseDn, searchQuery, searchControls);
      if (searchResult.hasMore()) {
        result=searchResult.nextElement();
      }
    } finally {
      if (ldapContext!=null) {
        ldapContext.close();
      }
    }
    LOG.info("remote_service={} operation={} parameters=\"user:{},userFound:{}\" status=ok " + "duration_ms={}",
               LDAP_SERVICE_NAME,
               "searchUser",
               user.getUserName(),
               result!=null ? result.getAttributes().get(this.userAttributeUserName) : null,
               System.currentTimeMillis() - startTime);
    
    return result;
  }

  public void setEnabled(User user) throws NamingException {
    if (ldapType.equals("ad")) {
      LOG.debug("Push enabled status {} to AD for user {}", user.isEnabled(), user.getUserName());
      SearchResult adUser = null;
      try {
        adUser = searchUser(user);
      } catch (NamingException e) {
        LOG.error("Unable to connect LDAP", e);
      }
      if (adUser != null) {
        int userAccountControlOrig = Integer.parseInt((String)adUser.getAttributes().get(this.userAccountControl).get());
        int userAccountControlValue;
        if (user.isEnabled()) {
          //enable in AD
          userAccountControlValue=userAccountControlOrig & ~UF_ACCOUNTDISABLE;
        } else {
          //disable in AD
          userAccountControlValue = userAccountControlOrig | UF_ACCOUNTDISABLE;
        }
        if (userAccountControlOrig != userAccountControlValue) {
          List<ModificationItem> mods = new ArrayList<>();
          mods.add(new ModificationItem(DirContext.REPLACE_ATTRIBUTE, new BasicAttribute(this.userAccountControl,
                                                                                          ""+userAccountControlValue)));
          long startTime = System.currentTimeMillis();
          InitialDirContext ldapContext=null;
          SearchResult result=null;
          try {
            ldapContext = new InitialDirContext(env);
            ldapContext.modifyAttributes(adUser.getNameInNamespace(),mods.stream().toArray(ModificationItem[]::new));
            this.forceImportEntityForUser(user);
            LOG.info("remote_service={} operation={} parameters=\"user:{},enabled={},oldUserAccountControl={},"
                         + "newUserAccountControl={}\" status=ok duration_ms={}",
                     LDAP_SERVICE_NAME,
                     "setEnabled",
                     user.getUserName(),
                     user.isEnabled(),
                     userAccountControlOrig,
                     userAccountControlValue,
                     System.currentTimeMillis() - startTime);
          } finally {
            if (ldapContext!=null) {
              ldapContext.close();
            }
          }
        }
      }
    } else {
      LOG.debug("User status changed, but ldapType is ldap, so it is not possible to push the modification");
    }
  }
  
  public void saveModification(User user) {
    LOG.debug("Push modification to AD for user {}",user.getUserName());
    SearchResult adUser=null;
    try {
      adUser = searchUser(user);
    } catch (NamingException e) {
      LOG.error("Unable to connect LDAP",e);
    }
    if (adUser == null) {
      try {
        LOG.debug("User is not present in AD, need to add it");
        addUser(user);
      } catch (NamingException e) {
        LOG.error("Unable to connect LDAP",e);
      }
    } else {
      try {
        updateUser(user, adUser);
      }  catch (NamingException e) {
        LOG.error("Unable to connect LDAP",e);
      }
    }
  }
  
  private void updateUser(User user, SearchResult adUser) throws NamingException {
    long startTime = System.currentTimeMillis();
    InitialDirContext ldapContext=null;
    SearchResult result=null;
    try {
      ldapContext = new InitialDirContext(env);
      List<ModificationItem> mods = new ArrayList<>();
      
      if(!(this.ldapType.equals(LDAP_TYPE) && this.userAttributeFirstName.equals("cn"))) {
        //in ldap implementation, the cn is part of the dn
        //we must make a rename in this case
        if (!user.getFirstName().equals(adUser.getAttributes().get(this.userAttributeFirstName).get())) {
          Attribute attribute = new BasicAttribute(this.userAttributeFirstName, user.getFirstName());
          mods.add(new ModificationItem(DirContext.REPLACE_ATTRIBUTE, attribute));
        }
      }
      if (this.ldapType.equals(AD_TYPE)) {
        String dc =  this.userBaseDn.toLowerCase()
                                    .substring(this.userBaseDn.toLowerCase().indexOf("dc=")+3)
                                    .replace(",dc=",".");
        String userPrincipalName = user.getUserName()+"@"+dc;
        if (adUser.getAttributes().get("userPrincipalName")==null) {
          Attribute attribute = new BasicAttribute("userPrincipalName", userPrincipalName);
          mods.add(new ModificationItem(DirContext.ADD_ATTRIBUTE,attribute));
        } else if (!userPrincipalName.equals(adUser.getAttributes().get("userPrincipalName").get())) {
          Attribute attribute = new BasicAttribute("userPrincipalName", userPrincipalName);
          mods.add(new ModificationItem(DirContext.REPLACE_ATTRIBUTE,attribute));
        }

      }
      if (!user.getLastName().equals(adUser.getAttributes().get(this.userAttributeLastName).get())) {
        Attribute attribute = new BasicAttribute(this.userAttributeLastName, user.getLastName());
        mods.add(new ModificationItem(DirContext.REPLACE_ATTRIBUTE,attribute));
      }
      if (!user.getEmail().equals(adUser.getAttributes().get(this.userAttributeEmail).get())) {
        Attribute attribute = new BasicAttribute(this.userAttributeEmail, user.getEmail());
        mods.add(new ModificationItem(DirContext.REPLACE_ATTRIBUTE,attribute));
      }
      ldapContext.modifyAttributes(adUser.getNameInNamespace(),mods.stream().toArray(ModificationItem[]::new));

      List<String> currentGroups=new ArrayList<>();
      try {
        adUser.getAttributes().get("memberOf").getAll().asIterator().forEachRemaining(o -> currentGroups.add(((String)o).toLowerCase()));
      } catch (Exception e) {
        LOG.error("Error when reading groups of user {}",adUser.getNameInNamespace());
      }
      defaultLdapGroups.stream().forEach(groupDn -> {
        if (!currentGroups.contains(groupDn.toLowerCase())) {
          addInLdapGroup(adUser.getNameInNamespace(), groupDn.toLowerCase());
        }
      });

      if(this.ldapType.equals(LDAP_TYPE) && this.userAttributeFirstName.equals("cn") &&
          !user.getFirstName().equals(adUser.getAttributes().get(this.userAttributeFirstName).get())) {
  
        String oldDn = adUser.getNameInNamespace();
        String newDn = oldDn.replace("cn="+adUser.getAttributes().get(this.userAttributeFirstName).get(),
                                     "cn="+user.getFirstName());
        ldapContext.rename(oldDn,newDn);
      }


  
      this.forceImportEntityForUser(user);
      LOG.info("remote_service={} operation={} parameters=\"user:{}\" status=ok " + "duration_ms={}",
               LDAP_SERVICE_NAME,
               "updateUser",
               user.getUserName(),
               System.currentTimeMillis() - startTime);
    } finally {
      if (ldapContext!=null) {
        ldapContext.close();
      }
    }
    
  
  }
  
  private void addUser(User user) throws NamingException {
    long startTime = System.currentTimeMillis();
    InitialDirContext ldapContext=null;
    SearchResult result=null;
    try {
      ldapContext = new InitialDirContext(env);
  
      BasicAttributes attrs = new BasicAttributes();
      Attribute classes = new BasicAttribute("objectclass");
      if (this.ldapType.equals(LDAP_TYPE)) {
        classes.add("inetOrgPerson");
      } else {
        classes.add("user");
      }
      attrs.put(classes);
      attrs.put(this.userAttributeUserName, user.getUserName());
      attrs.put(this.userAttributeEmail, user.getEmail());
      attrs.put(this.userAttributeFirstName, user.getFirstName());
      attrs.put(this.userAttributeLastName, user.getLastName());

      if (this.ldapType.equals(LDAP_TYPE)) {
        attrs.put(encodeLdapPassword(user.getPassword(),this.userAttributePassword));
      } else {
        attrs.put(encodeAdPassword(user.getPassword(),this.userAttributePassword));
        attrs.put(USER_ACCOUNT_CONTROL,"512");
        String dc =  this.userBaseDn.toLowerCase()
                                    .substring(this.userBaseDn.toLowerCase().indexOf("dc=")+3)
                                    .replace(",dc=",".");
        String userPrincipalName = user.getUserName()+"@"+dc;
        attrs.put("userPrincipalName",userPrincipalName);
      }
      
      String dn = computeDn(user);
      ldapContext.createSubcontext(dn,attrs);

      defaultLdapGroups.stream().forEach(groupDn -> {
        addInLdapGroup(dn, groupDn);
      });
  
      this.forceImportEntityForUser(user);
      LOG.info("remote_service={} operation={} parameters=\"user:{}\" status=ok " + "duration_ms={}",
               LDAP_SERVICE_NAME,
               "addUser",
               user.getUserName(),
               result!=null ? result.getAttributes().get(this.userAttributeUserName) : null,
               System.currentTimeMillis() - startTime);
  
    } finally {
      if (ldapContext!=null) {
        ldapContext.close();
      }
    }
    
    
  }

  private void addInLdapGroup(String userDn, String groupDn) {
    InitialDirContext groupLdapContext = null;
    try {
      groupLdapContext = new InitialDirContext(env);

      ModificationItem[] mods = new ModificationItem[1];
      Attribute mod = new BasicAttribute("member", userDn);
      mods[0] = new ModificationItem(DirContext.ADD_ATTRIBUTE, mod);
      groupLdapContext.modifyAttributes(groupDn, mods);
    } catch (NamingException e) {
      LOG.error("Error adding user {} in group {}",userDn, groupDn, e);
    } finally {
      if (groupLdapContext!=null) {
        try {
          groupLdapContext.close();
        } catch (NamingException e) {

        }
      }
    }
  }

  private String computeDn(User user) {
    String dn = "cn=";
    if (this.ldapType.equals(LDAP_TYPE)) {
      dn+=user.getFirstName();
    } else {
      dn+=user.getDisplayName();
    }
    dn+=","+this.userBaseDn;
    return dn;
    
  }
  
  private Attribute encodeLdapPassword(String password, String passwordAttribute) {
    Attribute mod0 = new BasicAttribute(passwordAttribute, password);
    return mod0;
  }
  
  private Attribute encodeAdPassword(String password, String passwordAttribute) {
    String quotedPassword = "\"" + password + "\"";
    char unicodePwd[] = quotedPassword.toCharArray();
    byte pwdArray[] = new byte[unicodePwd.length * 2];
    for (int i=0; i<unicodePwd.length; i++) {
      pwdArray[i*2 + 1] = (byte) (unicodePwd[i] >>> 8);
      pwdArray[i*2 + 0] = (byte) (unicodePwd[i] & 0xff);
    }
    Attribute mod0 = new BasicAttribute(passwordAttribute, pwdArray);
    return mod0;
  }
  
  public void forceImportEntityForUser(User user) {
    try {
      this.externalStoreImportService.importEntityToInternalStore(IDMEntityType.USER,
                                                                  user.getUserName(),
                                                                  true,
                                                                  true);
    } catch (Exception e) {
      LOG.error("Error when forcing sync",e);
    }
    //as import user service will not see difference between exo user and LDAP user,
    //the event USER_MODIFIED_FROM_EXTERNAL_STORE will be not fired, and the sync date not added.
    //so we launch it manually
    try {
      // The user information creation listener triggering is optional,
      // thus this is surrounded by try/catch
      listenerService.broadcast(IDMExternalStoreService.USER_MODIFIED_FROM_EXTERNAL_STORE, externalStoreImportService, user);
    } catch (Exception e) {
      LOG.warn("Error while triggering event on user '" + user.getUserName() + "' data import (modification) from external store", e);
    }
    
  }


}
