/*
 * Copyright (C) 2025 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.listener;

import io.meeds.analytics.model.StatisticData;
import io.meeds.analytics.utils.AnalyticsUtils;
import jakarta.annotation.PostConstruct;
import org.exoplatform.antimalware.service.MalwareDetectionService;
import org.exoplatform.services.listener.Asynchronous;
import org.exoplatform.services.listener.Event;
import org.exoplatform.services.listener.Listener;
import org.exoplatform.services.listener.ListenerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import java.util.Map;

@Asynchronous
@Component
@Profile("analytics")
public class AntiMalwareAnalyticsListener extends Listener<String, Object> {

  @Autowired
  private ListenerService         listenerService;

  @Autowired
  private MalwareDetectionService malwareDetectionService;

  private static final String     SCAN_COMPLETED              = "anti-malware-scan-completed";

  private static final String     INFECTED_FILE_REMOVED       = "anti-malware-infectedFile-removed";

  private static final String     INFECTED_FILE_CLEANED       = "infected-file-cleaned";

  private static final String     INFECTED_FILE_NAME          = "infectedItemName";

  private static final String     INFECTED_FILE_LAST_MODIFIER = "infectedItemLastModifier";

  private static final String     INFECTED_FILE_PATH          = "infectedItemPath";

  public static final String      INFECTED_FILE_ID            = "infectedItemId";

  private final String[]          eventTypes                  = { SCAN_COMPLETED, INFECTED_FILE_REMOVED, INFECTED_FILE_CLEANED };

  @PostConstruct
  public void init() {
    for (String eventType : eventTypes) {
      listenerService.addListener(eventType, this);
    }
  }

  @Override
  public void onEvent(Event<String, Object> event) throws Exception {
    if (event.getEventName().equals(SCAN_COMPLETED)) {
      String connectorType = event.getSource();
      int infectedFilesCount = (int) event.getData();
      addAntimalwareScanStatistics(connectorType, infectedFilesCount);
    } else if (event.getEventName().equals(INFECTED_FILE_REMOVED) || event.getEventName().equals(INFECTED_FILE_CLEANED)) {
      String connectorType = malwareDetectionService.getDefaultMalwareDetectionConnector().getType();
      Map<String, String> infectedItem = (Map<String, String>) event.getData();
      addAntimalwareInfectedFileRemoveStatistics(connectorType, infectedItem);
    }
  }

  private static void addAntimalwareScanStatistics(String connectorType, int infectedFilesCount) {
    StatisticData scanStatistics = new StatisticData();
    scanStatistics.setModule("AntiMalware");
    scanStatistics.setOperation("antimalwareCheck");
    scanStatistics.addParameter("infectedFilesCount", infectedFilesCount);
    scanStatistics.addParameter("connectorType", connectorType);
    AnalyticsUtils.addStatisticData(scanStatistics);
  }

  private static void addAntimalwareInfectedFileRemoveStatistics(String connectorType, Map<String, String> infectedItem) {
    StatisticData fileRemovalStatistics = new StatisticData();
    fileRemovalStatistics.setModule("AntiMalware");
    fileRemovalStatistics.setOperation("InfectedFileRemoval");
    fileRemovalStatistics.addParameter("fileId", infectedItem.get(INFECTED_FILE_ID));
    fileRemovalStatistics.addParameter("fileName", infectedItem.get(INFECTED_FILE_NAME));
    fileRemovalStatistics.addParameter("filePath", infectedItem.get(INFECTED_FILE_PATH));
    fileRemovalStatistics.addParameter("fileAuthor", infectedItem.get(INFECTED_FILE_LAST_MODIFIER));
    fileRemovalStatistics.addParameter("connectorType", connectorType);
    AnalyticsUtils.addStatisticData(fileRemovalStatistics);
  }
}
