/*
 * Copyright (C) 2025 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */

package org.exoplatform.antimalware.connector;

import jakarta.annotation.PostConstruct;
import org.exoplatform.antimalware.service.MalwareDetectionService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.stream.Stream;

@Component
public class ClamAVMalwareDetectionConnector extends MalwareDetectionConnector {

  private static final Log        LOG        = ExoLogger.getExoLogger(ClamAVMalwareDetectionConnector.class);

  @Value("${exo.malwareDetection.connector.clamav.report.path:${exo.data.dir}/clamav-report.txt}")
  private String                  reportPath = null;

  @Value("${exo.malwareDetection.connector.clamav.isDefault:false}")
  private boolean                 isDefault  = false;

  private String                  type       = "ClamAV";

  @Autowired
  private MalwareDetectionService malwareDetectionService;

  @PostConstruct
  public void init() {
    setType(type);
    setDefault(isDefault);
    malwareDetectionService.addMalwareDetectionConnector(this);
  }

    @Override
    public List<String> getInfectedItemsPaths() {
      if (reportPath == null) {
        return List.of();
      }

      Path path = Paths.get(reportPath);
      if (!Files.exists(path) || isFileEmpty(path)) {
        return List.of();
      }

      List<String> infectedPaths;
      try (Stream<String> lines = Files.lines(path)) {
        infectedPaths = lines
                .map(String::trim)
                .filter(line -> line.endsWith("FOUND"))
                .map(line -> {
                    String[] parts = line.split(": ", 2);
                    return (parts.length >= 1) ? parts[0].trim() : null;
                })
                .filter(p -> p != null && !p.isEmpty())
                .toList();

      } catch (IOException e) {
        LOG.error("Error reading infected items paths from ClamAV report", e);
        return List.of();
      }

      // Clear the report only after successful parsing
      clearReportFile(path);

      LOG.info("ClamAV scan found {} infected item(s)", infectedPaths.size());
      return infectedPaths;
  }

  private void clearReportFile(Path path) {
    try {
      Files.newBufferedWriter(path, StandardOpenOption.TRUNCATE_EXISTING).close();
      LOG.debug("ClamAV report file truncated after processing: {}", reportPath);
    } catch (IOException ex) {
      LOG.warn("Failed to truncate ClamAV report file: {}", reportPath, ex);
    }
  }

  private boolean isFileEmpty(Path path) {
    try {
      return Files.size(path) == 0;
    } catch (IOException e) {
      LOG.warn("Failed to check if ClamAV report file is empty: {}", reportPath, e);
      return false;
    }
  }
}
