/*
 * Copyright (C) 2003-2007 eXo Platform SAS.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see<http://www.gnu.org/licenses/>.
 */
package org.exoplatform.services.workflow.impl.bonita;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import hero.interfaces.BnNodeLocal;
import hero.interfaces.BnNodePropertyLocal;
import hero.interfaces.BnProjectLocal;
import hero.interfaces.BnProjectLocalHome;
import hero.interfaces.BnProjectPK;
import hero.interfaces.BnProjectPropertyLocal;
import hero.interfaces.BnProjectUtil;

import org.exoplatform.services.workflow.Form;

/**
 * This class represents a Form that is automatically generated by eXo
 * 
 * Created by Bull R&D
 * @author Brice Revenant
 * Dec 30, 2005
 */
//@SuppressWarnings({"unchecked"})
public class AutomaticFormImpl implements Form {
  /** Localized Resource Bundle corresponding to this Form */
  private ResourceBundle bundle;

  /** Name of the State corresponding to this Form */
  private String stateName;
  
  /** Submit buttons corresponding to this Form */
  private List<Map<String, Object>> submitButtons;
  
  /** Variables corresponding to this Form */
  private Hashtable<String, Map<String, Object>> variables;
  
  /**
   * This constructor is invoked to create a new Automatic Form.
   * The Process instance identifier is not specified, which means propaged
   * properties are not displayed. 
   * 
   * @param processId identifies the Process to which the State belongs
   * @param stateName name of the State corresponding to the Form to be created
   * @param locale    locale of the Form
   */
  public AutomaticFormImpl(String processId,
                           String stateName,
                           Locale locale) {
    try {
      // Retrieve the Project Model in Bonita
      BnProjectLocalHome projectHome = BnProjectUtil.getLocalHome();
      BnProjectLocal projectModel    = projectHome.findByPrimaryKey(
          new BnProjectPK(processId));
      
      if(ProcessData.START_STATE_NAME.equals(stateName)) {
        // The Form corresponds to a Start Process one
        this.stateName = ProcessData.START_STATE_NAME;
        // TODO Translate me
        this.bundle    = new AutomaticFormBundle("start");
        this.variables = new Hashtable<String, Map<String, Object>>();
        
        /*
         * By convention, eXo considers that a panel corresponds to a start
         * Process one when it contains no submit button. In addition there
         * needs to be a bundle string for the default button.
         */ 
        this.submitButtons = new ArrayList<Map<String, Object>>();
        ((AutomaticFormBundle) this.bundle).addDefaultButton();
        
        // Only project variables need to be processed when starting a process
        initializeProjectVariables(projectModel);
      }
      else {
        // The Form corresponds to an Activity. Retrieve the corresponding Node.    
        BnNodeLocal node = projectModel.getBnNode(stateName);
        this.stateName   = node.getName();
        this.bundle      = new AutomaticFormBundle(this.stateName);
        this.variables   = new Hashtable<String, Map<String, Object>>();
        
        /*
         * In Bonita, we consider this is the Workflow duty to determine which
         * activity comes next hence a single default button for all activities.
         */
        this.submitButtons = new ArrayList<Map<String, Object>>();
        Map<String, Object> attributes = new HashMap<String, Object>();
        // TODO Translate me
        attributes.put("name", "submit");
        attributes.put("transition", "");
        this.submitButtons.add(attributes);
        ((AutomaticFormBundle) this.bundle).addButton(attributes);
        
        // Project variables are processed last so they have a higher priority
        initializeNodeVariables(node);
        initializeProjectVariables(projectModel);
      }
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#getCustomizedView()
   */
  public String getCustomizedView() {
    // The form does not correspond to a customized view as it is automatic
    return null;
  }
  
  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#getIconURL()
   */
  public String getIconURL() {
    // No image is rendered when the empty string is returned
    return "";
  }

  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#getResourceBundle()
   */
  public ResourceBundle getResourceBundle() {
    return this.bundle;
  }
  
  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#getStateImageURL()
   */
  public String getStateImageURL() {
    // No image is rendered when the empty string is returned
    return "";
  }  

  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#getStateName()
   */
  public String getStateName() {
    return stateName;
  }

  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#getSubmitButtons()
   */
  public List getSubmitButtons() {
    return submitButtons;
  }

  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#getVariables()
   */
  public List getVariables() {
    // Need to create a List from the Hashtable values
    List ret = new ArrayList<Map<String, Object>>();
    
    for(Map<String, Object> attributes : variables.values()) {
      ret.add(attributes);
    }
    
    return ret;
  }
  
  private void initializeNodeVariables(BnNodeLocal node) {
    Collection<BnNodePropertyLocal> properties = node.getBnProperties();
    
    for(BnNodePropertyLocal property : properties) {
      // Retrieve the name
      String name = property.getTheKey();
      
      // Create variable attributes
      Map<String, Object> attributes = new HashMap<String, Object>();
      attributes.put("name",      name);
      attributes.put("editable",  "true");
      attributes.put("mandatory", "false");
      
      Collection<String> possibleValues = property.getPossibleValues();
      if(possibleValues != null) {
        /*
         * In case of multiple values, the component is select and different
         * values are specified in the variable attributes, so that the bundle
         * can retrieve them. Constant variables in UITask.java cannot be
         * referenced since this class is currently not contained by a jar.
         */
        attributes.put("component", "select");
        attributes.put("possible-values", possibleValues);
      } else {
        // The default component is text
        attributes.put("component", "text");
      }
      
      // Include the variable in the variables list and in the bundle
      this.variables.put(name, attributes);
      ((AutomaticFormBundle) this.bundle).addVariable(attributes);
    }
  }
  
  private void initializeProjectVariables(BnProjectLocal projectModel) {
    Collection<BnProjectPropertyLocal> properties =
      projectModel.getBnProperties();
    
    for(BnProjectPropertyLocal property : properties) {
      // Retrieve the name
      String name = property.getTheKey();

      // Create variable attributes
      Map<String, Object> attributes = new HashMap<String, Object>();
      attributes.put("name",      name);
      attributes.put("editable",  "true");
      attributes.put("mandatory", "false");
      
      Collection<String> possibleValues = property.getPossibleValues();
      if(possibleValues != null) {
        /*
         * In case of multiple values, the component is select and different
         * values are specified in the variable attributes, so that the bundle
         * can retrieve them. Constant variables in UITask.java cannot be
         * referenced since this class is currently not contained by a jar.
         */
        attributes.put("component", "select");
        attributes.put("possible-values", possibleValues);
      } else {
        // The default component is text
        attributes.put("component", "text");
      }
      
      // Include the variable in the variables list and in the bundle
      this.variables.put(name, attributes);
      ((AutomaticFormBundle) this.bundle).addVariable(attributes);
    }
  }

  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#isCustomizedView()
   */
  public boolean isCustomizedView() {
    // The form does not correspond to a customized view as it is automatic
    return false;
  }

  /* (non-Javadoc)
   * @see org.exoplatform.services.workflow.Form#isDelegatedView()
   */
  public boolean isDelegatedView() {
    // There is no delegated view as the form is automatic
    return false;
  }
}
