/*
 * Copyright (C) 2003-2008 eXo Platform SAS.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see<http://www.gnu.org/licenses/>.
 */
package org.exoplatform.services.jcr.ext.replication.recovery;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.exoplatform.services.jcr.ext.replication.ChannelManager;
import org.exoplatform.services.jcr.ext.replication.ReplicationService;
import org.exoplatform.services.jcr.ext.replication.priority.AbstractPriorityChecker;
import org.exoplatform.services.jcr.ext.replication.priority.DynamicPriorityChecker;
import org.exoplatform.services.jcr.ext.replication.priority.MemberListener;
import org.exoplatform.services.jcr.ext.replication.priority.StaticPriorityChecker;
import org.exoplatform.services.jcr.storage.WorkspaceDataContainer;
import org.exoplatform.services.log.ExoLogger;
import org.jgroups.Address;
import org.jgroups.Channel;
import org.jgroups.ChannelListener;
import org.jgroups.MembershipListener;
import org.jgroups.View;

/**
 * Created by The eXo Platform SAS.
 * 
 * @author <a href="mailto:alex.reshetnyak@exoplatform.com.ua">Alex Reshetnyak</a>
 * @version $Id: ConectionFailDetector.java 111 2008-11-11 11:11:11Z rainf0x $
 */

public class ConnectionFailDetector implements ChannelListener, MembershipListener, MemberListener {
  /**
   * The apache logger.
   */
  private static Log                    log          = ExoLogger.getLogger("ext.ConnectionFailDetector");

  /**
   * Definition the BEFORE_CHECK timeout.
   */
  private static final int              BEFORE_CHECK = 10000;

  /**
   * Definition the BEFORE_INIT timeout.
   */
  private static final int              BEFORE_INIT  = 60000;

  /**
   * Definition the AFTER_INIT timeout.
   */
  private static final int              AFTER_INIT   = 60000;

  /**
   * The ChannalManager will be transmitted or receive the Packets.
   */
  private final ChannelManager          channelManager;

  /**
   * The channel name.
   */
  private String                        channelName;

  /**
   * The ReconectTtread will be initialized reconnect to cluster.
   */
  private ReconectTtread                reconectTtread;

  /**
   * Start value for lastViewSize.
   */
  private int                           lastViewSize = 2;

  /**
   * Start value for allInited.
   */
  private boolean                       allInited    = false;

  /**
   * The WorkspaceDataContainer will be used to workspace for set state 'read-only'.
   */
  private final WorkspaceDataContainer  dataContainer;

  /**
   * The RecoveryManager will be initialized cluster node synchronization. 
   */
  private final RecoveryManager         recoveryManager;

  /**
   * The list of address for suspect members.
   */
  private List<Address>                 suspectMembers;

  /**
   * The own priority value.
   */
  private final int                     ownPriority;

  /**
   * The own name in cluster. 
   */
  private final String                  ownName;

  /**
   * The list of names to other participants in cluster.
   */
  private final List<String>            otherPartisipants;

  /**
   * The priority checker (static or dynamic).
   */
  private final AbstractPriorityChecker priorityChecker;

  /**
   * ConnectionFailDetector  constructor.
   *
   * @param channelManager
   *          the ChannelManager
   * @param dataContainer
   *          the WorkspaceData
   * @param recoveryManager
   *          the RecoveryManager
   * @param ownPriority
   *          the own priority
   * @param otherParticipants
   *          the list of names to other participants in cluster
   * @param ownName
   *          the own name in cluster
   * @param priprityType
   *          the priority type (dynamic or static)s
   */
  public ConnectionFailDetector(ChannelManager channelManager,
                                WorkspaceDataContainer dataContainer,
                                RecoveryManager recoveryManager,
                                int ownPriority,
                                List<String> otherParticipants,
                                String ownName,
                                String priprityType) {
    this.channelManager = channelManager;
    this.channelManager.setChannelListener(this);

    this.dataContainer = dataContainer;
    this.recoveryManager = recoveryManager;

    this.ownPriority = ownPriority;

    this.ownName = ownName;
    this.otherPartisipants = new ArrayList<String>(otherParticipants);

    if (priprityType.equals(ReplicationService.PRIORITY_STATIC_TYPE))
      priorityChecker = new StaticPriorityChecker(channelManager,
                                                  ownPriority,
                                                  ownName,
                                                  otherParticipants);
    else
      priorityChecker = new DynamicPriorityChecker(channelManager,
                                                   ownPriority,
                                                   ownName,
                                                   otherParticipants);

    priorityChecker.setMemberListener(this);
  }

  /**
   * {@inheritDoc}
   */
  public void channelClosed(Channel channel) {
    if (log.isDebugEnabled())
      log.debug("Channel closed : " + channel.getClusterName());
  }

  /**
   * {@inheritDoc}
   */
  public void channelConnected(Channel channel) {
    if (log.isDebugEnabled())
      log.debug("Channel connected : " + channel.getClusterName());

    channelName = channel.getClusterName();
  }

  /**
   * {@inheritDoc}
   */
  public void channelDisconnected(Channel channel) {
    if (log.isDebugEnabled())
      log.debug("Channel disconnected : " + channel.getClusterName());
  }

  /**
   * {@inheritDoc}
   */
  public void channelReconnected(Address address) {
  }

  /**
   * {@inheritDoc}
   */
  public void channelShunned() {
  }

  /**
   * {@inheritDoc}
   */
  public void block() {
  }

  /**
   * {@inheritDoc}
   */
  public void suspect(Address adrress) {
    if (log.isDebugEnabled())
      log.debug(" ------->>> MembershipListener.suspect : " + adrress.toString());

    if (suspectMembers == null)
      suspectMembers = new ArrayList<Address>();

    if (!suspectMembers.contains(adrress))
      suspectMembers.add(adrress);
  }

  /**
   * {@inheritDoc}
   */
  public void viewAccepted(View view) {
    log.info(" Memebers view :" + view.printDetails());

    priorityChecker.informAll();

    if (view.size() > 1)
      allInited = true;

    if (allInited == true)
      lastViewSize = view.size();

    if (priorityChecker instanceof StaticPriorityChecker || otherPartisipants.size() == 1) {

      if (log.isDebugEnabled()) {
        log.debug("lastViewSize == 1 && !priorityChecker.isMaxPriority() == "
            + (lastViewSize == 1 && !priorityChecker.isMaxPriority()));
        log.debug("lastViewSize > 1 && !priorityChecker.isMaxOnline() == "
            + (lastViewSize > 1 && !priorityChecker.isMaxOnline()));
      }

      if (lastViewSize == 1 && !priorityChecker.isMaxPriority()) {
        if (reconectTtread == null || reconectTtread.isStoped() == true) {
          reconectTtread = new ReconectTtread(true);
          reconectTtread.start();
          memberSuspect();
        }
      } else if (reconectTtread != null && priorityChecker.isAllOnline()) {
        reconectTtread.setStop(false);
        reconectTtread = null;
      } else if (lastViewSize > 1 && !priorityChecker.isMaxOnline()) {
        if (reconectTtread == null || reconectTtread.isStoped() == true) {
          reconectTtread = new ReconectTtread(true);
          reconectTtread.start();
          memberSuspect();
        }
      }
    } else if (priorityChecker instanceof DynamicPriorityChecker && otherPartisipants.size() > 1) {

      if (lastViewSize == 1 && !priorityChecker.isMaxPriority()) {
        if (reconectTtread == null || reconectTtread.isStoped() == true) {
          reconectTtread = new ReconectTtread(true);
          reconectTtread.start();
          memberSuspect();
        }
      } else if (reconectTtread != null && priorityChecker.isAllOnline()) {
        reconectTtread.setStop(false);
        reconectTtread = null;
      }
    }
  }

  /**
   * The ReconectTtread will be initialized reconnect to cluster.
   */
  private class ReconectTtread extends Thread {
    /**
     * The 'isStop' is a flag to run() stop. 
     */
    private boolean isStop;

    /**
     * ReconectTtread  constructor.
     *
     * @param isStop
     *          the 'isStop' value
     */
    public ReconectTtread(boolean isStop) {
      this.isStop = isStop;
    }

    /**
     * {@inheritDoc}
     */
    public void run() {
      while (isStop) {
        try {
          log.info("Connect to channel : " + channelName);
          Thread.sleep(BEFORE_CHECK);

          int curruntOnlin = 1;

          if (channelManager.getChannel() != null)
            curruntOnlin = channelManager.getChannel().getView().size();

          if (curruntOnlin <= 1 || ((curruntOnlin > 1) && !priorityChecker.isMaxOnline())) {
            channelManager.closeChannel();

            Thread.sleep(BEFORE_INIT);

            channelManager.init();
            channelManager.connect();
          } else {
            isStop = false;
          }
          Thread.sleep(AFTER_INIT);
        } catch (Exception e) {
          log.info(e, e);
        }
      }
    }

    /**
     * setStop.
     *
     * @param isStop
     *          the 'isStop' value
     */
    public void setStop(boolean isStop) {
      this.isStop = isStop;
    }

    /**
     * isStoped.
     *
     * @return boolean
     *           return the 'isStop' value
     */
    public boolean isStoped() {
      return !isStop;
    }
  }

  /**
   * {@inheritDoc}
   */
  public void memberRejoin() {

    dataContainer.setReadOnly(false);
    log.info(dataContainer.getName() + " set not read-only");
    log.info(dataContainer.getName() + " recovery start ...");
    recoveryManager.startRecovery();
  }

  /**
   * Call this method if maxPriority member was suspected.
   * 
   */
  public void memberSuspect() {
    log.info(dataContainer.getName() + " set read-only");
    dataContainer.setReadOnly(true);
  }
}
