/*
 * Copyright (C) 2003-2008 eXo Platform SAS.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see<http://www.gnu.org/licenses/>.
 */

package org.exoplatform.services.jcr.webdav.command;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.jcr.AccessDeniedException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.lock.LockException;
import javax.xml.namespace.QName;

import org.exoplatform.common.util.HierarchicalProperty;
import org.exoplatform.services.jcr.webdav.WebDavStatus;
import org.exoplatform.services.jcr.webdav.command.order.OrderMember;
import org.exoplatform.services.jcr.webdav.command.order.OrderPatchResponseEntity;
import org.exoplatform.services.jcr.webdav.util.TextUtil;
import org.exoplatform.services.jcr.webdav.xml.WebDavNamespaceContext;
import org.exoplatform.services.rest.Response;

/**
 * Created by The eXo Platform SAS. Author : Vitaly Guly <gavrikvetal@gmail.com>
 * 
 * @version $Id: OrderPatchCommand.java 35184 2009-08-07 14:14:37Z pnedonosko $
 */

public class OrderPatchCommand {

  public OrderPatchCommand() {
  }

  public Response orderPatch(Session session, String path, HierarchicalProperty body, String baseURI) {
    try {
      Node node = (Node) session.getItem(path);

      List<OrderMember> members = getMembers(body);

      WebDavNamespaceContext nsContext = new WebDavNamespaceContext(session);
      URI uri = new URI(TextUtil.escape(baseURI + node.getPath(), '%', true));

      if (doOrder(node, members)) {
        return Response.Builder.ok().build();
      }

      OrderPatchResponseEntity orderPatchEntity = new OrderPatchResponseEntity(nsContext,
                                                                               uri,
                                                                               node,
                                                                               members);
      return Response.Builder.withStatus(WebDavStatus.MULTISTATUS).entity(orderPatchEntity).build();
    } catch (PathNotFoundException exc) {
      return Response.Builder.notFound().build();
    } catch (LockException exc) {
      return Response.Builder.withStatus(WebDavStatus.LOCKED).build();
    } catch (Exception exc) {
      return Response.Builder.serverError().build();
    }

  }

  protected List<OrderMember> getMembers(HierarchicalProperty body) {
    ArrayList<OrderMember> members = new ArrayList<OrderMember>();
    List<HierarchicalProperty> childs = body.getChildren();
    for (int i = 0; i < childs.size(); i++) {
      OrderMember member = new OrderMember(childs.get(i));
      members.add(member);
    }
    return members;
  }

  protected boolean doOrder(Node parentNode, List<OrderMember> members) {
    boolean success = true;
    for (int i = 0; i < members.size(); i++) {
      OrderMember member = members.get(i);

      int status = WebDavStatus.OK;

      try {
        parentNode.getSession().refresh(false);
        String positionedNodeName = null;

        if (!parentNode.hasNode(member.getSegment())) {
          throw new PathNotFoundException();
        }

        if (!new QName("DAV:", "last").equals(member.getPosition())) {
          NodeIterator nodeIter = parentNode.getNodes();
          boolean finded = false;

          while (nodeIter.hasNext()) {
            Node curNode = nodeIter.nextNode();

            if (new QName("DAV:", "first").equals(member.getPosition())) {
              positionedNodeName = curNode.getName();
              finded = true;
              break;
            }

            if (new QName("DAV:", "before").equals(member.getPosition())
                && curNode.getName().equals(member.getPositionSegment())) {
              positionedNodeName = curNode.getName();
              finded = true;
              break;
            }

            if (new QName("DAV:", "after").equals(member.getPosition())
                && curNode.getName().equals(member.getPositionSegment())) {
              if (nodeIter.hasNext()) {
                positionedNodeName = nodeIter.nextNode().getName();
              }
              finded = true;
              break;
            }
          }

          if (!finded) {
            throw new AccessDeniedException();
          }
        }

        parentNode.getSession().refresh(false);
        parentNode.orderBefore(member.getSegment(), positionedNodeName);
        parentNode.getSession().save();

      } catch (LockException exc) {
        status = WebDavStatus.LOCKED;

      } catch (PathNotFoundException exc) {
        status = WebDavStatus.FORBIDDEN;

      } catch (AccessDeniedException exc) {
        status = WebDavStatus.FORBIDDEN;

      } catch (RepositoryException exc) {
        status = WebDavStatus.INTERNAL_SERVER_ERROR;

      }

      member.setStatus(status);
      if (status != WebDavStatus.OK) {
        success = false;
      }
    }

    return success;
  }

}
