/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.exoplatform.services.jcr.impl.core.query;

import org.apache.lucene.index.*;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.WildcardQuery;
import org.apache.lucene.util.ReaderUtil;
import org.exoplatform.commons.utils.ClassLoading;
import org.exoplatform.commons.utils.PropertyManager;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.container.configuration.ConfigurationManager;
import org.exoplatform.management.annotations.Managed;
import org.exoplatform.management.annotations.ManagedDescription;
import org.exoplatform.management.annotations.ManagedName;
import org.exoplatform.management.jmx.annotations.NameTemplate;
import org.exoplatform.management.jmx.annotations.Property;
import org.exoplatform.services.document.DocumentReaderService;
import org.exoplatform.services.jcr.RepositoryService;
import org.exoplatform.services.jcr.config.*;
import org.exoplatform.services.jcr.core.WorkspaceContainerFacade;
import org.exoplatform.services.jcr.core.nodetype.NodeTypeDataManager;
import org.exoplatform.services.jcr.dataflow.ItemDataConsumer;
import org.exoplatform.services.jcr.dataflow.ItemState;
import org.exoplatform.services.jcr.dataflow.ItemStateChangesLog;
import org.exoplatform.services.jcr.dataflow.persistent.MandatoryItemsPersistenceListener;
import org.exoplatform.services.jcr.datamodel.*;
import org.exoplatform.services.jcr.impl.Constants;
import org.exoplatform.services.jcr.impl.backup.*;
import org.exoplatform.services.jcr.impl.backup.rdbms.DataRestoreContext;
import org.exoplatform.services.jcr.impl.backup.rdbms.DirectoryRestore;
import org.exoplatform.services.jcr.impl.checker.InspectionReport;
import org.exoplatform.services.jcr.impl.core.LocationFactory;
import org.exoplatform.services.jcr.impl.core.NamespaceRegistryImpl;
import org.exoplatform.services.jcr.impl.core.SessionDataManager;
import org.exoplatform.services.jcr.impl.core.SessionImpl;
import org.exoplatform.services.jcr.impl.core.query.lucene.*;
import org.exoplatform.services.jcr.impl.core.value.NameValue;
import org.exoplatform.services.jcr.impl.core.value.PathValue;
import org.exoplatform.services.jcr.impl.core.value.ValueFactoryImpl;
import org.exoplatform.services.jcr.impl.dataflow.persistent.WorkspacePersistentDataManager;
import org.exoplatform.services.jcr.impl.util.io.DirectoryHelper;
import org.exoplatform.services.jcr.impl.util.io.FileCleanerHolder;
import org.exoplatform.services.jcr.storage.WorkspaceDataContainer;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.rpc.RPCException;
import org.exoplatform.services.rpc.RPCService;
import org.exoplatform.services.rpc.RemoteCommand;
import org.picocontainer.Startable;

import javax.jcr.Node;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.query.InvalidQueryException;
import javax.jcr.query.Query;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Created by The eXo Platform SAS.
 * 
 * <br>Date:
 *
 * @author <a href="karpenko.sergiy@gmail.com">Karpenko Sergiy</a> 
 * @version $Id: SearchManager.java 1008 2009-12-11 15:14:51Z nzamosenchuk $
 */
@Managed
@NameTemplate(@Property(key = "service", value = "SearchManager"))
public class SearchManager implements Startable, MandatoryItemsPersistenceListener, Suspendable, Backupable
{

   /**
    * Logger instance for this class
    */
   private static final Log LOG = ExoLogger.getLogger("exo.jcr.component.core.SearchManager");

   /**
    * Used to display date and time for JMX components 
    */
   private static final DateTimeFormatter sdf  =DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

   protected final QueryHandlerEntry config;

   /**
    * Text extractor for extracting text content of binary properties.
    */
   protected final DocumentReaderService extractor;

   /**
    * QueryHandler where query execution is delegated to
    */

   protected QueryHandler handler;

   /**
    * The shared item state manager instance for the workspace.
    */
   protected final ItemDataConsumer itemMgr;

   /**
    * The namespace registry of the repository.
    */
   protected final NamespaceRegistryImpl nsReg;

   /**
    * The node type registry.
    */
   protected final NodeTypeDataManager nodeTypeDataManager;

   /**
    * QueryHandler of the parent search manager or <code>null</code> if there
    * is none.
    */
   protected final SearchManager parentSearchManager;

   protected IndexingTree indexingTree;

   private final ConfigurationManager cfm;

   protected LuceneVirtualTableResolver virtualTableResolver;

   protected IndexerChangesFilter changesFilter;

   protected final FileCleanerHolder cleanerHolder;

   /**
    * The Repository name.
    */
   protected final String repositoryName;

   /**
    * The Workspace name.
    */
   protected final String workspaceName;

   /**
    * Indicates if workspace is system or not.
    */
   protected final boolean isSystem;

   /**
    * The repository service.
    */
   protected final RepositoryService rService;

   /**
    * The unique name of the related workspace
    */
   protected final String wsId;

   /**
    * Whether to enable remote RPC calls or not
    */
   protected final boolean enableRemoteCalls;

   /**
    * The unique name of the workspace container
    */
   protected final String wsContainerId;

   /**
    * Component responsible for executing commands in cluster nodes.
    */
   protected final RPCService rpcService;

   /**
    * Indicates if component suspended or not.
    */
   protected final AtomicBoolean isSuspended = new AtomicBoolean(false);

   /**
    * Indicates that node keep responsible for resuming.
    */
   protected final AtomicBoolean isResponsibleForResuming = new AtomicBoolean(false);

   /**
    * Suspend remote command.
    */
   private RemoteCommand suspend;

   /**
    * Resume remote command.
    */
   private RemoteCommand resume;

   /**
    * Index recovery.
    */
   private final IndexRecovery indexRecovery;

   /**
    * Request to all nodes to check if there is someone who responsible for resuming.
    */
   private RemoteCommand requestForResponsibleForResuming;

   /**
    * Switches index between online and offline modes
    */
   private RemoteCommand changeIndexState;

   private final ExoContainerContext ctx;

   private String hotReindexingState = "not stated";

   private final String SUFFIX_ASYNC_REINDEXING = "-async-reindexing";

   /**
    * Name of max clause count property.
    */

   private static final String LUCENE_BOOLEAN_QUERY_MAX_CLAUSE_COUNT = "org.apache.lucene.maxClauseCount";


   static
   {
      String max = PropertyManager.getProperty(LUCENE_BOOLEAN_QUERY_MAX_CLAUSE_COUNT);
      int value = Integer.MAX_VALUE;
      if (max != null)
      {
         try
         {
            value = Integer.valueOf(max);
         }
         catch (NumberFormatException e)
         {
            LOG.warn("The value of the property '" + LUCENE_BOOLEAN_QUERY_MAX_CLAUSE_COUNT
               + "' must be an integer, the default value will be used.");
         }
      }
      BooleanQuery.setMaxClauseCount(value);
   }

   public SearchManager(ExoContainerContext ctx, WorkspaceEntry wEntry, RepositoryEntry rEntry,
      RepositoryService rService, QueryHandlerEntry config, NamespaceRegistryImpl nsReg, NodeTypeDataManager ntReg,
      WorkspacePersistentDataManager itemMgr, SystemSearchManagerHolder parentSearchManager,
      DocumentReaderService extractor, ConfigurationManager cfm,
      final RepositoryIndexSearcherHolder indexSearcherHolder, FileCleanerHolder cleanerHolder)
      throws RepositoryException, RepositoryConfigurationException
   {
      this(ctx, wEntry, rEntry, rService, config, nsReg, ntReg, itemMgr, parentSearchManager, extractor, cfm,
         indexSearcherHolder, null, cleanerHolder);
   }

   /**
    * Creates a new <code>SearchManager</code>.
    * 
    * @param ctx
    *            The eXo Container context in which the SearchManager is registered
    * @param rEntry
    *            repository configuration
    * @param rService
    *            repository service  
    * @param config
    *            the search configuration.
    * @param nsReg
    *            the namespace registry.
    * @param ntReg
    *            the node type registry.
    * @param itemMgr
    *            the shared item state manager.
    * @param parentSearchManager
    *            the parent search manager or <code>null</code> if there is no
    *            parent search manager.
    * @param rpcService
    *             the service for executing commands on all nodes of cluster           
    * @throws RepositoryException
    *             if the search manager cannot be initialized
    * @throws RepositoryConfigurationException
    */
   public SearchManager(ExoContainerContext ctx, WorkspaceEntry wEntry, RepositoryEntry rEntry,
      RepositoryService rService, QueryHandlerEntry config, NamespaceRegistryImpl nsReg, NodeTypeDataManager ntReg,
      WorkspacePersistentDataManager itemMgr, SystemSearchManagerHolder parentSearchManager,
      DocumentReaderService extractor, ConfigurationManager cfm,
      final RepositoryIndexSearcherHolder indexSearcherHolder, RPCService rpcService, FileCleanerHolder cleanerHolder)
      throws RepositoryException, RepositoryConfigurationException
   {
      this.ctx = ctx;
      this.wsContainerId = ctx.getName();
      this.rpcService = rpcService;
      this.repositoryName = rEntry.getName();
      this.workspaceName = wEntry.getName();
      this.isSystem = rEntry.getSystemWorkspaceName().equals(workspaceName);
      this.rService = rService;
      this.wsId = wEntry.getUniqueName();
      this.extractor = extractor;
      this.cleanerHolder = cleanerHolder;
      indexSearcherHolder.addIndexSearcher(this);
      this.config = config;
      this.nodeTypeDataManager = ntReg;
      this.nsReg = nsReg;
      this.itemMgr = itemMgr;
      this.cfm = cfm;
      this.virtualTableResolver = new LuceneVirtualTableResolver(nodeTypeDataManager, nsReg);
      this.parentSearchManager = parentSearchManager != null ? parentSearchManager.get() : null;
      this.enableRemoteCalls = wEntry.getContainer()
                                     .getParameterBoolean(WorkspaceDataContainer.ENABLE_RPC_SYNC,
                                                          WorkspaceDataContainer.RPC_CALLS_ENABLED_DEFAULT);
      if (parentSearchManager != null)
      {
         ((WorkspacePersistentDataManager)this.itemMgr).addItemPersistenceListener(this);
      }

      if (rpcService == null) {
         this.indexRecovery = null;
      } else {
         initRemoteCommands();
         this.indexRecovery = new IndexRecoveryImpl(rpcService, this);
      }
   }

   public void createNewOrAdd(String key, ItemState state, Map<String, List<ItemState>> updatedNodes)
   {
      List<ItemState> list = updatedNodes.get(key);
      if (list == null)
      {
         list = new ArrayList<ItemState>();
         updatedNodes.put(key, list);
      }
      list.add(state);

   }

   /**
    * Creates a query object from a node that can be executed on the workspace.
    * 
    * @param session
    *            the session of the user executing the query.
    * @param sessionDataManager
    *            the item manager of the user executing the query. Needed to
    *            return <code>Node</code> instances in the result set.
    * @param node
    *            a node of type nt:query.
    * @return a <code>Query</code> instance to execute.
    * @throws InvalidQueryException
    *             if <code>absPath</code> is not a valid persisted query (that
    *             is, a node of type nt:query)
    * @throws RepositoryException
    *             if any other error occurs.
    */
   public Query createQuery(SessionImpl session, SessionDataManager sessionDataManager, Node node)
      throws InvalidQueryException, RepositoryException
   {
      AbstractQueryImpl query = createQueryInstance();
      query.init(session, sessionDataManager, handler, node);
      return query;
   }

   /**
    * Creates a query object that can be executed on the workspace.
    * 
    * @param session
    *            the session of the user executing the query.
    * @param sessionDataManager
    *            the item manager of the user executing the query. Needed to
    *            return <code>Node</code> instances in the result set.
    * @param statement
    *            the actual query statement.
    * @param language
    *            the syntax of the query statement.
    * @return a <code>Query</code> instance to execute.
    * @throws InvalidQueryException
    *             if the query is malformed or the <code>language</code> is
    *             unknown.
    * @throws RepositoryException
    *             if any other error occurs.
    */
   public Query createQuery(SessionImpl session, SessionDataManager sessionDataManager, String statement,
      String language) throws InvalidQueryException, RepositoryException
   {
      AbstractQueryImpl query = createQueryInstance();
      query.init(session, sessionDataManager, handler, statement, language);
      return query;
   }

   /**
    * Check index consistency. Iterator goes through index documents and check, does each document have
    * according jcr-node. If index is suspended then it will be temporary resumed, while check is running
    * and suspended afterwards.
    */
   public void checkIndex(final InspectionReport report, final boolean isSystem) throws RepositoryException,
      IOException
   {
      if (isSuspended.get())
      {
         try
         {
            if (isSystem && parentSearchManager != null && parentSearchManager.isSuspended.get())
            {
               parentSearchManager.resume();
            }
            resume();

            handler.checkIndex(itemMgr, isSystem, report);
         }
         catch (ResumeException e)
         {
            throw new RepositoryException("Can not resume SearchManager for inspection purposes.", e);
         }
         finally
         {
            // safely return the state of the workspace
            try
            {
               suspend();
               if (isSystem && parentSearchManager != null && !parentSearchManager.isSuspended.get())
               {
                  parentSearchManager.suspend();
               }
            }
            catch (SuspendException e)
            {
               LOG.error(e.getMessage(), e);
            }
         }

      }
      else
      {
         // simply run checkIndex, if not suspended
         handler.checkIndex(itemMgr, isSystem, report);
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<String> getFieldNames() throws IndexException
   {
      final Set<String> fildsSet = new HashSet<String>();
      if (handler instanceof SearchIndex)
      {
         IndexReader reader = null;
         try
         {
            reader = ((SearchIndex)handler).getIndexReader();
            FieldInfos infos = ReaderUtil.getMergedFieldInfos(reader);
            for (FieldInfo info : infos)
            {
               fildsSet.add(info.name);
            }
         }
         catch (IOException e)
         {
            throw new IndexException(e.getLocalizedMessage(), e);
         }
         finally
         {
            try
            {
               if (reader != null)
               {
                  Util.closeOrRelease(reader);
               }
            }
            catch (IOException e)
            {
               throw new IndexException(e.getLocalizedMessage(), e);
            }
         }

      }
      return fildsSet;
   }

   /**
    * just for test use only
    */
   public QueryHandler getHandler()
   {

      return handler;
   }

   public Set<String> getNodesByNodeType(final InternalQName nodeType) throws RepositoryException
   {

      return getNodes(virtualTableResolver.resolve(nodeType, true));
   }

   /**
    * Return set of uuid of nodes. Contains in names prefixes maped to the
    * given uri
    * 
    * @param uri
    * @return
    * @throws RepositoryException
    */
   public Set<String> getNodesByUri(final String uri) throws RepositoryException
   {
      Set<String> result;
      final int defaultClauseCount = BooleanQuery.getMaxClauseCount();
      try
      {

         // final LocationFactory locationFactory = new
         // LocationFactory(this);
         final ValueFactoryImpl valueFactory = new ValueFactoryImpl(new LocationFactory(nsReg), cleanerHolder);
         BooleanQuery.setMaxClauseCount(Integer.MAX_VALUE);
         BooleanQuery query = new BooleanQuery();

         final String prefix = nsReg.getNamespacePrefixByURI(uri);
         query.add(new WildcardQuery(new Term(FieldNames.LABEL, prefix + ":*")), Occur.SHOULD);
         // name of the property
         query.add(new WildcardQuery(new Term(FieldNames.PROPERTIES_SET, prefix + ":*")), Occur.SHOULD);

         result = getNodes(query);

         // value of the property

         try
         {
            final Set<String> props = getFieldNames();

            query = new BooleanQuery();
            for (final String fieldName : props)
            {
               if (!FieldNames.PROPERTIES_SET.equals(fieldName))
               {
                  query.add(new WildcardQuery(new Term(fieldName, "*" + prefix + ":*")), Occur.SHOULD);
               }
            }
         }
         catch (final IndexException e)
         {
            throw new RepositoryException(e.getLocalizedMessage(), e);
         }

         final Set<String> propSet = getNodes(query);
         // Manually check property values;
         for (final String uuid : propSet)
         {
            if (isPrefixMatch(valueFactory, uuid, prefix))
            {
               result.add(uuid);
            }
         }
      }
      finally
      {
         BooleanQuery.setMaxClauseCount(defaultClauseCount);
      }

      return result;
   }

   /**
    * {@inheritDoc}
    */
   public void onSaveItems(final ItemStateChangesLog itemStates)
   {
      //skip empty
      if (itemStates.getSize() > 0)
      {
         //Check if SearchManager started and filter configured
         if (changesFilter != null && parentSearchManager != null)
         {
            changesFilter.onSaveItems(itemStates);
         }
      }
   }

   public void start()
   {

      if (LOG.isDebugEnabled())
      {
         LOG.debug("start");
      }
      try
      {
         if (indexingTree == null)
         {
            NodeData indexingRootNodeData = (NodeData)itemMgr.getItemData(Constants.ROOT_UUID);
            indexingTree = new IndexingTree(indexingRootNodeData, isSystem ? Constants.JCR_SYSTEM_PATH : null);
         }
         initializeQueryHandler();
      }
      catch (RepositoryException e)
      {
         handler = null;
         throw new RuntimeException(e.getLocalizedMessage(), e);
      }
      catch (RepositoryConfigurationException e)
      {
         handler = null;
         throw new RuntimeException(e.getLocalizedMessage(), e);
      }
   }

   public void stop()
   {
      handler.close();

      // ChangesFiler instance is one for both SearchManagers and close() must be invoked only once,  
      if (parentSearchManager != null)
      {
         changesFilter.close();
      }

      if (indexRecovery != null)
      {
         indexRecovery.close();
      }
         
      unregisterRemoteCommands();

      if (LOG.isDebugEnabled())
      {
         LOG.debug("Search manager stopped");
      }
   }

   /**
    * {@inheritDoc}
    */
   public void updateIndex(final Set<String> removedNodes, final Set<String> addedNodes) throws RepositoryException,
      IOException
   {
      final ChangesHolder changes = getChanges(removedNodes, addedNodes);
      apply(changes);
   }

   public void apply(ChangesHolder changes) throws RepositoryException, IOException
   {
      if (handler != null && changes != null && (!changes.getAdd().isEmpty() || !changes.getRemove().isEmpty()))
      {
         handler.apply(changes);
      }
   }

   /**
    * Extracts all the changes and returns them as a {@link ChangesHolder} instance
    */
   public ChangesHolder getChanges(final Set<String> removedNodes, final Set<String> addedNodes)
   {
      if (handler != null)
      {
         Iterator<NodeData> addedStates = new Iterator<NodeData>()
         {
            private final Iterator<String> iter = addedNodes.iterator();

            public boolean hasNext()
            {
               return iter.hasNext();
            }

            public NodeData next()
            {

               // cycle till find a next or meet the end of set
               do
               {
                  String id = iter.next();
                  try
                  {
                     ItemData item = itemMgr.getItemData(id);
                     if (item != null)
                     {
                        if (item.isNode())
                        {
                           if (!indexingTree.isExcluded(item))
                           {
                              return (NodeData)item;
                           }
                        }
                        else
                        {
                           LOG.warn("Node not found, but property " + id + ", " + item.getQPath().getAsString()
                              + " found. ");
                        }
                     }
                     else
                     {
                        LOG.warn("Unable to index node with id " + id + ", node does not exist.");
                     }

                  }
                  catch (RepositoryException e)
                  {
                     LOG.error("Can't read next node data " + id, e);
                  }
               }
               while (iter.hasNext()); // get next if error or node not found

               return null; // we met the end of iterator set
            }

            public void remove()
            {
               throw new UnsupportedOperationException();
            }
         };

         Iterator<String> removedIds = new Iterator<String>()
         {
            private final Iterator<String> iter = removedNodes.iterator();

            public boolean hasNext()
            {
               return iter.hasNext();
            }

            public String next()
            {
               return nextNodeId();
            }

            public String nextNodeId() throws NoSuchElementException
            {
               return iter.next();
            }

            public void remove()
            {
               throw new UnsupportedOperationException();

            }
         };

         if (removedNodes.size() > 0 || addedNodes.size() > 0)
         {
            return handler.getChanges(removedIds, addedStates);
         }
      }
      return null;
   }

   protected QueryHandlerContext createQueryHandlerContext(QueryHandler parentHandler)
      throws RepositoryConfigurationException
   {
      WorkspaceContainerFacade container;
      try
      {
         container = rService.getRepository(repositoryName).getWorkspaceContainer(workspaceName);
      }
      catch (RepositoryException e)
      {
         throw new RepositoryConfigurationException(e);
      }

      // Recovery Filters are 
      String changesFilterClassName = config.getParameterValue(QueryHandlerParams.PARAM_CHANGES_FILTER_CLASS, null);
      boolean recoveryFilterUsed = false;

      if (changesFilterClassName != null)
      {
         try
         {
            Class<?> changesFilterClass = ClassLoading.forName(changesFilterClassName, this);
            // Set recoveryFilterUsed, if changes filter implements LocalIndex strategy 
            if (changesFilterClass != null)
            {
               recoveryFilterUsed = LocalIndexMarker.class.isAssignableFrom(changesFilterClass);
            }
         }
         catch (ClassNotFoundException e)
         {
            throw new RepositoryConfigurationException(e.getMessage(), e);
         }
      }

      return new QueryHandlerContext(container, itemMgr, indexingTree, nodeTypeDataManager, nsReg, parentHandler,
             getIndexDirectory().getAbsolutePath(), extractor, true, recoveryFilterUsed, enableRemoteCalls,
            virtualTableResolver, indexRecovery, rpcService, repositoryName, wsId, cleanerHolder);
   }

   /**
    * Creates a new instance of an {@link AbstractQueryImpl} which is not
    * initialized.
    * 
    * @return an new query instance.
    * @throws RepositoryException
    *             if an error occurs while creating a new query instance.
    */
   protected AbstractQueryImpl createQueryInstance() throws RepositoryException
   {
      try
      {
         String queryImplClassName = handler.getQueryClass();
         Object obj = ClassLoading.forName(queryImplClassName, this).newInstance();
         if (obj instanceof AbstractQueryImpl)
         {
            return (AbstractQueryImpl)obj;
         }
         else
         {
            throw new IllegalArgumentException(queryImplClassName + " is not of type "
               + AbstractQueryImpl.class.getName());
         }
      }
      catch (Throwable t) //NOSONAR
      {
         throw new RepositoryException("Unable to create query: " + t.toString(), t);
      }
   }

   /**^
    * Returns "index-dir" parameter from configuration. 
    */
   protected String getIndexDirParam() throws RepositoryConfigurationException
   {
      String dir = config.getParameterValue(QueryHandlerParams.PARAM_INDEX_DIR, null);
      if (dir == null)
      {
         LOG.warn(QueryHandlerParams.PARAM_INDEX_DIR + " parameter not found. Using outdated parameter name "
            + QueryHandlerParams.OLD_PARAM_INDEX_DIR);
         dir = config.getParameterValue(QueryHandlerParams.OLD_PARAM_INDEX_DIR);
      }
      return dir;
   }

   /**
    * @return the indexingTree
    */
   protected IndexingTree getIndexingTree()
   {
      return indexingTree;
   }

   /**
    * @return the ctx
    */
   public ExoContainerContext getExoContainerContext()
   {
      return ctx;
   }

   /**
    * Initialize changes filter.
    * @throws RepositoryException
    * @throws RepositoryConfigurationException
    * @throws SecurityException
    */
   @SuppressWarnings("unchecked")
   protected IndexerChangesFilter initializeChangesFilter() throws RepositoryException,
      RepositoryConfigurationException

   {
      IndexerChangesFilter newChangesFilter = null;
      Class<? extends IndexerChangesFilter> changesFilterClass = DefaultChangesFilter.class;
      String changesFilterClassName = config.getParameterValue(QueryHandlerParams.PARAM_CHANGES_FILTER_CLASS, null);
      try
      {
         if (changesFilterClassName != null)
         {
            changesFilterClass =
               (Class<? extends IndexerChangesFilter>)ClassLoading.forName(changesFilterClassName, this);
         }
         Constructor<? extends IndexerChangesFilter> constuctor =
            changesFilterClass.getConstructor(SearchManager.class, SearchManager.class, QueryHandlerEntry.class,
               IndexingTree.class, IndexingTree.class, QueryHandler.class, QueryHandler.class,
               ConfigurationManager.class);
         if (parentSearchManager != null)
         {
            newChangesFilter =
               constuctor.newInstance(this, parentSearchManager, config, indexingTree,
                  parentSearchManager.getIndexingTree(), handler, parentSearchManager.getHandler(), cfm);
         }
      }
      catch (SecurityException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (IllegalArgumentException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (ClassNotFoundException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (NoSuchMethodException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (InstantiationException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (IllegalAccessException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (InvocationTargetException e)
      {
         throw new RepositoryException("Unable to initialize change filter " + changesFilterClassName, e);
      }
      return newChangesFilter;
   }

   /**
    * Initializes the query handler.
    * 
    * @throws RepositoryException
    *             if the query handler cannot be initialized.
    * @throws RepositoryConfigurationException
    */
   protected void initializeQueryHandler() throws RepositoryException, RepositoryConfigurationException
   {
      // initialize query handler
      String className = config.getType();
      if (className == null)
      {
         throw new RepositoryConfigurationException("Content hanler       configuration fail");
      }

      try
      {
         Class<?> qHandlerClass = ClassLoading.forName(className, this);
         try
         {
            // We first try a constructor with the workspace id
            Constructor<?> constuctor =
               qHandlerClass.getConstructor(String.class, QueryHandlerEntry.class, ConfigurationManager.class);
            handler = (QueryHandler)constuctor.newInstance(wsContainerId, config, cfm);
         }
         catch (NoSuchMethodException e)
         {
            // No constructor with the workspace id can be found so we use the default constructor
            Constructor<?> constuctor =
               qHandlerClass.getConstructor(QueryHandlerEntry.class, ConfigurationManager.class);
            handler = (QueryHandler)constuctor.newInstance(config, cfm);
         }
         QueryHandler parentHandler = (this.parentSearchManager != null) ? parentSearchManager.getHandler() : null;
         QueryHandlerContext context = createQueryHandlerContext(parentHandler);
         handler.setContext(context);

         if (parentSearchManager != null)
         {
            changesFilter = initializeChangesFilter();
            parentSearchManager.setChangesFilter(changesFilter);
         }
      }
      catch (SecurityException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (IllegalArgumentException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (ClassNotFoundException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (NoSuchMethodException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (InstantiationException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (IllegalAccessException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
      catch (InvocationTargetException e)
      {
         throw new RepositoryException(e.getMessage(), e);
      }
   }

   /**
    * Inserts the instance of {@link IndexerChangesFilter} into the {@link SearchManager}. 
    * Used to set instance for {@link SystemSearchManager}.
    * 
    * @param changesFilter
    */
   protected void setChangesFilter(IndexerChangesFilter changesFilter)
   {
      if (this.changesFilter == null)
      {
         this.changesFilter = changesFilter;
      }
   }

   /**
    * @param query
    * @return
    * @throws RepositoryException
    */
   private Set<String> getNodes(final org.apache.lucene.search.Query query) throws RepositoryException
   {
      Set<String> result = new HashSet<String>();
      QueryHits hits = null;
      try
      {
         hits = handler.executeQuery(query);

         ScoreNode sn;
         while ((sn = hits.nextScoreNode()) != null)
         {
            result.add(sn.getNodeId());
         }
         return result;
      }
      catch (IndexOfflineIOException e)
      {
         throw new IndexOfflineRepositoryException(e.getMessage(), e);
      }
      catch (IOException e)
      {
         throw new RepositoryException(e.getLocalizedMessage(), e);
      }
      finally
      {
         if (hits != null)
         {
            try
            {
               hits.close();
            }
            catch (IOException e)
            {
               LOG.error("Can not close QueryHits.", e);
            }
         }
      }
   }

   private boolean isPrefixMatch(final InternalQName value, final String prefix) throws RepositoryException
   {
      return value.getNamespace().equals(nsReg.getNamespaceURIByPrefix(prefix));
   }

   private boolean isPrefixMatch(final QPath value, final String prefix) throws RepositoryException
   {
      for (int i = 0; i < value.getEntries().length; i++)
      {
         if (isPrefixMatch(value.getEntries()[i], prefix))
         {
            return true;
         }
      }
      return false;
   }

   /**
    * @param valueFactory
    * @param uuid
    * @param prefix
    * @throws RepositoryException
    */
   private boolean isPrefixMatch(final ValueFactoryImpl valueFactory, final String uuid, final String prefix)
      throws RepositoryException
   {

      final ItemData node = itemMgr.getItemData(uuid);
      if (node != null && node.isNode())
      {
         final List<PropertyData> props = itemMgr.getChildPropertiesData((NodeData)node);
         for (final PropertyData propertyData : props)
         {
            if (propertyData.getType() == PropertyType.PATH || propertyData.getType() == PropertyType.NAME)
            {
               for (final ValueData vdata : propertyData.getValues())
               {
                  final Value val = valueFactory.loadValue(vdata, propertyData.getType());
                  if (propertyData.getType() == PropertyType.PATH)
                  {
                     if (isPrefixMatch(((PathValue)val).getQPath(), prefix))
                     {
                        return true;
                     }
                  }
                  else if (propertyData.getType() == PropertyType.NAME)
                  {
                     if (isPrefixMatch(((NameValue)val).getQName(), prefix))
                     {
                        return true;
                     }
                  }
               }
            }
         }
      }
      return false;
   }

   /**
    * {@inheritDoc}
    */
   public boolean isTXAware()
   {
      return false;
   }

   public String getWsId()
   {
      return wsId;
   }

   /**
    * {@inheritDoc}
    */
   public void suspend() throws SuspendException
   {
      if (suspend != null)
      {
         isResponsibleForResuming.set(true);

         try
         {
            rpcService.executeCommandOnAllNodes(suspend, true);
         }
         catch (SecurityException e)
         {
            throw new SuspendException(e);
         }
         catch (RPCException e)
         {
            throw new SuspendException(e);
         }
      }
      else
      {
         suspendLocally();
      }
   }

   /**
    * {@inheritDoc}
    */
   public boolean isSuspended()
   {
      return isSuspended.get();
   }

   /**
    * Switches index into corresponding ONLINE or OFFLINE mode. Offline mode means that new indexing data is
    * collected but index is guaranteed to be unmodified during offline state. Passing the allowQuery flag, can
    * allow or deny performing queries on index during offline mode. AllowQuery is not used when setting index
    * back online. When dropStaleIndexes is set, indexes present on the moment of switching index offline will be
    * marked as stale and removed on switching it back online.
    * 
    * @param isOnline
    * @param allowQuery
    *          doesn't matter, when switching index to online
    * @param dropStaleIndexes
    *          doesn't matter, when switching index to online
    * @throws IOException
    */
   public void setOnline(boolean isOnline, boolean allowQuery, boolean dropStaleIndexes) throws IOException
   {
      handler.setOnline(isOnline, allowQuery, dropStaleIndexes);
   }

   public boolean isOnline()
   {
      return handler.isOnline();
   }

   /**
    * {@inheritDoc}
    */
   public void resume() throws ResumeException
   {
      if (resume != null)
      {
         try
         {
            rpcService.executeCommandOnAllNodes(resume, true);
         }
         catch (SecurityException e)
         {
            throw new ResumeException(e);
         }
         catch (RPCException e)
         {
            throw new ResumeException(e);
         }

         isResponsibleForResuming.set(false);
      }
      else
      {
         resumeLocally();
      }
   }

   /**
    * Public method, designed to be called via JMX. Optimizes lucene index.
    * See {@link IndexWriter#forceMerge(int)} for details.
    */
   @Managed
   @ManagedDescription("Index optimization ")
   public void optimize()
   {
      if (handler instanceof SearchIndex)
      {
         try
         {
            if (isSuspended.get())
            {
               resume();

               try
               {
                  ((SearchIndex)handler).getIndex().optimize();
               }
               finally
               {
                  suspend();
               }
            }
            else
            {
               ((SearchIndex)handler).getIndex().optimize();
            }
         }
         catch (IOException e)
         {
            LOG.error(e.getMessage(), e);
         }
         catch (ResumeException e)
         {
            LOG.error(e.getMessage(), e);
         }
         catch (SuspendException e)
         {
            LOG.error(e.getMessage(), e);
         }
      }
      else
      {
         LOG.error("This kind of QuerHandler class doesn't support index optimization.");
      }
   }

   /**
    * Public method, designed to be called via JMX. Checks if index has deletions. 
    * If it is true, it is recommended to call optimize to complete them.
    */
   @Managed
   @ManagedDescription("Checks if index has deletions.")
   public boolean hasDeletions()
   {
      if (handler instanceof SearchIndex)
      {
         try
         {
            if (isSuspended.get())
            {
               resume();

               try
               {
                  return ((SearchIndex)handler).getIndex().hasDeletions();
               }
               finally
               {
                  suspend();
               }
            }
            else
            {
               return ((SearchIndex)handler).getIndex().hasDeletions();
            }
         }
         catch (IOException e)
         {
            LOG.error(e.getMessage(), e);
         }
         catch (ResumeException e)
         {
            LOG.error(e.getMessage(), e);
         }
         catch (SuspendException e)
         {
            LOG.error(e.getMessage(), e);
         }
      }
      else
      {
         LOG.error("This kind of QuerHandler class doesn't support 'hasDeletions' checking.");
      }

      return false;
   }

   /**
    * Public method, designed to be called via JMX, to perform "HOT" reindexing of the workspace
    * @param dropExisting use the same index directory  (if "true") and all queries will throw
    *                     an exception while task is running. Otherwise (if "false") Server can continue
    *                     working as expected while index is recreated (the new index is created under
    *                     a new index folder).
    * @param nThreads indexing Thread Pool Size, default value is 1.
    *
    * @throws IllegalStateException
    */
   @Managed
   @ManagedDescription("Starts hot async reindexing")
   public void reindex(@ManagedName("dropExisting") final boolean dropExisting, @ManagedName("nThreads") final int nThreads)
           throws IllegalStateException, ExecutionException, InterruptedException {
     reindexWorkspace(dropExisting, nThreads);
   }

  /**
   * Perform hot reindexing of the workspace
   * @param dropExisting use the same index directory  (if "true") and all queries will throw
   *                     an exception while task is running. Otherwise (if "false") Server can continue
   *                     working as expected while index is recreated (the new index is created under
   *                     a new index folder).
   * @param nThreads indexing Thread Pool Size, default value is 1.
   * @return a Future which can be used to get the result of the reindexing (true means successful)
   * @throws IllegalStateException
   */
   public CompletableFuture<Boolean> reindexWorkspace(final boolean dropExisting, int nThreads) throws IllegalStateException
   {
      // checks
      if (handler == null || handler.getIndexerIoModeHandler() == null || changesFilter == null)
      {
         throw new IllegalStateException("Index might have not been initialized yet.");
      }
      if (handler.getIndexerIoModeHandler().getMode() != IndexerIoMode.READ_WRITE)
      {
         throw new IllegalStateException(
            "Index is not in READ_WRITE mode and reindexing can't be launched. Please start reindexing on coordinator node.");
      }
      if (isSuspended.get() || !handler.isOnline())
      {
         throw new IllegalStateException("Can't start reindexing while index is "
            + ((isSuspended.get()) ? "SUSPENDED." : "already OFFLINE (it means that reindexing is in progress).") + ".");
      }

      LOG.info("Starting hot reindexing on the " + handler.getContext().getRepositoryName() + "/"
         + handler.getContext().getContainer().getWorkspaceName() + ", with" + (dropExisting ? "" : "out")
         + " dropping the existing indexes.");

      // starting new thread, releasing JMX call
      ExecutorService executorService = Executors.newSingleThreadExecutor(runnable -> new Thread(runnable,
              "HotReindexing-" + handler.getContext().getRepositoryName() + "-"
                      + handler.getContext().getContainer().getWorkspaceName()));
      CompletableFuture<Boolean> reindexFuture = CompletableFuture.supplyAsync(() -> doReindexing(dropExisting, nThreads), executorService);
      reindexFuture.thenRun(() -> executorService.shutdown());
      return reindexFuture;
   }

  /**
   * Perform workspace reindexing
   * @param dropExisting
   * @param nThreads
   */
  protected boolean doReindexing(boolean dropExisting, int nThreads) {
    boolean successful = false;
    Integer oldPoolSize = ((SearchIndex) handler).getIndexingThreadPoolSize();
    String newIndexPath = null;

    hotReindexingState = "Running. Started at " + LocalDateTime.now().format(sdf);
    try {

      isResponsibleForResuming.set(true);

      //hot async reindexing with drop exiting index data
      // set offline index state
      if (dropExisting) {
        // set offline cluster wide (will make merger disposed and volatile flushed)
        if (changeIndexState != null && changesFilter.isShared()) {
          rpcService.executeCommandOnAllNodes(changeIndexState, true, false, false);
        } else {
          handler.setOnline(false, false, true);
        }
      }
      // launch reindexing thread safely, resume nodes if any exception occurs
      if (handler instanceof SearchIndex) {
        if (dropExisting) {
          ((SearchIndex) handler).getIndex().reindex(itemMgr);
        } else {
          String oldIndexPath = ((SearchIndex) handler).getPath();
          newIndexPath = oldIndexPath + SUFFIX_ASYNC_REINDEXING;

          //try remove new index directory if already exist
          cleanIndexDirectory(oldIndexPath + SUFFIX_ASYNC_REINDEXING);

          //hot async reindexing without remove old index
          //create new index and register it
          MultiIndex newIndex = ((SearchIndex) handler).createNewIndex(SUFFIX_ASYNC_REINDEXING);
          ErrorLog errorLog = ((SearchIndex) handler).doInitErrorLog(newIndexPath);
          ((SearchIndex) handler).getIndexRegister().register(newIndex, errorLog);
          ((SearchIndex) handler).setIndexingThreadPoolSize(nThreads > 1 ? nThreads : 1);

          //set offline the new index
          newIndex.setOnline(false, true, false);

          //start create new index
          newIndex.reindex(itemMgr);

          //set  online to flush offline index
          newIndex.setOnline(true, true, false);

          //Suspend component, on cluster mode stop all index nodes
          //All the working threads will be suspended until the resume operation is performed.
          suspend();
          ((SearchIndex) handler).getIndexRegister().unregister(newIndex);

          final File indexDir = new File(oldIndexPath);
          final File newIndexDir = new File(oldIndexPath + SUFFIX_ASYNC_REINDEXING);

          //try remove old index directory
          LOG.info("Try to clean old lucene indexes of the workspace '" + workspaceName + "'");
          cleanIndexDirectory(oldIndexPath);

          //try rename new index folder .new
          LOG.info("try to rename new lucene indexes of the workspace '" + workspaceName + "'");
          DirectoryHelper.renameFile(newIndexDir, indexDir);


           // Resume all the working threads that have been previously suspended.
          resume();
        }

        successful = true;
      } else {
        LOG.error("This kind of QuerHandler class doesn't support hot reindxing.");
      }
    } catch (RepositoryException e) {
      LOG.error("Error while reindexing the workspace", e);
    } catch (SecurityException e) {
      LOG.error("Can't change state to offline.", e);
    } catch (RPCException e) {
      LOG.error("Can't change state to offline.", e);
    } catch (IOException e) {
      LOG.error("Error while reindexing the workspace", e);
    }
    // safely change state back
    catch (SuspendException e) {
      LOG.error("Can't suspend SearchIndex component, " +
              "impossible to switch to newly created index, the old index still used", e);

    } catch (ResumeException e) {
      LOG.error("Can't resume SearchIndex component, impossible to set the index online", e);
    } finally {
      if (dropExisting) {
        // finish, setting indexes back online
        if (changeIndexState != null &&  changesFilter.isShared()) {
          try {
            // if dropExisting, then queries are no allowed
            rpcService.executeCommandOnAllNodes(changeIndexState, true, true, true);
          } catch (SecurityException e) {
            LOG.error("Error setting index back online in a cluster", e);
          } catch (RPCException e) {
            LOG.error("Error setting index back online in a cluster", e);
          }
        } else {
          try {
            handler.setOnline(true, true, true);
          } catch (IOException e) {
            LOG.error("Error setting index back online locally");
          }
        }
      } else {
        ((SearchIndex) handler).setIndexingThreadPoolSize(oldPoolSize);
      }
      if (successful) {
        hotReindexingState = "Finished at " + LocalDateTime.now().format(sdf);
        LOG.info("Reindexing finished successfully.");
      } else {
        hotReindexingState = "Stopped with errors at " + LocalDateTime.now().format(sdf);
        LOG.info("Reindexing halted with errors.");
        if (!dropExisting) {
          try {
            if (newIndexPath != null)
              cleanIndexDirectory(newIndexPath);
          } catch (IOException e) {
            LOG.error("Error while removing the folder of the index used for the hot reindexing", e);

          }
        }
      }
      isResponsibleForResuming.set(false);

      return successful;
    }
  }

   @Managed
   @ManagedDescription("Hot async reindexing state")
   public String getHotReindexingState()
   {
      return hotReindexingState;
   }

   @Managed
   @ManagedDescription("Index IO mode (READ_ONLY/READ_WRITE)")
   public String getIOMode()
   {
      if (handler == null || handler.getIndexerIoModeHandler() == null)
      {
         return "not initialized";
      }
      return (handler.getIndexerIoModeHandler().getMode() == IndexerIoMode.READ_WRITE) ? "READ_WRITE" : "READ_ONLY";
   }

   @Managed
   @ManagedDescription("Index state (Online/Offline(indexing))")
   public String getState()
   {
      if (handler == null)
      {
         return "not initialized";
      }
      return handler.isOnline() ? "Online" : "Offline (indexing)";
   }

   @Managed
   @ManagedDescription("QueryHandler class")
   public String getQuerHandlerClass()
   {
      if (handler != null)
      {
         return handler.getClass().getCanonicalName();
      }
      else
      {
         return "not initialized";
      }
   }

   @Managed
   @ManagedDescription("ChangesFilter class")
   public String getChangesFilterClass()
   {
      if (changesFilter != null)
      {
         return changesFilter.getClass().getCanonicalName();
      }
      else
      {
         return "not initialized";
      }
   }

   /**
    * Register remote commands.
    */
   private void initRemoteCommands()
   {
      if (!this.enableRemoteCalls) {
        return;
      }
      // register commands
      suspend = rpcService.registerCommand(new RemoteCommand()
      {

         public String getId()
         {
            return "org.exoplatform.services.jcr.impl.core.query.SearchManager-suspend-" + wsId + "-"
               + (parentSearchManager == null);
         }

         public Serializable execute(Serializable[] args) throws Throwable
         {
            suspendLocally();
            return null;
         }
      });

      resume = rpcService.registerCommand(new RemoteCommand()
      {

         public String getId()
         {
            return "org.exoplatform.services.jcr.impl.core.query.SearchManager-resume-" + wsId + "-"
               + (parentSearchManager == null);
         }

         public Serializable execute(Serializable[] args) throws Throwable
         {
            resumeLocally();
            return null;
         }
      });

      requestForResponsibleForResuming = rpcService.registerCommand(new RemoteCommand()
      {

         public String getId()
         {
            return "org.exoplatform.services.jcr.impl.core.query.SearchManager-requestForResponsibilityForResuming-"
               + wsId + "-" + (parentSearchManager == null);
         }

         public Serializable execute(Serializable[] args) throws Throwable
         {
            return isResponsibleForResuming.get();
         }
      });

      changeIndexState = rpcService.registerCommand(new RemoteCommand()
      {
         public String getId()
         {
            return "org.exoplatform.services.jcr.impl.core.query.SearchManager-changeIndexerState-" + wsId + "-"
               + (parentSearchManager == null);
         }

         public Serializable execute(Serializable[] args) throws Throwable
         {
            boolean isOnline = (Boolean)args[0];
            boolean allowQuery = (args.length == 2) ? (Boolean)args[1] : false;
            SearchManager.this.setOnline(isOnline, allowQuery, true);
            return null;
         }
      });

   }
   
   /**
    * Unregister remote commands.
    */
   private void unregisterRemoteCommands()
   {
      if (suspend != null) {
        rpcService.unregisterCommand(suspend);
      }
      if (resume != null) {
        rpcService.unregisterCommand(resume);
      }
      if (requestForResponsibleForResuming != null) {
        rpcService.unregisterCommand(requestForResponsibleForResuming);
      }
      if (changeIndexState != null) {
        rpcService.unregisterCommand(changeIndexState);
      }
   }

   protected void suspendLocally() throws SuspendException
   {
      if (handler != null && !handler.isOnline())
      {
         throw new SuspendException("Can't suspend index, while reindexing in progeress.");
      }

      if (!isSuspended.get())
      {
         if (handler instanceof Suspendable)
         {
            ((Suspendable)handler).suspend();
         }

         isSuspended.set(true);
      }
   }

   protected void resumeLocally() throws ResumeException
   {
      if (isSuspended.get())
      {
         if (handler instanceof Suspendable)
         {
            ((Suspendable)handler).resume();
         }

         isSuspended.set(false);
      }
   }

   /**
    * {@inheritDoc}}
    */
   public void clean() throws BackupException
   {
      LOG.info("Start to clean lucene indexes of the workspace '"+workspaceName+"'");
      try
      {
         final File indexDir = getIndexDirectory();

         DirectoryHelper.removeDirectory(indexDir);

      }
      catch (IOException e)
      {
         throw new BackupException(e);
      }
      catch (RepositoryConfigurationException e)
      {
         throw new BackupException(e);
      }
   }

   /**
    * {@inheritDoc}}
    */
   public void backup(final File storageDir) throws BackupException
   {
      LOG.info("Start to backup lucene indexes of the workspace '"+workspaceName+"'");
      try
      {
         final File indexDir = getIndexDirectory();
         if (!indexDir.exists())
         {
            throw new IOException("Can't backup index. Directory " + indexDir.getCanonicalPath()
                                      + " doesn't exists");
         }
         else
         {
            File destZip = new File(storageDir, getStorageName() + ".zip");
            DirectoryHelper.compressDirectory(indexDir, destZip);
         }
      }
      catch (RepositoryConfigurationException e)
      {
         throw new BackupException(e);
      }
      catch (IOException e)
      {
         throw new BackupException(e);
      }
   }

   /**
    * Returns the index directory.
    * 
    * @return File
    * @throws RepositoryConfigurationException
    */
   protected File getIndexDirectory() throws RepositoryConfigurationException
   {
      return new File(getIndexDirParam());
   }

   /**
    * Returns storage name of index.
    * 
    * @return String
    */
   protected String getStorageName()
   {
      return "index";
   }

   public boolean isEnableRemoteCalls() {
      return enableRemoteCalls;
   }

   /**
    * {@inheritDoc}}
    */
   public DataRestore getDataRestorer(DataRestoreContext context) throws BackupException
   {
      try
      {
         File zipFile = new File((File)context.getObject(DataRestoreContext.STORAGE_DIR), getStorageName() + ".zip");

         if (zipFile.exists())
         {
            return new DirectoryRestore(getIndexDirectory(), zipFile);
         }
         else
         {
            // try to check if we have deal with old backup format
            zipFile = new File((File)context.getObject(DataRestoreContext.STORAGE_DIR), getStorageName());
            if (zipFile.exists())
            {
               return new DirectoryRestore(getIndexDirectory(), zipFile);
            }
            else
            {
               throw new BackupException("There is no backup data for index");
            }
         }
      }
      catch (RepositoryConfigurationException e)
      {
         throw new BackupException(e);
      }
   }

   /**
    * {@inheritDoc}
    */
   public int getPriority()
   {
      return PRIORITY_NORMAL;
   }

   /**
    * remove index directory if exist
    * @param path index directory path
    * @throws IOException
    */
   private void cleanIndexDirectory(String path) throws IOException
   {
      File newIndexFolder = new File(path);
      if(newIndexFolder.exists())
      {
         DirectoryHelper.removeDirectory(newIndexFolder);
      }
   }
}
