/***************************************************************************
 * Copyright 2001-2003 The eXo Platform SARL         All rights reserved.  *
 * Please look at license.txt in info directory for more license detail.   *
 **************************************************************************/
package org.exoplatform.maven.plugin.exobuild;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.installer.ArtifactInstaller;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.resolver.filter.AndArtifactFilter;
import org.apache.maven.artifact.resolver.filter.IncludesArtifactFilter;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.artifact.filter.collection.ArtifactsFilter;
import org.apache.maven.toolchain.Toolchain;
import org.apache.maven.toolchain.ToolchainManager;
import org.codehaus.plexus.component.repository.exception.ComponentLookupException;
import org.codehaus.plexus.util.cli.CommandLineException;
import org.codehaus.plexus.util.cli.CommandLineUtils;
import org.codehaus.plexus.util.cli.Commandline;
import org.codehaus.plexus.util.cli.StreamConsumer;
import org.exoplatform.maven.plugin.exec.Classpath;
import org.exoplatform.maven.plugin.exec.StreamLog;
import org.exoplatform.maven.plugin.exec.UrlUtils;
import org.exoplatform.maven.plugin.exobuild.utils.DependencyStatusSets;
import org.exoplatform.maven.plugin.exobuild.utils.DependencyUtil;
import org.exoplatform.maven.plugin.exobuild.utils.filters.DestFileFilter;
import org.exoplatform.maven.plugin.exobuild.utils.markers.DefaultFileMarkerHandler;

/**
 * @goal exobuild
 * @phase install
 * @requiresDependencyResolution runtime
 * @description Packager : mvn exobuild:exobuild -Dproduct=portal -Ddeploy=tomcat
 */
public class ExobuildMojo extends AbstractFromDependenciesMojo {

  /**
   * @parameter expression="${exo.working.dir}"
   *            default-value="${exo.projects.directory.working}"
   * @required
   * @since 1.0
   */
  protected File                      exoWorkingDir;

  /**
   * @parameter expression="${exo.dep.dir}"
   *            default-value="${exo.projects.directory.dependencies}"
   * @required
   * @since 1.0
   */
  protected File                      exoDependenciesDir;

  /**
   * @parameter expression="${exo.projects.app.tomcat.version}"
   * @optional
   * @since 1.0
   */
  protected String                      exobuildTomcatVersion;

  /**
   * @parameter expression="${exo.projects.app.jboss.version}"
   * @optional
   * @since 1.0
   */
  protected String                      exobuildJBossVersion;

  /**
   * @parameter expression="${exo.base.dir}"
   *            default-value="${exo.projects.directory.base}"
   * @optional
   * @since 1.0
   */
  protected String                      exoBaseDir;

  // //////////////////////////

  /**
   * @parameter expression="${product}"
   * @required
   */
  private String                      product;

  /**
   * @parameter expression="${deploy}"
   * @required
   */
  private String                      deploy;

  /**
   * @parameter expression="${release}"
   */
  private String                      release;

  /**
   * @parameter expression="${database}"
   */
  private String                      database;

  /**
   * @parameter expression="${dbsetup}"
   */
  private String                      dbsetup;

  /**
   * @parameter expression="${enable-workflow}"
   */
  private String                      enable_workflow;

  /**
   * @parameter expression="${maven.home}"
   */
  private String                      mavenHome;

  // //////////////////////////

  /**
   * Default location used for mojo unless overridden in ArtifactItem
   * 
   * @parameter expression="${outputDirectory}"
   *            default-value="${project.build.directory}/packager"
   * @optional
   * @since 1.0
   */
  protected File                      outputDirectoryPackager;

  /**
   * Default location used for mojo unless overridden in ArtifactItem
   * 
   * @parameter expression="${outputDirectory}"
   *            default-value="${project.build.directory}/packager-conf"
   * @optional
   * @since 1.0
   */
  protected File                      outputDirectoryPackagerConf;

  /**
   * @component
   */
  protected ArtifactInstaller         installer;

  /**
   * @component
   */
  protected ArtifactRepositoryFactory repositoryFactory;

  // //////////////////////////////
  // Exec
  // //////////////////////////////
  /**
   * The executable. Can be a full path or a the name executable. In the latter
   * case, the executable must be in the PATH for the execution to work.
   * 
   * @parameter expression="${exec.executable}" default-value="java"
   * @optional
   */
  private String                      executable;

  /**
   * Program standard and error output will be redirected to the file specified
   * by this optional field. If not specified the standard maven logging is
   * used.
   * 
   * @parameter expression="${exec.outputFile}"
   */
  private File                        outputFile;

  /**
   * Can be of type <code>&lt;argument&gt;</code> or
   * <code>&lt;classpath&gt;</code> Can be overriden using "exec.args" env.
   * variable
   * 
   * @parameter
   */
  private List                        arguments;

  /**
   * @parameter expression="${basedir}"
   * @required
   * @readonly
   * @since 1.0
   */
  private File                        basedir;

  /**
   * Environment variables to pass to the executed program.
   * 
   * @parameter
   * @since 1.1-beta-2
   */
  private Map                         environmentVariables = new HashMap();

  /**
   * The current build session instance. This is used for toolchain manager API
   * calls.
   * 
   * @parameter expression="${session}"
   * @required
   * @readonly
   */
  private MavenSession                session;

  /**
   * Exit codes to be resolved as successful execution for non-compliant
   * applications (applications not returning 0 for success).
   * 
   * @parameter
   * @since 1.1.1
   */
  private List                        successCodes;

  private boolean                     longClasspath;

  public static final String          CLASSPATH_TOKEN      = "%classpath";

  // AbstractExec

  /**
   * This folder is added to the list of those folders containing source to be
   * compiled. Use this if your plugin generates source code.
   * 
   * @parameter expression="${sourceRoot}"
   */
  private File                        sourceRoot;

  /**
   * This folder is added to the list of those folders containing source to be
   * compiled for testing. Use this if your plugin generates test source code.
   * 
   * @parameter expression="${testSourceRoot}"
   */
  private File                        testSourceRoot;

  /**
   * Arguments for the executed program
   * 
   * @parameter expression="${exec.args}"
   */
  private String                      commandlineArgs;

  /**
   * Defines the scope of the classpath passed to the plugin. Set to
   * compile,test,runtime or system depending on your needs. Since 1.1.2, the
   * default value is 'runtime' instead of 'compile'.
   * 
   * @parameter expression="${exec.classpathScope}" default-value="runtime"
   */
  protected String                    classpathScope;

  /** @see #parseCommandlineArgs */
  private static final char           PARAMETER_DELIMITER  = ' ';

  /** @see #parseCommandlineArgs */
  private static final char           STRING_WRAPPER       = '"';

  /** @see #parseCommandlineArgs */
  private static final char           ESCAPE_CHAR          = '\\';

  /**
   * Main entry into mojo. Gets the list of dependencies and iterates through
   * calling copyArtifact.
   * 
   * @throws MojoExecutionException with a message if an error occurs.
   * @see #getDependencies
   * @see #copyArtifact(Artifact, boolean)
   */
  public void execute() throws MojoExecutionException {
    System.out.println("Welcome to packager !!");
    System.out.println("* Product=" + product);
    System.out.println("* Deploy=" + deploy);

    DependencyStatusSets dss = getDependencySets(this.failOnMissingClassifierArtifact);
    Set artifacts = dss.getResolvedDependencies();

    for (Iterator i = artifacts.iterator(); i.hasNext();) {
      Artifact artifact = (Artifact) i.next();
      if (artifact.getType().equals("js")) {
        String destFileName = DependencyUtil.getFormattedFileName(artifact, true);
        File destDir = DependencyUtil.getFormattedOutputDirectory(useSubDirectoryPerType,
                                                                  useSubDirectoryPerArtifact,
                                                                  useRepositoryLayout,
                                                                  stripVersion,
                                                                  outputDirectoryPackagerConf,
                                                                  artifact);
//        System.out.println("JS=" + artifact.getFile());
        File destFile = new File(destDir, destFileName);
        copyFile(artifact.getFile(), destFile);
      }

      if (artifact.getArtifactId().contains("package") && artifact.getType().equals("zip")) {
        File destDir = DependencyUtil.getFormattedOutputDirectory(useSubDirectoryPerType,
                                                                  useSubDirectoryPerArtifact,
                                                                  useRepositoryLayout,
                                                                  stripVersion,
                                                                  outputDirectoryPackager,
                                                                  artifact);
        unpack(artifact.getFile(), destDir, null, null);
        DefaultFileMarkerHandler handler = new DefaultFileMarkerHandler(artifact,
                                                                        this.markersDirectory);
        handler.setMarker();
      }
    }
    
    String[] deployList = deploy.split(",");
    for (int i = 0; i < deployList.length; i++) {
      String deployTarget = deployList[i];
//      System.out.println("EXOBUILD preparing :" +deployTarget);
      executeExobuild(deployTarget);
    }
  }

  /**
   * Copy the pom files associated with the artifacts.
   */
  public void copyPoms(File destDir, Set artifacts, boolean removeVersion) throws MojoExecutionException

  {
    Iterator iter = artifacts.iterator();
    while (iter.hasNext()) {
      Artifact artifact = (Artifact) iter.next();
      Artifact pomArtifact = getResolvedPomArtifact(artifact);

      // Copy the pom
      if (pomArtifact.getFile() != null && pomArtifact.getFile().exists()) {
        File pomDestFile = new File(destDir, DependencyUtil.getFormattedFileName(pomArtifact,
                                                                                 removeVersion));
        if (!pomDestFile.exists()) {
          copyFile(pomArtifact.getFile(), pomDestFile);
        }
      }
    }
  }

  protected Artifact getResolvedPomArtifact(Artifact artifact) {
    Artifact pomArtifact = this.factory.createArtifact(artifact.getGroupId(),
                                                       artifact.getArtifactId(),
                                                       artifact.getVersion(),
                                                       "",
                                                       "pom");
    // Resolve the pom artifact using repos
    try {
      this.resolver.resolve(pomArtifact, this.remoteRepos, this.local);
    } catch (Exception e) {
      getLog().info(e.getMessage());
    }
    return pomArtifact;
  }

  protected ArtifactsFilter getMarkedArtifactFilter() {
    return new DestFileFilter(this.overWriteReleases,
                              this.overWriteSnapshots,
                              this.overWriteIfNewer,
                              this.useSubDirectoryPerArtifact,
                              this.useSubDirectoryPerType,
                              this.useRepositoryLayout,
                              this.stripVersion,
                              this.outputDirectory);
  }

  // <executable>java</executable>
  // <workingDirectory>${basedir}</workingDirectory>
  // <arguments>
  // <argument>-Dexo.package.home=${basedir}/target/packager</argument>
  // <argument>-Dexo.current.dir=${basedir}</argument>
  // <argument>-Dexo.base.dir=${exo.projects.directory.base}</argument>
  // <argument>-Dexo.conf.dir=${basedir}/target/packager-conf</argument>
  // <argument>-Dexo.working.dir=${exo.working.dir}</argument>
  // <!--argument>-Dexo.src.dir=NONE</argument-->
  // <argument>-Dexo.dep.dir=${exo.projects.directory.dependencies}</argument><!--
  // to get the server ref install -->
  // <argument>-Dexo.m2.repos=file:${settings.localRepository},http://maven2.exoplatform.org/rest/maven2,http://repository.jboss.org/maven2</argument>
  // <argument>-Dclean.server=${exo.projects.app.jboss.version}</argument>
  // <argument>-Dexo.m2.home=${maven.home}</argument>
  // <argument>-Xshare:auto</argument>
  // <argument>-Xms128m</argument>
  // <argument>-Xmx512m</argument>
  // <argument>-classpath</argument>
  // <argument>${basedir}/target/packager/lib/js.jar</argument>
  // <argument>org.mozilla.javascript.tools.shell.Main</argument>
  // <argument>${basedir}/target/packager/javascript/eXo/eXo.js</argument>
  // <argument>exobuild</argument>
  // <argument>--product=portal</argument>
  // <argument>--deploy=jbossear</argument>
  // </arguments>

  /**
   * priority in the execute method will be to use System properties arguments
   * over the pom specification.
   * @param deployTarget TODO
   * 
   * @throws MojoExecutionException if a failure happens
   */
  public void executeExobuild(String deployTarget) throws MojoExecutionException {

    if (basedir == null) {
      throw new IllegalStateException("basedir is null. Should not be possible.");
    }
    if (exoWorkingDir == null) {
      throw new IllegalStateException("exo.working.dir is null. Please define it as property or exoWorkingDir in maven-exobuild-plugin configuration.");
    }

//    System.out.println("exoWorkingDir: " + exoWorkingDir);
//    System.out.println("exoDependenciesDir: " + exoDependenciesDir);
//    System.out.println("exoJBossVersion: " + exobuildJBossVersion);

    List arguments = new ArrayList();
    arguments.add("-Dexo.package.home=" + basedir + "/target/packager");
    arguments.add("-Dexo.current.dir=" + basedir);
    arguments.add("-Dexo.base.dir=" + exoBaseDir);
    arguments.add("-Dexo.conf.dir=" + basedir + "/target/packager-conf");
    arguments.add("-Dexo.working.dir=" + exoWorkingDir);// ${exo.working.dir}
    arguments.add("-Dexo.dep.dir=" + exoDependenciesDir); // ${exo.projects.directory.dependencies}
    arguments.add("-Dexo.m2.repos=file:" + local.getBasedir());
    if (deployTarget.equals("tomcat"))
      arguments.add("-Dclean.server=" + exobuildTomcatVersion);
    if (deployTarget.startsWith("jboss"))
      arguments.add("-Dclean.server=" + exobuildJBossVersion);
    // ${exo.projects.app.jboss.version},
    // ${exo.projects.app.tomcat.version}
    arguments.add("-Dexo.m2.home=" + mavenHome);
    arguments.add("-DrepoUrl=xxx"); //To allow exobuild to call back maven
    arguments.add("-Xshare:auto");
    arguments.add("-Xms128m");
    arguments.add("-Xmx512m");
    arguments.add("-classpath");
    arguments.add(basedir + "/target/packager/lib/js.jar");
    arguments.add("org.mozilla.javascript.tools.shell.Main");
    arguments.add(basedir + "/target/packager/javascript/eXo/eXo.js");
    arguments.add("exobuild");
    arguments.add("--product=" + product);
    if (release != null)
      arguments.add("--release=" + deployTarget);
    //If not release then deploy (with same parameter than deploy
    if (release == null && deployTarget != null)
      arguments.add("--deploy=" + deployTarget);
    if (database != null)
      arguments.add("--database=" + database);
    if (dbsetup != null)
      arguments.add("--dbsetup=" + dbsetup);
    if (enable_workflow != null)
      arguments.add("--enable-workflow=" + enable_workflow);

    String[] args = (String[]) arguments.toArray(new String[arguments.size()]);
//    System.out.println("Parameters: ");
//    for (int i = 0; i < args.length; i++) {
//      System.out.print(" " + args[i]);
//    }
    Commandline commandLine = new Commandline();
    commandLine.setExecutable(getExecutablePath());
    commandLine.addArguments(args);

    File workingDirectory = basedir;
    if (!workingDirectory.exists()) {
      getLog().debug("Making working directory '" + workingDirectory.getAbsolutePath() + "'.");
      if (!workingDirectory.mkdirs()) {
        throw new MojoExecutionException("Could not make working directory: '"
            + workingDirectory.getAbsolutePath() + "'");
      }
    }

    commandLine.setWorkingDirectory(workingDirectory.getAbsolutePath());

    if (environmentVariables != null) {
      Iterator iter = environmentVariables.keySet().iterator();
      while (iter.hasNext()) {
        String key = (String) iter.next();
        String value = (String) environmentVariables.get(key);
        commandLine.addEnvironment(key, value);
      }
    }

    final Log outputLog = getExecOutputLog();

    StreamConsumer stdout = new StreamConsumer() {
      public void consumeLine(String line) {
        outputLog.info(line);
      }
    };

    StreamConsumer stderr = new StreamConsumer() {
      public void consumeLine(String line) {
        outputLog.info(line);
      }
    };

    try {
      getLog().debug("Executing command line: " + commandLine);

      int resultCode = executeCommandLine(commandLine, stdout, stderr);

      if (isResultCodeAFailure(resultCode)) {
        throw new MojoExecutionException("Result of " + commandLine + " execution is: '"
            + resultCode + "'.");
      }
    } catch (CommandLineException e) {
      throw new MojoExecutionException("Command execution failed.", e);
    }

    registerSourceRoots();
  }

  boolean isResultCodeAFailure(int result) {
    if (successCodes == null || successCodes.size() == 0)
      return result != 0;
    for (Iterator it = successCodes.iterator(); it.hasNext();) {
      int code = Integer.parseInt((String) it.next());
      if (code == result)
        return false;
    }
    return true;
  }

  private Log getExecOutputLog() {
    Log log = getLog();
    if (outputFile != null) {
      try {
        if (!outputFile.getParentFile().exists() && !outputFile.getParentFile().mkdirs()) {
          getLog().warn("Could not create non existing parent directories for log file: "
              + outputFile);
        }
        PrintStream stream = new PrintStream(new FileOutputStream(outputFile));
        log = new StreamLog(stream);
      } catch (Exception e) {
        getLog().warn("Could not open " + outputFile + ". Using default log", e);
      }
    }

    return log;
  }

  /**
   * Compute the classpath from the specified Classpath. The computed classpath
   * is based on the classpathScope. The plugin cannot know from maven the phase
   * it is executed in. So we have to depend on the user to tell us he wants the
   * scope in which the plugin is expected to be executed.
   * 
   * @param specifiedClasspath Non null when the user restricted the
   *          dependenceis, null otherwise (the default classpath will be used)
   * @return a platform specific String representation of the classpath
   */
  private String computeClasspathString(Classpath specifiedClasspath) {
    List resultList = computeClasspath(specifiedClasspath);
    StringBuffer theClasspath = new StringBuffer();

    for (Iterator it = resultList.iterator(); it.hasNext();) {
      String str = (String) it.next();
      addToClasspath(theClasspath, str);
    }

    return theClasspath.toString();
  }

  /**
   * Compute the classpath from the specified Classpath. The computed classpath
   * is based on the classpathScope. The plugin cannot know from maven the phase
   * it is executed in. So we have to depend on the user to tell us he wants the
   * scope in which the plugin is expected to be executed.
   * 
   * @param specifiedClasspath Non null when the user restricted the
   *          dependenceis, null otherwise (the default classpath will be used)
   * @return a list of class path elements
   */
  private List computeClasspath(Classpath specifiedClasspath) {
    List artifacts = new ArrayList();
    List theClasspathFiles = new ArrayList();
    List resultList = new ArrayList();

    collectProjectArtifactsAndClasspath(artifacts, theClasspathFiles);

    if ((specifiedClasspath != null) && (specifiedClasspath.getDependencies() != null)) {
      artifacts = filterArtifacts(artifacts, specifiedClasspath.getDependencies());
    }

    for (Iterator it = theClasspathFiles.iterator(); it.hasNext();) {
      File f = (File) it.next();
      resultList.add(f.getAbsolutePath());
    }

    for (Iterator it = artifacts.iterator(); it.hasNext();) {
      Artifact artifact = (Artifact) it.next();
      getLog().debug("dealing with " + artifact);
      resultList.add(artifact.getFile().getAbsolutePath());
    }

    return resultList;
  }

  private static void addToClasspath(StringBuffer theClasspath, String toAdd) {
    if (theClasspath.length() > 0) {
      theClasspath.append(File.pathSeparator);
    }
    theClasspath.append(toAdd);
  }

  private List filterArtifacts(List artifacts, Collection dependencies) {
    AndArtifactFilter filter = new AndArtifactFilter();

    filter.add(new IncludesArtifactFilter(new ArrayList(dependencies))); // gosh

    List filteredArtifacts = new ArrayList();
    for (Iterator it = artifacts.iterator(); it.hasNext();) {
      Artifact artifact = (Artifact) it.next();
      if (filter.include(artifact)) {
        getLog().debug("filtering in " + artifact);
        filteredArtifacts.add(artifact);
      }
    }
    return filteredArtifacts;
  }

  String getExecutablePath() {
    File execFile = new File(executable);
    if (execFile.exists()) {
      getLog().debug("Toolchains are ignored, 'executable' parameter is set to " + executable);
      return execFile.getAbsolutePath();
    } else {
      Toolchain tc = getToolchain();

      // if the file doesn't exist & toolchain is null, the exec is
      // probably in the PATH...
      // we should probably also test for isFile and canExecute, but the
      // second one is only
      // available in SDK 6.
      if (tc != null) {
        getLog().info("Toolchain in exec-maven-plugin: " + tc);
        executable = tc.findTool(executable);
      }
    }

    return executable;
  }

  private static boolean isEmpty(String string) {
    return string == null || string.length() == 0;
  }

  //
  // methods used for tests purposes - allow mocking and simulate automatic
  // setters
  //

  protected int executeCommandLine(Commandline commandLine,
                                   StreamConsumer stream1,
                                   StreamConsumer stream2) throws CommandLineException {
    return CommandLineUtils.executeCommandLine(commandLine, stream1, stream2);
  }

  void setExecutable(String executable) {
    this.executable = executable;
  }

  String getExecutable() {
    return executable;
  }

  void setArguments(List arguments) {
    this.arguments = arguments;
  }

  void setBasedir(File basedir) {
    this.basedir = basedir;
  }

  void setProject(MavenProject project) {
    this.project = project;
  }

  protected String getSystemProperty(String key) {
    return System.getProperty(key);
  }

  public void setSuccessCodes(List list) {
    this.successCodes = list;
  }

  public List getSuccessCodes() {
    return successCodes;
  }

  private Toolchain getToolchain() {
    Toolchain tc = null;

    try {
      if (session != null) // session is null in tests..
      {
        ToolchainManager toolchainManager = (ToolchainManager) session.getContainer()
                                                                      .lookup(ToolchainManager.ROLE);

        if (toolchainManager != null) {
          tc = toolchainManager.getToolchainFromBuildContext("jdk", session);
        }
      }
    } catch (ComponentLookupException componentLookupException) {
      // just ignore, could happen in pre-2.0.9 builds..
    }
    return tc;
  }

  /**
   * Create a jar with just a manifest containing a Main-Class entry for
   * SurefireBooter and a Class-Path entry for all classpath elements. Copied
   * from surefire (ForkConfiguration#createJar())
   * 
   * @param classPath List&lt;String> of all classpath elements.
   * @return
   * @throws IOException
   */
  private File createJar(List classPath, String mainClass) throws IOException {
    File file = File.createTempFile("maven-exec", ".jar");
    file.deleteOnExit();
    FileOutputStream fos = new FileOutputStream(file);
    JarOutputStream jos = new JarOutputStream(fos);
    jos.setLevel(JarOutputStream.STORED);
    JarEntry je = new JarEntry("META-INF/MANIFEST.MF");
    jos.putNextEntry(je);

    Manifest man = new Manifest();

    // we can't use StringUtils.join here since we need to add a '/' to
    // the end of directory entries - otherwise the jvm will ignore them.
    String cp = "";
    for (Iterator it = classPath.iterator(); it.hasNext();) {
      String el = (String) it.next();
      // NOTE: if File points to a directory, this entry MUST end in '/'.
      cp += UrlUtils.getURL(new File(el)).toExternalForm() + " ";
    }

    man.getMainAttributes().putValue("Manifest-Version", "1.0");
    man.getMainAttributes().putValue("Class-Path", cp.trim());
    man.getMainAttributes().putValue("Main-Class", mainClass);

    man.write(jos);
    jos.close();

    return file;
  }

  /**
   * Collects the project artifacts in the specified List and the project
   * specific classpath (build output and build test output) Files in the
   * specified List, depending on the plugin classpathScope value.
   * 
   * @param artifacts the list where to collect the scope specific artifacts
   * @param theClasspathFiles the list where to collect the scope specific
   *          output directories
   * @throws NullPointerException if at least one of the parameter is null
   */
  protected void collectProjectArtifactsAndClasspath(List artifacts, List theClasspathFiles) {

    if ("compile".equals(classpathScope)) {
      artifacts.addAll(project.getCompileArtifacts());
      theClasspathFiles.add(new File(project.getBuild().getOutputDirectory()));
    } else if ("test".equals(classpathScope)) {
      artifacts.addAll(project.getTestArtifacts());
      theClasspathFiles.add(new File(project.getBuild().getTestOutputDirectory()));
      theClasspathFiles.add(new File(project.getBuild().getOutputDirectory()));
    } else if ("runtime".equals(classpathScope)) {
      artifacts.addAll(project.getRuntimeArtifacts());
      theClasspathFiles.add(new File(project.getBuild().getOutputDirectory()));
    } else if ("system".equals(classpathScope)) {
      artifacts.addAll(project.getSystemArtifacts());
    } else {
      throw new IllegalStateException("Invalid classpath scope: " + classpathScope);
    }

    getLog().debug("Collected project artifacts " + artifacts);
    getLog().debug("Collected project classpath " + theClasspathFiles);
  }

  /**
   * Parses the argument string given by the user. Strings are recognized as
   * everything between STRING_WRAPPER. PARAMETER_DELIMITER is ignored inside a
   * string. STRING_WRAPPER and PARAMETER_DELIMITER can be escaped using
   * ESCAPE_CHAR.
   * 
   * @return Array of String representing the arguments
   * @throws MojoExecutionException for wrong formatted arguments
   */
  protected String[] parseCommandlineArgs() throws MojoExecutionException {
    if (commandlineArgs == null) {
      return null;
    }

    boolean inString = false;
    String arguments = commandlineArgs.trim();

    List argumentList = new LinkedList();

    int chr = 0;
    char curChar;
    StringBuffer arg = new StringBuffer();
    while (chr < arguments.length()) {
      curChar = arguments.charAt(chr);
      if (curChar == ESCAPE_CHAR) {
        // A char is escaped, replace and skip one char
        if (arguments.length() == chr + 1) {
          // ESCAPE_CHAR is the last character... this should be an
          // error (?)
          // char is just ignored by now
          getLog().warn(ESCAPE_CHAR
              + " was the last character in your command line arguments. Verify your parameters.");
        } else {
          // In a string only STRING_WRAPPER is escaped
          if (!inString || arguments.charAt(chr + 1) == STRING_WRAPPER) {
            chr++;
          }

          arg.append(arguments.charAt(chr));
        }
      } else if (curChar == PARAMETER_DELIMITER && !inString) {
        // Next parameter begins here
        argumentList.add(arg.toString());
        arg.delete(0, arg.length());
      } else if (curChar == STRING_WRAPPER) {
        // Entering or leaving a string
        inString = !inString;
      } else {
        // Append this char to the current parameter
        arg.append(curChar);
      }

      chr++;
    }
    if (inString) {
      // Error string not terminated
      throw new MojoExecutionException("args contains not properly formatted string");
    } else {
      // Append last parameter
      argumentList.add(arg.toString());
    }

    Iterator it = argumentList.listIterator();
    String[] result = new String[argumentList.size()];
    int index = 0;
    while (it.hasNext()) {
      result[index] = (String) it.next();
      index++;
    }

    getLog().debug("Args:");
    it = argumentList.listIterator();
    while (it.hasNext()) {
      getLog().debug(" <" + (String) it.next() + ">");
    }
    getLog().debug(" parsed from <" + commandlineArgs + ">");

    return result;
  }

  /**
   * @return true of the mojo has command line arguments
   */
  protected boolean hasCommandlineArgs() {
    return (commandlineArgs != null);
  }

  /**
   * Register compile and compile tests source roots if necessary
   */
  protected void registerSourceRoots() {
    if (sourceRoot != null) {
      getLog().info("Registering compile source root " + sourceRoot);
      project.addCompileSourceRoot(sourceRoot.toString());
    }

    if (testSourceRoot != null) {
      getLog().info("Registering compile test source root " + testSourceRoot);
      project.addTestCompileSourceRoot(testSourceRoot.toString());
    }
  }

}
