/*
 * Copyright (C) 2003-2011 eXo Platform SAS.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package org.exoplatform.social.core.chromattic.entity;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.chromattic.api.RelationshipType;
import org.chromattic.api.annotations.Create;
import org.chromattic.api.annotations.DefaultValue;
import org.chromattic.api.annotations.FormattedBy;
import org.chromattic.api.annotations.Id;
import org.chromattic.api.annotations.ManyToOne;
import org.chromattic.api.annotations.MappedBy;
import org.chromattic.api.annotations.Name;
import org.chromattic.api.annotations.NamingPrefix;
import org.chromattic.api.annotations.OneToMany;
import org.chromattic.api.annotations.OneToOne;
import org.chromattic.api.annotations.Owner;
import org.chromattic.api.annotations.Path;
import org.chromattic.api.annotations.PrimaryType;
import org.chromattic.api.annotations.Property;
import org.chromattic.ext.format.BaseEncodingObjectFormatter;
import org.exoplatform.social.core.storage.query.PropertyLiteralExpression;

/**
 * @author <a href="mailto:alain.defrance@exoplatform.com">Alain Defrance</a>
 * @version $Revision$
 */
@PrimaryType(name = "soc:activity", orderable = true)
@FormattedBy(BaseEncodingObjectFormatter.class)
@NamingPrefix("soc")
public abstract class ActivityEntity implements NamedEntity {

  @Id
  public abstract String getId();

  @Name
  public abstract String getName();

  @ManyToOne
  public abstract ActivityEntity getParentActivity();

  /**
   * All comments of the identity. The child is the posted time stamp.
   */
  @Owner
  @OneToMany
  public abstract List<ActivityEntity> getComments();
  
  @OneToMany(type = RelationshipType.REFERENCE)
  @MappedBy("ref")
  public abstract Collection<ActivityRef> getActivityRefs();

  @ManyToOne
  public abstract ActivityDayEntity getDay();

  /**
   * The identity whose activity stream contains an activity.
   */
  @MappedBy("soc:identity")
  @ManyToOne(type = RelationshipType.REFERENCE)
  public abstract IdentityEntity getIdentity();
  public abstract void setIdentity(IdentityEntity identity);
  public static final PropertyLiteralExpression<String> identity =
      new PropertyLiteralExpression<String>(String.class, "soc:identity");

  /**
   * The identity of the user who creates an activity.
   */
  @MappedBy("soc:posterIdentity")
  @ManyToOne(type = RelationshipType.REFERENCE)
  public abstract IdentityEntity getPosterIdentity();
  public abstract void setPosterIdentity(IdentityEntity posterIdentity);
  public static final PropertyLiteralExpression<String> poster =
    new PropertyLiteralExpression<String>(String.class, "soc:posterIdentity");

  /**
   * The string which specifies the primary text of an activity.
   */
  @Property(name = "soc:title")
  public abstract String getTitle();
  public abstract void setTitle(String title);
  public static final PropertyLiteralExpression<String> title =
    new PropertyLiteralExpression<String>(String.class, "soc:title");

  /**
   * The title Id of an activity.
   */
  @Property(name = "soc:titleId")
  public abstract String getTitleId();
  public abstract void setTitleId(String titleId);

  /**
   * The application Id which creates an activity.
   */
  @Property(name = "soc:appId")
  public abstract String getAppId();
  public abstract void setAppId(String appId);

  /**
   * The string which specifies the body template message Id in the gadget specification. The body is an optional extended version of an activity.
   */
  @Property(name = "soc:body")
  public abstract String getBody();
  public abstract void setBody(String body);

  /**
   * The body Id of an activity.
   */
  @Property(name = "soc:bodyId")
  public abstract String getBodyId();
  public abstract void setBodyId(String bodyId);

  /**
   * The application Id which creates an activity.
   */
  @Property(name = "soc:type")
  public abstract String getType();
  public abstract void setType(String type);

  /**
   * An optional string Id which is generated by the posting application.
   */
  @Property(name = "soc:externalId")
  public abstract String getExternalId();
  public abstract void setExternalId(String externalId);

  /**
   * The URL to access an activity.
   */
  @Property(name = "soc:url")
  public abstract String getUrl();
  public abstract void setUrl(String url);

  /**
   * A float number between '0' and '1' which represents the relative priority level of an activity in relation to other activities from the same source.
   */
  @Property(name = "soc:priority")
  public abstract Float getPriority();
  public abstract void setPriority(Float priority);

  /**
   * The list of identity Ids who like the activity.
   */
  @Property(name = "soc:likes")
  public abstract String[] getLikes();
  public abstract void setLikes(String[] title);
  public static final PropertyLiteralExpression<String> likes =
    new PropertyLiteralExpression<String>(String.class, "soc:likes");

  /**
   * Specify if an activity is a comment or not. The default value is false, meaning that it is a normal activity.
   */
  @Property(name = "soc:isComment")
  @DefaultValue("false")
  public abstract Boolean isComment();
  public abstract void setComment(Boolean isComment);
  public static final PropertyLiteralExpression<Boolean> isComment =
      new PropertyLiteralExpression<Boolean>(Boolean.class, "soc:isComment");

  /**
   * The number which specifies the time at which an activity took place in milliseconds since the epoch.
   */
  @Property(name = "soc:postedTime")
  public abstract Long getPostedTime();
  public abstract void setPostedTime(Long postedTime);
  public static final PropertyLiteralExpression<Long> postedTime =
      new PropertyLiteralExpression<Long>(Long.class, "soc:postedTime");
  
  @Property(name = "soc:lastUpdated")
  public abstract Long getLastUpdated();
  public abstract void setLastUpdated(Long lastUpdated);
  public static final PropertyLiteralExpression<Long> lastUpdated =
      new PropertyLiteralExpression<Long>(Long.class, "soc:lastUpdated");

  /**
   * The list of identity Ids who mentioned.
   */
  @Property(name = "soc:mentioners")
  public abstract String[] getMentioners();
  public abstract void setMentioners(String[] mentionerId);
  public static final PropertyLiteralExpression<String> mentioners =
    new PropertyLiteralExpression<String>(String.class, "soc:mentioners");

  /**
   * The list of identity Ids who commented.
   */
  @Property(name = "soc:commenters")
  public abstract String[] getCommenters();
  public abstract void setCommenters(String[] commenterId);
  public static final PropertyLiteralExpression<String> commenters =
    new PropertyLiteralExpression<String>(String.class, "soc:commenters");
  
  /**
   * The activity parameters.
   * @return
   */
  @MappedBy("soc:params")
  @OneToOne
  @Owner
  public abstract ActivityParameters getParams();
  public abstract void setParams(ActivityParameters params);

  @Path
  public abstract String getPath();

  @Create
  public abstract ActivityEntity createComment(String name);

  @Create
  public abstract ActivityParameters createParams();

  public void putParams(Map<String, String> parameters) {

    ActivityParameters params = getParams();
    if (params == null) {
      setParams(params = createParams());
    }

    //
    Map<String, String> chromatticMap = params.getParams();
    for (Map.Entry<String, String> entry : parameters.entrySet()) {
      if (!entry.getKey().startsWith("jcr:"))
      chromatticMap.put(entry.getKey(), entry.getValue());
    }
    
  }
  
  @Override
  public String toString() {
    return String.format("\nAct{id=%s,type=%s,lastUpdated=%s,title='%s',poster=%s,streamOwner=%s}",
                         getId(),
                         getTitleId(),
                         getLastUpdated(),
                         getTitle(),
                         getPosterIdentity().getRemoteId(),
                         getIdentity().getRemoteId());
  }
}
