/*
 * Isomorphic SmartClient
 * Version v9.0_2013-07-03 (2013-07-03)
 * Copyright(c) 1998 and beyond Isomorphic Software, Inc. All rights reserved.
 * "SmartClient" is a trademark of Isomorphic Software, Inc.
 *
 * licensing@smartclient.com
 *
 * http://smartclient.com/license
 */




//>	@class	ListGrid
// A ListGrid is a +link{DataBoundComponent} that displays a list of objects in a grid, where
// each row represents one object and each cell in the row represents one property. 
//
//  @implements DataBoundComponent    
//  @treeLocation Client Reference/Grids
//  @visibility external
//<
// Make ListGrid a subclass of VLayout. This allows us to change the order of the sub-components
// - show summary row between header and body, etc.

isc.ClassFactory.defineClass("ListGrid", "VLayout", "DataBoundComponent");

// Synonym for backCompat.  NOTE: define an alias rather than make a subclass, otherwise, attempts
// to skin the class using the old name would only affect the subclass!
isc.addGlobal("ListViewer", isc.ListGrid);
// define groups for documentation purposes

    //> @groupDef data
    //<

    //> @groupDef databinding
    // DataBinding means the automatic, highly customizable process of 'binding' a UI component
    // to a DataSource, so that a UI component displays, edits and saves DataSource records
    // using appropriate formatters, editors, validation rules, and persistence logic.
    // 
    // @see interface:DataBoundComponent
    // @title DataBinding
    //<

    //> @groupDef sorting
    //< 

    //> @groupDef editing
    // Data being displayed by a grid may be edited within the grid, by showing editing
    // interfaces embedded inside the cells of the grid.
    // <P>
    // <b>Enabling editing</b>
    // <P>
    // Editing is enabled when +link{listGrid.canEdit,canEdit} is <code>true</code>.  When enabled,
    // the user can begin editing via the 
    // +link{listGrid.editEvent,editEvent}, typically click or double-click.  Editing can also be triggered
    // programmatically by a call to +link{listGrid.startEditing,startEditing()} or
    // +link{listGrid.startEditingNew,startEditingNew()}.
    // <P>
    // <b>New record creation</b>
    // <P>
    // By default, editing is restricted to existing records.  Setting +link{listGrid.listEndEditAction} to
    // "next" allows the user to create new records by simply navigating off the end of the dataset
    // with the keyboard.  Editing of new records can also be initiated with 
    // +link{listGrid.startEditingNew()}, for example, from a button outside the grid.  See the
    // +link{group:unsavedRecords,Unsaved Records Overview} for special concerns when dealing
    // with unsaved records.
    // <P>
    // <b>Saving changes</b>
    // <P>
    // Saving of changes is triggered automatically when the user navigates out of the row or cell
    // being edited (based on +link{listGrid.saveByCell}) or when the user ends editing.   For
    // a "mass update" interface, automatic saving of changes can be disabled entirely via 
    // +link{listGrid.autoSaveEdits,autoSaveEdits:false}, in which case a manual call to 
    // +link{listGrid.saveEdits,saveEdits()} or +link{listGrid.saveAllEdits,saveAllEdits()} is required
    // to trigger saving. 
    // <P>
    // If a grid has no DataSource, saving means that the properties of the +link{ListGridRecord}s
    // in +link{listGrid.data,grid.data} are directly changed.
    // <P>
    // For a grid with a DataSource, saving will be accomplished by using DataSource "update"
    // operations for existing records, and DataSource "add" operations for new records.  If multiple
    // records have been edited and +link{listGrid.saveAllEdits,saveAllEdits()} is called, 
    // +link{rpcManager.startQueue,request queuing} will be automatically used to enable all
    // edits to be saved in one HTTP turnaround (if using the SmartClient Server).
    // <P>
    // By default, a grid will send only updated fields and primaryKey fields as part of 
    // +link{dsRequest.data} so that the server can discern which fields the user actually changed.
    // However, the grid always includes the original field values in the
    // dsRequest as +link{dsRequest.oldValues}.
    // <P>
    // Note that although it is possible to load DataSource data without actually declaring a
    // +link{dataSourceField.primaryKey,primaryKey field}, a primaryKey must be declared for
    // editing and saving.  The values of primaryKey fields is how SmartClient identifies the
    // changed record to the server.
    // <P>
    // <b>Validation</b>
    // <P>
    // Any time saving is attempted, validation is automatically triggered.  Values entered by the
    // user will be checked against the +link{listGridField.validators} and the
    // +link{dataSourceField.validators}. Any invalid values abort an attempted save.
    // <P>
    // Similar to editing and saving, validation can be done on row transitions or on cell
    // transitions by setting +link{listGrid.validateByCell,validateByCell}, or can be disabled entirely
    // via +link{listGrid.neverValidate,neverValidate:true}.
    // <P>
    // <b>Editability of cells</b>
    // <P>
    // Editors will either be shown for the complete row or for a single cell based on 
    // +link{listGrid,editByCell,editByCell}.  Whether a cell can be edited can be controlled on a
    // per field basis by setting +link{listGridField.canEdit,field.canEdit}, or on a per-record basis
    // by setting +link{listGrid.recordEditProperty,recordEditProperty} on a 
    // +link{ListGridRecord,record}, or can be controlled on an arbitrary, programmatic basis via
    // an override of +link{listGrid.canEditCell()}.
    // <P>
    // Cells which are not editable just display the cell's current value.
    // <P>
    // <b>Keyboard Navigation</b>
    // <P>
    // Full keyboard navigation is supported by default, including Tab and Shift-Tab to navigate
    // between cells in a row, and Up Arrow and Down Arrow to traverse rows.  Several properties
    // on both grids and fields, all named *EditAction, control navigation behavior of certain keys
    // (eg Enter).
    // <P>
    // You can use +link{listGrid.startEditing,startEditing(<i>rowNum</i>, <i>colNum</i>)} to 
    // programmatically move editing to a particular cell, for example, during a 
    // +link{listGridField.changed,field.changed()} event.
    // <P>
    // <b>editValues (unsaved changes)</b>
    // <P>
    // The term "editValues" means changes that the user has made to the dataset which have not
    // been saved.  The grid manages and stores editValues separately from the data itself in order
    // to allow the user to revert to original values, and in order to enable to grid to send only
    // updated fields to the server.  
    // <P>
    // Because editValues are stored separately, if you directly access the dataset (eg via 
    // <code>grid.getData().get()</code>) you will see the records without the user's unsaved changes.
    // Many APIs exist for retrieving and managing editValues (search for editValue).
    // For the common case of needing to access the record-as-edited, you can call 
    // +link{listGrid.getEditedRecord,grid.getEditedRecord(rowNum)}.
    // <P>
    // When accessing and manipulating edited data, you should think carefully about whether
    // you want to be working with the original data or with the edited version.  Values entered
    // by the user may not have been validated yet, or may have failed validation, hence you may
    // find a String value in a field of type "date" or "int", which could cause naive formatters or
    // totaling functions to crash.
    // <P>
    // Setting editValues via APIs such as +link{listGrid.setEditValue()} is fully equivalent
    // to the user making changes to data via the editing UI.  If you <i>also</i> allow editing
    // external to the grid, setting editValues is one way to combine changes from external
    // editors into the grid's edits, so that you can do a single save.
    // <P>
    // <b>Customizing Cell Editors</b>
    // <P>
    // When a cell is being edited, the editor displayed in the cell will be a +link{class:FormItem}.
    // The editor type for the cell will be determined by +link{listGrid.getEditorType()} based on the
    // specified +link{ListGridField.editorType} or +link{ListGridField.type, data type} for the field in
    // question.
    // <P>
    // You can customize the editor by setting +link{listGridField.editorProperties} to a set of
    // properties that is valid for that FormItem type.  Custom FormItem classes are also allowed,
    // for example, you may use +link{formItem.icons} to create an icon that launches a separate
    // +link{Dialog} in order to provide an arbitrary interface that allows the user to select the
    // value for a field.
    // <P>
    // <b>Events</b>
    // <P>
    // Editing triggers several events which you can provide handlers for in order to customize
    // editing behavior.  Some of the most popular are +link{listGridField.change,field.change()},
    // +link{listGridField.changed,field.changed()} for detecting changes made by the user,
    // +link{listGrid.cellChanged()} for detecting changes that have been successfully saved,
    // and +link{listGrid.editorEnter,editorEnter} and +link{listGrid.editorExit,editorExit()}
    // for detecting user navigation during editing.
    // <P>
    // <var class="smartclient">
    // You can also install event handlers directly on the FormItem-based editors used in the grid
    // via +link{listGridField.editorProperties,editorProperties} as mentioned above.  When handling
    // events on items, or which involve items, be aware that in addition to standard 
    // +link{FormItem} APIs, editors have the following properties:
    // <P>
    // - <code>rowNum</code>: The rowNum of the record being edited.<br>
    // - <code>colNum</code>: The colNum of the cell being edited.<br>
    // - <code>grid</code>: A pointer back to the listGrid containing the record.
    // </var>
    // <var class="smartgwt">
    // <code>ListGridField.setEditorType()</code> can be used to customize the editors shown
    // for each field, including providing FormItem-specific event handlers.  However,
    // ListGrid-provided event APIs should be used wherever possible (for example, use
    // <code>EditorEnterEvent</code> rather than <code>FocusEvent</code>).  If, in a FormItem
    // event handler, you need access to the ListGrid, you can either declare the event handler
    // as a Java "inner class" in a scope where the ListGrid is available as a final variable,
    // or you can use <code>event.getItem().getContainerWidget()</code>.  Note the ListGrid APIs
    // +link{listGrid.getEditRow,getEditRow()} and +link{listGrid.getEditCol,getEditCol()}
    // indicate what cell is being edited.
    // <P>
    // For more dynamic editor customization, include changing the type of editor used on a
    // per-row basis, use +link{listGrid.setEditorCustomizer()}.
    // <P>
    // <b>NOTE:</b> with both APIs, in effect several FormItems are generated from the
    // customized FormItem you provide - see the docs for
    // +link{DataSourceField.setEditorType()} for special coding patterns that apply in this
    // case.
    // </var>
    // <P>
    // <b>Binary Fields</b>
    // <P>
    // The ListGrid will automatically show "view" and "download" icon buttons for binary field
    // types (see +link{type:ListGridFieldType}).  However, you cannot use an upload control
    // embedded within a ListGrid row to upload files (regardless of whether you use FileItem or
    // UploadItem).  This is because, in the browser's security model, native HTML upload
    // controls cannot be re-created and populated with a chosen file, and the ListGrid needs
    // to be able to re-create editors at any time in order to handle loading data on scroll,
    // scrolling editors in and out of view, adding new rows, changing sort direction, and
    // other use cases.
    // <P>
    // However you <i>can</i> create an editor with a +link{formItem.icons,FormItem icon} that
    // pops up a separate Window containing a FileItem in a DynamicForm, so long as the form in
    // the Window saves the uploaded file immediately rather than trying to have the grid
    // perform the save.
    //
    // @title Grid Editing
    // @treeLocation Client Reference/Grids/ListGrid
    // @visibility external
    //<
    

    
    //> @groupDef unsavedRecords
    // APIs such as +link{listGrid.startEditingNew(),startEditingNew()} or 
    // +link{listGrid.listEndEditAction,listEndEditAction:"next"} allow editing records that have not
    // been saved to the server.  These unsaved records are special in several ways:
    // <ul> 
    // <li> there is no actual Record object in the dataset for them: <code>getRecord(rowNum)</code>
    // will return null, instead, <code>getEditValues(rowNum)</code> allows access to field values for
    // the unsaved record 
    // <li> rows for editing these records always appear at the end of the grid and do not sort with
    // other rows
    // <li> because unsaved records lack an actual Record object and lack a
    // +link{dataSourceField.primaryKey} value, they have limited functionality: they cannot be
    // selected, and do not support +link{listGrid.showRecordComponents} and certain other features.
    // </ul>
    // <P>
    // If you need to work with unsaved records and have all ListGrid features apply to them, this is
    // usually a sign that you should re-think your UI for adding new records.  Consider the following
    // approaches - which works best will depend on the application:
    // <ul>
    // <li> actually save a new record to persistent storage, then start editing it.  This has the
    // advantage that the user will never lose data by exiting the application with unsaved
    // records, which can be important if there is a lot of data entry before the record is ready to
    // save (for example, a new issue report in an issue-tracking applications, or a new blog entry).
    // This is also a good approach if the user may want to get a unique ID for the new record 
    // right away (again useful for a new issue report or blog entry).
    // <P>
    // If values for several fields are required before the record should be visible on other screens
    // or to other users, you can add a field to the record to flag it as incomplete so that it is not
    // shown on other screens.  Alternatively, require certain fields to be entered via an external
    // form or dialog before the record is added to the grid.  
    // <P>
    // Saving a new record and editing it can be done via +link{DataSource.addData()} followed by a call to
    // +link{listGrid.startEditing()} once the record has been saved.
    // <li> edit new records via a separate +link{form,DynamicForm} instead, possibly in a modal
    // +link{Window} - then unsaved records never need to be shown in the grid.  Similar to the
    // approach above, this modal form might have only certain minimum fields to make a valid
    // new record, then further editing could continue in the grid.
    // <li> use a +link{dataSource.clientOnly,clientOnly DataSource} so that records can be saved
    // immediately without contacting the server.  This is a good approach if several unsaved records
    // need to be manipulated by multiple components before they are finally saved.
    // <li> use +link{DataSource.updateCaches()} with an "add" DSResponse to cause a new record to be
    // added to the grid due to +link{ResultSet,automatic cache synchronization}.  At this point the
    // grid will believe the record exists on the server and it will be treated like any other saved
    // record.  This means your server code will need to handle the fact that the ListGrid will submit
    // "update" DSRequests for any subsequent edits.
    // </ul>
    // <b>NOTE about validation:</b> by design, SmartClient assumes that any record that has been
    // saved is valid and does not validate field values that appear in records loaded from the
    // server.  This includes records added to a clientOnly DataSource via
    // +link{DataSource.setCacheData()} as well as records added due to a call to
    // +link{DataSource.updateCaches()}.  
    // <P>
    // Usually the best approach is to avoid this situation by editing such records in a form or other
    // control until they are valid rather than showing invalid records in a grid.  However, if such
    // records need to be considered invalid, one approach is to take field values and add them as
    // editValues via +link{listGrid.setEditValues()}.  At this point the ListGrid will consider the
    // values as user edits and will validate them.
    // 
    // @title Handling Unsaved Records
    // @visibility external
    //<
    
    //> @groupDef imageColumns
    // Columns that show images either as their only appearance or in addition to text.
    //<
    
    //> @groupDef formulaFields
    // Fields with values calculated from other fields in the grid.
    //<

isc.defineClass("GridBody", isc.GridRenderer).addProperties({
    // suppress adjustOverflow while pending a redraw so we can resize smaller without seeing 
    // a scrollbar flash in and out of existence
    adjustOverflowWhileDirty:false,
    
    canSelectRecord : function(record) {
        return this.grid.canSelectRecord(record);
    },

    // adjustOverflow() - overridden to support 'autoFitData' behavior
    adjustOverflow : function (reason, a,b,c,d) {
        // If we get naively called while undrawn just call Super which will bail.
        if (!this.isDrawn()) return this.Super("adjustOverflow", arguments);
        // we call 'getDelta' from this method which can fall back through to 'adjustOverflow'
        // Avoid infinite looping if we hit this case.
        if (this._calculatingDelta) return;
        
        
        if (this.grid._updatingRecordComponents) {
            return this.Super("adjustOverflow", arguments);
        }
        
        
        var grid = this.grid;
        
        
        if (grid == null) return this.Super("adjustOverflow", arguments);

        var data = grid.data, isLoading = false;;
            
        if (isc.isA.ResultSet(data) && !data.lengthIsKnown()) {
            if (grid.emptyMessageHeight == null) {
                return this.invokeSuper(isc.GridBody, "adjustOverflow", reason,a,b,c,d);
            }
            isLoading = true;
        }
        
        var initialWidth = this.getWidth(), initialHeight = this.getHeight();
        var fitVertical = (this.autoFitData == "both"),
            fitHorizontal = fitVertical,
            frozen = grid && grid.frozenFields != null,
            isFrozenBody = frozen && grid && (grid.frozenBody == this);

        if (!fitVertical) fitVertical = (this.autoFitData == "vertical");
        if (!fitHorizontal) fitHorizontal = (this.autoFitData == "horizontal");
        // If we have frozen fields, the frozen body never shows scrollbars and always
        // gets sized to match the widths of the fields it contains (done as part of
        // setBodyFieldWidths). Don't worry about trying to run special auto-fit logic
        // on the frozen body.
        // - We do run auto-fit logic on the unfrozen body and take the size of the frozen
        //   body into account when doing so.
        // - We do still need to ensure the header layout is sized correctly when the frozen
        //   body is resized
        
        if (fitHorizontal || fitVertical) {
            var height, width, rowHeights, hscrollOn, vscrollOn, dX, dY;
            
            if (fitVertical) {
                var minHeight = this.grid.getAutoFitMinBodyHeight();
                height = minHeight;
                var totalRows = isLoading ? 0 : this.getTotalRows(),
                    rows = totalRows;
                    
                rowHeights = 0;
                // ignore autoFitMaxRecords if set to zero - this means fit to all records!
                if (this.autoFitMaxRecords) {
                    rows = Math.min(rows, this.autoFitMaxRecords);
                }
                if (rows > 0) {
                    // We need to handle variable rowHeights so we're going to have to look at
                    // the table element to determine the heights - we already have a method to
                    // do that
                    var drawnRowHeights = this._getDrawnRowHeights();
                    // If we have any undrawn rows assume calculated sizes
                    
                    var firstDrawnRow = this._firstDrawnRow,
                        lastDrawnRow = this._lastDrawnRow;
                    
                    
                    
                    // fdr / ldr unset implies no drawn rows - set such that we calculate
                    // theoretical heights only
                    if (this._firstDrawnRow == null) {
                        firstDrawnRow = rows;
                        lastDrawnRow = rows;
                    }
                    // _isFrozenBody defined in GridRenderer
                    var isFrozenBody = this._isFrozenBody();
                    
                    if (firstDrawnRow > 0) {
                        firstDrawnRow = Math.min(firstDrawnRow, rows);
                        for (var i = 0; i < firstDrawnRow; i++) {
                            rowHeights += this.getRowHeight ? 
                                            this.getRowHeight(this.grid.getRecord(i), i, isFrozenBody) 
                                            : this.cellHeight;
                        }
                    }
                    var lastLogicalRow = rows-1;
                    if (lastDrawnRow < lastLogicalRow) {
                        for (var i = lastDrawnRow+1; i < lastLogicalRow+1; i++) {
                            rowHeights += this.getRowHeight ? 
                                            this.getRowHeight(this.grid.getRecord(i), i, isFrozenBody) 
                                            : this.cellHeight;
                        }
                    }
                    // Measure the rendered rows and add up the heights.
                    // Note that getDrawnRowHeights() just returns an array of the heights of
                    // rendered rows so the first drawn row is the first entry in the array, not
                    // the _firstDrawnRow'th entry
                    lastDrawnRow = Math.min(lastDrawnRow, lastLogicalRow);
                    for (var i = 0; i <= lastDrawnRow-firstDrawnRow; i++) {   
                        rowHeights += drawnRowHeights[i];                        
                    }
                    // If we are clipping off any rows we know we have a v-scrollbar
                    vscrollOn = totalRows > rows;
                    
                    // Treat autoFitMaxHeight:0 as unspecified - resize as large as necessary
                    var autoFitMaxHeight = this.getAutoFitMaxHeight();
                    if (autoFitMaxHeight && rowHeights > autoFitMaxHeight) {
                        rowHeights = autoFitMaxHeight;
                        vscrollOn = true;
                    }
//                     this.logWarn("total rows to show:"+ rows +
//                      ", rendered:" + [this._firstDrawnRow,this._lastDrawnRow] +
//                      ", rowHeights total up to:"+ rowHeights + 
//                      ", current height:" + this.getHeight() + 
//                      ", body height based on ListGrid specified height:" + height);
                    
                } else {
                    // The emptyMessage renders in the available space. If emptyMessageHeight
                    // is explicitly set, leave that much space for it.
                    
                    if (this.grid.emptyMessageHeight != null) {
                        rowHeights = this.grid.emptyMessageHeight;
                    }
                }
                 

                // add some extra height if autoFitExtraRecords is set
                
                if (this.autoFitExtraRecords && this.autoFitExtraRecords > 0) {
                    var extraHeight = Math.round(this.autoFitExtraRecords * this.cellHeight);
                    rowHeights += extraHeight;
                }
 
            } else {
                vscrollOn = this.getScrollHeight() > this.getHeight();
            }
  
            if (fitHorizontal && !isFrozenBody) {
                var width = this.grid.getInnerWidth(),
                    frozenBodyWidth;
                
                if (frozen) {
                    var frozenWidths = this.grid.getFrozenSlots(this.grid._fieldWidths);
                    frozenBodyWidth = frozenWidths.sum();
                    width -= frozenBodyWidth;
                    
                    // if the frozenWidths exceed the specified width for the grid as a whole,
                    // apply an arbitrary small positive min width for the unfrozen body
                }
                
                  
                // Note that we're calling getColumnSizes on the GridRenderer
                // So if we the LG is frozen body this gives us the cols within the 
                // appropriate body, not the total set of cols in the grid.
                var colSizes = this.getColumnSizes(),
                    contentWidth = colSizes.sum();
                if (this.autoFitMaxColumns) {
                    var maxCols = this.autoFitMaxColumns;
                    // bit of a hack - how to deal with maxCols specified as a number <= the
                    // number of frozen fields.
                    // For now we just enforce at least one unfrozen field
                    if (frozen) {
                        maxCols = Math.max(1, maxCols-this.grid.frozenFields.length);
                    }
                    
                    if (maxCols < colSizes.length) {
                        colSizes = colSizes.slice(0, maxCols);
                    }
                }
                
                var colWidths = colSizes.sum();  
                if (this.autoFitMaxWidth) {
                    var maxWidth = this.autoFitMaxWidth;
                    if (frozen) maxWidth = Math.max(20, maxWidth - frozenBodyWidth);
                    colWidths = Math.min(maxWidth, colWidths);
                }
                hscrollOn = (this.overflow == isc.Canvas.SCROLL) ? true : 
                            (this.overflow == isc.Canvas.AUTO) ? (contentWidth > Math.max(width, colWidths)) :
                            false;
                                
            } else {
                hscrollOn = this.overflow == isc.Canvas.SCROLL ? true :
                            this.overflow == isc.Canvas.AUTO  ? this.getScrollWidth() > this.getWidth() :
                            false;
            }
            // Now we know if we have an h-scrollbar, adjust height and width for scrollbars /
            // borders / margin if appropriate
            if (fitVertical && rowHeights != null) {
                rowHeights += this.getVBorderPad() + this.getVMarginSize();
                if (hscrollOn) {
                    rowHeights += this.getScrollbarSize();
                    var autoFitMaxHeight = this.getAutoFitMaxHeight()
                    if (autoFitMaxHeight && rowHeights > autoFitMaxHeight) {
                        rowHeights = autoFitMaxHeight;
                    }
                }
                // Resize vertically if rowHeights (+ border etc) > the auto fit min height
                // (which is derived from the ListGrid's specified height)
                if (rowHeights > height) {
                    height = rowHeights;
                    this._vAutoFit = true;
                } else {
                    if (this._vAutoFit) delete this._vAutoFit; 
                }
            }
            if (fitHorizontal && !isFrozenBody && colWidths != null) {
                
                colWidths += this.getHBorderPad() + this.getHMarginSize();      
                if (vscrollOn || this.alwaysShowVScrollbar) {
                    colWidths += this.getScrollbarSize();
                    if (this.autoFitMaxWidth) {
                        var maxWidth = this.autoFitMaxWidth;
                        if (frozen) maxWidth = Math.max(20, maxWidth - frozenBodyWidth);
                        colWidths = Math.min(maxWidth, colWidths);
                    }
                }
                // Resize horizontally if colWidths > width
                if (colWidths > width) {
                    width = colWidths;
                    this._hAutoFit = true;
                } else {
                    if (this._hAutoFit) delete this._hAutoFit;
                }
            }

            // Calculate the delta with our current size.
            this._calculatingDelta = true;
            dY = this.getDelta(this._$height, height, this.getHeight());   
                     
            dX = this.getDelta(this._$width, width, this.getWidth());
            delete this._calculatingDelta;
            // If necessary resize to accommodate content!
            if (dY != null || dX != null) {
                this.resizeBy(dX, dY, null, null, true);
            }
            
            // if width change != null, resize header to match body
            // Note that if isFrozenBody is true we skipped the dX calculation so
            // always resize the headerLayout to match
            if (dX != null || (isFrozenBody && fitHorizontal)) {
                var lg = this.grid,
                    sortArrowSize = lg._showSortButton() ? lg.getScrollbarSize() : 0,
                    headerWidth = (width - sortArrowSize),
                    totalHeaderWidth = headerWidth;
                if (frozen && lg.headerLayout) {
                    
                    if (isFrozenBody) {
                        totalHeaderWidth = this.getWidth() + lg.body.getWidth();
                        // If we go past the autoFitMaxWidth limit, run adjustOverflow on the body
                        // to force it to shrink/start scrolling
                        if (lg.autoFitMaxWidth != null && 
                            (totalHeaderWidth + lg.getHBorderPad() + 
                                lg.getHMarginSize() > lg.autoFitMaxWidth)) 
                        {
                            // don't bother to go on and resize the header - we'll do that
                            // when the body adjust overflow method runs
                            return lg.body.adjustOverflow();
                        }
                        totalHeaderWidth -= sortArrowSize;
                        
                    } else {
                        totalHeaderWidth = headerWidth + lg.frozenBody.getWidth();
                    }
                    lg.headerLayout.setWidth(totalHeaderWidth);
                }
                
                // We can skip resizing the frozen header - this is handled in setBodyFieldWidths
                if (!isFrozenBody) {
                    var header = lg.header;
    
                    if (header && header.isDrawn()) {
                        header.setWidth(headerWidth);
                    }
                }
            }
        
        // if autoFitData is null but we don't match our 'specified size', assume the property
        // has been modified and reset to specified size
        }
        
        // catch the case where autoFitData has been cleared in either direction and
        // reset to specified size.
        var verticalChanged = (!fitVertical && this._vAutoFit),
            horizontalChanged = (!fitHorizontal && this._hAutoFit);
        if (verticalChanged || horizontalChanged) {
            delete this._vAutoFit;
            delete this._hAutoFit;
            
            var standardHeight = verticalChanged ? this.grid.getAutoFitMinBodyHeight() : null,
                standardWidth = horizontalChanged ? 
                                (!frozen ? this.grid.getInnerWidth() : 
                                    (this.grid.getInnerWidth() - this.grid.frozenBody.getWidth()) )
                                                  : null;
            this.resizeTo(width,height);
            // reset field widths on the grid to resize the header to match the body
            this.grid._updateFieldWidths("autoFitData mode changed");
        }
        var returnVal = this.invokeSuper(isc.GridBody, "adjustOverflow", reason, a,b,c,d);
        // if size changed, refresh recordComponents to account for new draw area
        if ((fitVertical || fitHorizontal) &&
            (this.getWidth() != initialWidth || this.getHeight() != initialHeight))
        {
            this.grid.updateRecordComponents(true);
        }
        
        // Fire the "bodyOverflowed" observation. This updates frozen body end space and
        // summary row body right space to so these can keep in sync with body scrolling
        // even though the viewport sizes are different.
        if (!isFrozenBody) this.grid.bodyOverflowed();
        return returnVal;
    },
    
    getAutoFitMaxHeight : function () {
        return this.grid ? this.grid.getAutoFitMaxBodyHeight() : null;
    },
    
    resizeBy : function (deltaX, deltaY, animating, suppressHandleUpdate, autoFitSize) {
        
        // autoFitSize parameter: When autoFitData is true for this grid, we resize the
        // body to fit the data, and pass in the autoFitSize parameter to this method.
        // In the case of an explicit resize outside the autoFitData system, hang onto the
        // specified size so we can reset to it if the data shrinks, etc
        if (!autoFitSize) {
            this._specifiedWidth = this.getWidth() + (deltaX != null ? deltaX : 0);
        }

        // Note that return value of resizeBy indicates whether the size actually changed
        var returnVal = this.invokeSuper(isc.GridBody, "resizeBy", 
                                deltaX, deltaY, animating, suppressHandleUpdate, autoFitSize);
        // we usually update _userWidth/_userHeight as part of layout.childResized to
        // store the explicit width, which then stops the member reacting to the layout's
        // subsequent resizes.
        // However, if we're autoFitting the (unfrozen) body to content, we want a
        // subsequent resize of the grid as a whole to still cause the body to expand
        // further. 
        // Therefore yank out this _userSize flag in this case.
        if (autoFitSize && deltaX != null && !this.frozen) {
            delete this._userWidth;
        }                                
        return returnVal;                                
    },
    
	// context menus (NOTE: ListGrid-level handling is identical for cell vs row context click)
    cellContextClick : function (record, rowNum, colNum) {
        var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
        return this.grid._cellContextClick(record, rowNum, gridColNum);
    },
    
    // Override _rowClick: If a record is marked as disabled this suppresses all events, but
    // if the user clicks in the "remove" field of an already removed record we actually want
    // to react to this and unmarkAsRemoved()
    _rowClick : function (rowNum, colNum) {
        if (!this.grid) return;
        var returnVal;
        var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
        
        var rec = rowNum >= 0 ? this.grid.getRecord(rowNum) : null,
            field = gridColNum >= 0 ? this.grid.getField(gridColNum) : null,
            isRemoveClick = false
        ;
        if (field && field.isRemoveField) {
            // If the user clicks inside the remove field on a group or summary row, don't fire
            // the remove-click!
            isRemoveClick = !rec || (!rec._isGroup && !rec.isGroupSummary && !rec.isGridSummary);
        }

        if (isRemoveClick) {
            this.grid.removeRecordClick(rowNum,colNum);
            returnVal = false;
        } else {
            returnVal = this.Super("_rowClick", arguments);
        }
        return returnVal;
    },

    _getCellHoverComponent : function (record, rowNum, colNum) {
        if (this.grid && isc.isA.ListGrid(this.grid)) {
            var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
            return this.grid._getCellHoverComponent(record, rowNum, gridColNum);
        }
    },

    
    // this ensures that if we're not showing any records we can still scroll the header fields
    // into view.
    expandEmptyMessageToMatchFields:true,
    applyHSpaceToEmptyMessage:true,
    
    getInnerHTML : function () {
        // call bodyDrawing on the LG if we are the primary body
        this.grid.bodyDrawing(this);
        return this.Super("getInnerHTML", arguments);
    },
    
    // ------------------------------------------------------
    //
    // PrintHTML
    // This needs some tweaking to handle the following:
    // - printHTML can be generated asynchronously in 2 ways:
    //  - if number of rows exceeds printMaxRows we use timers to break up the HTML generation
    //  - if we have embeddedComponents fetching their printHTML may also be asynchronous
    //
    // In either case, 'getTableHTML()' will be fired more than once, asynchronously.
    // In the case of async embedded component printHTML generation, this is the standard
    // mechanism - see 'gotComponentPrintHTML' in GridRenderer.
    // In the case of splitting the printing into chunks, the _printingChunk
    // flag will be set and startRow/endRow will be shifted, then getTableHTML will be called
    // on a timer, repeatedly until all the rows' HTML is generated.
    //
    // We need to fire the 'prepareForPrinting' and 'donePrinting' methods on the ListGrid
    // around each of these blocks - this is required as the ListGrid relies on the body to
    // handle generating header HTML and if there are frozen fields, HTML from the frozen
    // body, and does so by setting various flags on the GR body which'll be read by
    // getTableHTML()
    //
    
    getTablePrintHTML : function (context) {
        // context contains startRow, endRow, callback, printProperties, printWidths
        var startRow = context.startRow,
            endRow = context.endRow,
            totalRows = endRow != null ? (endRow - startRow) : this.getTotalRows(),
            maxRows = this.printMaxRows,
            printWidths = context.printWidths,
            printProps = context.printProps;
            
        var asyncPrintCallback = {
            target:this,
            methodName:"gotTablePrintHTML",
            printContext:context,
            printCallback:context.callback
        }
        
        context.callback = asyncPrintCallback;
        
        if (maxRows < totalRows) {
            this.logDebug("get table print html - breaking HTML into chunks", "printing");
            if (startRow == null) startRow = context.startRow = 0;
            if (endRow == null) endRow = context.endRow = this.getTotalRows();
            this.getPrintHTMLChunk(context);
            
            return null;
        }
        
        // No chunks - can only be asynchronous due to getTableHTML directly going asynch
        // to get embeddedComponentHTML
        var suspendPrintingContext = this.grid._prepareForPrinting(printWidths, printProps);
        var printHTML = this.getTableHTML(null, startRow, endRow, null, asyncPrintCallback);
       
        // restore settings
        this.grid._donePrinting(suspendPrintingContext);
        return printHTML;
    },
    
    gotTablePrintHTML : function (HTML, asyncCallback) {
        var callback = asyncCallback.printCallback;
        if (callback) {
            this.fireCallback(callback, "HTML,callback", [HTML,callback]);
        }
    },
    
    // This is called repeatedly, asynchronously for each "chunk"
    // The first chunk may include fetches for component tableHTML so can also be asynchonous
    // itself.
    
    getPrintHTMLChunk : function (context, returnSynchronous) {
        
        var suspendPrintingContext = this.grid._prepareForPrinting(context.printWidths);
        // printing chunk flag - used by the GR to avoid writing out the outer table tags for each
        // chunk.
        this._printingChunk = true;
        
        // Second flag to indicate we are printing chunks. This is used only by
        // gotComponentPrintHTML() to reset the _printingChunk flag before calling
        // getTableHTML
        this._gettingPrintChunkHTML = true;
        
        var startRow = context.startRow, 
            endRow = context.endRow, 
            maxRows = this.printMaxRows, 
            callback = context.callback;
    
        this.currentPrintProperties = context.printProps;
            
        if (!context.html) context.html = [];
    
        var chunkEndRow = context.chunkEndRow = Math.min(endRow, (startRow + maxRows)),
            chunkHTML = this.getTableHTML(null, startRow, chunkEndRow, null, 
                {target:this, methodName:"gotPrintChunkHTML",
                    printContext:context, printCallback:context.callback
                });
            
        // restore settings
        this.grid._donePrinting(suspendPrintingContext);
        this._printingChunk = false;
        
        // chunkHTML will only be null if getTableHTML went asynchronous - can happen on the
        // first chunk while retrieving embedded componentHTML
        if (chunkHTML != null) {
            delete this._gettingPrintChunkHTML;
            this.gotPrintChunkHTML(chunkHTML, {printContext:context});
            if (returnSynchronous) {
                return chunkHTML;
            }
        }
    },
    gotPrintChunkHTML : function (HTML, callback) {
        var context = callback.printContext,
            startRow = context.startRow,
            endRow = context.endRow,
            chunkEndRow = context.chunkEndRow,
            maxRows = this.printMaxRows,
            gotHTMLCallback = context.callback;
            
        context.html.add(HTML);        
        
        if (chunkEndRow < endRow) {
            context.startRow = chunkEndRow;
            return this.delayCall("getPrintHTMLChunk", [context], 0);
        }
    
        if (gotHTMLCallback != null) {
            var html = context.html.join(isc.emptyString);
            this.fireCallback(gotHTMLCallback, "HTML,callback", [html,gotHTMLCallback]);
        }
    },
    
    // In GridRenderer.getTableHTML(), when printing, we generate all embedded components'
    // print HTML up front, then slot it into the actual HTML for the table.
    // component printHTML may be asynchronously generated in which case this callback is
    // fired when we have the component HTML - default implementation re-runs getTableHTML
    // which now recognizes it's got component HTML and continues to get the actual table
    // HTML then fire the async callback.
    // Overridden to call 'prepareForPrinting()' on the grid and reset the '_printingChunk'
    // flag if necessary
    gotComponentPrintHTML : function (HTML, callback) {
        
        var asyncCallback = callback.context.asyncCallback,
            context = asyncCallback.printContext;
            
        var printWidths = context.printWidths;
            
        var suspendPrintingContext = this.grid._prepareForPrinting(printWidths);
        if (this._gettingPrintChunkHTML) {
            this._printingChunk = true;
        }
        
        var HTML = this.Super("gotComponentPrintHTML", arguments);
        if (this._printingChunk) delete this._printingChunk;
        
        if (HTML != null) {
            delete this._gettingPrintChunkHTML;
        } else {
            this.grid._donePrinting(suspendPrintingContext);
        }
        
    },
    
    // Row Spanning Cells
    // ----------------------------

    
    
    
    refreshCellValue : function (rowNum, colNum) {        
        var lg = this.grid;
        if (rowNum >= 0 && colNum >= 0 && lg && lg.allowRowSpanning && lg.useRowSpanStyling) 
        {
            // If we're asked to refresh a logical cell that's not rendered
            // (actually spanned by another cell) we could either refuse or refresh the
            // spaning cell. Refresh the spanning cell in case its value is calculated
            // from the cell in question.
            var startRow = this.getCellStartRow(rowNum, colNum);
            if (startRow != rowNum) {
                rowNum = startRow;
            }
        }
        isc.GridRenderer._instancePrototype.refreshCellValue.call(this, rowNum, colNum);
    },

    
    // Cell Alignment
    // ---------------------------------------------------------------------------------------
    
    // cellAlignment - override to account for the fact that with frozen fields, body 
    // colNum may be offset from ListGrid colNum
    getCellVAlign : function (record, field, rowNum, colNum) {
        if (this.grid && this.grid.getCellVAlign) {
            var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
            return this.grid.getCellVAlign(record, rowNum, gridColNum);
        }
    },
    getCellAlign : function (record, field, rowNum, colNum) {
        
        if (this.grid && this.grid.getCellAlign != null) {
            var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
            return this.grid.getCellAlign(record, rowNum, gridColNum);
            
        } else return field.cellAlign || field.align;
    },
 
    // Single Cell rows   
    // ---------------------------------------------------------------------------------------

    // if this is removed, DONTCOMBINE directive no longer needed in GridRenderer.js
    _drawRecordAsSingleCell : function (rowNum, record,c) {
        var lg = this.grid;
        if (lg.showNewRecordRow && lg._isNewRecordRow(rowNum)) return true;
        
        return isc.GridRenderer._instancePrototype.
            _drawRecordAsSingleCell.call(this, rowNum,record,c);    
        //return this.Super("_drawRecordAsSingleCell", arguments);
    },
    
    // showSingleCellCheckboxField()
    // If this record is showing a single cell value, should a checkbox field also show up next
    // to the record?
    showSingleCellCheckboxField : function (record) {
        var lg = this.grid;
        return lg && lg.showSingleCellCheckboxField(record);
    },
    
    // This method is called on records where _drawRecordAsSingleCell is true
    // returns the start/end col the single cell value should span.
    // Typically just spans all the cells we render out but if we're showing the
    // checkbox field we may want to NOT span over that field
    _getSingleCellSpan : function (record, rowNum, startCol, endCol) {
        // Span all columns if we're not showing a checkbox field
        
        if (rowNum == this._animatedShowStartRow ||
            !this.showSingleCellCheckboxField(record) || 
            (this.grid && this.grid.frozenBody != null && this.grid.frozenBody != this)) 
        {
            return [startCol,endCol];
        }
        
        
        return [Math.max(startCol, 1), endCol];
    },
    
    // Scrolling / Scroll Sync
    // ---------------------------------------------------------------------------------------

    // Have v-scrolling occur on the frozen body on mouseWheel
    // This essentially duplicates the mouseWheel handler at the Canvas level for
    // widgets with visible scrollbars.
    mouseWheel : function () {
        if (this.frozen && this.grid != null) {
            var wheelDelta = this.ns.EH.lastEvent.wheelDelta;
            var scrollTo = this.scrollTop + Math.round(wheelDelta * isc.Canvas.scrollWheelDelta);
            // Scroll the main body (we'll scroll in response to that) rather than
            // scrolling the frozen body directly.
            this.grid.body.scrollTo(null, scrollTo, "frozenMouseWheel");
            return false;
        }
        return this.Super("mouseWheel", arguments);
    },

    // Override _getDrawRows()
    // Have the frozen body rely on the unfrozen body to handle drawAhead / quickDrawAhead
    // etc and keep set of drawn rows in synch
    
    _getDrawRows : function () {
        if (this.frozen && this.grid) {
            var grid = this.grid;
            return grid.body._getDrawRows();
        }
        return this.Super("_getDrawRows", arguments);
    },

    // doneFastScrolling: ensure *both* bodies redraw without draw-ahead direction
    doneFastScrolling : function () {
        // we only expect to see this fire on the unfrozen body - the frozen body doesn't
        // show a scrollbar so won't get the thumb drag which initializes this method
        if (!this.frozen && this.grid != null && this.grid.frozenBody != null) {
            
            var redrawFrozenBody = this._appliedQuickDrawAhead;
            this.Super("doneFastScrolling", arguments);
            if (redrawFrozenBody) {
                this.grid.frozenBody._suppressDrawAheadDirection = true;
                this.grid.frozenBody.markForRedraw("Done fast scrolling on unfrozen body");
            }
        }
    },

    // observe the scroll routine of the body so we can sync up
    scrollTo : function (left, top, reason, animating) {
        if (isc._traceMarkers) arguments.__this = this;
        // Clamp the positions passed in to the edges of the viewport
        // (avoids the header from getting out of synch with the body.)
        
        if (left != null) {
            var maxScrollLeft = this.getScrollWidth() - this.getViewportWidth();
            left = Math.max(0, Math.min(maxScrollLeft, left));
        }
        if (top != null) {
            var maxScrollTop = this.getScrollHeight() - this.getViewportHeight();
            top = Math.max(0, Math.min(maxScrollTop, top));
        }            
        var lg = this.grid;
        
        
        this.invokeSuper(isc.GridBody, "scrollTo", left,top,reason,animating);

        //this.logWarn("body.scrollTo: " + this.getStackTrace());
        // dontReport when we're being called in response to bodyScrolled
        // observation!
        var dontReport = this._noScrollObservation;
        if (!dontReport) lg.bodyScrolled(left, top, this.frozen);
        
        // If the body scrolled without forcing a redraw, ensure any visible edit form 
        // items are notified that they have moved.
        
        if (!this.isDirty() && lg._editorShowing) {
            lg._editRowForm.itemsMoved();
        }

    },
    
    // Embedded Components
    // ---------------------------------------------------------------------------------------

    // embedded components can be per row or per cell.
    // When per-cell the GR APIs act by colNum only, not by field name.
    // However for us to handle field reorder, show/hide, etc it's useful to hang fieldName
    // onto the embeddedComponents as well
    addEmbeddedComponent : function (component, record, rowNum, colNum, position) {
        var comp = this.invokeSuper(isc.GridBody, "addEmbeddedComponent", component, record, 
                                    rowNum, colNum, position);
        if (component._currentColNum != null && component._currentColNum != -1 && this.grid) {
            var grid = this.grid,
                colNum = component._currentColNum,
                fieldName = grid.getFieldName(colNum);
            component._currentFieldName = this.grid.getFieldName(colNum);

            // set up a map of embedded components per column (fieldName)
            // This will make lookup quicker.
            if (grid._columnComponentsMap == null) {
                grid._columnComponentsMap = {};
            }
            if (grid._columnComponentsMap[fieldName] == null) {
                grid._columnComponentsMap[fieldName] = {};
            }
            grid._columnComponentsMap[fieldName][component.getID()] = true;
            // skip "%" sized components since they should never expand a field.
            if (component._percent_width == null) {
                grid._fieldComponentWidthsChanged(fieldName);
            }
        }
        
        return component;
    },
    removeEmbeddedComponent : function (record, component, suppressRedraw) {
        var grid = this.grid;
        if (grid) {
            var fieldName = component._currentFieldName;
            if (fieldName != null) {
                if (grid._columnComponentsMap && grid._columnComponentsMap[fieldName]) {
                    delete grid._columnComponentsMap[fieldName][component.getID()];
                }
                if (component._percent_width == null) {
                    grid._fieldComponentWidthsChanged(fieldName);
                }
            }
        }
        this.invokeSuper(isc.GridBody, "removeEmbeddedComponent", record, component, suppressRedraw);
    },

    
    // Override _getExtraEmbeddedComponentHeight() / upateHeightForEmbeddedComponents to
    // respect listGrid.recordComponentHeight if specified, even if there are no
    // embedded components for this record.
    
    updateHeightForEmbeddedComponents : function (record, rowNum, height) {
  
        if (record && !this.grid._hasEmbeddedComponents(record) && this.grid.showRecordComponents
            && this.grid.recordComponentHeight != null) 
        {
            // Reimplementing the superClass version, except that this logic is running even
            // when there are no embeddedComponents on the row.
            var details = this._getExtraEmbeddedComponentHeight(record, rowNum);
            if (details.allWithin && details.extraHeight > 0) {
                height = Math.max(height,details.extraHeight);
                //this.logWarn("in updateHeightForEmbeddedComponents ("+this.grid+"): details are "+isc.echoAll(details)+"\nheight is "+height);
            } else {
                height += details.extraHeight;
                //this.logWarn("in updateHeightForEmbeddedComponents ("+this.grid+"): details are "+isc.echoAll(details)+"\nheight is "+height);
            }
            
            return height;
        }
        
        return this.invokeSuper(isc.GridBody, "updateHeightForEmbeddedComponents", record, rowNum, height);
    },
    
    _getExtraEmbeddedComponentHeight : function (record, rowNum) {
        var heightConfig = this.invokeSuper(isc.GridBody, "_getExtraEmbeddedComponentHeight",
                                        record, rowNum);
        if (this.grid.showRecordComponents && this.grid.recordComponentHeight != null) {
            heightConfig.extraHeight = Math.max(heightConfig.extraHeight,
                                            this.grid.recordComponentHeight);
        }
        return heightConfig;
    },
    _writeEmbeddedComponentSpacer : function (record) {
        if (record && this.grid && this.grid.showRecordComponents 
            && this.grid.recordComponentHeight != null)
        {
            return true;
        }
        return this.invokeSuper(isc.GridBody, "_writeEmbeddedComponentSpacer", record);
    },
    
    _placeEmbeddedComponents : function () {
        
        if (this.grid && this.grid._autoFittingFields) {
            return;
        }
        return this.Super("_placeEmbeddedComponents", arguments);
    },
    
    getAvgRowHeight : function () {
        if (this.grid) return this.grid.getAvgRowHeight(this);
        return this.Super("getAvgRowHeight", arguments);
    },
    
    // override shouldShowAllColumns() - we can avoid showing all columns if row height
    // is variable *only* because of an expansion component expanding the entire row since
    // the heights won't vary per-cell.
    shouldShowAllColumns : function () {
        if (this.showAllColumns) {
            return true;
        }
        if (!this.fixedRowHeights && !this.showAllRows) {
            if (this.grid.canExpandRecords && this.grid._specifiedFixedRecordHeights) {
                return false;
            }
            return true;
        }
        if (this.overflow == isc.Canvas.VISIBLE) {
            return true;
        }
        return false;
         
    },

    // Editing
    // ---------------------------------------------------------------------------------------

    //> @method listGrid.markForRedraw()
    // @include canvas.markForRedraw
    // @visibility external
    //<
    
    // Redraw overridden:
	// - Update the editRow form items (we don't create more items than we need when
	//   rendering incrementally)
	// - Update the values in the edit row form.
    redraw : function (reason,b,c,d) {
        // flag to note we're redrawing - this is used by getDrawnFields() 
        this._redrawing = true;
        
        // If alwaysShowEditors is marked as true, but editorShowing is false it implies our
        // attempt to start editing on draw() failed - presumably there were no
        // editable cells in view.
        // See if we can start editing now in this case
        var lg = this.grid;
        if (lg.alwaysShowEditors && !lg._editorShowing) {
            lg.startEditing(null,null,true,null,true);
        }
        
        
        var editForm = lg._editRowForm,
            editing = lg._editorShowing,
            editColNum, editRowNum, editRecord,
            completeWidths,
            fieldsToRemove;
        
        // If the grid is showing inactive Editor HTML for any cells, we'll clear it
        // (and potentially regenerate it) as part of redraw(). Notify the grid so it can clear
        // up inactive contexts stored on the edit form items
        
        lg._clearingInactiveEditorHTML();
        
        // if body redraw came from data change, folder opening, or resizing of content,
        // it's likely to introduce a v-scrollbar.
        // If leaveScrollbarGap is false, call '_updateFieldWidths()' before the redraw occurs so
        // we leave a gap for the v-scrollbar, rather than redrawing with both V and H scrollbar,
        // then resizing the fields and redrawing without an H-scrollbar.
        if (!lg.leaveScrollbarGap && lg.predictScrollbarGap && (this.overflow == isc.Canvas.AUTO)) {
            var vScrollOn = this.vscrollOn,
                
                vScrollWillBeOn = !lg.isEmpty() && 
                                  (lg.getTotalRows() * lg.cellHeight)  > this.getInnerHeight();
            
            if (vScrollOn != vScrollWillBeOn) {
                // ensure we don't try to recalculate field widths a second time by clearing
                // the _fieldWidthsDirty flag
                delete this._fieldWidthsDirty;
                lg._updateFieldWidths("body redrawing with changed vertical scroll-state");
                
            }
        } 

        var nativeRowRefocus = false;

        if (editing) {
            
            this.logInfo("redraw with editors showing, editForm.hasFocus: " +
                         editForm.hasFocus, "gridEdit");
            editColNum = lg.getEditCol();                             
            
            // See comments near _storeFocusForRedraw() for how edit form item focus is handled
            // on redraw
            this._storeFocusForRedraw();

            // This will add the new edit items corresponding to the newly displayed fields
            // and return the items that need to be removed (after the body is actually redrawn,
            // which will hide them)
            // It also fires the "redrawing" notification on any items that are about to be redrawn
            fieldsToRemove = this._updateEditItems();
        } else if (isc.screenReader) {
            nativeRowRefocus = this.hasFocus || (this.editForm && this.editForm.hasFocus())
        }   
        
        // refresh field widths if necessary
        if (this._fieldWidthsDirty != null) {
            
            var fwReason = this._fieldWidthsDirty;
            delete this._fieldWidthsDirty;
            lg._updateFieldWidths(fwReason);
        }
        // store the new drawArea
        var newDrawArea = this.getDrawArea();
        
        
        var grid = this.grid,
            drawArea = this._oldDrawArea;

        if (!drawArea) drawArea = this._oldDrawArea = [0,0,0,0];

        var grid = this.grid,
            firstRecord = grid.getRecord(newDrawArea[0]),
            lastRecord = grid.getRecord(newDrawArea[1]),
            dataPresent = (firstRecord && firstRecord != Array.LOADING) 
                && (lastRecord && lastRecord != Array.LOADING);
        ;
        if (dataPresent && !drawArea.equals(newDrawArea)) 
        {
            // the old and new drawAreas differ and the extents of the new data are present - 
            // fire the notification method and update the stored _oldDrawArea
            
            if (!this.frozen) { 
                grid._drawAreaChanged(drawArea[0], drawArea[1], drawArea[2], drawArea[3], this);
                this._oldDrawArea = newDrawArea;
            }
        }
        
        // Always update all recordComponents on redraw().
        // don't rely on the draw area changing since we may be showing the same set of
        // rowNum/colNums but having underlying data or field meaning changes.
        // Note: updateRecordComponents() updates components in frozen and unfrozen bodies.
        // If this is a redraw of the frozen body, don't call updateRecordComponents() if
        // the redraw was tripped by scrolling or data change as in this case we'll also
        // get a redraw of the unfrozen body which can handle updating the RC's.
        // (DO still call the method in other cases as it may imply the fields in the frozen
        // body have changed, etc).
        if (!(this.frozen && this._suppressRecordComponentsUpdate)) {
            grid.updateRecordComponents();
        }
        this._suppressRecordComponentsUpdate = false;

        
        delete this._drawnEditItems;

        this.invokeSuper(isc.GridBody, "redraw", reason,b,c,d);

        // clear the "redrawing" flag since the HTML is now up to date
        delete this._redrawing;

        if (editing) {
            // Remove the items that correspond to fields that are no longer rendered in the DOM
            if (fieldsToRemove != null && fieldsToRemove.length > 0) {
                editForm.removeItems(fieldsToRemove);
            }
            
            // Fire the method to notify form items that they have been drawn() / redrawn()
            // or cleared()
            
            lg._editItemsDrawingNotification(null, true, this);
            
            /*
            var itemColArray = [],
                items = lg._editRowForm.getItems();
            for (var i =0; i < items.length; i++) {
                itemColArray.add(items[i].colNum + " - " + items[i].name);
            }
            this.logWarn("After redraw - edit form covers these cols:" + itemColArray);
            */

        	
            lg.updateEditRow(lg.getEditRow());           
            
            // If the editRowForm is currently marked as having focus, or we took focus from it
            // before a redraw and no other widget has subsequently picked up focus, restore focus
            // to it
            // This catches both a simple redraw and the case where the user scrolls the edit item
            // out of view, it is cleared due to incremental rendering, then scrolls back into view
            // and it gets redrawn
        	
            if (editForm.hasFocus || 
                (this._editorSelection && isc.EH.getFocusCanvas() == null)) 
            {
                this._restoreFocusAfterRedraw(editColNum);
            } else {
                delete this._editorSelection;
            }
            
        } else {
            if (editForm != null) {       
            // notify the form that it's items have been cleared() (will no-op if they're
            // not currently drawn)
            lg._editItemsDrawingNotification(null, null, this);
            
            
            }
            if (isc.screenReader) {
            // _nativeFocusRow was remembered last time putNativeFocusInRow was called
            
                this._putNativeFocusInRow(this.getNativeFocusRow(), !nativeRowRefocus);
            }
        }

    },
    
    redrawOnScroll : function (immediate) {
        if (this.frozen) this._suppressRecordComponentsUpdate = true;
        return this.Super("redrawOnScroll", arguments);
    },

    
    // force redraw on setDisabled() if we're showing an edit form to ensure we 
    // redraw the items in enabled/disabled state
    setHandleDisabled : function (disabled) {
        var lg = this.grid;
        if (this.isDrawn() && lg && lg._editorShowing) {
            this.markForRedraw("Grid body disabled while editing");
        }
        return this.Super("setHandleDisabled", arguments);
    },
    
    // Add edit items corresponding to newly displayed fields (displayed due to incremental
    // rendering)
    // If any fields are to be hidden, do not remove these here, but return them in an array so
    // they can be removed from the form after the redraw completes
    // Note that the order of items in the form will not match the order of fields necessarily - 
    // acceptable since developers will be interacting with the items' colNum attribute rather than
    // index in the edit form fields array.
    
    _updateEditItems : function () {
    
        // We keep the set of items in the editForm in synch with the set of 
        // visible columns for performance.
        // Determine which items need to be created or removed here.
        var lg = this.grid, editForm = lg.getEditForm(),
            fieldsToRemove = [],
            editItems = editForm.getItems();
        if (!lg.editByCell) {

            // set up the vars used in creating form items
            var editRowNum = lg.getEditRow(),
                editRecord = lg.getRecord(editRowNum),
                
                completeWidths = lg.getEditFormItemFieldWidths(editRecord);
         
            // Determine what fields are rendered into the body
            // If we have frozen columns, we will always be showing them, in addition to whatever
            // fields are visible
            var editItems = editForm.getItems(),
                itemNames = editItems.getProperty(this.fieldIdProperty),
                fields = lg.getDrawnFields(),
                fieldNames = fields.getProperty(this.fieldIdProperty);
            
            // minor optimization - if possible, avoid iterating through both arrays
            var lengthsMatch = editItems.length == fields.length,
                changed = false;
                
            // fields that are no longer drawn should be removed
            for (var i = 0; i < editItems.length; i++) {
                // don't actually remove the items until they have been removed from the DOM via
                // redraw
                var index = fieldNames.indexOf(itemNames[i]);
                if (index == -1) {
                    changed = true;
                    fieldsToRemove.add(editItems[i]);
                } else {
                    // If we're keeping the item, just update width, and notify the item we're
                    // about to redraw
                    editItems[i].width = completeWidths[editItems[i].colNum];
                    editItems[i]._size = null;
                    editItems[i].redrawing();
                }
            }
            // newly rendered fields should be added
            if (!lengthsMatch || changed) {      
                var editedVals = lg.getEditedRecord(editRowNum, 0);
                for (var i = 0; i < fields.length; i++) {                    
                    if (!itemNames.contains(fieldNames[i])) {

                        var colNum = lg.fields.indexOf(fields[i]);
                        var item = lg.getEditItem(
                                        fields[i],
                                        editRecord, editedVals, editRowNum,
                                        colNum, completeWidths[colNum]
                                   );
                        editForm.addItem(item);
                    }
                }
            }
        } 
        // if editByCell is true this is not necessary - we consistantly have the editForm contain
        // only the necessary cell
        return fieldsToRemove;
    },  

    // _storeFocusForRedraw()
    // called when the edit form is showing and the body is being redrawn.
    // remember the current focus state / selection of the edit form so we can reset it after
    // redrawing the item in the DOM
    // blur the item (suppressing the handler if the item will be refocused after redraw) 


    
    
        
    _storeFocusForRedraw : function () {

        var lg = this.grid,
            editForm = lg.getEditForm(),
            editColNum = lg.getEditCol();
        if (editForm.hasFocus) {
            var focusItem = editForm.getFocusSubItem();
            if (focusItem) {
                focusItem.updateValue();
                var origFocusItem = focusItem;
                
                // We may be focused in a sub item, in which case we need to use the
                // parentItem to get the field name wrt our fields array
                while (focusItem.parentItem != null) {
                    focusItem = focusItem.parentItem;
                }
                
                // blur the focus item before removing it from the DOM.
                // If canEditCell for the current focus item returns false, we will
                // not redisplay it at the end of this method, so allow it to fire the
                // standard blur-handler
                
                if (!lg.canEditCell(focusItem.rowNum, focusItem.colNum) ||
                    editColNum != focusItem.colNum) {
                    
                    editForm.blur();
                } else {
                    
                    if (focusItem.hasFocus) {
                        // remember the current selection, so we can reset it after the redraw
                        // and refocus. [will have no effect if the item is not a text-item]
                        focusItem.rememberSelection();
                        this._editorSelection = 
                            [focusItem._lastSelectionStart, focusItem._lastSelectionEnd];
                    }
                    editForm._blurFocusItemWithoutHandler();

                }
                
            }
        }
        
        editForm._setValuesPending = true;        
    },
    
    // If the editForm is visible during a body redraw() this method ensures that after the
    // redraw completes, and the form items are present in the DOM, focus / selection is restored
    // to whatever it was before the redraw
    _restoreFocusAfterRedraw : function (editColNum) {
        var lg = this.grid,
            editForm = lg.getEditForm(),
            editItem = editForm.getItem(lg.getEditorName(lg.getEditRow(), editColNum));
        if (editItem != null && editItem.isDrawn()) {
            var scrollLeft = lg.body.getScrollLeft(), 
                scrollTop = lg.body.getScrollTop(),
                viewportWidth = lg.body.getViewportWidth(), 
                viewportHeight = lg.body.getViewportHeight(),
                rect = editItem.getRect(),
                // If we are partially out of the viewport, don't put focus into item -
                // forces a native scroll which can interfere with user scrolling.
                // Note: partially out of viewport actually could be ok for text items
                // where focus will only cause a scroll if the actual text is offscreen.
                
                outOfViewport = rect[0] < scrollLeft ||
                                rect[1] < scrollTop ||
                                rect[0] + rect[2] > (scrollLeft + viewportWidth) ||
                                rect[1] + rect[3] > (scrollTop + viewportHeight);
                                
            if (!outOfViewport) {
                // Avoid selecting the focused value - we don't want rapid keypresses
                // to kill what was previously entered  
                editForm._focusInItemWithoutHandler(editItem);
                // Reset the selection / text insertion point to whatever was 
                // remembered before the redraw.
                
                if (this._editorSelection && this._editorSelection[0] != null) {
                    editItem.setSelectionRange(this._editorSelection[0], this._editorSelection[1]);
                }
                // clear up the _editorSelection flag so we don't try to restore focus again on
                // scroll
                delete this._editorSelection;
            }
        }
    },
    // If programmatic 'focus' from "syntheticTabIndex" (IE - tab keypress when the clickMask is up)
    // is called on the ListGrid body and the editor is showing, it
    // makes sense to focus in the edit form.
    focusAtEnd : function (start) {
        var grid = this.grid,
            editForm = grid ? grid.getEditForm() : null;
        if (editForm) {
            editForm.focusAtEnd(start);
        } else {
            return this.Super("focusAtEnd", arguments);
        }
    },
    
    // Override cellMove: We need to be able to show validation error HTML
    cellMove : function (record,rowNum,colNum) {
       
        var nativeTarget = isc.EH.lastEvent ? isc.EH.lastEvent.nativeTarget : null;
        if (nativeTarget && nativeTarget.getAttribute != null &&
            (nativeTarget.getAttribute("isErrorIcon") == "true")) 
        {
            // adjust for frozen fields
            colNum = this.grid.getFieldNumFromLocal(colNum, this);
            if (this.grid._overErrorIcon != null) {
                var lastRow = this.grid._overErrorIcon[0],
                    lastCol = this.grid._overErrorIcon[1];
                if (lastRow != rowNum || lastCol != colNum) {
                    this.grid._handleErrorIconOut();
                }
            }
            
            if (this.grid._overErrorIcon == null) {
                this.grid._handleErrorIconOver(rowNum,colNum);
            }
        } else {
            if (this.grid._overErrorIcon != null) {
                this.grid._handleErrorIconOut();
            }
        }
    },

    // Override shouldShowRollOver to avoid styling the current edit cell with the over
    // style.
    // This avoids the issue where if you roll over the edit form items, the rollover style
    // would flash off as the body receives a mouseout (looks very weird).        
    shouldShowRollOver : function (rowNum, colNum,a,b) {

        //if (!this.invokeSuper(isc.GridBody, "shouldShowRollOver", rowNum,colNum,a,b)) return false;
        
        if (!this.grid.showRollOver || this._rowAnimationInfo) return false;
 
        // don't show rollover for the edit row if editing the whole row
        var lg = this.grid;
        if (lg._editorShowing && !lg.editByCell && rowNum == lg._editRowNum) return false;
        
        return true;
    },

    updateRollOver : function (rowNum, colNum, movingToNewCell) {

        var lg = this.grid;
        
        if (lg.showRollOverCanvas) {            
            // movingToNewCell param passed when the user rolled off one cell and over another
            // and this method is being called to clear the first cell hilight.
            // we can no-op in this case since we'll update the rollOverCanvas on the subsequent
            // call to this method, and that will avoid a clear/draw cycle (and flash)   
            
            if (!movingToNewCell) {
                var leaving = !(this.lastOverRow == rowNum && this.lastOverCol == colNum);
                lg.updateRollOverCanvas(rowNum, colNum, leaving);
            }
            // no support for frozen body / rollOverCanvas yet

        }

        this.setRowStyle(rowNum, null, this.canSelectCells || this.useCellRollOvers ? 
                         colNum : null);

        // frozen fields: if there is another body, force its rollover row to match and
        // update it
        var otherBody = (this == lg.body ? lg.frozenBody : lg.body);
        if (otherBody && !this.canSelectCells) {
            otherBody.lastOverRow = this.lastOverRow;
            otherBody.lastOverCol = this.lastOverCol;
            otherBody.setRowStyle(rowNum, null, (this.useCellRollOvers ? colNum : null));
        }
    },

    _selectCellOnMouseDown : function (record, rowNum, colNum) {
        // remember the last cell clicked (used for keyboard navigation)
        // (Note: we use the same cell used for selection rather than the actual position
        // of the event as that's where the user will see the visual indication)
        this._lastSelectedRow = rowNum;
        this._lastSelectedCol = colNum;

        this.grid._lastSelectedBody = this;
    
        //this.logWarn("mouseDown at: " + [rowNum, colNum]);

        if (this.useRowSpanStyling) {
            // rowSpan-sensitive selection 
            var gridSelection = this.grid.selection;

            // selected cells are designed by the coordinates where spanning starts.
            // NOTE: this means that calling isSelected() with return false for any cell
            // coordinates where no cell exists in the DOM (because it was spanned over by a
            // rowSpanning cell in a previous row).  This is true even when the spanning cell
            // that eliminated that DOM cell is in facet selected.
            var startRow = this.getCellStartRow(rowNum, colNum);

            // for ctrl-click just select/deselect the clicked cell
            if (isc.EH.modifierKeyDown()) {
                var gridColumn = this.grid.getFieldNumFromLocal(colNum, this);
                gridSelection.selectOnMouseDown(this, startRow, gridColumn);
                return;
            }

            // deselect everything - shift modifiers, dragging or other cases not supported in
            // this mode
            gridSelection.deselectAll();

            // select all cells to the right that are partially or wholly spanned by the
            // clicked cell

            var gridBody, startCol, span,
                mode = this.rowSpanSelectionMode;

            if (mode == "forward" || mode == null) {
                // field num where the click landed, in terms of listGrid.fields
                startCol = this.grid.getFieldNumFromLocal(colNum, this);
                // cells spanned by the clicked cell
                span = this.getCellRowSpan(startRow, colNum);
            } else if (mode == "outerSpan") {
                // optionally, start the selection from the span of the first column, which
                // creates a selection behavior similar to the default row-level selection
                // behavior, based on the concept of the span in the first column defining the
                // "row"
                startCol = 0;
                gridBody = this.grid.getFieldBody(0);
                startRow = gridBody.getCellStartRow(rowNum, 0);
                span = gridBody.getCellRowSpan(startRow, 0);
            } else if (mode == "both") {
                // use the span of the clicked cell
                span = this.getCellRowSpan(startRow, colNum);
                // but go through all columns
                startCol = 0;
            } else {  
                this.logWarn("unrecognized rowSpanSelectionMode: '" + mode + "'");
                return;
            }

            //this.logWarn("start cell: " + [startRow, startCol] +
            //             " spans: " + span);
            // for each column to the right of the starting column
            for (var column = startCol; column < this.grid.fields.length; column++) {
                // get the body that contains the field and the field's index within that body
                var bodyToSelect = this.grid.getFieldBody(column),
                    localFieldNum = this.grid.getLocalFieldNum(column);

                // for each row spanned by the starting cell
                for (var i = startRow; i < startRow+span; i++) {
                    // when selecting, select the coordinates of the beginning of the span
                    var cellStartRow = bodyToSelect.getCellStartRow(i, localFieldNum);
                    gridSelection.selectCell(cellStartRow, column);
                    // and skip past cells that were spanned over
                    i += (bodyToSelect.getCellRowSpan(cellStartRow, localFieldNum) - 1);
                }
            }
            return;
        }
        
        // if we're in the body, select rows or cells
        this.selection.selectOnMouseDown(this, rowNum, colNum);
    },
    
    // override selectOnMouseDown/Up to disable selection when a row is clicked anywhere
    // besides the checkbox when selectionAppearance is checkbox.
    selectOnMouseDown : function (record, rowNum, colNum) {
        var shouldSelect = true,
            selApp = this.grid.selectionAppearance,
            cbSel = (selApp == "checkbox");
        if (cbSel) {
            // if frozen fields are showing, the cb field will show up in the frozen body!
            if ((this.grid.frozenFields != null && this.grid.frozenBody != this) || 
                (this.grid.getCheckboxFieldPosition() != colNum)) 
            {
                shouldSelect = false;
            }
        }
        
        if (shouldSelect) {
            if (this.canSelectCells) this._selectCellOnMouseDown(record, rowNum, colNum);
            else this.invokeSuper(isc.GridBody, "selectOnMouseDown", record, rowNum, colNum);
        }

        if (isc.screenReader) {
            this._putNativeFocusInRow(rowNum);
        }
    },
    
    // Override mouseUp.
    // If the mouseUp occurred over a CanvasItem, ignore it.
    
    mouseUp : function () {
        var target = isc.EH.getTarget();
        
        if (this.grid && target != null && target != this && this.grid._editorShowing) {
            var editForm = this.grid.getEditForm();
            while (target != this && target != null && target != editForm) {
                if (target.canvasItem && editForm.items.contains(target.canvasItem)) {
                    return;
                }
                target = isc.isA.FormItem(target) ? target.containerWidget : target.parentElement;
            }
        }
        return this.Super("mouseUp", arguments);
    },
    
    selectOnMouseUp : function (record, rowNum, colNum) {
        var cbColNum = this.grid.getCheckboxFieldPosition(),
            selApp = this.grid.selectionAppearance;
        if (selApp != "checkbox" || (selApp == "checkbox" && cbColNum == colNum)) {
            this.invokeSuper(isc.GridBody, "selectOnMouseUp", record, rowNum, colNum);    
        }
    },
    
    // Override handleSelectionChanged() to fire our viewStateChanged method
    handleSelectionChanged : function (record,state) {
        if (this.grid.suppressSelectionChanged) return;
        var returnVal = this.Super("handleSelectionChanged", arguments);
        this.grid.handleViewStateChanged();
        return returnVal;
    },

    setSelection : function (selection) {                            
        this.clearSelection();
        this.Super("setSelection", arguments);
    },

    clearSelection : function () {
        var selection = this.selection;
        this.Super("clearSelection", arguments);        
        // if selection not inherented from ListGrid, destroy it
        if (selection && selection.isA("DependentCellSelection")) {
            selection.destroy();
        }
    },

    _setSeparateCellSelection : function (selection, firstCol) {
        this.clearSelection();
        if (selection) {
            this.selection = selection.getDependentCellSelection(this.fields.length);
            this.selection._updateDependency(firstCol);
            this.observe(this.selection, "selectionChanged",
                         "observer._cellSelectionChanged(observed.changedCells)");
        }
    },

    // When refreshing cellStyle, notify our edit items that the underlying cell style changed
    // so they can update if necessary
    _updateCellStyle : function (record, rowNum, colNum, cell, className, a,b,c) {
        this.invokeSuper(isc.GridBody, "_updateCellStyle", record, rowNum,colNum,cell,className,a,b,c);
        var lg = this.grid;        
        if (lg && lg.getEditRow() == rowNum) {
            var fieldName = lg.getFieldName(lg.getFieldNumFromLocal(colNum, this)),
                form = lg.getEditForm(),
                item = form ? form.getItem(fieldName) : null;                
            if (item && item.gridCellStyleChanged) {
                if (className == null) className = this.getCellStyle(record,rowNum,colNum);
                item.gridCellStyleChanged(record, rowNum, colNum, className);
            }
        }
        
    },
    
    // hovers: override getHoverTarget to return a pointer to our grid - this allows 
    // the developer to call 'updateHover' directly on the grid.
    getHoverTarget : function () {
        return this.grid;
    },

	// direct keyPresses to the ListGrid as a whole to implement arrow navigation,
	// selection, etc
	
    keyPress : function (event, eventInfo) {
        return this.grid.bodyKeyPress(event, eventInfo);
    },
    
    // getters for the current keyboard focus row for key-events.
    
    getFocusRow : function () {
        return this.grid.getFocusRow();
    },
    getFocusCol : function () {
        var colNum = this.grid._getKeyboardClickNum();
        return this.grid.getLocalFieldNum(colNum);
    },
    
	// Override _focusChanged to implement 'editOnFocus' - start editing the first
	// editable cell if appropriate.
	// See comments in 'editOnFocus' jsdoc comment for details of how this should work.
    _focusChanged : function (hasFocus) {
        
    	// use the Super implementation to set up this.hasFocus BEFORE we further 
    	// manipulate focus due to editing.
        var returnVal = this.Super("_focusChanged", arguments);

        var lastEvent = isc.EH.lastEvent;

        // if we're acquiring focus because we're in the middle of a click sequence on the body, 
        // the mouse handlers will correctly start editing or place focus on whatever row was hit, and we
        // should do nothing
        if (lastEvent.target == this &&
              (lastEvent.eventType == isc.EH.MOUSE_DOWN ||
               lastEvent.eventType == isc.EH.MOUSE_UP ||
               lastEvent.eventType == isc.EH.CLICK ||
               lastEvent.eventType == isc.EH.DOUBLE_CLICK)) return returnVal;
    
        // otherwise, entering focus due to a key event (tab, shift-tab) or something else (programmatic
        // including re-focus due to clickMask blur).
        var editCell,
            parent = this.grid;
        if (hasFocus && parent.isEditable()) {      
        
        	// editOnFocus enabled, but not currently editing
            if (parent.editOnFocus && parent.isEditable() &&     
                parent.getEditRow() == null)
            {
                if (this.logIsInfoEnabled("gridEdit")) {
            	    this.logInfo("Editing on focus: eventType: " + lastEvent.eventType +
            	                 ", lastTarget " + lastEvent.target, "gridEdit");
                }
            
            	// If we're explicitly suppressing edit on focus, don't start editing.
                if (parent._suppressEditOnFocus) {
                    delete parent._suppressEditOnFocus;
                } else {
                    // this'll run through every cell in every record until it finds one that's
                    // editable
                    editCell = parent.findNextEditCell(0,0,true,true);

                    if (editCell != null) 
                        parent.handleEditCellEvent(editCell[0], editCell[1], isc.ListGrid.FOCUS);
                }
            }
        }
          
        // In screenReader mode, if focus is moving into the grid but we're not going into editing mode,
        // put focus onto the row element rather than onto the GR body.
        
        if (isc.screenReader) {
            if (hasFocus) {
                if (editCell == null) {
                    var lastEvent = isc.EH.lastEvent,
                        eventType = lastEvent.eventType;
        
                    // find the last hilited row if there is one
                    var rowNum = this.getNativeFocusRow();
                    
                    //this.logWarn("focus entering body - focusing in native row: " + rowNum +
                    //             ", focus row was: " + parent.getFocusRow());
                    this._putNativeFocusInRow(rowNum); 
                }
            } else {
                parent.clearLastHilite();
            }
        }

        return returnVal;
    },
    
    // override putNativeFocusInRow to ensure we hilight the focus row
    _putNativeFocusInRow : function (rowNum, suppressFocus) {
        var parent = this.grid;
        // if suppressFocus is passed, we're not actually focussing into the row so don't 
        // hilite the row.
        if (parent && parent.hiliteOnNativeRowFocus && !suppressFocus) parent._hiliteRecord(rowNum);
        return this.Super("_putNativeFocusInRow", arguments);
    },

    // override updateRowSelection to update selectionCanvas if necessary
    updateRowSelection : function (rowNum) {
        var lg = this.grid;
        if (!lg) return;
        
        
        
        if (lg.showSelectionCanvas) lg.updateSelectionCanvas();
        if (lg._dontRefreshSelection) {        
            return;
        }
        
        this.invokeSuper(isc.GridBody, "updateRowSelection", rowNum);

        if (isc.Canvas.ariaEnabled() && lg.selection) {
            this.setRowAriaState(rowNum, "selected", lg.selection.isSelected(lg.getRecord(rowNum)));
        }
        
        // with selectionAppearance:"checkbox", detect all rows selected and update checkbox in
        // header
        if (lg.getCurrentCheckboxField() != null) {
            var cellNum = lg.getCheckboxFieldPosition();
            if (lg) {
                lg.refreshCell(rowNum, cellNum);
                lg.updateCheckboxHeaderState();
            }
            
        } else if (lg.getTreeFieldNum && lg.selectionAppearance == "checkbox") {
            // in the TreeGrid, refresh the tree cell because that's where the checkbox is shown
            var treeCellNum = lg.getTreeFieldNum();
            lg.refreshCell(rowNum, treeCellNum);
        }
    },
    
    // ditto with _cellSelectionChanged
    _cellSelectionChanged : function (cellList,b,c,d) {
        var lg = this.grid;     
        if (lg && lg.showSelectionCanvas) lg.updateSelectionCanvas();
        return this.invokeSuper(isc.GridBody, "_cellSelectionChanged", cellList, b,c,d);
    },

    // remove any dynamic references that point to us if we're being destroyed
    destroy : function () {
        var grid = this.grid;
        if (this == grid._lastSelectedBody)       grid._lastSelectedBody       = null;
        if (this == grid._lastKeyboardHiliteBody) grid._lastKeyboardHiliteBody = null;
        this.Super("destroy", arguments);
    },
    
    // Embedded components
    // -----------------------
    
    // animateShow selectionCanvas / rollOverCanvas if appropriate
    shouldAnimateEmbeddedComponent : function (component) {
        var grid = this.grid;
        if (component == grid.selectionCanvas) return grid.animateSelection;
        if (component == grid.selectionUnderCanvas) return grid.animateSelectionUnder;
        if (component == grid.rollOverCanvas) return grid.animateRollOver;
        if (component == grid.rollUnderCanvas) return grid.animateRollUnder;
            
        return false;
    },
    
    
    _handleEmbeddedComponentResize : function (component, deltaX, deltaY) {
        this.Super("_handleEmbeddedComponentResize", arguments);
        
        // Notify the grid - allows us to update the other body if we're showing
        // both a frozen and an unfrozen body
        this.grid._handleEmbeddedComponentResize(this, component, deltaX, deltaY);
    },
    
    // Override draw() to scroll to the appropriate cell if 'scrollCellIntoView' was called
	// before the body was drawn/created
	// Also update the edit form item rows if we're already editing.
    draw : function (a,b,c,d) {
        var lg = this.grid;
                
        if (lg.getEditRow() != null) {
            
            var rowNum = lg.getEditRow(),
                record = lg.getRecord(rowNum),
                fieldNum = lg.getEditCol(),
                form = lg._editRowForm,
                items = lg.getEditRowItems(record, rowNum, fieldNum, lg.editByCell),
                liveItems = form.getItems();
               
            var setItems = liveItems == null || items.length != liveItems.length;
            if (!setItems) {
                var liveItemNames = liveItems.getProperty("name");
                for (var i = 0; i < items.length; i++) {
                    if (!liveItemNames.contains(items[i].name)) {
                        setItems = true;
                        break;
                    }
                }
            }
            if (setItems) {
                this.logDebug("calling setItems on form from body draw","gridEdit");
                form.setItems(items);
            } else {
                this.logDebug("Skipping setItems() on form from body draw", "gridEdit");
            }
            
            
            form._setValuesPending = true;
            
        }

        
        delete this._drawnEditItems;
        
        this.invokeSuper(isc.GridBody, "draw", a,b,c,d);
        
        // If we are showing any edit form items, notify them that they have been written
        // into the DOM.
        
        if (lg._editRowForm) {
            lg._editItemsDrawingNotification(null, null, this);
        }
        // Tell the form to update its values (setItemValues())
        // (do this after the items have been notified that they're drawn to ensure items'
        // element values are set)
        lg.updateEditRow(lg.getEditRow());

        if (lg._scrollCell != null) {
            var scrollRowNum = isc.isAn.Array(lg._scrollCell) ? lg._scrollCell[0] : lg._scrollCell,
                scrollColNum = isc.isAn.Array(lg._scrollCell) ? lg._scrollCell[1] || 0 : 0
            ;
            lg.scrollToCell(scrollRowNum, scrollColNum);
            delete lg._scrollCell;
        }

        // Call 'updateRecordComponents()' on initial draw to set up recordComponents
        // If this is a ResultSet rather than an array, the updateRecordComponents method
        // will be able to skip all records and we'll render out the components on redraw.
        this.grid.updateRecordComponents();

        if (!this._updatingExpansionComponents) this.grid.updateExpansionComponents();
    },

	// rerun ListGrid-level layout if the body's scrolling state changes, to allow sizing
	// the header appropriately
    layoutChildren : function (reason,a,b,c) {
        this.invokeSuper(isc.GridBody, "layoutChildren", reason,a,b,c);
        // This method may be called with "scrolling state change" when a bodyLayout is 
        // currently undrawn but drawing out its children - we've seen this in FF 3
        // In this case bail now since if _updateFieldWidths() is fired on an undrawn body it
        // bails, leaving the body mis sized
        
        if (!this.isDrawn() || (this.grid.frozenFields && !this.grid.bodyLayout.isDrawn())) {   
            return;
        }
        var isScrollStateChanged = (reason == "scrolling state changed"),
            isNewScrollbars = (reason == "introducing scrolling");
        if (isScrollStateChanged || isNewScrollbars) {
            
            if (this.isRTL() && !this._animatedShowStartRow) {
                this._placeEmbeddedComponents();
            }
        }
        
        if (isScrollStateChanged) {
                
            if (this._rowHeightAnimation == null) {
                this.grid.layoutChildren("body scroll changed");
                delete this._scrollbarChangeDuringAnimation;
           
    
            
            } else {
                this._scrollbarChangeDuringAnimation = true;
            }
        }
    },
    
    // Override rowAnimationComplete to call layoutChildren on the ListGrid if
    // scrollbars were introduced or cleared during animation.
    _rowAnimationComplete : function () {
        this.Super("_rowAnimationComplete", arguments);
        if (this._scrollbarChangeDuringAnimation) {    
            this.grid.layoutChildren("body scroll changed during animation");
            delete this._scrollbarChangeDuringAnimation;
        }
    },

    
    // Override moved to notify any edit form items that they have moved.
    handleMoved : function (a,b,c,d) {
        this.invokeSuper(isc.GridBody, "handleMoved", a,b,c,d);
        
        var lg = this.grid;
        if (lg._editorShowing) {
            lg._editRowForm.itemsMoved();
        }
        
    },
    
    handleParentMoved : function (a,b,c,d) {
        this.invokeSuper(isc.GridBody, "handleParentMoved", a,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) {
            lg._editRowForm.itemsMoved();
        }            
    },
    
    // Override show() / hide() / parentVisibilityChanged() / clear() to notify the Edit 
    // form items that they have been shown / hidden.
    setVisibility : function (newVisibility,b,c,d) {
        this.invokeSuper(isc.GridBody, "setVisibility", newVisibility,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) lg._editRowForm.itemsVisibilityChanged();
    },
    
    parentVisibilityChanged : function (newVisibility,b,c,d) {
        this.invokeSuper(isc.GridBody, "parentVisibilityChanged", newVisibility,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) lg._editRowForm.itemsVisibilityChanged();
    },
    
    clear : function () {
        var lg = this.grid;
        lg._clearingInactiveEditorHTML();

        
        delete this._drawnEditItems;
        this.Super("clear", arguments);
        if (lg._editorShowing) {
            // If we're showing the editRow form, notify the items that they have
            // been removed from the DOM.
            lg._editItemsDrawingNotification(null, null, this);

            // Separate mechanism to notify the form that items are no longer visible.
            
            lg._editRowForm.itemsVisibilityChanged();
        }
    },
    
    // also notify the edit form items of z index change
    zIndexChanged : function () {
        this.Super("zIndexChanged", arguments);
        var lg = this.grid;
        // Note: setZIndex can be called at init time to convert "auto" to a numeric
        // zIndex - we therefore can't assume that we've been added to the ListGrid as
        // a child yet.
        if (lg && lg._editorShowing) lg._editRowForm.itemsZIndexChanged();
        
    },
    parentZIndexChanged : function (a,b,c,d) {
        this.invokeSuper(isc.GridBody, "zIndexChanged", a,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) lg._editRowForm.itemsZIndexChanged();        
    },
    
    // Implement 'redrawFormItem()' - if one of the edit form items asks to redraw
    // we can simply refresh the cell rather than having the entire body redraw
    redrawFormItem : function (item, reason) {
        var lg = this.grid;
        if (lg && (item.form == lg._editRowForm)) {
            // determine which cell
            var row = lg.getEditRow(), col = lg.getColNum(item.getFieldName());
            
            // If the user has edited the cell, or setValue() has been called on the item
            // we don't want a call to redraw() on the item to drop that value
            if (lg.getEditCol() == col) {
                lg.storeUpdatedEditorValue();
            }
            
            if (row >= 0 && col >= 0) lg.refreshCell(row, col, false, true);
            
        } else 
            return this.markForRedraw("Form Item Redraw " + (reason ? reason : isc.emptyString));
    },
	
    
    sizeFormItem : function (item) {
        var lg = this.grid;
        var width = item.width,
            finalWidth;
        
        if (isc.isA.String(width)) {
            var fieldWidths = lg.getEditFormItemFieldWidths(item.record),
                fieldWidth = fieldWidths[lg.getFieldNum(item.getFieldName())];
             if (width == "*") {
                finalWidth = fieldWidth;
             } else if (width[width.length-1] == "%") {
                var percentWidth = parseInt(width);
                if (isc.isA.Number(percentWidth)) {
                    finalWidth = Math.floor(fieldWidth * (percentWidth / 100));
                }
            }
        }
        
        var height = item.height,
            finalHeight;
        if (isc.isA.String(height)) {
            var cellHeight = lg.cellHeight;
            if (width == "*") {
                finalHeight = cellHeight;
            } else if (height[height.length-1] == "%") {
               var percentHeight = parseInt(height);
               if (isc.isA.Number(percentHeight)) {
                   finalHeight = Math.floor(cellHeight * (percentHeight / 100));
               }
            }
        }
        // Hang the calculated values on the _size attribute as we do when running
        // normal stretch-resize policy in form items.
        
        if (finalHeight != null || finalWidth != null) {
            item._size = [finalWidth == null ? item.width : finalWidth,
                          finalHeight == null ? item.height : finalHeight];
        }
        
    },
	
	//>Animation
    // Override startRowAnimation - if doing a delayed redraw to kick off a row animation
    // to close an open folder, we need to temporarily open the folder again to get the
    // appropriate HTML for the animation rows.
    startRowAnimation : function (show, startRow, endRow, callback, speed, duration, 
                                  effect, slideIn, delayed) 
    {
        this.finishRowAnimation();
        
        var shouldOpenFolder = (delayed && (this._openFolder != null)),
            tg = this.grid;
            
        if (shouldOpenFolder) {
            var wasSuppressed = tg._suppressFolderToggleRedraw;
            tg._suppressFolderToggleRedraw = true;
            tg.data.openFolder(this._openFolder);
            tg._suppressFolderToggleRedraw = wasSuppressed;
        }
        this.Super("startRowAnimation", arguments);
        if (shouldOpenFolder) {            
            var wasSuppressed = tg._suppressFolderToggleRedraw;
            tg._suppressFolderToggleRedraw = true;
            tg.data.closeFolder(this._openFolder);
            tg._suppressFolderToggleRedraw = wasSuppressed;
        }        
        delete this._openFolder;
    }
    //<Animation
});


isc.ListGrid.addClassProperties({
    

	//>	@type SortArrow
	//			Do we display an arrow for the sorted field ?
	//			@group	sorting, appearance
    //	@value	"none"   Don't show a sort arrow at all.
    //	@value	"corner" Display sort arrow in the upper-right corner (above the scrollbar) only.
	CORNER:"corner",	
    //	@value	"field"  Display sort arrow above each field header only.
	FIELD:"field",			
    //	@value	"both"   Display sort arrow above each field header AND in corner above scrollbar.
	//BOTH:"both", // NOTE: Canvas establishes this constant
    // @visibility external
	//<
	// NOTE: Canvas established the constant NONE ( == "none")

    //> @type ReorderPosition
    //  Controls where a drag-item should be dropped in relation to the target row
    //  @group dragdrop
    //  @value  ListGrid.BEFORE  Drop the drag-item before the target-row
    BEFORE:"before",
    //  @value  ListGrid.AFTER   Drop the drag-item after the target-row
    AFTER:"after",
    //  @value  ListGrid.OVER    Drop the drag-item over (onto) the target-row
    OVER:"over",
    // @visibility external
	//<
     
	//> @type RowEndEditAction
	//  While editing a ListGrid, what cell should we edit when the user attempts to navigate 
	//  into a cell past the end of an editable row, via a Tab keypress, or a programmatic
	//  saveAndEditNextCell() call?
	//
	// @value   "same"   navigate to the first editable cell in the same record
	// @value   "next"   navigate to the first editable cell in the next record
	// @value   "done"   complete the edit.
	// @value   "stop"   Leave focus in the cell being edited (take no action)
	// @value   "none"   take no action
	//
	// @visibility external
	// @group editing
	// @see ListGrid.rowEndEditAction
	//
	//<
    
	//>	@type EnterKeyEditAction
	// What to do when a user hits enter while editing a cell
	// @value "done" end editing (will save edit values if +link{listGrid.autoSaveEdits} 
	//  is true).
	// @value "nextCell" edit the next editable cell in the record
	// @value "nextRow" edit the same field in the next editable record
	// @value "nextRowStart" edit the first editable cell in next editable record
	//
	// @group editing
	// @visibility external
	//<
    
    //> @type EscapeKeyEditAction
    // What to do if the user hits escape while editing a cell.
    // @value "cancel" cancels the current edit and discards edit values
    // @value "done" end editing (will save edit values if +link{listGrid.autoSaveEdits} 
	//  is true).
    // @value "exit" exit the editor (edit values will be left intact but not saved).
    //
    // @group editing
    // @visibility external
    //<

	//>	@type EditCompletionEvent
	//			What event / user interaction type caused cell editing to complete.
	//			@visibility external
	//			@group	editing
	//
	//          @value  isc.ListGrid.CLICK_OUTSIDE  User clicked outside editor during edit.
	//          @value  isc.ListGrid.CLICK  User started editing another row by clicking on it
	//          @value  isc.ListGrid.DOUBLE_CLICK  User started editing another row by double
    //                               clicking 
	//          @value  isc.ListGrid.ENTER_KEYPRESS Enter pressed.
	//          @value  isc.ListGrid.ESCAPE_KEYPRESS    User pressed Escape.
	//          @value  isc.ListGrid.UP_ARROW_KEYPRESS  Up arrow key pressed.
	//          @value  isc.ListGrid.DOWN_ARROW_KEYPRESS    down arrow key.
	//          @value  isc.ListGrid.TAB_KEYPRESS   User pressed Tab.
	//          @value  isc.ListGrid.SHIFT_TAB_KEYPRESS   User pressed Shift+Tab.
	//          @value  isc.ListGrid.EDIT_FIELD_CHANGE      Edit moved to a different field (same row)
	//          @value  isc.ListGrid.PROGRAMMATIC   Edit completed via explicit function call
	// @visibility external
	//<
    CLICK_OUTSIDE:"click_outside",
    CLICK:"click",
    DOUBLE_CLICK:"doubleClick",
    ENTER_KEYPRESS:"enter",
    ESCAPE_KEYPRESS:"escape",
    UP_ARROW_KEYPRESS:"arrow_up",
    DOWN_ARROW_KEYPRESS:"arrow_down",
    // left/right only used in conjunction with moveEditorOnArrow
    LEFT_ARROW_KEYPRESS:"arrow_left",
    RIGHT_ARROW_KEYPRESS:"arrow_right",
    TAB_KEYPRESS:"tab",
    SHIFT_TAB_KEYPRESS:"shift_tab",
    EDIT_FIELD_CHANGE:"field_change",
    EDIT_ROW_CHANGE:"row_change",
    PROGRAMMATIC:"programmatic",
    // Focus is not a valid edit completion event - focusing in the grid can start an edit
    // if editOnFocus is true but this should not kill an existing edit.
    FOCUS:"focus",

	// GridRenderer passthrough
	// --------------------------------------------------------------------------------------------

	// the following properties, when set on the ListGrid, are applied to the GridBody
    _gridPassthroughProperties : [
    	// pass it a selection object (this enables selection behaviors)
        "selection", 
        "selectionType",
        "canSelectCells",
        "canDragSelect",
        "canSelectOnRightMouse",
        "recordCanSelectProperty",
                    
    	// D&D
        "canDrag",
        "canAcceptDrop",
        "canDrop",

    	// table geometry
        "autoFit",
        "wrapCells",
        "cellSpacing",
        "cellPadding",
        "cellHeight",
        "enforceVClipping",
        // autoFitData behavior implemented on GridBody class, not GR class
        "autoFitData",
        "autoFitMaxRecords",
        "autoFitMaxWidth",
        "autoFitMaxColumns",
        "autoFitMaxHeight",
        "autoFitExtraRecords",
        
        "allowRowSpanning",
        
    	// incremental rendering
    	// "showAllRecords" -> showAllRows done elsewhere
        "showAllColumns",
        "drawAllMaxCells",
        "drawAheadRatio",
        "quickDrawAheadRatio",
        "instantScrollTrackRedraw",
        "scrollRedrawDelay",
        
        // printing
        "printMaxRows",
        
        //>Animation
        // If we're doing a speed rather than duration based row animation allow the cap to
        // be specified on the ListGrid / TreeGrid
        // (Note that this is documented in the TreeGrid class).
        "animateRowsMaxTime",
        //<Animation

    	// documented by default setting
        "fastCellUpdates",

    	// rollover
        "showRollOver",
        "useCellRollOvers",

    	// hover
        "canHover",
        "showHover",
        "showClippedValuesOnHover",
        "hoverDelay", 
        "hoverWidth", 
        "hoverHeight", 
        "hoverAlign", 
        "hoverVAlign", 
        "hoverStyle",
        "hoverOpacity",
        "hoverMoveWithMouse",
        
        "hoverByCell",
        "keepHoverActive",
        "cellHoverOutset",

    	// empty message
        "showEmptyMessage",
        "emptyMessageStyle",
        "emptyMessageTableStyle",    

    	// offline message
        "showOfflineMessage",
        "offlineMessageStyle",
        
        // special presentation of records
        "useCellRecords",
        "singleCellValueProperty",
        "isSeparatorProperty",

    	// Focus things -- note no need to pass tabIndex through - layouts should auto-manage
    	// their members' tab-orders correctly
        "accessKey",
        "canFocus",
        "_useNativeTabIndex",
        "tableStyle",
        "baseStyle",
        "recordCustomStyleProperty",
        "showSelectedStyle",
        
        // whether to use rowSpan-oriented cell styling behaviors
        "useRowSpanStyling",
        // selection mode when rowSpanning is active
        "rowSpanSelectionMode",

        "showFocusOutline"
    ],

	// the following methods, when called on the LV, will call the same-named method on the
	// GridRenderer (this.body). 
    _lv2GridMethods : [
        // this makes it easier to override getCellStyle at the LV level, since you can call
        // these helpers as this.getCellStyleName()
        "getCellStyleName",
        "getCellStyleIndex",

        "_getShowClippedValuesOnHover",

        //> @method listGrid.cellValueIsClipped()
        // @include gridRenderer.cellValueIsClipped()
        //<
        "cellValueIsClipped",

        // setFastCellUpdates explicitly handled
        // in a method which keeps lg.fcu in synch with the 
        // body property value 
        //"setFastCellUpdates",
        
    	// checking table geometry   
        "getRowTop",
        "getRowPageTop",
        "getRowSize",
        "getDrawnRowHeight",
        "getCellPageRect",

        // row span information
        "getCellStartRow",
        "getCellRowSpan",

        //> @method listGrid.getVisibleRows 
    	// @include gridRenderer.getVisibleRows()
        // @visibility external
    	//<
        "getVisibleRows",

        //> @method listGrid.getDrawnRows
    	// @include gridRenderer.getDrawnRows()
        // @visibility external
    	//<
        "getDrawnRows"
    ],

	// styling
	
	//>	@method	listGrid.getCellStyle()
	// @include gridRenderer.getCellStyle()
    // @see listGrid.getBaseStyle()
	//<

	//>	@method	listGrid.getCellCSSText()
	// @include gridRenderer.getCellCSSText()
    // @example addStyle
	//<

	// refresh
	//>	@method	listGrid.refreshCellStyle()    
	//  @include    gridRenderer.refreshCellStyle()
	//<

	// events
	//>	@method	listGrid.cellOver()
	// @include gridRenderer.cellOver()
	//<
	//>	@method	listGrid.rowOver()
	// @include gridRenderer.rowOver()
	//<

	//>	@method	listGrid.cellOut()
	// @include gridRenderer.cellOut()
	//<
	//>	@method	listGrid.rowOut()
	// @include gridRenderer.rowOut()
	//<
    
	//>	@method	listGrid.cellHover()
	// @include gridRenderer.cellHover()
	//<
    //> @method listGrid.cellValueHover() ([A])
    // @include gridRenderer.cellValueHover()
    //<
	//>	@method	listGrid.rowHover()
	// @include gridRenderer.rowHover()
	//<
	//>	@method	listGrid.cellHoverHTML()
	// @include gridRenderer.cellHoverHTML()
	//<
    //> @method listGrid.cellValueHoverHTML()
    // @include gridRenderer.cellValueHoverHTML()
    //<

	//>	@method	listGrid.cellContextClick()
	// @include gridRenderer.cellContextClick()
    // @example cellClicks
	//<
	//>	@method	listGrid.rowContextClick()
	// @include gridRenderer.rowContextClick()
    // @example recordClicks
	//<
    
	//>	@method	listGrid.cellMouseDown()
	// @include gridRenderer.cellMouseDown()
	//<
	//>	@method	listGrid.rowMouseDown()
	// @include gridRenderer.rowMouseDown()
	//<

	//>	@method	listGrid.cellMouseUp()
	// @include gridRenderer.cellMouseUp()
	//<
	//>	@method	listGrid.rowMouseUp()
	// @include gridRenderer.rowMouseUp()
	//<

	//>	@method	listGrid.cellClick()
    // Called when a cell receives a click event.
    // <P>
    // Note that returning false from this method will not prevent any
    // specified +link{listGrid.rowClick} handler from firing.
    //
    // @group   events    
    // @param   record  (ListGridRecord)    Record object returned from getCellRecord()    
    // @param	rowNum	(number)	row number for the cell
    // @param	colNum	(number)	column number of the cell
    // @return	(boolean)	whether to cancel the event
    // @visibility external
    // @example cellClicks
	//<

	//>	@method	listGrid.cellDoubleClick()
	// @include gridRenderer.cellDoubleClick()
    // @example cellClicks
	//<
    
    // Geometry
	//>	@method	listGrid.getRowTop()    
	// @include gridRenderer.getRowTop()
    // @visibility external
	//<
    
	//>	@method	listGrid.getRowPageTop()    
	// @include gridRenderer.getRowPageTop()
    // @visibility external
	//<

    // ListGrid / GridBody passthroughs
    // ---------------------------------------------------------------------------------------

	// the following methods, when called on the GridRenderer used as LV.body, call the same-named
	// method on the ListGrid instance itself
    _grid2LVMethods : [
        
        "getTotalRows",
        "isEmpty",
        "cellIsEnabled",
        "willAcceptDrop",

    	// passed scroll change notification through
        "scrolled",

    	// native element naming
        "getTableElementId",
        "getRowElementId",
        "getCellElementId",

    	// shouldFixRowHeight - enables us to override the ListGrid level 'fixedRecordHeights'
    	// for individual rows
        "shouldFixRowHeight",

        "getEmptyMessage",
        "getCanHover",
        // bubble stopHover on the GR up to stopHover here.
        "stopHover",
        
        "updateEmbeddedComponentZIndex"

        // NOTE: These methods pick up their parameters from the stringMethodRegistry on the
        // GridRenderer class. If expanding this list ensure that any methods that take parameters
        // are registered as stringMethods on that class
    ],

    // used by _invokeKeyboardCopyPasteShortcut for copy/paste between ListGrids
    _cellClipboard : null                                   
});

isc.ListGrid.addClassMethods({
    makeBodyMethods : function (methodNames) {
        var funcTemplate = this._funcTemplate;
        if (funcTemplate == null) {
            funcTemplate = this._funcTemplate = [
                ,
                
                
                "this.grid._passthroughBody = this;" + 
                "var returnVal = this.grid.",,"(",,");" +
                "this.grid._passthroughBody=null;" +
                "return returnVal;"];
        }

        var methods = {},
            method;

		for (var i = 0; i < methodNames.length; i++) {
			var methodName = methodNames[i],
                argString = isc.GridRenderer.getArgString(methodName);
            
            if (isc.contains(argString, "colNum")) {
                // if there's a colNum argument, map it to the field index in the master
                funcTemplate[0] = "if (this.fields[colNum]) colNum = this.fields[colNum].masterIndex;"

            } else if (isc.isAn.emptyString(argString)) {
                // if there are no arguments, pass the body itself as a means of identifying
                // the calling body
                argString = "body";
                funcTemplate[0] = "body = this;";
            } else {
                funcTemplate[0] = null;
            }
    
			// create a function that routes a function call to the target object
            funcTemplate[2] = methodName;
            funcTemplate[4] = argString;
            var functionText = funcTemplate.join(isc.emptyString);
            
            //this.logWarn("for method: " + methodName + " with argString :"  + argString +  
            //             " function text is: " + functionText);

            methods[methodName] = method = new Function(argString, functionText);
            method._isPassthroughMethod = true;
		}
        
        return methods;
    },
    
  
    classInit : function () {
    	// create functions to have methods on the ListGrid's body call methods on the ListGrid
    	// itself.  This is partly legacy support: the way to customize body rendering used to
    	// be to install functions that controlled body rendering directly on the ListGrid
    	// itself.

    	// make certain grid methods appear on the LV for convenience, so you don't have to go
    	// this.body.someGridMethod()
    	
    	
        this.addMethods(isc.ClassFactory.makePassthroughMethods(
            this._lv2GridMethods, "body"));

		// ----------------------------------------------------------------------------------------
    	// create methods that can be installed on the body to call methods on the LV itself, for:
        var passthroughMethods = {};

    	// - handlers (like cellOver) and overrides (like getCellCSSText) that we allow to be
    	//   defined on the LV but are really grid APIs
        var gridAPIs = isc.getKeys(isc.GridRenderer._gridAPIs),
            passthroughMethods = isc.ListGrid.makeBodyMethods(gridAPIs);

    	// - methods the grid needs to fulfill as the drag/drop target, which are really implemented
    	//   on the LV
        isc.addProperties(passthroughMethods,
                          isc.ListGrid.makeBodyMethods(this._grid2LVMethods));

        this._passthroughMethods = passthroughMethods;

    	

    	// create methods on the ListGrid to act as Super implementations for per-instance
    	// overrides of methods where we want to call the original GridRenderer implementation
    	// as Super.
        var passBackMethods = {},
            funcTemplate = [
                ,
                // _passthroughBody is set up by the body function that called back up the
                // the grid method - if present, we use it to ensure we call the original
                // implementation on the correct body.
                "var _passthroughBody = this._passthroughBody || this.body;" +
                " if (_passthroughBody == null) {" +
                    "return;" +
                "}" +
                "if(_passthroughBody.__orig_",,")return _passthroughBody.__orig_",,"(",,")"],
            origPrefix  = "__orig_",
            gridProto = isc.GridRenderer.getPrototype();
        for (var i = 0; i < gridAPIs.length; i++) {
            var methodName = gridAPIs[i],
                argString = isc.GridRenderer.getArgString(methodName);
            if (isc.ListGrid.getInstanceProperty(methodName) == null) {
                 
                if (isc.contains(argString, "colNum")) {
                    // if there's a colNum argument, map it to the field index in the body
                    
                    funcTemplate[0] = "if (colNum != null && colNum >= 0) colNum = this.getLocalFieldNum(colNum);"
                } else {
                    funcTemplate[0] = null;
                }
                funcTemplate[2] = funcTemplate[4] = methodName;
                funcTemplate[6] = argString
                
                passBackMethods[methodName] = new Function(argString, 
                    funcTemplate.join(isc.emptyString));
                // XXX this would also work, but imposes another Super call penalty, and is
                // odd (call to Super from outside of the object)
                //"return this.body.Super('" + methodName + "', arguments);");
            }
        	
            gridProto[origPrefix + methodName] = gridProto[methodName];
        }
        this._passBackMethods = passBackMethods;
        this.addMethods(passBackMethods);

    },

    // retrieve possibly sorted list of coordinates from a coordinate record
    _getCoordinateList : function (coordinateRecord, sortByCoordinate) {
        var list = [];
        for (var coordinate in coordinateRecord) {
            if (coordinateRecord.hasOwnProperty(coordinate)){
                list.add(parseInt(coordinate));
            }
        }
        if (sortByCoordinate) {
            list.sort(function (a, b) { return a - b; });
        }
        return list;
    }
});

// add default properties to the class
isc.ListGrid.addProperties( {
    
	//>	@attr	listGrid.styleName		(CSSStyleName : "listGrid" : IRW)
	// Default CSS class
	// @group	appearance
    // @visibility external
	//<
    styleName:"listGrid",

	//>	@attr	listGrid.data		(List of ListGridRecord : null : IRW)
	// A List of ListGridRecord objects, specifying the data to be used to populate the
    // ListGrid.  In ListGrids, the data array specifies rows. Note that ListGrids
    // automatically observe changes to the data List and redraw accordingly.
    // <p>
    // This property is settable directly only as part of a +link{ListGrid} constructor.  If
    // you want to change the +link{ListGrid}'s data after initial creation, call
    // +link{listGrid.setData}.
    // <p>
    // This property will typically not be explicitly specified for databound ListGrids, where
    // the data is returned from the server via databound component methods such as
    // +link{fetchData()}. In this case the data objects will be set to a 
    // +link{class:ResultSet,resultSet} rather than a simple array.
	//
	// @group	data
    // @see ListGridRecord
    // @setter ListGrid.setData()
	// @visibility external
    // @example inlineData
    // @example localData
	//<
    
    // useCellRecords - Is our data model going to be one record per cell or one record per row?
    useCellRecords:false,

    //> @object ListGridRecord
    // A ListGridRecord is a JavaScript Object whose properties contain values for each
    // +link{ListGridField}.  A ListGridRecord may have additional properties which affect the
    // record's appearance or behavior, or which hold data for use by custom logic or other,
    // related components.
    // <p>
    // For example a ListGrid that defines the following fields:
    // <pre>
    // fields : [
    //     {name: "field1"},
    //     {name: "field2"}
    // ],
    // </pre>
    // Might have the following data:
    // <pre>
    // data : [
    //     {field1: "foo", field2: "bar", customProperty:5},
    //     {field1: "field1 value", field2: "field2 value", enabled:false}
    // ]
    // </pre>
    // Each line of code in the <code>data</code> array above creates one JavaScript Object via
    // JavaScript {type:ObjectLiteral,object literal} notation.  These JavaScript Objects are
    // used as ListGridRecords.
    // <P>
    // Both records shown above have properties whose names match the name property of a
    // ListGridField, as well as additional properties.  The second record will be disabled due to
    // <code>enabled:false</code>; the first record has a property "customProperty" which will
    // have no effect by default but which may be accessed by custom logic.
    // <P>
    // After a ListGrid is created and has loaded data, records may be accessed via
    // +link{listGrid.data}, for example, listGrid.data.get(0) retrieves the first record.
    // ListGridRecords are also passed to many events, such as
    // +link{ListGrid.cellClick,cellClick()}.
    // <P>
    // A ListGridRecord is always an ordinary JavaScript Object regardless of how the grid's
    // dataset is loaded (static data, java server, XML web service, etc), and so supports the
    // normal behaviors of JavaScript Objects, including accessing and assigning to properties
    // via dot notation:
    // <pre>
    //     var fieldValue = record.<i>fieldName</i>;
    //     record.<i>fieldName</i> = newValue;
    // </pre>
    // <P>
    // Note however that simply assigning a value to a record won't cause the display to be
    // automatically refreshed - +link{listGrid.refreshCell()} needs to be called.  Also,
    // consider +link{group:editing,editValues vs saved values} when directly modifying
    // ListGridRecords.
    // <P>
    // See the attributes in the API tab for the full list of special properties on
    // ListGridRecords that will affect the grid's behavior.
    //
    // @treeLocation Client Reference/Grids/ListGrid
    // @see ListGrid.data
    // @inheritsFrom Record
    // @visibility external
    //<
    

    
    //> @attr listGrid.recordEnabledProperty (string : "enabled" : IR)
    // Property name on a record that will be checked to determine whether a record is enabled.
    // <P>
    // Setting this property on a record will effect the visual style and interactivity of
    // the record.  If set to <code>false</code> the record (row in a +link{ListGrid} or
    // +link{TreeGrid}) will not highlight when the mouse moves over it, nor will it respond to
    // mouse clicks.
    //
    // @see listGridRecord.enabled
    // @example disabledRows
    // @visibility external
    //<
    recordEnabledProperty: "enabled",

    //> @attr listGridRecord.enabled (boolean : null : IR)
    //
    // Default property name denoting whether this record is enabled. Property name may be 
    // modified for some grid via +link{listGrid.recordEnabledProperty}.
    //
    // @visibility external
    // @example disabledRows
    //<

    //> @attr listGrid.canExpandRecordProperty (string : "canExpand" : IR)
    // Property name on a record that will be checked to determine whether a record can be 
    // expanded.
    //
    // @see listGridRecord.canExpand
    // @group expansionField
    // @visibility external
    //<
    canExpandRecordProperty: "canExpand",

    //> @attr listGridRecord.canExpand (boolean : null : IR)
    //
    // Default property name denoting whether this record can be expanded. Property name may be 
    // modified for the grid via +link{listGrid.canExpandRecordProperty}.
    //
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.recordCanRemoveProperty (String : "_canRemove" : IRA)
    // If set to false on a record and +link{ListGrid.canRemoveRecords,canRemoveRecords} is
	// true, removal of that record is disallowed in the UI. The icon in the remove field
    // is not shown.
	// @group  editing
    // @visibility external
    //< 
    recordCanRemoveProperty:"_canRemove",

    //> @attr listGridRecord._canRemove (boolean : null : IRW)
    //
    // Default property name denoting whether this record can be removed. Property name may be 
    // modified for the grid via +link{listGrid.recordCanRemoveProperty}.
    //
    // @group  editing
    // @visibility external
    //<

    //> @attr listGridRecord.isSeparator (boolean : null : IR)
    //
    // Default property name denoting a separator row.<br>
    // When set to <code>true</code>, defines a horizontal separator in the listGrid
    // object. Typically this is specified as the only property of a record object, since a
    // record with <code>isSeparator:true</code> will not display any values.<br>
    // Note: this attribute name is governed by +link{ListGrid.isSeparatorProperty}.
    // @visibility external
    //<

    //> @attr listGridRecord.customStyle (CSSStyleName : null : IRW)
    // Name of a CSS style to use for all cells for this particular record.  
    // <P>
    // Note that using this property assigns a single, fixed style to the record, so rollover
    // and selection styling are disabled.  To provide a series of stateful styles for a record
    // use +link{listGridRecord._baseStyle} instead.
    // <P>
    // See +link{listGrid.getCellStyle()} for an overview of various ways to customize styling,
    // both declarative and programmatic.
    // <P>
    // If this property is changed after draw(), to refresh the grid call
    // +link{listGrid.refreshRow()} (or +link{listGrid.markForRedraw()} if several rows are
    // being refreshed).
    // <P>
    // If your application's data uses the "customStyle" attribute for something else, the
    // property name can be changed via +link{listGrid.recordCustomStyleProperty}.
    //
    // @visibility external
    //<

    //> @attr listGridRecord._baseStyle (CSSStyleName : null : IRW)
    // Name of a CSS style to use as the +link{listGrid.baseStyle} for all cells for this
    // particular record.  
    // <P>
    // The styleName specified with have suffixes appended to it as the record changes state
    // ("Over", "Selected" and so forth) as described by +link{listGrid.getCellStyle()}.  For a
    // single, fixed style for a record, use +link{listGridRecord.customStyle} instead.
    // <P>
    // See +link{listGrid.getCellStyle()} for an overview of various ways to customize styling,
    // both declarative and programmatic.
    // <P>
    // If this property is changed after draw(), to refresh the grid call
    // +link{listGrid.refreshRow()} (or +link{listGrid.markForRedraw()} if several rows are
    // being refreshed).
    // <P>
    // If your application's data uses the "_baseStyle" attribute for something else, the
    // property name can be changed via +link{listGrid.recordBaseStyleProperty}.
    //
    // @visibility external
    //<
    
    //> @attr listGridRecord.singleCellValue (HTML : null : IRW)
    // Default property name denoting the single value to display for all fields of this row.
    // If this property is set for some record, the record will be displayed as a single 
    // cell spanning every column in the grid, with contents set to the value of this
    // property.<br>
    // Note: this attribute name is governed by +link{ListGrid.singleCellValueProperty}.
    // @visibility external
    //<


    //> @attr listGridRecord.canDrag (boolean : null : IR)
    //
    // When set to <code>false</code>, this record cannot be dragged. If canDrag is false for
    // any record in the current selection, none of the records will be draggable.
    //
    // @visibility external
    //<

    //> @attr listGridRecord.canAcceptDrop (boolean : null : IR)
    //
    // When set to <code>false</code>, other records cannot be dropped on (i.e., inserted
    // via drag and drop) immediately before this record.
    //
    // @visibility external
    //<

	//> @attr listGridRecord.linkText (string : null : IRW)
    //
    //  The HTML to display in this row for fields with fieldType set to link. This overrides
    //  +link{attr:listGridField.linkText}.
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
    //  @see attr:listGridField.linkText
    //  @see attr:listGrid.linkTextProperty
	//  @group  display_values
	//  @visibility external
	//<
    
	// Animation
    // ---------------------------------------------------------------------------------------
    // These apply to ListGrid grouping, which basically makes the data model into a Tree where animation
    // is applied for folder open/close.
    
    //> @attr listGrid.animateFolders (Boolean : true : IRW)
    // If true, when folders are opened / closed children will be animated into view.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    //<
    animateFolders:true,
    
    //> @attr listGrid.animateFolderMaxRows (integer : null : IRW)
    // If +link{animateFolders} is true for this grid, this number can be set to designate
    // the maximum number of rows to animate at a time when opening / closing a folder.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @see getAnimateFolderMaxRows()
    // @group animation 
    // @visibility external
    //<

    //> @attr listGrid.animateFolderTime (number : 100 : IRW)
    // When animating folder opening / closing, if +link{treeGrid.animateFolderSpeed} is not
    // set, this property designates the duration of the animation in ms.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    // @see listGrid.animateFolderSpeed
    //<
    animateFolderTime:100,

    //> @attr listGrid.animateFolderSpeed (number : 3000 : IRW)
    // When animating folder opening / closing, this property designates the speed of the
    // animation in pixels shown (or hidden) per second. Takes precedence over the 
    // +link{treeGrid.animateFolderTime} property, which allows the developer to specify a
    // duration for the animation rather than a speed.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    // @see listGrid.animateFolderTime
    //<    
    animateFolderSpeed:3000,
    
    //> @attr listGrid.animateFolderEffect (AnimationAcceleration : null : IRW)
    // When animating folder opening / closing, this property can be set to apply an
    // animated acceleration effect. This allows the animation speed to be "weighted", for
    // example expanding or collapsing at a faster rate toward the beginning of the animation
    // than at the end.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    //<    

    //> @attr listGrid.animateRowsMaxTime (number : 1000 : IRW)
    // If animateFolderSpeed is specified as a pixels / second value, this property will cap
    // the duration of the animation.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation_advanced
    //<    
    animateRowsMaxTime:1000,

    // external: doc'd on TreeGrid
    shouldAnimateFolder : function (folder) {
        if (!this.animateFolders || !this.isDrawn()) return false;
        
        if (this.autoFitData == "vertical" || this.autoFitData== "both") return false;
        
        var children = this.data.isFolder(folder) ? this.data.getOpenList(folder) : null;
        
        // No children - bit arbitrary whether we "animate" or not!
        
        if (children == null || children.length <= 1) return false;
        return (children.length <= this.getAnimateFolderMaxRows());
    },
    
    // external: doc'd on TreeGrid
    getAnimateFolderMaxRows : function () {
        var maxRows = this.animateFolderMaxRows;
        if (maxRows == null) {
            var vfRs = this.body ? this.body._getViewportFillRows() : [0,0];
            maxRows = Math.min(75, (vfRs[1]-vfRs[0]) * 3);
        }
        return maxRows
    },
    
	
	// DataBinding
	// ----------------------------------------------------------------------------------------

	//>	@attr listGrid.fields (Array of ListGridField : null : [IRW])
	// An array of field objects, specifying the order, layout, formatting, and
	// sorting behavior of each field in the listGrid object.  In ListGrids, the fields
	// array specifies columns.  Each field in the fields array is a ListGridField object. 
    // Any listGrid that will display data should have at least one visible field.
    // <p>
    // If +link{ListGrid.dataSource} is also set, this value acts as a set of overrides as
    // explained in +link{attr:DataBoundComponent.fields}.
    //
	// @see    ListGridField
	// @see    setFields()
    // @group databinding
	// @visibility external
    // @example listGridFields
    // @example mergedFields
	//<
    
    //> @attr listGrid.defaultFields (Array of ListGridField Properties : null : IRA)
    // An array of listGrid field configuration objects.  When a listGrid is initialized, if this
    // property is set and there is no value for the <code>fields</code> attribute, this.fields will
    // be defaulted to a generated array of field objects duplicated from this array.
    // <P>
    // This property is useful for cases where a standard set of fields will be displayed
    // in multiple listGrids - for example a subclass of ListGrid intended to display a particular
    // type of data:<br>
    // In this example we would not assign a single +link{listGrid.fields} array directly to the
    // class via <code>addProperties()</code> as every generated instance of this class would
    // then point to the same fields array object. This would cause unexpected behavior such as
    // changes to the field order in one grid effecting other grids on the page.<br>
    // Instead we could use <code>addProperties()</code> on our new subclass to set
    // <code>defaultFields</code> to a standard array of fields to display. Each generated instance
    // of the subclass would then show up with default fields duplicated from this array.
    // @visibility external
    //<
    
	//>	@attr	listGrid.dataSource		(DataSource or ID : null : IRW)
    // @include dataBoundComponent.dataSource
	//<
    
	//> @attr listGrid.autoFetchDisplayMap (Boolean : true : [IRW])
	// If true, for fields where +link{listGridField.optionDataSource} is specified,
    // a valueMap will be automatically created by making a +link{dataSource.fetchData()} call
    // against the specified dataSource and extracting a valueMap from the returned records
    // based on the displayField and valueField.
    // <P>
    // If set to false, valueMaps will not be automatically fetched.  In this case, setting
    // field.optionDataSource is effectively a shortcut for setting optionDataSource on
    // the editor via +link{listGridField.editorProperties}.
    // <P>
    // Can also be disabled on a per-field basis with +link{listGridField.autoFetchDisplayMap}.
    //
	// @group display_values
	// @see listGridField.autoFetchDisplayMap
	// @see listGridField.optionDataSource
    // @visibility external
	//<
    autoFetchDisplayMap:true,
    
    //> @attr listGrid.warnOnUnmappedValueFieldChange (Boolean : true : IRWA)
    // If a field has +link{listGridField.displayField} specified and has no
    // +link{listGridField.optionDataSource}, this field will display the value from the
    // <code>displayField</code> of each record by default (for more on this behavior
    // see +link{listGridField.optionDataSource}).
    // <P>
    // If such a field is editable, changing the edit value for the field on some record, 
    // without updating the edit value for the associated display field on the same record 
    // would mean the user would continue to see the unchanged display field value.
    // Developers can resolve this situation by programmatically setting an edit value for
    // the display field as well as the data field, or avoid it by specifying an optionDataSource
    // and ensuring +link{listGrid.autoFetchDisplayMap} is true, or setting an explicit valueMap
    // for the field.
    // <P>
    // By default, when the edit value on a field with a specified displayField and 
    // no optionDataSource is set, we log a warning to notify the developer. This warning may
    // be disabled by setting <code>warnOnUnmappedValueFieldChange</code> to <code>false</code>.
    // <P>
    // Note: There are actually a couple of cases in which the system will automatically
    // derive a new display-field value and apply it to the record:
    // <ol><li>If the edit value was changed by a user actually editing the record 
    // (rather than a programmatic call to setEditValue()), and the edit-item had
    // a valueMap or optionDataSource set, we automatically pick up the display value from
    // that item and store it as an edit-value for the displayField of the record</li>
    //     <li>If the listGrid has a loaded record in its data set whose valueField value matches
    // the edit value for the valueField, we automatically apply the displayField value from that
    // record as an edit value for the displayField on the newly edited record.</li></ol>
    // In either case, the display value for the record is updated automatically 
    // (and the warning would not be logged).
    // @visibility external
    //<
    warnOnUnmappedValueFieldChange:true,
    
    //or <code>autoFetchDisplayMap</code> is false at
    // the +link{listGrid.autoFetchDisplayMap,listGrid}
    // or +link{listGridField.autoFetchDisplayMap,field} level, the field will display the
    // record value from the +link{

    //> @attr ListGrid.saveLocally (boolean : null : IRA)
    // For grids with a specified +link{ListGrid.dataSource}, this property can be set to 
    // <code>true</code> to prevent the grid from attempting to save / retrieve data from the
    // server.  In this case the grid's data should be specified as an array via 
    // the +link{listGrid.data} attribute, and the datasource will simply act as a schema to
    // describe the set of fields visible in the grid. 
    // +link{listGrid.canEdit,Inline edits}, or removals via the
    // +link{listGrid.canRemoveRecords} mechanism will update the local data array rather than
    // attempting to perform operations against the dataSource.
    // @see useRemoteValidators
    // @visibility external
    // @group databinding
    //<
    
    
    //> @attr ListGrid.saveRequestProperties (DSRequest Properties : null : IRWA)
    // For editable grids with a specified +link{listGrid.dataSource}, where 
    // +link{listGrid.saveLocally} is false, this attribute may be used to specify standard
    // DSRequest properties to apply to all save operations performed by this grid (whether
    // triggered by user interaction, or explicit saveEdits or saveAllEdits call).
    // <P>
    // An example usage would be to customize the prompt displayed while saving is in progress
    // if +link{listGrid.waitForSave} is true.
    // <P>
    // Note that for more advanced customization of save operations,
    // +link{dataBoundComponent.addOperation} and +link{dataBoundComponent.updateOperation}
    // are available to developers, allowing specification of an explicit +link{operationBinding} 
    // for the add / update operation performed on save. 
    //
    // @visibility external
    // @group dataBinding
    // @group editing
    //<
    
    //> @attr ListGrid.useRemoteValidators (boolean : null : IRWA)
    // If +link{listGrid.saveLocally} is specified, but this grid is bound to a DataSource which
    // includes remote field validators, by default edits will be saved synchronously and
    // these validators will not be executed.<br>
    // Set this property to <code>true</code> to ensure these remote validators are called when 
    // saving edits in saveLocally mode. Note that since these remote validators need to run on 
    // the server, saving with this property set is asynchronous, even though the data that
    // ultimately gets updated is already present on the client.
    // @visibility external
    // @group databinding
    //<

	//>	@attr listGrid.useAllDataSourceFields (boolean : null : IRW)
    // @include dataBoundComponent.useAllDataSourceFields
    // @group databinding
	//<

	//>	@attr listGrid.showDetailFields (Boolean : true : IR)
    // Whether to include fields marked <code>detail:true</code> from this component's 
    // <code>DataSource</code>.
    // <P>
    // Fields may also be included directly in this component's <code>fields</code> array in
    // which case they will be present regardless of the <code>detail</code> attribute.
    // <p>
    // When this property is <code>true</code>, <code>ListGrid</code>s will include all 
    // detail fields, but they will be initially hidden.  The user may show
    // these fields via the default header context menu (+link{listGrid.showHeaderContextMenu}).
    // Setting this property to false will completely exclude all detail fields from the list
    // grid's fields array, such that they cannot be shown by the user or programmatically.
    // <P>
    // To override the visibility of individual fields, use the standard
    // +link{listGrid.showField()}, +link{listGrid.hideField()} and +link{listGridField.showIf}
    // APIs, for example, set showIf:"true" to show a detail field initially.
    // 
    // @group databinding
    // @visibility external
    //<
    showDetailFields:true,
    
    //> @attr ListGrid.titleField (string : see below : IRW)
    // Best field to use for a user-visible title for an individual record from this grid.
    // If +link{ListGrid.dataSource} is non null, this property may be specified on the 
    // dataSource instead.
    // <p>
    // If not explicitly set, titleField looks for fields named "title", "name", and "id" 
    // in that order.  If a field exists with one of those names, it becomes the titleField.  
    // If not, then the first field is designated as the titleField.
    //  @visibility external
    //<

    
    //> @attr listGrid.dataProperties (object : null :IRWA)
    // For databound ListGrids, this attribute can be used to customize the +link{ResultSet}
    // object created for this grid when data is fetched
    // @group databinding
    // @visibility external
    //<
    
    // Grouping
    // ---------------------------------------------------------------------------------------
    
    //> @object groupNode
    //
    // An auto-generated subclass of +link{TreeNode} representing the group nodes
    // in a grouped listgrid. 
    //
    // @see listGrid.groupBy()
    // @treeLocation Client Reference/Grids/ListGrid
    // @group grouping
    // @visibility external    
    //< 
    
    //> @attr groupNode.groupMembers (Array of ListGridRecord or GroupNode : see below : R)
    // Array of ListGridRecord that belong to this group, or, for multi-field grouping, array
    // of groupNodes of subgroups under this groupNode.
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @attr groupNode.groupTitle (HTML : see below : R)
    // The computed title for the group, which results from +link{listGridField.getGroupTitle()}
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @attr groupNode.groupValue (any : see below : R)
    // The value from which groups are computed for a field, 
    // which results from +link{listGridField.getGroupValue()}
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @attr listGrid.originalData (object : null : R)
    // When grouped, a copy of the original ungrouped data.
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<

    //> @attr listGrid.groupTree (AutoChild Tree : null : R)
    // The data tree that results from a call to  +link{listGrid.groupBy()}.
    // This will be a +link{ResultTree} if +link{listGrid.dataSource} is
    // present, otherwise it will be a +link{Tree}.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<

    //> @type GroupStartOpen 
    // Possible values for the state of ListGrid groups when groupBy is called
    //
    // @value "all" open all groups
    // @value "first" open the first group
    // @value "none" start with all groups closed
    // @visibility external
    //<
    
    //> @attr listGrid.groupStartOpen (GroupStartOpen || Array : "first" : IRW)
    // Describes the default state of ListGrid groups when groupBy is called. 
    //
    // Possible values are:
    // <ul>
    // <li>"all": open all groups
    // <li>"first": open the first group
    // <li>"none": start with all groups closed 
    // <li>Array of values that should be opened
    // </ul>
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    groupStartOpen:"first",
    
    //> @attr listGrid.canCollapseGroup (Boolean : true : IR)
    // Can a group be collapsed/expanded? When true a collapse/expand icon is shown
    // (+link{groupIcon,groupIcon}) and clicking the icon or double-clicking the group title
    // will collapse or expand the group.
    //
    // When false the group icon is not shown and double-clicking on the title does
    // not change group state. Additionally +link{groupStartOpen,groupStartOpen} is 
    // initialized to "all".
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    canCollapseGroup:true,

    //> @attr listGrid.groupTitleField (String : null : IR)
    // When a list grid is +link{listGrid.groupBy(),grouped}, each group shows
    // under an auto generated header node. By default the title of the group will be
    // shown, with a hanging indent in this node, and will span all columns in the grid.
    // Setting this property causes the titles of auto-generated group nodes to appear as
    // though they were values of the designated field instead of spanning all columns
    // and record values in the designated groupTitleField will appear indented under 
    // the group title in a manner similar to how a TreeGrid shows a Tree.
    // <P>
    // Note if +link{listGrid.showGroupSummaryInHeader} is true, the header nodes will not show
    // a single spanning title value by default - instead they will show the summary values for
    // each field. In this case, if groupTitleField is unset, a 
    // +link{listGrid.showGroupTitleColumn,groupTitleColumn} can be automatically generated to
    // show the title for each group.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    getGroupTitleField : function () {
        return this.groupTitleField;
    },
    
    //> @attr listGrid.showGroupTitleColumn (Boolean : true : IR)
    // If this grid is +link{listGrid.groupBy(),grouped} and +link{listGrid.showGroupSummaryInHeader}
    // is true, instead of group header nodes showing up with a single cell value spanning the full
    // set of columns, summaries for each field will show up in the appropriate columns of the 
    // header node.
    // <P>
    // In this case there are 2 options for where the group title will show up. Developers may
    // specify an existing field to put the title values into via +link{listGrid.groupTitleField}.
    // If no groupTitleField is specified, this property may be set to <code>true</code>
    // which causes a <code>groupTitleColumn</code> to be automatically generated.
    // Each group header will show the group title in this column (records within the group will
    // not show a value for this column). The column appears in the leftmost position within the
    // grid (unless +link{listGrid.showRowNumbers} is true, in which case this column shows up
    // in the second-leftmost position), and by default will auto-fit to its data.
    // <P>
    // To customize this field, developers may modify 
    // +link{listGrid.groupTitleColumnProperties} 
    // <var class="SmartClient">or
    // +link{listGrid.groupTitleColumnDefaults} at the class level.</var>
    // @visibility external
    //<
    showGroupTitleColumn:true,
   
    //> @attr listGrid.groupTitleColumnProperties (ListGridField properties : null : IR)
    // Custom properties for the automatically generated <code>groupTitleColumn</code>.
    // <P>
    // See +link{listGrid.showGroupTitleColumn} for an overview of the groupTitleColumn.
    // @visibility external
    //<
    //groupTitleColumnProperties:null,
    
    //> @attr listGrid.groupTitleColumnDefaults (ListGridField properties : object : IR)
    // Default properties for the automatically generated <code>groupTitleColumn</code>.
    // Default object includes properties to enable autoFitWidth to group title values.
    // <P>
    // To modify the behavior or appearance of this column, developers may set
    // +link{listGrid.groupTitleColumnProperties} at the instance level, or override this
    // object at the class level. If overriding this object, we recommend using
    // +link{class.changeDefaults()} rather than replacing this object entirely.
    // <P>
    // See +link{listGrid.showGroupTitleColumn} for an overview of the groupTitleColumn.
    // @visibility external
    //<
    groupTitleColumnDefaults:{
        canEdit:false,
        canFilter:false,
        canHide:false,
        canReorder:false,
        showDefaultContextMenu:false,
        autoFreeze:true,
        
        sortNormalizer:function (recordObject,fieldName,context) {
            return recordObject.groupTitle;
        },
        
        autoFitWidth:true,
        autoFitWidthApproach:"value",
        title:"&nbsp;"
    },
    
    
    // We actually show the special group title column if
    // - we're showing the group summary in the header
    // - we have no explicitly specified group title field
    // - the showGroupTitleColumn flag is true
    showingGroupTitleColumn : function () {
        return (this.isGrouped && this.showGroupSummary && this.showGroupSummaryInHeader 
                && this.showGroupTitleColumn && this.getGroupTitleField() == null);
    },
    
    // groupTitleColumnName: This could be modified to display an actual field within the
    // grid data, but the developer might as well use groupTitleField instead. 
    // Leaving unexposed for now.
    groupTitleColumnName:"groupTitle",
    
    getGroupTitleColumn : function () {
        var grid = this;
        var groupTitleColumn = isc.addProperties(
            {   _isGroupTitleColumn:true,
                // 'grid' available through closure
                getAutoFreezePosition: function () { return grid.getGroupTitleColumnPosition() }
            }, 
            this.groupTitleColumnDefaults, 
            this.groupTitleColumnProperties
        );
     
        if (groupTitleColumn.name == null) {
            groupTitleColumn.name = this.groupTitleColumnName;
        }
        return groupTitleColumn;
    },
    
    getGroupTitleColumnPosition : function () {
        // This is really just a sanity check - we don't expect to be calling this method when
        // we're not showing the special groupTitleColumn
        if (!this.showingGroupTitleColumn()) return -1;

        
        var pos = 0;        
        if (this.shouldShowRowNumberField()) pos++;
        if (this.shouldShowCheckboxField()) pos++;
        if (this.shouldShowExpansionField()) pos++;
        return pos;
    },
    
    singleCellGroupHeaders : function () {
        if (this.getGroupTitleField() != null) return false;
        if (this.showGroupSummary && this.showGroupSummaryInHeader) return false;
        
        return true
    },
    
    //> @attr listGrid.showGroupSummaryInHeader (Boolean : false : IRW)
    // If this grid is +link{listGrid.groupBy(),grouped}, and +link{listGrid.showGroupSummary}
    // is true, setting this property causes field summary values for each group to be displayed
    // directly in the group header node, rather than showing up at the bottom of each
    // expanded group.
    // <P>
    // Note that this means the group header node will be showing multiple field values
    // rather than the default display of a single cell spanning all columns containing the
    // group title. Developers may specify an explicit +link{listGrid.groupTitleField}, or
    // rely on the automatically generated +link{listGrid.showGroupTitleColumn,groupTitleColumn}
    // to have group titles be visible in as well as the summary values.
    // <P>
    // Also note that multi-line group summaries are not supported when showing
    // the group summary in the group header. If multiple 
    // +link{listGridField.summaryFunction,field summary functions} are defined for some field
    // only the first will be displayed when this property is set to true.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    showGroupSummaryInHeader:false,
    
    //> @attr listGrid.showCollapsedGroupSummary (Boolean : false : IRW)
    // Should group summaries be visible when the group is collapsed?
    // <P>
    // This property only applies to +link{listGrid.groupBy(),grouped} grids showing
    // +link{listGrid.showGroupSummary,group summary rows}. When set to true, the
    // group summary row(s) for each group will show up under the group header nodes when
    // the group is collapsed, or at then end of the grouped set of data if the group
    // is expanded. 
    // <P>
    // This property has no effect if +link{showGroupSummaryInHeader} is true.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    showCollapsedGroupSummary:false,
    
    //> @method listGridField.getGroupValue()
    // Return the value which records should be grouped by.  
    // <P>
    // All records for which getGroupValue() returns the same value appear in the same
    // group.  Default is the result of +link{listGrid.getCellValue}.
    // <P>
    // While any type of value may be returned, avoiding the use of string values may
    // result in improved performance. In this case, +link{listGridField.getGroupTitle()}
    // may be implemented to map a numeric group value into a legible string.
    //
    // @param   value (any)   raw value for the cell, from the record for the row
    // @param   record (ListGridRecord) 
    //   Record object for the cell. Note: If this is a new row that has not been saved, in an 
    //   editable grid, it has no associated record object. In this case the edit values will
    //   be passed in as this parameter (see +link{listGrid.getEditValues()})
    // @param   field (Object)    Field object for which to get group value
    // @param   fieldName (String)    The name of the field
    // @param   grid (ListGrid) the ListGrid displaying the cell
    // @return (any) Group value to which this record belongs
    //
    // @see listGrid.groupBy()
    // @see listGridField.getGroupTitle()
    // @group grouping
    // @visibility external
    // @example customGrouping
    //<
    
    //> @method listGridField.getGroupTitle()
    // Return the title that should be shown to the user for the group with the
    // <code>groupValue</code> passed as a parameter.
    // <P>
    // Default title is the groupValue itself. 
    //
    // @group grouping
    //
    // @param   groupValue (any)   the value from the group is created, the result of 
    //  +link{listGridField.getGroupValue()}
    // @param   groupNode (groupNode) the node in the grid containing the group. 
    // @param   field (Object)    Field object for which to get group value
    // @param   fieldName (String)    The name of the field
    // @param   grid (ListGrid) the ListGrid displaying the cell
    // @return (any) Group value to which this record belongs
    //
    // @see listGrid.groupBy()
    // @see listGridField.getGroupValue()
    // @visibility external
    // @example customGrouping
    //<

    //> @attr listGridField.groupingModes (ValueMap : null : IR)
    // If set, provides several possible styles of grouping that are valid for this field.  For
    // example, a field of type:"date" might be able to be grouped by week, month, or by day of
    // week.
    // <P>
    // If <code>groupingModes</code> are present and
    // +link{listGrid.canGroupBy,grouping is enabled}, the menu for this field includes a
    // submenu of possible grouping modes generated from the <code>groupingModes</code> valueMap.
    // When the user selects a particular grouping mode,
    // +link{listGridField.groupingMode,field.groupingMode} is set to the user's chosen mode,
    // and this choice can be detected via the <code>field</code> parameter to
    // +link{listGridField.getGroupValue()} in order to provide different modes of grouping.
    // <P>
    // The user may also choose to group records without specifying a grouping mode, in this case,
    // the +link{listGridField.defaultGroupingMode} is used.
    // <P>
    // Note that <code>getGroupValue</code>, <code>groupingModes</code> et al can be specified on
    // +link{SimpleType} declarations, and the different grouping modes that are offered
    // automatically for various common types are defined 
    // +link{class:GroupingMessages, this way}.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.groupingMode (identifier : null : IR)
    // For a field that allows multiple +link{listGridField.groupingModes,grouping modes}, 
    // the current grouping mode.
    // <P>
    // This property is set when a user chooses a particular grouping mode, and may be set on
    // ListGrid creation to affect the initial grouping.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.defaultGroupingMode (identifier : null : IR)
    // Default groupingMode used when the user does not specify a mode or grouping is triggered
    // programmatically and +link{listGridField.groupingMode,field.groupingMode} is unset.
    // See +link{listGridField.groupingModes,field.groupingModes}.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.groupPrecision (integer : null : IR)
    // For fields of type:"float" or derived from float, number of digits after the decimal point
    // to consider when grouping.
    // <P>
    // For example, <code>groupPrecision:2</code> indicates that 45.238 and 45.231 group together,
    // but 45.22 and 45.27 are separate.
    // <P>
    // See also +link{listGridField.groupGranularity,groupGranularity} for grouping by broader
    // ranges.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.groupGranularity (integer : null : IR)
    // Granularity of grouping for numeric fields.
    // <P>
    // Groups will be formed based on ranges of values of size <code>groupGranularity</code>.  For
    // example, if groupGranularity were 1000, groups would be 0-1000, 1000-2000, etc.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.canHilite (boolean : null : IRW)
    // Determines whether this field can be hilited.  Set to false to prevent this
    // field from appearing in HiliteEditor.
    // 
    // @visibility external
    //<

    //> @attr listGridField.canGroupBy (Boolean : true : IRW)
    // Determines whether this field will be groupable in the header context menu.
    // 
    // @see listGrid.groupBy()
    // @visibility external
    //<

    //> @attr listGridField.canSortClientOnly (Boolean : false : IRW)
    // When true, this field can only be used for sorting if the data is entirely client-side.
    // 
    // @visibility external
    //<

    //> @attr listGridField.showDefaultContextMenu (Boolean : true : IRW)
    // When set to false, this field will not show a context menu in its header.
    // 
    // @visibility external
    //<

    //> @attr listGridField.filterOperator (OperatorId : null : IR)
    // With the +link{listGrid.showFilterEditor,FilterEditor} showing, the +link{Operator} to 
    // use when matching values for this field.
    // <P>
    // Note that you can set all FilterEditor fields to default to either substring or exact 
    // match via +link{listGrid.autoFetchAsFilter}, but if you want a mix of exact vs substring 
    // match on different fields, you need to use this property, and your ListGrid will produce
    // +link{AdvancedCriteria} rather than the simpler +link{Criteria} format.  This is
    // automatically and transparently handled by the SmartClient Server's SQLDataSource and
    // HibernateDataSource in Power Edition or above, but if you have your own filtering
    // implementation, it will need to be able to handle AdvancedCriteria.
    // @visibility external
    //<

    //> @attr listGridField.canExport (Boolean : null : IR)
    //	Dictates whether the data in this field be exported.  Explicitly set this
    //  to false to prevent exporting.  Has no effect if the underlying 
    //  +link{dataSourceField.canExport, dataSourceField} is explicitly set to 
    //  canExport: false.
    //
    // @visibility external
    //<

    //> @attr listGridField.exportRawValues (Boolean : null : IR)
    //	Dictates whether the data in this field should be exported raw by 
    // +link{DataBoundComponent.exportClientData, exportClientData()}.  If set to true for a 
    // field, the values in the field-formatters will not be executed for data in this field.
    // Decreases the time taken for large exports.
    //
    // @visibility external
    //<

    //> @attr listGridField.summaryValue (HTMLString : null : IRW)
	// The value to display for a ListGridField when it appears in the +link{summaryRow}.  The
    // default for normal fields is null and for special fields, like the +link{checkboxField},
    // the default is to show a blank value (a non-breaking space).
    // @visibility external
    //<
    
    //> @attr listGrid.groupNodeStyle (String : "groupNode" : IRW)
    // The CSS style that group rows will have
    //
    // @group grouping
    // @see group:grouping
    // @visibility external
    //<
    groupNodeStyle: "groupNode",
	
    //> @attr listGrid.groupIcon (SCImgURL : "[SKINIMG]/TreeGrid/opener.gif" : IRW)
    // The URL of the base icon for the group icons in this treegrid.
    //
    // @group grouping
    // @see group:grouping
    // @visibility external
    //<
    groupIcon: "[SKINIMG]/TreeGrid/opener.gif",
    
    //> @attr listGrid.groupIconSize (Number : 16 : IRW)
    // Default width and height of group icons for this ListGrid.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    groupIconSize: 16,
    
    //> @attr listGrid.groupIndentSize (Number : 20 : IRW)
    // Default number of pixels by which to indent subgroups relative to parent group.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @see listGrid.getGroupNodeHTML
    // @visibility external
    //<
    groupIndentSize: 20,
    
    //> @attr listGrid.groupLeadingIndent (Number : 10 : IRW)
    // Default number of pixels by which to indent all groups.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @see listGrid.getGroupNodeHTML
    // @visibility external
    //<
    groupLeadingIndent: 10,

    //> @attr listGrid.canGroupBy (Boolean : true : IRW)
    // If false, grouping via context menu will be disabled.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    canGroupBy: true,

    //> @attr listGrid.groupByMaxRecords (int : 1000 : IRW)
    // Maximum number of records to which a groupBy can be applied. If there are more records,
    // grouping will not be available via the default header context menu, and calls to 
    // +link{listGrid.groupBy()} will be ignored.
    // <P>
    // The maximum exists because ListGrid grouping is performed in-browser, hence requires loading of
    // all records that match the current filter criteria before records can be grouped.  The default
    // maximum represents a number of records which are safe to load in legacy browsers such as Internet
    // Explorer 8 (modern browsers can handle far more), and is also a good upper limit from the
    // perspective of loading data from a database.
    // <P>
    // Going beyond this limit can cause "script running slowly" errors from legacy browsers (as well as
    // high database load).  To build an interface for grouping that handles arbitrary data volume, use
    // a TreeGrid with +link{treeGrid.loadDataOnDemand} with server-side grouping code.
    // 
    // @group grouping
    // @see groupBy
    // @visibility external
    //<
    groupByMaxRecords: 1000,

    //> @attr listGrid.groupByAsyncThreshold (int : 50 : IRW)
    // When grouping is requested with this number of records or more, an asynchronous approach is
    // used to avoid the browser showing a "script is running slowly.." message prompting the
    // user to stop execution of JavaScript.
    // <p>
    // Note that +link{groupByMaxRecords} must be set at least as high as +link{groupByAsyncThreshold}
    // or asynchronous grouping will never be used.
    // <p>
    // During asynch grouping, interactivity is blocked and the +link{asynchGroupingPrompt} is shown
    // to the user, then hidden when grouping completes.
    // <p>
    // Note that this async processing covers grouping <b>only</b> - it does not cover whole grid or
    // per-group summaries, client-side sort or filter, or other operations that may cause the browser
    // to show the "script is running slowly" prompt when working with very large sets of records in a
    // grid.
    // <p>
    // At this time, there is no generally effective way to avoid this warning dialog appearing with very
    // large datasets in Microsoft's Internet Explorer (IE).  IE's severely flawed detection algorithm for
    // runaway scripts has been shown to interrupt computations after only 0.2 seconds elapsed time
    // even if the computation would have finished in 0.3 seconds.  Optimizations that reduce
    // execution time can sometimes trigger the "script running slowly" dialog sooner.  Since not
    // every operation can reasonably be made asychronous, the current recommendation is to avoid
    // working with overly large datasets until the affected versions of IE are obsoleted.
    // @visibility external
    //<
    groupByAsyncThreshold: 50,

    //> @attr listGrid.asynchGroupingPrompt (String : "${loadingImage}&nbsp;Grouping data..." : IR)
    // The prompt to display while interactivity is blocked during +link{groupByAsyncThreshold,asynchronous grouping}.
    // @group i18nMessages
    // @visibility external
    //<
    asynchGroupingPrompt: "${loadingImage}&nbsp;Grouping data...",

    //> @attr listGrid.isGrouped (Boolean : false : R)
    // True if this listgrid is grouped, false otherwise
    //
    // @group grouping
    // @visibility external
    // @see     groupBy
    //<
    
    //> @attr listGrid.nullGroupTitle (String : '-none-' : IRW)
    // Default alias to use for groups with no value
    //
    // @group grouping
    // @visibility external
    // @see     groupBy
    //<
    nullGroupTitle: "-none-",
    
    //> @attr listGrid.groupByField (String | Array of String : see below : IR)
    // List of fields to group grid records. If only a single field is used, that field
    // may be specified as a string. After initialization, use +link{listGrid.groupBy()} 
    // to update the grouping field list, instead of modifying groupByField directly.
    // @group grouping
    // @visibility external
    // @see groupBy
    // @example dynamicGrouping
    //<
    

    // ----------------------
    // Value icons
    // The valueIcons object is a mapping between values and image URLs - when specified
    // we show the valueIcon image either next to, or instead of the normal cell value.
    
    //> @attr listGridField.valueIcons (object : null : IRW)
    // This property is a mapping between data values to +link{SCImgURL, SCImgURLs} for icons to display
    // for those data values.  For example, given a field named "status" with possible values
    // "Normal", "Slow", "Offline", the follow definition would show various icons for that
    // field:
    // <P>
    // <pre>
    // fields : [
    //     { name:"status", 
    //       valueIcons: {
    //           Normal : "greenIcon.png",
    //           Slow : "yellowIcon.png",
    //           Offline : "redIcon.png"
    //       }
    //     },
    //     ... other fields ...
    // ]
    // </pre>
    // <P>
    // <code>valueIcons</code> can either be displayed alongside the normal value or can
    // replace the normal field value so that only the icon is shown.  See
    // +link{listGridField.showValueIconOnly}.
    // <P>
    // If inline editing is enabled for this field, editors displayed for this field will also
    // show valueIcons.  This may be overridden by explicitly setting
    // +link{listGridField.editorValueIcons}.
    // <P>
    // Note that the following attributes related to valueIcon styling will also be picked up
    // by the editor from the ListGridField object unless explicitly specified via the
    // equivalent <code>editor_</code> attributes:<br>
    // +link{listGridField.valueIconWidth}<br>
    // +link{listGridField.valueIconHeight}<br>
    // +link{listGridField.valueIconSize}<br>
    // +link{listGridField.valueIconLeftPadding}<br>
    // +link{listGridField.valueIconRightPadding}<br>            
    // +link{listGridField.imageURLPrefix}<br>    
    // +link{listGridField.imageURLSuffix}
    //
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGrid.valueIconSize (number : 16 : IRW)
    // Default width and height of value icons for this ListGrid.
    // Can be overridden at the listGrid level via explicit +link{ListGrid.valueIconWidth} and
    // +link{ListGrid.valueIconHeight}, or at the field level via +link{ListGridField.valueIconSize},
    // +link{ListGridField.valueIconWidth} and {ListGridField.valueIconHeight}
    // @visibility external
    // @group imageColumns
    // @see ListGrid.valueIconWidth
    // @see ListGrid.valueIconHeight
    // @see ListGridField.valueIconSize
    //<
    valueIconSize:16,
    
    
    //> @attr listGrid.valueIconWidth (number : null : IRW)
    // Width for value icons for this listGrid.
    // Overrides +link{ListGrid.valueIconSize}.
    // Can be overridden at the field level
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGrid.valueIconHeight (number : null : IRW)
    // Height for value icons for this listGrid.
    // Overrides +link{ListGrid.valueIconSize}.
    // Can be overridden at the field level
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGridField.valueIconSize (number : null : IRW)
    // Default width and height of value icons in this field.
    // Takes precedence over valueIconWidth, valueIconHeight and valueIconSize specified at
    // the ListGrid level.
    // Can be overridden via +link{ListGridField.valueIconWidth} and {ListGridField.valueIconHeight}
    // @visibility external
    // @group imageColumns
    // @see ListGrid.valueIconSize
    // @see ListGridField.valueIconWidth
    // @see ListGridField.valueIconHeight
    //<
    
    //> @attr listGridField.valueIconWidth (number : null : IRW)
    // Width for value icons for this listGrid field.
    // Overrides +link{ListGrid.valueIconSize}, +link{ListGrid.valueIconWidth}, and
    // +link{ListGridField.valueIconSize}.
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGridField.valueIconHeight (number : null : IRW)
    // Height for value icons for this listGrid field.
    // Overrides +link{ListGrid.valueIconSize}, +link{ListGrid.valueIconHeight}, and
    // +link{ListGridField.valueIconSize}.
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr   listGridField.valueIconLeftPadding (number : null : IRW)
    // How much padding should there be on the left of valueIcons for this field
    // Overrides +link{listGrid.valueIconLeftPadding}
    // @group imageColumns
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    
    //> @attr   listGridField.valueIconRightPadding (number : null : IRW)
    // How much padding should there be on the right of valueIcons for this field
    // Overrides +link{listGrid.valueIconRightPadding}
    // @group imageColumns
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    
    //> @attr listGridField.editorValueIcons (object : null : IRW)
    // When some cell in this field is being edited, setting this property will specify the
    // value icons to display in the cell's editor. If unset, the editor's valueIcons
    // will be determined in the same way as it would be for a static cell.
    // @group imageColumns
    // @visibility external
    //<

    //> @attr listGridField.editorValueIconWidth (number : null : IRW)
    // When some cell in this field is being edited, setting this property will specify the
    // width for value icons in the cell's editor. If unset, the editor's valueIcon width and
    // height will be determined in the same way as it would be for a static cell.
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGridField.editorValueIconHeight (number : null : IRW)
    // When some cell in this field is being edited, setting this property will specify the
    // height for value icons in the cell's editor. If unset, the editor's valueIcon width and
    // height will be determined in the same way as it would be for a static cell.
    // @group imageColumns
    // @visibility external
    //<    
    
    //> @attr listGridField.showValueIconOnly (boolean : null : IRW)
    // If this field has a valueIcons property specified, setting this property causes 
    // the valueIcon for each value to be displayed in the cell without also showing the
    // record's value for the field.
    // <P>
    // If unset the default behavior is to show the icon only if an explicit valueMap is
    // specified as well in addition to a valueIcons map, otherwise show both the valueIcon and
    // value for the cell.
    // <P>
    // Note that if this field is editable +link{FormItem.showValueIconOnly} will be passed
    // through to editors displayed in this field.
    //
    // @group imageColumns
    // @see listGridField.valueIcons
    // @see listGridField.suppressValueIcon
    // @visibility external
    //<
    
    // NOTE: showValueIconOnly: the use cases are:
    // - represent a value as an icon only to minimize space
    // - show text, but add an icon as decoration, either to all values, or to emphasize some
    //   values for quicker scanning
    // The property 'showValueIconOnly' allows the developer to explicitly show the valueIcon
    // with or without text.  If showValueIconOnly is unset, we make the assumption that:
    // - if the field is *not* constrained to a fixed set of values (has no valueMap), there's 
    //   no way to have icons for all the values, so the purpose of the icons is to add
    //   emphasis to certain values [so we show both text and images]
    // - otherwise the developer has an icon for every possible value, so there is no need for
    //   the value to also be displayed - we size the field large enough to accommodate the icon
    //   only, and suppress the text.
        
    //> @attr   listGridField.suppressValueIcon (boolean : null : IRW)
    // If this field has a valueIcons property specified, setting this property to true will 
    // prevent the valueIcon being written out into this field's cells.
    // <P>
    // Note this property may also be set to false to avoid showing the standard
    // +link{listGrid.booleanTrueImage} and +link{listGrid.booleanFalseImage} for fields of type
    // <code>boolean</code>.
    //
    // @group imageColumns
    // @see listGridField.valueIcons
    // @see listGridField.showValueIconOnly
    // @visibility external
    //<
    
    //> @attr   listGridField.valueIconOrientation (string : null : IRW)
    // If we're showing a valueIcon for this field should it appear to the left or the right
    // of the text?  By default the icon will appear to the left of the textual value - 
    // set this to "right" to show the icon on the right of the text.
    // Has no effect if +link{listGridField.showValueIconOnly} is true
    // @visibility external
    // @group imageColumns
    //<
    
    
    //> @attr   listGrid.valueIconLeftPadding (number : 2 : IRW)
    // How much padding should there be on the left of valueIcons by default
    // Can be overridden at the field level
    // @group imageColumns
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    valueIconLeftPadding:2,
    
    //> @attr   listGrid.valueIconRightPadding (number : 2 : IRW)
    // How much padding should there be on the right of valueIcons by default
    // @group imageColumns
    // Can be overridden at the field level    
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    valueIconRightPadding:2,
    
    // ------------
    // Hilite Icons
    // ------------
     
    //> @attr listGrid.hiliteIcons (Array of String : ["[SKINIMG]/Dialog/notify.png", "[SKINIMG]/Dialog/warn.png", "[SKINIMG]/actions/approve.png"] : IR)
    // @include dataBoundComponent.hiliteIcons
    // @group hiliting
    // @visibility external
    //<

    //> @attr listGrid.hiliteIconPosition (HiliteIconPosition : "before" : IR)
    // @include dataBoundComponent.hiliteIconPosition
    // @group hiliting
    // @visibility external
    //<

    //> @attr listGrid.hiliteIconSize (number : 12 : IRW)
    // @include dataBoundComponent.hiliteIconSize 
    // @group hiliting
    // @see hiliteIconWidth
    // @see hiliteIconHeight
    // @see ListGridField.hiliteIconSize
    // @visibility external
    //<
        
    //> @attr listGrid.hiliteIconWidth (number : null : IRW)
    // @include dataBoundComponent.hiliteIconWidth
    // @group hiliting
    // @visibility external
    //<

    //> @attr listGrid.hiliteIconHeight (number : null : IRW)
    // @include dataBoundComponent.hiliteIconHeight
    // @group hiliting
    // @visibility external
    //<
    
    //> @attr   listGrid.hiliteIconLeftPadding (number : 2 : IRW)
    // @include dataBoundComponent.hiliteIconLeftPadding
    // @group hiliting
    // @visibility external
    //<

    //> @attr   listGrid.hiliteIconRightPadding (number : 2 : IRW)
    // @include dataBoundComponent.hiliteIconRightPadding
    // @group hiliting
    // @visibility external
    //<
  
    //> @attr listGridField.hiliteIconPosition (HiliteIconPosition : null : IR)
    // When +link{listGrid.hiliteIcons} are present, where the hilite icon will be placed 
    // relative to the field value.  See +link{type:HiliteIconPosition}.
    // Overrides +link{listGrid.hiliteIconPosition}.
    // @group hiliting
    // @visibility external
    //<

    //> @attr listGridField.hiliteIconSize (number : null : IRW)
    // Default width and height of +link{listGrid.hiliteIcons, hilite icons} in this field.
    // Takes precedence over hiliteIconWidth, hiliteIconHeight and hiliteIconSize specified at
    // the component level.
    // Can be overridden via +link{ListGridField.hiliteIconWidth} and 
    // +link{ListGridField.hiliteIconHeight}
    // @group hiliting
    // @see ListGrid.hiliteIconSize
    // @see ListGridField.hiliteIconWidth
    // @see ListGridField.hiliteIconHeight
    // @visibility external
    //<

    //> @attr listGridField.hiliteIconWidth (number : null : IRW)
    // Width for hilite icons for this field.
    // Overrides +link{listGrid.hiliteIconSize}, +link{listGrid.hiliteIconWidth}, and
    // +link{ListGridField.hiliteIconSize}.
    // @group hiliting
    // @visibility external
    //<
        
    //> @attr listGridField.hiliteIconHeight (number : null : IRW)
    // Height for hilite icons for this field.
    // Overrides +link{listGrid.hiliteIconSize}, +link{listGrid.hiliteIconHeight}, and
    // +link{ListGridField.hiliteIconSize}.
    // @group hiliting
    // @visibility external
    //<

    //> @attr   listGridField.hiliteIconLeftPadding (number : null : IRW)
    // How much padding should there be on the left of +link{DataBoundComponent.hiliteIcons, hilite icons} 
    // for this field?
    // Overrides +link{listGrid.hiliteIconLeftPadding}
    // @group hiliting
    // @visibility external
    //<
        
    //> @attr   listGridField.hiliteIconRightPadding (number : null : IRW)
    // How much padding should there be on the right of +link{DataBoundComponent.hiliteIcons, hilite icons} 
    // for this field?
    // Overrides +link{listGrid.hiliteIconRightPadding}
    // @group hiliting
    // @visibility external
    //<


    
    
    //> @attr   ListGridField.imageURLPrefix (string : null : IRWA)
    // If this field has type [+link{type:ListGridFieldType}] set to <code>"image"</code>
    // and the URL for the image displayed is not absolute, the path of the URL will be relative 
    // to this string<br>
    // Alternatively, if this field displays any valueIcons, this prefix will be applied to
    // the beginning of any +link{ListGridField.valueIcons} when determining the
    // URL for the image.
    // @group imageColumns
    // @visibility external
    // @example imageType
    //<
    
    //> @attr   ListGridField.imageURLSuffix (string : null : IRWA)
    // If any cells in this field are showing a value icon (see: +link{ListGridField.valueIcons})
    // or this is has +link{type:ListGridFieldType} set to <code>"image"</code>, this the value
    // of this property will be appended to the end of the URL for the icon displayed.<br>
    // Typical usage might be to append a file type such as <code>".gif"</code> to the
    // filename of the image.<br>
    // For editable fields, this property will also be passed through to any editors as
    // +link{FormItem.imageURLSuffix}.
    // @group imageColumns
    // @visibility external
    // @example imageType
    //<    
    
    //> @attr   ListGridField.editorImageURLPrefix (string : null : IRWA)
    // When some cell in this field is being edited, this property can be used to apply 
    // an explicit +link{FormItem.imageURLPrefix} to the editor in question. 
    // This can be used to modify the valueIcons within the editor.<br>
    // If unset, but +link{ListGridField.imageURLPrefix} is specified, that will be used
    // instead.
    // @group editing
    // @visibility external
    //<
    
    //> @attr   ListGridField.editorImageURLSuffix (string : null : IRWA)
    // When some cell in this field is being edited, this property can be used to apply 
    // an explicit +link{FormItem.imageURLSuffix} to the editor in question. 
    // This can be used to modify the valueIcons within the editor.<br>
    // If unset, but +link{ListGridField.imageURLPrefix} is specified, that will be used
    // instead.
    // @group editing
    // @visibility external
    //<

  	//>	@attr	listGrid.imageSize (number : 16 : IRW)
    // Default size of thumbnails shown for fieldTypes image and imageFile.  Overrideable on a
    // per-field basis via +link{attr:ListGridField.imageSize} or
    // +link{attr:ListGridField.imageWidth}/+link{attr:ListGridField.imageHeight}
    // 
    // @group imageColumns
    // @visibility external
	//<
    imageSize: 16,

  	//>	@attr	listGridField.imageSize (number : 16 : IRW)
    // Size of images shown for fieldTypes image and imageFile in this field.
    // This setting overrides the global ListGrid default +link{attr:ListGrid.imageSize}.
    // <P>
    // If set to a String, assumed to be a property on each record that specifies the image
    // height.  For example, if <code>field.imageSize</code> is "logoSize",
    // <code>record.logoSize</code> will control the size of the image.
    //
    // @see attr:ListGridField.imageWidth
    // @see attr:ListGridField.imageHeight
    // 
    // @group imageColumns
    // @visibility external
	//<

  	//>	@attr	listGridField.imageWidth (number : 16 : IRW)
    // Width of images shown for fieldTypes image and imageFile in this field.
    // <P>
    // If set to a String, assumed to be a property on each record that specifies the image
    // width.  For example, if <code>field.imageWidth</code> is "logoWidth",
    // <code>record.logoWidth</code> will control the width of the image.
    //
    // @see attr:ListGrid.imageSize
    // @see attr:ListGridField.imageSize
    // @see attr:ListGridField.imageHeight
    // 
    // @group imageColumns
    // @visibility external
	//<

  	//>	@attr	listGridField.imageHeight (number : 16 : IRW)
    // Height of image shown for fieldTypes image and imageFile in this field.
    // <P>
    // If set to a String, assumed to be a property on each record that specifies the image
    // height.  For example, if <code>field.imageHeight</code> is "logoHeight",
    // <code>record.logoHeight</code> will control the height of the image.
    //
    // @see attr:ListGrid.imageSize
    // @see attr:ListGridField.imageSize
    // @see attr:ListGridField.imageWidth
    // 
    // @group imageColumns
    // @visibility external
	//<

    // ListGridField
    // ---------------------------------------------------------------------------------------

	//  -- Define the 'listGridField' pseudo class for doc

	//>	@object ListGridField
    // An ordinary JavaScript object containing properties that configures the display of
    // and interaction with the columns of a +link{ListGrid}.
	//
	// @see ListGrid.fields
	// @see ListGrid.setFields
	// @treeLocation Client Reference/Grids/ListGrid
	// @visibility external    
	//<

    //> @type ListGridFieldType
    // ListGrids format data for viewing and editing based on the <i>type</i> attribute of the
    // field.  This table describes how the ListGrid deals with the various built-in types.
    //
    // @value "text"    Simple text rendering for view.  For editing a text entry field is shown.
    // If the length of the field (as specified by the +link{attr:dataSourceField.length}
    // attribute) is larger than the value specified by +link{attr:listGrid.longTextEditorThreshold}, a
    // text input icon is shown that, when clicked on (or field is focused in) opens a larger
    // editor that expands outside the boundaries of the cell (textarea by default, but
    // overrideable via +link{ListGrid.longTextEditorType}).
    //
    // @value "boolean" For viewing and editing a checkbox is shown with a check mark for the
    // <code>true</code> value and no check mark for the <code>false</code> value. This behavior
    // may be suppressed by setting link{listGridField.suppressValueIcon} for the field. See
    // +link{ListGrid.booleanTrueImage} for customization.
    //
    // @value "integer" Same as <code>text</code>.  Consider setting
    // +link{listGridField.editorType,editorType} to use a +link{SpinnerItem}.
    //
    // @value "float" Same as <code>text</code>.  Consider setting
    // +link{listGridField.editorType,editorType} to use a +link{SpinnerItem}.
    //
    // @value "date" Field value should be a <code>Date</code> instance representing a logical
    // date, with no time of day information.  See +link{group:dateFormatAndStorage} for 
    // details of the logical date type and how it is represented and manipulated.
    // <P>
    // Dates will be formatted using +link{listGridField.dateFormatter,ListGridField.dateFormatter} 
    // if specified, otherwise
    // +link{ListGrid.dateFormatter,ListGrid.dateFormatter}.  
    // If both these attributes are unset, dates are formatted
    // using the standard +link{Date.setShortDisplayFormat(),short display format} for dates.
    // <P>
    // For editing, by default a +link{DateItem} is used with +link{DateItem.useTextField} set
    // to true, providing textual date entry plus a pop-up date picker. The
    // +link{DateItem.dateFormatter, dateFormatter} and +link{DateItem.inputFormat, inputFormat}
    // for the editor will be picked up from the ListGridField, if specified.
    //
    // @value "time" Field value should be a <code>Date</code> instance representing a logical
    // time, meaning time value that does not have a specific day and also has no timezone.  See
    // +link{group:dateFormatAndStorage} for details of the logical time type and how it is
    // represented and manipulated.
    // <P>
    // Times will be formatted using +link{listGridField.timeFormatter,ListGridField.timeFormatter}
    // if specified, 
    // otherwise +link{ListGrid.timeFormatter,ListGrid.timeFormatter}.
    // <P>
    // If both these attributes are unset, times are formatted using the standard 
    // +link{Time.shortDisplayFormat,short display format} for times.
    // <P>
    // For editing, by default a +link{TimeItem} is used. The
    // +link{TimeItem.timeFormatter, timeFormatter} for the editor will be picked up from 
    // the ListGridField, if specified.
    //
    // @value "datetime" Field value should be a <code>Date</code> instance representing a
    // specific date and time value.  See +link{group:dateFormatAndStorage} for details of the
    // datetime type and how it is represented and manipulated.
    // <P>
    // Dates will be formatted using +link{listGridField.dateFormatter,ListGridField.dateFormatter}
    // if specified, otherwise
    // +link{ListGrid.datetimeFormatter,ListGrid.datetimeFormatter}.  
    // If both these attributes are unset, dates are formatted
    // using the standard +link{Date.setShortDateTimeDisplayFormat(),short display format} for
    // datetime values.
    // <P>
    // For editing, by default a +link{DateTimeItem} is used, providing textual date entry plus
    // a pop-up date picker.  The +link{DateItem.dateFormatter, dateFormatter} and
    // +link{DateItem.inputFormat, inputFormat} for the editor will be picked up from the
    // ListGridField, if specified.
    //
    // @value "sequence" Same as <code>text</code>
    //
    // @value "link"     Renders a clickable html link (using an HTML anchor tag: &lt;A&gt;).
    // The target URL is the value of the field, which is also the default display value.  You
    // can override the display value by setting +link{attr:listGridRecord.linkText} or
    // +link{attr:listGridField.linkText}.
    // <P>
    // Clicking the link opens the URL in a new window by default.  To change this behavior,
    // you can set <code>field.target</code>, which works identically to the "target"
    // attribute on an HTML anchor (&lt;A&gt;) tag.  See +link{listGridField.target} for more
    // information.
    // <P>
    // In inline edit mode, this type works like a text field.
    // <P>
    // To create a link not covered by this feature, consider using
    // +link{listGridField.formatCellValue()} along with +link{Canvas.linkHTML()}, or simply
    // +link{listGrid.getCellStyle,styling the field} to look like a link, and providing
    // interactivity via +link{listGridField.recordClick,field.recordClick()}.
    //
    // @value "image"   Renders a different image in each row based on the value of the field.  If
    // this URL is not absolute, it is assumed to be relative to
    // +link{ListGridField.imageURLPrefix} if specified. The size of the image is controlled by
    // +link{attr:listGridField.imageSize}, +link{attr:listGridField.imageWidth},
    // +link{attr:listGridField.imageHeight} (and by the similarly-named global default
    // attributes on the ListGrid itself).
    // <P>
    // You can also specify the following attributes on the field: <code>activeAreaHTML</code>, and
    // <code>extraStuff</code> - these are passed to +link{method:canvas.imgHTML} to generate the
    // final URL.
    // <P>
    // Note if you want to display icons <b>in addition to</b> the normal cell value, you
    // can use +link{listGridField.valueIcons,valueIcons} instead.
    // 
    // @value "icon" Shows +link{listGridField.icon,field.icon} in every cell, and also in the
    // header.  Useful for a field that is used as a button, for example, launches a detail
    // window or removes a row.  Implement a +link{listGridField.recordClick,field.recordClick}
    // to define a behavior for the button.
    // <P>
    // NOTE: for a field that shows different icons depending on the field value, see
    // +link{listGridField.valueIcons}.
    // <P>
    // <code>type:"icon"</code> also defaults to a small field width, accommodating just the icon
    // with padding, and to a blank header title, so that the header shows the icon only.  
    // <P>
    // +link{listGridField.iconWidth,field.iconWidth} and related properties configure
    // the size of the icon both in the header and in body cells.
    // <P> 
    // If you want the icon to appear only in body cells and not in the header, set
    // +link{listGridField.cellIcon,field.cellIcon} instead, leaving field.icon null.
    // 
    // @value "binary"  For viewing, the grid renders a 'view' icon (looking glass) followed by a
    // 'download' icon and then the name of the file is displayed in text.  If the user clicks the
    // 'view' icon, a new browser window is opened and the file is streamed to that browser
    // instance, using +link{dataSource.viewFile()}.  For images and other file types with
    // known handlers, the content is typically displayed inline - otherwise the browser will
    // ask the user how to handle the content.  If the download icon is clicked,
    // +link{dataSource.downloadFile()} is used to cause the browser to show a "save" dialog.
    // There is no inline editing mode for this field type.
    //
    // @value "imageFile"   Same as <code>binary</code>
    //
    // @value "summary" Show a calculated summary based on other field values within the
    //  current record. See +link{listGridField.recordSummaryFunction} for more information
    // 
    // 
    // @see attr:listGridField.type
    // @see type:FieldType
    // @visibility external
    // @example gridsDataTypes
    //<    

	//> @attr listGridField.type (ListGridFieldType : "text" : [IR])
    //  ListGrids picks a renderer for the view and edit mode of a field based on this attribute.
    //  See +link{ListGridFieldType} for a summary of how types are rendered.
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
	//  @group  appearance
	//  @visibility external
	//<
    
	//> @attr listGridField.name (identifier : null : [IR])
	// Name of this field.  Must be unique within this ListGrid as well as a valid JavaScript identifier,
    // as specified by ECMA-262 Section 7.6 (the <var class="smartclient">+link{String.isValidID()}</var>
    // <var class="smartgwt">StringUtil.isValidID()</var> function can be used to test whether
    // a name is a valid JavaScript identifier).
    // <P>
    // The name of field is also the property in each record which holds the value for that
    // field.
    // <P>
    // If a +link{listGrid.dataSource} is specified and the DataSource has a field with the
    // same name, the ListGridField and DataSourceField are merged so that properties on the
    // ListGridField
    // 
	// @group data
	// @visibility external
	//<
	
	//> @attr listGridField.dataPath (string : null : [IRA])
	// dataPath for this field. This property allows the grid to display details of nested data
	// structures in a flat list of columns. 
	// @group data
	// @visibility external
	//<

	//> @attr listGridField.title (string : null : [IRW])
	// A title for this field, to display in the header for the field and in other
    // contexts such as the +link{canPickFields,menu for picking visible fields}.
    // <P>
    // Note: if you want to use HTML tags to affect the display of the header, you should do so
    // via +link{listGridField.headerTitle} instead so that other places where the title
    // appears in the UI are not affected.  For example, you might set <code>headerTitle</code>
    // to an empty string to suppress the header title on a narrow column, but you would retain
    // the normal title in the <code>title</code> property to avoid a blank menu item in the
    // field picker menu, +link{editHilites(),hilite editor} and other contexts.
    // <P>
    // Alternately you can specify a +link{getFieldTitle()} method on the field to return the
    // HTML for the field title.
    //
	// @group  appearance
    // @see method:listGridField.getFieldTitle()
	// @visibility external
	//<
	
	//> @attr listGridField.showTitle (boolean : null : [IRW])
	// This property may be set to <code>false</code> to explicitly suppress display of
	// the field title in the column header button for the field.
	// @visibility external
	//<

    //> @method listGridField.getFieldTitle()
    // If your derivation of the field title is more complex than specifying a static string,
    // you can specify a getFieldTitle() method on your field to return the title string.
    // Otherwise you can use the +link{title} attribute on the field to specify the title.
    // <P>
    // You can use +link{listGrid.setFieldProperties,setFieldProperties()} to dynamically
    // update the title.
    //
    // @param viewer (ListGrid) pointer back to the ListGrid
    // @param fieldNum (number) index of this field in the grid's fields array.
    // @return	(string) Field title.
    // @group appearance
    // @see attr:listGridField.title
    // @visibility external
    //<
    
    //> @attr listGridField.wrap (Boolean : null : [IRW])
    // Should the field title wrap if there is not enough space horizontally to 
    // accomodate it. (Note that this is a soft-wrap - if set the title will wrap
    // at word boundaries).
    // @visibility external
    //<

    //> @attr listGridField.target (string : "_blank" : IRW)
    // By default, clicking a link rendered by this item opens it in a new browser window.  You 
    // can alter this behavior by setting this property.  The value of this property will be 
    // passed as the value to the <code>target</code> attribute of the anchor tag used to render 
    // the link.
    // <P>
    // If you set listGridField.target to "javascript", the default behavior is to catch and 
    // consume mouse-clicks that would result in the link being followed.  Instead, the
    // +link{listGrid.cellClick()} event is fired for the containing cell.
    // 
    // @visibility external
    //<
    
	//> @method listGridField.showIf()
	// An optional +link{group:stringMethods,stringMethod} which if provided, is evaluated to
    // conditionally determine whether this field should be displayed. 
    // Evaluated on initial draw, then reevaluated on explicit 
	// calls to <code>listGrid.refreshFields()</code> or <code>listGrid.setFields()</code>. 
    // <P>
    // Use 'showIf:"false"' to set a ListGrid field to initially hidden.
    // <P>
    // Note that explicit calls to +link{listGrid.showField,grid.showField()} or hideField()
    // will wipe out the <code>showIf</code> expression, as will the end user showing and
    // hiding columns via the +link{listGrid.showHeaderContextMenu,header contextMenu}.
    // <P>
    // Also note that fields marked as +link{DataSourceField.detail,detail:true} will be hidden by
    // default even if +link{ListGrid.showDetailFields} is <code>true</code>. To show detail fields
    // inherited from a DataSource, include an explicit field definition for the field and
    // set this property to return <code>true</code>.
    //
    // @param list (ListGrid) A pointer to the listGrid containing the field
    // @param field (ListGridField) the ListGridField object
    // @param fieldNum (integer) the index of the field
    // @return (boolean) whether the field should be shown
    // 
	// @group appearance
	// @see method:ListGrid.refreshFields
	// @visibility external
	//<

    //> @attr listGridField.frozen (boolean : null : IR)
    // Whether this field should be "frozen" for the purposes of horizontal scrolling.  See
    // +link{group:frozenFields}.
    // @group frozenFields
    // @visibility external
    //<

    //> @attr listGridField.canFreeze (boolean : null : IR)
    // Whether this field should display freezing/unfreezing options in its header context menu. 
    // See +link{group:frozenFields}.
    // @see method:listGrid.getHeaderContextMenuItems()
    // @group frozenFields
    // @visibility external
    //<

    //> @attr listGridField.autoFreeze (boolean : null : IR)
    // Whether this field should be automatically frozen when other fields are frozen.  When
    // true, the field will be automatically frozen to the extreme of the grid.  The
    // automatically generated +link{listGrid.checkboxField, checkbox},
    // +link{listGrid.expansionField, expansion} and 
    // +link{listGrid.rowNumberField, rowNumber} fields are examples of fields that specify
    // <code>autoFreeze: true</code>.
    // <P>
    // You can control the position of this field in the array of frozen fields by providing a
    // +link{listGridField.getAutoFreezePosition} implementation.
    // @group frozenFields
    // @visibility external
    //<

    //> @method listGridField.getAutoFreezePosition()
    // When a field has +link{listGridField.autoFreeze, autoFreeze} set to true, developers can
    // implement this method to indicate where in the frozen-fields array this field should 
    // appear.
    // <P>
    // Some automatically generated fields, such as 
    // +link{listGrid.rowNumberField, rowNumberField}, 
    // +link{listGrid.expansionField, expansionField} and
    // +link{listGrid.checkboxField, checkboxField}, provide default implementations of this 
    // method.
    // @return (number) the index at which this autoFreeze field should appear in the frozen body
    // @group frozenFields
    //<

    //> @attr listGridField.canHide (boolean : null : IR)
    // Whether this field can be hidden via the header context menu. 
    //
    // @see method:listGrid.getHeaderContextMenuItems()
    // @group appearance
    // @visibility external
    //<

    //> @attr listGridField.canDragResize (boolean : null : IR)
    // Whether this field can be dragResized using the mouse.  If unset, the default behavior
    // is governed by +link{listGrid.canResizeFields}.
    // @visibility external
    //<

    //> @attr listGridField.canReorder (boolean : null : IR)
    // Whether this field can be reordered using the mouse.  If unset, the default behavior is
    // governed by +link{listGrid.canReorderFields}.  Note that setting this property to 
    // <code>false</code> will lock this field from being moved - that is, the user is 
    // prevented from moving this field directly by dragging with the mouse, or by dropping another 
    // field onto this field.  However, unless this field is at the extreme edge of the grid, 
    // or all fields between it and the extreme edge of the grid are also
    // <code>canReorder: false</code>, (ie, if there are unlocked fields on either side of this
    // field), then it is possible that this locked field may be 
    // reordered automatically, as a result of the user dragging one unlocked field onto another 
    // unlocked field.
    // @visibility external
    //<

    //> @attr listGridField.ignoreKeyboardClicks (boolean : null : IRW)
    // If the user is navigating through the grid using the keyboard, record click or double click
    // events may be generated via keyboard interactions (see +link{listGrid.generateClickOnSpace}, 
    // +link{listGrid.generateClickOnEnter}, +link{listGrid.generateDoubleClickOnSpace}, 
    // +link{listGrid.generateDoubleClickOnEnter} and +link{listGrid.arrowKeyAction}). 
    // <P>
    // These synthetic events have both a target row and column.
    // Setting this flag to true ensures that this field will never be considered the target for
    // a keyboard click event.
    // @group events
    // @visibility external
    //<
    
    

    //> @attr listGridField.excludeFromState
    // @include dataSourceField.excludeFromState
    // @see ListGrid.getViewState()
    //<

    // Grid, Group and Record-level summaries
    // ---------------------------------------------------------------------------------------

    //> @attr listGridField.showGridSummary (Boolean : null : IR)
    // If +link{listGrid.showGridSummary} is true, should this field show a summary value.
    // If unset, this field will show a summary value in the summary row if an
    // explicit +link{listGridField.summaryFunction} is specified or if a
    // +link{SimpleType.getDefaultSummaryFunction(),default summary function} is defined
    // for the specified field type.
    // @visibility external
    //<
    
    //> @attr listGridField.showGroupSummary (boolean : null : IR)
    // If +link{listGrid.showGroupSummary} is true, should this field show a summary value
    // in a summary row when the grid is grouped?
    // If unset, this field will show a summary value in the summary row if an
    // explicit +link{listGridField.summaryFunction} is specified or if a
    // +link{SimpleType.getDefaultSummaryFunction(),default summary function} is defined
    // for the specified field type.
    // @visibility external
    //<
    
    //> @attr listGridField.summaryFunction (SummaryFunction  or Array of SummaryFunction : null : IR)
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true, 
    // this attribute can be used to specify
    // an explicit +link{type:SummaryFunction} for calculating the summary value to
    // display.
    // <P>
    // If an array of summaryFunctions is specified, they will be executed in turn and the
    // grid will show multiple summary rows at the grid or group level (or both)
    // containing the resulting values.
    // @visibility external
    //<
    
    //> @attr listGridField.summaryValueTitle (String : null : IR)
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true and the
    // +link{listGridField.summaryFunction} is set to <code>"title"</code>, this attribute may be
    // set to a string to display in the group and/or grid summary. If unspecified the
    // +link{listGridField.title} will show up in the summary.
    // @visibility external
    //<
    
    //> @method listGridField.getGridSummary() [A]
    // If +link{listGrid.showGridSummary} is true, and this method is specified it will be
    // called to generate the summary value to be displayed in the grid summary row. Note that
    // this is called instead of making use of the +link{listGridField.summaryFunction}.
    // <P>
    // As with +link{listGrid.getGridSummary()} this method may return an array of results - 
    // in this case each result will show up in a separate row in the +link{listGrid.summaryRow}
    // grid.
    // <P>
    // If this grid is grouped, and +link{listGrid.showGroupSummary} is true, this method
    // will be passed a third parameter - an array of group-level summaries.
    // @param records (Array of ListGridRecord) records for which a summary is being generated
    // @param field (ListGridField) pointer to the field for which summary value is being generated
    // @param [groupSummaries] (Array of objects) If this grid is grouped and
    //  +link{listGrid.showGridSummary} is specified, this parameter contains an array of already-
    //  calculated summary values for each group in the grid. Each element in this array will
    //  be an object containing calculated summary values for each field in the grid, as well as
    //  a specified groupValue and groupName, allowing the developer to determine which group this
    //  summary value comes from
    // @return (any) summary value to display.
    // @visibility external
    //<
    
    //> @attr listGridField.formatGridSummary (stringMethod : null : IR)
    // Optional stringMethod to format the summary value displayed
    // in the +link{listGrid.showGridSummary,grid summary}.
    // Takes a single parameter <code>value</code> and should return the formatted version
    // of that value. If specified this will be applied instead of any formatting logic applied
    // via +link{listGridField.formatCellValue()}, +link{listGrid.formatCellValue()}, etc.
    // <P>
    // Note that for fields with a specified summary function of "count", if no custom formatting
    // is applied, we default to formatting the count value by appending 
    // <code>field.pluralTitle</code> if defined, otherwise <code>field.title</code> to the
    // numeric count value returned by the standard count function. To change this behavior for
    // such fields, specify an explicit 'formatGridSummary' and/or 'formatGroupSummary' method
    // @visibility external
    //<
    
    //> @method listGridField.getGroupSummary() [A]
    // If +link{listGrid.showGroupSummary} is true, and this method is specified it will be
    // called to generate the field summary value to be displayed for each group level summary row.
    // Note that this is called instead of making use of the +link{listGridField.summaryFunction}.
    // <P>
    // This method may return an array of results - in this case the group will show multiple summary
    // rows, with each entry in the array showing up in a different record.
    //
    // @param records (Array of ListGridRecord) records for which a summary is being generated
    //  (so all records in the group).
    // @param field (ListGridField) pointer to the field for which summary value is being generated
    // @param [groupNode] (object) object with specified groupValue and groupName for this group
    // @return (any) summary value to display
    // @visibility external
    //<
    
    //> @attr listGridField.formatGroupSummary (stringMethod : null : IR)
    // Optional stringMethod to format the group level summary values for this field displayed via
    // +link{listGrid.showGroupSummary}.
    // Takes a single parameter <code>value</code> and should return the formatted version
    // of that value.  If specified this will be applied instead of any formatting logic applied
    // via +link{listGridField.formatCellValue()}, +link{listGrid.formatCellValue()}, etc.
    // <P>
    // Note that for fields with a specified summary function of "count", if no custom formatting
    // is applied, we default to formatting the count value by appending 
    // <code>field.pluralTitle</code> if defined, otherwise <code>field.title</code> to the
    // numeric count value returned by the standard count function. To change this behavior for
    // such fields, specify an explicit 'formatGridSummary' and/or 'formatGroupSummary' method
    // @visibility external
    //<

    //> @method listGridField.getRecordSummary() [A]
    // Only applies to +link{listGridFieldType,summary type} fields. If specified , this
    // method will be called to generated the record summary value to be displayed for each row
    // in this field.
    // <P>
    // Note that if specified, this is called instead of making use of the
    // +link{listGridField.recordSummaryFunction}.
    // <P>
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true, this
    // field's value in the summary row[s] will still be calculated by calling this method.
    // In this case, the record object passed in will contain summary values for each field.
    // If custom handling is required for this case, it may be detected by checking the
    // record object's +link{listGridRecord.isGroupSummary} and +link{listGridRecord.isGridSummary}
    // attributes.
    // @param record (ListGridRecord) record for which a summary is being generated
    // @param field (ListGridField) pointer to the summary type field
    // @return (any) summary value to display
    // @visibility external
    //<
    
    //> @attr listGridField.recordSummaryFunction (RecordSummaryFunction : null : IR)
    // Only applies to fields of type <code>"summary"</code>.
    // Fields of this type will display a calculated value based on the other field values
    // within the current record.
    // <P>
    // This attribute specifies how the summary field value will be calculated. See 
    // +link{type:RecordSummaryFunction} for valid options.
    // <P>
    // A subset of the ListGrid's fields will be passed to the RecordSummaryFunction. 
    // Which fields to include is determined based on +link{listGridField.includeInRecordSummary}
    // <P>
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true, this
    // field's value in the summary row[s] will still be calculated by calling this method.
    // In this case, the record object passed in will contain summary values for each field.
    // If custom handling is required for this case, it may be detected by checking the
    // record object's +link{listGridRecord.isGroupSummary} and +link{listGridRecord.isGridSummary}
    // attributes.
    // @visibility external
    //<
    
    //> @attr listGridField.partialSummary (boolean : null : IR)
    // Only applies to fields of type <code>"summary"</code>.
    // This attribute is set on a summary field, when calculating the summary value from
    // some record, the summary function will only be passed the fields before this summary field.
    // This may be useful for displaying running totals across a record.
    // <P>
    // Note that this feature would typically be used with
    // +link{listGrid.canReorderFields,canReorderFields:false}
    // @visibility external
    //<
    
    //> @attr listGridField.includeInRecordSummary (boolean : null : IR)
    // If a listGrid is showing a field of type summary, should this field be passed to the
    // recordSummaryFunction when calculating the summary value to display.
    // If unset, fields are included if they are of type "integer" or "float" only (since most
    // summary functions perform numeric calculations). See also
    // +link{listGridField.includeInRecordSummaryFields}.
    // @visibility external
    //<
    
    //> @attr listGridField.includeInRecordSummaryFields (array of fieldNames : null : IR)
    // If this listGrid has any fields of type <code>"summary"</code> and
    // this field will be +link{listGridField.includeInRecordSummary,included} in summary calculations
    // by default, this attribute provides an opportunity to explicitly specify which summary fields
    // the record should be displayed in.
    // <P>
    // Specified as an array of fieldNames. If set, this field value will only be included for
    // record summary value calculations for summary fields whos name is included in this array.
    // @visibility external
    //<

    //> @attr listGridField.applyAfterSummary (Boolean : null : IRW)
    // If +link{listGrid.field.userFormula} is set for this field, and this grid is showing
    // +link{listGrid.showGroupSummary,group summaries} or a
    // +link{listGrid.showGridSummary,grid summary}, this property determines what field value
    // should be present in those summary rows. Should the field apply the user-formula to the
    // calculated summary row, or should it apply a standard grid or group summary to the
    // user-formula values displayed in the grid?
    // <P>
    // Default behavior may be specified at the grid level via +link{listGrid.applyFormulaAfterSummary}
    // @visibility external
    //<

    // Header button icons
    // ---------------------------------------------------------------------------------------
    // Include all relevant docs from StatefulCanvas
    
    //> @attr listGridField.icon (SCImgURL: null : [IR])
    // Optional icon to show next to the title for this field. 
    // Should be set to a URL to an image. Relative paths will be evaluated starting at
    // the imgDir of this component. This URL is partial - it may be updated to indicate
    // the current disabled (etc) state of the field.
    // <p>
    // If +link{listGridField.type,field.type} is set to "icon", this icon will also be shown
    // in every cell of this field - see also +link{listGridField.cellIcon,field.cellIcon}.
    // <p>
    // To change this property after fields have been passed to +link{listGrid.setFields()},
    // use +link{listGrid.setFieldIcon()}.
    // 
    // @visibility external
    //<

    //> @attr listGridField.iconSize (integer : null : [IR])
    // If +link{listGridField.icon} is specified, this property can be used to specify the 
    // size of the icon to be displayed in the ListGrid header button. 
    // (See +link{StatefulCanvas.iconSize})
    // @see listGridField.icon
    // @visibility external
    //<
    
    //> @attr listGridField.iconWidth (integer : null : [IR])
    // If +link{listGridField.icon} is specified, this property can be used to specify the 
    // width of the icon to be displayed in the ListGrid header button. 
    // (See +link{StatefulCanvas.iconWidth})<br>
    // If this field is editable, and +link{ListGridField.editorIconWidth} is unset, this 
    // property will be passed onto the editors for this field as +link{FormItem.iconWidth}, 
    // which will effect the default size for +link{ListGridField.icons, icons} displayed 
    // in the editor. 
    // @see listGridField.icon
    // @see listGridField.icons
    // @visibility external
    //<

    //> @attr listGridField.iconHeight (integer : null : [IR])
    // If +link{listGridField.icon} is specified, this property can be used to specify the 
    // height of the icon to be displayed in the ListGrid header button. 
    // (See +link{StatefulCanvas.iconHeight})<br>
    // If this field is editable, and +link{ListGridField.editorIconHeight} is unset, this 
    // property will be passed onto the editors for this field as +link{FormItem.iconWidth}, 
    // which will effect the default size for +link{ListGridField.icons, icons} displayed 
    // in the editor. 
    // @see listGridField.icon
    // @see listGridField.icons
    // @visibility external
    //<

    //> @attr listGridField.iconOrientation (string : "left" : [IR])
    // If this field is showing an icon, should it appear to the left or right of the title?<br>
    // Valid options are <code>"left"</code> or <code>"right"</code>
    // @see listGridField.icon    
    // @visibility external
    //<
    // iconOrientation JS doc not included from statefulCanvas as that refers to 
    // setIconOrientation(), and we don't have an exposed way to get at the ListGrid field
    // header button at runtime.

    //> @attr listGridField.iconSpacing (int : 6 : [IR])
    // @include statefulCanvas.iconSpacing   
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showDisabledIcon (Boolean : true : [IR])
    // @include statefulCanvas.showDisabledIcon
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showRollOverIcon (Boolean : false : [IR])
    // @include statefulCanvas.showRollOverIcon
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showFocusedIcon (Boolean : false : [IR])
    // @include statefulCanvas.showFocusedIcon
    // @see listGridField.icon        
    // @visibility external
    //<
    
    //> @attr listGridField.showDownIcon (Boolean : false : [IR])
    // @include statefulCanvas.showDownIcon
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showSelectedIcon (Boolean : false : [IR])
    // @include statefulCanvas.showSelectedIcon   
    // @see listGridField.icon        
    //  @visibility external
    //<

    //> @attr listGridField.cellIcon (SCImgURL : null : [IR])
    // For a field of type:"icon" only, set the icon that appears in body cells.  Unless
    // setting +link{listGridField.icon,field.icon}, setting field.cellIcon will not show an
    // icon in the header.
    // <p>
    // To change this property after fields have been passed to +link{listGrid.setFields()},
    // use +link{listGrid.setFieldCellIcon()}.
    // 
    // @visibility external
    //<

    //> @attr listGridField.showFileInline (boolean : null : [IR])
    // For a field of type:"imageFile", indicates whether to stream the image and display it
    // inline or to display the View and Download icons.
    // 
    // @visibility external
    //<

    // FormItem icons
    // ---------------------------------------------------------------------------------------
    
    //> @attr listGridField.icons (Array of FormItemIcon Properties: null : [IRA]) 
    // If this field is editable, this property can be used to specify 
    // +link{FormItem.icons, icons} to be displayed in the editors displayed for this field
    // @group editing
    // @visibility external
    //<
    
    //> @attr listGridField.editorIconWidth (number : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconWidth}.<br>
    // If this property unset, the iconWidth property from the editor can be picked up from 
    // +link{listGridField.iconWidth} instead.
    // @see listGridField.icons
    // @group editing    
    // @visibility external
    //<

    //> @attr listGridField.editorIconHeight (number : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconHeight}.<br>
    // If this property unset, the iconHeight property from the editor can be picked up from 
    // +link{listGridField.iconHeight} instead.
    // @see listGridField.icons
    // @group editing    
    // @visibility external
    //<
    
    //> @attr listGridField.defaultIconSrc (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.defaultIconSrc}.
    // @see listGridField.icons
    // @group editing    
    // @visibility external
    //<
    
    //> @attr listGridField.iconPrompt (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconPrompt}.
    // @see listGridField.icons
    // @group editing        
    // @visibility internal
    //<

    //> @attr listGridField.iconHSpace (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconHSpace}.
    // @see listGridField.icons
    // @group editing        
    // @visibility internal
    //<    
    
    //> @attr listGridField.iconVAlign (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconVAlign}.
    // @see listGridField.icons
    // @group editing        
    // @visibility external
    //<    

    // editor picker icon
    
    //> @attr listGridField.showPickerIcon (boolean : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.showPickerIcon}.
    // @group editing    
    // @visibility pickerIcon
    //<
    
    //> @attr listGridField.pickerIconSrc (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.pickerIconSrc}.
    // @group editing        
    // @visibility pickerIcon
    //<
    
    //> @attr listGridField.pickerIconWidth (integer : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.pickerIconWidth}.
    // @group editing        
    // @visibility pickerIcon
    //<
    
    //> @attr listGridField.pickerIconHeight (integer : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.pickerIconHeight}.
    // @group editing        
    // @visibility pickerIcon
    //<

    // Summary Title
    // ---------------------------------------------------------------------------------------
    
    //> @attr listGridField.summaryTitle (string : null : [IRWA])
    // Optional long summary title for this field, provided in addition to 
    // +link{listGridField.title}. This gives the developer an option to use a very short,
    // or empty title for the ListGrid column (where space may be a factor), but have a longer 
    // value available to be used elsewhere.<br>
    // By default this value will be used for the title of the context-menu item
    // for showing/hiding the listGrid field when the user right-clicks on the ListGrid header.
    //
    // @group appearance
    // @see attr:listGridField.title
    // @deprecated Rather than customizing the summaryTitle developers should typically use
    //  the +link{listGridField.headerTitle} attribute to show a different
    //  title in the column header button than the title used elsewhere.
    // @visibility external
    //<

    //> @method listGridField.getSummaryTitle() [A]
    // Optional string method to return a long summary title for this field, if a dynamic
    // summary title is required for this field.  
    //
    // @param viewer (listGrid) pointer back to the ListGrid
    // @param field (listGridField) pointer to the field object
    // @group appearance
    // @see attr:listGridField.summaryTitle
    // @see attr:listGridField.title
    // @deprecated Rather than customizing the summaryTitle developers should typically use
    //  the +link{listGridField.headerTitle} attribute to show a different
    //  title in the column header button than the title used elsewhere.
    // @visibility external
    //<

    // Header Appearance
    // ---------------------------------------------------------------------------------------
    
	//> @attr listGridField.width (Number or String : "*" : [IRW])
	// The width of this field, specified as either an absolute number of pixels,
	// a percentage of the remaining space like "25%", or "*" to split remaining space among
    // all fields which have "*".
    // <P>
    // See also +link{listGrid.minFieldWidth} to ensure no field goes below a minimum size.
    // <P>
    // Use +link{listGrid.resizeField} to programmatically change field width after creation.
    // <P>
    // Use +link{listGrid.getFieldWidth} to access the rendered field width after
    // the ListGrid is drawn.
    //
	// @group appearance
	// @visibility external
	//<

    //> @attr   listGridField.align (Alignment : null : [IRW])
    // Horizontal alignment for field's column header: "left", "right"
    // or "center". Applied to the column header title and cells by default. A separate
    // alignment for cells can be specified via +link{listGridField.cellAlign}.<br>
    // If null, values are left-aligned. If this field is editable, the
    // alignment of cells in the body will also be reflected in any editors for the field.
    //  @group  appearance
    //  @visibility external
    //<
    
    //> @attr listGridField.headerBaseStyle (CSSClass : null : [IRW])
    // Custom base style to apply to this field's header button instead of 
    // +link{listGrid.headerBaseStyle}.<br>
    // Note that depending on the header button constructor, you may have to override
    // +link{listGridField.headerTitleStyle} as well.
    // @group appearance
    // @visibility external
    //<
    
    //> @attr listGridField.headerTitleStyle (CSSClass : null : [IRW])
    // Custom titleStyle to apply to this field's header button instead of 
    // +link{listGrid.headerTitleStyle}.<br>
    // Note that this will typically only have an effect if 
    // +link{listGrid.headerButtonConstructor} is set to +link{class:StretchImgButton} or a subclass 
    // thereof.
    // @group appearance
    // @visibility external
    //<
    
    //> @attr listGridField.headerTitle (string : null : IR)
	// Optional title for the header button for this field. If specified this will be 
	// displayed in the header button instead of +link{listGridField.title} or 
	// +link{listGridField.name}. Set to an empty string to suppress the title in the
	// header button entirely.
	// @group appearance
	// @see listGridField.title
	// @visibility external
	//<

    

    // Header Spans
    // ---------------------------------------------------------------------------------------
    // - known limitations
    //   - can't reorder a column to before or after a spanned set of columns, if the spanned
    //   columns are at the start or end of the visible fields.
    //   - several uses of this.Super(), instead of the faster this.invokeSuper() approach.
    //   Attempt to use invokeSuper() failed, likely because the header is not a discrete
    //   class, but an instance of Toolbar, and my guess (Alex) is that Class.invokeSuper()
    //   doesn't handle this particular case.


    //> @attr listGrid.headerSpans (Array of HeaderSpan : null : IR)
    // Header spans are a second level of headers that appear above the normal ListGrid headers,
    // spanning one or more listGrid fields in a manner similar to a column-spanning cell in an 
    // HTML table.
    // <P>
    // A header span can be created by simply naming the fields the header should span.  The
    // example below creates a headerSpan that spans the first two fields of the ListGrid.
    // <var class="smartclient">
    // <pre>
    //    isc.ListGrid.create({
    //        headerHeight:40,
    //        fields : [
    //            { name:"field1" },
    //            { name:"field2" },
    //            { name:"field3" }
    //        ],
    //        headerSpans : [
    //            { 
    //                fields: ["field1", "field2"],
    //                title: "Field 1 and 2"
    //            }
    //        ]
    //    });
    // </pre>
    // </var>
    // <var class="smartgwt">
    // <pre>
    //      ListGrid grid = new ListGrid();
    //      grid.setHeaderHeight(40);
    //      grid.setFields(new ListGridField[] {
    //          new ListGridField("field1"),
    //          new ListGridField("field2"),
    //          new ListGridField("field3")
    //      });
    //      grid.setHeaderSpans(new HeaderSpan[] {
    //          new HeaderSpan("Field 1 and 2", new String[] {"field1", "field2"})
    //      });
    // <pre>
    // </var>
    // Header spans can be nested, allowing fields to be grouped by multiple levels of 
    // granularity. See +link{headerSpan.spans} for further information on nesting spans.
    // <P>
    // Header spans will automatically react to resizing of the headers they span, and will be
    // hidden automatically when all of the spanned fields are hidden.
    // <P>
    // Header spans appear in the +link{listGrid.header,header} area of the ListGrid, sharing space
    // with the existing headers, so it's typical to set +link{listGrid.headerHeight} to
    // approximately double its normal height when using headerSpans, or if using nested header 
    // spans, the default header height multiplied by the number of levels of header spans to be 
    // shown.
    // <P>
    // See +link{headerSpan} for many properties that allow the control of the appearance of
    // headerSpans. 
    // <var class="smartclient">
    // Note that headerSpans are created via the +link{AutoChild} pattern, hence
    // you can change the SmartClient component being used, or any of it's properties.
    // </var>
    // <P>
    // Neither headerSpans themselves nor the fields within them may be drag reordered, but other
    // unspanned headers may be.
    // <P>
    // A span can only span adjacent fields - if a span is defined and the spanned fields don't
    // sit next to each other in the specified fields array, the fields array will be automatically 
    // reorded to match the order specified in the span's +link{headerSpan.fields} array.
    // <P>
    // Note that headerSpans primarily provide a visual cue for grouping multiple headers 
    // together.  If you have an OLAP, data "cube" or multi-dimensional data model, the
    // +link{CubeGrid} component is the right choice.
    //
    // @group headerSpan
    // @visibility external
    //<    

    //> @attr listGrid.headerSpanHeight (integer : null : IR)
    // Default height for a +link{listGrid.headerSpans,headerSpan} with no height specified.
    // <P>
    // If <code>headerSpanHeight</code> is not specified (the default), headerSpans will be 1/2
    // of +link{listGrid.headerHeight}.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.headerSpanVAlign (vAlign : "center" : IR)
    // Default alignment for +link{listGrid.headerSpans,headerSpans} with no
    // +link{headerSpan.valign} specified.
    //
    // @group headerSpan
    // @visibility external
    //<
    headerSpanVAlign: "center",

    //> @attr listGrid.unspannedHeaderVAlign (vAlign : null : IR) 
    // When +link{listGrid.headerSpans,headerSpans} are in use, this property sets the default
    // vertical alignment for for fields which do <b>not</b> have a headerSpan.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.headerSpanConstructor (SCClassName : null : IR)
    // +link{SCClassName,SmartClient Class} to use for headerSpans.  Typically a +link{Button} or
    // +link{StretchImgButton} subclass.
    // <P>
    // If unset, headerSpans will be created using the +link{listGrid.headerButtonConstructor}.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.headerSpan (MultiAutoChild StatefulCanvas : null : IR)
    // +link{listGrid.headerSpans,headerSpans} are created via the +link{AutoChild} pattern, hence
    // <code>headerSpanConstructor</code>, <code>headerSpanDefaults</code> and
    // <code>headerSpanProperties</code> are valid.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @object HeaderSpan
    // A header span appears as a second level of headers in a ListGrid, spanning one or more
    // ListGrid columns and their associated headers.
    // <P>
    // See +link{listGrid.headerSpans}.
    // <P>
    // In addition to the properties documented here, all other properties specified on the
    // headerSpan object will be passed to the +link{Class.create,create()} method of the
    // +link{listGrid.headerSpanConstructor}.  This allows you to set properties such as
    // +link{button.baseStyle} or +link{stretchImgButton.src} directly in a
    // <code>headerSpan</code>.  
    //
    // @group headerSpan
    // @treeLocation Client Reference/Grids/ListGrid
    // @visibility external
    //<

    //> @attr headerSpan.name (identifier : null : IR)
    // Name for this headerSpan, for use in APIs like +link{listGrid.setHeaderSpanTitle()}.
    // <P>
    // Name is optional, but if specified, must be unique for this ListGrid (but not globally
    // unique) as well as a valid JavaScript identifier, as specified by ECMA-262 Section 7.6
    // (the <var class="smartclient">+link{String.isValidID()}</var><var class="smartgwt">StringUtil.isValidID()</var>
    // function can be used to test whether a name is a valid JavaScript identifier).
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.fields (Array of String : null : IR)
    // List of fields that this header spans.  Fields should be identified by their value for
    // +link{listGridField.name}.
    // <P>
    // Developers may define multiple levels of header-spans by specifying +link{headerSpan.spans}
    // however a span cannot be specified with both <code>fields</code> and <code>spans</code>.
    //
    // @group headerSpan
    // @visibility external
    //< 
    
    //> @attr headerSpan.spans (Array of HeaderSpan : null : IR)
    // This property allows developer to "nest" header spans, grouping fields together by
    // multiple layers of granularity.
    // <P>
    // For example a group of fields could be nested within two layers of header spans as follows:
    // <var class="smartclient">
    // <pre>
    // { title:"Europe", spans:[
    //      {title:"France", fields:["Paris", "Lyon"]},
    //      {title:"UK", fields:["London", "Glasgow"]},
    //      {title:"Spain", fields:["Barcelona"]}
    //  ]
    // }
    // </pre>
    // </var>
    // <var class="smartgwt">
    // <pre>
    //      HeaderSpan france = new HeaderSpan("France", new String[] {"Paris", "Lyon"});
    //      HeaderSpan uk = new HeaderSpan("UK", new String[] {"London", "Glasgow"});
    //      HeaderSpan spain = new HeaderSpan("Spain", new String[] {"Barcelona"});
    //
    //      HeaderSpan europe = new HeaderSpan();
    //      europe.setTitle("Europe");
    //      europe.setSpans(france, uk, spain);
    // </pre>
    // </var>
    // Note that a span definition can not include both <code>spans</code> 
    // and +link{headerSpan.fields,fields}.
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.title (String : null : IR)
	// A title for this headerSpan, to display in the headerSpan button for this headerSpan
    // and in other contexts such as the +link{canPickFields,menu for picking visible fields}.
    //                                
    // Note: if you want to use HTML tags to affect the display of the header, you should do so
    // via +link{headerSpan.headerTitle} instead so that other places where the title
    // appears in the UI are not affected.  Refer to discussion at +link{listGridField.title}.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.headerTitle (String : null : IR)
	// Optional title for the headerSpan button for this headerSpan. If specified this will be 
	// displayed in the headerSpan button instead of +link{headerSpan.title}. Set to an empty
    // string to suppress the title in the header button entirely.
    //
    // @group headerSpan
    // @visibility external
    //<
    
    //> @attr headerSpan.headerBaseStyle (CSSClass : null : [IRW])
    // Custom base style to apply to the header button created for this span instead
    // of +link{listGrid.headerBaseStyle}.
    // <P>
    // Note that depending on the header button constructor, you may have to specify
    // +link{headerSpan.headerTitleStyle} as well.
    // @group appearance
    // @visibility external
    //<
    
    //> @attr headerSpan.headerTitleStyle (CSSClass : null : [IRW])
    // Custom titleStyle to apply to the header button created for this span instead of 
    // +link{listGrid.headerTitleStyle}.
    // <p>
    // Note that this will typically only have an effect if 
    // +link{listGrid.headerButtonConstructor} is set to +link{class:StretchImgButton} or a subclass 
    // thereof.
    // @see headerSpan.headerBaseStyle
    // @group appearance
    // @visibility external
    //<
    
    //> @attr headerSpan.height (integer : null : IR)
    // Height of this headerSpan.  Defaults to +link{listGrid.headerSpanHeight}.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.valign (VerticalAlignment: null : IR)
    // Vertical alignment of the title of this headerSpan.
    // <P>
    // Defaults to listGrid.headerSpanVAlign if unset.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.align (Alignment: "center" : IR)
    // Horizontal alignment of the title of this headerSpan.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.showTreeColumnPicker (Boolean : true : IR)
    // When +link{listGrid.headerSpans} are in use, whether to show a hierarchical column picker
    // that includes both headerSpans and normal headers, with normal headers indented under
    // headerSpans similarly to how a +link{TreeGrid} displays a Tree.
    // <P>
    // If <code>showTreeColumnPicker</code> is false, no column picker will be shown on the
    // headerSpan itself, and the column picker for a clicked on a normal field header will include
    // only normal fields.
    //
    // @group headerSpan
    // @visibility external
    //<
    showTreeColumnPicker: true,


    // Cell Styling
    // ---------------------------------------------------------------------------------------
    
    //> @attr listGridField.cellAlign (Alignment : null : [IRW])
    // Horizontal alignment for cells in this field's column: "left", "right"
	// or "center".<br>
    // If null, alignment is derived from +link{ListGridField.align}. If this field is editable, 
    // the alignment of cells in the body will also be reflected in any editors for the field.
	//  @group  appearance
	//  @visibility external
	//<
	
    //> @attr listGrid.reverseRTLAlign (Boolean : true : [IRW])
    // If a page is rendered in +link{isc.Page.isRTL(),RTL mode}, should
    // cell alignments specified +link{listGridField.cellAlign} be reversed (so
    // an <code>align:"right"</code> field will have content aligned on the left and
    // vice versa)?
    // <P>
    // This is true by default to match user expectation that text flows from
    // start-to end and is aligned with the start of text flow (left in LTR mode, 
    // right in RTL mode) by default. May be set to false to have the specified 
    // alignments be taken literally in RTL mode.
    // @visibility external
    //<
    reverseRTLAlign:true,

    
    //> @attr   listGridField.baseStyle (CSSClass : null : [IRW])
    //  Custom base style to apply to all cells in this field instead of +link{ListGrid.baseStyle}
    //  To override the baseStyle at the row level, use 
    //  +link{ListGrid.recordBaseStyleProperty, record[listGrid.recordBaseStyleProperty]}
    //  instead.
    // @see ListGrid.recordBaseStyleProperty    
    // @group appearance
    // @visibility external
    // @example gridCells
    //<
    
    // Sorting (per field)   
    // ---------------------------------------------------------------------------------------

	//> @attr   listGridField.canSort (Boolean : true : [IRW])
	//  Enables or disables sorting by this column. If false, interactive
	//  sorting via header-clicks or menu-items will have no effect, but direct scripted calls
	//  to +link{ListGrid.sort, sort()} or +link{ListGrid.setSort, setSort()} will work as
    //  expected.
	//  @group  sorting
	//  @see    method:ListGrid.sort
	//  @see    attr:ListGrid.canSort
	//  @visibility external
	//<
    
    //> @attr listGridField.sortDirection (SortDirection : null : [IRW])
    // Specifies the default sorting direction for this column. If specified on the
    // +link{listGrid.sortField,default sort field} for the listGrid, sorting occurs
    // automatically, otherwise this will be the default direction when the user clicks the
    // field header, or calls +link{ListGrid.sort()} without specifying an explicit sort
    // direction.
    // <P>
    // Overrides ListGrid.sortDirection
	//  @group  sorting
	//  @see type:SortDirection
	//  @visibility external
	//<
    
	//> @method listGridField.sortNormalizer() (A)
	// Optional function to return the value that should be used when sorting this field.
    // <P>
    // Note that, if the dataset exceeds +link{ListGrid.dataPageSize} and hence paging is
    // introduced, the grid relies on the server to provide sorting, and the sortNormalizer
    // will no longer be called.
    //
	// @param recordObject    (object)    record to normalize
	// @param fieldName       (string)    name of the field on which sorting occurred
    // @param context (ListGrid) A pointer back to the list grid displaying this field will
    //   be available as the <code>context</code> argument. Note that you can also get a pointer
    //   to the field definition object by calling <code>context.getField(fieldName)</code>
	//  @return (any)   normalized value for sorting
	//  @group  sorting
	//  @visibility external
    //  @example dataTypes
	//<

	// Editing (per field)
	// ----------------------------------------------------------------------------------------

	//> @attr listGridField.canEdit (boolean : null : [IRW])
	// Can this field be edited?  May be overridden by setting the 'canEdit' property at the
    // listGrid level. If not explicitly set and this grid is bound to a dataSource, the
    // +link{listGrid.canEditFieldAttribute} may be used to set default editability at the field level.
    // <p>
    // For more dynamic control over whether fields can be edited, see
    // +link{listGrid.canEditCell}.
    //
	// @group  editing
	// @see attr:listGrid.canEdit
    // @see attr:listGrid.recordEditProperty
    // @see method:listGrid.canEditCell
	// @visibility external
    // @example disableEditing
	//<

    
    //> @attr listGridField.alwaysShowEditors (boolean : null : [IRW])
    // When this attribute is set, editors will be rendered into every row of the grid for
    // this field, rather than showing up in a single record at a time.
    // This attribute is only valid when +link{listGrid.editByCell} is false
    // @group editing
    //<
    

    
    //> @attr listGridField.defaultValue (any : null : [IRW])
    // If this field +link{listGridField.canEdit, can be edited}, this property can be used to
    // specify a default value for this field's editor when adding new rows to the grid.
    // @see listGrid.startEditingNew()
    // @group editing
    // @visibility external
    //<
    
    //> @method listGridField.defaultDynamicValue()
    // If this field +link{listGridField.canEdit, can be edited}, this property can be used to
    // set a dynamic default value which will show up in editors for this field. 
    // Will be applied to the editor for the field as +link{FormItem.defaultDynamicValue}
    //
    // @param	item	(FormItem)  The editor for the cell itself (also available as "this").
    //                              Note that in addition to the standard FormItem APIs available
    //                              on the editor, it also has:<br>
    //                              - a pointer back to the containing listGrid 
    //                              [<code>item.grid</code>]<br>
    //                              - the colNum being edited [<code>item.colNum</code>]<br>
    //                              - the rowNum being edited [<code>item.rowNum</code>]
    // @param	form    (DynamicForm) the managing DynamicForm instance
    // @param   values  (Object)      the current set of values for the form as a whole
    // @group editing
    // @visibility external
    //<
    
	//> @attr   listGridField.enterKeyEditAction (EnterKeyEditAction : "done" : [IRW])
	// What to do when a user hits enter while editing this field?<br>
	// Overrides the <code>enterKeyEditAction</code> as specified at the listGrid level while 
	// focus is in this field.
	//  @group  editing
	//  @visibility external
	//<
    
	//> @attr   listGridField.escapeKeyEditAction (EscapeKeyEditAction : "cancel" : [IRW])
	// What to do when a user hits escape while editing this field?<br>
	// Overrides the <code>escapeKeyEditAction</code> as specified at the listGrid level while 
	// focus is in this field.
	//  @group  editing
	//  @visibility external
	//<

    
	//> @attr   listGridField.nextTabColNum (number : null : [IRWA])
	// If specified, when the user hits tab while editing this field, editing will move to the 
	// specified colNum in the next row (or the first editable field after it), rather than the 
	// next editable field in this row.
	//  @group  editing
	//  @visibility advancedInlineEdit
	//<
    
	//> @attr   listGridField.previousTabColNum (number : null : [IRWA])
	// If specified, when the user hits shift+tab while editing this field, editing will move 
	// to the specified colNum in the previous row (or the first editable field before it), 
	// rather than the previous editable field in this row.
	//  @group  editing
	//  @visibility advancedInlineEdit
	//<
    
	//> @attr   listGridField.editorType (FormItem className : null : [IRWA])
	//      Name of form item class to use for the form item created to edit this field. 
	//      (Only used if this field is editable).<br>
	//      Note: If this is not specified, the edit-form item type may be derived from the
	//      <code>editorType</code> property, typically inherited from datasource fields, or 
	//      from the <code>type</code> of the field (showing the appropriate form item for 
	//      the data-type). See the +link{group:editing} overview for more on editing ListGrid
    //      fields.
	//  @group  editing
	//  @see attr:listGrid.canEdit
	//  @visibility external
    //  @example customEditors
	//<
    // link to editing group documentation included as that describes the additional
    // "rowNum", "colNum" and "grid" properties stored on the editor.
    
	//> @attr   listGridField.editorProperties (FormItem properties : null : [IRWA])
	// Properties to apply the the form item created to edit this field. (Only used if
	// this field is editable).
    // <P>
    // For example, if you have a field "shoeSize" with +link{dataSourceField.editorType} set
    // to "SpinnerItem" in order to use a SpinnerItem as your field editor, and you want to pass the
    // +link{spinnerItem.step} property to the created SpinnerItem:
    // <pre>
    //    fields : [
    //        { name:"shoeSize", editorType:"SpinnerItem", 
    //          editorProperties : { step:0.5 } },
    //        ... other fields ...
    //    ]
    // </pre>
    //
	// @group editing
	// @visibility external
    // @example customEditors
	//<
    
	//> @attr   listGrid.modalEditing (boolean : null : [IRWA])
	//      If this property is true, any mouse click outside of the open cell editors
    //      will end editing mode, hiding the cell editors and saving any changes to those
    //      cell values.
	// @group  editing
    // @visibility external
    // @example modalEditing
	//<

	//> @method listGridField.editorEnter (A)
	// Callback fired when the user first starts editing a cell.
    // <P>
    // This callback is typically used to establish dynamic default values via
    // +link{listGrid.setEditValue()} or +link{listGrid.setEditValues()}.
	//
	// @param record (ListGridRecord) record for the cell being edited.  <b>Will be null</b>
    //                                for a new, unsaved record.
	// @param value (any) value for the cell being edited
	// @param rowNum (int)	row number for the cell
	// @param colNum (int)	column number of the cell
    // @param grid (ListGrid) ListGrid to which this field belongs
	// @group editing
	// @visibility external
	//<

	//> @method listGridField.editorExit (A)
	// Callback fired when the user attempts to navigate away from the current edit cell, 
	// or complete the current edit.<br>
	// Return false from this method to cancel the default behavior (Saving / cancelling the
	// current edit / moving to the next edit cell)
	//
	// @param   editCompletionEvent (EditCompletionEvent)  What interaction triggered this
	//                                                          edit cell exit
	// @param	record     (ListGridRecord)	record for the cell being edited
	// @param	newValue   (any)    new value for the cell being edited
	// @param	rowNum	   (int)	row number for the cell
	// @param	colNum	   (int)	column number of the cell
    // @param   grid    (ListGrid)  ListGrid to which this field belongs    
    // @return  (boolean)   Returning false from this method will cancel the default behavior
    //                      (for example saving the row) and leave the editor visible and focus
    //                      in this edit cell.
	//  @group  editing
	//  @see listGrid.editorExit
	// @visibility external
	//<
        
    //> @method listGridField.cellChanged()
	// Callback fired when field changes value as the result of a cell edit.  Fired only on
    // successful save of edit, when the new value doesn't match the value before editing.<br>
    // <p>
    // Same signature as +link{method:listGrid.cellChanged()}, but defined on a per-field
    // basis.
    //
	// @param	record     (ListGridRecord)	record for the cell being changed
	// @param	newValue   (any)    new value for the cell
	// @param	oldValue   (any)    old value for the cell
	// @param	rowNum	   (number)	row number for the cell
	// @param	colNum	   (number)	column number of the cell
	// @param	grid       (ListGrid)	grid where cell was changed.
    //
	// @group  editing
	// @see method:listGrid.cellChanged()
	// @visibility external
	//<
    
    //> @attr listGridField.validators (array of Validator : null : [IRW])
    // Array of +link{class:Validator} objects for this field.  When the user edits cells in
    // this field, these validators will be applied to the edited value.<br>
    // Note: for databound listGrids, this property may be specified on the 
    // +link{class:DataSourceField}, enabling both client and server side validation.
    // @see class:Validator
    // @see listGridField.required
    // @group gridValidation 
    // @visibility external
    // @example dataValidation
    //<
    
    //> @attr listGridField.validateOnChange (boolean : null : [IRW])
    // If set to true, any +link{listGridField.validators} for this field will be run whenever
    // the value of the field is changed.
    // <P>
    // Analogous to the +link{FormItem.validateOnChange} property.
    // @group gridValidation
    // @visibility external
    //<
    
    //> @attr listGridField.required (boolean : null : [IRW])
    // When the user edits cells in this field, is this value required to be non-empty 
    // in order for validation to pass.<br>
    // Note: for databound listGrids, this property may be specified on the 
    // +link{class:DataSourceField}, enabling both client and server side validation.
    //
    // @see listGridField.validators
    // @group gridValidation
    // @visibility external
    //<
    
    //> @attr listGridField.displayFormat (varies : null : [IRWA])
    // For fields of type <code>"date"</code>, set this property to a valid
    // +link{dateDisplayFormat} to specify how the date should be formatted.<br>
    // For fields of type <code>"time"</code>, set this property to a valid 
    // +link{type:TimeDisplayFormat, TimeDisplayFormat} to specify how the time should be formatted.<br>
    // If unset, display format may be set at the listGrid level via +link{ListGrid.dateFormatter}
    // or +link{ListGrid.timeFormatter}.
    // <p>
    // If this field is editable the displayFormat will also be passed to the editor created
    // to edit this field.  For dates you may also need to set +link{listGridField.inputFormat}.
    //
    // @see listGridField.inputFormat
    // @see listGrid.dateFormatter
    // @see listGrid.timeFormatter
    // @deprecated Use +link{listGridField.dateFormatter} and +link{listGridField.timeFormatter} 
    //  instead.
    // @visibility external
    //<
    
    //> @attr listGridField.dateFormatter (DateDisplayFormat : null : [IRW])
    // Display format to use for date type values within this field. 
    // <P>
    // The +link{listGridField.timeFormatter} may also be used to format underlying Date values as
    // times (ommitting the date part entirely). If both <code>dateFormatter</code> and
    // <code>timeFormatter</code> are specified on a field, for
    // fields specified as +link{listGridField.type,type "time"} the
    // <code>timeFormatter</code> will be used, otherwise the <code>dateFormatter</code>
    // <P>
    // If <code>field.dateFormatter</code> and <code>field.timeFormatter</code> is unspecified,
    // date display format may be defined at the component level via
    // +link{ListGrid.dateFormatter,ListGrid.dateFormatter}, or for fields of type <code>"datetime"</code>
    // +link{ListGrid.datetimeFormatter,ListGrid.datetimeFormatter}. Otherwise the
    // default is to use the system-wide default short date format, configured via
    // +link{Date.setShortDisplayFormat()}.  Specify any valid +link{type:DateDisplayFormat} to 
    // change the format used by this item.
    // <P>
    // If this field is editable the dateFormatter will also be passed to the editor created
    // to edit this field. Note that you can also specify an explicit +link{listGridField.inputFormat}
    // which will be passed through to the editor as well, though this is not typically required
    // as the input format should be automatically derived by the SmartClient system
    // for standard DateDisplayFormats.
    //
    // @see listGrid.dateFormatter
    // @see listGrid.datetimeFormatter
    // @see listGridField.timeFormatter
    // @visibility external
    //<
    
    //>	@attr listGridField.timeFormatter (TimeDisplayFormat : null : [IRWA])
    // Time-format to apply to date type values within this field.  If specified, any
    // dates displayed in this field will be formatted as times using the appropriate format.
    // This is most commonly only applied to fields specified as type <code>"time"</code> though
    // if no explicit +link{listGridField.dateFormatter} is specified it will be respected for other 
    // fields as well.
    // <P>
    // If unspecified, a timeFormatter may be defined 
    // +link{ListGrid.timeFormatter,at the component level} and will be respected by fields
    // of type <code>"time"</code>.
    // <P>
    // If this field is editable, the timeFormatter will also be passed to the editor created to
    // edit this field as +link{formItem.timeFormatter}.
    //
	// @group appearance
    // @visibility external
	//<
    //timeFormatter:null

    //> @attr listGridField.decimalPrecision (number : null : [IRW])
    // @include dataSourceField.decimalPrecision
    //
    // @group appearance
    // @serverDS allowed
    // @visibility external
    //<

    //> @attr listGridField.decimalPad (number : null : [IRW])
    // @include dataSourceField.decimalPad
    //
    // @group appearance
    // @serverDS allowed
    // @visibility external
    //<

    //> @attr listGridField.inputFormat (DateInputFormat : null : [IRWA])
    // For fields of type <code>"date"</code> or <code>"datetime"</code>, if this is an editable 
    // listGrid, this property allows you to specify the +link{DateItem.inputFormat, inputFormat}
    // applied to the editor for this field.
    // @see listGridField.dateFormatter
    // @visibility external
    //<
    
    //> @attr listGridField.isRemoveField (boolean : null : [IRA])
    // If set to true and +link{listGrid.canRemoveRecords} is true, this field will be rendered
    // as the remove-field for this grid. In most common usage scenarios this field will essentially
    // be a placeholder indicating where the remove field should be rendered, meaning properties 
    // other than <code>isRemoveField</code>, such as <code>name</code> or <code>title</code>, may
    // be left unset.
    // @see listGrid.canRemoveRecords
    // @visibility external
    //<

    //> @method listGridField.recordClick()
    //
    // Executed when this field is clicked on.  Note that if +link{ListGrid.recordClick()} is
    // also defined, it will be fired for fields that define a recordClick handler if the
    // field-level handler returns true. Return false to prevent the grid-level handler from firing.
    //
    // @param	viewer		(ListGrid)	the listGrid that contains the click event
    // @param	record		(ListGridRecord)	the record that was clicked on
    // @param	recordNum	(number)	number of the record clicked on in the current set of
    //                                  displayed records (starts with 0)
    // @param	field		(ListGridField)	the field that was clicked on (field definition)
    // @param	fieldNum	(number)	number of the field clicked on in the listGrid.fields
    //                                  array
    // @param	value       (object)    value of the cell (after valueMap, etc. applied)
    // @param	rawValue	(object)	raw value of the cell (before valueMap, etc applied)
    // @return	(boolean)	false to stop event bubbling
    //
    // @group	events
    //
	// @see method:listGrid.recordClick()
	// @visibility external
    // @example recordClicks
	//<

    //> @method listGridField.recordDoubleClick()
    //
    // Executed when this field is double-clicked.  Note that if
    // +link{ListGrid.recordDoubleClick()} is also defined, it will be fired for fields that define
    // a recordDoubleClick handler if the field-level handler returns true. Return false to prevent
    // the grid-level handler from firing.
    //
    //
    // @param	viewer		(ListGrid)	the listGrid that contains doubleclick event
    // @param	record		(ListGridRecord)	the record that was double-clicked
    // @param	recordNum	(number)	number of the record clicked on in the current set of
    //                                  displayed records (starts with 0)
    // @param	field		(ListGridField)	the field that was clicked on (field definition)
    // @param	fieldNum	(number)	number of the field clicked on in the listGrid.fields
    //                                  array
    // @param	value       (object)    value of the cell (after valueMap, etc. applied)
    // @param	rawValue	(object)	raw value of the cell (before valueMap, etc applied)
    // @return	(boolean)	false to stop event bubbling
    //
    // @group	events
    //
	// @see method:listGrid.recordClick()
	// @visibility external
    // @example recordClicks
	//<

    // Filtering
    // ---------------------------------------------------------------------------------------

	//> @attr   listGridField.canFilter (boolean : null : [IRW])
	//      If showing a filter row for this ListGrid, should the filter criteria for this 
	//      field be editable
	//  @group  filterEditor
	//  @visibility external
    //  @example disableFilter
	//<

	//> @attr listGridField.filterEditorValueMap (object : null : [IRW])
	//  If this listGrid is showing a filter row, this property can be used to specify a
	//  mapping of internal data to/from display values to be in the appropriate filter
	//  row form item.
	//  @visibility external
	//  @group filterEditor
	//<
    
	//> @attr   listGridField.filterEditorType (FormItem className : null : [IRWA])
	//      If this ListGrid is showing a filter row, this property can be used to
	//      specify the form item class to use for the filter form item associated with this 
	//      field
	//      (Only used if this field is not canFilter:false).<br>
	//      Note: If this is not specified, the edit-form item type may be derived from the
	//      'editorType' property, typically inherited from datasource fields, or from the
	//      'type' of the field (showing the appropriate form item for the data-type).
	//  @group  filterEditor
	//  @visibility external
	//<
    
    //> @attr   listGridField.defaultFilterValue (any : null : [IRWA])
    // If this ListGrid is showing a filter row, this property can be used to apply a default
    // value to show in the filter editor for this field.
    // @group filterEditor
    // @visibility external
    //<
    

	//> @attr   listGridField.filterEditorProperties (FormItem properties : null : [IRWA])
	// If this ListGrid is showing a filter row
    // (+link{listGrid.showFilterEditor,showFilterEditor}:true), this property
    // can be used to specify properties for the appropriate filter form item.
	// @group filterEditor
    // @visibility external
	//<
        
	//>	@attr     listGridField.filterOnKeypress (boolean : null : [IRWA])
	// If we're showing the filterEditor (listGrid.showFilterEditor is true), this property 
	// determines whether this list should be filtered every time the user edits the value of
    // the filter editor for this field.
	// @group filterEditor
	// @visibility external
	//<
    
    //> @attr listGrid.fetchDelay (number : 300 : IRWA)
	// If we're showing the filterEditor (+link{listGrid.showFilterEditor} is true), and we're
	// re-filtering on every keypress (+link{listGrid.filterOnKeypress} is true), this 
    // property is the delay in milliseconds between the user changing the filter and the 
    // filter request being kicked off. If multiple changes are made to the filter 
    // within this fetch delay, only the most recent will actually cause a re-filter
    // @group filterEditor
    // @visibility external
    //<
    fetchDelay:300,

	//>	@attr listGridField.shouldPrint (boolean : null : IRW)
    // Whether this field should be included in the printable representation of the grid.
	//
    // @group printing
	// @visibility external
	//<

    // AutoComplete
    // ---------------------------------------------------------------------------------------

	//>	@attr listGridField.autoComplete (AutoComplete : null : IRW)
	// Whether to do inline autoComplete when editing this field.
	// <p>
	// If unset, defaults to listGrid.autoComplete
	//
	// @see listGrid.autoComplete
	// @visibility autoComplete
	//<

	//>	@attr listGridField.uniqueMatch (boolean : null : IRW)
	// When autoComplete is enabled, whether to offer only unique matches to the user.
	// <p>
	// If unset, defaults to listGrid.uniqueMatch.
	//
	// @see listGrid.uniqueMatch
	// @visibility autoComplete
	//<
    uniqueMatch:true,
    

	// Formatting (per field)
	// --------------------------------------------------------------------------------------------

	//> @method listGridField.getCellValue()    
	// A stringMethod which returns the cell value to display for this field for some record.
	// If defined, called by ListGrid.getCellValue().  Called in the scope of the field object.
    //
    // Deprecated as of Jan 12 05 in favor of field.formatCellValue, because 'getCellValue()'
    // is a lower-level API which handles (for example) returning the HTML for editors within
    // the cell.
	//  
	// @param  viewer  (ListGrid)  the ListGrid for which we're returning a cellValue
	// @param  record  (object)    the current record object
	// @param  recordNum   (nubmer)    row-index of the current record
	// @param  field   (listGridField) current field object
	// @param  fieldNum    (number)    column-index of the current field
	// @param  value   (any)   unformatted value for this field, determined via 
	//                          ListGrid.getRawCellValue()
	// @see    method:listGrid.getCellValue
	// @see method:listGridField.formatCellValue
	// @group  display_values
	// @visibility external
	// @return (any)   value to display in the ListGrid cell
	// @deprecated As of SmartClient 5.5, use +link{listGridField.formatCellValue}.
	//<
    
	// We provide separate formatters for the raw value displayed in a static cell, and the
	// value displayed in an editor.
	// This makes sense because:
	// - developers are likely to want to apply different formats - for example including some
	//   raw HTML in the static value, but not in the value displayed in a text based editor.
	// - the more common 'formatCellValue()' needs no parser to convert from the formatted value
	//   back to the raw value
	// If a developer wishes to apply the same formatter in both cases, the suggested approach
	// would be to write a single formatter function and have it be called from both methods.
    
	//> @method listGridField.formatCellValue()
	// Return the HTML to display in cells of this field.
    // <P>
    // Given the raw value for this field as taken from the record Formatter to apply to the
    // static values displayed in cells for this field.
    // <P>
	// <i>Example usage</i>: formatting a currency value stored in cents (so "100" to "$1.00")<br>
	// The value passed to this method is the raw value for the cell.<br>
	// Takes precedence over <code>formatCellValue</code> defined at the grid level for cells
	// in this field.
    // <P>
	// Note: this formatter will not be applied to the values displayed in cells being edited.
	// The +link{listGridField.formatEditorValue,formatEditorValue()} is provided for that purpose.
	//
	// @group display_values
	//
	// @param   value (any)   raw value for the cell, from the record for the row
	// @param   record   (ListGridRecord) 
    //   Record object for the cell. Note: If this is a new row that has not been saved, in an 
    //   editable grid, it has no associated record object. In this case the edit values will
    //   be passed in as this parameter (see +link{listGrid.getEditValues()})
	// @param   rowNum  (number)    row number for the cell
	// @param   colNum  (number)    column number for the cell.
	// @param   grid    (ListGrid) the ListGrid displaying the cell
	// @return (String) HTML to display in the cell
	//
	// @see listGrid.formatCellValue()
	// @see listGridField.formatEditorValue()
	// @visibility external
    // @example formatValues
	//<
	
	//> @attr listGridField.escapeHTML (boolean : null : IRW)
	// By default HTML values in ListGrid cells will be interpreted by the browser.
    // Setting this flag to true will causes HTML characters to be escaped, meaning the
    // raw value of the field (for example <code>"&lt;b&gt;AAA&lt;/b&gt;"</code>) is displayed
    // to the user rather than the interpreted HTML (for example <code>"<b>AAA</b>"</code>)
    // @visibility external
    //<
    
	//> @attr listGridField.linkText (String : null : IRW)
    // The HTML to display in cells of this field if the fieldType is set to link. 
    // <P>
    // This property sets linkText that will be the same for all records.  You can set linkText
    // on a per-record basis via +link{attr:listGridRecord.linkText}.
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
    //  @see attr:listGridRecord.linkText
    //  @see attr:listGrid.linkTextProperty
    //  @see attr:listGridField.linkTextProperty
	//  @group  display_values
	//  @visibility external
    //  @example linkImage
	//<
	
	//> @attr listGridField.linkTextProperty (string : null : IRW)
    // Name of the property in a ListGridRecord that holds the HTML to display in cells of this
    // field if the fieldType is set to "link".
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
    //  @see attr:listGridRecord.linkText
    //  @see attr:listGridField.linkText
    //  @see attr:listGrid.linkTextProperty
	//  @group  display_values
	//  @visibility external    
	//<
	
	//> @attr listGridField.linkURLPrefix (string : null : IRWA)
    // If this field has type [+link{type:ListGridFieldType}] set to <code>"link"</code>,
    // setting this property will apply a standard prefix to the link URL for cells in this field.
    // @visibility external
    //<
    
    
	//> @attr listGridField.linkURLSuffix (string : null : IRWA)
    // If this field has type [+link{type:ListGridFieldType}] set to <code>"link"</code>,
    // setting this property will apply a standard suffix to the link URL for cells in this field.
    // @visibility external
    //<
    
    // --------------------
    // Editing 
    
	//> @method listGridField.formatEditorValue
	// Return the value to display in cells of this field which are being edited.
    // <P>
	// <i>Example usage</i>: converting a stored value in cents (100) to a dollar-and-cents 
	// value in the editor (1.00)
    // <P>
	// The value passed to this method is the raw value for the cell.
    // <P>
	// <code>formatEditorValue</code> takes precedence over +link{listGrid.formatEditorValue()}
    // defined at the grid level for cells in this field.
    // <P>
	// To convert the formatted value displayed within an editor back to a raw value, implement
	// +link{listGridField.parseEditorValue} as well.
	//
	// @group editing
	//
	// @param   value (any)   raw value for the cell being edited
	// @param   record   (ListGridRecord) 
    //   Record object for the cell. Note: If this is a new row that has not been saved, in an 
    //   editable grid, it has no associated record object. In this case the edit values will
    //   be passed in as this parameter.
	// @param   rowNum  (number)    row number for the cell
	// @param   colNum  (number)    column number for the cell.
	// @param   grid    (ListGrid instance) A pointer to the ListGrid displaying the cell
	// @return (any) formatted value to display in the editor
	//
	// @see listGridField.formatCellValue()
	// @see listGrid.formatEditorValue()
	// @see listGridField.parseEditorValue()
	//
	// @visibility external
	//<
    
	//> @method listGridField.parseEditorValue
	// Method used to convert the value displayed in an editor for some cell in this field into
	// a raw value for saving.<br>
	// Takes precedence over <code>parseEditorValue</code> defined at the grid level.
	//
	// @group editing
	//
	// @param   value (any)   value displayed in the editor for the cell
	// @param   record (object) record object for the row being edited. May be null if this
	//                          is a new row being added to the end of the list.
	// @param   rowNum  (number)    row number for the cell
	// @param   colNum  (number)    column number for the cell.
	// @param   grid    (ListGrid instance) A pointer to the ListGrid displaying the cell
	// @return (any) raw value for the field derived from formatted value in editor
	// @see listGrid.parseEditorValue()
	// @see listGridField.formatEditorValue()
	// @visibility external    
	//<

	//> @attr listGridField.valueMap (Array of Object : null : IRW)
    // Array of legal values for this field, or an Object where each property maps a stored
    // value to a user-displayable value.<br>
    // Note that if this field is editable (see +link{listGrid.canEdit}, 
    // +link{listGridField.canEdit}), editors displayed for this field will pick up their
    // valueMap either from this value or from +link{listGridField.editorValueMap}.
    // <P>
    // See also +link{dataSourceField.valueMap}.
    //
	// @group display_values
    // @see ListGrid.setValueMap()
    // @see ListGrid.getDisplayValue()
	// @visibility external
    // @example listType
	//<
	
	//> @attr listGridField.sortByMappedValue (boolean : null : IRW)
	// If +link{listGridField.valueMap} is set, and the grid is +link{listGrid.setSort(),sorted}
	// by this field, should the data be sorted by the underlying data value or the
	// mapped display value. If unset, will sort by display value. Set to <code>false</code>
	// to sort by underlying data value. Note that this has no effect if
	// a +link{listGridField.sortNormalizer} has been specified.
	// @visibility external
	//<
	
	//> @attr listGridField.multiple (Boolean : null : IR)
	// Indicates that this field should always be Array-valued. This property will be 
	// passed through to the generated edit-item when editing the field - so if
	// +link{listGridField.valueMap} is set, the default editor will be a +link{SelectItem} with
	// +link{SelectItem.multiple} set to true.
	// <P>
	// Note that for databound grids it typically makes sense to set +link{DataSourceField.multiple}
	// rather than setting the property directly on the ListGridField object.
	// @group editing
	// @visibility external
	//<
    
    //> @attr listGridField.editorValueMap (Array or Object : null : IRW)
    // A valueMap to use for editors shown for this field.  By default if this is not
    // specified +link{listGridField.valueMap,field.valueMap} will be used instead.
    // <P>
    // Dynamic valueMaps can be provided by implementing +link{listGrid.getEditorValueMap()}.
    // 
    // @group editing
    // @visibility external
    // @see listGrid.getEditorValueMap()
    // @see listGrid.setEditorValueMap()
    //<
    
    //> @method listGridField.change()
    // If this field is editable, any +link{formItem.change, change} handler specified
    // on the ListGridField will be passed onto the editors for this field.
    // <P>
    // Note that if +link{listGridField.canToggle} is true, the user may change the value of
    // a boolean field without going into edit mode by single clicking on the field. In this 
    // case the +link{listGridField.change()} and +link{listGridField.changed()} handlers will
    // fire but the <code>form</code> and <code>item</code> parameters will be null.
    //
    // @param	form    (DynamicForm) the managing DynamicForm instance
    // @param	item	(FormItem)    the editor (form item) itself (also available as "this").
    //                              Note that in addition to the standard FormItem APIs available
    //                              on the editor, it also has:<br>
    //                              - a pointer back to the containing listGrid 
    //                              [<code>item.grid</code>]<br>
    //                              - the colNum being edited [<code>item.colNum</code>]<br>
    //                              - the rowNum being edited [<code>item.rowNum</code>]
    // @param   value   (any)         The new value of the form item
    // @param   oldValue    (any)     The previous value of the form item
    // @return (Boolean) The change may be cancelled <var class="SmartClient">by returning false</var>
    // @see listGridField.changed()
    // @see listGrid.cellChanged()
    // @group editing
    // @visibility external
    //<
    
    //> @method listGridField.changed()
    // If this field is editable, any +link{formItem.changed, changed} handler specified
    // on the ListGridField will be passed onto the editors for this field.
    // Note that if +link{listGridField.canToggle} is true, the user may change the value of
    // a boolean field without going into edit mode by single clicking on the field. In this 
    // case the +link{listGridField.change()} and +link{listGridField.changed()} handlers will
    // fire but the <code>form</code> and <code>item</code> parameters will be null.
    //
    // @param	form    (DynamicForm) the managing DynamicForm instance
    // @param	item	(FormItem)    the editor (form item) itself (also available as "this").
    //                              Note that in addition to the standard FormItem APIs available
    //                              on the editor, it also has:<br>
    //                              - a pointer back to the containing listGrid 
    //                              [<code>item.grid</code>]<br>
    //                              - the colNum being edited [<code>item.colNum</code>]<br>
    //                              - the rowNum being edited [<code>item.rowNum</code>]
    // @param   value   (any)         The current value (after the change).
    // @see listGridField.change()
    // @see listGrid.cellChanged()
    // @group editing
    // @visibility external
    //<
    
	//> @attr listGridField.emptyCellValue (HTMLString : "&amp;nbsp;" : IRW)
	// The value to display for a cell whose value is null or the empty
	// string after applying formatCellValue and valueMap (if any).
    // <p>
    // This is the field-specific attribute.  You may also set the emptyCellValue at the grid
    // level to define the emptyCellValue for all empty fields in the grid.
    // 
	// @group display_values
    // @see listGrid.emptyCellValue
	// @visibility external
    // @example emptyValues
	//<
    
	// Field.optionDataSource
	// --------------------------------------------------------------------------------------------

	//> @attr listGridField.autoFetchDisplayMap (boolean : null : [IRW])
	// If true, automatically fetches records and derives a valueMap from
    // +link{listGridField.optionDataSource}.
    // <p>
    // Same as +link{listGrid.autoFetchDisplayMap}, but defined on a per-field basis.
    //
	// @group display_values
	// @see listGrid.autoFetchDisplayMap
    // @visibility external
	//<
	
	//> @attr listGridField.displayValueFromRecord (boolean : null : IRWA)
	// If a +link{listGridField.displayField} is set, should this field show record values from
	// the <code>displayField</code>?
	// <P>
	// If +link{listGridField.displayField} is specified, and there is no separate
	// +link{listGridField.optionDataSource}, by default we will show display-field values
	// from the same record. Setting this property to false would disable this behavior.
	// <P>
	// Alternatively, if there is a +link{listGridField.optionDataSource} (and 
	// +link{listGridField.autoFetchDisplayMap} is false), the displayField would be ignored
	// for the field and the underlying +link{listGridField.name,record[fieldName] value} would
	// be displayed to the user. This property may be set to true to override this behavior and
	// pick up values from the <code>displayField</code> for display in this field even when there
	// is an optionDataSource set.
	// <P>
	// Note that this property has no effect on fields with an explicitly specified valueMap, or
	// with an optionDataSource where +link{listGridField.autoFetchDisplayMap} is true.
	// @visibility external
	//<
	
    //> @attr listGridField.optionTextMatchStyle (TextMatchStyle : null : [IR])
    // For fields with an +link{listGridField.optionDataSource}, where
    // +link{listGridField.autoFetchDisplayMap} is true, this property will govern
    // the <code>textMatchStyle</code> attribute of the +link{DSRequest} parameter passed to 
    // +link{DataSource.fetchData()} when retrieving the remote data set to be used as 
    // a basis for this field's valueMap.
    // @group display_values
    // @visibility external
    //<
    
    //> @attr listGridField.optionFilterContext (DSRequest Properties : null : [IR])
    // If this field has an optionDataSource specified and 
    // +link{listGridField.autoFetchDisplayMap,autoFetchDisplayMap} is set, this attribute
    // provides a way to customize the dataSource request issued to fetch the display map from
    // the option dataSource.
    // @group display_values
    // @visibility external
    //<
    
    //> @attr listGridField.optionOperationId (string : null : [IR])
    // If this field has an optionDataSource specified and 
    // +link{listGridField.autoFetchDisplayMap,autoFetchDisplayMap} is set, this attribute
    // provides a way to customize the +link{DSRequest.operationId} passed to 
    // <code>dataSource.fetchData()</code> when retrieving the display map from the option 
    // dataSource.
    // @group display_values
    // @visibility external
    //<
    
	//> @attr listGridField.optionDataSource (String : null : [IRW])
	// Derive a +link{valueMap} by fetching records from another DataSource and extracting
    // the +link{listGridField.valueField,valueField} and 
    // +link{listGridField.displayField,displayField} in the loaded records, to derive one
    // valueMap entry per record loaded from the optionDataSource.
    // <P>
    // Unlike the similar use of +link{pickList.optionDataSource} for +link{PickList,pickLists}
    // used during editing or filtering, <code>listGridField.optionDataSource</code> causes the
    // <b>entire set of records from the optionDataSource to be fetched</b>, without paging.
    // Hence listGridField.optionDataSource is appropriate only for smaller valueMaps.  For very
    // large valueMap situations, such as an accountId field that should be displayed as an
    // accountName where there are thousands of accounts, the correct approach is:
    // <ul>
    // <li> do not set listGridField.optionDataSource
    // <li> declare two fields in the DataSource, eg "accountId" and "accountName".
    // <li> Set the +link{ListGridField.displayField} attribute on the data field to the
    //      name of the display field.
    // <li> When fetching records for display in a grid, have your server send back values for 
    //      both fields, but show only the data field ("accountId") in the grid.
    // </ul>
    // In this case the cells in the accountId field will show the record value from the 
    // accountName field.  This approach means the valueMap will never be loaded in its
    // entirety, instead, each loaded record contains the valueMapping for that one record, as
    // a pair of fields within the record.  
    // <p>
    // If you are using the SmartClient Server Framework with the SQL or JPA/Hibernate built-in
    // connectors, this entire approach can be achieved very easily using the
    // +link{dataSourceField.includeFrom} setting - see the 
    // +link{dataSourceField.includeFrom,docs for includeFrom} for details.
    // <P>
    // Notes:
    // <ul>
    // <li>When using the above approach, it is key that the server return <b>both</b>
    // the underlying stored value <b>and</b> the display value, as suggested above.
    // This approach allows the +link{pickList.optionDataSource} property to be used to 
    // provide paged valueMaps during inline editing and 
    // +link{ListGrid.showFilterEditor,inline filtering}. This can be achieved by setting the
    // <code>optionDataSource</code> attribute on the form item used to edit the field
    // via +link{listGridField.editorProperties} (for editing) or
    // +link{listGridField.filterEditorProperties,field.filterEditorProperties} (for
    // filtering), without specifying an optionDataSource at the listGridField level. 
    // Alternatively developers can use +link{listGridField.autoFetchDisplayMap} to 
    // suppress the fetch against the optionDataSource at the listGrid level.</li>
    // <li>Setting <code>listGridField.optionDataSource</code> to the same dataSource as the
    // listGrid is not the same as omitting the optionDataSource setting entirely. Unless
    // +link{listGridField.autoFetchDisplayMap} has been set to explicitly disable fetching,
    // a fetch will be performed against the dataSource to build a valueMap which will be used
    // as the definitive mapping from data to display values, rather than picking up the display
    // values from the records themselves. This distinction is required to support cases where
    // the +link{listGridField.valueField} points to a different field in the grid (useful for
    // hierarchical relationships, for example), or where +link{listGridField.optionCriteria} or
    // +link{listGridField.optionOperationId} are specified and return different data from
    // the records displayed within the grid.</li>
    // <li>If a displayField is specified, with no associated optionDataSource, and the field is
    // editable, updating the edit value for the field may not automatically update the displayField
    // edit value, meaning the user may not realize the edit value has been modified.
    // If the new value came from the user editing the field, and the edit item has
    // a valueMap or optionDataSource specified, the display value is picked up automatically and
    // stored out on the displayField for the record. However if the value was set programmatically,
    // the developer should also set the edit value for the display field to ensure the displayed
    // value reflects the new edit value. Note that when this occurs, a warning will be logged
    // which can be disabled via +link{listGrid.warnOnUnmappedValueFieldChange}.</li>
    // <li>For very advanced usage a developer can use +link{listGridField.displayValueFromRecord}
    // to explicitly tell the grid whether or not to display the display field value for the record
    // in this field when a displayField is specified. See documentation on that property for more
    // information</li>
    // </ul>
    //
	// @group display_values
	// @visibility external
	//<

	//> @attr listGridField.valueField (string : null : [IRW])
	// Specifies the +link{listGridField.optionDataSource} field used to retrieve the stored
    // values that are to be mapped to the display values (specified by
    // +link{listGridField.displayField}). Note that if this field is editable this will also
    // be applied to this field's editors.
    //
	// @group display_values
	// @visibility external
	//<

	//> @attr listGridField.displayField (string : null : [IRW])
	// Specifies the +link{listGridField.optionDataSource} field used to retrieve the display
    // values that are to be mapped from the internal values specified by +link{valueField}.
    // <P>
    // If no <code>optionDataSource</code> is defined for the field, the cell will display
    // the displayField value for the current record instead of the underlying value for
    // this field.  This approach can be used for situations where field values need a stored
    // value to displayed value mapping, but the set of all possible values is too large to
    // load as a +link{ValueMap} - see +link{listGridField.optionDataSource} for more details
    // on this approach.  Note that if this field is editable this will also be applied to this
    // field's editors.  +explorerExample{largeValueMapSQL,This sample} illustrates this
    // approach achieved via a server-side SQL join.
    // <p>
    // The display value for a record with a specified <code>displayField</code> can be 
    // picked up via +link{ListGrid.getDisplayValue()}.
    //
	// @group display_values
	// @visibility external
	//<
	
    //> @attr listGridField.sortByDisplayField (boolean : null : [IRW])
    // For a field with <code>displayField</code> configured, should client-side sorting
    // be performed on the display field value? Unless explicitly set to <code>false</code>
    // the display field value is used.
    //
    // @group display_values
    // @visibility external
    //<
	
	//> @attr listGridField.optionCriteria (Criteria : null : [IRW])
	// If +link{optionDataSource} is set for this ListGridField, criteria specified in this
	// attribute will be passed to the dataSource when performing the fetch operation to
	// determine data-value to display-value mappings
	// @group display_values
	// @visibility external
	//<
	
	
	// ---------
	
	//> @attr listGridField.includeFrom (String : null : [IR])
	// Indicates this field's values should be fetched from another, related DataSource.  
	// The individual field will inherit settings such as +link{listGridField.type,field.type}
	// and +link{listGridField.title,field.title} from the related DataSource just like
	// fields from the primary DataSource.
	// <P>
	// When +link{fetchData()} is called, the automatically created +link{DSRequest} will 
	// specify +link{dsRequest.additionalOutputs} requesting the field, and any +link{Criteria} 
	// generated by the component will likewise refer to the field from the related DataSource.
	// <P>
	// It's an error to use this property if the ListGrid does not have a DataSource at all.
	// The related DataSource must be loaded or a warning will be logged and the field 
	// definition ignored.
	// <P>
	// This value is expected to be set to the following format 
	// <code>dataSourceID.fieldName</code> where <i>dataSourceID</i> is the ID of the
	// related dataSource and <i>fieldName</i> is the name of the field from that dataSource
	// from which you wish to retrieve values. Note that if this property is set and
	// +link{field.name} is not explicitly specified, this field's <code>name</code> will
	// default to the <i>fieldName</i> value from this property.
	// <p>
	// Note about automatic cache updates: "update" and "add" operations
	// submitted automatically by this ListGrid will include
	// <code>dsRequest.additionalOutputs</code> to ensure all data necessary
	// for cache updates is returned by the server.
	// <p>
	// If your grid shows data that can be edited elsewhere in the UI (not by
	// inline editing), to avoid problems with +link{ResultSet} automatic
	// cache synchronization, you may need to switch from using
	// <code>listGridField.includeFrom</code> to
	// <code>dataSourceField.includeFrom</code>.  This is because
	// server responses to "add" and "update" operations which are initiated
	// outside of this grid do not know about the
	// <code>listGridField.includeFrom</code> setting, and so will not
	// automatically return data for fields included in this way.  Switching
	// to <code>dataSourceField.includeFrom</code> ensures the field is
	// always included in server responses, avoiding the issue.
	//
	// @group display_values
	// @visibility crossDS
	//<
	
	// ----------------------------------------------------------------------------------------
	// Don't show scrollbars -- scrolling occurs in the body
    overflow:isc.Canvas.HIDDEN,	

	//>	@attr	listGrid.backgroundColor		(string : "white" : IRW)
	//		@group	appearance
	//<
	backgroundColor:"white",
    
	//>	@attr	listGrid.minHeight		(number : 50 : IRW)
	// Minimum height for the entire list (smaller than this doesn't tend to work very well).
	//		@group	sizing
	//<
    minHeight:50,							

    defaultWidth:200,

	//>	@attr listGrid.fieldIdProperty (string : "name" : IRWA)
	//  Property to be used as field identifier on listGridField objects.
	//  The ID of the field is also the property in each record which holds the value 
	//  for that field.
	//		@group	data
	//<
    // defaulted on Canvas
    

	// GridRenderer properties
	// ---------------------------------------------------------------------------------------

	//>	@attr listGrid.showAllRecords (Boolean : false : [IRW])
	// @include gridRenderer.showAllRows
    // @example autofitRows
	//<
	//showAllRecords:false,

	//>	@attr listGrid.showAllColumns (Boolean : false : IR)
	// @include gridRenderer.showAllColumns
	//<
	//showAllColumns:false,

	//>	@attr listGrid.drawAllMaxCells (int : 250 : IRWA)
	// @include gridRenderer.drawAllMaxCells
    // @group performance
    // @visibility external
	//<
	drawAllMaxCells:250,


	//>	@attr listGrid.drawAheadRatio (float : 1.3 : IRW)
	// @include gridRenderer.drawAheadRatio
    // @group performance
    // @example databoundFetch
	//<
    drawAheadRatio:1.3,

    //> @attr listGrid.quickDrawAheadRatio (float : 1.0 : IRW)
    // @include gridRenderer.quickDrawAheadRatio
    // @group performance
    //<
    quickDrawAheadRatio:1.0,

    //> @attr listGrid.instantScrollTrackRedraw (Boolean : true : IRW)
    // @include gridRenderer.instantScrollTrackRedraw
    // @group performance
    // @visibility external
    //<

    //> @attr listGrid.scrollRedrawDelay (int : 75 : IRW)
    // @include gridRenderer.scrollRedrawDelay
    // @group performance
    // @visibility external
    //<
    scrollRedrawDelay:75,

    //>	@attr listGrid.virtualScrolling (boolean : null : [IRA])
    // When incremental rendering is switched on and there are variable record heights, the virtual
    // scrolling mechanism manages the differences in scroll height calculations due to the
    // unknown sizes of unrendered rows to make the scrollbar and viewport appear correctly.
    // <P>
    // virtualScrolling is switched on automatically when fixedRecordHeights is false and when
    // using the +link{listGrid.showRecordComponents,recordComponents subsystem}, as
    // recordComponents expand the rows that contain them. This flag should be manually enabled
    // when calling +link{listGrid.addEmbeddedComponent}(...) if embedded components can
    // cause record sizes to expand beyond specified cellHeight.
    // @visibility external
    //<

    //> @attr listGrid.dataPageSize (integer : 75 : IRW)
    // @include dataBoundComponent.dataPageSize
    // @group performance
    // @visibility external
    // @example databoundFetch
    //<
    
    //> @attr listGrid.dataFetchMode (FetchMode : "paged" : IR)
    // @include dataBoundComponent.dataFetchMode
    //<

    // configures ResultSet.fetchDelay, delay in MS before fetches are triggered
    
    dataFetchDelay : 300,

    //> @attr listGrid.body (MultiAutoChild GridRenderer : null : R)
    // GridRenderer used to render the dataset.
    // <p>
    // Note that this is a multi-instance component when there are frozen fields because in
    // addition to the primary body AutoChild, a "frozen body" AutoChild is created to render
    // the frozen portion of the dataset.
    // @see ListGrid.getBody()
    // @visibility external
    //<
    body:null,

    bodyConstructor:"GridBody",

	//>	@attr listGrid.bodyOverflow (Overflow : isc.Canvas.AUTO : [IRWA])
	// Overflow setting for the "body", that is, the area of the grid where data values are
    // rendered.
    // <P>
    // By setting both the grid itself and the body to overflow:visible, it is possible to
    // "auto-fit" to the rendered height or width of the rows.  Note that in this case
    // <code>grid.width</code> and <code>grid.height</code> act as minimums.
    //
	//      @visibility external
	//		@group	sizing
    //      @example autofitRows
	//<
    bodyOverflow:isc.Canvas.AUTO,
    

	//>	@attr listGrid.bodyBackgroundColor (string : "white" : IRW)
    // Background color applied to the ListGrid body (that is, the area of the grid where
    // data values are rendered).<br>
    // Note that this will typically not be visible to the user unless there are few enough
    // rows that there is visible space in the body below the last row. To style data cells,
    // override +link{ListGrid.baseStyle} instead.
	//		@group	appearance
    // @visibility external
	//<
    bodyBackgroundColor:"white",			
    
	//>	@attr listGrid.bodyStyleName (CSSStyleName : null : IRW)
	// CSS style used for the body of this grid.  If applying a background-color to the body
    // via a CSS style applied using this property, be sure to set 
    // +link{ListGrid.bodyBackgroundColor} to <code>null</code>.
    //		@group	appearance
    // @visibility external
	//<
	//bodyStyleName:null,
    
    // whether to allow the body and header to have different border sizes and automatically
    // adjust the body column sizes to compensate such that column boundaries line up.
    allowMismatchedHeaderBodyBorder : true,

	//>	@attr listGrid.emptyCellValue (HTMLString : "&nbsp;" : IRW)
	// The value to display for cells whose value is null or the empty string after applying
    // formatCellValue and valueMap (if any).
    // <p>
    // This is the grid-wide attribute.  You may also set the emptyCellValue on a per-field basis.
    //
	// @group cellStyling
    // @see listGridField.emptyCellValue
    // @visibility external
    // @example emptyValues
	//<
	emptyCellValue:"&nbsp;",
    
	//>	@attr listGrid.cellHeight (number : 20 : [IRW])
	// @include gridRenderer.cellHeight
    // @example multilineValues
	//<
    cellHeight:20,
    
    //> @attr listGrid.normalCellHeight (number : 20 : [IRWA])
    // If +link{listGrid.baseStyle} is unset, base style will be derived from 
    // +link{listGrid.normalBaseStyle} if this grid has fixed row heights and 
    // the specified +link{listGrid.cellHeight} matches this value. Otherwise
    // +link{listGrid.tallBaseStyle} will be used.
    // @visibility external
    //<
    normalCellHeight:20,

	//>	@attr listGrid.fixedRecordHeights (Boolean : true : IRWA)
    // Should we vertically clip cell contents, or allow rows to expand vertically to show all
    // contents?
    // <P>
    // If we allow rows to expand, the row height as derived from
    // +link{gridRenderer.getRowHeight(),getRowHeight()} or the
    // default +link{cellHeight} is treated as a minimum.
    // <P>
    // <b>NOTE:</b> by default, for performance reasons, clipping is not enforced for some
    // kinds of content (such as images) on all browsers.  Set
    // +link{enforceVClipping,enforceVClipping:true} to enforce clipping for all types of
    // content on all browsers.
    //
	// @include gridRenderer.fixedRowHeights
    // @example autofitValues
	//<
	fixedRecordHeights:true,				

    //> @attr listGrid.enforceVClipping (Boolean : false : IRW)
    // For performance reasons, even when +link{fixedRecordHeights} is set, vertical clipping
    // is not enforced by default for some kinds of content (such as images) on all browsers.
    // Set +link{enforceVClipping,enforceVClipping:true} to enforce clipping for all types of
    // content on all browsers.
    // <P>
    // This additional setting is likely to be phased out as browsers improve.
    //
    // @visibility external
    //<

	//>	@attr listGrid.fixedFieldWidths (Boolean : true : IRWA)
	// Should we horizontally clip cell contents, or allow columns to expand horizontally to
	// show all contents?
	// <P>
	// If we allow columns to expand, the column width is treated as a minimum.
	// <P>
	// NOTE: the header does not automatically respond to expanded field widths. 
	// If your grid is showing a header we'd recommend developers consider 
	// setting +link{listGrid.autoFitFieldWidths} rather than using this attribute.
    // @group cellStyling
    // @visibility external
	//<
	// NOTE: doc is duplicated here because in the ListGrid we need to discuss the header.
	fixedFieldWidths:true,

    // autoFit attributes

    //> @type Autofit 
    // Possible values for to change the behavior of how data will fill the ListGrid.
    // @value "vertical" expand vertically to accommodate records.
    // @value "horizontal" expand horizontally to accommodate fields.
    // @value "both" expand horizontally and vertically to accommodate content.
    // @group autoFitData
    // @visibility external
    //<

    //> @attr listGrid.autoFitData (Autofit : null : IRW)
    // Should this ListGrid automatically expand to accommodate the size of records and fields?
    // <P>
    // Valid settings are
    // <ul><li><code>"vertical"</code>: expand vertically to accommodate records.</li>
    //     <li><code>"horizontal"</code>: expand horizontally to accommodate fields.</li>
    //     <li><code>"both"</code>: expand horizontally and vertically to accommodate content.</li>
    // </ul>
    // How far the ListGrid will expand may be limited via the following properties:
    // +link{ListGrid.autoFitMaxHeight}, +link{ListGrid.autoFitMaxRecords},
    // +link{ListGrid.autoFitMaxWidth}, +link{ListGrid.autoFitMaxColumns}.
    // <P>
    // Note that this property causes the grid as a whole to expand to fit records or fields.
    // To have the fields or records themselves expand to fit cell contents, see 
    // +link{listGrid.autoFitFieldWidths} and +link{listGrid.fixedRecordHeights}.
    // @group autoFitData
    // @visibility external
    //<
    
    //> @attr listGrid.autoFitMaxHeight (number : null : IRW) 
    // If +link{listGrid.autoFitData} is set to <code>"vertical"</code> or <code>"both"</code> this
    // property provides an upper limit on how far the ListGrid will expand vertically to accommodate
    // its content. If content exceeds this height, scrollbars will be introduced as usual.  
    // In addition to this property, +link{ListGrid.autoFitMaxRecords} allows you to limit vertical
    // expansion based on the number of rows to be rendered.
    // @group autoFitData
    // @visibility external
    //<

    //> @attr ListGrid.autoFitMaxRecords (number : 50 : IRW)
    // If +link{listGrid.autoFitData} is set to <code>"vertical"</code> or <code>"both"</code> this
    // property provides the maximum number of records for which the ListGrid will expand. If more
    // records are present, scrolling will be introduced to reach them as normal.
    // If unset, by default the ListGrid will expand to accommodate as many records as are present.
    // @group autoFitData
    // @visibility external
    //<
    autoFitMaxRecords:50,

    //> @attr ListGrid.autoFitExtraRecords (number : null : IRW)
    // If +link{listGrid.autoFitData} is set to <code>"vertical"</code> or <code>"both"</code> 
    // this property specifies the number of additional records for which the grid 
    // will expand. If more
    // records are present, scrolling will be introduced to reach them as normal.
    // @group autoFitData
    // @visibility external
    //<
//    autoFitExtraRecords:0,

    //> @attr listGrid.autoFitMaxWidth (number : null : IRW) 
    // If +link{listGrid.autoFitData} is set to <code>"horizontal"</code> or <code>"both"</code>
    // this property provides an upper limit on how far the ListGrid will expand horizontally to
    // accommodate its content. 
    // If content exceeds this width, scrollbars will be introduced as usual.  
    // In addition to this property, +link{ListGrid.autoFitMaxColumns} allows you to limit 
    // horizontal expansion based on the number of columns to be rendered.
    // @group autoFitData
    // @visibility external
    //<
    
    //> @attr ListGrid.autoFitMaxColumns (number : 50 : IRW)
    // If +link{listGrid.autoFitData} is set to <code>"horizontal"</code> or <code>"both"</code>
    // this property provides the maximum number of columns for which the ListGrid will expand.
    // If more columns are present, scrolling will be introduced to reach them as normal.
    // If unset the ListGrid will expand to accommodate as many columns as are defined for the
    // grid.
    // @group autoFitData
    // @visibility external
    //<    
    autoFitMaxColumns:50,
  
    
    //> @attr listGrid.canAutoFitFields (Boolean : true : IRW)
    // Whether the user able to autofit specific columns
    // to their data and/or title via a context menu item or +link{listgrid.headerAutoFitEvent}.
    // <P>
    // Autofitting can also be programmatically enabled by setting +link{field.autoFitWidth}.
    // @visibility external
    // @group autoFitFields
    //<
    canAutoFitFields:true,
    
    //>	@type AutoFitEvent
    // Event on a listGrid header to trigger auto-fit of the listgrid field.
    // @value "doubleClick" React to a double click on the listGrid header.
    // @value "click" React to a click on the listGrid header.
    // @value "none" No event will trigger auto-fit.
    // @group autoFitFields
    // @visibility external
    //<
        
    //> @attr listGrid.headerAutoFitEvent (AutoFitEvent : "doubleClick" : IR)
    // Event on a ListGrid header that triggers auto fitting to data and/or title.
    // <P>
    // Note that if sorting is enabled for the field and the headerAutoFitEvent is "click", both
    // sorting and autofit occur on a click.
    // @visibility external
    // @group autoFitFields
    //<
    headerAutoFitEvent:"doubleClick",
    
    //> @attr listGridField.autoFitWidth (Boolean : null : IR)
    // Enables autofitting to values or titles for this field.. This overrides
    // the +link{listGrid.autoFitFieldWidths} attribute on a per-field basis.
    //
    // @visibility external
    // @group autoFitFields
    //<
        
    //> @attr listGrid.autoFitFieldWidths (Boolean : null : IR)
    // Enables autofitting of fields to values or titles. This property may be overridden
    // on a per-field basis via +link{listGridField.autoFitWidth}.
    // <P>
    // The +link{listGrid.autoFitWidthApproach} controls whether fitting is to values, titles 
    // or both. This property may also be overridden on a per field basis.
    // <P>
    // If +link{field.width} is also set on the field, it will be taken as a minimum width.
    // +link{ListGrid.minFieldWith} will also be respected. Developers may also use
    // the +link{listGrid.autoFitFieldsFillViewport} setting to ensure that all available
    // space is used even if the field content isn't wide enough to fill the listGrid
    // viewport.
    // <P>
    // Autofitting will be performed:
    // <ul>
    //  <li> whenever the dataset is completely changed or rows are added or removed
    //  <li> whenever a field which is autofitting is changed
    //  <li> on a manual call to +link{listGrid.autoFitField()} or
    //       +link{listGrid.autoFitFields()}
    // </ul>
    // Autofitting behavior continues until the user resizes the field manually, at which
    // point it stops. The user can also perform a one-time auto-fit of fields via
    // the header context menu if +link{listGrid.canAutoFitFields} is enabled.
    // <P>
    // When autofitting to column values, +link{listGrid.getDefaultFieldWidth()} will be
    // called to determine the space required for a field's values. This method
    // uses values from the rendered set of rows to
    // calculate the required column width. The values used not match the complete set of data
    // for the grid when rendering rows incrementally. See +link{listGrid.showAllRecords} and
    // +link{listGrid.drawAheadRatio}) to control incremental rendering of rows.
    // <P>
    // Note that for <code>icon</code> type fields, the +link{listGrid.autoFitIconFields} 
    // property setting may turn on auto-fit-width behavior for specific fields by default,
    // even if <code>autoFitFieldWidths</code> is false for the grid as a whole.
    // <P>
    // Using this feature has a performance penalty roughly comparable to always rendering 
    // one additional field per field where autofitting is enabled.  Specifically, enabling it
    // for all fields would be comparable to <i>both</i> doubling the number of fields
    // <i>and</i> disabling +link{listGrid.showAllColumns,horizontal incremental rendering}.
    // In a grid where only half the fields are normally visible and hence only half are
    // normally rendered, this would be roughly 4 times slower overall.
    // <P>
    // This performance penalty is a result of +link{getDefaultFieldWidth()} having to
    // render out the data set offscreen and measure the rendered content - it does not apply
    // for cases where this method can return a simple fixed values (as with icon fields).
    // <P>
    // Which fields are currently autofitting is saved as part of the 
    // +link{getViewState,view state} of the ListGrid.
    //
    // @visibility external
    // @group autoFitFields
    //<
        
    //> @attr listGrid.autoSizeHeaderSpans (Boolean : false : IR)
    // If this listGrid has specified +link{listGrid.headerSpans}, setting this
    // attribute to true will cause spans to expand to accomodate long titles if necessary.
    // @visibility external
    // @group headerSpan
    // @group autoFitFields
    //<
    autoSizeHeaderSpans:false,

    
    //> @attr listGrid.autoFitClipFields (Array of String : null : IR)
    // If +link{listGrid.autoFitFieldWidths} is enabled and the calculated field sizes
    // are wide enough that horizontal scrolling would be introduced, this attribute may be
    // set to an array of fieldNames, causing those fields to be clipped rather than
    // forcing horizontal scrollbars to appear.
    // <P>
    // Note: If any +link{ListGridField.frozen,frozen columns} are included in this list they
    // will not be clipped.
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @attr listGrid.autoFitFieldsFillViewport (Boolean : true : IR) 
    // If +link{listGrid.autoFitFieldWidths} is enabled, and extra space is available after
    // autofitting all fields, should the grid automatically expand one field to fill the extra
    // space.
    // <P>
    // When enabled, the field to expand may be specified via +link{autoFitExpandField}.
    // <P>
    // Note this logic will not expand a +link{ListGridField.frozen,frozen column}.
    // 
    // @group autoFitFields
    // @visibility external
    //<
    autoFitFieldsFillViewport:true,
    
    //> @attr listGrid.autoFitExpandField (String : null : IR)
    // The field to expand if +link{listGrid.autoFitFieldWidths} and 
    // +link{autoFitFieldsFillViewport} are enabled and
    // auto-fitting will not fill all available horizontal space.
    // <P>
    // If unset, will default to the text field with the longest 
    // +link{dataSourceField.length} if length is set, otherwise, the first text
    // field with no width specified.
    // <P>
    // Note that expanding +link{ListGridField.frozen,frozen columns} is not supported.
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @type AutoFitWidthApproach
    // How should field width be determined when +link{listGridField.autoFitWidth} is true?
    // @value "value" Size field to fit to the data value(s) contained in the field.
    // @value "title" Size field to fit the field title
    // @value "both" Size field to fit either the field title or the data values in the field
    //  (whichever requires more space).
    //
    // @group autoFitFields
    // @visibility external
    //<

    //> @attr listGrid.autoFitWidthApproach (AutoFitWidthApproach : "value" : [IRW])
    // When a user requests column autofitting via the 
    // +link{getHeaderContextMenuItems,header contextmenu} or via a 
    // +link{headerAutoFitEvent,mouse gesture}, what autofit approach is used.
    // @group autoFitFields
    // @visibility external
    //<
    autoFitWidthApproach:"value",
    
    // If we're autoFitting to values, and we're showing a grid summary, should the 
    // grid summary values be taken into account when sizing columns?
    // By default they are - set this flag to false to disable this behavior.
    includeGridSummaryInAutoFitWidth:true,
    
    //> @attr listGridField.autoFitWidthApproach (AutoFitWidthApproach : null : [IRW])
    // When a user requests column autofitting via the 
    // +link{getHeaderContextMenuItems,header contextmenu} or via a 
    // +link{headerAutoFitEvent,mouse gesture}, what autofit approach is used. If set, this
    // setting overrides the autoFitWidthApproach specified at the ListGrid level.
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @type AutoFitIconFieldType
    // How should fields of +link{listGridFieldType,type:"icon"} be sized by default?
    // @value "none" Apply no special sizing to icon fields - treat them like any other 
    //   field in the grid
    // @value "iconWidth" size the field to accommodate the width of the icon
    // @value "title" size the field to accommodate the title (or the width of the icon if
    //   it exceeds the width of the title.
    //
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @attr listGrid.autoFitIconFields (AutoFitIconFieldType : "title" : [IRW])
    // SmartClient listGrids have special logic to automatically size fields that
    // are displayed as an icon - that is fields with 
    // +link{listGridFieldType,type:"icon"}, fields displaying only
    // +link{listGridField.showValueIconOnly,value icons}, and boolean fields (which
    // are rendered as a checkmark type icon by default.
    // <P>
    // This attribute controls this behavior - governing whether icon fields should
    // be sized to fit their content (icon), title, or whether to disable this 
    // behavior. Setting this value to <code>"title"</code> or <code>"iconWidth"</code>
    // will cause +link{listGridField.autoFitWidth} to be enabled by default for all 
    // icon fields with the +link{listGridField.autoFitWidthApproach} set to 
    // <code>"value"</code> or <code>"both"</code> as appropriate. Note that the
    // width required for the icons is calculated by +link{listGrid.getDefaultFieldWidth()}
    // which performs a simple calculation based on the specified icon width for these
    // types of fields.
    // <P>
    // This setting governs default behavior for icon fields - for specific fields within
    // a grid, this default behavior can be overridden by setting an explicit
    // +link{listGridField.width} or
    // explicitly enabling +link{listGridField.autoFitWidth} and setting
    // +link{listGridField.autoFitWidthApproach} on the field in question.
    // @see listGrid.autoFitFieldWidths
    // @group autoFitFields
    // @visibility external
    //<
    autoFitIconFields:"title",
    
    //> @attr listGrid.leaveScrollbarGap (Boolean : true : IRW)
	// Whether to leave a gap for the vertical scrollbar, even when it's not present.
    // <P>
    // Note that if leaveScrollbarGap is false and vertical scrolling is introduced, fields
    // will be resized to fit the smaller body area if possible, in order to avoid horizontal
    // scrolling also being required.
    //
    // @group appearance
    // @visibility external
    // @example autofitRows
    //<
    
	leaveScrollbarGap:true,
	
    // leaveScrollbarGap has meaning at the Layout level. We don't expect to ever show a vscrollbar
    // at the actual ListGrid level so override 'getBreadth' to avoid leaving a gap outside the
    // actual component members (header, body etc).
    getBreadth : function () {
        return this.getInnerWidth();
    },

	// if leaveScrollbarGap is false, whether to resize fields when vscrolling is introduced
    resizeFieldsForScrollbar:true,

	//>	@attr listGrid.autoFit (boolean : false : IRWA)
	// If true, make columns only wide enough to fit content, ignoring any widths specified.
	// Overrides fixedFieldWidths.
	// <P>
	// NOTE: the header does not automatically respond to expanded field widths
	//	@group	sizing
	//<
	
	//autoFit:false,

	//>	@attr listGrid.wrapCells (Boolean : false : IRWA)
    // Should content within cells be allowed to wrap?
    // <P>
    // Even if content is allowed to wrap, if +link{fixedRecordHeights} is set, the content
    // will be clipped off at the cell boundary.  Either set a larger, fixed +link{cellHeight}
    // to reveal more content, or set +link{fixedRecordHeights} to false to allow auto-sizing.
    // 
    // @example autofitValues
    // @visibility external
	//<
	//wrapCells:false,

    //> @attr listGrid.showClippedValuesOnHover (Boolean : null : IRA)
    // @include gridRenderer.showClippedValuesOnHover
    //<
    showClippedValuesOnHover:null,

	//>	@attr listGrid.cellSpacing (number : 0 : [IRW])
	// @include gridRenderer.cellSpacing
    // @visibility internal
	//<
    
	cellSpacing:0,

	//>	@attr listGrid.cellPadding (number : 2 : [IRW])
	// @include gridRenderer.cellPadding
	//<
	cellPadding:2,

    //> @attr listGrid.dateFormatter (DateDisplayFormat : null : [IRW])
    // How should Date type values be displayed in this ListGrid by default?
    // <P>
    // This property specifies the default DateDisplayFormat to apply to Date values
    // displayed in this grid for all fields except those of +link{listGridField.type,type "time"}
    // (See also +link{listGrid.timeFormatter}).<br>
    // If +link{listGrid.datetimeFormatter} is specified, that will be applied by default
    // to fields of type <code>"datetime"</code>.
    // <P>
    // Note that if +link{listGridField.dateFormatter} or +link{listGridField.timeFormatter} are
    // specified those properties will take precedence over the component level settings.
    // <P>
    // If unset, date values will be formatted according to the system wide 
    // +link{Date.setShortDisplayFormat(),short display format} or 
    // +link{Date.setShortDatetimeDisplayFormat(),short datetime display format} for datetime type
    // fields.
    // <P>
    // If this field is editable the dateFormatter will also be passed to the editor created
    // to edit this field as +link{DateItem.dateFormatter, dateFormatter}.
    // In this case you may also need to set +link{listGrid.dateInputFormat}.
    //
    // @visibility external
    //<
    //dateFormatter:null,
    
    //> @attr listGrid.datetimeFormatter (DateDisplayFormat : null : [IRW])
    // Display format to use for fields specified as type 'datetime'.  Default is to use the
    // system-wide default date time format, configured via
    // +link{Date.setShortDatetimeDisplayFormat()}.  Specify any
    // valid +link{type:DateDisplayFormat} to change the display format for datetimes used by this grid.
    // <var class="smartclient">
    // May be specified as a function. If specified as  a function, this function will be executed in the scope of the Date
    // and should return the formatted string.
    // </var>
    // <P>
    // May also be specified at the field level via
    // +link{listGridField.dateFormatter}
    // <P>
    // If this field is editable the dateFormatter will also be passed to the editor created
    // to edit this field as +link{DateItem.dateFormatter, dateFormatter}.
    // In this case you may also need to set +link{listGrid.dateInputFormat}.
    // 
    // @see listGridField.dateFormatter
	// @group appearance
    // @visibility external
    //<
    
    //> @attr listGrid.dateInputFormat (DateInputFormat : null : [IRWA])
    // If this is an editable listGrid, this property will specify the 
    // +link{DateItem.inputFormat, inputFormat} applied to editors for fields of type 
    // <code>"date"</code>. May be overridden per field via +link{listGridField.inputFormat}.
    // @see listGrid.dateFormatter
    // @visibility external
    //<

	// function to call appropriate date formatter
	// Note: this is executed in the scope of a field object - see 'applyFieldDefaults'
    _formatDateCellValue : function (value, field, grid, record, rowNum, colNum) {
    
        if (isc.isA.Date(value)) {
            // A developer may force a "date" or "datetime" type field value to be displayed as time
            // by specifying a timeFormatter and no dateFormatter on the field.
            if (grid._formatAsTime(field)) {
                var formatter = grid._getTimeFormatter(field);
                var isLogicalTime = isc.SimpleType.inheritsFrom(field.type, "time");
                return isc.Time.toTime(value, formatter, isLogicalTime);
            }

            var isDatetime = field && isc.SimpleType.inheritsFrom(field.type, "datetime"),
                isLogicalDate = !isDatetime && isc.SimpleType.inheritsFrom(field.type, "date"),
                formatter = grid._getDateFormatter(field);
            
            // rely on date.toShortDateTime() / toShortDate() to handle applying the
            // custom formatter if specified, otherwise picking up the appropriate system-wide
            // default for the data type.
            // The second parameter to toShortDateTime() explicitly causes the date to be displayed
            // in the custom timezone set up in Time.setDefaultDisplayTimezone
            if (isDatetime) return value.toShortDateTime(formatter, true);
            return value.toShortDate(formatter, !isLogicalDate);
        }
        return value;
    },
    
    // Date formatting helpers:
    // Called from formatDateCellValue() / formatTimeCellValue() [which are type-specific formatters
    // applied to fields as part of field init], and also as a catch-all for
    // Date type values in fields of some other specified data-type.
    // We use a consistent pattern across DataBoundComponents:
    // - developer can specify explicit dateFormatter / timeFormatter per field and they'll be used.
    //   if both are specified dateFormatter takes precedence except in "time" type fields.
    // - developer can specify dateFormatter, datetimeFormatter and timeFormatter per component and
    //   they'll be used if no per-field settings are found.

    // If a field has a JS Date value, should we format it as a time? True for "time" type fields
    // or fields with an explicit time formatter only.
    _formatAsTime : function (field) {
        if (field == null) return false;
        
        // If at the field level the timeFormatter is defined *(and there's no date formatter)
        // respect it.
        if (field.timeFormatter != null && field.dateFormatter == null) return true;
        // timeFormatter null and dateFormatter non-null --> format as date
        
        if (field.dateFormatter != null && field.timeFormatter == null) return false;
        return isc.SimpleType.inheritsFrom(field.type, "time");
    },
    
    _getDateFormatter : function (field) {
        
        if (field == null) return this.dateFormatter;
        
        if (field.dateFormatter != null) return field.dateFormatter;
        // displayFormat is back-compat at this point - only applies to fields of type
        // date or datetime (or subtypes thereof)
        if (field.displayFormat != null && isc.SimpleType.inheritsFrom(field.type, "date")) {
            return field.displayFormat;
        }
        
        if (this.datetimeFormatter != null && isc.SimpleType.inheritsFrom(field.type, "datetime")) {
            return this.datetimeFormatter;
        }
        
        return this.dateFormatter;
    },
    
    // This picks up *explicit* dateInputFormat to pass through to the edit-item. No need to
    // include logic to derive from the display format if no explicit input format was specified,
    // that'll be handled by the FormItem code.
    _getDateInputFormat : function (field) {
        var inputFormat;
        if (field) inputFormat = field.inputFormat
        if (!inputFormat) inputFormat = this.dateInputFormat;
        return inputFormat;
    },

    // function to call appropriate number formatter
    // If no number formatter is defined, the default formatter will be used [standard 'toString']
    // is the default
    _formatNumberCellValue : function (value, field, grid, record, rowNum, colNum) {
        if (isc.isA.Number(value)) {
            if (isc.SimpleType.inheritsFrom(field.type, "float") && 
                (field.decimalPrecision != null || field.decimalPad != null)) 
            {
                return isc.Canvas.getFloatValueAsString(
                        value, field.decimalPrecision, field.decimalPad);
            } else if (field.precision != null) {
                return isc.Canvas.getNumberValueAsString(value, field.precision, field.type);
            } else {
                var formatter = (field.numberFormatter || field.formatter || grid.numberFormatter);
                return value.toFormattedString(formatter);
            }
        }

        // If passed a non-number just return it
        return value;
    },
    
	//>	@attr listGrid.timeFormatter (TimeDisplayFormat : "toShortPaddedTime" : [IRW])
	// Display format to use for fields specified as type 'time'.  May also be specified at 
    // the field level via +link{listGridField.timeFormatter}.<br>
    // If unset, time fields will be formatted based on the system wide 
    // +link{Time.shortDisplayFormat}.<br>
    // If this field is editable, the timeFormatter will also be passed to the editor
    // created to edit any time type fields as +link{TimeItem.timeFormatter}
	// @group appearance
    // @visibility external
	//<
    timeFormatter:"toShortPaddedTime",
    
    _getTimeFormatter : function (field) {
        if (field != null) {
            if (field.timeFormatter != null) return field.timeFormatter;
            if (field.displayFormat != null && isc.SimpleType.inheritsFrom(field.type, "time")) {
                return field.displayFormat;
            }
        }
        return this.timeFormatter;
    },

	// function to call appropriate time formatter
	// Note: this is executed in the scope of a field object - see 'applyFieldDefaults'
    _formatTimeCellValue : function (value, field, grid, record, rowNum, colNum) {
        var time = value;
        if (isc.isA.String(time)) {
            // Pass in the 'validTime' param - If we're given a string which doesn't 
            // parse to a time we don't want to display "12:00 am"
            time = isc.Time.parseInput(time, true);
        }
        if (isc.isA.Date(time)) {
            // If dateFormatter is set on the field, and timeFormatter is not, we respect it
            // even for fields of explicit type "time"
            if (!grid._formatAsTime(field)) {
                
                return time.toShortDate(grid._getDateFormatter(field), true);
            }
            var formatter = grid._getTimeFormatter(field);
            
            // If we're passed an invalid formatter
            return isc.Time.toTime(time, formatter, true);
        }
        return value;
    },
    
    _formatBinaryCellValue : function (value, field, grid, record, rowNum, colNum) {
        
        if (isc.isA.String(value)) return value;
        if (record == null) return null;

        var fieldName = field.name,
            ds = grid.getDataSource(),
            filenameField = (ds ? ds.getFilenameField(fieldName) : null) || fieldName + "_filename",
            fileName = record[filenameField],
            value
        ;

        if (field.type=="imageFile" && field.showFileInline == true) {
            var urlProperty = fieldName + "_imgURL";
            
            if (!record[urlProperty]) {
                var dimensions = isc.Canvas.getFieldImageDimensions(field, record), 
                    image = grid.getDataSource().streamFile(record, field.name);
                    
                dimensions.width = dimensions.width || grid.imageSize;
                dimensions.height = dimensions.height || grid.imageSize;
                value = record[urlProperty] = 
                    isc.Canvas.imgHTML(image, dimensions.width, dimensions.height);
            } else 
                value = record[urlProperty];
        } else {
            if (field.showFileInline == true) { // non-imageFile field
	            this.logWarn("_formatBinaryCellValue(): Unsupported field-type for showFileInline: "+field.type);
            }

            
            if (!field.filenameSuppressed && (fileName == null || isc.isAn.emptyString(fileName))) {
                return this.emptyCellValue;
            }

            var viewAction = "'" + grid.getID() +".view",
                dlAction = "'" + grid.getID() +".download",
                completion = "";
            if (field && field.name) {
                completion = "Cell(" + rowNum + ", \"" + field.name + "\")'";
            } else {
                completion = "Row(" + rowNum + ")'";
            }
            var viewIconHTML = isc.Canvas.imgHTML("[SKIN]actions/view.png", 16, 16, null,
                            "style='cursor:"+isc.Canvas.HAND+"' onclick="+viewAction+completion);
            var downloadIconHTML = isc.Canvas.imgHTML("[SKIN]actions/download.png", 16, 16, null,
                            "style='cursor:"+isc.Canvas.HAND+"' onclick="+dlAction+completion);

            value = viewIconHTML + "&nbsp;" + downloadIconHTML + 
                (fileName ? "&nbsp;" + fileName : "");
        }
        
        return value;
    },

    // value is rendered as an anchor - the href and name of the anchor is the value.  By default
    // opens in a new browser window - this can be overridden by setting the 'target' property on
    // the record.
    //
    // The name of the link can be overridden by setting the 'linkName' property on the record.  By
    // default we use the value.
    _$linkTemplate:[
        "<a href='",
        ,   // 1: HREF
        "' target='",
        ,   // 3: name of target window
        // onclick handler enables us to prevent popping a window if (EG) we're masked.        
        //                      5: ID              
        "' onclick='if(window.",     ,") return ",
                //  7:ID                         9:rowNum,     11:colNum        
                         ,"._linkClicked(event,",        ,",",          ,");'>",
        ,   // 13: link text
        "</a>"
    ],
    _$doubleEscapedQuote:"\\'",
    _$_blank:"_blank",
    
    _formatLinkCellValue : function (value, field, grid, record, rowNum, colNum) {
        if (value == null || isc.is.emptyString(value)) return value;

        // target window
        var target = field.target ? 
                        field.target.replaceAll(grid._$singleQuote, grid._$doubleEscapedQuote) : 
                        grid._$_blank;
        // get the linkText property. If defined on the field, use that, otherwise 
        // use the linkTextProperty from the grid.
        var linkTextProp = field.linkTextProperty ? field.linkTextProperty : grid.linkTextProperty;
        var linkText = (record && record[linkTextProp]) ? record[linkTextProp] 
                                                            : field.linkText || value;

        // link URL
        var href = value;

        if (field.linkURLPrefix) href = field.linkURLPrefix + href;
        if (field.linkURLSuffix) href = href + field.linkURLSuffix;

        href = href.replaceAll(grid._$singleQuote, grid._$doubleEscapedQuote);

        if (target == "javascript") {
            // target is "javascript" - make the link inert and have the cellClick event fired 
            // instead
            href = "javascript:void";
        }

        // combine
        var template = grid._$linkTemplate;
        template[1] = href;
        template[3] = target;
        var ID = grid.getID();
        template[5] = ID;
        template[7] = ID;
        template[9] = rowNum;
        template[11] = colNum;
        template[13] = linkText;

        return template.join(isc.emptyString);
    },
    
    _linkClicked : function (event, rowNum, colNum) {
        // don't allow the click if the cell should not be interactive.
        var mustCancel = (this.destroyed || !this.isDrawn() || !this.isVisible() ||
                          isc.EH.targetIsMasked(this.body) || !this.recordIsEnabled(rowNum, colNum));

        var record = this.getRecord(rowNum),
            field = this.getField(colNum);

        if (event.target == "javascript" || field.target == "javascript") {
            mustCancel=true;
            this.cellClick(record, rowNum, colNum);
        }

        if (mustCancel) {
            
            if (!isc.Browser.isIE) {
                event.preventDefault();
            }
            
            return false;
        }
        return true;
    },
    
    //> @attr listGrid.linkTextProperty (string : "linkText" : [IRW])
	// Property name on a record that will hold the link text for that record.
	// <P>
	// This property is configurable to avoid possible collision with data values in the
    // record.
    // <P>
    // Use +link{listGridField.linkTextProperty} if you have more than one link field and 
	//
	// @see type:ListGridFieldType
    // @see type:FieldType
    // @see attr:listGridField.linkText
    // @see attr:listGridField.linkTextProperty
	// @group  display_values
	// @visibility external
	//<
    linkTextProperty : "linkText",
    
    // value is a URL to an image
    _formatImageCellValue : function (value, field, grid, record, rowNum, colNum) {
        // if no value is stored, just return an empty string so we don't render a broken image
        if (value == null || value == isc.emptyString) return isc.emptyString;

        // if any of field.imageWidth/Height/Size are set as strings, assume they are property
        // names on the record
        
        var dimensions = isc.Canvas.getFieldImageDimensions(field, record);

        dimensions.width = dimensions.width || grid.imageSize;
        dimensions.height = dimensions.height || grid.imageSize;

        
        var src = value,
            prefix = field.imageURLPrefix || field.baseURL || field.imgDir;

        // If imageURLSuffix is specified, apply it to the value
        if (field.imageURLSuffix != null) src += field.imageURLSuffix;            
         
        

        return isc.Canvas.imgHTML(src, dimensions.width, dimensions.height, null, field.extraStuff, 
                                  prefix, field.activeAreaHTML);
    },

    // show field.icon in the cell
    _formatIconCellValue : function (value, field, grid, record, rowNum, colNum) {
        // prevent an icon from being shown in the filter editor if the field has canFilter
        if (isc.isA.RecordEditor(grid) && grid.isAFilterEditor() && field.canFilter == false) return null;
        
        if (field._iconHTML) return field._iconHTML;
        
        field._iconHTML = isc.Canvas.imgHTML(field.cellIcon || field.icon, 
                                             field.iconWidth || field.iconSize || grid.imageSize,
                                             field.iconHeight || field.iconSize || grid.imageSize);
        return field._iconHTML;
    },

	// CSS styles
	// --------------------------------------------------------------------------------------------

    //> @attr listGrid.fastCellUpdates (Boolean: true : I)
    // @include gridRenderer.fastCellUpdates
    // @group performance
    //<
    // explicitly set fastCellUpdates at the LG level 
    // this will be passed through to our body and allows us to check
    // this.fastCellUpdates directly rather than looking at the attribute on the body
    
    fastCellUpdates:isc.Browser.isIE && !isc.Browser.isIE9,
    
    //> @method listGrid.setFastCellUpdates()
    // @include gridRenderer.setFastCellUpdates()
    // @visibility external
    //<
    // explicit implementation keeps this.fastCellUpdates in synch with the version in the
    // body so we can check it directly in this.getBaseStyle
    setFastCellUpdates : function (fcu) {
        if (this.body != null) {
            this.body.setFastCellUpdates(fcu);
            // if the body refused to set to the specified 
            // value, respect that.
            fcu = this.body.fastCellUpdates;
        }
        if (this.frozenBody != null) {
            this.frozenBody.setFastCellUpdates(fcu);
        }
        this.fastCellUpdates = fcu;
    },
	
	//>	@attr listGrid.baseStyle (CSSStyleName : null : [IR])
	// +link{gridRenderer.baseStyle,base cell style} for this listGrid.
	// If this property is unset, base style may be derived from +link{listGrid.normalBaseStyle}
	// or +link{listGrid.tallBaseStyle} as described in
	// +link{listGrid.getBaseStyle()}.
    // @visibility external
    // @group appearance
	//<
	
	//>	@attr listGrid.normalBaseStyle (CSSStyleName : "cell" : [IR])
	// "Normal" baseStyle for this listGrid. Only applies if +link{listGrid.baseStyle} is 
	// set to null.
	// <P>
	// If <code>baseStyle</code> is unset, this
	// property will be used as a base cell style if the grid is showing fixed height rows, and
	// the specified cellHeight matches +link{listGrid.normalCellHeight} (and in Internet Explorer,
	// +link{listGrid.fastCellUpdates} is false). Otherwise +link{listGrid.tallBaseStyle} will
	// be used.
	// <P>
	// Having separate styles defined for fixed vs. variable height rows allows the developer
	// to specify css which is designed to render at a specific height (typically using
	// background images, which won't scale), without breaking support for styling rows
	// of variable height.
	//
	// @see listGrid.getBaseStyle()
	// @visibility external
	//<
	normalBaseStyle:"cell",

	//>	@attr listGrid.tallBaseStyle (CSSStyleName : "cell" : [IR])
	// "Tall" baseStyle for this listGrid. Only applies if +link{listGrid.baseStyle} is 
	// set to null.
	// <P>
	// If <code>baseStyle</code> is unset, this
	// property will be used as a base cell style unless the grid is showing fixed height
	// rows with a specified cellHeight that matches +link{listGrid.normalCellHeight}, in
	// which case +link{listGrid.normalBaseStyle} will be used. Note that in Internet Explorer
	// if +link{listGrid.fastCellUpdates} is true, <code>tallBaseStyle</code> will also be
	// used even if the cellHeight matches the specified <code>normalCellHeight</code> for the
	// grid.
	//
	// @see listGrid.getBaseStyle()
	// @visibility external
	//<
	tallBaseStyle:"cell",

	
    
    
    //>	@attr listGrid.editFailedBaseStyle (CSSStyleName : null : [IRWA])
	//  A base name for the CSS class applied to cells when editing has failed.<br>
	//  If this listGrid is editable, this style will be applied to any edited cells for which
	//  validation failed.<br>
	//  As with the default 'baseStyle' property, this style will have "Dark", "Over", "Selected", 
	//  or "Disabled" appended to it according to the state of the cell.<br>
    // If null, cells for which editing has failed will be rendered using the normal base style
    // classNames, but with custom CSSText applied as derived from <code>this.editFailedCSSText</code>
	// @visibility external
	// @group	appearance
	// @see baseStyle
    // @see editFailedCSSText
	//<
    editFailedBaseStyle:null,   //"cellEditFailed",
    
	//>	@attr listGrid.editFailedCSSText (string : "color:red;border:1px solid red;" : [IRWA])
	//  Custom CSS text to be applied to cells when editing has failed.<br>
	//  If this listGrid is editable, this css text will be applied to any edited cells for which
	//  validation failed, on top of the base style for the cell.<br>
    // For further customization of styling for cells that failed editing validation, use
    // <code>this.editFailedBaseStyle</code> instead.
	// @visibility external
	// @group	appearance
    // @see editFailedBaseStyle
	//<
    editFailedCSSText:"color:red;border:1px solid red;",
    
	//>	@attr listGrid.editPendingBaseStyle (CSSStyleName : null : [IRA])
	// A base name for the CSS class applied to cells containing pending (unsaved) edits<br>
	// As with the default 'baseStyle' property, this style will have "Dark", "Over", "Selected", 
	// or "Disabled" appended to it according to the state of the cell.
    // <P>
    // If this property is null (the default setting), cells with pending edits will pick up
    // custom css text to be applied on top of the normal base style from
    // <code>this.editPendingCSSText</code>.
    //
	// @group appearance
	// @see baseStyle
	// @visibility external
	//<    
    editPendingBaseStyle:null, //"cellEditPending",

	//>	@attr listGrid.editPendingCSSText (string : "color:#0066CC;" : [IRWA])
	// Custom CSS text to be applied to cells with pending edits that have not yet been
    // submitted.<br>
    // For further customization of styling for cells with pending edits use
    // <code>this.editPendingBaseStyle</code> instead.
	// @group appearance
    // @see editFailedBaseStyle
	// @visibility external
	//<
    editPendingCSSText:"color:#0066CC;",

    //> @attr listGrid.recordCustomStyleProperty (String : "customStyle" : IRW)
    // @include GridRenderer.recordCustomStyleProperty
    // @visibility external
    // @see listGrid.getCellStyle()
    // @see listGrid.recordBaseStyleProperty
    //<
    recordCustomStyleProperty:"customStyle",
    
    //> @attr listGrid.recordBaseStyleProperty (string : "_baseStyle" : [IRWA])
    // This attribute allows custom base styles to be displayed on a per-record basis.
    // To specify a custom base-style for some record set 
    // <code>record[listGrid.recordBaseStyleProperty]</code> to the desired base style name - 
    // for example if <code>recordBaseStyleProperty</code> is <code>"_baseStyle"</code>, set
    // <code>record._baseStyle</code> to the custom base style name.
    // 
    // @visibility external
    // @group appearance
    // @see listGrid.baseStyle
    //<
    recordBaseStyleProperty:"_baseStyle",
    
    //> @attr listGrid.frozenBaseStyle (string : null : [IRW])
    // If this listGrid contains any frozen fields, this property can be used to apply a custom
    // baseStyle to all cells in those frozen fields. If unset, the standard base style will be
    // used for both frozen and unfrozen cells.
    // @visibility external
    // @group appearance, frozenFields
    // @see listGrid.baseStyle
    // @see listGridField.frozen
    //<
    
    //> @attr listGrid.shrinkForFreeze (Boolean : false : IRWA)
    // If this list grid is showing any +link{listGridField.frozen,frozen} fields, and a horizontal
    // scrollbar is visible at the bottom of the liquid columns, should an equivalent scrollbar gap
    // be left visible below the frozen columns?<br>
    // Note that if set to <code>true</code> any backgroundColor or border applied to the ListGrid
    // will show up below the bottom row of the frozen column(s).
    // @group frozenFields
    // @visibility external
    //<
    shrinkForFreeze:false,
    
    //> @attr listGrid.alternateRecordStyles (Boolean : false : [IRW])
    // @include gridRenderer.alternateRowStyles
	// @group cellStyling
    // @example gridCells
    //<
	//alternateRecordStyles:false,
    
    //> @attr listGrid.alternateBodyStyleName (CSSStyleName : null : [IRWA])
    // Optional css style to apply to the body if +link{listGrid.alternateRecordStyles} is true 
    // for this grid. If unset +link{listGrid.bodyStyleName} will be used to style the body
    // regardless of the +link{listGrid.alternateRecordStyles,alternateRecordStyles} setting.
    // @visibility external
    //<

	//>	@attr listGrid.alternateRecordFrequency (number : 1 : [IRW])
	// @include gridRenderer.alternateRowFrequency
	// @group cellStyling
	//<
	alternateRecordFrequency:1,

    // property you can set per-record to add custom CSSText
    recordCSSTextProperty : "cssText",

	//>	@attr listGrid.showHiliteInCells (boolean : false : IRWA)
	// When cell styling is being updated (updateCellStyle()), should the HTML content of the
	// cell also be updated?  If false, only the cell's CSS styling will be updated. 
	//
	// You should turn this on if you've implemented a <code>formatCellValue</code> function 
	// that adds styling cues to a cell (like an inline image), which need be updated as the 
	// cell switches states.
	// (eg, if you would use different HTML for a selected cell's contents).
	//		@group	hiliting, drawing
	//<
	//showHiliteInCells:false,				

    //> @attr listGrid.hiliteCanReplaceValue (boolean : null : IR)
    // If set, end users can create advanced hiliting rules that will use the
    // +link{hilite.replacementValue} feature to cause values in hilited cells
    // to be replaced with a user-entered value.  For example, a user could create a hilite rule
    // that replaces numeric values ranging from 0.5 to 1.0 with the text "LOW".
    // <p>
    // Specifically, when the "Add Advanced Rule" button is pressed and
    // <code>hiliteCanReplaceValue</code> is true, the user will see a text entry field titled
    // "Replace value with" (+link{hiliteReplaceValueFieldTitle}) and if they enter a value, that
    // value will appear in the grid cell in lieue of the cell's original value.
    //
    // @group hiliting
    // @visibility external
    //<

    //> @attr listGrid.hiliteReplaceValueFieldTitle (string : "Replace value with" : IR)
    // Title used for the text box shown when +link{listGrid.hiliteCanReplaceValue} is set.
    // @group i18nMessages
    // @visibility external
    //<
    hiliteReplaceValueFieldTitle : "Replace value with",

    //> @attr listGrid.showSelectedStyle (Boolean : true : IRW )
    // @include gridRenderer.showSelectedStyle
    //<
    showSelectedStyle : true,

    // Keyboard handling
    // ---------------------------------------------------------------------------------------

    //> @attr listGrid.generateClickOnSpace (Boolean : true : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits space, 
    // the cell will respond to a click event.
    // @visibility external
    //<
    generateClickOnSpace : true,
		
    //> @attr listGrid.generateClickOnEnter (Boolean : false : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits Enter, 
    // the cell will respond to a click event.
    // @visibility external
    //<
    //generateClickOnEnter : false,

    //> @attr listGrid.generateDoubleClickOnSpace (Boolean : false : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits Space, 
    // the cell will respond to a double click event.
    // @visibility external
    //<
    // generateDoubleClickOnSpace : false,
    
    //> @attr listGrid.generateDoubleClickOnEnter (Boolean : true : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits Enter, 
    // the cell will respond to a double click event.
    // @visibility external
    //<
    generateDoubleClickOnEnter : true,

    //> @attr listGrid.arrowKeyAction (string : null : IRWA)
    // Action to perform when the listGrid has keyboard focus (but not editing focus) and a user
    // presses the up or down arrow key. Possible values are:
    // <ul>
    // <li><code>"select"</code> : select the next row in the list (calls <code>recordClick</code> handler)</li>
    // <li><code>"focus"</code> : move focus to the next row in the list without changing the selection</li>
    // <li><code>"activate"</code> : select and activate the next row in the list (calls
    //  <code>recordDoubleClick</code> handler)</li>
    // <li><code>"none"</code> : no action</li>
    // <li> <code>null</code> : if +link{listGrid.selectionAppearance} is "checkbox", behaves as if set
    //  to "focus"; otherwise, behaves as if set to "select"</li>
    // </ul>
    // @group events
    // @visibility external
    //<
    arrowKeyAction: null,

    //> @attr listGrid.showRecordComponents (boolean : null : IRWA)
    // If set to true, this listGrid should create and show an embedded component in every
    // row of the grid.
    // <P>
    // Developers using this feature should implement the 
    // +link{listGrid.createRecordComponent()} and +link{listGrid.updateRecordComponent()}
    // methods.
    // <P>
    // createRecordComponent() will be called by the grid as rows are rendered to create the
    // record components to use per record or per cell.  Your implementation should return a
    // component to embed in the record passed in. Note that this method should create and
    // return a new component each time it is called.
    // <P>
    // This feature also supports reusing components in different rows in the grid. If 
    // +link{listGrid.recordComponentPoolingMode} is set to <code>"recycle"</code>,
    // components created by the <code>createRecordComponent</code>
    // method will become available for reuse when they are no longer associated with a record.
    // The system will automatically store these in a pool. When a record with no
    // associated component is rendered, if there are any recordComponents in this pool, 
    // the system will call +link{listGrid.updateRecordComponent()}, and pass in the component.
    // This allows the developer to apply record-specific attributes to an already created
    // component and render it out into the new record. This greatly improves performance for 
    // large grids as it allows a small number of components to be created and reused rather 
    // than maintaining potentially one record component for every cell in the grid.
    // <P>
    // Record components are refreshed according to the recordComponentPooling mode. If set to 
    // <code>data</code> components will be maintained as long as their associated record remains
    // present in the data set (but this can require a component to be created for every record so
    // is not desirable for large data sets). Otherwise record components are refreshed as they
    // are scrolled into view. Note that you can explicitly refresh record components via
    // +link{listGrid.invalidateRecordComponents()} and +link{listGrid.refreshRecordComponent()}
    // <P>
    // NOTE: recordComponents can have an impact on row height and therefore may require
    // +link{listGrid.virtualScrolling}.
    // This is not supported in conjunction with +link{ListGridField.frozen,frozen fields}.
    // If you are using recordComponents in a listGrid with frozenFields, you can specify an
    // explicit +link{listGrid.recordComponentHeight} to ensure every row in the grid renders
    // tall enough to accommodate the recordComponents, and as such virtual scrolling is not
    // required.
    //
    // @see recordComponentPosition
    // @see showRecordComponentsByCell
    // @see recordComponentPoolingMode
    // @see showRecordComponent()
    // @see createRecordComponent()
    // @see updateRecordComponent()
    //
    // @visibility external
    //<
    
    //> @type EmbeddedPosition
    // How a component should be embedded within its record or cell
    // @value "expand" component should be placed underneath normal record or cell content,
    //        expanding the records.  Expanding records can result in variable height rows,
    //        in which case +link{listGrid.virtualScrolling,virtualScrolling} should be
    //        enabled.
    // @value "within" component should be placed within the normal area of the record or cell.
    //        Percentage sizes will be treated as percentages of the record and
    //        +link{Canvas.snapTo} positioning settings are also allowed and refer to the
    //        rectangle of the record or cell. Note that for components embedded within cells,
    //        cell align and vAlign will be used if snapTo is unset (so top / left alignment
    //        of cell content will map to snapTo of "TL", etc).
    // @visibility external
    //<

    //> @attr listGrid.recordComponentPosition (EmbeddedPosition : null : IRW)
    // if +link{listGrid.showRecordComponents} is true, how should the component appear within
    // the cell. Valid options are 
    // <ul><li><code>"within"</code>: the component will be rendered inside the record / cell.
    //  +link{canvas.snapTo} may be set to specify where the component should render within
    //  the row or cell, and +link{canvas.snapOffsetTop} / +link{canvas.snapOffsetLeft} may
    //  be set to indent recordComponents within their parent cells.
    //  Note that if unset, the component will show up at the top/left edge
    //  for components embedded within an entire row, or for per-cell components, cell
    //  align and valign will be respected.  Note also that, when rendering components "within"
    //  cells, specified component heights will be respected and will change the height of the 
    //  row.  However, if you want components to completely fill a cell at it's default height, 
    //  set height: "100%" or rows will render at the default height of the component. </li> 
    // <li><code>"expand"</code>: the component will be written into the cell below the
    //  normal cell content, causing the cell to expand vertically to accommodate it.
    // <li><code>null</code>: If this attribute is unset, we will default to showing
    //  recordComponents with position <code>"within"</code> if 
    //  +link{showRecordComponentsByCell} is true, otherwise using <code>"expand"</code>
    //  logic.
    // </ul>
    // @see showRecordComponents
    // @visibility external
    //<
    
//    recordComponentPosition:"expand",
    
    
    //> @attr listGrid.showRecordComponentsByCell (boolean : null : IRWA)
    // If true, shows +link{listGrid.showRecordComponents, recordComponents} in cells, rather 
    // than just in records.
    // @visibility external
    //<

    //> @type RecordComponentPoolingMode
    // The method of component-pooling to employ for +link{showRecordComponents,recordComponents}.
    // <P>
    // @value "viewport" components are destroyed when the record is not being rendered.  Best 
    //        for large datasets where embedded components differ greatly per record.
    // @value "data" components are +link{canvas.clear,clear()ed} when not in the viewport, but
    //        stay with a record until the record is dropped from cache.  Best for guaranteed
    //        small datasets.
    // @value "recycle" components are pooled and will be passed to 
    //       +link{listGrid.updateRecordComponent,updateRecordComponent()} with the
    //       <code>recordChanged</code> parameter set to true.  Best for large datasets where
    //       embedded components are uniform across different records and can be efficiently
    //       reconfigured to work with a new record
    // @visibility external
    //<

    //> @attr listGrid.recordComponentPoolingMode (RecordComponentPoolingMode : "viewport" : IRWA)
    // The method of +link{type:RecordComponentPoolingMode, component-pooling} to employ for 
    // +link{showRecordComponents,recordComponents}.
    // <P>
    // The default mode is "viewport", which means that recordComponents are destroyed as soon 
    // their record leaves the viewport.  
    // <P>
    // For the most efficient implementation, switch to "recycle" mode, which pools components 
    // when records leave the viewport and re-uses them in other records.  In this mode, you 
    // should implement +link{listGrid.updateRecordComponent, updateRecordComponent()} to 
    // apply any changes to make reused components applicable to the new record they appear in, 
    // if necessary.  If you have components of different types in different columns and still
    // want to take advantage of component recycling, you can 
    // set +link{listGrid.poolComponentsPerColumn} to ensure that components intended for one 
    // column are not recycled for use in another column that should have a different component.
    // <P>
    // Note that, if different records have distinctly different components embedded 
    // in them, or multiple columns in each record embed different components, you should 
    // leave the recordComponentPoolingMode at "viewport" if your dataset is very large or 
    // use "data" otherwise.
    // @visibility external
    //<
    recordComponentPoolingMode:"viewport",

    //> @attr listGrid.poolComponentsPerColumn (Boolean : true : IRW)
    // Should recycled +link{listGrid.showRecordComponents,record components}, be pooled
    // per column or per record. Only applies if +link{listGrid.showRecordComponentsByCell} is true.
    // <P>
    // When +link{listGrid.recordComponentPoolingMode} is "recycle" and you have components of 
    // different types in different columns, set this property to true to ensure that 
    // components intended for one column are not recycled for use in another column that 
    // should have a different component.
    // <P>
    // If no components applicable to a particular column are available in the pool, the system
    // calls +link{listGrid.createRecordComponent, createRecordComponent}.
    // 
    // @visibility external
    //<
    
    poolComponentsPerColumn:true,

	// Rollover
	// --------------------------------------------------------------------------------------------
	//>	@attr listGrid.showRollOver (Boolean : true : IRW)
	// Should we show different styling for the cell the mouse is over?
    // <br>
    // If true, the cell style will have the suffix "Over" appended.
	// @group appearance
    // @visibility external
	//<
    
	showRollOver:!isc.Browser.isTouch,
    
    //> @attr listGrid.showRollOverCanvas (boolean : null : IRWA)
    // If +link{listGrid.showRollOver} is true, this property provides an option to show roll over
    // styling with the +link{listGrid.rollOverCanvas} and +link{listGrid.rollUnderCanvas}
    // rather than using css styling.
    // <p>
    // This setting is not compatible with +link{canFreezeFields,frozenFields}, as it would
    // require two separate elements and would limit styling options.
    //
    // @group rowEffects
    // @visibility external
    //<
    //showRollOverCanvas:null,
    
    //> @attr listGrid.rollOverCanvas (AutoChild Canvas : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showRollOver} is true, and
    // +link{listGrid.showRollOverCanvas} is true.  This autoChild canvas will be created and 
    // displayed above the current rollOver row.
    // <P>
    // Note that the rollOverCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the rollOverCanvas<br>
    // - <code>this.record</code> - a pointer to the current roll over record object in the grid.
    // 
    // @group rowEffects
    // @visibility external
    //<

	//> @attr listGrid.rollUnderCanvas (AutoChild Canvas : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showRollOver} is true, and
    // +link{listGrid.showRollOverCanvas} is true.  This autoChild canvas will be created and 
    // displayed behind the current rollOver cell in the page's z-order, meaning it will only be
    // visible if the cell styling is transparent. 
    // <P>
    // Note that the rollUnderCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the rollUnderCanvas<br>
    // - <code>this.record</code> - a pointer to the current roll over record object in the grid.
    // <p>
    // This setting is not compatible with +link{canFreezeFields,frozenFields}, as it would
    // require two separate elements and would limit styling options.
    //
    // @group rowEffects
    // @visibility external
    //<

    //> @attr listgrid.showBackgroundComponent (Boolean : false : IRW)
    // If <code>true</code> this grid will create and show per-row backgroundComponents 
    // as detailed +link{listGrid.backgroundComponent,here}.
    // @visibility external
    //<

	//> @attr listGrid.backgroundComponent (MultiAutoChild Canvas : null : IR)
	// Has no effect unless +link{listGrid.showBackgroundComponent} is <code>true</code>.
	// <P>
    // Canvas created and embedded in the body behind a given record.   When 
    // +link{listGridRecord.backgroundComponent} is set, this autoChild canvas 
    // will be constructed (if listGridRecord.backgroundComponent is not already a Canvas) and 
    // its properties combined with those of listGridRecord.backgroundComponent and then 
    // displayed behind a specific record in the page's z-order, meaning 
    // it will only be visible if the cell styling is transparent.
    // @group rowEffects
    // @visibility external
    //<
    backgroundComponentDefaults: {
        snapTo:"TL",
        autoDraw: false,
        opacity: "50%"
    },

    //>Animation
    
    //> @attr listGrid.animateRollOver (Boolean : false : IRWA)
    // If +link{listGrid.showRollOverCanvas} is <code>true</code> setting this property to true
    // ensures that when the rollOver canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}. Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //> @attr listGrid.animateRollUnder (Boolean : false : IRWA)
    // If +link{listGrid.showRollOverCanvas} is <code>true</code> setting this property to true
    // ensures that when the rollUnder canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}. Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //<Animation
    
    //> @attr listGrid.useCellRollOvers (boolean : null : IRW)
    // Are rollovers cell-level or row-level?
    // @visibility external
    //<
	
	// Hover
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.canHover (boolean : null : [IRW])
	// @include gridRenderer.canHover
    // @group hovers
    // @see attr:listGrid.showHover
    // @see attr:listGridField.showHover
    // @example valueHoverTips
	//<
	// are hover events and hover popups enabled?
	//canHover:false,							

	//>	@attr listGrid.showHover (Boolean : true : [IRW])
	// @include gridRenderer.showHover
    // @group hovers
	//<
	// if canHover:true, should we show hover popups?
	showHover: true,							

    //> @attr listGrid.showClippedHeaderTitlesOnHover (boolean : true : [IRA])
    // If true and a header button's title is clipped, then a hover containing the full field
    // title is enabled.
    // @group hovers
    // @see ListGrid.headerTitleClipped()
    // @see ListGrid.headerHoverHTML()
    // @visibility external
    //<
    showClippedHeaderTitlesOnHover: true,

	//>	@attr listGridField.showHover (boolean : null : IRW)
	// Whether to show hovers for this field.  The default hover will be the contents of the
    // cell the user is hovering over, and can be customized via
    // +link{listGridField.hoverHTML,field.hoverHTML()}.
    // <P>
    // +link{ListGrid.canHover} can be set to true to cause hovers to be shown for all fields
    // by default.  In this case, <code>field.showHover</code> can be set to false to suppress
    // hovers for an individual field.
    // <P>
    // All hovers can be disabled, regardless of other settings, by setting
    // +link{ListGrid.showHover} to false.
    // @visibility external
    // @example valueHoverTips
	//<
    
	// can be set to false to cause hover to be per-row instead of per-cell
	//hoverByCell:true,
	
	// if canHover:true, should an active hover remain active until we leave the listGrid?
	// default behavior is to clear/deactivate the hover on each cellOut/rowOut
	//keepHoverActive:false,	

	// the space between the borders of the cell and the hover, in pixels
    cellHoverOutset:5,
    
    // Note: hoverWidth, hoverStyle, et al will be picked up by the grid renderer when showing 
    // cell hovers (handled by GridRenderer class)

    //> @attr listGrid.hoverStyle (CSSStyleName : "gridHover" : [IRWA])
    // Style to apply to hovers shown over this grid.
    // @see listGrid.showHover
    // @group hovers
    // @visibility external
    //<
    hoverStyle:"gridHover",

	//>	@attr listGridField.prompt (String : null : IR)
    // Causes a tooltip hover to appear on the header generated for this field (effectively
    // sets +link{canvas.prompt} for the header).
    //
    // @visibility external
    //<

	// Selection
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.selection (Selection : null : [RA])
	// The Selection object associated with the listGrid.
	// @group  selection
	// @visibility external
	//<

    //> @attr listGrid.selectionAppearance (SelectionAppearance : "rowStyle" : IRW)
    // How selection of rows should be presented to the user.
    // <P>
    // For <code>selectionAppearance:"checkbox"</code> with multiple selection
    // allowed, you would typically use +link{listGrid.selectionType}:"simple" (the default).  Because 
    // <code>selectionType</code> and <code>selectionAppearance</code> are unrelated, 
    // the combination of <code>selectionAppearance:"checkbox"</code> and
    // <code>selectionType:"multiple"</code> results in a grid where multiple selection can
    // only be achieved via shift-click or ctrl-click.  
    // <P>
    // If using <code>"checkbox"</code> for a +link{listGrid}, see also
    // +link{listGrid.checkboxField} for customization APIs.
    // <P>
    // If using <code>"checkbox"</code> for a +link{treeGrid}, an extra icon,
    // +link{treeGrid.getExtraIcon} is not supported. Additionally only
    // +link{listGrid.selectionType}:"simple" and "single" are supported.
    // You can also toggle the display of a disabled checkbox on a treegrid, displayed 
    // when the node can't be selected, via +link{TreeGrid.showDisabledSelectionCheckbox}.
    // @group selection
    // @visibility external
    //<
    selectionAppearance: "rowStyle",

    //>	@attr listGrid.canSelectAll (boolean : null : [IRW])
    // Controls whether a checkbox for selecting all records appears in the header with 
    // +link{listGrid.selectionAppearance, selectionAppearance} set to "checkbox"
    // 
    // @group selection
    // @visibility external
    //<
    
    //>	@attr listGrid.selectionType (SelectionStyle : null : [IRW])
	// Defines a listGrid's clickable-selection behavior.  
    // <P>
    // The default selection appearance is governed by +link{listGrid.selectionAppearance}: if
    // selectionAppearance is "checkbox", this will be "simple", otherwise, this will be
    // "multiple".
	// 
    // @group	selection, appearance
	// @see type:SelectionStyle
	//      @visibility external
    // @example multipleSelect
	//<
    
	//>	@attr listGrid.selectionProperty (string : null : IRA)
	// If specified, the selection object for this list will use this property to mark records
	// as selected.  In other words, if this attribute were set to <code>"isSelected"</code>
	// any records in the listGrid data where <code>"isSelected"</code> is <code>true</code>
	// will show up as selected in the grid. Similarly if records are selected within the grid
	// after the grid has been created, this property will be set to true on the selected
	// records.
	//
	// @group  selection, appearance
	// @visibility external
	//<    
	//selectionProperty:null,	
	
    //> @attr listGrid.reselectOnUpdate (boolean : true : IRA)
    // If true, when an update operation occurs on a selected record in a 
    // +link{listGrid.dataSource,databound} listGrid, ensure the updated record is 
    // re-selected when the operation completes.
    // The +link{listGrid.reselectOnUpdateNotifications} attributes governs whether
    // +link{listGrid.selectionUpdated()} and +link{listGrid.selectionChanged()} will fire
    // when this occurs.
    // @visibility external
    //<
    // This property is passed to the selection object in DBC.createSelectionModel()
    reselectOnUpdate:true,

    //> @type SelectionNotificationType
    // Enum to indicate selection change notification types. Used by
    // +link{listGrid.reselectOnUpdateNotification}
    // @value "none" No selection change notification should fire
    // @value "selectionChanged" +link{listGrid.selectionChanged()} should fire but
    //  +link{listGrid.selectionUpdated()} should not fire.
    // @value "selectionUpdated" +link{listGrid.selectionChanged()} and 
    //  +link{listGrid.selectionUpdated()} should both fire.
    // @visibility external
    //<

    //> @attr listGrid.reselectOnUpdateNotifications (SelectionNotificationType : "selectionChanged" : IRWA)
    // if +link{listGrid.reselectOnUpdate} is true, this property governs what 
    // selection changed notifications should be triggered when a selected record
    // is edited then automatically reselected when the edited data is merged into
    // the data set.
    // @visibility external
    //<
    reselectOnUpdateNotifications:"selectionChanged",
	
    //> @attr listGrid.recordCanSelectProperty (String : "canSelect" : IRA)
    // If set to false on a record, selection of that record is disallowed.
    // @visibility external
    //< 
	recordCanSelectProperty:"canSelect",

    //> @attr listGridRecord.canSelect (boolean : null : IR)
    //
    // Default property name denoting whether this record can be selected. Property name may be 
    // modified for the grid via +link{listGrid.recordCanSelectProperty}.
    //
    // @visibility external
    //<

    //> @attr listGrid.canSelectCells (Boolean : false : IR)
    // Enables cell-level selection behavior as well as 
    // +link{useCellRollOvers,cell-level rollover}.
    // <P>
    // To query and manipulate cell-level selections, use +link{getCellSelection()} to retrieve 
    // the +link{CellSelection}.  
    // <P>
    // Note that the ListGrid has a data model of one +link{Record} per row, unlike the
    // +link{CubeGrid} which supports one +link{CellRecord} per cell.  For this reason 
    // record-oriented APIs that act on the selection will act on entire Records that have
    // <i>any</i> selected cells (examples include drag and drop and transferSelectedData()).
    // <P>
    // More generally, <code>canSelectCells</code> is primarily intended to enable developers 
    // to build Excel-like interactions on local datasets, by using +link{setData()} plus
    // +link{saveLocally}:true rather than record-oriented DataSources and data binding.
    // <P>
    // The following keyboard selection behaviors are enabled with this property:
    // <P>
    // SHIFT +        [Arrow Key]:  begin or continue incremental selection
    // <P>
    // SHIFT + CTRL + [Arrow Key]:  incremental selection to the end of row or column
    // <P>
    // CTRL  + A: select all cells (enabled only with +link{listGrid.canSelectAll})
    // <P>
    // Incremental selection allows selection of rows and columns of cells via keyboard
    // or mouse provided the shift key is down.  Behavior is designed to match Excel.
    // Thus, if a previous selection has begun, cells will be selected from that origin.
    //
    // @visibility external
    //<
	//canSelectCells:false,

    //> @attr listGrid.useCopyPasteShortcuts (Boolean : true : IRW)
    // Enables copy/paste shortcuts, provided +link{listGrid.canSelectCells} is true
    // <p>
    // The default setting of true means that the value of +link{listGrid.canSelectCells}
    // determines whether copy/paste shortcuts are enabled.  Setting this property to
    // false disables said shortcuts unconditionally.
    // <p>
    // Copying is done via copying to a SmartClient-wide "clipboard" - <b>not</b> the OS-level
    // clipboard, which is innaccessible on some browsers - via the methods
    // +link{getSelectedCellData()} and +link{applySelectedCellData()}.  To copy data to and
    // from applications outside of the browser, use the technique shown in the
    // +explorerExample{gridToExcel,Grid to Excel} and
    // +explorerExample{excelToGrid,Excel to Grid} samples.
    // <p>
    // The following shortcuts are available:
    // <p>
    // CTRL + D: copy cell values from top row of selected cells down to all rows
    // <p>
    // CTRL + R: copy cell values from left column of selected cells right to all columns
    // <p>
    // CTRL + C: copy selected cell values into shared clipboard
    // <p>
    // CTRL + V: paste from shared clipboard into current selection
    //
    // @visibility external
    //<
	useCopyPasteShortcuts: true,

    //> @attr listGrid.originBaseStyle (CSSStyleName : null : IRW)
    // Name of a CSS Style to use as the +link{listGrid.baseStyle} for a cell that 
    // is currently a selection origin for shifted incremental cell selection.
    // Only has an effect if +link{listGrid.canSelectCells} is true.
    // <P>
    // @visibility external
    //<
    //originBaseStyle: null,

    //> @attr listGrid.copyEmptyCells (boolean : true : IRW)
    // Determines whether empty cells (those with an undefined value) are present
    // in the records generated by +link{listGrid.getSelectedCellData}, If true, an
    // empty cell in the selection will be added to the record with a null value.
    // If false, an empty cell will not be represented in the record at all.
    //<
    copyEmptyCells: true,

	//>	@attr listGrid.canDragSelect (Boolean : false : IRW)
	//  If this property is true, users can drag the mouse to select several rows or cells.
	//  This is mutually exclusive with rearranging rows or cells by dragging.
	//  @group	selection
	//  @visibility	external
    //  @example dragListSelect
	//<
	//canDragSelect:false,

	//>	@attr listGrid.canDragSelectText (Boolean : false : IRW)
	// If this property is true, users can drag the mouse to select text within grid rows.
	// This is mutually exclusive with 
    // +link{canReorderRecords,rearranging rows or cells by dragging}, and with 
    // +link{canDragSelect,drag selection of rows}.
    //
    // @group selection
	// @visibility external
	//<
	//canDragSelectText:false,
    
    //>	@attr listGrid.canDropInEmptyArea (Boolean : true : IRW)
	// If set to false, dropping over an empty part of the grid body is disallowed and the 
    // no-drop indicator is displayed.
    // @group dragdrop
    // @visibility external
    //<
	//canDropInEmptyArea: true,			

    //> @attr listGrid.showSelectionCanvas (boolean : null : IRWA)
    // If +link{listGrid.selectionType} is set to <code>"single"</code>, setting this property to
    // true means selection will be displayed to the user with the +link{listGrid.selectionCanvas} 
    // and +link{listGrid.selectionUnderCanvas} rather than using css styling.
    // <p>
    // This setting is not compatible with +link{canFreezeFields,frozenFields}, as it would
    // require two separate elements and would limit styling options.
    //
    // @group rowEffects
    // @visibility external
    //<
    //showSelectionCanvas:null,
    
    //> @attr listGrid.selectionCanvas (AutoChild Canvas : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showSelectionCanvas} is true,
    // and +link{listGrid.selectionType} is <code>"single"</code>.
    // This autoChild canvas will be created and displayed above the selected record on every 
    // selection change.
    // <P>
    // Note that the selectionCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the selectionCanvas<br>
    // - <code>this.record</code> - a pointer to the currently selected record object in the grid.
    // @group rowEffects
    // @visibility external
    //<
    
    //> @attr listGrid.selectionUnderCanvas (AutoChild Canvas : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showSelectionCanvas} is true,
    // and +link{listGrid.selectionType} is <code>"single"</code>.
    // displayed behind the current selected cell in the page's z-order, meaning it will only be
    // visible if the cell css styling is transparent. 
    // <P>
    // Note that the selectionUnderCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the selectionUnderCanvas<br>
    // - <code>this.record</code> - a pointer to the current selected record object in the grid.
    // @group rowEffects
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxField (AutoChild ListGridField : null : IR)
    // Returns the specially generated checkbox field used when +link{selectionAppearance} is
    // "checkbox".  Created via the +link{AutoChild} pattern so that
    // <code>checkboxFieldDefaults</code> and <code>checkboxFieldProperties</code> are available
    // for skinning purposes. Note that +link{listGridField.shouldPrint} is <code>false</code>
    // for the checkboxField by default - if you want this column to show up in the grid's print
    // view, use <code>checkboxFieldProperties</code> to set this property to true.
    // <P>
    // This field will render an icon to indicate the selected state of each row, which, when
    // clicked will toggle the selection state. The icon src may be configured using 
    // +link{listGrid.checkboxFieldTrueImage} and +link{listGrid.checkboxFieldFalseImage}, as
    // well as +link{listGrid.checkboxFieldImageWidth} and +link{listGrid.checkboxFieldImageHeight}.
    // <P>
    // The checkboxField can be detected by calling +link{listGrid.isCheckboxField()} on any
    // ListGridField object.
    //
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldTrueImage (SCImgURL : null :IRWA)
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // determines the image to display in the checkbox field for a selected row.
    // If unset, the +link{listGrid.booleanTrueImage} will be used. Note that the special
    // value "blank" means that no image will be shown.
    // @see listGrid.checkboxFieldFalseImage
    // @see listGrid.checkboxFieldImageWidth
    // @see listGrid.checkboxFieldImageHeight
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldFalseImage (SCImgURL : null :IRWA)
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // determines the image to display in the checkbox field for an unselected row.
    // If unset, the +link{listGrid.booleanFalseImage} will be used. Note that the special
    // value "blank" means that no image will be shown.
    // @see listGrid.checkboxFieldTrueImage
    // @see listGrid.checkboxFieldImageWidth
    // @see listGrid.checkboxFieldImageHeight
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldPartialImage (SCImgURL : null :IRWA)
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // determines the image to display in the checkbox field for a partially selected row.
    // If unset, the +link{listGrid.booleanPartialImage} will be used. Note that the special
    // value "blank" means that no image will be shown.
    // @see listGrid.checkboxFieldTrueImage
    // @see listGrid.checkboxFieldImageWidth
    // @see listGrid.checkboxFieldImageHeight
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldImageWidth (integer : null : IR) 
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // may be set to govern the width of the checkbox image displayed to indicate whether a row is
    // selected. If unset, the checkboxField image will be sized to match the
    // +link{listGrid.booleanImageWidth} for this grid.
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldImageHeight (integer : null : IR) 
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // may be set to govern the height of the checkbox image displayed to indicate whether a row is
    // selected. If unset, the checkboxField image will be sized to match the
    // +link{listGrid.booleanImageHeight} for this grid.
    // @group checkboxField
    // @visibility external
    //<
          
    //>Animation
    
    //> @attr listGrid.animateSelection (Boolean : false : IRWA)
    // If +link{listGrid.showSelectionCanvas} is <code>true</code> setting this property to true
    // ensures that when the selection canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}.  Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //> @attr listGrid.animateSelectionUnder (Boolean : false : IRWA)
    // If +link{listGrid.showSelectionCanvas} is <code>true</code> setting this property to true
    // ensures that when the selectionUnder canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}. Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //<Animation

	// Empty and loading messages
	// --------------------------------------------------------------------------------------------

    //>	@attr listGrid.showEmptyMessage (Boolean : true : [IRW])
    // @include gridRenderer.showEmptyMessage
    // @example emptyGrid
    //<
	showEmptyMessage:true,
    
	//>	@attr listGrid.emptyMessage (HTMLString : "No items to show." : [IRW])
    // @include gridRenderer.emptyMessage
    // @example emptyGrid
	//<
	emptyMessage:"No items to show.",

	//>	@attr listGrid.emptyMessageStyle (CSSStyleName : "emptyMessage" : [IRW])
    // The CSS style name applied to the +link{emptyMessage} if displayed.
    // @group emptyMessage
    // @visibility external
	//<
	emptyMessageStyle:"emptyMessage",

    //> @attr listGrid.filterButtonPrompt (String : "Filter" : [IR])
    // The prompt to show when the mouse hovers over the Filter button in the FilterEditor.
    // @group i18nMessages
    // @visibility external
    //<
    filterButtonPrompt : "Filter",


	//>	@attr listGrid.loadingDataMessage (HTMLString : "\${loadingImage}&amp;nbsp;Loading data..." : [IRW])
    // The string to display in the body of a listGrid while data is being loaded.
    // Use <code>"\${loadingImage}"</code> to include +link{Canvas.loadingImageSrc,a loading image}.
    // @see loadingDataMessageStyle
    // @group emptyMessage, i18nMessages
    // @visibility external
	//<
    loadingDataMessage : "${loadingImage}&nbsp;Loading data...",

	//>	@attr listGrid.loadingDataMessageStyle (CSSStyleName : "loadingDataMessage" : [IRW])
    // The CSS style name applied to the loadingDataMessage string if displayed.
    // @group emptyMessage
    // @visibility external    
	//<
    loadingDataMessageStyle: "loadingDataMessage",

	//>	@attr listGrid.loadingMessage (string : "&amp;nbsp;" : IR)
    // If you have a databound listGrid and you scroll out of the currently loaded dataset, by
    // default you will see blank rows until the server returns the data for those rows.  The
    // loadingMessage attribute allows you to specify arbitrary html that will be shown in each
    // such "blank" record while the data for that record is loading.
    //
	// @group emptyMessage, i18nMessages
    // @visibility external
	//<
	loadingMessage:"&nbsp;",	

    // Separator / Single Cell rows
    // ---------------------------------------------------------------------------------------

    //> @attr listGrid.singleCellValueProperty (string : "singleCellValue" : IRW)
    // If <code>record[this.singleCellValueProperty]</code> is set for some record, the 
    // record will be displayed as a single cell spanning every column in the grid, with 
    // contents set to the value of <code>record[this.singleCellValueProperty]</code>.
    // @visibility external
    //<
    singleCellValueProperty:"singleCellValue",

    //> @attr listGrid.isSeparatorProperty (string : "isSeparator" : IRW)
    // If <code>record[this.isSeparatorProperty]</code> is set for some record, the 
    // record will be displayed as a simple separator row.
    // @visibility external
    //<
    isSeparatorProperty:"isSeparator",
 
    // Filter Editor (aka QBE)   
    // ---------------------------------------------------------------------------------------

	//> @attr listGrid.showFilterEditor (boolean : null : IRW)
	// Should this listGrid display a filter row.  If true, this ListGrid
	// will be drawn with a single editable row, (separate from the body) with a filter button.
    // <P>
    // Values entered into this row are used as filter criteria to filter this List's data on
    // enter-keypress or filter button click. +link{listGrid.autoFetchTextMatchStyle} determines
    // the textMatchStyle for the request passed to +link{listGrid.fetchData()}.
    // <P>
    // Note that if +link{listGrid.filterData()} or +link{listGrid.fetchData()} is called directly
    // while the filter editor is showing, the filter editor values will be updated to reflect the
    // new set of criteria. If you wish to retain the user entered filter criteria and 
    // programmatically modify a subset of field values programmatically this can be achieved by
    // deriving new criteria by copying the existing set of criteria and adding other changes - 
    // something like this:
    // <pre><code>
    //   var newCriteria = myListGrid.getFilterEditorCriteria();
    //   isc.addProperties(newCriteria, {
    //      field1:"new value1",
    //      field2:"new value2"
    //   });
    //   myListGrid.setCriteria(newCriteria);
    // </code></pre>
    // In this example code we're using +link{listGrid.getFilterEditorCriteria()} rather than 
    // +link{listGrid.getCriteria()} - this ensures that if the user has typed a new value into
    // the filter editor, but not yet clicked the filter button, we pick up the value the user
    // entered.
    // <P>
    // If you call <code>filterData()</code> and pass in criteria for dataSource 
    // fields that are not present in the ListGrid, these criteria will continue to be applied along
    // with the user visible criteria.
    // <P>
    // <b>filterEditor and advanced criteria</b>: If a developer calls <code>filterData()</code>
    // on a ListGrid and passes in +link{AdvancedCriteria}, expected behavior of the filter 
    // editor becomes ambiguous, as  AdvancedCriteria supports far more complex filter 
    // expressions than the ordinary filterEditor is capable of expressing.
    // <br>
    // The above example code assumes simple criteria, but if we wanted to apply advanced 
    // criteria to the grid we could call +link{dataSource.combineCriteria()} rather than doing
    // a simple addProperties() on the criteria object.
    // <P>
    // Default behavior for AdvancedCriteria will combine the AdvancedCriteria with the values 
    // in the filter editor as follows:
    // <ul>
    // <li>If the top level criteria has operator of type "and":<br>
    //  Each field in the top level
    //  criteria array for which a 'canFilter' true field is shown in the listGrid will show up
    //  if the specified operator matches the default filter behavior 
    //  (based on the +link{listGrid.autoFetchTextMatchStyle}).<br>
    //  If the user enters values in the filter editor, these will be combined with the
    //  existing AdvancedCriteria by either replacing or adding field level criteria at the top 
    //  level.</li>
    // <li>If the top level criteria is a single field-criteria:<br>
    //  If the field shows up in the listGrid and is canFilter:true, it will be displayed to
    //  the user (if the operator matches the default filter behavior for the field).<br>
    //  If the user enters new filter criteria in the filterEditor, they will be combined with
    //  this existing criterion via a top level "and" operator, or if the user modifies the
    //  field for which the criterion already existed, it will be replaced.</li>
    // <li>Otherwise, if there are multiple top level criteria combined with an "or" operator,
    //  these will not be shown in the filter editor. Any filter parameters the user enters will
    //  be added to the existing criteria via an additional top level "and" operator, meaning
    //  the user will essentially filter a subset of the existing criteria</li>
    // </ul>
    //  @group filterEditor
	//  @visibility external
    //  @example filter
	//<
	
	//showFilterEditor:null
    
    //> @attr listGrid.filterEditor (ListGrid AutoChild : null : R)
    // If +link{listGrid.showFilterEditor} is set to true, the <code>filterEditor</code>
    // is automatically created as an AutoChild. It's implemented as a specialized 
    // ListGrid in edit mode.
    // @visibility external
    //<
    
    //> @attr listGrid.filterEditorProperties (ListGrid properties : null : IR)
    // Properties to apply to the automatically generated +link{listGrid.filterEditor}
    // if +link{listGrid.showFilterEditor} is true.
    // @visibility external
    //<
    
    
    
    filterEditorDefaults : { shouldPrint:false }, 

    
    //> @attr listGrid.filterButtonProperties (Button properties : null : IR)
    // If +link{listGrid.showFilterEditor} is true, this attribute may be used to customize the
    // filter button shown to the right of the filterEditor row.
    // @visibility external
    //<

	//> @attr listGrid.filterEditorHeight (number : 22 : IRW)
    // Height for the filterEditor, if shown.
	//    
	//  @group filterEditor
	//  @visibility external
	//<
    filterEditorHeight:22,
    
    //> @attr listGrid.autoFetchAsFilter (boolean : null : IR)
    // Determines whether +link{filterData()} or +link{fetchData()} should be called when this 
    // grid is initially filtered via +link{autoFetchData}, or filtered by the user via the
    // +link{showFilterEditor,filterEditor}. 
    // @group dataBinding
    // @deprecated in favor of listGrid.autoFetchTextMatchStyle
    //<
    // Note: This was exposed in the 7.0 release candidate builds only.
    // Leaving supported (deprecated) but not publically documented
    // If set, at init time, it will set textMatchStyle to exact or substring (handled in Canvas.js)
    
    //> @attr listGrid.autoFetchTextMatchStyle (TextMatchStyle : "substring" : IR)
    // When this grid is initially filtered via +link{autoFetchData}, or filtered by the user 
    // via the +link{showFilterEditor,filterEditor}, this attribute can be used to set the
    // <code>textMatchStyle</code> on the dsRequest passed to <code>fetchData()</code>.
    // @group dataBinding
    // @visibility external
    //<
    // implemented in canvas.getInitialFetchContext() and recordEditor.performAction()
    autoFetchTextMatchStyle:"substring",
    
	// Editing
	// --------------------------------------------------------------------------------------------
	//> @attr listGrid.canEdit (Boolean : null : [IRW])
	//      Can the user edit cells in this listGrid? Can be set for the listGrid, and overridden for 
	//      individual fields.<br>
	//      If 'canEdit' is false at the listGrid level, fields can never be edited - in this case
	//      the canEdit property on individual fields will be ignored.<br>
	//      If 'canEdit' is set to true at the listGrid level, setting the 'canEdit' property to
	//      false at the field level will prevent the field from being edited inline.<br>
	//      If 'canEdit' is not set at the listGrid level, setting 'canEdit' to true at the field 
	//      level enables the field to be edited inline.
	//      @visibility external
	//      @group  editing
	//      @see    startEditing()
    //      @see listGridField.canEdit
    //      @see listGrid.recordEditProperty
    //      @see listGrid.canEditCell()
	//      @see    fields
    //      @example editByRow
	//<
	//canEdit:null,

    //> @attr listGrid.canEditNew (Boolean : null : [IRW])
	//  Can the user add new rows?
	//  @group  editing
	//  @see attr:listGrid.canEdit
    //  @see attr:listGrid.recordEditProperty
	//  @visibility internal
	//<
    canEditNew: false,

	//> @attr listGrid.recordEditProperty (string : "_canEdit" : [IRWA])
	// Property name on a record that should be checked to determine whether the record may be
	// edited.
	// <br>
	// This property is configurable to avoid possible collision with data values in record.
	// With the default setting of "_canEdit", a record can be set non-editable by ensuring
	// record._canEdit == false.
	// <br>
	// For controlling editability for the entire grid or for a field, set grid.canEdit or
	// field.canEdit.
	//
	//  @group  editing
	//  @see attr:listGrid.canEdit
	//  @see attr:listGridField.canEdit
    //  @see method:listGrid.canEditCell
	//  @visibility external
	//<
    recordEditProperty:"_canEdit",
    
    //> @attr listGridRecord._canEdit (boolean : null : IR)
    //
    // Default property name denoting whether this record can be edited. Property name may be 
    // modified for the grid via +link{listGrid.recordEditProperty}.
    //
    // @group  editing
    // @visibility external
    //<

	// Name for property used by internal '_testRowEditData' method to track whether
	// records have been compared to edit-data in order to map rowNums to edit values.
	// Customizable, in case of collision with record data - but unlikely to be overridden.
    editValuesTestedProperty:"_editValuesTested",

    //> @attr listGrid.alwaysShowEditors (boolean : null : [IR])
    // When this attribute is set, editors will be rendered into every row of the grid rather than
    // showing up in a single record at a time.
    // This attribute is only valid when +link{listGrid.editByCell} is false
    // @group editing
    // @visibility external
    //<

	//>	@attr listGrid.editByCell (boolean : null : [IRW])
	// Determines whether when the user edits a cell in this listGrid the entire row becomes
	// editable, or just the cell that received the edit event.
    // <P>
	// No effect if this.canEdit is false or null.
    //
	// @group editing
	// @see listGrid.canEdit
    // @example editByCell
	// @visibility external
	//<

	//>	@attr listGrid.saveByCell (boolean : null : [IRW])
	// Whether edits should be saved whenever the user moves between cells in the current edit
	// row.
	// <P>
	// If unset, defaults to +link{editByCell,this.editByCell}.
	// <P>
	// To avoid automatic saving entirely, set +link{autoSaveEdits}:false.
	//
	//  @group  editing
	//  @visibility external
	//  @see listGrid.editByCell
	//<
    
	//>	@attr listGrid.validateByCell (boolean : null : [IRW])
	// Whether client-side validation checks should be performed when the user moves between
	// cells in the current edit row.  If unset, defaults to +link{listGrid.editByCell}.
    // <P>
    // Note that validation always occurs when a row is to be saved, so setting
    // +link{saveByCell}:true forces validation on cell transitions.  To completely disable
    // automatic validation, set +link{neverValidate}:true.
    //
    // @group gridValidation
    // @visibility external
    // @see group:editing
    //<
    

    // autoValidate will disable validation on row-transitions, so validation will only
    // occur on save attempts.  Not currently exposed.
    autoValidate:true,
    
    //> @attr listGrid.validateOnChange (boolean : null : [IRW])
    // If true, validation will be performed on each edited cell when each editor's 
    // "change" handler is fired.
    //
    // @see ListGridField.validateOnChange
    // @group gridValidation
    // @visibility external
    //<
    //validateOnChange:null
    
    //> @attr listGrid.neverValidate (boolean : null : [IRWA])
	// If true, validation will not occur as a result of cell editing for this grid.
	//	@group	gridValidation
    // @visibility external
	//<
    //neverValidate:null,
    
    
    //> @attr listGrid.canRemoveRecords (Boolean : false : IR)
    // If set, provide UI for the user to remove records from the grid as an additional field
    // showing the +link{listGrid.removeIcon}, which, when clicked, will call
    // +link{listGrid.removeRecordClick()} which removes the row from the data set (or if
    // +link{deferRemoval} is true changes the +link{listGrid.markRecordRemoved()} status
    // for the record). Individual records can be marked to prevent removal - see 
    // +link{listGrid.recordCanRemoveProperty}.
    // <P>
    // To add a confirmation dialog before a record is removed, set
    // +link{listGrid.warnOnRemoval}.
    // <P>
    // If deferring removal, the record will appear marked with the +link{removedCSSText} 
    // until the removal is committed via a call to +link{saveEdits()}. 
    // Otherwise, the record will disappear from view. If +link{listGrid.animateRemoveRecord}
    // is true, the removed record will appear to shrink out of view when it is removed.
    // <P>
    // By default the field will display the +link{listGrid.removeIcon} next to each record, and
    // will be rendered as the rightmost column. Two mechanisms exist to further modify this field:
    // <ul>
    // <li>To change the position of the remove-field, include an explicitly specified field with
    //     the attribute +link{listGridField.isRemoveField,isRemoveField:true} set. This will then
    //     be used as the remove field instead of adding a field to the beginning of the set of
    //     columns.</li>
    // <li>Additional direct configuration of the remove field may be achieved by modifying
    //     +link{listGrid.removeFieldProperties}.</li>
    // </ul>
    // If +link{deferRemoval} is true, when a record is marked as removed, the the icon will 
    // change to display the +link{listGrid.unremoveIcon} for this row. Clicking on this icon
    // will call +link{listGrid.unmarkRemovedRecord()} to mark the record as no longer pending
    // deletion.
    // @group databinding
    // @visibility external
    //<
    
    //> @attr listGrid.warnOnRemoval (Boolean : false : IRW)
    // If +link{listGrid.canRemoveRecords} is true, when the user clicks the remove icon
    // for some record, should we show a warning message 
    // (defined as +link{listGrid.warnOnRemovalMessage}) and allow the user to cancel removal?
    // @visibility external
    //<
    warnOnRemoval:false,
    //> @attr listGrid.warnOnRemovalMessage (string : "Are you sure you want to delete this record?" : IRW)
    // Warning message to show the user on a click on the 'remove' icon 
    // if +link{listGrid.canRemoveRecords} is true and
    // +link{listGrid.warnOnRemoval} is true.
    // @visibility external
    // @group i18nMessages
    //<
    warnOnRemovalMessage:"Are you sure you want to delete this record?",

    shouldShowRemoveField : function () {
        if (this.fieldSourceGrid != null) return this.fieldSourceGrid.shouldShowRemoveField();
        return this.canRemoveRecords;
    },
    
    //> @attr listGrid.recordRemovedProperty (string : "_removed" : IR)
    // Property name used as an edit-value on a record that has been marked for removal via
    // +link{listGrid.markRecordRemoved()}. 
    //
    // @see listGridRecord.removed
    // @visibility internal
    //<
    // This property is internal - the "_removed" flag is obfuscated and unlikely to collide
    // with real world data but could be configued to be something different if this became an issue.
    recordRemovedProperty: "_removed",
    
    //> @method listGrid.markRecordRemoved()
    // Marks a record deleted such that a later call to +link{saveEdits()} or +link{saveAllEdits()}
    // will cause a "remove" +link{DSRequest} to be submitted.
    // <P>
    // A removed record is disabled and non-editable, and uses +link{removedCSSText} for its CSS
    // style, which by default will show strikethrough text.
    // <P>
    // Contrast this method with removeSelectedData(), which immediately submits a DSRequest to
    // remove the selected records from the dataset.
    // <P>
    // Records that have been marked for removal using this method may be 'unmarked' via a call to
    // +link{listGrid.unmarkRecordRemoved()}, or by discarding edit values (+link{discardEdits()}).
    //
    // @param rowNum (number) row number for the record to mark
    //
    // @group editing
    // @visibility external
    //<
    markRecordRemoved : function (rowNum, suppressRefresh) {
        if (!isc.isA.Number(rowNum)) rowNum = this.findRowNum(rowNum);
        // hide the edit form if we're mid-edit.
        if (this.getEditRow() == rowNum) this.hideInlineEditor();
        // deselect if we're selected.
        var record= this.getRecord(rowNum);
        this.selection.deselect(record);
        
        this.setEditValue(rowNum, this.recordRemovedProperty, true, false, false);
        if (!suppressRefresh) this.refreshRow(rowNum);
    },
    
    //> @method listGrid.recordMarkedAsRemoved()
    // Returns true if the specified record is marked as removed via a call to 
    // +link{markRecordRemoved()}
    // @param rowNum (int) index of row to verify
    // @return (Boolean) true if the specified record has been marked for removal
    // @visibility external
    // @group editing
    //<
    recordMarkedAsRemoved : function (rowNum) {
        if (rowNum == null) return false;
        if (!isc.isA.Number(rowNum)) {
            
            rowNum = this.getEditSessionRowNum(rowNum);
            if (rowNum == null) return false;
        }
        return (this.getEditValue(rowNum, this.recordRemovedProperty) == true);
    },
    
    //> @method listGrid.unmarkRecordRemoved()
    // Reverses a previous call to +link{markRecordRemoved()}.
    // <P>
    // Note that a record that is marked for removal and then un-marked retains any uncommited
    // edits from before it was marked for removal.  These can be discarded with
    // +link{discardEdits()}.
    // @param rowNum (int) index of record to clear the 'removed' 
    // @group editing
    // @visibility external
    //<
    unmarkRecordRemoved : function (rowNum, suppressRefresh) {
        if (!isc.isA.Number(rowNum)) rowNum = this.findRowNum(rowNum);
        this.clearEditValue(rowNum, this.recordRemovedProperty);
        if (!suppressRefresh) this.refreshRow(rowNum);
    },
    
    
    
	//>	@attr listGrid.removedCSSText (string : "text-decoration:line-through;" : [IRWA])
	//  Custom CSS text to be applied to records that have been 
	// +link{listGrid.markRecordRemoved(),marked for removal}.
	// <P>
	// This CSS text will be applied on top of standard disabled styling for the cell.
    //
	// @visibility external
	// @group	appearance
	//<
    removedCSSText:"text-decoration:line-through;",

    //> @method listGrid.markSelectionRemoved()
    // Marks the currently selected records as removed, as though +link{markRecordRemoved()} had
    // been called. 
    //
    // @group editing
    // @visibility external
    //<
    markSelectionRemoved : function () {
        var records = this.getSelectedRecords();
        if (records != null) {
            for (var i = 0; i < records.length; i++) {
                // We can pass the record object into markRecordRemoved even though it's doc'd
                // as taking just rowNum
                this.markRecordRemoved(records[i], true);
            }
        }
        this.markForRedraw();
            
    },
    
    //> @attr listGrid.deferRemoval (boolean : null : IR)
    // When enabled, the field shown by +link{listGrid.canRemoveRecords} causes records to be
    // marked for future removal via +link{markRecordRemoved()} instead of immediately being 
    // removed.
    // <P>
    // When a record has been marked for removal, an icon in the
    // <code>canRemoveRecords</code> field allowing it to be unmarked will be displayed.
    // <P>
    // If not explicitly specified by this property, removal of records will be deferred if
    // +link{autoSaveEdits} is false for the grid.
    // @group editing
    // @visibility external
    //<

    shouldDeferRemoval : function () {
        if (this.deferRemoval != null) return this.deferRemoval;
        return !this.autoSaveEdits;
    },
    
    //> @attr listGrid.removeIcon (SCImgURL : "[SKIN]/actions/remove.png" : IR)
    // When +link{ListGrid.canRemoveRecords} is enabled, default icon to show in
    // the auto-generated field that allows removing records.
    // @visibility external
    //<
    removeIcon:"[SKIN]/actions/remove.png",
    
    //> @attr listGrid.unremoveIcon (SCImgURL : "[SKIN]/actions/undo.png" : IR)
    // When +link{ListGrid.canRemoveRecords} is enabled, this icon will be shown in the
    // auto generated field fro removing records if the record has been marked as removed via
    // +link{listGrid.markRecordRemoved()}. At this point, clicking on the icon will
    // unmark the record as removed.
    // @visibility external
    //<
    unremoveIcon:"[SKIN]/actions/undo.png",

    //> @attr listGrid.removeIconSize (Number : 16 : IRW)
    // Default width and height of +link{removeIcon,remove icons} for this ListGrid.
    //
    // @visibility external
    //<
    removeIconSize: 16,
    
    //> @attr listGrid.animateRemoveRecord (Boolean : true : IRW)
    // When +link{ListGrid.canRemoveRecords} is enabled, should records be animated out of view
    // when they are removed by the user? 
    // @visibility external
    //<
    // When showing alternate records styles, the styles will essentially be reassigned after the
    // animation completes which means we finish our smooth animation with what looks a little like
    // a jump - not clear how to avoid this, but we could warn about this in the attribute
    // description.
    animateRemoveRecord:true,
    
    //> @attr listGrid.animateRemoveTime (number : 100 : IRW)
    // When animating record removal 
    // +link{listGrid.animateRemoveRecord,(see animateRemoveRecord)}, if 
    // +link{listGrid.animateRemoveSpeed} is not
    // set, this property designates the duration of the animation in ms.
    // @group animation
    // @visibility animation
    // @see listGrid.animateRemoveRecord
    //<
    animateRemoveTime:100,

    //> @attr listGrid.animateRemoveSpeed (number : 200 : IRW)
    // When +link{listGrid.animateRemoveRecord, animating record removal}, this property 
    // designates the speed of the animation in pixels per second. Takes precedence over the
    // +link{listGrid.animateRemoveTime} property, which allows the developer to specify a
    // duration for the animation rather than a speed.
    // @group animation
    // @visibility animation
    // @see listGrid.animateRemoveRecord
    //<    
    animateRemoveSpeed:200,

    //> @attr listGrid.removeFieldTitle (String : "&nbsp;" : IRWA)
    // The title to use for the +link{listGrid.removeFieldDefaults, remove field}.
    // <P>
    // By default this title is not displayed in the remove column header button as the
    // +link{listGrid.removeFieldDefaults} sets +link{listGridField.showTitle} to <code>false</code>.
    // @visibility external
    //<
    removeFieldTitle: isc.nbsp,

    //> @attr listGrid.removeFieldDefaults (ListGridField properties : {...} : IR)
    // Default configuration properties for the "remove field" 
    // displayed when +link{ListGrid.canRemoveRecords} is enabled.
    // +link{classMethod:class.changeDefaults()} should be used when modifying this object. 
    // <P>
    // The default configuration includes a +link{listGridField.recordClick()} handler which
    // calls +link{listGrid.removeData()} to actually perform the data removal.
    // @visibility external
    //<
    removeFieldDefaults:{
        type:"icon",
        width:21,
        showDefaultContextMenu:false,
        canEdit:false,
        canHide:false,
        canSort:false,
        canGroupBy:false,
        canFilter:false,
        showTitle:false,
        canExport: false,
        ignoreKeyboardClicks:true,
        showGridSummary: false,
        showGroupSummary: false,
        summaryValue: "&nbsp;"
        // No need for a recordClick handler.
        // We explicitly override '_rowClick()' on the gridBody class to fire the removeRecordClick 
        // event. This handles the case where the icon is marked for removal already, therefore
        // disabled, so we wouldn't get a field.recordClick notification.
        
        // In setFields() we override 'formatCellValue' to return the appropriate icon
        // (removeIcon / unremoveIcon).

    },
    
    //> @attr listGrid.removeFieldProperties (ListGridField properties : null : IR)
    // Configuration properties for the "remove field" displayed when
    // +link{ListGrid.canRemoveRecords} is enabled.
    // <var class="SmartClient">These configuration settings will be overlaid
    // on top of the +link{listGrid.removeFieldDefaults}.</var>
    // @visibility external
    //<

	//>	@attr listGrid.filterByCell (boolean : null : [IRWA])
	// If we're showing the filterEditor (this.showFilterEditor is true), this property 
	// determines whether this list should be filtered every time the user puts focus in
	// a different field in the filter editor.
	// @group filterEditor
	// @visibility external
	//<
	
	//>	@attr listGrid.filterOnKeypress (boolean : null : [IRWA])
	// If we're showing the filterEditor (this.showFilterEditor is true), this property 
	// determines whether this list should be filtered every time the user modifies the value
    // in a field of the filter-editor. Can also be set at the field level.
	// @group filterEditor
	// @visibility external
	//<
    
	//> @attr listGrid.waitForSave (Boolean : false : [IRWA])
	// If this is an editable listGrid, this property determines whether the user will be
	// able to dismiss the edit form, or navigate to another cell while the save is in 
	// process (before the asynchronous server response returns).
	//	@group	editing
	// @visibility external
	//<
	//waitForSave:false,

	//> @attr listGrid.stopOnErrors (Boolean : false : [IRWA])
	// If this is an editable listGrid, this property determines how failure to save due to 
	// validation errors should be displayed to the user.
	// <P>
	// If this property is true, when validation errors occur the errors will be displayed
	// to the user in an alert, and focus will be returned to the first cell to fail validation.
	// <P>
	// If false, the cells that failed validation will be silently styled with the 
	// editFailedBaseStyle.
    // <p>
	// <b>Note:</b> stopOnErrors being set to true implies that 'waitForSave' is also true.
	// We will not dismiss the editor until save has completed if stopOnErrors is true.
	//
	// @group editing
	// @see waitForSave
	// @visibility external
	//<
	//stopOnErrors:false,

    //> @attr listGrid.autoSaveEdits (Boolean : true : [IRWA])
    // If this ListGrid is editable, should edits be saved out when the user finishes editing
    // a row (or a cell if +link{ListGrid.saveByCell} is true).
    // <P>
    // The default of <code>true</code> indicates that edits will be
    // +link{saveByCell,automatically saved} as the
    // user navigates through the grid and/or +link{enterKeyEditAction,hits 'Enter'} to end
    // editing.  See the +link{group:editing,Grid Editing} overview for details. 
    // <P>
    // Setting <code>autoSaveEdits</code> false creates a "mass update" / "mass delete"
    // interaction where edits will be retained for all edited cells (across rows if
    // appropriate) until +link{saveEdits()} is called to save a particular row, or
    // +link{saveAllEdits()} is called to save all changes in a batch.
    //
    // @group editing
    // @visibility external
    //<
    autoSaveEdits:true,
    
    // ListGrid validation error icon. Very similar API to the FormItem class validation error 
    // icon.
    
    //> @attr ListGrid.showErrorIcons (boolean : true : IRW)
    //  If this grid is editable, and an edit has caused validation failure for some cell,
    //  should we show an icon to indicate validation failure?
    //  @group  errorIcon
    //  @visibility internal
    //<
    showErrorIcons : true,
    
    //> @attr ListGrid.errorIconHeight (number : 16 : IRW)
    //      Height of the error icon, if we're showing icons when validation errors occur.
    //  @group  errorIcon
    //  @visibility internal
    //<
    errorIconHeight : 16,
    
    //> @attr ListGrid.errorIconWidth (number : 16 : IRW)
    //      Height of the error icon, if we're showing icons when validation errors occur.
    //  @group  errorIcon
    //  @visibility internal
    //<    
    errorIconWidth : 16,
    
    //> @attr ListGrid.errorIconSrc (SCImgURL : "[SKIN]/ListGrid/validation_error_icon.png" : IRW)
    //      Src of the image to show as an error icon, if we're showing icons when validation
    //      errors occur.
    //  @group  errorIcon
    //  @visibility internal
    //<    
    errorIconSrc : "[SKIN]/validation_error_icon.png",    
        
	//> @attr listGrid.confirmCancelEditing (Boolean : false : [IRW])
	// If this is an editable listGrid, when the user attempts to cancel an edit, should we
    // display a confirmation prompt before discarding the edited values for the record?
	//
	//  @visibility external
	//	@group	editing
	//<
	//confirmCancelEditing:false,

	//> @attr listGrid.cancelEditingConfirmationMessage (string : Cancelling this edit will discard unsaved changes for this record. Continue? : [IRW])
	// If this is an editable listGrid, and <code>this.confirmCancelEditing</code> is true
    // this property is used as the message to display in the confirmation dismissal prompt.
    //
	//  @visibility external
	//	@group	editing, i18nMessages
	//<
	cancelEditingConfirmationMessage:"Cancelling this edit will discard unsaved changes for this record. Continue?",

    //> @attr listGrid.confirmDiscardEdits (Boolean : true : [IRW])
	// For editable listGrids, outstanding unsaved edits when the user performs a new filter
    // or sort will be discarded. This flag determines whether we should display a confirmation
    // dialog with options to save or discard the edits, or cancel the action in this case.
	//
	//  @visibility external
	//	@group	editing
	//<
	confirmDiscardEdits:true,
    
	//> @attr listGrid.confirmDiscardEditsMessage (string : "This action will discard unsaved changes for this list.": [IRW])
	// If <code>this.confirmDiscardEdits</code> is true, this property can be used to customize the
    // error message string displayed to the user in a dialog with options to 
    // cancel the action, or save or discard pending edits in response to sort/filter actions
    // that would otherwise drop unsaved edit values.
	// @visibility external
	// @group editing, i18nMessages
	//<
	confirmDiscardEditsMessage:"This action will discard all unsaved changes for this list.",

    //> @attr listGrid.discardEditsSaveButtonTitle (string :"Save" : IRW)
    // If +link{listGrid.confirmDiscardEdits} is true this is the title for the save button
    // appearing in the lost edits confirmation dialog. Override this for localization if necessary.
    // @visibility external
    // @group editing, i18nMessages
    //<
    discardEditsSaveButtonTitle:"Save",
    
	//> @attr listGrid.addNewBeforeEditing (boolean : false : IRWA)
	// When creating a new edit record via 'startEditingNew()' [or tabbing beyond the end
	// of the last editable field in a list], should we contact the server to create a
	// server-side record before editing begins?
	// @group	editing
	// @visibility advancedInlineEdit
	//<
	//addNewBeforeEditing:false,

	//> @attr listGrid.rowEndEditAction (RowEndEditAction : null : IRW)
	// If the user is editing a record in this listGrid, and attempts to navigate to a field
	// beyond the end of the row, via tab (or shift-tab off the first editable field), this 
	// property determines what action to take:<ul>
	// <li>"next": start editing the next (or previous) record in the list
	// <li>"same": put focus back into the first editable field of the same record.
	// <li>"done": hide the editor
	// <li>"stop": leave focus in the cell being edited
	// <li>"none": no action
	// </ul>
	// @group editing
	// @visibility external
	//<                                            
    //rowEndEditAction:"next",

	//> @attr listGrid.listEndEditAction (RowEndEditAction : null : IRW)
	// If the user is editing the last record in this listGrid, and attempts to navigate 
	// beyond the last row either by tabbing off the last editable field, or using the down
	// arrow key, this property determines what action to take:<ul>
	// <li>"next": start editing a new record at the end of the list.
	// <li>"done": hide the editor
	// <li>"stop": leave focus in the cell being edited
	// <li>"none": no action
	// </ul>
    // <P>
    // See the +link{group:editing,Grid Editing overview} and also the
    // +link{group:unsavedRecords,Editing Unsaved Records overview} for context about how newly
    // added records behave.
    //
	// @group editing
	// @visibility external
    // @example enterNewRows
	//<     
	
    //listEndEditAction:"stop",
    
    //> @attr listGrid.showNewRecordRow (boolean    : null  : [IRW])
    // If this is an editable ListGrid, this property determines whether an extra row should
    // be displayed below the last record, which can be used to add a new record to the dataset
    //<
    
	//>	@attr listGrid.newRecordRowMessage (string : "-- Add New Row --" : IR)
    // If this listGrid is showing the 'newRecordRow' (used for adding new rows to the end
    // of the data), this property determines what message should be displayed in this row.
    // @group editing, i18nMessages
    //<
    newRecordRowMessage:"-- Add New Row --",
    
	//> @attr listGrid.enterKeyEditAction (EnterKeyEditAction : "done" : IRW)
	// What to do when a user hits enter while editing a cell:
    // <ul>
	// <li>"nextCell": start editing the next editable cell in this record (or the first
	//     editable cell in the next record if focus is in the last editable cell in the row)
	// <li>"nextRow": start editing the same field in the next row (skipping any rows where
	//      that would be a non-editable cell.
	// <li>"nextRowStart": start editing the first editable cell in the next row.
	// <li>"done": hide the editor (editing is complete)
    // </ul>
    // Note that if this.autoSaveEdits is true, this may cause a save of the current edit values
	// @group	editing
	// @visibility external
	//<
    enterKeyEditAction:"done",
    
    //> @attr listGrid.escapeKeyEditAction (EscapeKeyEditAction : "cancel" : IRW)
    // What to do when a user hits escape while editing a cell:<ul>
    // <li>"cancel": close the editor and discard the current set of edit values
    // <li>"done": just close the editor (the edit is complete, but the edited values are retained).
    // </ul>
    // Note that if +link{autoSaveEdits} is true, this may cause a save of the current edit values
    // @group editing
    // @visibility external
    //<
    escapeKeyEditAction:"cancel",
    
	//> @type ListGridEditEvent	
    // Event that will trigger inline editing.
    //
    // @value "click"       A single mouse click triggers inline editing
    // @value "doubleClick" A double click triggers inline editing
    // @value "none"        No mouse event will trigger editing.  Editing must be
    //                      programmatically started via +link{listGrid.startEditing()}
    //                      (perhaps from an external button) or may be triggered by 
    //                      keyboard navigation if +link{listGrid.editOnFocus} is set.
    //
	// @group editing
	// @visibility external
    //<

	//> @attr listGrid.editEvent (ListGridEditEvent : "doubleClick" : IRW)
    // Event that will trigger inline editing, see +link{type:ListGridEditEvent} for options.
    // <P>
    // Note this setting has no effect unless +link{canEdit} has been set to enable editing.
    // <P>
    // See also +link{editOnFocus} and +link{startEditing}.
    //
	// @group editing
	// @visibility external
    // @example editByRow
	//<
	editEvent:isc.EH.DOUBLE_CLICK,

	//> @attr listGrid.editOnFocus (boolean : null : [IRWA])
	// Should we start editing when this widget receives focus (if this ListGrid supports
	// editing)?
    // <P>
	// Note that this property being set to true will cause editing to occur on a single
	// click, even if +link{editEvent} is <code>"doubleClick"</code>, because single clicking
    // the grid will place keyboard focus there automatically.
    //
	// @group editing
	// @visibility external
	//<
	// Note - editOnFocus behavior is slightly more complicated than might be imagined. Actual
	// behavior:
	// - focus must go to the body (not the header) to start editing.
	// - if we are currently editing, getting focus will not trigger a new edit.
	// - if the focus is a result of clicking on the listGrid body, we will only start editing
	//   if the user clicked on an editable cell -- this is the same behavior as with 
	//   editEvent:"click"
	// - otherwise when this widget receives focus, the first editable row will become editable.
    
    //> @attr listGrid.editOnF2Keypress (Boolean : true : [IRWA])
    // Should we start editing when the widget has focus and the user presses the "f2" key
    // (if this ListGrid supports editing)?
    // <P>
    // Note that if +link{listGrid.editEvent} is set to <code>"click"</code> or
    // <code>"doubleClick"</code>, the <code>Space</code> or <code>Enter</code> key may
    // also be used to start editing, depending on the value for +link{generateClickOnSpace},
    // +link{generateDoubleClickOnSpace}, +link{generateClickOnEnter} and 
    // +link{generateDoubleClickOnEnter}.
    // <P>
    // If +link{listGrid.canEdit} is false, or +link{listGrid.editEvent} is set to "none" this
    // property has no effect.
    //
    // @group editing
    // @visibility external
    //<
    // Tested on:
    // FF 4.0.1 (Mac)
    // Chrome 11.0.696.77 (Mac)
    // Safari 5.0.5 (Mac)
    // IE 8 (XP)
    editOnF2Keypress:true,
    
    
	//> @attr listGrid.editOnKeyPress (boolean : null : [IRWA])
	// If set to true, when this grid has focus, if the user starts typing character keys
    // we'll start editing the focused cell.
    // @group editing
	// @visibility internal
	//<
    
    //> @attr listGrid.moveEditorOnArrow (boolean : null : [IRWA])
	// If +link{listGrid.editOnKeyPress, editOnKeyPress} is true, once the user starts editing
    // a cell by typing while focused in it, should left / right arrow keys cause the 
    // edit cell to shift horizontally?
    // @group editing
	// @visibility internal
	//< 
    
    	        
    //> @attr listGrid.selectOnEdit (Boolean : true : [IRWA])
    //  When the user starts editing a row, should the row also be selected?
    //  <P>
    //  Note that  when this attribute is set to <code>true</code>, other all other rows in the grid
    //  will be deselected when a record is selected due to editing. 
    // @group editing
    // @visibility external
    //<
    selectOnEdit : true,
    
    //> @attr listGridField.canToggle (Boolean : varies : IRWA)
    // Allows a boolean or +link{valueMap,valueMapped} field to be edited without going into
    // edit mode. When this attribute is set, clicking on the field will change the value - for
    // boolean fields toggling between <code>true</code> and <code>false</code>, and for valueMapped
    // fields, advancing the value to the next option in the valueMap.
    // <P>
    // To enable this feature, +link{listGrid.canEdit} must be set to true.
    // For boolean type fields <code>canToggle</code> is true by default, meaning setting
    // <code>canEdit</code> to true implies the user can toggle the value via a single click
    // without going into edit mode. You can disable this by explicitly setting
    // <code>canToggle</code> to false for a boolean field.<br>
    // Note that you can enable toggling only (without allowing the user to edit other fields)
    // by just setting +link{listGrid.editEvent,grid.editEvent:"none"}.
    // <P>
    // If +link{listGrid.editEvent} is set to "click", when the user clicks on the field, 
    // the value will be toggled, and inline editing will be triggered as usual.
    // Otherwise the toggled value will be saved immediately to the server, or if 
    // +link{listGrid.autoSaveEdits} has been set to false, will be stored as an edit value
    // for the record.
    //
    // @group editing
    // @visibility external
    //<
    
    //> @attr listGrid.enumCriteriaAsInitialValues (Boolean : true : IR)
    // In a ListGrid that has a DataSource and has filter criteria that include values for
    // fields declared as +link{DSFieldType,type "enum"} in the DataSource, by default a newly
    // edited row will use those filter criteria as initial values.
    // <P>
    // For example, if a ListGrid is showing all Accounts that have status:"Active" and a new row
    // is created, the new row will default to status:"Active" unless this flag is set to false.
    //
    // @group editing
    // @visibility external
    //<
    enumCriteriaAsInitialValues:true,
	
	//> @attr listGrid.application (application : null : IRW)
	//		Application to use when saving edited values in a databound ListGrid
	//	@group	editing
	//<

	//> @type AutoComplete
	// Possible autoComplete modes.
	//
	// @value   "smart"    Enable SmartClient autoComplete feature.  Suppresses browser's
	//                     built-in autoComplete feature where applicable.
	// @value   "native"   Allow browser's built-in autoComplete feature to function
	// @value   "none"     Suppress brower's built-in autoComplete feature
	// @group autoComplete
	// @visibility autoComplete
	//<

	//>	@attr listGrid.autoComplete (AutoComplete : null : IRW)
	// Whether to do inline autoComplete in text fields during inline editing<br>
    // Overridden by +link{ListGridField.autoComplete} if specified.
    // If unset picks up the default from the appropriate editor class (subclass of FormItem).
    // 
	// @see listGridField.autoComplete
	// @group autoComplete
	// @visibility autoComplete
	//<
    //autoComplete:null,

	//>	@attr listGrid.uniqueMatch (boolean : true : IRW)
	// When SmartClient autoComplete is enabled, whether to offer only unique matches to the
	// user.
	// <p>
	// Can be individually enabled per TextItem, or if set for the grid as a whole, can
	// be set differently for individual fields.
	//
	// @see listGridField.uniqueMatch
	// @group autoComplete
	// @visibility autoComplete
	//<

    // autoSelectEditors - if true when the user starts editing a cell always select the content
    // of the cell
    autoSelectEditors:true,
	// defaults for the form used for inline editing
    editFormDefaults: {
    	
        canSelectText:true,
        autoDraw:false,
        
        // show error icons on the left by default
        errorOrientation:"left",
        showErrorText:false,
        showErrorStyle:false,
        
        itemKeyPress:function (item, keyName, characterValue) {
            return this.grid.editorKeyPress(item, keyName, characterValue);
        }
    }, 
    
    //> @attr listGrid.longTextEditorThreshold (int : 255 : IRW) 
    // When the length of the field specified by +link{attr:dataSourceField.length} exceeds this
    // value, the ListGrid shows an edit field of type +link{attr:listGrid.longTextEditorType}
    // rather than the standard text field when the field enters inline edit mode.
    //
    // @group editing
    // @visibility external
    //<
    longTextEditorThreshold : 255,
    
    //> @attr listGrid.longTextEditorType (string : "PopUpTextAreaItem" : IRW) 
    // When the length of the field specified by +link{attr:dataSourceField.length} exceeds 
    // <code>this.longTextEditorThreshold</code> show an edit field of this type
    // rather than the standard text field when the field enters inline edit mode.
    //
    // @group editing
    // @visibility external
    //<
    longTextEditorType : "PopUpTextAreaItem",
    
	// functions installed into FormItems used for inline editing.  
	// Note - these will be executed in the scope of the form item (not in the scope of the
	// ListGrid).
	//
	// Set up keyboard handling on form items to handle user navigation via "Enter", "Escape",
	// "Tab" keypresses, etc.
    //
	// NOTE: in Moz, if the user is holding down tab and we are cancelling the field change
	// *while logging to an open Log window*, we'll be stuck in the field indefinitely.  This
	// is probably a non-bug but can look like a freeze.
	//
	// row editing: Cancel tab to prevent focus cycling through the visible form items, since
	// for the first/last editable or visible item, we want to place focus in newly drawn
	// editors (on next/previous row, or just in newly drawn area that we scrolled into)
    editorKeyDown : function (item, keyName) {
        
        if (isc.Browser.isMoz && item && item.multiple && 
            isc.isA.NativeSelectItem(item) && keyName == "Enter") 
        {
            item._selectedOnEnterKeydown = item.getValue();
        }
    },

    editorKeyPress : function (item, keyName, characterValue) {

    	// We will return false to cancel native behavior on any key event for the keys
    	// used for navigating around the edit cells (arrow keys, tab, etc.)
        var EH = isc.EH,
            returnValue,
            editEvent;        
        
        var isMultiLineEditor =  //>PopUpTextAreaItem
                    isc.isA.PopUpTextAreaItem(item) ||     //<PopUpTextAreaItem
                    (isc.RichTextItem && isc.isA.RichTextItem(item)) ||
                    isc.isA.TextAreaItem(item) || isc.isA.SpinnerItem(item);

        if (keyName == "Tab") {
        
        	// Always cancel the native event that would take focus from this item.
        	// This is appropriate as we will always focus in the next item programmatically, 
        	// even if we are currently showing the entire edit row.
        	
            var shift = EH.shiftKeyDown();
                
            // If this is a container item, with sub items, or has a number of focusable
            // icons, we may be moving focus within the item, rather than going to another
            // edit cell.  This is handled by _moveFocusWithinItem().
            if (this.ns.isA.ContainerItem(item) || 
                (item.icons != null && item.icons.length > 0 
                    // If the event occurred on a PopUpTextAreaItem, native focus is either 
                    // on the icon, or within the TextArea in another form.
                    // In this case we always navigate to the next cell
                    && !this.ns.isA.PopUpTextAreaItem(item))
               ) 
            {
                if (!this._moveFocusWithinItem(item, shift)) {
                    return false;
                }
            }
            
            editEvent = shift ? isc.ListGrid.SHIFT_TAB_KEYPRESS 
                                            : isc.ListGrid.TAB_KEYPRESS;
            returnValue = false;
                    
        } else if (keyName == "Enter") {
        	// If the event occurred over an icon, we don't want to interfere with it, as
        	// enter will activate the link (for accessibility)
            if (item.getFocusIconIndex() != null) return;
            
            // allow enter to work normally for text areas.  Alt + Enter overrides.
            if (isMultiLineEditor && isc.EH.altKeyDown() == false) {
                return returnValue;
            }
            if (item._selectedOnEnterKeydown != null) {
                var oldVal = item._selectedOnEnterKeydown;
                delete item._selectedOnEnterKeydown;
                item.setValue(oldVal);
            }
            editEvent = isc.ListGrid.ENTER_KEYPRESS;
            returnValue = false;

        } else if (keyName == "Escape") {
            editEvent = isc.ListGrid.ESCAPE_KEYPRESS;
            returnValue = false;

        // By default move to a new row in response to
        //  - arrow keypress (up / down)
        //  - ctrl + arrow keypress (up / down), but not if the shift key is also down
        // Exceptions:
        //  - Text area use alt+arrowKeypress only
        //  - explicitly avoid row change on alt+arrow key in SelectItem, where this is used
        //    to show / hide the pickList
        //
        
        } else if (keyName == "Arrow_Up") {
            var textArea = isMultiLineEditor;
            if (textArea && !isc.EH.altKeyDown()) return returnValue;
            
            if (isc.isA.SelectItem(item) && !isc.EH.ctrlKeyDown()) return returnValue;
            
            if (isc.EH.ctrlKeyDown() && isc.EH.shiftKeyDown()) return returnValue;
            editEvent = isc.ListGrid.UP_ARROW_KEYPRESS;
            returnValue = false;

        } else if (keyName == "Arrow_Down") {

            var textArea =  isMultiLineEditor;
            if (textArea && !isc.EH.altKeyDown()) return returnValue;
            
            if (isc.isA.SelectItem(item) && !isc.EH.ctrlKeyDown()) return returnValue;
            
            if (isc.EH.ctrlKeyDown && isc.EH.shiftKeyDown()) return returnValue;
            
            editEvent = isc.ListGrid.DOWN_ARROW_KEYPRESS;
            returnValue = false;
    
        // if the user started editing via editOnKeyPress and the 'moveEditorOnArrow' flag is true
        // we shift cells on left/right arrow
        } else if (this.moveEditorOnArrow && this._editSessionFromKeyPress) {
            if (keyName == "Arrow_Left") {
                editEvent = isc.ListGrid.LEFT_ARROW_KEYPRESS;
                returnValue = false;
            } else if (keyName == "Arrow_Right") {
                editEvent = isc.ListGrid.RIGHT_ARROW_KEYPRESS;
                returnValue = false;            
            }
        }
        
        if (editEvent != null) {
            
            if (isc.EH.clickMaskUp()) {
                isc.EH.setMaskedFocusCanvas(null, isc.EH.clickMaskRegistry.last());
            }
    
            // Fire cellEditEnd to handle saving out the value / moving to the next cell as 
            // appropriate
            this.cellEditEnd(editEvent);
        }
        
        return returnValue;            
    },

    // _moveFocusWithinItem() Helper method fired when the user hits tab / shift+tab while
    // focused in some edit item. This will move focus to the appropriate icon or sub item
    // if necessary.  A return value of false indicates focus was moved within the item, so
    // should not move to another edit cell.
    _moveFocusWithinItem : function (item, shift) {
        
        if (!item) return true;
        return (!item._moveFocusWithinItem(!shift));
    },

	// Override elementFocus on the form items:
	// If we're editing the whole row, and the user clicks in a new field to focus in it,
	// call 'editCellEnd' method to perform validation / saving on the previous
	// edit field (if required).
	
    _editFormItem_elementFocus : function (suppressHandlers) {
        var cell = this._setLGEditCellForFocus(suppressHandlers);
        this.Super("elementFocus", arguments);
        this._fireLGEditorEnter(cell[0], cell[1]);
    },
    _editFormItem_setLGEditCellForFocus : function (suppressHandlers) {

        

        var form = this.form,
            lg = form.grid;
    	    
        
        var rowNum, colNum, fieldName, fieldChanged;
        if (lg._editorShowing) {
            rowNum = lg._editRowNum;
            if (!lg.editByCell) {
                rowNum = lg._editRowNum;
                fieldName = this.getFieldName();
                colNum = lg.fields.findIndex(lg.fieldIdProperty, fieldName);
                
                if (!suppressHandlers) {
                    fieldChanged = (lg._editColNum != colNum);
                    // If the user has clicked in another field in the edit form,
                    // fire editField on the appropriate field
                    if (fieldChanged) {
                	// store the new edit cell
                        lg.setNewEditCell(rowNum, colNum);
                	// fire 'cellEditEnd' to save / validate before moving to the new cell
                        lg.cellEditEnd(isc.ListGrid.EDIT_FIELD_CHANGE);
                    
                	// Note - if we could cancel the focus here, it might make sense, as the 
                	// cellEditEnd callback method will re-focus in the new focus cell, but we can't
                	// cancel the focus by simply returning false from this method.
                	// Therefore allow the focus to proceed, and fall through to the super 
                	// implementation of this method which will fire focus handlers, show any
                	// 'showOnFocus' icons, etc.
                    }
                }
            } else {
                colNum = lg._editColNum;         
            }
        }
        return [rowNum, colNum];
    },
    _editFormItem_fireLGEditorEnter : function (rowNum, colNum) {
        var form = this.form,
            lg = form.grid;
        
        if (lg._editorShowing) {
        	// If this is the current edit field, and hasn't yet fired its 'editorEnter' handlers
        	// fire them now.
            var rowEnter = this._rowEnterOnFocus,
                cellEnter = this._cellEnterOnFocus;
            // Note: we're clearing out the flags before we fire the handlers. If the
            // handler trips a change of edit row, etc., we want editorExit to fire.
            delete this._rowEnterOnFocus;
            delete this._cellEnterOnFocus;
            
            
            var editVals = isc.addProperties(
                {}, 
                lg.getCellRecord(rowNum,colNum),
                lg._getEditValues(rowNum,colNum)
            );
            
            if (cellEnter) {
                var fieldName = lg.getFieldName(colNum);
                lg._handleEditorEnter(this, rowNum, colNum, editVals[fieldName]);
            }
            if (rowEnter) lg._handleRowEditorEnter(this, rowNum, editVals);

        } else {
            lg.logWarn("suppressing editorEnter handlers on focus as listGrid._editorShowing is null");
        }
    },
    

	// Header
	// ----------------------------------------------------------------------------------------

    //> @groupDef gridHeader
    // Properties and methods related to the ListGrid header. ListGrid headers are implemented
    // as a +link{class:Toolbar} of buttons shown at the top of the ListGrid 
    // (one button per column).<br>
    // The toolbar header provides UI for interacting with the ListGrid fields directly (sorting,
    // resizing, reordering columns, etc).
    // @visibility external
    //<

    //> @attr listGrid.header (AutoChild Layout : null : R)
    // A Toolbar used to manager the headers shown for each column of the grid.
    // @group gridHeader
    // @visibility external
    //<

	//>	@attr listGrid.canTabToHeader (boolean : null : IR)
	// Should the header be included in the tab-order for the page? If not explicitly specified,
	// the header will be included in the tab order for the page if +link{isc.setScreenReaderMode()} is
    // called.
	// @group accessibility
	// @visibility external
	//<
	//canTabToHeader:null,
    
	//>	@attr listGrid.headerHeight (number : 22 : [IRW])
	//          The height of this listGrid's header, in pixels.
    //      @setter listGrid.setHeaderHeight()
	//      @visibility external
	//      @group  gridHeader, sizing
	//<
    // Note: we treat a headerHeight of zero as an equivalent of showHeader:false
	headerHeight:22,

    //> @attr listGrid.minFieldWidth (int : 15 : IRW)
    // Minimum size, in pixels, for ListGrid headers.
    // @visibility external
    //<
    minFieldWidth:15,

	//>	@attr listGrid.showHeader (Boolean: true : [IRW])
	// Should we show the header for this ListGrid?
    // @group gridHeader, appearance
	// @visibility external
	//<
	showHeader:true,

	//>	@attr listGrid.headerBarStyle (CSSStyleName : null : IR)
	// Set the CSS style used for the header as a whole.
	// @group	gridHeader, appearance
    // @visibility external
	//<
    //headerBarStyle:null,
    
    //>	@attr listGrid.headerBackgroundColor (Color: "#CCCCCC" : IRW)
	// BackgroundColor for the header toolbar. Typically this is set to match the color
    // of the header buttons.
	//		@group	gridHeader, appearance
    // @visibility external
    //<
	headerBackgroundColor:"#CCCCCC",		
 
    headerDefaults : {
        // immediately reposition headers during drag resizing, don't delay
        instantRelayout:true,
        // when the user resizes buttons, don't try to fit them into the available space -
        // allow the user to introduce hscrolling
        enforcePolicy:false, 

        // when the header is clicked, have it call headerClick() on us
		itemClick : function (button, buttonNum) {
			this.Super("itemClick",arguments);
		    this.grid._headerClick(buttonNum, this);
		},
		
		itemDoubleClick : function (button, buttonNum) {
            this.Super("itemDoubleClick", arguments);
            this.grid._headerDoubleClick(buttonNum, this);
        },

        showContextMenu : function () {
            return this.grid.headerBarContextClick(this);
        },
        
		backgroundRepeat:isc.Canvas.NO_REPEAT,
        
        // don't print the header, we handle this as part of the body instead, to ensure column
        // alignment
        shouldPrint:false
    },
    
    //> @attr listGrid.headerButtonConstructor (Class : null : IR)
    // Widget class for this ListGrid's header buttons. If unset, constructor will be 
    // picked up directly from the standard +link{class:Toolbar} button constructor.
	// @group	gridHeader, appearance
    // @visibility external
    //<
    
    //>	@attr listGrid.headerBaseStyle (CSSStyleName : null : IR)
	// +link{Button.baseStyle} to apply to the buttons in the header, and the sorter, for 
    // this ListGrid.
    // Note that, depending on the +link{listGrid.headerButtonConstructor, Class} of the header
    // buttons, you may also need to set +link{listGrid.headerTitleStyle}.
	// @group	gridHeader, appearance
    // @visibility external
	//<
    
    //> @attr listGrid.headerTitleStyle (CSSStyleName : null : IR) 
    // +link{StretchImgButton.titleStyle} to apply to the buttons in the header, and the sorter,
    // for this ListGrid.
    // Note that this will typically only have an effect if 
    // +link{listGrid.headerButtonConstructor} is set to +link{class:StretchImgButton} or a subclass 
    // thereof.
	// @group	gridHeader, appearance
    // @visibility external
	//<
    
	//>	@attr listGrid.frozenHeaderBaseStyle (CSSStyleName : null : IR)
    // If this listGrid contains any frozen fields, this property can be used to apply a custom
    // headerBaseStyle to the frozen set of fields. If unset, the standard headerBaseStyle
    // will be used for both frozen and unfrozen cells.
    // @visibility external
    // @group gridHeader, appearance, frozenFields
    // @see listGrid.headerBaseStyle
    // @see listGridField.frozen
    //<
    
    //>	@attr listGrid.frozenHeaderTitleStyle (CSSStyleName : null : IR)
    // If this listGrid contains any frozen fields, this property can be used to apply a custom
    // headerTitleStyle to the frozen set of fields. If unset, the standard headerTitleStyle
    // will be used for both frozen and unfrozen cells.
    // @visibility external
    // @group gridHeader, appearance, frozenFields
    // @see listGrid.headerTitleStyle
    // @see listGridField.frozen
    //<
    

    //>	@attr listGrid.headerButtonDefaults (Button Properties: {...} : IRA)
	// Defaults to apply to all header buttons. To modify this object, 
    // use +link{class.changeDefaults(), ListGrid.changeDefaults()} 
    // rather than replacing with an entirely new object.
	// @group	gridHeader, appearance
    // @visibility external
	//<
	headerButtonDefaults:{
        clipTitle: true,
        showClippedTitleOnHover: true,

        titleHover : function headerButtonDefaultTitleHoverImpl() {
            var header = this.parentElement,
                grid = header.grid;
            if (grid && grid.headerHover) return grid.headerHover(this.masterIndex);
        },
        titleHoverHTML : function headerButtonDefaultTitleHoverHTMLImpl() {
            var header = this.parentElement,
                grid = header.grid;
            return (grid == null
                    ? null
                    : grid.headerHoverHTML(this.masterIndex,
                                           grid.defaultHeaderHoverHTML(this.masterIndex)));
        },

        // override getCurrentCursor to show default (non pointer) for canSort:false fields
        getCurrentCursor : function () {
            var grid = this.parentElement ? this.parentElement.grid : null;
            var currentCursor = isc.Canvas.DEFAULT;
            if (grid && this.masterIndex != null) {
                var field = grid.getField(this.masterIndex),
                    canSort = grid._canSort(field) != false;
                if (canSort) currentCursor = isc.Canvas.HAND;
            } else {
                if (this.isSorterButton) {
                    if (!grid && isc.isA.ListGrid(this.parentElement)) grid = this.parentElement;
                    if (grid) {
                        var canSort = grid._canSort(grid._getSortFieldNum()) != false;
                        if (canSort) currentCursor = isc.Canvas.HAND;
                    }
                } else {
                    currentCursor = this.getClass().getPrototype().cursor;
                }
            }
            this.cursor = currentCursor;
            return this.Super("getCurrentCursor", arguments);
        },

        dragScrollType:"parentsOnly",
        minWidth:20,
        hoverDelay:500
    },

	//>	@attr listGrid.headerButtonProperties (Button Properties: null : IRA)
	// Properties to apply to all header buttons.
    // Overrides defaults applied via  +link{ListGrid.headerButtonDefaults}.
    // @group	gridHeader, appearance
    // @visibility external
	//<

    //> @attr listGrid.clipHeaderTitles (Boolean : varies : IRA)
    // Whether the ListGrid should manage the clipping of titles of header buttons.
    // <p>
    // In some cases this may be preferable to the button component's default clipping behavior
    // because if a +link{ListGrid.showSortArrow,sort arrow} or sort numeral is displayed for
    // a header, then the button's default clipping behavior may clip the sort arrow/numeral
    // whereas ListGrid-managed title clipping utilizes special HTML which keeps the sort
    // arrow/numeral visible.
    // <p>
    // This feature is automatically enabled if supported by the browser. The only supported
    // use of this attribute is to <em>disable</em> the feature by setting clipHeaderTitles
    // to false.
    // @group gridHeader, appearance
    // @visibility external
    //<
    
    clipHeaderTitles: (!isc.Browser.isIE ||
                       (isc.Browser.version > 6 && isc.Browser.isStrict) ||
                       isc.Browser.version >= 10),

    //> @attr listGrid.sorterConstructor (Class : Button : IR)
    // Widget class for the corner sort button, if showing.  This button displays the current 
    // sort direction of the primary sort field (either the only sorted field or the first in a
    // +link{listGrid.canMultiSort, multi-sort} grid) and reverses the direction of that field
    // when clicked.  For consistent appearance, this
    // is usually set to match +link{listGrid.headerButtonConstructor}
    // @group gridHeader, appearance
    // @visibility external
    //<
    sorterConstructor:isc.Button,
    
    
    //> @attr listGrid.sorterButtonTitle (string : "corner menu" : IR)
    // The title for the corner sort button.  The title will only 
    // +link{Class.changeDefaults(), ListGrid.changeDefaults()} rather than replacing with an
    // entirely new object.
    // @group i18nMessages, gridHeader, appearance
    // @visibility external
    //<
    sorterButtonTitle: "corner menu",

    //> @attr listGrid.sorterDefaults (object : {...} : IRA)
    // Defaults to apply to the corner sort button. To modify this object, use
    // +link{Class.changeDefaults(), ListGrid.changeDefaults()} rather than replacing with an
    // entirely new object.
    // @group gridHeader, appearance
    // @visibility external
    //<
    sorterDefaults:{
        _redrawWithParent:false,
        getTitle : function () {return this.creator.getSortArrowImage() },
        click : function () { return this.creator._sorterClick() },
        showContextMenu : function() { return this.creator._sorterContextClick() },
        isSorterButton: true,
        align: "center",
        canFocus:false  // no need to focus, since we allow focus-ing on the header buttons
    },

    //>	@attr listGrid.sorterProperties (Button Properties: null : IRA)
    // Properties to apply to the sorter button. Overrides defaults applied via 
    // +link{ListGrid.sorterDefaults}.
    // @group gridHeader, appearance
    // @visibility external
    //<

    // Sorting
    // --------------------------------------------------------------------------------------------

    //>	@attr listGrid.canSort (Boolean : true : [IRW])
    // Enables or disables interactive sorting behavior for this listGrid. Does not
    // affect sorting by direct calls to the +link{listGrid.sort, sort} or 
    // +link{listGrid.setSort, setSort} methods.
    // 
    // @group sorting
    // @visibility external
    //<
    canSort:true,

    //>	@attr listGrid.canUnsort (boolean : false : [IRW])
    // When set to true, the third click on a column header removes the sort indicator
    // from the field.
    //
    // @group sorting
    // @visibility internal
    //<
    canUnsort:false,

    //> @attr listGrid.invalidateCacheOnUnsort (boolean : false : [IRW])
    // If true, and +link{listGrid.canUnsort} is also true and the grid is databound to a
    // +link{ResultSet}, then unsort causes the grid to drop the current client-side
    // data cache and refetch the current range of records from the server.
    //
    // @group sorting
    // @visibility internal
    //<
    invalidateCacheOnUnsort: false,

    //> @attr listGrid.selectHeaderOnSort (Boolean : true : [IRW])
    // If true, show the field-header for the sorted field (or the first field in a 
    // +link{listGrid.canMultiSort, multi-sort} grid) in the selected state.
    //
    // @group sorting
    // @visibility external
    //<
    selectHeaderOnSort: true,

    //>	@attr listGrid.sortFieldNum (number : null : [IRW])
    // Specifies the number of the field by which to sort this listGrid. Column numbers
    // start at 0 for the left-most column.
    // @group sorting
    // @example sort
    // @visibility external
    // @deprecated as of version 7.0 in favor of +link{listGrid.sortField}
    //<
    //sortFieldNum:null,

    //> @attr listGrid.sortField (String or integer : null : IR)
    // Specifies the field by which this grid should be initially sorted. Can be set to either a
    // +link{listGridField.name,field name} or the index of the field in the fields Array.
    // <P>
    // ListGrids also support being initialized with multiple-field sort via 
    // +link{listGrid.initialSort}. If initialSort is specified, it will be used in preference
    // to this property.
    // <P>
    // To sort the grid after it has been initialized, use +link{ListGrid.sort()} or 
    // +link{listGrid.setSort()}. Details about the current sort of a live grid can be
    // retrieved by calling +link{listGrid.getSortField()} or +link{listGrid.getSort()}
    //
    // @group sorting
    // @example sort
    // @visibility external
    //<

	//>	@attr listGrid.keyboardClickField (string|number : null : [IRW])
	// When simulating click events listGrid rows as a result of keyboard events
	// (navigating using the arrow keys, space, enter for doubleClick), which column
	// should the event be generated upon?
	// Should be set to the name or index of the desired column.
	// If null, defaults to the first column where +link{listGridField.ignoreKeyboardClicks}
	// is not false.
	// @group  events
	//<

	//>	@attr listGrid.sortDirection (SortDirection : Array.ASCENDING : [IRW])
    // Sorting direction of this ListGrid. If specified when the ListGrid is initialized,
    // this property will be the default sorting direction for the +link{listGrid.sortField}.
    // May be overridden by specifying +link{ListGridField.sortDirection}.
    // <P>
    // After initialization, this property will be updated on +link{ListGrid.sort()} or
    // +link{ListGrid.setSort()} to reflect the current sort direction of the grid. When
    // this grid is sorted by multiple fields, the grid's sortDirection reflects the
    // sort direction of the primary sort field.
    // @group  sorting
    // @see type:SortDirection
    // @example sort
    // @setter listGrid.sort()
	// @visibility external
	//<
   	
	sortDirection:Array.ASCENDING,

	//>	@attr listGrid.showSortArrow (SortArrow : null : [IRW])
	// Indicates whether a sorting arrow should appear for the listGrid, and its
	// location. See +link{SortArrow} for details.
    // <P>
	// Clicking the sort arrow reverses the direction of sorting for the current sort
	// column (if any), or sorts the listGrid by its first sortable column. The arrow
	// image on the button indicates the current direction of sorting.
    // If undefined, the sort arrow will show up in the sorted field, and the
    // corner sort button will be displayed if a vertical scrollbar is being displayed
    //
	// @group sorting, appearance
	// @visibility external
	//<
	
    //showSortArrow:null,

    //> @attr listGrid.canPickFields (Boolean : true : [IRW])
    // Indicates whether the field picker item and submenu should be present in the header
    // context menu. This menu allows the user to hide visible fields and show hidden fields.
    // By default only fields explicitly included in the +link{listGrid.fields} array will
    // be available in this menu, unless +link{listGrid.canPickOmittedFields} is set to true for
    // a databound grid.
    //
    // @visibility external
    //<
    canPickFields: true,
    
    //> @attr listGrid.canPickOmittedFields (Boolean : false : [IR])
    // If this grid has a specified +link{listGrid.dataSource}, and
    // +link{listGrid.useAllDataSourceFields} is false, setting this property to true
    // will cause all dataSource fields not included in the specified set of fields to
    // show up in the +link{listGrid.canPickFields,field picker menu item}.
    // <P>
    // Has no effect if +link{listGrid.fields} is unset (as in this case all dataSource fields
    // will be displayed by default), or if +link{listGrid.canPickFields} is false.
    // @visibility external
    //<
    canPickOmittedFields:false,
    
    // Frozen Fields (aka Frozen Columns)
    // ---------------------------------------------------------------------------------------
    
    //> @groupDef frozenFields
    // Frozen fields are fields that do not scroll horizontally with other fields, remaining on
    // the screen while other fields may be scrolled off.  This feature is typically used to
    // allow basic identifying information (like an "accountId") to remain on screen while the
    // user scrolls through a large number of related fields.
    // <P>
    // Fields can be programmatically frozen via setting
    // +link{listGridField.frozen,field.frozen} to true when the grid is created, or
    // dynamically frozen and unfrozen via +link{listGrid.freezeField,freezeField()} and
    // +link{listGrid.unfreezeField,unfreezeField()}.
    // The setting +link{listGrid.canFreezeFields,canFreezeFields} enables a user interface to
    // allow end users to dynamically freeze and unfreeze fields.
    // <P>
    // The frozen fields feature is not compatible with the following features:
    // <ul>
    // <li> variable height auto-sizing records
    //      (+link{listGrid.fixedRecordHeights,fixedRecordHeights:false})
    // <li> +link{autoFitData}:"horizontal", as well as headers that autoFit to titles
    //      (normally enabled via <code>field.overflow:"visible"</code>)
    // <li> the +link{CubeGrid} subclass of ListGrid
    // <li> nested grids
    // </ul>
    // The frozen fields feature <b>is</b> compatible with column resize and reorder, selection
    // and multi-selection, loading data on demand, inline editing, drag and drop and reorder
    // of records, the +link{TreeGrid} subclass of ListGrid, and all dynamic styling-related and
    // formatting-related features.
    // 
    // @title Frozen Fields
    // @visibility external
    //<

    //> @attr listGrid.canFreezeFields (boolean : null : IRW)
    // Whether an interface should be shown to allow user is allowed to dynamically "freeze" or
    // "unfreeze" columns with respect to horizontally scrolling. If unset, this property defaults
    // to <code>true</code> unless:<ul>
    // <li>+link{listGrid.fixedRecordHeights,this.fixedRecordHeights} is <code>false</code></li>
    // <li>+link{listGrid.bodyOverflow,this.bodyOverflow} is <code>"visible"</code></li>
    // <li>+link{listGrid.autoFitData,this.autoFitData} is set to <code>"horizontal"</code> or 
    // <code>"both"</code></li>
    // <li>Any field has overflow set to <code>"visible"</code></li></ul>
    // <P>
    // Note that the <code>canFreezeFields</code> setting enables or disables the user
    // interface for freezing and unfreezing fields only.  Fields can be programmatically
    // frozen via setting +link{listGridField.frozen,field.frozen} to true when the grid is
    // created, or dynamically frozen and unfrozen via +link{freezeField()} and
    // +link{unfreezeField()}.
    //
    // @group frozenFields
    // @visibility external
    //<
    // Note that fixedColumnWidths:false will also disable canFreezeFields but this
    // is not currently public.
    
    // -------------------------
    // Formula / summary fields (picked up from databoundcomponent)
    
    //> @attr listGrid.canAddFormulaFields (boolean : false : IRW)
    // @include dataBoundComponent.canAddFormulaFields
    // @visibility external
    //<
    
    //> @method listGrid.getFormulaFieldValue()
    // @include dataBoundComponent.getFormulaFieldValue()
    // @visibility external
    //<
    
    //> @attr listGrid.canAddSummaryFields (boolean : false : IRW)
    // @include dataBoundComponent.canAddSummaryFields
    // @visibility external
    //<
    
    //> @method listGrid.getSummaryFieldValue()
    // @include dataBoundComponent.getSummaryFieldValue()
    // @visibility external
    //<



	// Context Menus
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.showCellContextMenus (Boolean : false : [IRW])	
	// Whether to show a context menu with standard items for all context clicks on rows in the
	// body.
    // @visibility external
	//<
	//showCellContextMenus:false,
    
    //> @attr listGrid.openRecordEditorContextMenuItemTitle (string : "Edit" : [IRW])
    // If +link{listGrid.canOpenRecordEditor} is true and +link{listGrid.showCellContextMenus}
    // is true, this property specifies the title for the context menu item shown allowing the
    // user to perfom editing on a row via an embedded form.
    // @group i18nMessages
    // @visibility nextedGrid
    //<
    openRecordEditorContextMenuItemTitle:"Edit",

    //> @attr listGrid.dismissEmbeddedComponentContextMenuItemTitle (string : "Dismiss" : IRW)
    // If +link{listGrid.showCellContextMenus} is true, and we are currently showing either
    // an embedded editor (see +link{listGrid.canOpenRecordEditor}) or an embedded
    // detail grid (see +link{listGrid.canOpenRecordDetailGrid}, this property
    // specifies the title for the context menu item shown allowing the user to dismiss the
    // embedded component.
    // @group i18nMessages
    // @visibility nextedGrid
    //<
    dismissEmbeddedComponentContextMenuItemTitle:"Dismiss",

    //> @attr listGrid.deleteRecordContextMenuItemTitle (string : "Delete" : IRW)
    // If +link{listGrid.showCellContextMenus} is true, this property
    // specifies the title for the context menu item shown allowing the user to delete the
    // record on which the contextMenu was shown.
    // @group i18nMessages
    // @visibility experimental
    //<    
    deleteRecordContextMenuItemTitle:"Delete",


	//> @attr listGrid.canOpenRecordDetailGrid (boolean : true : [IRW])
	// Whether context menu items will be shown for viewing records from related DataSources in
	// grids embedded in the record.
	// <P>
	// Valid only when <code>showCellContextMenus</code> is true.
	// @visibility nestedGrid
	//<
    canOpenRecordDetailGrid:true,

	//> @attr listGrid.recordDetailGridProperties (Object : null : [IR])
	// Properties for detail grids shown embedded inside rows.
	// @visibility nestedGrid
	//<

	//> @attr listGrid.canOpenRecordEditor (boolean : true : [IRW])
	// Whether a context menu item will be shown for editing records with a form embedded in
	// the record.
	// <P>
	// Valid only when <code>showCellContextMenus</code> is true.
	// @visibility nestedGrid
	//<
    canOpenRecordEditor:true,

	//> @attr listGrid.recordEditorProperties (Object : null : [IR])
	// Properties for editor forms shown embedded inside rows.
    // @see listGrid.canOpenRecordEditor
	// @visibility nestedGrid
	//<
    
    //> @attr listGrid.recordEditorSaveButtonTitle (string : "Save" : [IRW])
    // Title for the Save button shown in the editor form embedded inside rows if 
    // +link{listGrid.canOpenRecordEditor} is true.
    // @see listGrid.canOpenRecordEditor
    // @group i18nMessages
    // @visibility nestedGrid
    //<
    recordEditorSaveButtonTitle:"Save", 

    //> @attr listGrid.recordEditorCancelButtonTitle (string : "Cancel" : [IRW])
    // Title for the Cancel button shown in the editor form embedded inside rows if 
    // +link{listGrid.canOpenRecordEditor} is true.
    // @see listGrid.canOpenRecordEditor
    // @group i18nMessages
    // @visibility nestedGrid
    //<
    recordEditorCancelButtonTitle:"Cancel",


	//>!BackCompat 2007.02.02
    // showCornerContextMenu was never externally documented and we have no in-code comments
    // about having ever exposed this property, so it may be safe to get rid of this
    // back-compat

	//>	@attr listGrid.showCornerContextMenu (boolean : null : [IR])	
	// Whether to allow a context menu on the sorter with standard items for showing and hiding
	// fields.
    // @deprecated as of 5.6 in favor of +link{attr:listGrid.showHeaderContextMenu}
	//<
    //<!BackCompat
    
	//>	@attr listGrid.showHeaderContextMenu (Boolean : true : [IR])	
	// Whether to show a context menu on the header with standard items for showing and hiding
	// fields.
    // @group gridHeader
    // @see method:listGrid.displayHeaderContextMenu()
    // @see method:listGrid.getHeaderContextMenuItems()
    // @visibility external
	//<
    // NOTE: avoid crashing if Menu class isn't loaded by defaulting to false.
	// when we load the Menu class, we override this default.
	//showHeaderContextMenu:false,
    
    // headerMenuButton
    // ----------------------------
    //>	@attr listGrid.showHeaderMenuButton (Boolean : false : [IR])	
    // If set to true and +link{listGrid.showHeaderContextMenu,showHeaderContextMenu} is true, the
    // +link{listgrid.headerMenuButton} will be displayed when the user rolls
    // over the header buttons in this grid.
    // @group headerMenuButton    
    // @visibility external
	//<
    // As with showHeaderContextMenu, this default should not be set to true until we know
    // for sure that Menu has been loaded (see Menu.js)
    //showHeaderMenuButton:true,
    
    //> @attr listGrid.leaveHeaderMenuButtonSpace (boolean : null : [IWA])
    // If +link{listGrid.showHeaderMenuButton} is true, when auto-fitting fields to 
    // the title width via +link{listGrid.autoFitFieldWidths} or +link{listGridField.autoFitWidth}, 
    // should the button be sized such that there is enough space for the heeader menu button to
    // show without covering the field title?
    // <P>
    // May be explicitly specified at the +link{listGridField.leaveHeaderMenuButtonSpace,field level}
    // or at the +link{listGrid.leaveHeaderMenuButtonSpace,grid level}. If not explicitly
    // specified space will be left for fields with 
    // +link{listGridField.align} set to <code>"left"</code> or <code>"right"</code>, but not for
    // fields with align set to <code>"center"</code>.
    //
    // @group headerMenuButton
    // @visibility external
    //<
    leaveHeaderMenuButtonSpace:null,
    
    //> @attr listGridField.leaveHeaderMenuButtonSpace (boolean : null : [IWA])
    // If +link{listGrid.showHeaderMenuButton} is true, when auto-fitting fields to 
    // the title width via +link{listGrid.autoFitFieldWidths} or +link{listGridField.autoFitWidth}, 
    // should the button be sized such that there is enough space for the heeader menu button to
    // show without covering the field title?
    // <P>
    // May be explicitly specified at the +link{listGridField.leaveHaderMenuButtonSpace,field level}
    // or at the +link{listGrid.leaveHeaderMenuButtonSpace,grid level}. If not explicitly
    // specified space will be left for fields with 
    // +link{listGridField.align} set to <code>"left"</code> or <code>"right"</code>, but not for
    // fields with align set to <code>"center"</code>.
    //
    // @group headerMenuButton
    // @visibility external
    //<
    
    //> @attr listGrid.headerMenuButtonConstructor (className : null : [IRA])
    // Constructor for the  +link{listGrid.headerMenuButton}. If unset a standard "Button" will
    // be rendered out. Note that this property may be overridden by different skins.
    // @group headerMenuButton
    // @visibility external
    //<
    //headerMenuButtonConstructor: "StretchImgButton",
    
    //> @attr listGrid.headerMenuButton (AutoChild StatefulCanvas : null : [RA])
    // If +link{showHeaderMenuButton} is true, when the user rolls over the header buttons in this
    // grid the headerMenuButton will be shown over the header button in question. When clicked
    // this button will display the standard header context menu (see
    // +link{listGrid.displayHeaderContextMenu}).
    // <P>
    // +link{group:headerMenuButton,Several properties} exist to customize the appearance of the
    // headerMenuButton. Also see the +link{type:AutoChild} documentation for information on how 
    // to make freeform modifications to autoChild widgets
    // @group headerMenuButton
    // @visibility external
    //<
    
    //>	@attr listGrid.headerMenuButtonIcon (URL : "[SKIN]/ListGrid/sort_descending.gif" : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the icon shown on the
    // auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonIcon:"[SKIN]/ListGrid/headerMenuButton_icon.gif",
    
    //>	@attr listGrid.headerMenuButtonIconWidth (number : 7 : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the width of the icon
    // shown on the auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonIconWidth:7,
    
    //>	@attr listGrid.headerMenuButtonIconHeight (number : 7 : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the height of the icon
    // shown on the auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonIconHeight:7,
    
    //>	@attr listGrid.headerMenuButtonWidth (number : 16 : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the width of the 
    // auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonWidth:16,

    //>	@attr listGrid.headerMenuButtonHeight (measure : "100%" : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the height of the 
    // auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonHeight:"100%",
    

	// Drag Resize / Reorder / Drag and Drop
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.canDragRecordsOut (Boolean : false : [IRW])	
	// Indicates whether records can be dragged from this listGrid and dropped elsewhere.
	// @visibility external
	// @group  dragging
    // @see ListGridRecord.canDrag
    // @see ListGridRecord.canAcceptDrop
    // @example dragListMove
	//<
	canDragRecordsOut:false,
    
	//>	@attr listGrid.canAcceptDroppedRecords (Boolean : false : [IRW])
	// Indicates whether records can be dropped into this listGrid.
	// @visibility external
	// @group  dragging
    // @see ListGridRecord.canDrag
    // @see ListGridRecord.canAcceptDrop    
    // @example dragListMove
	//<
	//canAcceptDroppedRecords:false,
    
	//>	@attr listGrid.canReorderRecords (Boolean : false : [IRW])
	// Indicates whether records can be reordered by dragging within this listGrid.
	// @visibility external
    // @group  dragging
    // @see ListGridRecord.canDrag
    // @see ListGridRecord.canAcceptDrop    
    // @example dragListMove
    // @example gridsDragReorder
	//<
	//canReorderRecords:false,
    
	//>	@attr listGrid.canReorderFields (Boolean : true : [IRW])
	// Indicates whether fields in this listGrid can be reordered by dragging and
	// dropping header fields.
	// @visibility external
	// @group  dragging
    // @example columnOrder
	//<
	canReorderFields:true,
    
	//>	@attr listGrid.canResizeFields (Boolean : true : [IRW])
	// Indicates whether fields in this listGrid can be resized by dragging header
	// fields.
	// @visibility external
	// @group  dragging
    // @example columnSize
	//<
	canResizeFields:true,

	// for dragging records out, use the drag tracker
    dragAppearance:isc.EH.TRACKER,

    // if you set canDragResize to true on the grid show an outline rather than the tracker!    
    dragResizeAppearance:isc.EH.OUTLINE,
    
    //> @type DragTrackerMode
    // When records are being dragged from within a ListGrid, what sort of drag-tracker
    // should be displayed?
    // @value "none" Don't display a drag tracker at all
    // @value "icon" Display an icon to represent the record(s) being dragged. Icon src is
    //              derived from +link{ListGrid.getDragTrackerIcon()}
    // @value "title" Display a title for the record being dragged. Title derived from
    //              +link{ListGrid.getDragTrackerTitle()}
    // @value "record" Display the entire record being dragged
    // @group dragTracker
    // @visibility external
    //<
    
    //> @attr listGrid.dragTrackerMode (DragTrackerMode : "icon" : [IRA])
    // When records are being dragged from within a ListGrid, what sort of drag-tracker
    // should be displayed?<br>
    // Note that if multiple records are being dragged the displayed tracker will be
    // based on the first selected record.
    // @group dragTracker
    // @visibility external
    //<
    dragTrackerMode:"title",
    
    
	//>	@attr listGrid.resizeFieldsInRealTime (Boolean : isc.Browser.isIE && isc.Browser.isWin : IRWA)
	//		True == we redraw the list viewer in real time as fields are being resized.
	//		This can be slow with a large list and/or on some platforms.
    //
	//		@group	dragging
    //      @visibility external
	//<
	resizeFieldsInRealTime: (isc.Browser.isIE && isc.Browser.isWin) 
                            || (isc.Browser.isFirefox && isc.Browser.geckoVersion >= 20080529)
                            // Safari 3.0+, Google Chrome
                            || (isc.Browser.isSafari && isc.Browser.safariVersion >= 500), 
	
    //>	@attr listGrid.dragDataAction		
    // @include dataBoundComponent.dragDataAction
    //<
    
	// Embedded Components
	// --------------------------------------------------------------------------------------------
    embeddedComponentIndent: 25,

	// Nested Master-Detail
	// --------------------------------------------------------------------------------------------
    nestedGridDefaults : {
        height:150
    },

	// Skinning
	// --------------------------------------------------------------------------------------------
	//>	@attr listGrid.skinImgDir (URL : "images/ListGrid/" : IRWA)
	// Where do 'skin' images (those provided with the class) live?
	// @group appearance, images
    // @visibility external
	//<
	skinImgDir:"images/ListGrid/",		

	//>	@attr listGrid.sortAscendingImage (ImgProperties : {...} : IRWA)
	// Image to show when sorting ascending. See +link{class:ImgProperties} for format.
	// @group appearance
    // @visibility external
	//<
	sortAscendingImage:{src:"[SKIN]sort_ascending.gif", width:7, height:7},

	//>	@attr listGrid.sortDescendingImage (ImgProperties : {...} : IRWA)
	// Image to show when sorting descending. See +link{class:ImgProperties} for format.
	// @group appearance
    // @visibility external
	//<
	sortDescendingImage:{src:"[SKIN]sort_descending.gif", width:7, height:7},

	//>	@attr listGrid.trackerImage (ImgProperties : {...} : IRWA)
	// Default image to use for the dragTracker when things are dragged within or out of this 
    // list. See +link{class:ImgProperties} for format.
    //
	// @group dragTracker
    // @see listGrid.dragTrackerMode
    // @see listGrid.getDragTrackerIcon()
    // @visibility external
	//<	
    trackerImage:{src:"[SKIN]tracker.gif", width:16, height:16},

    //> @attr listGrid.booleanBaseStyle (CSSStyleName : null : IRA)
    // An optional CSS style to apply to the checkbox image. If supplied, and the checkbox is
    // enabled, the base style is suffixed with "True", "False", or "Partial" if the checkbox
    // is selected, unselected, or partially selected; if the checkbox is disabled, the suffix
    // is "TrueDisabled", "FalseDisabled", or "PartialDisabled".
    // <p>
    // <b>NOTE:</b> This attribute is not supported by +link{TreeGrid}.
    // @group imageColumns
    // @visibility external
    //<

    //> @attr listGrid.booleanTrueImage (SCImgURL : null : IRWA)
    // Image to display for a true value in a boolean field. The special value "blank" means
    // that no image will be shown.
    // <P>
    // To turn this off explicitly set +link{listGridField.suppressValueIcon} to true.
    // <P>
    // If this, +link{listGrid.booleanFalseImage} and +link{listGrid.booleanPartialImage}
    // are undefined, this will be set to +link{checkboxItem.checkedImage}.
    // @see listGrid.booleanFalseImage
    // @see listGrid.booleanPartialImage
    // @group imageColumns
    // @visibility external
    //<
    booleanTrueImage:null,
    
    //> @attr listGrid.booleanFalseImage (SCImgURL : null : IRWA)
    // Image to display for a false value in a boolean field. Default <code>null</code> value
    // or the special value "blank" means no image will be displayed.
    // <P>
    // To turn this off explicitly set +link{listGridField.suppressValueIcon} to true
    // <P>
    // If this, +link{listGrid.booleanTrueImage} and +link{listGrid.booleanPartialImage}
    // are undefined, this will be set to +link{checkboxItem.uncheckedImage}.
    // @group imageColumns
    // @see listGrid.booleanTrueImage
    // @see listGrid.booleanPartialImage
    // @visibility external
    //<
    booleanFalseImage:null,
    
    //> @attr listGrid.booleanPartialImage (SCImgURL : null : IRWA)
    // Image to display for a partially true value in a boolean field (typically selection).
    // The special value "blank" means that no image will be shown.
    // <P>
    // To turn this off explicitly set +link{listGridField.suppressValueIcon} to true.
    // <P>
    // If this, +link{listGrid.booleanTrueImage} and +link{listGrid.booleanFalseImage}
    // are undefined, this will be set to +link{checkboxItem.partialSelectedImage}.
    // @see listGrid.booleanTrueImage
    // @see listGrid.booleanFalseImage
    // @group imageColumns
    // @visibility external
    //<
    booleanPartialImage:null,

    //> @attr listGrid.booleanImageWidth (number : 16 : IRWA)
    // Width for the +link{listGrid.booleanTrueImage}, +link{listGrid.booleanFalseImage}
    // and +link{listGrid.booleanPartialImage}.
    // Note: If +link{listGrid.booleanTrueImage} is unset, the +link{checkboxItem.checkedImage}
    // will be used to indicate a true value in a boolean field. In this case this property is
    // ignored in favor of +link{checkboxItem.valueIconWidth}.
    // @group imageColumns
    // @visibility external
    //<
    
    booleanImageWidth:16,
    
    //> @attr listGrid.booleanImageHeight (number : 16 : IRWA)
    // Height for the +link{listGrid.booleanTrueImage}, +link{listGrid.booleanFalseImage}
    // and +link{listGrid.booleanPartialImage}.
    // Note: If +link{listGrid.booleanTrueImage} is unset, the +link{checkboxItem.checkedImage}
    // will be used to indicate a true value in a boolean field. In this case this property is
    // ignored in favor of +link{checkboxItem.valueIconHeight}.
    // @group imageColumns
    // @visibility external
    //<
    booleanImageHeight:16,
    
    
    //> @attr listGrid.mozBodyOutlineColor (string : "white" : IRWA)
    // If we're in Moz Firefox 1.5 or above, and showing a header, what color should the 
    // dotted focus outline show around the body. Must be a color that contrasts with the 
    // header of the ListGrid.
    // @visibility internal
    //<
    mozBodyOutlineColor:"white",
    //> @attr listGrid.mozBodyNoHeaderOutlineColor (string : "red" : IRWA)
    // If we're in Moz Firefox 1.5 or above, and we're not showing a header, what color 
    // should the dotted focus outline show around the body. Must be a color that contrasts 
    // with the header of the ListGrid.
    // @visibility internal
    //<
    mozBodyNoHeaderOutlineColor:"red",


    
// -----------------------------------------------------------------------------------------
// row numbers
//

    //> @attr listGrid.showRowNumbers (boolean : null : IRWA)
    // When set to true, shows an additional field at the beginning of the field-list 
    // (respecting RTL) that displays the current rowNum for each record.
    // @group rowNumberField
    // @visibility external
    //<

    //> @attr listGrid.rowNumberStyle (CSSStyleName : "cellDark" : IRWA)
    // The CSS Style name for the +link{listGrid.rowNumberField}.
    // @group rowNumberField
    // @visibility external
    //<
    rowNumberStyle: "specialCol",

    //> @attr listGrid.rowNumberStart (number : 1 : IRWA)
    // The number to start the row-count from - default value is 1.
    // @group rowNumberField
    // @visibility external
    //<
    rowNumberStart: 1,

    //> @attr listGrid.rowNumberField (AutoChild ListGridField : null : IRWA)
    // An automatically generated field that displays the current row number when 
    // +link{listGrid.showRowNumbers, showRowNumbers} is true.
    // @group rowNumberField
    // @visibility external
    //<
    rowNumberFieldDefaults: {
        name: "_rowNumberField",
        excludeFromState:true,
        canEdit: false,
        canFilter:false,
        canGroupBy: false,
        canSort: false,
        canExport: false,
        canHide: false,
        canReorder: false,
        canDragResize: false,
        // make this special field canHilite false so we don't see it in HiliteEditors by default
        canHilite: false,
        showAlternateStyle: false,
        _isRowNumberField: true,
        showDefaultContextMenu: false,
        keyboardFiresRecordClick: false,
        showGroupSummary:false,
        showGridSummary:false,
        summaryValue: "&nbsp;",
        formatCellValue : function (value, record, rowNum, colNum, grid) {
            if (grid.isGrouped) {
                if (record == null || record._isGroup) return "&nbsp;";
                
                var groupedRowNum = grid.getGroupedRecordIndex(record);
                // skip any records we can't find in the group tree (EG summary rows)
                if (groupedRowNum == -1) return null;
                return (grid.rowNumberStart + groupedRowNum);
            } else {
                return this.rowNumberStart + rowNum;
            }
        },
        autoFreeze: true
    },

    // helper method to get index of the group in which a record exists
    getParentGroupIndex : function (record) {
        // bail if we're not grouped (return group 0)
        if (!this.isGrouped) return 0;

        // find out which group this record is in
        var tree = this.groupTree,
            parentNode = tree.getParent(record),
            rootChildren = tree.getChildren(tree.getParent(parentNode)),
            groupCount=0
        ;

        for (var i=0; i<rootChildren.length; i++) {
            var child = rootChildren[i];
            if (child.groupValue == parentNode.groupValue) {
                groupCount = i;
                break;
            }
        }

        return groupCount;
    },

    // helper method to get the true row-number for a grouped record, whether or not prior groups are closed
    getGroupedRecordIndex : function (record) {
        // bail if we're not grouped
        if (!this.isGrouped) return -1;
        // find the true index of this record in a grouped grid - indexOf doesn't cater for 
        // closed groups
        var tree = this.groupTree,
            
            parentNode = tree == null ? null : tree.getParent(record);
            
        // Checking for parentNode == null allows us to skip group-summary nodes which
        // shouldn't be counted anyway.
        if (parentNode == null) return -1;

        var rootChildren = tree.getChildren(tree.getParent(parentNode)),
            groupCount=0,
            trueIndex=0
        ;

        for (var i=0; i<rootChildren.length; i++) {
            var child = rootChildren[i];

            if (child.groupValue == parentNode.groupValue) {
                var siblings = tree.getChildren(child);
                for (var j=0; j<siblings.length; j++) {
                    if (this.objectsAreEqual(siblings[j], record)) {
                        return trueIndex + j;
                    }
                }
            }
            var prevSiblings = tree.getChildren(child),
                length = prevSiblings.length;
            // Don't count group summary rows - these show up at the end of the
            // group [and we support an arbitrary number of them]
            if (this.showGroupSummary && !this.showGroupSummaryInHeader) {
                for (var ii = length-1; ii >=0; ii--) {
                    if (prevSiblings[ii][this.groupSummaryRecordProperty]) length-=1;
                    else break;
                }
            }
            trueIndex += length;
        }

        return trueIndex;
    },

    // helper method to compare the properties on two objects
    objectsAreEqual : function (object1, object2) {
        for (var key in object1) {
            if (object1[key] != object2[key]) return false;
        }
        return true;
    },

    _rowNumberFieldWidth: 30,
    getRowNumberField : function () {
        var grid = this,
            rnField = {
                // default the width 
                width:this._rowNumberFieldWidth,
                baseStyle: this.rowNumberStyle,
                rowNumberStart: this.rowNumberStart,
                getAutoFreezePosition: function () { return grid.getRowNumberFieldPosition() }
            }
        ;
        isc.addProperties(rnField, this.rowNumberFieldDefaults, this.rowNumberFieldProperties);

        rnField.title = isc.nbsp;

        return rnField;
    },

    getCurrentRowNumberField : function () {
        var fields = this.completeFields || this.fields,
            rnFields = fields.find(this.fieldIdProperty, "_rowNumberField");
        return !rnFields ? null : isc.isAn.Array(rnFields) ? rnFields[0] : rnFields;
    },

    //> @method listGrid.isRowNumberField()
    // Identifies whether the passed-in field is the specially generated
    // +link{listGrid.rowNumberField, rowNumberField} used when +link{showRowNumbers} is
    // true.  Use this method in your custom event handlers to avoid inappropriately
    // performing actions when the rowNumberField is clicked on.
    // 
    // @param field (ListGridField) field to test
    // @return (Boolean) whether the provided field is the rowNumberField
    // @group rowNumberField
    // @visibility external
    //<
    isRowNumberField : function (field) {
        if (!field || !field._isRowNumberField) return false;
        else return true;
    },

    // helper function to get the rowNumber field position
    // Appears at the far left of the grid, to the left of the other special fields which are
    // - group summary title field
    // - expansion component icon field
    // - checkbox selection field
    getRowNumberFieldPosition : function () {
        if (this.fieldSourceGrid) return this.fieldSourceGrid.getRowNumberFieldPosition();
        if (!this.shouldShowRowNumberField()) return -1;
        return 0;
    },

    shouldShowRowNumberField : function () {
        // fieldSourceGrid: for cases like the summaryRow / filterEditor where we
        // share field objects across grids (and don't necessarily percolate settings like 
        // 'showRowNumbers').
        return this.fieldSourceGrid ? this.fieldSourceGrid.shouldShowRowNumberField() 
                                    : (this.showRowNumbers == true);
    },

    //> @attr listGrid.exportRawValues (Boolean : null : IR)
    // Dictates whether the data in this grid should be exported raw by 
    // +link{DataBoundComponent.exportClientData, exportClientData()}.  If set to true,  
    // data will not be processed by field-formatters during exports.
    // Decreases the time taken for large exports.  This property can also be set at the
    // +link{listGridField.exportRawValues, field level}.
    //
    // @visibility external
    //<

// -----------------------------------------------------------------------------------------
// Expando Rows
//

    //> @attr listGrid.canExpandRecords (Boolean : false : IRWA)
    // When set to true, shows an additional field at the beginning of the field-list 
    // (respecting RTL) to allow users to expand and collapse individual records.
    // See +link{listGrid.expandRecord()} and +link{listGrid.expansionMode} for details
    // on record expansion.
    // <P>
    // If expanded records will be variable height,
    // you should switch on +link{listGrid.virtualScrolling, virtualScrolling}.
    // <P>
    // Note that expanded records are not currently supported in conjunction 
    // with +link{listGridField.frozen,frozen fields}.
    
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.canExpandMultipleRecords (Boolean : true : IRWA)
    // When +link{listGrid.canExpandRecords} is true, this property indicates whether multiple
    // records can be expanded simultaneously.  If set to false, expanding a record will
    // automatically collapse any record which is already expanded.  The default value is 
    // <code>true</code>.
    // 
    // @group expansionField
    // @visibility external
    //<
    canExpandMultipleRecords: true,

    //> @attr listGrid.maxExpandedRecords (integer : null : IRWA)
    // When +link{listGrid.canExpandRecords} and +link{listGrid.canExpandMultipleRecords} are
    // both true, this property dictates the number of
    // records which can be expanded simultaneously.  If the expanded record count hits the
    // value of this property, further attempts to expand records will result in a popup
    // warning (see +link{listGrid.maxExpandedRecordsPrompt}) and expansion will be cancelled.
    // <P>
    // The default value is null, meaning there is no limit on the number of expanded records.
    // 
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.maxExpandedRecordsPrompt (String : "This grid is limited to \${count} simultaneously expanded records.  Please collapse some expanded records and retry." : IR)
    // This is a dynamic string - text within <code>\${...}</code> will be evaluated as JS code
    // when the message is displayed. Note that the local variable <code>count</code> will be
    // available and set to this.maxExpandedRecords. The string will be executed in the scope of the
    // ListGrid so <code>this</code> may also be used to determine other information about this grid.
    // <P>
    // Default value returns <P>
    // <code>
    // <i>This grid is limited to <code>[+link{listGrid.maxExpandedRecords}]</code> simultaneously 
    // expanded records.  Please collapse some expanded records and retry.</i>
    // </code>
    // @visibility external
    // @group i18nMessages
    //<
    maxExpandedRecordsPrompt: "This grid is limited to \${count} simultaneously expanded records.  Please collapse some expanded records and retry.",

    //> @type ExpansionMode
    // When +link{ListGrid.canExpandRecords, canExpandRecords} is true, ExpansionMode
    // dictates the type of UI to be displayed in the expanded portion of the row.
    // <P>
    // There are a number of builtin ExpansionModes and you can override 
    // +link{listGrid.getExpansionComponent, getExpansionComponent()} to create your own
    // expansion behaviors.
    //  
    //  @value  "detailField"  Show a single field's value in an +link{class:HtmlFlow}. Field 
    //      to use is +link{listGrid.detailField}.
    //  @value  "details"   Show a +link{class:DetailViewer} displaying those fields from the 
    //      record which are not already displayed in the grid.
    //  @value  "related"    Show a separate +link{class:ListGrid} containing related-records.
    //      See +link{ListGridRecord.detailDS} and +link{ListGrid.recordDetailDSProperty} for 
    //      more information.
    //  @value  "editor"    Show a +link{class:DynamicForm} to edit those fields from the 
    //      record which are not already present in the grid.  If the record is collapsed with
    //      unsaved changes and +link{listGrid.expansionEditorShowSaveDialog} is not set, Edits
    //      will be saved automatically, or stored as +link{group:editing,editValues} if 
    //      +link{listGrid.autoSaveEdits} is false.  Otherwise, a confirmation dialog is
    //      displayed.  Can optionally show a 
    //      +link{listGrid.showExpansionEditorSaveButton,save button} and
    //      +link{listGrid.expansionEditorCollapseOnSave,auto-collapse} when save is pressed. 
    //      If a record fails validation on save and the field in question is not visible in
    //      the grid, the record is automatically expanded and validated to show the errors.
    //  @value  "detailRelated"    Show a +link{class:DetailViewer} displaying those fields
    //      from the record not already displayed in the grid, together with a separate
    //      +link{class:ListGrid} containing related-records.
    // @group expansionField
    // @visibility external
	//<

    //> @attr listGrid.detailField (String : null : IRWA)
    // The field whose contents to show in the expanded portion of a record when 
    // +link{listGrid.canExpandRecords, canExpandRecords} is <code>true</code> and 
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>detailField</code>.
    // @group expansionField
    // @visibility external
    //<
    
    //> @attr listGrid.expansionMode (ExpansionMode : null : IRWA)
    // The +link{ExpansionMode} for records in this grid.
    // Default <code>null</code> value means no expansion.
    // @group expansionField
    // @visibility external
    //<
    
    //> @attr listGrid.expansionField (AutoChild ListGridField : null : IRWA)
    // The field providing the facility to expand and collapse rows.
    // @group expansionField
    // @visibility external
    //<
    expansionFieldDefaults: {
        name: "_expansionField",
        canEdit: false,
        canFilter:false,
        canGroupBy: false,
        canSort: false,
        canExport: false,
        canHide: false,
        canReorder: false,
        canDragResize: false,
        // make this special field canHilite false so we don't see it in HiliteEditors
        canHilite: false,
        _isExpansionField: true,
        showDefaultContextMenu: false,
        keyboardFiresRecordClick: false,
        cellAlign: "center",
        summaryValue: "&nbsp;",
        recordClick: function (viewer, record, recordNum, field, fieldNum, value, rawValue) {
            if (!viewer.canExpandRecords || !field._isExpansionField) return;
            if (!viewer._canExpandRecord(record, recordNum)) return;
            if (viewer.isExpanded(record)) viewer.collapseRecord(record);
            else viewer.expandRecord(record);
        },
        formatCellValue : function (value, record, rowNum, colNum, grid) {
            // This ensures that if we're looking at an edit row for a new record we
            // don't show the expansion icon
            record = grid.getCellRecord(rowNum, colNum);
            if (record == null || record._isGroup) return null;
            if (!grid._canExpandRecord(record, rowNum)) return null;
            return grid.getValueIconHTML(
                grid.isExpanded(record) ? grid.expansionFieldTrueImage : grid.expansionFieldFalseImage, 
                this
            );
        },
        autoFreeze: true
    },

    // Helper method - should this grid show the special expansion field when canExpandRecords is true
    shouldShowExpansionField : function () {
        return this.fieldSourceGrid ? this.fieldSourceGrid.shouldShowExpansionField() 
                                    : this.canExpandRecords;
    },

    //> @attr listGrid.expansionFieldTrueImage (SCImgURL : null :IRWA)
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // determines the image to display in the expansion field for expanded rows.
    // If unset, the +link{listGrid.booleanTrueImage} will be used.
    // @see listGrid.expansionFieldFalseImage
    // @see listGrid.expansionFieldImageWidth
    // @see listGrid.expansionFieldImageHeight
    // @group expansionField
    // @visibility external
    //<
    expansionFieldTrueImage: "[SKINIMG]/ListGrid/group_opened.gif",

    //> @attr listGrid.expansionFieldFalseImage (SCImgURL : null :IRWA)
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // determines the image to display in the expansion field for collapsed rows.
    // If unset, the +link{listGrid.booleanFalseImage} will be used.
    // @see listGrid.expansionFieldTrueImage
    // @see listGrid.expansionFieldImageWidth
    // @see listGrid.expansionFieldImageHeight
    // @group expansionField
    // @visibility external
    //<
    expansionFieldFalseImage: "[SKINIMG]/ListGrid/group_opening.gif",

    //> @attr listGrid.expansionFieldImageWidth (integer : null : IR) 
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // may be set to govern the width of the expansion image displayed to indicate whether a row 
    // is expanded. If unset, the expansionField image will be sized to match the
    // +link{listGrid.booleanImageWidth} for this grid.
    // @group expansionField
    // @visibility external
    //<
    
    //> @attr listGrid.expansionFieldImageHeight (integer : null : IR) 
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // may be set to govern the height of the expansion image displayed to indicate whether a 
    // row is expanded. If unset, the expansionField image will be sized to match the
    // +link{listGrid.booleanImageHeight} for this grid.
    // @group expansionField
    // @visibility external
    //<

    //> @method listGrid.getExpansionField()
    // Returns the specially generated expansion field used when +link{canExpandRecords} is
    // true.
    // <P>
    // Called during +link{setFields()}, this method can be overridden to add advanced dynamic
    // defaults to the expansion field (call Super, modify the default field returned by Super,
    // return the modified field).  Normal customization can be handled by just setting
    // +link{AutoChild} properties, as mentioned under the docs for +link{listGrid.expansionField}.
    //
    // @return (ListGridField)
    // @group expansionField
    // @visibility external
    //<
    // the amount to add to the icon width to get the expansion field width
    expansionFieldExtraWidth: 16,
    getExpansionField : function () {
        var grid = this,
            expField = {
                excludeFromState:true,
                // default the width to the width of the icon plus an arbitrary buffer
                width: this._getExpansionFieldImageWidth() + this.expansionFieldExtraWidth,
                getAutoFreezePosition: function () { return grid.getExpansionFieldPosition() }
            }
        ;

        // if expansionFieldImageWidth/Height are set on this grid, pass them through to the field
        expField.valueIconWidth = this._getExpansionFieldImageWidth();
        expField.valueIconHeight = this._getExpansionFieldImageHeight();

        // combine the properties for the field using the autoChild pattern
        isc.addProperties(expField, this.expansionFieldDefaults, this.expansionFieldProperties);

        expField.title = isc.nbsp;

        return expField;
    },

    getCurrentExpansionField : function () {
        var fields = this.completeFields || this.fields,
            expFields = fields.find(this.fieldIdProperty, "_expansionField");         
        return !expFields ? null : isc.isAn.Array(expFields) ? expFields[0] : expFields;
    },

    _getExpansionFieldImageWidth : function () {
        return this.expansionFieldImageWidth || this.booleanImageWidth ||
                (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null);
    },
    _getExpansionFieldImageHeight : function () {
        return this.expansionFieldImageHeight || this.booleanImageHeight ||
                (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconHeight") : null);
    },

    //> @method listGrid.isExpansionField()
    // Identifies whether the passed-in field is the specially generated
    // +link{listGrid.expansionField,expansionField} used when +link{canExpandRecords} is
    // true.  Use this method in your custom event handlers to avoid inappropriately
    // performing actions when the expansionField is clicked on.
    // 
    // @param field (ListGridField) field to test
    // @return (Boolean) whether the provided field is the expansion field
    // @group expansionField
    // @visibility external
    //<
    isExpansionField : function (field) {
        if (!field || !field._isExpansionField) return false;
        else return true;
    },

    // helper function to get the expansion field position
    // This is one of the fields that appears "stuck" to the left of the grid - these are
    // - row number field
    // - group summary title field
    // - expansion field
    // - checkbox selection field
    getExpansionFieldPosition : function () {
        if (this.fieldSourceGrid != null) return this.fieldSourceGrid.getExpansionFieldPosition();
        
        if (!this.shouldShowExpansionField()) return -1;
        
        var pos = 0;
        if (this.shouldShowRowNumberField()) pos++;
        if (this.shouldShowCheckboxField()) pos++;
        return pos;
    },

    _canExpandRecord:function (record,rowNum) {
        if (record == null) record = this.getRecord(rowNum);
        if (record == null) return false;
        return this.canExpandRecord(record,rowNum);
    },
    //> @method listGrid.canExpandRecord()
    // Indicates whether a given record or rowNum can be expanded.  The default implementation
    // checks the value of +link{listGrid.canExpandRecords} and 
    // <code>record[+link{listGrid.canExpandRecordProperty}]</code>.
    // <P>
    // Override this method for more specific control over individual record expansion.
    // <P>
    // <b>Note:</b> Rows with no underlying record in the data array - for example newly 
    // added edit rows that have not yet been saved - cannot be expanded.
    // 
    // @param record (ListGridRecord) record to work with
    // @param rowNum (Number) rowNum of the record to work with
    // @return (boolean) true if the record can be expanded
    // @group expansionField
    // @visibility external
    //<
    canExpandRecord : function (record, rowNum) {
        return record[this.canExpandRecordProperty] == false ? false : 
            true && (this.canExpandRecords != false);
    },
    
    //> @method listGrid.setCanExpandRecords()
    // Setter for +link{listGrid.canExpandRecords}
    // @param canExpand (boolean) new value for listGrid.canExpandRecords.
    // @visibility external
    //<
    setCanExpandRecords : function (canExpand) {
        if (this.canExpandRecords == canExpand) return;
        if (!canExpand) {
            var data = this.data;
            if (data) {
                var expandedRows = this._getExpandedRows();
                if (expandedRows != null) {
                    for (var i = 0; i < expandedRows.length; i++) {
                        this.collapseRecord(expandedRows[i]);
                    }
                }
            }
        }
        this.canExpandRecords = canExpand;
        this.refreshFields();
    },

    //> @method listGrid.expandRecord()
    // Expands a given +link{ListGridRecord, record} by creating a subcomponent and inserting it 
    // in to the record's grid-row.  A number of built-in +link{ExpansionMode, expansionModes} 
    // are supported by the default implementation of 
    // +link{listGrid.getExpansionComponent, getExpansionComponent()} and you can override
    // that method to provide your own expansion behavior.
    // <P>
    // Once a record has been expanded, the currently visible expansion component may be
    // retrieved via +link{getCurrentExpansionComponent()}.
    // 
    // @param record (ListGridRecord) record to expand
    // @group expansionField
    // @visibility external
    //<
    _expandedRecordCount: 0,
    expandRecord : function (record) {
        if (!this.body) {
            // called before draw - add the record to an internal array, this method
            // will be called again with this record after the grid has drawn
            if (!this._recordsToExpand) this._recordsToExpand = [];
            this._recordsToExpand.add(record);
            return;
        }
        
        var component,
            rowNum = this.getRecordIndex(record)
        ;

        if (!this.isExpanded(record)) {

            // notification method / cancellation point
            
            if (this.onExpandRecord != null && !this.onExpandRecord(record)) return;

            if (!this.canExpandMultipleRecords) {
                // can only expand one record - if one's expanded already, collapse it now
                if (this._currentExpandedRecord)
                    this.collapseRecord(this._currentExpandedRecord);
                this._currentExpandedRecord = record;
            } else if (this.maxExpandedRecords != null) {
                // limited number of expanded records allowed - if we've hit that number, show the
                // maxExpandedRecordsPrompt and return
                if (this._expandedRecordCount >= this.maxExpandedRecords) {
                    var message = this.maxExpandedRecordsPrompt.evalDynamicString(this, { 
                        count: this.maxExpandedRecords 
                    });
                    isc.say(message);
                    return;
                }
            }
            
            // create an appropriate subcomponent and bind it
            component = this.getExpansionComponent(record);
            var layout = this.createAutoChild("expansionLayout", 
                {
                    layoutLeftMargin: this.embeddedComponentIndent,
                    members: [ component ]
                }
            );
            layout.isExpansionComponent = true;
            layout.removeOnHideField = true;
            this.addEmbeddedComponent(layout, record, this.data.indexOf(record));
            this._setExpanded(record, true);
            this._setExpansionComponent(record, true);
            this._expandedRecordCount++;
            if (!this.canExpandMultipleRecords) this._currentExpandedRecord = record;
        }

        this.delayCall("markForRedraw", ["Expanded Record"]);
        this.refreshRow(rowNum);
    },

    //> @method listGrid.collapseRecord()
    // Collapses a given +link{ListGridRecord, record} which has been previously expanded using
    // +link{listGrid.expandRecord}.
    // 
    // @param record (ListGridRecord) record to collapse
    // @group expansionField
    // @visibility external
    //<
    collapseRecord : function (record) {
        var component = 
                (record && this._hasEmbeddedComponents(record)) ? 
                        this._getEmbeddedComponents(record).find("isExpansionComponent", true) : null
        ;

        if (isc.isA.Layout(component)) {
            var member = component.getMember(0);
            if (member) {
                if (!isc.isA.DynamicForm(member)) member = member.formMember;
                if (isc.isA.DynamicForm(member) && member.valuesHaveChanged()) {
                    this.saveAndCollapseRecord(member, component, record, true, true);
                    return;
                }
            }
        }

        this._collapseRecord(record, component);
    },

    saveAndCollapseRecord : function (member, component, record, shouldShowSaveDialog, shouldCollapse) {
        if (this.autoSaveEdits == true) {
            var _this = this;

            if (this.expansionEditorShowSaveDialog && shouldShowSaveDialog) {
                isc.confirm(this.expansionEditorSaveDialogPrompt,
                    function (yes) {
                        if (yes) {
                            _this.saveExpansionDetail(member, component, record, shouldCollapse);
                        } else {
                            if (shouldCollapse) _this._collapseRecord(record, component);
                        }
                    }
                );
            } else {
                this.saveExpansionDetail(member, component, record, shouldCollapse);
            }
            return;
        } else {
            var _values = member.getChangedValues(),
                rowNum = this.getRecordIndex(record),
                _this = this
            ;

            if (this.expansionEditorShowSaveDialog && shouldShowSaveDialog) {
                isc.confirm("You have unsaved changes - do you want to save them now?",
                    function (yes) {
                        if (yes) {
                            _this._saveExpansionEditorValues(rowNum, _values);
                        }
                        if (shouldCollapse) _this._collapseRecord(record, component);
                    }
                );
                return;
            } else {
                _this._saveExpansionEditorValues(rowNum, _values);
                if (shouldCollapse) this._collapseRecord(record, component);
                return;
            }
        }
    },

    
    _saveExpansionEditorValues : function (rowNum, values) {
        for (var key in values) {
            this.setEditValue(rowNum, key, values[key]);
        }
    },

    saveExpansionDetail : function (member, component, record, shouldCollapse) {
        var _this = this;
        member.saveData(
            function (dsResponse, data, dsRequest) {
                if (data) {
                    record = data;
                    if (shouldCollapse) _this._collapseRecord(record, component);
                    else member.editRecord(record);
                }
            }, { showPrompt: true, promptStyle: "cursor" }
        );
    },
    
    _collapseRecord : function (record, component) {
        component = component || 
            (record && this._hasEmbeddedComponents(record) ? 
                    this._getEmbeddedComponents(record).find("isExpansionComponent", true) : null);

        if (this.isExpanded(record)) {
            // notification method / cancellation point
            
            if (this.onCollapseRecord != null && !this.onCollapseRecord(record)) return;

            if (this._currentExpandedRecord && this._currentExpandedRecord == record)
                delete this._currentExpandedRecord;

            this.removeEmbeddedComponent(record, component ? component : this.frozenFields ? this.frozenFields.length : 0);
            this._expandedRecordCount--;
        }
        this._setExpanded(record, false);
		
        //this._remapEmbeddedComponents();
        this.markForRedraw();
    },

    

    _$expandedPrefix:"_expanded_",
 
    //> @method listGrid.isExpanded()
    // Whether a given +link{ListGridRecord, record} is expanded or collapsed.
    //
    // @param record (ListGridRecord) record in question
    // @group expansionField
    // @return  (Boolean)           true if the node is expanded
    // @visibility external 
    //<
    isExpanded : function (record) {
        return record[this._$expandedPrefix + this.ID];
    },

    _setExpanded : function (record, value) {
        record[this._$expandedPrefix + this.ID] = value;
    },
    _getExpandedRows : function () {
        return this.data.findAll(this._$isExpandedPrefix + this.ID, true);
    },

    _$hasExpansionComponentPrefix:"_hasExpansionComponent_",
    _hasExpansionComponent : function (record) {
        return record[this._$hasExpansionComponentPrefix + this.ID];
    },
    _setExpansionComponent : function (record, value) {
        record[this._$hasExpansionComponentPrefix + this.ID] = value;
    },

    
    _$embeddedComponentsPrefix:"_embeddedComponents_",

    _hasEmbeddedComponents : function (record) {
        var ids = record[this._$embeddedComponentsPrefix + this.ID];
        return (ids != null && ids.length > 0);
    },
    _getEmbeddedComponents : function (record) {
        // Convert array of IDs into an array of component references
        var ids = record[this._$embeddedComponentsPrefix + this.ID],
            components = []
        ;
        if (!ids) return null;
        for (var i = 0; i < ids.length; i++) {
            components[i] = isc.Canvas.getById(ids[i]);
        }   
        return components;
    },
    _setEmbeddedComponents : function (record, value) {
        record[this._$embeddedComponentsPrefix + this.ID] = value;
    },
    
    _addEmbeddedComponent : function (record, component) {
        if(!record[this._$embeddedComponentsPrefix + this.ID]) {
            record[this._$embeddedComponentsPrefix + this.ID] = [];
        }
        record[this._$embeddedComponentsPrefix + this.ID].add(component.getID());
    },
    _removeEmbeddedComponent : function (record, component) {
    
        var ids = record[this._$embeddedComponentsPrefix + this.ID];
        if (ids == null) return;
        if (ids.length == 0) {
            record[this._$embeddedComponentsPrefix + this.ID] = null;
            return;
        }
        ids.remove(component.getID());
        if (ids.length == 0) {
            record[this._$embeddedComponentsPrefix + this.ID] = null;
        }
    },
    
    _deleteEmbeddedComponents : function (record, value) {
        delete record[this._$embeddedComponentsPrefix + this.ID];
    },

    
    _$recordComponentsPrefix:"_recordComponents_",
    _hasRecordComponents : function (record) {
        return (record[this._$recordComponentsPrefix + this.ID] != null);
    },
    _getRecordComponents : function (record) {
        if (!record) return null;

        // Convert array of IDs into an array of component references
        var ids = record[this._$recordComponentsPrefix + this.ID],
            components = {}
        ;
        if (ids) {
            for (var key in ids) {
                if (ids[key].isNullMarker) {
                    components[key] = ids[key];
                } else {
                    components[key] = isc.Canvas.getById(ids[key]);
                }
            }   
        }
        return components;
    },
    _addRecordComponent : function (record, fieldName, component) {
        if(!record[this._$recordComponentsPrefix + this.ID]) {
            record[this._$recordComponentsPrefix + this.ID] = {};
        }
        if (component.isNullMarker) {
            record[this._$recordComponentsPrefix + this.ID][fieldName] = component;
        } else {
            record[this._$recordComponentsPrefix + this.ID][fieldName] = component.getID();
        }
    },
    _deleteRecordComponent : function (record, fieldName) {
        var ids = record[this._$recordComponentsPrefix + this.ID];
        if (ids == null) return;
        if (isc.isAn.emptyObject(ids)) {
            record[this._$recordComponentsPrefix + this.ID] = null;
            return;
        }
        delete ids[fieldName];
        if (isc.isAn.emptyObject(ids)) {
            record[this._$recordComponentsPrefix + this.ID] = null;
        }
    },

    //> @attr listGrid.expansionDetailField (MultiAutoChild HTMLFlow : null : RA)
    // Automatically generated +link{class:HTMLFlow} for displaying the contents of 
    // +link{listGrid.detailField, a specified field} in a record's expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>detailField</code>.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionDetailFieldProperties</code> and 
    // <code>listGrid.expansionDetailFieldDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionDetailFieldDefaults: {
        _constructor: isc.HTMLFlow,
        autoDraw: false,
        width: "100%",
        height: "100%"
    },
    //> @attr listGrid.expansionDetails (MultiAutoChild DetailViewer : null : RA)
    // Automatically generated +link{class:DetailViewer} for displaying the details of a record
    // in its expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>details</code>.  Note that 
    // only those fields
    // which do not already appear in the grid are displayed in the expanded section.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionDetailsProperties</code> and 
    // <code>listGrid.expansionDetailsDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionDetailsDefaults: {
        _constructor: isc.DetailViewer,
        autoDraw: false,
        width: "100%"
    },
    //> @attr listGrid.expansionRelated (MultiAutoChild ListGrid : null : RA)
    // Automatically generated +link{class:ListGrid} for displaying data related to a record
    // in its expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>related</code>.  
    // The +link{class:DataSource} containing the related data is provided by 
    // +link{listGrid.getRelatedDataSource, getRelatedDataSource()} which, by default,
    // returns the DataSource referred to in +link{listGridRecord.detailDS}.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionRelatedProperties</code> and 
    // <code>listGrid.expansionRelatedDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionRelatedDefaults: {
        _constructor: isc.ListGrid,
        autoDraw: false,
        width: "100%",
        autoFitData: "vertical",
        autoFitMaxRecords: 4
    },

    //> @attr listGrid.expansionEditor (MultiAutoChild DynamicForm : null : RA)
    // Automatically generated +link{class:DynamicForm} for editing the details of a record
    // in its expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>editor</code>.  Note that only 
    // those fields which do not already appear in the grid will appear in the expanded section.
    // <P>
    // According to the value of +link{expansionEditorShowSaveButton}, a save button is shown
    // beneath the editor.  You can save the values in the editor by clicking this button
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionEditorProperties</code> and 
    // <code>listGrid.expansionEditorDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionEditorDefaults: {
        _constructor: "DynamicForm",
        autoDraw: false,
        numCols: 4,
        colWidths: ["*", "*", "*", "*"],
        width: "100%",
        saveOperationType: "update"
    },

    //> @attr listGrid.expansionEditorShowSaveDialog (boolean : null : IR)
    // When +link{listGrid.canExpandRecords, canExpandRecords} is true and
    // +link{listGrid.expansionMode, expansionMode} is <i>editor</i>, whether a dialog should be
    // displayed when an expanded row is collapsed while it's nested editor has changed values.
    //
    // @group expansionField
    // @visibility external
    //<
    //expansionEditorShowSaveDialog: null,

    //> @attr listGrid.expansionEditorSaveDialogPrompt (string : "You have unsaved changes - do you want to save them now?" : IR)
    // When +link{listGrid.canExpandRecords, canExpandRecords} is true and
    // +link{listGrid.expansionMode, expansionMode} is <i>editor</i>, the prompt to display
    // in a dialog when an expanded row is collapsed while it's nested editor has changed values.
    //
    // @group expansionField,i18nMessages
    // @visibility external
    //<
    expansionEditorSaveDialogPrompt: "You have unsaved changes - do you want to save them now?",

    //> @attr listGrid.expansionEditorCollapseOnSave (Boolean : true : RW)
    // When +link{expansionMode} is <i>editor</i>, should the row be collapsed following a 
    // save initiated by the expansion-component's +link{expansionEditorSaveButton, save button}.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionEditorCollapseOnSave: true,

    //> @attr listGrid.showExpansionEditorSaveButton (Boolean : true : RW)
    // When +link{expansionMode} is <i>editor</i>, should a Save button be shown below the
    // the expanded editor?
    // <P>
    // Note that if an expanded-row containing an editor is collapsed while changes are 
    // oustanding, changes will be either be automatically updated to the grid, or will first
    // show a confirmation dialog, according to the value of 
    // +link{expansionEditorShowSaveDialog}.
    // 
    // @group expansionField
    // @visibility external
    //<
    showExpansionEditorSaveButton: true,

    //> @attr listGrid.expansionEditorSaveButton (MultiAutoChild IButton : null : RA)
    // Automatically generated +link{class:IButton} for saving the values in the expanded
    // portion of a ListGrid row.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionEditorSaveButtonProperties</code> and 
    // <code>listGrid.expansionEditorSaveButtonDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionEditorSaveButtonDefaults: {
        _constructor: "IButton",
        autoFit: true,
        layoutAlign: "center",
        click : function () {
            if (!this.grid || !this.form) return;
            var grid = this.grid,
                component = 
                (this.record && grid._hasEmbeddedComponents(this.record)) ? 
                    grid._getEmbeddedComponents(this.record).find("isExpansionComponent", true) : 
                    null
            ;

            this.grid.saveAndCollapseRecord(this.form, component, this.record, false, 
                this.grid.expansionEditorCollapseOnSave
            );
        }
    },

    //> @attr listGrid.expansionEditorSaveButtonTitle (String : "Save" : RWA)
    // The title for the +link{expansionEditorSaveButton}.
    //
    // @group expansionField,i18nMessages
    // @visibility external
    //<
    expansionEditorSaveButtonTitle: "Save",

    //> @attr listGrid.expansionDetailRelated (MultiAutoChild HLayout : null : RA)
    // Automatically generated +link{class:HLayout} appearing in a record's expanded section
    // when +link{type:ExpansionMode, listGrid.expansionMode} is <code>detailRelated</code>.  
    // This component contains two other autoChild components,
    // a +link{class:DetailViewer} for viewing fields from the record which are not already 
    // present in the grid and a separate embedded +link{class:ListGrid} for displaying other 
    // data related to this record via record.detailDS.  See +link{listGrid.expansionDetails} 
    // and +link{listGrid.expansionRelated} for more information.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionDetailRelatedProperties</code> and 
    // <code>listGrid.expansionDetailRelatedDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionDetailRelatedDefaults: {
        _constructor: isc.HLayout,
        autoDraw: false,
        width: "100%",
        height: "100%"
    },

    //> @attr listGrid.expansionLayout (MultiAutoChild VLayout : null : RA)
    // Automatically generated +link{class:VLayout} which fills a record's expanded section
    // and contains other builtin +link{ExpansionMode, expansion-components}.  You can also
    // override +link{listGrid.getExpansionComponent, getExpansionComponent()} to provide
    // components of your own specification.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionLayoutProperties</code> and 
    // <code>listGrid.expansionLayoutDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record),
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionLayoutDefaults: {
        _constructor: isc.VLayout,
        autoDraw: false,
        width: "100%",
        height: 10,
        overflow: "visible"
    },
    
    //> @method listGrid.getCurrentExpansionComponent()
    // Returns the expansion component derived from +link{listGrid.getExpansionComponent()}
    // currently visible in some record, or null if the specified record is not showing
    // an expansion component.
    //
    // @param record (Integer Or ListGridRecord) rowNum or record to get the expansionComponent for
    // @return (Canvas) the currently visible expansion component for the expanded row.
    // @group expansionField
    // @visibility external
    //<
    getCurrentExpansionComponent : function (record) {
        if (isc.isA.Number(record)) record = this.getRecord(record);
        
        if (!this._hasExpansionComponent(record)) return null;
        
        // we actually hang 'isExpansionComponent' onto the layout containing the generated
        // expansion component so return its (only) member.
        var component = this._hasEmbeddedComponents(record) ? 
                    this._getEmbeddedComponents(record).find("isExpansionComponent", true) : null;
        if (component) {
            return component.members[0];
        }
        return null;
    },

    //> @method listGrid.getExpansionComponent()
    // Gets the embedded-component to show as a given record's expansionComponent.  This 
    // component is then housed in +link{listGrid.expansionLayout, a VLayout} and embedded 
    // into a record's row.
    // <P>
    // By default, this method returns one of a set of builtin components, according to the 
    // value of +link{type:ExpansionMode, listGrid.expansionMode}.  You can override this method 
    // to return any component you wish to provide as an expansionComponent.
    // <P>
    // As long as the record is expanded, this component may be retrieved via a call to
    // +link{getCurrentExpansionComponent()}.
    // 
    // @param record (ListGridRecord) record to get the expansionComponent for
    // @return (Canvas | Canvas Properties) the component to embed
    // @group expansionField
    // @visibility external
    //<
    // override getExpansionComponent() to do exactly that, returning some component other than
    // those provided by default
    getExpansionComponent : function (record) {
        return this._getStockEmbeddedComponent(record, true, false, this.getRecordIndex(record), 0);
    },

    //> @attr listGridRecord.backgroundComponent (Canvas : null : IR)
    // Has no effect unless +link{listGrid.showBackgroundComponent} is <code>true</code>.
	// <P>
    // Canvas created and embedded in the body behind a given record.   When set, either as
    // a Canvas or Canvas Properties, will be constructed if necessary, combined with the 
    // autoChild properties specified for +link{listGrid.backgroundComponent} and displayed 
    // behind this record in the page's z-order, meaning 
    // it will only be visible if the cell styling is transparent.
    // @group rowEffects
    // @visibility external
    //<

// Expando Rows - Record implementation

    //> @attr listGrid.detailDS (String : null : IRWA)
    // If +link{canExpandRecords} is true and +link{type:ExpansionMode,listGrid.expansionMode}
    // is <code>"related"</code>, this property specifies the dataSource for the 
    // related records grid to be shown embedded in expanded records.
    // <P>
    // This property may also be specified on a per-record basis - see 
    // +link{recordDetailDSProperty}
    // @visibility external
    //<

    //> @attr listGrid.recordDetailDSProperty (String : "detailDS" : IRWA)
    // The name of the ListGridRecord property that specifies the DataSource to use when 
    // +link{type:ExpansionMode, listGrid.expansionMode} is "related".  The default is 
    // +link{ListGridRecord.detailDS}. Note that you can set the +link{detailDS} at the
    // grid level instead if the same dataSource is to be used for all records.
    // 
    // @visibility external
    //<
    recordDetailDSProperty: "detailDS",

    //> @attr listGridRecord.detailDS (DataSource : null : IRWA)
    // The default value of +link{listGrid.recordDetailDSProperty}.
    // 
    // @visibility external
    //<
    
    //>	@method	listGrid.getRelatedDataSource() 
    // Returns the +link{class:DataSource} containing data related to the passed record.  Used
    // when +link{ListGrid.canExpandRecords} is true and +link{ExpansionMode} is "related". The
    // default implementation returns the DataSource specified in
    // +link{listGridRecord.detailDS} if set, otherwise +link{listGrid.detailDS}.
    // 
    // @param   record  (ListGridRecord)    The record to get the Related dataSource for.
    // @return (DataSource) The related DataSource for the "record" param
    // 
    // @visibility external
    //<
    getRelatedDataSource : function (record) {
        return isc.DS.getDataSource(record[this.recordDetailDSProperty]) ||
                    isc.DS.get(this.detailDS);
    }

    //> @attr listGrid.childExpansionMode (ExpansionMode : null : IRWA)
    // For +link{ExpansionMode, expansionModes} that show another grid or tree, what the 
    // child's expansionMode should be.
    // <P>Default value <code>null</code> means no further expansion.
    //
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.expansionCanEdit (Boolean : null : IRWA)
    // For +link{ExpansionMode, expansionModes} that show another grid or tree, is that 
    // component editable?
    // <P>The default value for this property is <code>false</code>.
    //
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.filterLocalData (boolean : null : IRW)
    // If this grid has its +link{listGrid.setData(),data property} set to a static array
    // of records, should +link{listGrid.filterData()} filter this array?
    // If this property is set to true, the local data array will be filtered according to
    // the specified criteria, and the results displayed. If false, a normal databound
    // fetch will occur, retriving records that match the specified criteria from this
    // component's +link{listGrid.dataSource}.
    // <P>
    // If this property is not explicitly set, default behavior will filter against the
    // dataSource unless the grid has a specified +link{listGrid.dataPath}, in which case
    // filtering will occur locally.
    // <P>
    // Note that a specified +link{listGrid.dataSource} is required for filtering to occur
    // even when filtering a local data array.
    // @visibility external
    //<
});



isc.ListGrid.addMethods({

// warnOnReusedFields -- will log a warning at init time if this.fields is set to an array of 
// field objects which are already displayed as fields in another grid.
// Internal - we only disable this for a couple of special cases where we really do want to share
// field objects across grids, like the record editor
warnOnReusedFields:true,

//> @method ListGrid.initWidget() (A)
// Initialize the canvas and call listGrid.setData().
// @param [all arguments] (object) objects with properties to override from default
//<
initWidget : function () {
    
    
    this.Super("initWidget", arguments);
    
    // call the setter on 'showRecordComponents()' - this disables the
    // drawAllMaxCells logic for the grid.
    if (this.showRecordComponents) {
        delete this.showRecordComponents;
        this.setShowRecordComponents(true);
    }
    
    // we want to avoid a fields array object getting re-used across multiple ListGrids.
    // This can happen if the developer does something like assigning a fields object to the
    // instance prototype.
    // To handle this we have another attribute 'defaultFields' which may be set on the instance
    // prototype and differs only from initializing with a fields block in that array will be
    // shallow cloned and assigned to this.fields rather than sharing the same object across
    // ListGrid instances
    
    if (this.fields == null && this.defaultFields != null) {
        this.fields = isc.shallowClone(this.defaultFields);
        //this.logWarn("fields picked up from this.defaultFields -- now set to:" + this.fields); 
    } else if (this.warnOnReusedFields && this.fields != null) {
        if (this.fields._initializedFieldsArray) {
            this.logWarn("ListGrid initialized with this.fields attribute set to an array " +
                "which is already being displayed in another ListGrid instance. To reuse " +
                "standard field configuration across multiple ListGrids, use " +
                "listGrid.defaultFields rather than assigning directly to listGrid.fields.");

        // Also catch the case where the developer re-uses individual fields within separate
        // arrays across grids
        } else {

            var dupedFields;
            if (isc.isAn.Array(this.fields)) {
                dupedFields = this.fields.findIndex("_initializedFieldObject", true) != -1;
            } else if (isc.isAn.Object(this.fields)) {
                for (var fieldName in this.fields) {
                    var field = this.fields[fieldName];
                    if (field && field._initializedFieldObject) {
                        dupedFields = true;
                        break;
                    }
                }
            }
            if (dupedFields) {
                 this.logWarn("ListGrid initialized with this.fields attribute set to an array " +
                "containing fields which are already being displayed in another ListGrid " +
                "instance. To reuse standard field configuration across multiple ListGrids, use " +
                "listGrid.defaultFields rather than assigning directly to listGrid.fields.");
            }
        }

        // Set a flag on the array and the individual field objects - this allows us catch the case
        // where a developer reuses a standard fields array across multiple grids rather than using
        // the defaultFields attribute
        this.fields._initializedFieldsArray = true;
        if (isc.isAn.Array(this.fields)) {
            this.fields.setProperty("_initializedFieldObject", true);
        } else if (isc.isAn.Object(this.fields)) {
            for (var fieldName in this.fields) {
                var field = this.fields[fieldName];
                if (field) {
                    field._initializedFieldObject = true;
                }
            }
        }
    }
    
    if (this.canEditNew) this.listEndEditAction = this.rowEndEditAction = "next";

    if (this.alwaysShowEditors) {
        this.editByCell = false;
        
        // disable selection
        
        this.selectionType = "none";
        this.selectOnEdit = false;
        
        if (this.canGroup != false) {
            this.logInfo("grouping functionality is not supported when alwaysShowEditors is true." +
                    " Explicitly disabling this.canGroup", "inactiveEditorHTML");
            this.canGroup = false;
        }
        if (this.modalEditing) {
            this.logInfo("modalEditing is not supported when alwaysShowEditors is true." + 
                " Explicitly setting this.modalEditing to fales", "inactiveEditorHTML");
            this.modalEditing = false;
        }
        
        // enforce editEvent:"click" - this means that if the user clicks on a cell which
        // isn't showing an inactive editor (for example where showEditorPlaceholder is true),
        // we still start editing that row/cell
        this.editEvent = "click";
        
        // If canEdit is unset, and we have no fields explicitly marked as canEdit:true,
        // flip the default to true.
        // This gives us the most intuitive behavior - if the developer specifies per-field
        // editablility we'll respect it, otherwise we'll default to canEdit true
        if (!this.isEditable()) {
            this.logInfo("alwaysShowEditors has been set for this grid but canEdit is unset and " +
              "no fields are explicitly marked as editable. Defaulting this.canEdit to true. " +
              "Note that to avoid this override developers can explicitly specify canEdit " +
              "at the grid or field level", "inactiveEditorHTML");
            this.canEdit = true;
        }
        
    }

    // if we have variable record heights and virtualScrolling is unset, switch it on
    
    if (this.canExpandRecords ||
        (this.fixedRecordHeights == false && this.virtualScrolling == null)) 
    {
        // the _specifiedFixedRecordHeights flag is used by shouldShowAllColumns - we know
        // that even though a row may exceed its cell height this isn't due to any cell's content
        // in the row, so we don't need to render out every column to get correctly sized rows.
        if (this.fixedRecordHeights) this._specifiedFixedRecordHeights = this.fixedRecordHeights;
        this.fixedRecordHeights = false;
        this.virtualScrolling = true;
    }

    // disable canAddFormulaField if the required component isn't present
    if (this.canAddFormulaFields && isc.FormulaBuilder == null) {
        this.logInfo("Required modules for adding formula fields not present - setting " +
                    "canAddFormulaFields to false.");
        this.canAddFormulaFields = false;
    }
    if (this.canAddSummaryFields && isc.SummaryBuilder == null) {
        this.logInfo("Required modules for adding summary fields not present - setting " +
                    "canAddSummaryFields to false.");
        this.canAddSummaryFields = false;
    }
    
    // default to canSelectCells mode if we're using rowSpan-sensitive styling
    if (this.useRowSpanStyling && this.canSelectCells == null) this.canSelectCells = true;

    // default cell-level roll overs to the value of canSelectCells
    if (this.useCellRollOvers == null) this.useCellRollOvers = this.canSelectCells;

	// force loading rows to contain at least &nbsp; otherwise row height may be reported as less
	// than the actual height.
    if (this.loadingMessage == null || this.loadingMessage == isc.emptyString) 
        this.loadingMessage = "&nbsp;";
    
    // default our overflow to "visible" if autoFitData is set
    if (this.autoFitData != null) {
        this._specifiedOverflow = this.overflow;
        this.setOverflow("visible");
    }

    // default our groupStartOpen to "all" if canCollapseGroup is false
    if (this.canCollapseGroup == false) this.groupStartOpen = "all";

    // store off the initial sortDirection - we'll revert to this when unsorting
    this._baseSortDirection = this.sortDirection ? "ascending" : "descending";

	// initialize the data object, setting it to an empty array if it hasn't been defined
	this.setData(this.data ? null : this.getDefaultData());

    // set up selectionType dynamic default
    this.setSelectionAppearance(this.selectionAppearance, true);

    this._setUpDragProperties(); 
},

getDefaultData : function () { return []; },

_storeDragProperties : function () {
	if (!this._initialDragProperties) {
		var iDP = this._initialDragProperties = {};
		iDP.canDrag = this.canDrag;
		iDP.canDrop = this.canDrop;
        iDP.canAcceptDrop = this.canAcceptDrop;
	}
},
_restoreDragProperties : function () {
    var iDP = this._initialDragProperties;
	if (iDP) {
		this.canDrag = iDP.canDrag;
		this.canDrop = iDP.canDrop;
        this.canAcceptDrop = iDP.canAcceptDrop;
	}
},
_setUpDragProperties : function () {
	if (!this._initialDragProperties) {
        // store the original dragProperties 
        this._storeDragProperties();
    }
	// set up our specific drag-and-drop properties

	this.canDrag = !this.canDragSelectText &&
                    (this.canDrag || this.canDragRecordsOut || this.canReorderRecords || 
                    this.canDragSelect);
	this.canDrop = (this.canDrop || this.canDragRecordsOut || this.canReorderRecords);
	this.canAcceptDrop = (this.canAcceptDrop || this.canAcceptDroppedRecords || 
                    this.canReorderRecords);

},

//> @method listGrid.setCanReorderRecords()
// Setter for the +link{listGrid.canReorderRecords} attribute.
// @param canReorderRecords (boolean) new value for <code>this.canReorderRecords</code>
// @visibility external
//<
setCanReorderRecords : function (canReorderRecords) {
    if (canReorderRecords != this.canReorderRecords) {
        this.canReorderRecords = canReorderRecords;
        this._restoreDragProperties();
        this._setUpDragProperties(); 
        this.redraw();
    }
},

getEmptyMessage : function () { 
    if (isc.ResultSet && isc.isA.ResultSet(this.data) && !this.data.lengthIsKnown()) {
        if (isc.Offline && isc.Offline.isOffline()) {
            return this.offlineMessage;
        }
        return this.loadingDataMessage == null ? "&nbsp;" : 
            this.loadingDataMessage.evalDynamicString(this, {
                loadingImage: this.imgHTML(isc.Canvas.loadingImageSrc, 
                                           isc.Canvas.loadingImageSize, 
                                           isc.Canvas.loadingImageSize)
            });
    }
    if (this.isOffline()) {
        return this.offlineMessage;
    }
    return this.emptyMessage == null ? "&nbsp;" : this.emptyMessage.evalDynamicString(this, {
        loadingImage: this.imgHTML(isc.Canvas.loadingImageSrc, 
                                   isc.Canvas.loadingImageSize, 
                                   isc.Canvas.loadingImageSize)
    });
},

isEmpty : function () {
    if (!this.data) return true;
    
    // treat having no fields as being empty so we don't attempt to write out and manipulate
    // an empty table 
    if (!this.fields || this.fields.length == 0) return true;
    
    
    if (isc.ResultSet && isc.isA.ResultSet(this.data)) {

        if (this.data.isPaged()) {
            if (!this.data.isEmpty()) return false;
            
            
            var editRows = this.getAllEditRows();
            if (editRows && editRows.length > 0) {
                for (var i = 0; i < editRows.length; i++) {
                    if (editRows[i] >= 0) return false;
                }
            }
            return true;
        } else {
            // If our length is not known we must be in the process of loading, so return the
            // loading message.
            return (!this.data.lengthIsKnown() || this.getTotalRows() == 0);
        }
    } else {
        return (this.getTotalRows() == 0);
    }
},

//>	@attr listGrid.preserveEditsOnSetData (boolean : null : IRWA)
// By default any edit values in an editable ListGrid are dropped when 'setData()' is called, 
// as previous edits are almost certainly obsoleted by the new data-set.
// This flag allows the developer to suppress this default behavior.
// @visibility internal
// @group data
//<
// Leave this internal for now - no known use cases for it, but seems like something that
// could come up.
//preserveEditsOnSetData : null,

//>	@method	listGrid.setData()    ([])
// Initialize the data object with the given array. Observes methods of the data object
// so that when the data changes, the listGrid will redraw automatically.
//      @visibility external
//		@group	data
//
//		@param	newData		(List of ListGridRecord)	data to show in the list
//<
setData : function (newData) {
	// if the current data and the newData are the same, bail
	//	(this also handles the case that both are null)
	if (this.data == newData) return;
	
    if (!this.preserveEditsOnSetData) this.discardAllEdits();
    
    // drop "lastHiliteRow" -no sense in hanging onto it
    this.clearLastHilite();
    
	// if we are currently pointing to data, stop observing it
	if (this.data) {
        this._ignoreData(this.data);
        // if the data was autoCreated, destroy it to clean up RS<->DS links
        if (this.data._autoCreated && isc.isA.Function(this.data.destroy))
            this.data.destroy();
    }

	// if newData was passed in, remember it
	if (newData) this.data = newData;
	
	// if data is not set, bail
	if (!this.data) return;

	// observe the data so we will update automatically when it changes
	this._observeData(this.data);
	
	// if we can regroup, do so. 
    
	this.regroup(true);
	
	// Don't redraw / refresh grid summary - this happens explicitly later.
	this.calculateRecordSummaries(null, true);
    
	// create a new selection if we don't have one or if we receive new data
    // Do this after grouping - if we created a new groupTree object we'll have already set up
    // an appropriate selection object
	if (!this.selection || (this.data != this.selection.data)) {
        this.createSelectionModel();
	}

    // Call this._remapEditRows() if we're hanging onto edit values
    // as we know they're now out of date.
    
	if (this.preserveEditsOnSetData) this._remapEditRows();
    // if this.alwaysShowEditors is set, and we have data, and we're not currently showing
    // editors, show them now.
    
    var fetching = isc.ResultSet && isc.isA.ResultSet(this.data) && !this.data.lengthIsKnown();
    if (!fetching && this._alwaysShowEditors() && !this._editorShowing) {
        this.startEditing(null,null,true,null,true);
    }

    var sortSpecifiers = this.getSort();
    if (sortSpecifiers) this.setSort(sortSpecifiers);

    if (this.hilites) {
        this.applyHilites();
    }

    
    if (isc.ResultSet && isc.isA.ResultSet(this.data) &&
        !(this.data.allRows && this.data.neverDropCache) &&
        this.body && this.body.overflow == "visible") 
    {
        this.body.showAllRows = false;
    }
    
    // Force a recalculation of the grid summaries since this method doesn't fire 
    // dataChanged() which is normally what causes this to occur.
    if (this.summaryRow && this.showGridSummary) {
        this.summaryRow._recalculateSummaries();
    }

    // clear out the current expanded row count, along with the current expanded record,
    // if there is one, since the expansionComponents have been dropped, along with the
    // records they were in
    this._expandedRecordCount=0;
    if (!this.canExpandMultipleRecords && this._currentExpandedRecord) 
        delete this._currentExpandedRecord;

    if (this._lastStoredSelectedState) {
        this.setSelectedState(this._lastStoredSelectedState);
        delete this._lastStoredSelectedState;
    }
    
    // if any fields are marked as autoFitWidth, recalculate their sizes
    this.updateFieldWidthsForAutoFitValue("setData called.");
    
	// mark us as dirty so we'll be redrawn if necessary
    this._markBodyForRedraw("setData");
},


// Determine which field to expand when autoFitFieldWidths is true and the 
// fields don't take up the full set of space available.
// If +link{listGrid.autoFitExpandField} is explicitly set use it.
// Otherwise, basically we want to find the first field that will not look odd to expand
// Implementation:
// - if a field is not of type text, ignore (don't want to expand images, dates, booleans, numbers)
// - If field has showValueIconsOnly set, ignore (even though it has text content it shouldn't
//   expand beyond the value icons width)
// - field.length:
//      - if all fields have field.length specified, expand the largest one
//      - if no fields have this property set, just expand the first text field
//      - if *some* fields have field.length set - the length may indicate a small field
//        (say 2 characters) which we don't want to expand.
//        Expand fields with no specified length in preference to those with
//        a small specified length.
autoFitExpandLengthThreshold : 10,
getAutoFitExpandField : function () {
    if (!this.autoFitFieldsFillViewport) return null;
    
    if (this.autoFitExpandField != null) {
        var field = this.getField(this.autoFitExpandField);
        // We don't support auto-expanding a frozen field at this time
        
        if (field != null && this.fields && this.fields.contains(field) && 
            (!this.frozenFields || !this.frozenFields.contains(field)))
        {
            return field;
        }
    }
    var fields = [], lengthFields = [];
    
    if (this.fields) {
        for (var i = 0; i < this.fields.length; i++) {
            var field = this.fields[i];
            if (!field.showValueIconOnly && 
               (isc.SimpleType.inheritsFrom(field.type, "text") || field.type == null)) 
            {
                if (!this._suppressedFrozenFields && field.frozen) continue;
                fields.add(field);
                if (fields[i] != null && fields[i].length != null) {
                   lengthFields.add(fields[i]);
                }
            
            }
        }
    }
    
    if (lengthFields.length > 0) {
        lengthFields.sortByProperty("length", Array.DESCENDING);
        if (lengthFields.last().length >= this.autoFitExpandLengthThreshold ||
            lengthFields.length == fields.length) 
        {
            return lengthFields[0];
        }
    }
    if (fields.length > 0) {
        var i = 0;
            field = fields[i]
        // Note: This conditional contains cases where we should *not* auto expand the field
        while (field != null && 
                // Respect explicitly specified pixel width
                ((field.width != null && field.width != "*") ||
                // Explicit length less than the min "expand" number of characters
                 (field.length != null &&
                    field.length < this.autoFitExpandLengthThreshold)) 
              )
        {
            i++;
            field = fields[i];
        }
        return field;
    }

    // Note that this could still return null - no text fields etc.
    return null;
},

// updateFieldWidthsForAutoFitValue()
// If this.autoFitWidthApproach means we size to
// column data and any fields are marked as autoFitWidth:true,
// this method will set the _fieldWidthsDirty flag on the gridRenderer, causing
// _updateFieldWidths() to be called on the next body redraw,
// Calling code should call this before causing a body redraw.
updateFieldWidthsForAutoFitValue : function (reason) {
    if (!this.body || this.body._fieldWidthsDirty) return;
    
    var fields = this.fields || [];
    for (var i = 0; i < fields.length; i++) {
        var shouldAutoFit = this.shouldAutoFitField(fields[i]);
        if (!shouldAutoFit) continue;
        var approach = this.getAutoFitWidthApproach(fields[i]);
        
        if (approach == "value" || approach == "both") {
            this.fields._appliedInitialAutoFitWidth = false;
            this.body._fieldWidthsDirty = "Updating field widths for field auto-fit" + 
                                      (reason ? (":" + reason) : ".");
            break;
        }
    }
},

invalidateCache : function () {
    // make sure that header checkbox is unchecked after refreshing cache
    if (this.getCheckboxFieldPosition() != -1) {
        this.deselectAllRecords();
        this._setCheckboxHeaderState(false);
    }

    // if we have a stored drawArea, delete it now so that the 
    // redraw recalculates it and reapplies recordComponents
    if (this.body && this.body._oldDrawArea) delete this.body._oldDrawArea;

    return this.Super("invalidateCache", arguments);
},

// use this rather than field.canSort to handle canSortClientOnly fields
_canSort : function (field) {
    if (!isc.isAn.Object(field)) {
        field = this.getSpecifiedField(field);
    }
    if (field == null) return false;

    var canSort = (field.canSort == false) ? false : this.canSort == false ? false : true;
    canSort = canSort && this._canSortData(field);

    return canSort;
},

_canSortData : function (fieldName) {
    
    var field = isc.isAn.Object(fieldName) ? fieldName : this.getSpecifiedField(fieldName);
    if (field == null) return false;

    if (isc.isAn.Array(this.data)) return true;
    if (field.canSortClientOnly == true) {
        if (isc.isA.ResultSet(this.data)) {
            if (!this.data.lengthIsKnown() || !this.data.canSortOnClient()) {
                return false;
            }
        } 
    }

    return true;
},

// helper to determine whether this grid can be sorted on multiple fields - returns false if
// either canMultiSort or canSort are false, or if the data itself doesn't support multiSort
_canMultiSort : function () {
    var canMultiSort = (this.canMultiSort != false) && this.canSort && this._canMultiSortData();
    return canMultiSort;
},

// helper to determine whether this grid's data supports sorting on multiple fields
_canMultiSortData : function () {
    var canMultiSort = this.canMultiSort;

    // if data doesn't support setSort (resultSet and Array do), disable multiSort
    if (!this.data.setSort) {
        canMultiSort = false;
    } else if (this.getDataSource() && canMultiSort != false) {
        canMultiSort = this.getDataSource().canMultiSort && this.canSort;
    }

    return canMultiSort;
},

// helper method to get an appropriate sortDirection for a field
_getFieldSortDirection : function (field) {
    var field = isc.isAn.Object(field) ? field : this.getSpecifiedField(field),
        direction;

    var fieldDir = field ? field.sortDirection : null;
    if (fieldDir != null) {
        direction = Array.shouldSortAscending(fieldDir) ? "ascending" : "descending";
    } else if (this.sortDirection != null) {
        direction = Array.shouldSortAscending(this.sortDirection) ? "ascending" : "descending";
    } else {
        direction = Array.shouldSortAscending(this._baseSortDirection) ? "ascending" : "descending";
    }

    return direction;
},

// Override createSelectionModel, from DataBoundComponent, to set the body's selection object
// with our selection object. Our body GridRenderer will then observe selection.setSelected.
createSelectionModel : function () {
    this.invokeSuper(isc.ListGrid, "createSelectionModel", arguments);
    if (isc.isA.Canvas(this.body)) {
        this.body.setSelection(this.selection);
        if (this.frozenBody) {
            this.frozenBody.setSelection(this.selection);
            // create separate selection objects for body/frozenBody
            this._separateDependentSelections();
        }
    }
},

destroySelectionModel : function () {
    if (this.body) this.body.clearSelection();
    if (this.frozenBody) this.frozenBody.clearSelection();
    return this.Super("destroySelectionModel", arguments);
},

// Method tripped to reselect a selected record that was updated in the DataSource
performReselectOnUpdate : function (updatedRecord) {
    if (this.reselectOnUpdateNotifications == "none") {
        this.suppressSelectionChanged = true;
    }
    this.selection.select(updatedRecord);
    if (this.reselectOnUpdatedNotifications == "selectionUpdated") {
        this.fireSelectionUpdated();
    }
    delete this.suppressSelectionChanged;
},

//>	@method	listGrid.setSelectionType()	[A]
// Changes selectionType on the fly.
// @param	selectionType (SelectionStyle)	New selection style. 
// @visibility external
//<
setSelectionType : function (selectionType, ignoreCheckbox) {
    // NOTE: this is sufficient because the Selection object dynamically inspects this property
    // on it's grid
    this.selectionType = selectionType;
    if (this.body) this.body.selectionType = selectionType;
},

//>	@method	listGrid.setSelectionAppearance()	
// Changes selectionAppearance on the fly.
// @param	selectionAppearance (String)	new selection appearance
// @visibility external
//<
setSelectionAppearance : function (selectionAppearance, isInit) {
    if (this.selectionAppearance == selectionAppearance && !isInit) return;
    
    if (selectionAppearance == "checkbox" && this.canSelectCells) {
        this.logWarn("ignoring selectionAppearance:'checkbox' since canSelectCells is set");
        selectionAppearance = "rowStyle";
    }

    this.selectionAppearance = selectionAppearance;

    // at initialization time, if selectionType wasn't explicitly set, default selectionType to
    // "simple" for "checkbox" appearance, otherwise "multiple"
    if (isInit && this.selectionType == null && !this.canSelectCells) {
        this.selectionType = (selectionAppearance == "checkbox" ? isc.Selection.SIMPLE : 
                                                                  isc.Selection.MULTIPLE);
    }
    
    // If this.completeFields is set (IE setFields has run already), we need to either add or
    // remove the checkboxField
    // Call setFields() with a duplicate of the current set of fields, less the
    // checkboxField (if there was one)
    // This will create a new checkboxField if necessary otherwise just clear the existing one
    if (this.completeFields != null) {
        var completeFields = [];
        for (var i = 0; i < this.completeFields.length; i++) {
            var field = this.completeFields[i];
            if (this.isCheckboxField(field)) continue;
            completeFields.add(field);
        }
        this.setFields(completeFields);
    }
},

//> @method listGrid.setBodyOverflow()  ([A])
// Update the +link{listGrid.bodyOverflow, bodyOverflow} for this listGrid.
// @param overflow (Overflow) new overflow setting for the body
// @visibility external
//<
setBodyOverflow : function (newOverflow) {
    this.bodyOverflow = newOverflow;
    if (this.body) this.body.setOverflow(this.bodyOverflow);
},

//> @method listGrid.setBodyStyleName()
// Update the +link{listGrid.bodyStyleName,bodyStyleName} for this listGrid.
// @param styleName (CSSStyleName) new body style name
// @visibility external
//<
setBodyStyleName : function (styleName) {
    this.bodyStyleName = styleName;
    if (this.body && (!this.alternateBodyStyleName || !this.alternateRecordStyles)) {
        this.body.setStyleName(styleName);
    }
},

//> @method listGrid.setAlternateBodyStyleName()
// Update the +link{listGrid.alternateBodyStyleName,alternateBodyStyleName} for this listGrid.
// @param styleName (CSSStyleName) new body style name when showing alternateRecordStyles
// @visibility external
//<
setAlternateBodyStyleName : function (styleName) {
    this.alternateBodyStyleName = styleName;
    if (this.body && this.alternateRecordStyles) {
        // if passed 'null', reset to this.bodyStyleName
        this.body.setStyleName(styleName || this.bodyStyleName);
    }
},

//> @method listGrid.setAlternateRecordStyles()
// Setter for +link{listGrid.alternateRecordStyles}
// @param alternateStyles (boolean) New value for <code>this.alternateRecordStyles</code>
// @visibility external
//<
setAlternateRecordStyles : function (alternateStyles) {
    if (this.alternateRecordStyles == alternateStyles) return;
    this.alternateRecordStyles = alternateStyles;
    
    if (this.body && (this.alternateBodyStyleName != null)) {
        if (alternateStyles) this.body.setStyleName(this.alternateBodyStyleName);
        else this.body.setStyleName(this.bodyStyleName);
    }
    
},

// Override hasInherentHeight / width: If we're autoFitting to our data, advertise inherent height
// This means that a layout will not expand us to fit the available space.

hasInherentHeight : function (a,b,c,d) {
    if (this.inherentHeight != null) return this.inherentHeight;
    if (this.autoFitData == isc.Canvas.VERTICAL || this.autoFitData == isc.Canvas.BOTH) {
        return true;
    }
    return this.invokeSuper(isc.ListGrid, "hasInherentHeight", a,b,c,d);
},

hasInherentWidth : function (a,b,c,d) {
    if (this.inherentWidth != null) return this.inherentWidth;
    if (this.autoFitData == isc.Canvas.HORIZONTAL || this.autoFitData == isc.Canvas.BOTH) {
        return true;
    }
    return this.invokeSuper(isc.ListGrid, "hasInherentWidth", a,b,c,d);
},

//> @method listGrid.setAutoFitData()
// Setter for +link{listGrid.autoFitData}.
// @param autoFitData (string) One of <code>"vertical"</code>, <code>"horizontal"</code>
//  or <code>"both"</code>. To disable auto fit behavior, pass in <code>null</code>.
// @group autoFitData
// @visibility external
//<
setAutoFitData : function (autoFitData) {
    this.autoFitData = autoFitData;
    
    if (this._autoDerivedCanFreeze && (autoFitData == "both" || autoFitData == "horizontal"
                                 
                                       || autoFitData == "vertical")) {
        delete this._autoDerivedCanFreeze;
        delete this.canFreezeFields;
    }
    if (autoFitData == null && this._specifiedOverflow) {
        this.setOverflow(this._specifiedOverflow);
    } else if (this.overflow != "visible") {
        this._specifiedOverflow = this.overflow;
        this.setOverflow("visible");
    }
    if (this.body) {
        this.body.autoFitData = this.autoFitData;
        this.body.adjustOverflow();
    }
},

//> @method listGrid.setAutoFitExtraRecords()
// Setter for +link{listGrid.autoFitExtraRecords}.
// @param extraRecords (integer) Number of extra rows beyond the data-size we'll expand to 
// accommodate if +link{listGrid.autoFitData,auto fit} is enabled vertically.
// @group autoFitData
// @visibility external
//<
setAutoFitExtraRecords : function (extraRecords) {
    this.autoFitExtraRecords = extraRecords;
    if (this.body) {
        this.body.autoFitExtraRecords = extraRecords;
        this.body.adjustOverflow();
    }
},


//> @method listGrid.setAutoFitMaxRecords()
// Setter for +link{listGrid.autoFitMaxRecords}.
// @param maxRecords (integer) Maximum number of rows we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled vertically.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxRecords : function (maxRecords) {
    this.autoFitMaxRecords = maxRecords;
    if (this.body) {
        this.body.autoFitMaxRecords = maxRecords;
        this.body.adjustOverflow();
    }
},

//> @method listGrid.setAutoFitMaxHeight()
// Setter for +link{listGrid.autoFitMaxHeight}.
// @param height (integer) Maximum height in px we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled vertically.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxHeight : function (height) {
    this.autoFitMaxHeight = height;
    if (this.body) {
        this.body.adjustOverflow();
    }
},
getAutoFitMaxBodyHeight : function () {
    if (this.autoFitMaxHeight == null) return null;
    var offset = this.getVBorderPad();
    if (this.showHeader) offset += this.getHeaderHeight();
    if (this.showFilterEditor) offset += this.filterEditorHeight;
    return this.autoFitMaxHeight - offset;
    
},

// When auto-fitting vertically, specified height for the grid acts as a minimum
getAutoFitMinBodyHeight : function () {
    var minHeight = this.getHeight(),
        offset = this.getVBorderPad();
    if (this.showHeader) offset += this.getHeaderHeight();
    if (this.showFilterEditor) offset += this.filterEditorHeight;
    return (minHeight - offset);
},

//> @method listGrid.setAutoFitMaxColumns()
// Setter for +link{listGrid.autoFitMaxColumns}.
// @param maxColumns (integer) Maximum number of fields we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled horizontally.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxColumns : function (maxColumns) {
    this.autoFitMaxColumns = maxColumns;
    if (this.body) {
        this.body.autoFitMaxColumns = maxColumns;
        this.body.adjustOverflow();
    }
},

//> @method listGrid.setAutoFitMaxWidth()
// Setter for +link{listGrid.autoFitMaxWidth}.
// @param width (integer) Width in px we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled horizontally.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxWidth : function (width) {
    this.autoFitMaxWidth = width;
    if (this.body) {
        this.body.autoFitMaxWidth = width;
        this.body.adjustOverflow();
    }
},

// --------------------------------
// AutoFitFields

//> @method listGrid.autoFitField()
// Programmatically cause a field to auto-fit horizontally to it's contents or title.
// <P>
// Does not establish permanent auto-fitting - use +link{listGrid.setAutoFitWidth()} to do so.
// @param fieldName (string)
// @return (int) new width in pixels
//
// @group autoFitFields
// @visibility external
//<
autoFitField : function (fieldName, scrollIntoView) {
    var field = this.getField(fieldName),
        fieldNum = this.getFieldNum(field);
    // avoid attempting to autofit hidden fields, or fields where
    // autoFitWidth is already set (should just happen dynamically!)
    if (field == null || fieldNum == -1) return;

    var widths = this.getFieldAutoFitWidths([field]);
    if (widths == null || widths[0] == null) return;
    var width = widths[0];
    // resize the field (as if the user drag-resized to the auto-fit size)
    var dontStoreWidth = this.shouldAutoFitField(field);
    if (width != this._fieldWidths[fieldNum]) {    
        this.resizeField(fieldName, width, !dontStoreWidth);
    } else if (!dontStoreWidth) field.width = width;
    if (dontStoreWidth) field._calculatedAutoFitWidth = this._fieldWidths[fieldNum];

    if (scrollIntoView) this.scrollToColumn(fieldNum, false);
    return width;
},

//> @method listGrid.autoFitFields()
// Perform a one-time horizontal auto-fit of the fields passed. Fields will be sized
// to match their contents or title (as specified in +link{listGrid.autoFitWidthApproach})
// Does not establish permanent auto-fitting - use +link{listGrid.setAutoFitWidth()} to do so.
// @param [fields] (Array of ListGridField) Array of fields to auto fit. If this parameter
//  is not passed, autoFitting will occur on all visible fields.
//
// @group autoFitFields
// @visibility external
//<

autoFitFields : function (fields) {
    

    if (this.body) this.body._fieldWidthsDirty = true;
    this.autoSizeHeaderSpans = false;
    if (fields == null) fields = this.fields;
    
    this._autoFittingFields = true;
    var finalFields = [];
    for (var i = 0; i < fields.length; i++) {
        var field = this.getField(fields[i]);
        if (field != null) {
            delete fields[i].width;
            delete fields[i]._calculatedAutoFitWidth;
            finalFields[finalFields.length] = field;
        }
    }
    // we've removed any hidden fields here.
    fields = finalFields;
    var widths = this.getFieldAutoFitWidths(fields);
    var fieldNums = [], resizeWidths = [], storeWidths = [];
    for (var i = 0; i < fields.length; i++) {

        if (widths[i] == null) continue;
        var width = widths[i],
            field = fields[i],
            fieldNum = this.getFieldNum(fields[i]),
            fieldName = fields[i].name;
        // resize the field (as if the user drag-resized to the auto-fit size)
        var dontStoreWidth = this.shouldAutoFitField(field);
        if (width != this._fieldWidths[fieldNum]) {

            fieldNums[fieldNums.length] = fieldNum;
            resizeWidths[resizeWidths.length] = width;
            storeWidths[storeWidths.length] = !dontStoreWidth;

        } else if (!dontStoreWidth) field.width = width;
    }

    this._resizeFields(fieldNums, resizeWidths, storeWidths);
    for (var i = 0; i < fieldNums.length; i ++) {
        if (!storeWidths[i]) {
            var fieldNum = fieldNums[i];
            this.getField(fieldNum)._calculatedAutoFitWidth = this._fieldWidths[fieldNum];
        }
    }

    this._autoFittingFields = false;

    // we suppressed placeEmbeddedComponents() during the field resizes, so
    // run it now to ensure any embedded components are correctly positioned and sized.
    if (this.frozenBody) this.frozenBody._placeEmbeddedComponents()
    if (this.body) this.body._placeEmbeddedComponents();
    // Normally we do this in resizeField() but we don't want a bunch of unnecessary
    // reflows so we delay until resize of fields is complete
    if (this.header && this.autoFitHeaderHeights) {
        this.dropCachedHeaderButtonHeights();
        this._updateHeaderHeight();
    }

    this.getFieldWidths();
    // This will adjust header heights if necessary to account for differently wrapped
    // content
    if (this.header) this.header._sizeSpans();
},


shouldAutoFitField : function (field) {
    if (field.autoFitWidth != null) return field.autoFitWidth;
    return this.autoFitFieldWidths;
},

// This method is called directly as part of resizeFields() and acts as a minimum, disallowing
// resizing below the min field width.

getMinFieldWidth : function (field, ignoreFieldWidth) {
    var minWidth = this.minFieldWidth || 1;
    if (!ignoreFieldWidth) {
        
        var fieldWidth = field.width;
        if (isc.isA.Number(fieldWidth)) {
            minWidth = Math.max(minWidth, field.width);
        }
    }
    var fieldName = field.name;
    // If we have embeddedComponents for the field we should treat that as a minimum
    if (this._columnComponentsMap && this._columnComponentsMap[fieldName]) {
        var componentMaxWidth = this._getFieldComponentMaxWidth(fieldName);
        if (componentMaxWidth > minWidth) {
            minWidth = componentMaxWidth;
        }
    }
    
    return minWidth;
},

_$within:"within",
_getFieldComponentMaxWidth : function (fieldName) {
    var field = this.getField(fieldName);
    if (field == null) return 0;
    
    if (field._maxComponentWidth != null) return field._maxComponentWidth;
    
    var components = this._columnComponentsMap[fieldName];
    var maxWidth = 0;
    for (var compID in components) {
        var component = window[compID];
//         this.logWarn("For field:" + fieldName + " contemplating:" + component + 
//             " which has width:" + component.getVisibleWidth());
        // skip "expand" components - these are always sized to fit the field.
        if (component && (component.embeddedPosition == this._$within) && 
            (component.getVisibleWidth() > maxWidth) )
        {
            maxWidth = component.getVisibleWidth();
        }
    }
    field._maxComponentWidth = maxWidth;
    return maxWidth;
},

// This method fires when we have embedded components which may overflow the
// specified field width horizontally, so we need to expand to fit.
_fieldComponentWidthsChanged : function (fieldName, newWidth) {
    var field = this.getField(fieldName);
    if (field) {
        // If we happen to know we just increased the size, we can update
        // the max-width without having to query sizes of all drawn components.
        // Otherwise just clear the cached value and we'll lazily recalculate.
        if (newWidth) field._maxComponentWidth = newWidth;
        else delete field._maxComponentWidth;
        
        // Use "fireOnPause" to actually resize the field to fit the components on a delay
        
        if (this._staleComponentWidthFields == null) {
            this._staleComponentWidthFields = {};
        }
        this._staleComponentWidthFields[fieldName] = true;
        this.fireOnPause(
            "checkFieldComponentOverflow", 
            {target:this,methodName:"_checkFieldComponentOverflow"}, 0
        );
    }
    
},
_checkFieldComponentOverflow : function () {
    if (this.fields == null) return;

    
    
    var currentWidths = this.getFieldWidths(),
        overflowedFields = [],
        newWidths = [],
        storeWidths = [],
        overflowed = false;
    for (var i = 0; i < this.fields.length; i++) {
        var name = this.fields[i].name;
        if (!this._staleComponentWidthFields[name]) continue;
        this._staleComponentWidthFields[name] = null;
        
        var minWidth = this.getMinFieldWidth(this.fields[i]);
        if (minWidth > currentWidths[i]) {
            overflowedFields.add(i);
            newWidths.add(minWidth);
            // We don't want to store any of these widths. If autoFitFieldWidths
            // is true, if the component is hidden etc we want to be able to resize
            // smaller again if _updateFieldWidthsForAutoFitValue() gets re-run.
            storeWidths.add(false);
        }
    }
    // Embedded components in some cell overflowed the available width - resize
    // the field to accomodate it.
    
    if (overflowedFields.length > 0) {
        this._resizeFields(overflowedFields, newWidths, storeWidths);
    }
},

getFieldAutoFitWidths : function (fields) {
    if (this.body == null || fields == null) return;
    if (!isc.isA.Array(fields)) {
        fields = [fields]
    }
    var widths = [],
        colNums = [],
        headers = [],
        bodyFields = [];
    
    for (var i = 0; i < fields.length; i++) {
        var minWidth = this.getMinFieldWidth(fields[i], true),
            field = fields[i],
            colNum = this.getColNum(field),
            approach = this.getAutoFitWidthApproach(field, true),        
            checkHeader = approach != "value",
            checkBody = approach != "title";
            
        // we'll use the colNums outside this loop when we pick up the
        // body col widths.
        colNums[i] = colNum;
        widths[i] = minWidth;

        if (checkHeader) {
            var header = this.getFieldHeaderButton(colNum);
            
            if (header != null) headers[i] = header;
        }            

        if (checkBody) {
            bodyFields.add(field);
        }
        
    }
    
    if (headers.length > 0) {
        //var startTime = isc.timeStamp();
        var headerWidths = this.getAutoFitTitleWidths(headers);
        //this.logWarn("Time to get header title widths:" + (isc.timeStamp() - startTime));

        for (var i = 0; i < headerWidths.length; i++) {
            
            if (headerWidths[i] == null) continue;
            widths[i] = Math.max(widths[i],headerWidths[i]);
        }
    }

    // for efficiency, pass all the body fields to getAutoFitValueWidths() at once.
    
    if (bodyFields.length > 0) {
        var bodyColWidthArr = this.getAutoFitValueWidths(bodyFields);
        for (var i = 0; i < fields.length; i++) {
            if (colNums[i] != null) {
                var colNum = colNums[i],
                    bodyColWidth = bodyColWidthArr ? bodyColWidthArr[colNum] : null;
                if (bodyColWidth != null) {
                    if (widths[i] == null || bodyColWidth > widths[i]) {
                        widths[i] = bodyColWidth;
                    }
                }
            }
        }
    }
    
    return widths;
},

getAutoFitTitleWidths : function (headers) {

    var testHTML = "",
        missingHeaders = {};
    for (var i = 0; i < headers.length; i ++) {
        
        if (headers[i] == null) {
            missingHeaders[i] = true;
            continue;
        }
        var header = headers[i],
            titleStyle = header.titleStyle;
        if (titleStyle == null) titleStyle = header.getStateName();
        
        testHTML += "<div style='position:absolute;" +
                (header.wrap ? "'" : "white-space:nowrap;'") +
                " class='" + titleStyle + "'>" 
                    
                    //+ headers[i].getInnerHTML() +
                    + this.getHeaderButtonTitle(header, false) + 
                "</div>";
    }

    if (isc.ListGrid.headerWidthsTester == null) {
        isc.ListGrid.headerWidthsTester = isc.Canvas.create({
            top:-100,
            autoDraw:true,
            overflow:"hidden",
            contents:testHTML
        })

    } else {
        isc.ListGrid.headerWidthsTester.setContents(testHTML);
    }
    if (isc.ListGrid.headerWidthsTester.isDirty()) {
        isc.ListGrid.headerWidthsTester.redraw();
    }
    
    var handle = isc.ListGrid.headerWidthsTester.getHandle(),
        childNodes = handle.childNodes;
    
    var widths = [];
    for (var i = 0, j=0; i < headers.length; i ++) {
        if (missingHeaders[i]) {
            widths[i] = null;
            continue;
        }
        
        var titleDiv = childNodes[j];
        widths[i] = titleDiv.offsetWidth;
        // If the title is written into a label account for the fact it's sized 
        // to fit within its actual button.
        if (headers[i].label != null) {
            widths[i] += (headers[i].getLabelHPad() + headers[i].getHBorderSize()) * 2;
        }
        j++;
    }
    
    if (widths.length != headers.length) {
        this.logWarn("getAutoFitTitleWidths(): Mismatch between " +
            "headers passed in and calculated widths");
    }
    return widths;
},

//> @method listGrid.setAutoFitWidth()
// Setter for +link{field.autoFitWidth}. Enables or disables dynamic autoFitWidth behavior
// on the specified field. Note if the field is currently autoFitWidth:true, and this method is
// disabling autoFit, the field will not be resized by default - if you wish to resize to an
// explicit width, use +link{listGrid.resizeField()}.
//
// @param fieldName (string) field to auto-fit
// @param autoFit (boolean) Should autoFitWidth be enabled or disabled? 
// @group autoFitFields
// @visibility external
//<
setAutoFitWidth : function (fieldName, autoFit) {
    var field = this.getField(fieldName);
    if (field == null) return;
    if (field.autoFitWidth == autoFit) return;
    
    field.autoFitWidth = autoFit;

    if (autoFit) this.autoFitField(field);
    // If we're turning off auto-fit, don't bother to resize
},

//> @method listGrid.setAutoFitFieldWidths()
// Setter for +link{listGrid.autoFitFieldWidths}. Modifies the default auto-fit-width behavior
// for fields in this grid. Note that this may be overridden at the field level via 
// +link{listGridField.autoFitWidth}.
// @param autoFit (boolean) New value for autoFitFieldWidths
// @param [dontResetWidths] (boolean) If autoFitFieldWidths was true, and is being set to false, 
//  should fields be resized to their originally specified size? 
//  Pass in this parameter to suppress this behavior.
// @visibility external
//<
setAutoFitFieldWidths : function (autoFit, dontResetWidths) {
    if (autoFit == this.autoFitFieldWidths) return;
    this.autoFitFieldWidths = autoFit
    if (autoFit) {
        this._updateFieldWidths("autoFitFieldWidths enabled");
    } else if (!dontResetWidths) {
        // If we're showing a header, we use it to handle converting specified
        // field widths into real sizes - running updateHeader will rebuild it and
        // perform this initial calculation for us.
        if (this.showHeader && this.headerHeight > 0) this.updateHeader();
        // Clear the flag indicating we've run through auto-fit logic and re-run
        // _updateFieldWidths() to revert the fields to specified sizes.
        this.fields._appliedInitialAutoFitWidth = false;
        this._updateFieldWidths("autoFitFieldWidths disabled");
    }
},

//> @method listGrid.setAutoFitWidthApproach()
// Setter for the +link{listGrid.autoFitWidthApproach}.
// @param approach (AutoFitWidthApproach) new AutoFitWidth approach
// @visibility external
//<
setAutoFitWidthApproach : function (approach) {
    if (this.autoFitWidthApproach == approach) return;
    this.autoFitWidthApproach = approach;

    // If we're showing a header, we use it to handle converting specified
    // field widths into real sizes - running updateHeader will rebuild it and
    // perform this initial calculation for us.
    
    if (this.showHeader && this.headerHeight > 0) this.updateHeader();
    // Clear the flag indicating we've run through auto-fit logic and re-run
    // _updateFieldWidths() to revert the fields to specified sizes.
    this.fields._appliedInitialAutoFitWidth = false;
    this._updateFieldWidths("autoFitFieldWidthApproach changed");
},

// mark the body for redraw, or if the body doesn't exist, the widget as a whole
_markBodyForRedraw : function (reason) {
	if (this.bodies) {
        this.bodies.map("markForRedraw", reason);
    } else if (this.body) {
        this.markForRedraw(reason);
    }        
},

redraw : function (a, b, c, d) {
    if (this.body) {
        
        if (this.body._scrollbarChangeDuringAnimation) {
            this._updateFieldWidths("scrollbar change during animation");
            delete this.body._scrollbarChangeDuringAnimation;
        }
        
        if (this._forceUpdateFieldWidths) this.body._fieldWidthsDirty = true;
    }

    this.invokeSuper(isc.ListGrid, "redraw", a, b, c, d);
    
},

//>	@method listGrid._observeData() (A)
//		observe methods on the data so we redraw automatically when data changes
//		called automatically by setData
//	@param	data	(object)		new data to be observed
//<
_observeData : function (data) {
	// redraw if the data changed
    
    if (!this.isObserving(data, "dataChanged")) {
        this.observe(data, "dataChanged",
                        "observer.dataChanged(" + 
                            (isc.ResultSet && isc.isA.ResultSet(data) ?
                            "arguments[0],arguments[1],arguments[2],arguments[3],arguments[4])" : 
                            ")")
        );
    }
 
    if (!this.isObserving(data, "dataArrived")) {
        if (isc.ResultSet && isc.isA.ResultSet(data)) {
            this.observe(data, "dataArrived", "observer._dataArrived(arguments[0],arguments[1])");
        } else if (isc.ResultTree && isc.isA.ResultTree(data)) {
            this.observe(data, "dataArrived", "observer._dataArrived(arguments[0])");
        }
    }

	// Note - we must check for data being a tree as if it is not defined, the inherited 
    // ListGrid.init() code will set it to an empty array, in which case this observation will 
    // fail.
    if (isc.isA.Tree(data)) {
        // update view in response to folders opening / closing
    	this.observe(data, "changeDataVisibility", "observer._folderToggleObservation(node, newState)");
    } 
},
//> @method listGrid.groupTreeChanged()
// Callback fired from group tree +link{listGrid.groupTree} dataChanged().
// <p>
// Handles remapping edit rows and forcing a redraw if necessary.
// @group grouping
//< 
// The groupTree may be a ResultTree or a Tree object.
// When a listGrid is grouped we still observe dataChanged on the underlying data object and react
// to it by updating or rebuilding the groupTree as required, and marking for redraw etc.
// Therefore if this method is fired from standard dataChanged() we typically need to take no
// action.
// Note that we explicitly disable databound cache-synch for the ResultTree and instead manage
// updating the ResultTree cache directly as part of ListGrid.dataChanged. This is appropriate since
// The ResultTree code for cache sync is organized around node ids and parent ids whereas the
// groupTree is a dynamic grouping based on records have the same values for a field.
// 
// This will actually fire in response to listGrid sort or direct manipulation of the groupTree
//
// Note that this method is only fired when an existing groupTree changes - not when regroup()
// is run, creating a new groupTree.
_$dataChanged:"dataChanged",
groupTreeChanged : function () {
    // If the groupTree was updated from underlying data change, no need to
    // redraw etc (already handled in dataChanged())
    if (this._handlingDataChanged) return;
    
    if (!this._savingEdits && !this.suppressEditRowRemap) this._remapEditRows();
    var lastRow = this.getTotalRows()-1;
    if (this.body) {
        
        if (this.body.lastOverRow > lastRow) delete this.body.lastOverRow;
        if (this.body.lastMouseOverRow > lastRow) delete this.body.lastMouseOverRow;
        if (this.body._lastHiliteRow > lastRow) delete this.body._lastHiliteRow;
    }
    if (this._lastRecordClicked > lastRow) delete this._lastRecordClicked;
    
    if (this.hilites) this.applyHilites();
    if (!this._suppressRedrawOnDataChanged) this.redrawForDataChanged();
    
},
//>	@method listGrid._observeGroupData() (A)
//      observe methods on the group tree object, so that changes to the group layout
//      can be detected
//	@param	data	(object)		new group tree to be observed
//  @visibility internal
//<
_observeGroupData : function (data) {
	// redraw if the data changed 
    this.observe(data, "dataChanged", "observer.groupTreeChanged()");
    this.observe(data, "changeDataVisibility", "observer._folderToggleObservation(node,newState)");
},

// METHODS MOVED FROM TREEGRID
// The following methods were moved from treegrid to allow the listgrid to support the tree
// as a data model for grouping. They will continue to be doc'd on treegrid for now

// Helper method - fired when folders open/close within the tree
_folderToggleObservation : function (node, newState) {
    // If we're loaded, reapply hilites, but pass in the flag to suppress marking for redraw
    // We need to do this since we won't get a dataChanged notification from the
    // children loading.
    // No need to redraw here since we will redraw (below) in response to the folder toggling.
    if (node != null && this.hilites && newState && this.data.getLoadState(node) == isc.Tree.LOADED) {
        this.applyHilites(true);
    }
    //>Animation
    // During animated folder open/close we suppress redraw in response to the folder toggling
    
    if (this._suppressFolderToggleRedraw) {            
        this._remapEditRows();

        // Re run auto fit logic expand our cols to fit the revealed content
        
        this.updateFieldWidthsForAutoFitValue("Folder Toggled");

        return;
    }
    
    // Cut short any currently running animated folder open / close 
    // Just call finishAnimation - this will no op if no animation is running
    
    if (this.body) this.body.finishRowAnimation();
    //<Animation
    
    // Length changes so we need to remap edit rows.
    this._remapEditRows();

    // Re run auto fit logic expand our cols to fit the revealed content
    this.updateFieldWidthsForAutoFitValue("Folder Toggled");

    // redraw to display the updated folder
    this._markBodyForRedraw('folderToggled');
},

//> @method treeGrid.toggleFolder()   ([])
//          Opens the folder specified by node if it's closed, and closes it if it's open.
//          TreeGrid will redraw if there's a change in the folder's open/closed state.
//
//      @visibility external
//      @param  node        (TreeNode)      node to toggle
//<
toggleFolder : function (node) {
    if (this.data.isOpen(node)) {
        this.closeFolder(node);
    } else {
        this.openFolder(node);
        
        if (this.frozenBody) this.frozenBody.markForRedraw();
    }
},

//> @method treeGrid.openFolder() ([A])
// Opens a folder.
// <p>
// Executed when a folder node receives a 'doubleClick' event. This handler must be
// specified as a function, whose single parameter is a reference to the relevant folder
// node in the tree's data.<br>
// See the ListGrid Widget Class for inherited recordClick and recordDoubleClick events.
//
// @param   node        (TreeNode)      node to open
// @see closeFolder()
// @see folderOpened()
// @see class:ListGrid
// @visibility external
//<
openFolder : function (node) {
    // CALLBACK API:  available variables:  "node"
    // Convert a string callback to a function
    if (this.folderOpened != null) {
        this.convertToMethod("folderOpened");
        if (this.folderOpened(node) == false) return false;
    }
    
    
    if (this.animateFolders) {
        this.animateOpen(node);
    } else {
        this.data.openFolder(node);
    }

},   

//> @method treeGrid.animateOpen()
// Animates a folder opening to display its children (which grow into view).
// Automatically triggered from <code>treeGrid.folderOpen()</code> if 
// <code>this.animateFolders</code> is true.
// @group animation
// @param folder (node) node to open
// @visibility animation_advanced
//<
animateOpen : function (folder) {
    var data = this.data;
    if (data.isOpen(folder)) return;

    // Open the data, but don't redraw with the new data visible (we'll handle redrawing 
    // when the animation completes).
    this._suppressFolderToggleRedraw = true;
    data.openFolder(folder);
    delete this._suppressFolderToggleRedraw;

    // parent may be null if we're looking at the root node
    var parent = data.getParent(folder);
    if (parent && !data.isOpen(parent)) return;
     
    if (data.getLoadState(folder) != isc.Tree.LOADED) {
        //this.logWarn("animation for LOD folder");
        // wait for dataChanged() to fire
        this._pendingFolderAnim = folder;
        return;
    }
   
    this._startFolderAnimation(folder);
},

//> @method treeGrid.closeFolder()
// Closes a folder.
//
// @param   node        (TreeNode)      node to close
// @see openFolder()
// @see folderClosed()
// @visibility external
//<
closeFolder : function (node) {
    // CALLBACK API:  available variables:  "node"
    // Convert a string callback to a function
    if (this.folderClosed != null) {
        this.convertToMethod("folderClosed");
        if (this.folderClosed(node) == false) return false;
    }

    // cancel editing of any nodes under this one
    if (this.getEditRow() != null) {
        var editRecord = this.getRecord(this.getEditRow());
        if (this.data.isDescendantOf(editRecord, node)) this.endEditing();
    }
    // now tell the data to close the folder
    if (this.shouldAnimateFolder(node)) 
        this.animateClose(node);
    else 
        this.data.closeFolder(node);
},

//> @method treeGrid.animateClose()
// Animates a folder closing to hide its children (which shrink out of view).
// Automatically triggered from <code>treeGrid.folderOpen()</code> if 
// <code>this.animateFolders</code> is true.
// @param folder (node) node to open
// @group animation
// @visibility animation_advanced
//<
animateClose : function (folder) {
    if (!this.data.isOpen(folder)) return;

    var parent = this.data.getParent(folder);
    if (parent && !this.data.isOpen(parent)) {
        return this.closeFolder(folder);
    }
    
    var data = this.data,
        folderIndex = data.indexOf(folder),
        numChildren = data.getOpenList(folder).getLength()-1;

    
    
    this.startRowAnimation( false, 
                            folderIndex+1, 
                            folderIndex + numChildren + 1, 
                            {target:this, methodName:"redraw"}, 
                            this.animateFolderSpeed, 
                            this.animateFolderTime,
                            this.animateFolderEffect,
                            true
                          );
    var wasSuppressed = this._suppressFolderToggleRedraw;
    this._suppressFolderToggleRedraw = true;
    this.data.closeFolder(folder);
    this._suppressFolderToggleRedraw = wasSuppressed;
    
    if (this.body && this.body._delayedRowAnimation != null) {
        this.body._openFolder = folder;
    }
    if (this.frozenBody && this.frozenBody._delayedRowAnimation != null) {
        this.frozenBody._openFolder = folder;
    }
    
},

_startFolderAnimation : function (folder) {
    // At this point we know we have all the children for the folder loaded - verify
    // that we actually should animate the folder into view - if we have too many children
    // we may not want to -- in this case just redraw.
    if (!this.shouldAnimateFolder(folder)) {
        this.markForRedraw();
        return;
    }
    var data = this.data,
        folderIndex = data.indexOf(folder),
        numChildren = data.getOpenList(folder).getLength()-1;

    // don't try to animate empty folders
    if (folderIndex < 0 || numChildren <= 0) return;
    
    this.startRowAnimation( true, 
                            folderIndex+1, 
                            (folderIndex + numChildren+1), 
                            {target:(this.bodyLayout || this.body), methodName:"redraw"}, 
                            this.animateFolderSpeed, 
                            this.animateFolderTime,
                            this.animateFolderEffect,
                            true
                          );
},

// Used to store open folder state in the groupTree
// (Also used by TreeGrid.getOpenState())

_addNodeToOpenState : function (tree, node, openState, isGroupTree) {
    if (!tree.isOpen(node) || !tree.isLoaded(node)) return false;
    var children = tree.getFolders(node),
        hasOpenChildren = false;
    if (children != null) {
        for (var i = 0; i < children.length; i++) {
            hasOpenChildren = this._addNodeToOpenState(tree, children[i], openState, isGroupTree) 
                              || hasOpenChildren;
        }
    }
    if (isGroupTree) {
        var folderInfo = {};
        folderInfo[node.groupName] = node.groupValue;
        openState.add(folderInfo);
    } else {
        openState[openState.length] = tree.getPath(node);
    }
    return true;
},

// END METHODS MOVED FROM TREEGRID


dataChanged : function (type, originalRecord, rowNum, updateData, filterChanged) {
    if (isc._traceMarkers) arguments.__this = this;
    
    if (this._asyncRegroupInProgress) {
        // We are currently in the middle of a regroup that, because of the size of the data,
        // is being executed asynchronously.  Respond to this dataChanged() after the regroup
        // operation is completed.
        var queue = (this._asyncRegroupDataChangedQueue = this._asyncRegroupDataChangedQueue || []);
        queue.push({
            type: type,
            originalRecord: originalRecord,
            rowNum: rowNum,
            updateData: updateData,
            filterChanged: filterChanged
        });
        return;
    }

    // set a flag so we know we're handling dataChanged
    // This prevents us from causing unnecessary additional redraws from dataChanged on the
    // groupTree if we're currently grouped by any field(s) 
    this._handlingDataChanged = true;
   
    // DataChanged fires in some cases where we don't want to reset autoFieldWidths
    // For example, scrolling through a paged resultset where the columns resizing on scroll
    // would be ugly.
    // Use ResultSet parameters to test for cases to react to:
    // - crud operations which will affect the data displayed
    // - filter changing (including invalidateCache calls)
    // Note that we have equivalent logic inline in removeData, cellChanged to handle
    // non-databound grids' data being changed through standard grid APIs
    var resetAutoFitWidths = filterChanged || type == "add" || type == "remove";
    if (!resetAutoFitWidths && type == "replace") {
        if (originalRecord == null || rowNum == null) resetAutoFitWidths = true;
        else {
            var updatedRecord = currData.get(rowNum);
            if (updatedRecord == null) resetAutoFitWidths = true;
            else {
                for (var i = 0; i < this.fields.length; i++) {
                    if (this.shouldAutoFitField(this.fields[i])) {
                        var fieldName = this.getFieldName(this.fields[i]);
                        if (updatedRecord[fieldName] != updateData[fieldName]) {
                            resetAutoFitWidths = true;
                            break;
                        }
                    }
                }
            }
        }
    }    
    
    // If we have any summary fields where we need to store summaries on the records, update
    // them now
    // Suppress redraw and grid summary update -- we explicitly handle these below.
    this.calculateRecordSummaries(null, false);

    // if a change was made to the groupBy field of a record, regroup
    var groupByFields = this.getGroupByFields();
    if (groupByFields != null && !this._markForRegroup) {
        // fully regroup for add/remove, or for an update where dataChanged is not passed the
        // originalRecord to figure out if the groupField was changed
        if (type == "add" || type == "remove" || type == "replace" ||
                (type == "update" && (originalRecord == null || rowNum == null)))
        {
            this._markForRegroup = true;
        // otherwise for other updates see if we need to regroup
        } else if (type == "update") { 
            var currData = this.data;
            if (this.data.isGroupedOutput && this.originalData) currData = this.originalData;
            
            var updatedRecord = currData.get(rowNum);
            // regroup fully if updatedRecord does not exist (was deleted)
            if (updatedRecord == null) this._markForRegroup = true;

            // if primary keys differ, the record was deleted via filtering. regroup
            if (!this._markForRegroup) {
                var pks = (this.dataSource != null 
                        ? this.getDataSource().getPrimaryKeyFieldNames()
                        : []);
                for (var i=0; i<pks.length; i++) {
                    if (originalRecord[pks[i]] != updatedRecord[pks[i]]) {
                        this._markForRegroup = true;
                        break;
                    }
                }
            }

            // regroup if group field values differ
            if (!this._markForRegroup) {
                for (var i = 0; i < groupByFields.length; i++) {
                    var undef, fieldName = groupByFields[i];
                    if (originalRecord[fieldName] !== undef && !this.fieldValuesAreEqual(
                            fieldName, originalRecord[fieldName], updatedRecord[fieldName])) {
                        // XXX incrementalRegroup can handle this case, but more testing is 
                        // necessary. change this post 7.0.
                        // Note, originalRecord must be changed to what node is below:
                        // this.data.find(pk) etc
                        this._incrementalRegroup(updatedRecord, originalRecord, rowNum, 
                              updateData, true);
                        this._ignoreRegroup = true;
                    }
                }
            }

            if (!this._markForRegroup) {
                var pks = this.getDataSource().getPrimaryKeyFieldNames(),
                    keyCriteria = {};
                for (var i = 0; i < pks.length; i++) {
                    keyCriteria[pks[i]] = updatedRecord[pks[i]];
                }
                
                // NOTE: In general, we explicitly do not support composite primaryKeys with 
                // Trees and TreeGrids.  However, ListGrid grouping is implemented by use of 
                // a Tree - "this.data" in the below call is a Tree, not the List or ResultSet
                // holding the grid's real data.  Fortunately, Tree supports a criteria search 
                // of its nodes, so we make use of that with a criteria object that happens to 
                // contain only primary keys
                var node = this.data.findAll(keyCriteria);
                for (var i = 0; i < node.length; i ++) {
                    // if we're grouped by the pk field we are likely to have header nodes
                    // with the same value for this field - only apply updates to the
                    // actual data node we care about!
                    if (node[i]._isGroup) continue;
                // apply all modified fields to the node.
                    isc.addProperties(node[i], updatedRecord);
                }
            }
        }
    }
    
    
    if (this._markForRegroup && !this._savingEdits &&
        // Skip attempting to regroup / reset selection if our resultSet is in mid-fetch
        (!isc.isA.ResultSet(this.data) || this.data.lengthIsKnown())) 
    {
        this._markForRegroup = false;
        this._lastStoredSelectedState = this.getSelectedState(true);
        this.regroup();
        if (this._lastStoredSelectedState) {
            this.setSelectedState(this._lastStoredSelectedState);
            delete this._lastStoredSelectedState;
        }
    }
    
    //>Animation
    // Call finishRowAnimation - will kill any show/hide row animations.
    // These animations assume the data remains constant for the duration
    // of the animation.
    // (No-ops if appropriate)
    if (this.body) this.body.finishRowAnimation();
    //<Animation
    
    // Call _remapEditRows() to ensure that editValues are associated with the (possibly
    // modified) rowNumbers using pointers between record primary key and edit values 
    
    if (!this._savingEdits && !this.suppressEditRowRemap) this._remapEditRows();

    // re-associate embeddedComponents with records which were not previously present in the 
    // cache but are now - set grid._shouldRetainEmbeddedComponents to false to have components
    // removed when their associated records are no longer in the cache.
    if (!isc.isA.ResultSet(this.data) || this.data.lengthIsKnown()) {
        this._remapEmbeddedComponents();
    }

    // if this.alwaysShowEditors is set, and we have data, and we're not currently showing
    // editors, show them now.
    // This handles us getting new data (from a fetch for example)
    if (this._alwaysShowEditors() && !this._editorShowing) {
        this.startEditing(null,null,true,null,true);
    }

    
    var lastRow = this.getTotalRows()-1;
    if (this.body) {
        if (this.body.lastOverRow > lastRow) delete this.body.lastOverRow;
        if (this.body.lastMouseOverRow > lastRow) delete this.body.lastMouseOverRow;
        if (this.body._lastHiliteRow > lastRow) delete this.body._lastHiliteRow;
    }
    if (this._lastRecordClicked > lastRow) delete this._lastRecordClicked;
    
    if (this.hilites) this.applyHilites();
    
    
    if (!this._suppressRedrawOnDataChanged) {
        // recalculate autoFitWidth field widths to fit the new data
        if (resetAutoFitWidths) this.updateFieldWidthsForAutoFitValue(this._$dataChanged);

        this.redrawForDataChanged();
    }

    // recalculate grid summaries
    
    if (this.summaryRow && this.showGridSummary) this.summaryRow._recalculateSummaries();

    // restore the selected state after a grouping operation
    // Skip this if this.data is an unloaded R.S.
    if (this._lastStoredSelectedState && 
        (!isc.isA.ResultSet(this.data) || this.data.lengthIsKnown())) 
    {
        this.setSelectedState(this._lastStoredSelectedState);
        delete this._lastStoredSelectedState;
    }

    // clear the _handlingDataChanged flag
    delete this._handlingDataChanged;
    
},

redrawForDataChanged : function () {
    if (this.frozenBody) this.frozenBody._suppressRecordComponentsUpdate = true;
    this._markBodyForRedraw(this._$dataChanged);
},

// wrap the call out to the dataArrived override point and handle sorter according to
// canSortClientOnly value and current data-state
_dataArrived : function (startRow, endRow) {
    var sortField = this._getSortFieldNum();
    
    if (startRow == 0) {
        this._forceUpdateFieldWidths = true;
    }
    if (sortField != null && sortField != -1) {
        var fieldNum = this.getFieldNum(sortField),
            field = this.getField(fieldNum)
        ;

        if (field && field.canSortClientOnly && !this._canSortData(field)) {
            this._setSortFieldNum(null);

            // tell that toolbar button to unselect / get rid of sort arrow
            if (sortField != null && this.header && isc.isA.Toolbar(this.header)) {
                this.header.deselectButton(sortField);
                var button = this.header.getButton(sortField);
                if (button) button.setTitle(this.getHeaderButtonTitle(button));
            }

            // Get rid of the sort arrow in the sorter
            if (this.sorter && this.sorter.setTitle) this.sorter.setTitle(this.sorter.getTitle());
        }
    }

    if (this.getCurrentCheckboxField()) {
        var cbPos = this.getCheckboxFieldPosition(),
            field = this.getField(cbPos);
        // if we are showing a checkbox select-all header, and we don't have 
        // a full cache, disable the checkbox header and show a hover prompt
        if (isc.ResultSet && isc.isA.ResultSet(this.data) 
            && !this.data.allMatchingRowsCached()) 
        {                                    
            var props = {
                disabled: true,
                showHover: true,
                prompt: this.selection.selectionRangeNotLoadedMessage,
                title: (this.canSelectAll == false || this.selectionType == "single") ? isc.nbsp :
                        this._getCheckboxValueIconHTML(false, true, true, field)
            }
            this.setFieldProperties(cbPos, props);
        // if we now have a full cache, enable the checkbox selectAll header
        } else {
            var props = {
                disabled: false,
                showHover: false,
                prompt: null,
                title: (this.canSelectAll == false || this.selectionType == "single") ? isc.nbsp :
                        this._getCheckboxValueIconHTML(false, true, false, field)
            }
            this.setFieldProperties(cbPos, props);
        }
    }

    
    if (isc.screenReader && this.body != null) {
        if (isc.isA.Tree(this.data)) {
            // in this case we're passed a single param, the parent node.
            var node = startRow;
            if (this.data.isOpen(node) && this.data.hasChildren(node)) {
                var children = this.data.getChildren(node);
                if (children && children.length > 0) node = children[0]
            }
            var rowNum = this.data.indexOf(node);
            // If we don't currently have focus, just remember the native focus row - this means
            // if we're showing a modal prompt / redrawing we should refocus on the right native
            // element...
            this.body._putNativeFocusInRow(rowNum, !this.hasFocus);
        }
    }

    this._markBodyForRedraw();
    
    this.dataArrived(startRow, endRow);
},


// doc'd in registerStringMethods block
dataArrived : function (startRow, endRow) {},

//>	@method listGrid._ignoreData() (A)
//		stop observing methods on data when it goes out of scope
//		called automatically by setData
//	@param	data	(object)		old data to be ignored
//<
_ignoreData : function (data) {
    //>Animation
    // Call finishRowAnimation - will kill any show/hide row animations
    // These animations assume the data remains constant for the duration
    // of the animation.    
    // (No-ops if appropriate)
    if (this.body) this.body.finishRowAnimation();
    //<Animation
	
    if (isc.isA.Tree(this.data)) this.ignore(data, "changeDataVisibility");

	this.ignore(data, "dataChanged");
    
    if (this.isObserving(data, "dataArrived")) {
        this.ignore(data, "dataArrived");
    }
    
	
	if (this.selection) this.selection.deselectAll();
	// NOTE: we don't ignore this.selection.setSelected because
	//			we're re-using the same selection object
},

//>	@method	listGrid.applyFieldDefaults()
//		@group	data
//         Derive default field sizes and formatters where possible, based on schema information.
//<
_generatedFieldNameCount:0,
applyFieldDefaults : function (fields) {
    if (fields == null) return;

	// apply ListGrid-specific defaults, like using toShortDate() for Date fields
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i];

        if (field == null) continue;
        
        // In general we can support un-named fields in dataBoundComponents if there is a dataPath
        if (!this.allowNamelessFields && field[this.fieldIdProperty] == null) {
            if (field.dataPath == null) {
                this.logWarn("unable to process field with no name / dataPath:" +
                             this.echo(field));
                continue;
            }
            // apply an arbitrary name - this gives us a straightforward way to map the
            // field object any generated editor item, etc.
            
            field[this.fieldIdProperty] = "field" + this._generatedFieldNameCount++;
        }
        // default the alignment of each field to left if not specified
        // In RTL mode default to right alignment (so text flows from start of value
        // outward as you'd expect). This means setting the property to "left" if
        // reverseRTLAlign is set, otherwise to "right".
        var defaultAlign = (this.isRTL() && !this.reverseRTLAlign) ? isc.Canvas.RIGHT 
                                                                   : isc.Canvas.LEFT;
            
        // "type" is used for align and default formatting. If we have a display field
        // we're pulling display-values from, use the type specified there rather than
        // on the underlying data values the display values represent.
        var type = this.getFieldDisplayType(field);
        var baseType = (type != null ? isc.SimpleType.getBaseType(type) : null);
        
            
        // note: needs to be first, as "image" type technically inherits from text
        if (isc.SimpleType.inheritsFrom(type, "image")) {
            field._typeFormatter = this._formatImageCellValue; 

    	// Attempt to size columns to fit their content
        } else if (baseType == this._$text) { 

            if (field.width == null && field.length != null) {
                if (field.length < 15 && !field.valueMap) {
                	// use minimal space for small text fields with no value map
                    field.width = field.length * 7;
                }
            } 
 
        } else if (baseType == "integer" || baseType == "float") {
            // align numbers right by default to line up decimal places       
            defaultAlign = (this.isRTL() && !this.reverseRTLAlign)  ? isc.Canvas.LEFT 
                                                                    : isc.Canvas.RIGHT;
            field._typeFormatter = this._formatNumberCellValue;

    	// by default size date columns fields to match the default shortDate format applied
        // to date fields
    	
        } else if (baseType == "date") {
            var canEdit = (this.canEdit == true && field.canEdit != false) ||
                          (this.canEdit != false && field.canEdit == true);
            // If the field has unspecified size, size to accommodate formatted date
            // (or editor if the field is editable)
            field.width = field.width || (canEdit ? 100 : 80);
            // right alignment lines up years if day/month values are numeric and not padded
            defaultAlign = (this.isRTL() && !this.reverseRTLAlign)  ? isc.Canvas.LEFT 
                                                                    : isc.Canvas.RIGHT;
            field._typeFormatter = this._formatDateCellValue;
                        
    	// by default size time columns fields to match the default format applied to time 
        // fields
        } else if (baseType == "time") {
            field.width = field.width || 80;
            field._typeFormatter = this._formatTimeCellValue;
            defaultAlign = (this.isRTL() && !this.reverseRTLAlign)  ? isc.Canvas.LEFT 
                                                                    : isc.Canvas.RIGHT;
                        
        } else if (type == "binary" || type == "blob" || type == "upload" || type == "imageFile") {
            field._typeFormatter = this._formatBinaryCellValue;
        } else if (type == "link") {
            field._typeFormatter = this._formatLinkCellValue;        
        } else if (type == "icon") {
            if (field.width == null && field.autoFitWidth == null) {
                if (this.autoFitIconFields != "none") {
                    field.autoFitWidth = true;
                    field.autoFitWidthApproach = 
                        (this.autoFitIconFields == "title") ? "both" : "value";
                }
            }
            
            // check autoFitIconFields -- if set, set min width to accommodate the
            // icons and set autoFitWidth:true / autoFitWidthApproach such that
            // it'll expand to accommodate the title if appropriate
            if (field.width == null && field.autoFitWidth == null) {
                if (this.autoFitIconFields != "none") {
                    field.autoFitWidth = true;
                    field.autoFitWidthApproach = 
                        (this.autoFitIconFields == "title") ? "both" : "value";
                    // set the default width (min width) to the 
                    // calculated default width
                    
                    field.width = this.getDefaultFieldWidth(field);
                }
            }
            field.align = field.align || "center";

            // install a formatter that will put button.icon into the cell
            field._typeFormatter = this._formatIconCellValue;

            // default title so that icon appears alone (otherwise would default to field name
            // if title was unset) 
            field.title = field.title || isc.nbsp;

        // turn on 'canToggle' for all boolean fields.
        // If 'canEdit' is also set to true these fields will be editable via a single
        // click
        } else if (type == "boolean" || type== "checkbox") {
            if (field.canToggle == null) field.canToggle = true;
        }
         
        // For boolean fields we show checkbox images by default
        // this is handled via the valueIcon system - see getValueIcon(), getValueIconWidth() and
        // showValueIconOnly()
        
        
        // If formatCellValue was passed to us as a string, convert it to a method
        if (field.formatCellValue != null && !isc.isA.Function(field.formatCellValue)) 
            isc.Func.replaceWithMethod(field, "formatCellValue", "value,record,rowNum,colNum,grid");
        
        if (this.showValueIconOnly(field)) {
            defaultAlign = isc.Canvas.CENTER;
            
            // apply the "icon" field logic to fields that show valueIcons - 
            // respect autoFitIconFields here too
            if (field.width == null && field.autoFitWidth == null) {
                if (this.autoFitIconFields != "none") {
                    field.autoFitWidth = true;
                    field.autoFitWidthApproach = 
                        (this.autoFitIconFields == "title") ? "both" : "value";
                    // set the default width (min width) to the 
                    // calculated default width
                    field.width = this.getDefaultFieldWidth(field);
                }
            }
        }
        // TODO: numeric quantities with range validators could be given specific sizes
        if (!field.align) field.align = defaultAlign;
        
        // For fields marked as multiple:true, set the "validateEachItem" flag.
        // This ensures that when validators run in an editable grid we will
        // validate each selected value
        if (field.multiple && field.validateEachItem == null) field.validateEachItem = true;
    }
},

// If this field has a specified display field, pick up the underlying display field object
// (from the optionDataSource if appropriate, otherwise from the grid itself, or the
// underlying DS the grid is bound to).
getDisplayField : function (field) {
    var displayField = field.displayField;
    if (displayField != null) {
        var ods = field.optionDataSource ? isc.DataSource.get(field.optionDataSource) : null;
        if (ods != null) displayField = ods.getField(displayField);
        else {
            displayField = this.getField(displayField);
            if (displayField == null && this.dataSource != null) {
                displayField = this.getDataSource().getField(field.displayField);
            }
        }
    }
    return displayField;
},

// Returns the specified type of the field's displayField, if there is one, otherwise of the
// field itself.
getFieldDisplayType : function (field) {
    var displayField = this.getDisplayField(field),
        type;
    if (displayField != null) type = displayField.type;
    if (type == null) type = field.type;
    return type;
},

// Helper method called on boolean fields to determine whether we should use the
// booleanTrueImage/booleanFalseImage and related settings, or fall back to the general
// valueIcons system, which may show a combination of text and icons
_$boolean:"boolean",
_formatBooleanFieldAsImages : function (field) {
    // If booleanTrue/FalseImage have been set null, always back off to showing text / valueIcons
    if (this.booleanTrueImage == null && this.booleanFalseImage == null) return false;
    
    var type = this.getFieldDisplayType(field),
        baseType = (type != null ? isc.SimpleType.getBaseType(type) : null);
    if (baseType != this._$boolean) return false;
   
    // read as: user has not tried to set valueIcon-specific flags.
    // Also note: it's commonly necessary to set a valueMap with a boolean field in order to
    // allow stored values like YES/NO/null to be mapped to boolean true/false, so a valueMap
    // doesn't indicate an intent to use valueIcons.  If you have a valueMap and there are more
    // values than true/false/unset, you shouldn't declare the field boolean, it should be
    // enum.
    return (!field.suppressValueIcon && field.showValueIconOnly == null &&
             field.valueIcons == null && field.formatCellValue == null);    
},


//>	@method	listGrid.setFieldProperties()
// Dynamically set properties for a particular field. This method will update the 
// fields header-button without having to explicitly reset the fields in the grid.
// <P>
// NOTE: Where explicit setters exist for field properties (such as +link{resizeField()},
// +link{setFieldTitle()}, +link{setFieldIcon()} etc, these should be used instead).
//
// @param	fieldNum (number or String) name of the field, or index.
// @param	properties (Button Properties) properties to apply to the header
// @visibility external
//<
// NOTE: little testing has been done on which properties can actually be set this way
setFieldProperties : function (fieldNum, properties) {
    var field, allFields = this.getAllFields();
    var origField = fieldNum;
    if (isc.isA.Number(fieldNum)) {
        // if an index was passed, use the visible fields
        field = this.getField(fieldNum);
    } else {
        // if a key was passed, use the complete fields to handle hidden columns
        var globalFieldNum = isc.Class.getArrayItemIndex(
                                fieldNum, allFields, this.fieldIdProperty);
        field = allFields[globalFieldNum];
        // map back to the fieldNum within this.fields (not within this.completeFields)
        fieldNum = this.getFieldNum(field);
    }    
    if (!field) return;
    isc.addProperties(field, properties);

    if (this.header != null && this.header.isDrawn()) {
        // getFieldHeader / getLocalFieldNum will account for frozen fields
        var header = this.getFieldHeader(fieldNum),
            headerButton = header.getMember(this.getLocalFieldNum(fieldNum));            
        if (headerButton) headerButton.setProperties(properties);
    }
},

//> @method listGrid.setFieldTitle()
// Change the title of a field after the grid is created.
//
// @param fieldNum (integer or String) name of the field, or index.
// @param title (String) new title
// @visibility external
//<
setFieldTitle : function (fieldNum, title) {
    this.setFieldProperties(fieldNum, {title:title});
},

//> @method listGrid.setFieldHeaderBaseStyle()
// Update the +link{field.headerBaseStyle} for a field within the grid at runtime.
// @param name (String) name of the field.
// @param newStyle (CSSClass) new baseStyle for the field header
// @visibility external
//<
setFieldHeaderBaseStyle : function (name, baseStyle) {
    var field = this.getField(name);
    if (field == null) {
        this.logWarn("setFieldHeaderBaseStyle() unable to find field:" + name);
        return;
    }
    field.headerBaseStyle = baseStyle;
    var button = this.getFieldHeaderButton(this.getFieldNum(field));
    if (button != null) {
    
        // Treat being passed null as an attempt to revert to default base style.
        if (baseStyle == null) {
            var buttonProperties = this.getHeaderButtonProperties();
            if (field.frozen && buttonProperties.frozenBaseStyle) {
                baseStyle = buttonProperties.frozenBaseStyle;
            } else if (buttonProperties.baseStyle) {
                baseStyle = buttonProperties.baseStyle;
            }
        }
        if (baseStyle == null) {
            baseStyle = button.getClass().getPrototype().baseStyle;
        }
        button.setBaseStyle(baseStyle);
    }
},

//> @method listGrid.setFieldHeaderTitleStyle()
// Update the +link{field.headerTitleStyle} for a field within the grid at runtime.
// @param name (String) name of the field.
// @param newStyle (CSSClass) new titleTyle for the field header
// @visibility external
//<
setFieldHeaderTitleStyle : function (name, titleStyle) {
    var field = this.getField(name);
    if (field == null) {
        this.logWarn("setFieldHeaderTitleStyle() unable to find field:" + name);
        return;
    }
    field.headerTitleStyle = titleStyle;
    var button = this.getFieldHeaderButton(this.getFieldNum(field));
    if (button != null) {
        if (titleStyle == null) {
            
            var buttonProperties = this.getHeaderButtonProperties();
            if (field.frozen && buttonProperties.frozenTitleStyle) {
                titleStyle = buttonProperties.frozenTitleStyle;
            } else if (buttonProperties.titleStyle) {
                titleStyle = buttonProperties.titleStyle;
            }
        }
        if (titleStyle == null) {
            titleStyle = button.getClass().getPrototype().titleStyle;
        }
        button.setTitleStyle(titleStyle);
    }
},

//> @method listGrid.setFieldIcon() 
// Change the +link{listGridField.icon} for a field after the grid is created
// @param fieldName (String) field to update
// @param icon (SCImgURL) icon for the field
// @visibility external
//<
setFieldIcon : function (fieldName, icon) {
    var field = this.getField(fieldName);
    this.setFieldProperties(fieldName, {icon:icon});
    if (field && field.type == "icon" && field.cellIcon == null) {
        delete field._iconHTML
        this.body.markForRedraw("Field icon changed");
    }
},

//> @method listGrid.setFieldCellIcon()
// Change the +link{listGridField.cellIcon} for a field after the grid is created
// @param fieldName (String) field to update
// @param cellIcon (SCImgURL) new cellIcon for the field
// @visibility external
//<
setFieldCellIcon : function (fieldName, icon) {
    this.setFieldProperties(fieldName, {cellIcon:icon});
    var field = this.getField(fieldName);
    if (field && field.type == "icon") {
        delete field._iconHTML
        this.body.markForRedraw("Field cell icon changed");
    }
},


// AutoComplete
// --------------------------------------------------------------------------------------------

//> @method listGrid.setAutoComplete()
// Change the autoCompletion mode for the grid as a whole.
//
// @param   newSetting (AutoComplete)  new setting
// @group autoComplete
// @visibility autoComplete
//<
setAutoComplete : function (newSetting) {
    this.autoComplete = newSetting;
},

//> @method listGrid.setFieldAutoComplete()
// Change the autoCompletion mode for an individual field.
//
// @param   newSetting (AutoComplete)  new setting
// @group autoComplete
// @visibility autoComplete
//<
setFieldAutoComplete : function (field, newSetting) {
    field = this.getField(field);
    if (field) field.autoComplete = newSetting;
},

// --------------------------------------------------------------------------------------------

//>	@method	listGrid.showFields()
// Force an array of fields to be shown. This method does not add new fields to the grid,
// it simply changes field visibility. If a field.showIf expression exists, it will be
// destroyed.
// <P>
// Note: for showing multiple fields it is more efficient to call this method than to call
// +link{showField()} repeatedly.
//
// @param	field           (Array of String or Array of ListGridField)	Fields to show.
// @param   [suppressRelayout] (boolean) If passed, don't resize non-explicitly sized columns
//                                       to fill the available space.
// @visibility external
// @example columnOrder
//<
// Actually this is a synonym for showField() - separated for ease of documentation
showFields : function (fields, suppressRelayout) {
    return this.showField(fields,suppressRelayout);
},

//>	@method	listGrid.showField()
// Force a field to be shown. This method does not add new fields to the grid,
// it simply changes field visibility. If a field.showIf expression exists, it will be
// destroyed.
// <P>
// Note: for showing multiple fields it is more efficient to call +link{showFields()} than
// to call this method repeatedly.
//
// @param	field           (field name or ListGridField)	field to show
// @param   [suppressRelayout] (boolean) If passed, don't resize non-explicitly sized columns
//                                       to fill the available space.
// @visibility external
// @example columnOrder
//<
showField : function (fields, suppressRelayout) {

    arguments.__this = this;
    
    if (!isc.isAn.Array(fields)) {
        fields = [fields];
    }
    
    var noFields = true,
        allVisible = true;
    
    var setFieldsCalled = this.completeFields != null,
        mustSetFields = (!setFieldsCalled || this.frozenFields || this._suppressedFrozenFields)
    ;

    for (var i = 0; i < fields.length; i++) {
    
        var field = fields[i],
            fieldObj = field;
        
        // Use getSpecifiedField() to retrieve the fieldObject from the fields / completeFields
        // array.
        // Note that this returns null for an invalid field object / ID
        fieldObj = this.getSpecifiedField(fieldObj);
        
        if (fieldObj == null) {
            fields[i] = null;
            this.logWarn("showField(): unable to find field object for field: " + field
                         + ". Taking no action. Call setFields() to add new fields.")
            continue;
        }
        
        noFields = false;
        
        // -- We always want to clear out any showIf property on the field, as even if the field is
        //    currently being shown, we want the field to continue to be shown from this point on
        //    regardless of any conditions in a showIf property
        if (fieldObj.showIf != null) fieldObj.showIf = null;
        
        if (fieldObj.frozen) mustSetFields = true;
        
        // if this field is in a headerSpan, we need to call setFields() to rebuild it
        if (this.spanMap && this.spanMap[fieldObj.name] != null) mustSetFields = true;

        if (mustSetFields) continue;
        
        // If this.fields contains the object, we can assume it's already visible if we're drawn
        // and will show up when we get drawn otherwise.
        if (this.fields.contains(fieldObj)) {
            fields[i] = null;
            continue;
        }

        // At this point we know we have at least one field that needs to be added
        // to the fields array (was previously hidden)
        allVisible = false;
        // hang onto the "live" fieldObj in the array
        fields[i] = fieldObj;
    }
    
    if (mustSetFields) {
         // Frozen fields: with frozen fields, the partial rebuild attempted below doesn't work.
         // This hasn't been looked into in detail yet.
         this.setFields(this.completeFields || this.fields);
         this.handleFieldStateChanged();
         return;
    }
    
    if (noFields || allVisible) return;
    
    
    // update this.fields
    this.deriveVisibleFields();
    
    // Empty slots may be present due to already visible fields
    fields.removeEmpty();
    
    var shownFieldNums = [],
        foundFields = 0;
        
    // Determine the position of the newly shown fields in our fields array.    
    for (var i = 0; i < this.fields.length; i++) {
        var index = fields.indexOf(this.fields[i]);
        if (index != -1) {
            shownFieldNums[index] = i;
            
            foundFields++;
            // stop when we've figured out the positon of all fields passed in.
            if (foundFields == fields.length) break;
        }
    }
    
    var header = this.header;
    // Update any UI to display the new field, if necessary
    

    // create the header button for the new column
    if (header != null) {
        if (!suppressRelayout) this.header.hPolicy = "fill";
        // undocumented feature - addButtons handles being pased an array of field indices
        // as well as an array of buttons
        this.header.addButtons(fields.duplicate(), shownFieldNums);
    }
    
    // tell the body about the new field
    if (this.body) {
    
        // If we're showing an editor in the new field we need to 
        // - create the form items for it and slot it into the edit form
        // - shift the colNum on other items, and the _editColNum of the grid as a whole.
        if (this._editorShowing) {
       
            var editRowNum = this.getEditRow(),
                record = this.getRecord(editRowNum),
    
                
                editedRecord = this.getEditedRecord(editRowNum),
                
                adjustedEditColNum = false,
                items = this.getEditForm().items,
                liveItemIndex = items.length-1,
                liveItem = items[liveItemIndex],
                itemColNum = liveItem.colNum;
        
            // sort the shownFieldNums
            // this allows us to update the live edit item colNum values easily.
            // Note that it means the shownFieldNums will no longer match up to the fields
            // entries in the shown fields array.
            shownFieldNums.sort();
            
            for (var i = shownFieldNums.length-1; i >= 0; i--) {
                var offset = i+1,
                    fieldNum = shownFieldNums[i],
                    threshold = (fieldNum - i);
                    
                if (!adjustedEditColNum && this._editColNum >= fieldNum) {
                    this._editColNum += offset;
                }
                
                // Update the edit form
                var fieldObj = this.fields[fieldNum],
                    width = this.getEditFormItemFieldWidths(record)[fieldNum],
                    item;
                
                while (liveItem != null && itemColNum >= threshold) {
                    liveItem.colNum += offset;
                    
                    liveItemIndex --;
                    liveItem = (liveItemIndex >= 0) ? items[liveItemIndex] : null;
                    itemColNum = (liveItem != null) ? liveItem.colNum : null;
                }
                
                // at this point we've updated any items with a colNum >= ours
                // Create an item for the newly shown field and slot it into the edit form
                //
                // liveItemIndex will now be set to the slot before the one we're interested in
                // (index of the item that previously matched our colNum, minus 1)
                //
                // Note if we're editing by cell this won't apply since if we're showing an editor
                // the field its showing on must already be visible.
                var drawnRange = this.body.getDrawArea();
                if (!this.editByCell && fieldNum >= drawnRange[2] && fieldNum <= drawnRange[3]) {
                    item = this.getEditItem(fieldObj, record, editedRecord, editRowNum, fieldNum, width);
                }
                if (item != null) {
                    this._editRowForm.addItems([item], liveItemIndex+1);
                }
            }
        }

        this.body.fields = this.normalFields || this.fields;
        this.setBodyFieldWidths(this.getFieldWidths());

        this._remapEmbeddedComponentColumns(this.body);

        // instant redraw rather than markForRedraw because we have to avoid dropping
        // values
        if (this.body.isDrawn()) this.body.redraw("show field");
    }

    // If we're auto-fitting vertically, allow the header to shrink or grow vertically
    // as appropriate
    
    if (this.header && this.autoFitHeaderHeights) {
        this.dropCachedHeaderButtonHeights();
        this._updateHeaderHeight();
    }

    
    // reset the sortFieldNum - we may have added or repositioned the primary sort field
    if (this.sortField != null) {
        this.sortFieldNum = null;
        this.sortFieldNum = this._getSortFieldNum()
    }
    
    // If we have a filterEditor showing, update its fields too
    if (this.filterEditor != null) this.filterEditor.showField(fields, suppressRelayout);

    // update recordSummaries as well as grid/group ones
    this.recalculateSummaries();
    if (this.summaryRow != null && this.showGridSummary) {
        this.summaryRow.showField(fields, suppressRelayout);
    }
    
    // reapply hilites - we don't store hilite information on hidden fields so if
    // hidden fields are displayed we'll need to reapply.
    if (this.hilites) this.applyHilites()

    this.markForRedraw("showField");

    this.handleFieldStateChanged();
},

//>	@method	listGrid.hideFields()
// Force an array of fields to be hidden.
// <P>
// NOTE: If a field.showIf expression exists, it will be destroyed.
// <P>
// When hiding multiple fields, this method should be called rather than
// calling +link{listGrid.hideField()} repeatedly for each field to hide.
//
// @param fields (Array of String or Array of ListGridField) fields to hide
// @param [suppressRelayout] (boolean) if passed, don't relayout non-explicit sized fields 
//                                      to fit the available space
// @visibility external
//<
hideFields : function (fields, suppressRelayout) {
    return this.hideField(fields, suppressRelayout);
},


//>	@method	listGrid.hideField()
// Force a field to be hidden.<br><br>
//
// NOTE: If a field.showIf expression exists, it will be destroyed.
// <P>
// Note also that if multiple fields are to be hidden it is more efficient to 
// call +link{hideFields()} passing in the array of fields to hide rather than to call 
// this method repeatedly.
// 
// @param	field           (field name or ListGridField)	field to hide
// @param [suppressRelayout] (boolean) if passed, don't relayout non-explicit sized fields 
//                                      to fit the available space
// @visibility external
// @example columnOrder
//<
hideField : function (fields, suppressRelayout) {
    arguments.__this = this;
    
    var noFields = true,
        allHidden = true;

    if (!isc.isAn.Array(fields)) {
        fields = [fields];
    }
    // we shuffle some stored colNums around on edit items - store out the index of each field
    // being hidden in an array to make this easier.
    var hiddenFieldNums = [];
    
    // If we need to update the UI, it's more efficient to do it here directly than
    // just calling setFields().
    // However if setFields has never been called we will have to call it
    //
    // Frozen fields: with frozen fields, the partial rebuild attempted below doesn't work.
    // This hasn't been looked into in detail yet.
    

    var mustSetFields = (this.completeFields == null || this.frozenFields);
       
    // If we're showing an editor we need to make certain changes to get rid of the
    // form item, etc.
    var editorShowing = this._editorShowing,
        editRow = editorShowing ? this.getEditRow() : null,
        editCol = editorShowing ? this.getEditCol() : null,
        hidEditCell = false;
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i],
            fieldObj = field;

        // Use getSpecifiedField() to pick up the field object from the completeFields array
        // (or if setFields has never been called, from the fields array).
        // Note - if we're passed an invalid field object, this method returns null.
        fieldObj = this.getSpecifiedField(fieldObj); 
        if (fieldObj == null) {
            this.logWarn("hideField(): unable to find field object for field: " + field
                         + ". Taking no action. To add this field use the setFields() method.");
            fields[i] = null;
            continue;
        }
        noFields = false;
            
        // -- Set showIf to always evaluate to false.
        fieldObj.showIf = this._$false;
        
        // If the field is not currently present in this.fields, we can safely assume it's already
        // hidden. No need to proceed in this case
        if (!this.fields.contains(fieldObj)) {
            fields[i] = null;
            continue;
        }
        allHidden = false;
        
        
        // If we're going to call setFields, we're done - we'll just call that
        // method outside this for-loop to update the completeFields array, edit values, etc.
        if (mustSetFields) continue;
        
        var fieldNum = this.fields.indexOf(fieldObj),
            fieldName = this.getFieldName(fieldNum);
        
        hiddenFieldNums.add(fieldNum);
        
        if (editorShowing) {
            if (editCol == fieldNum) hidEditCell = true;

            // Whatever happens we're going to clear the edit value from the cell being hidden
            // pass the additional 3rd parameter to avoid re-displaying the record's value
            // in the (about to be cleared) cell.
            var focusItem = this.getEditFormItem(fieldName);
            // (Item may not exist due to incremental rendering, non editable fields, editByCell)
            if (focusItem && focusItem.hasFocus) focusItem.blurItem();
            this.clearEditValue(editRow, fieldNum, true);
        }
    }
    // If we were passed an empty array, or an array containing already hidden fields
    // we can bail here.
    if (noFields || allHidden) return;

    // if we have to call setFields, simply call it and allow that method to do the work
    // of updating the fields array etc.
    if (mustSetFields) {
        this.setFields(this.completeFields || this.fields);
        this.handleFieldStateChanged();
        return;
    }
    
    fields.removeEmpty();
    if (editorShowing) {
        if (hidEditCell) {
            // If we're editing by cell, and hiding the current edit cell just kill the edit.
            if (this.editByCell) {
                this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
                editorShowing = false;
            } else {
                // If possible we want to shift edit focus to an adjacent field
                // findNextEditCell() is unaware of the cells we've just hidden so
                // reimplement the relevent part of this method to find the next editable and
                // still visible field. Check for closest field to the left first, then to the right
                var newEditCol = editCol-1,
                    foundEditCol = false;
                while (newEditCol >= 0) {
                    if (!hiddenFieldNums.contains(newEditCol) &&
                        this.canEditCell(editRow, newEditCol) &&
                        this._canFocusInEditor(editRow, newEditCol))
                    {
                        foundEditCol = true;
                        break;
                    }
                    newEditCol--;
                }
                if (!foundEditCol) {
                    newEditCol = editCol +1;
                    while (newEditCol < this.fields.length) {
                        if (!hiddenFieldNums.contains(newEditCol) &&
                            this.canEditCell(editRow, newEditCol) &&
                            this._canFocusInEditor(editRow, newEditCol))
                        {
                            foundEditCol = true;
                            break;
                        }
                        newEditCol++;
                    }
                }
                
                // If we don't have any other editable cells in the row, just cancel the edit
                if (!foundEditCol) {
                    this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
                    editorShowing = false;
                } else {
                    // focus in the adjacent field.
                    
                    this._startEditing(editRow, newEditCol,
                                            !this.getEditForm().hasFocus);
                }
            }
        }
    }
	// update this.fields
    this.deriveVisibleFields();
	// destroy the header button
    var header = this.header;
    if (header != null) {
        // Setting the hPolicy to "fill" will cause the header to relay it's buttons out to
        // fill the available space.
        if (!suppressRelayout) this.header.hPolicy = "fill";
        var buttons = [];
        for (var i = 0; i < hiddenFieldNums.length; i++) {
            var fieldNum = hiddenFieldNums[i];
        
            var button = this.header.getButton(fieldNum);
            buttons[buttons.length] = button;
            if (this.headerMenuButton && this.headerMenuButton.masterElement == button) {
                this.headerMenuButton.depeer();
            }
        }
        // removeButtons actually effects the display passed in, so duplicate it
        this.header.removeButtons(buttons.duplicate());
        buttons.map("destroy");
        
        // If we're auto-fitting vertically, allow the header to shrink or grow vertically
        // as appropriate
        if (this.autoFitHeaderHeights) {
            this.dropCachedHeaderButtonHeights();
            this._updateHeaderHeight();
        }
    }
	// If we're currently showing any edit form items for subsequent columns,
	// we must decrement their 'colNum' properties.
	// do this *before* we redraw the body, as the body redraw relies on these values being
	// accurate to create new items for fields that get shifted into view.
	// Also update the _editColNum if necessary. 
	
	
    var itemsToClear = [];
    if (editorShowing) {
        hiddenFieldNums.sort();
        
        var form = this._editRowForm,
            items = form.getItems(),
            
            itemIndex = items.length-1,
            item = items[itemIndex],
            itemColNum = item.colNum,
            
            adjustedEditColNum = false;
        
        for (var i = hiddenFieldNums.length-1; i >= 0; i--) {
        
            var offset = i+1,
                threshold = hiddenFieldNums[i];

            if (!adjustedEditColNum && this._editColNum > threshold) {
                this._editColNum -= offset;
                adjustedEditColNum = true;
            }
            
            while (item != null && itemColNum >= threshold) {
                if (itemColNum == threshold) itemsToClear.add(item);
                else item.colNum -= offset;
                
                itemIndex --;
                item = (itemIndex >= 0) ? items[itemIndex] : null;
                itemColNum = (item != null) ? item.colNum : null;
            }
        }
    }
	// tell the body about the removed fields
    if (this.body) {
        this.body.fields = this.normalFields || this.fields;
        this.setBodyFieldWidths(this.getFieldWidths());

        this._remapEmbeddedComponentColumns(this.body);
    	// instant redraw rather than markForRedraw because we have to avoid dropping
    	// values
        if (this.body.isDrawn()) this.body.redraw("hide field");
    }

    if (editorShowing && itemsToClear.length > 0) {
    	// If we're currently showing an edit form item for this field, remove it now (already
    	// been cleared from the DOM), and had the edit values cleared.
    	for (var i = 0; i < itemsToClear.length; i++) {
        	// Hide the actual item if there is one.
            var item = itemsToClear[i];
            this._editRowForm.removeItems([item]);
        }
    }
    // reset the sortFieldNum - we may have removed or repositioned the primary sort field
    if (this.sortField != null) {
        this.sortFieldNum = null;
        this.sortFieldNum = this._getSortFieldNum()
    }


	// If we have a filterEditor showing, update its fields too
    if (this.filterEditor != null) this.filterEditor.hideField(fields, suppressRelayout);
    if (this.summaryRow && this.showGridSummary) {
        this.summaryRow._recalculateSummaries();
        this.summaryRow.hideField(fields, suppressRelayout);
    }
    
    //this._remapEmbeddedComponents();
    this.handleFieldStateChanged();
},

//>	@method	listGrid.fieldIsVisible()
// Check whether a field is currently visible
//
// @param	field (String | ListGridField)	field to be checked
// @return (boolean) true if the field is currently visible, false otherwise.
// @visibility external
//<
fieldIsVisible : function (field) {
    var fieldObj = field;
	// If passed a field ID, look for it in the completeFields array rather than the fieldsArray
	// as it is may be defined, but not visible    
    if (!isc.isAn.Object(fieldObj)) fieldObj = this.getSpecifiedField(field);

    return this.fields.contains(fieldObj);
},

// ----------------------------------------------------------------------------
// panelHeader related methods

showActionInPanel : function (action) {
    // specifically add the "editNew" action, which is not added by default
    if (action.name == "editNew") return true;
    return this.Super("showActionInPanel", arguments);
},

//> @method listGrid.getTitleField()
// @include dataBoundComponent.getTitleField() 
//<

//>	@method	listGrid.setFields()  ([A])
// Sets the fields array and/or field widths to newFields and sizes, respectively.<br><br>
//
// If newFields is specified, it is assumed that the new fields may have nothing in common with
// the old fields, and the component is substantially rebuilt.  Consider the following methods
// for more efficient, more incremental changes: resizeField, reorderField, showField,
// hideField, setFieldProperty.
//
// @visibility external
//
// @param	[newFields]		(List of ListGridField)	array of fields to draw
//<
_$setFields:"set fields",
setFields : function (newFields) {

    if (isc._traceMarkers) arguments.__this = this;
    var recalculateSummaries = false;
    // If called with new fields (more than just an update of existing field state), reset
    // the flag indicating that we've used specified width as a minimum for autoFitWidth fields
    if (newFields != null && this.fields != null &&
        this.fields != newFields && this.completeFields != newFields) 
    {
        delete this.fields._appliedInitialAutoFitWidth;
    }
    // Check for any userFormulae / summary fields - we may have to "recalculateSummaries"
    if (isc.isAn.Array(newFields)) {
        for (var i = 0; i < newFields.length; i++) {
            if (newFields[i] && 
                (newFields[i].userFormula || newFields[i][this._$summary])) 
            {
                recalculateSummaries = true;
            }
        }
    }
    
    
    if (!newFields && this.getDataSource() && !this.getDataSource().hasFields()) {
        this.logWarn("ListGrid.setFields() : neither this ListGrid nor its dataSource have fields");    
    }
    // set boolean images to default checkboxItem images if unset
    if (!this.booleanFalseImage && !this.booleanTrueImage && !this.booleanPartialImage) { 
        this.booleanTrueImage = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("checkedImage") : null;
        this.booleanFalseImage = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("uncheckedImage") : null;
        this.booleanPartialImage = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("partialSelectedImage") : null;
        // set imageWidth from checkboxItem.valueIconWidth to avoid images having different sizes
        // when editing and not editing
        
        this.booleanImageWidth = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null;
        this.booleanImageHeight = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("valueIconHeight") : null;
                    
    }

    // if there is a sortFieldNum set, get a pointer to that field
	// we'll check later to reset the sort if we need to
	var sortFieldNum = this._getSortFieldNum(),
        oldSortField = (sortFieldNum != null && this.fields
                        ? this.fields[sortFieldNum] : null);
    
	// Interaction of setFields() with editing:
	// - If we have editValues for any row(s) [Pending unsaved edits], it is possible that
	//   some of the fields for which we have edit values will go away - in this case we need
	//   to drop those edit values, and any validation errors for those fields.
	// - If we are currently showing an editor for some row, we may also need to update the
	//   fields in the edit form.
	// * We'll drop the removed fields' edit values silently, [log at info level only] - this
	//   is expected / acceptable behavior when fields are removed.
    var editorShowing = this._editorShowing,
        editRowNum = this.getEditRow(),
        editColNum = this.getEditCol(),
        editForm = this._editRowForm,
        oldEditFieldName = this.fields ? this.getEditFieldName() : null
    ;
    
	// listGrid.completeFields and listGrid.fields are set to arrays which contain pointers
	// to the same set of "field" objects.
	// - Any fields with a showIf property that evaluates to false will be removed from
	//   this.fields, making it an array of only the list of *visible* fields.
	// - on a reorder or similar manipulation of the set of fields, you should manipulate 
	//   this.completeFields and pass the result to setFields().  Otherwise, you will lose any
	//   fields that aren't visible at the time.
	
    
	

	// on the first setFields() ever, create a new array for the visible fields
    if (this.completeFields == null) this.fields = [];
	// bind the passed-in fields to the DataSource and store
	// canPickOmittedFields - if true we'll pick up all dataSource fields but have them 
	// marked as showIf:false if useAllDataSourceFields is false
    this.completeFields = this.bindToDataSource(newFields, this.canPickOmittedFields);
    if (this.completeFields == null) this.completeFields = [];

    // rowNumberField displaying the current rowNum of each record
    if (this.shouldShowRowNumberField()) {
        var rnField = this.getCurrentRowNumberField(),
            rnPos = this.getRowNumberFieldPosition(),
            shouldAdd = !rnField
        ;
    
        // Exception: If the completeFields passed in (the new fields) contains the
        // special field already, no need to add it.
        
        if (shouldAdd && 
            newFields && newFields.find(this.fieldIdProperty, "_rowNumberField") != null) 
        {
            shouldAdd = false;
        }

        if (shouldAdd) rnField = this.getRowNumberField();

        if (shouldAdd) this.completeFields.addAt(rnField, rnPos);
        else this.completeFields.slideList([rnField], rnPos);
    } else {
        var rnField = this.getCurrentRowNumberField();
        if (rnField) this.completeFields.remove(rnField);
    }

    // checkboxField for selection
    if (this.shouldShowCheckboxField()) {
        var cbField = this.getCurrentCheckboxField(),
            cbPos = this.getCheckboxFieldPosition(),
            shouldAdd = !cbField
        ;
        if (shouldAdd && newFields && newFields.find(this.fieldIdProperty, "_checkboxField")) {
            shouldAdd = false;
        }

        if (shouldAdd) cbField = this.getCheckboxField();
        // make checkboxField frozen if we have any other frozen fields - note autoFreeze: true does this now

        // avoid adding the checkbox field at position 1 when there's nothing at position 0
        // (eg, when showingGroupTitleColumn() returns true, but groupBy() hasn't yet been called)
        if (cbPos > this.completeFields.length) cbPos = this.completeFields.length;

        if (shouldAdd) this.completeFields.addAt(cbField, cbPos);
        else this.completeFields.slideList([cbField], cbPos);
    } else {
        var cbField = this.getCurrentCheckboxField();
        if (cbField) this.completeFields.remove(cbField);
    }
    
    // expansionField for selection
    if (this.shouldShowExpansionField()) {
        var expField = this.getCurrentExpansionField(),
            expPos = this.getExpansionFieldPosition(),
            shouldAdd = !expField
        ;
        if (shouldAdd) expField = this.getExpansionField();
        // make expansionField frozen if we have any frozen fields - note autoFreeze: true does this now
        
        if (shouldAdd) this.completeFields.addAt(expField, expPos);
        else this.completeFields.slideList([expField], expPos);
    } else {
        var expField = this.getCurrentExpansionField();
        if (expField) this.completeFields.remove(expField);
    }

    // Add / update the removeField if this.canRemoveRecords is true
    if (this.shouldShowRemoveField()) {
        var removeFieldNum = this.completeFields.findIndex("isRemoveField", true),
            removeField = (removeFieldNum >= 0) ? this.completeFields[removeFieldNum] 
                                                : {excludeFromState:true, isRemoveField:true};
        
        if (removeFieldNum == -1 && newFields) {
            removeFieldNum = newFields.findIndex("isRemoveField", true);
        }
        if (!removeField._removeFieldInitialized) {
            
            isc.addProperties(removeField, this.removeFieldDefaults, this.removeFieldProperties);
            if (removeField.name == null) removeField.name = "_removeField";
            if (removeField.title == null) removeField.title = this.removeFieldTitle;

            // show removeIcon / unremoveIcon by default.
            // if cellIcon or formatCellValue are explicitly specified on the field
            // don't override those.
            if (removeField.cellIcon == null && removeField.formatCellValue == null) {
                removeField.formatCellValue = function (value, record, rowNum, colNum, grid) {
                    // If this record is explicitly marked to not allow removal show no icon
                    if (grid.recordCanRemoveProperty && record && record[grid.recordCanRemoveProperty] === false) return "&nbsp;";

                    if (!this.removeIconHTML) {
                         this.removeIconHTML = 
                            isc.Canvas.imgHTML(grid.removeIcon, grid.removeIconSize);
                         this.unremoveIconHTML = 
                            isc.Canvas.imgHTML(grid.unremoveIcon, grid.removeIconSize);
                    }
                    if (grid.recordMarkedAsRemoved(rowNum)) {
                        return this.unremoveIconHTML;
                    } else {
                        return this.removeIconHTML;
                    }
                }
                // Store out the iconSize - this makes autoFit more efficient
                removeField.iconSize = this.removeIconSize;
            }
            if (removeFieldNum == -1) {
                this.completeFields.add(removeField);
            }
            
            removeField._removeFieldInitialized = true;
        }
    }

    // If we have headerSpans ensure the order of fields works with the set of spans
    if (this.headerSpans) {
        
        // header spans can span fields that aren't next to each other in the fields array.
        // If this happens we'll need to render them next to each other for the spans to work of course
        this.reorderFieldsForHeaderSpans();
        
        this.buildSpanMap();
    }

    
    // sets things up to look up display maps when LG is drawn
    if (isc.DataSource) this._setOptionDataSources();    

    // set field state if necessary
    if (this.fieldState != null) this.setFieldState();
	// apply various type-based defaults
    this.applyFieldDefaults(this.completeFields);

	// determine which fields should be shown, and add them to the visible fields array
	// (this.fields)
    this.deriveVisibleFields();
    
    // warn about all fields being hidden.  Note it's valid to have no fields if you've created
    // a grid that will be bound to a DataSource or provided fields later.
    if (this.fields.length == 0 && this.completeFields.length > 0) {
        this.logWarn("All specified fields for this component are hidden. Note that fields " +
                     "may be hidden via 'showIf' or 'detail' attribute values. " +
                     "In order to display data this grid must have some visible fields.");
    }

	// wipe out the cached fieldWidths, if any
    this._fieldWidths = null;
    
    var newEditColNum, editFieldStillPresent, hadFocus;
	
    if (editorShowing) {
    	// make sure we get the latest value of the field being edited
        // (Not relevant if we're not drawn since the user can't have updated)
        this.storeUpdatedEditorValue();

    	// assume we should continue editing at the field with the same id
        newEditColNum = this.fields.findIndex(this.fieldIdProperty, oldEditFieldName);
        if (newEditColNum != -1 && !this.canEditCell(editRowNum, newEditColNum)) 
            newEditColNum = -1;
        
    	// if the field with same id isn't editable, find the next editable cell
        if (newEditColNum == -1) {
            var newEditCell;
            // extra param to suppress checking past this row            
            if (!this.editByCell) newEditCell = this.findNextEditCell(editRowNum, 0, 1, 
                                                                      true,true, false, true);
            if (newEditCell != null && newEditCell[0] == editRowNum) {
                newEditColNum = newEditCell[1];
            }

        	// Kill the edit if we're editing by cell, or can't find another editable field
        	// in the edit row.
            if (newEditColNum == -1) {
                this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
                editorShowing = false;
            }
        } else {
        	// field with the same name present and editable: blur and refocus after redraw
            var item = editForm.getItem(oldEditFieldName);
            if (item) {
                hadFocus = item.hasFocus;
                if (hadFocus) editForm._blurFocusItemWithoutHandler();
            }
            editFieldStillPresent = true;
        }
        
    	// Hide the editor if still editing.
    	// Note that this will fire a blur handler unless we have already blurred without 
    	// the handler
    	
        if (editorShowing) this.hideInlineEditor(false, true);
    }
    
    var autoCanFreeze = (this.canFreezeFields == null || this._autoDerivedCanFreeze) &&
                        this.fixedRecordHeights != false && this.fixedFieldWidths != false &&
                        this.autoFitData != "horizontal" && this.autoFitData != "both" &&
                        this.bodyOverflow != "visible";
    if (autoCanFreeze) {
        if (this.completeFields.getProperty("overflow").contains("visible")) autoCanFreeze = false;
    }
    if (autoCanFreeze) {
        this._autoDerivedCanFreeze = true;
        this.canFreezeFields = true;
    } else if (this._autoDerivedCanFreeze) {
        delete this._autoDerivedCanFreeze;
        this.canFreezeFields = null;
    }
	// if we're working with a cellSelection, it needs to know how many columns are visible
    if (this.canSelectCells) this.selection.numCols = this.fields.length;

	// if there is an oldSortField, try to find it in the new fields array
	if (oldSortField) {
        var newSortField = this.fields.indexOf(oldSortField);
        if (newSortField == -1) newSortField = null;
        this._setSortFieldNum(newSortField);
	}

	// if we are showing the header, rebuild it with the new set of fields
    if (this.showHeader && this.headerHeight > 0 && this.header != null) { 
    	// this method will actually update the header to match this.fields
        this.updateHeader();

    	// sync scroll position of new header with current body scroll position
        if (this.body != null) {
            this.syncHeaderScrolling(this.body.getScrollLeft());
        }
    }

    // If we added formula / summary fields, run the "recalculateSummaries" method to
    // apply calculated values to the records if necessary.
    if (recalculateSummaries) {
        this.calculateRecordSummaries(null, true);
    }

    this.updateBody();
  
    // If we have a filterEditor, update its fields
    
    if (this.filterEditor) {
         // update the ds on the filterEditor (and the edit-form) - we need a ds to support some 
        // expression-parsing features, like "matches other field" detecting fields by title as
        // well as name
        this.filterEditor.updateDataSource(this.getDataSource());
        var completeFields = this.completeFields || [];
        this.filterEditor.setFields(completeFields.duplicate());
    }
    
	// if we've rebuilt the header, this will place it and draw it.
	// this will also determine new field widths for the new fields (whether from the header or
	// otherwise)
    this.layoutChildren(this._$setFields);
    
	// Now the fields have been set, update the edit values field:
	// Update our editValues, and validation errors to account for any fields that have 
	// gone from the list.
    if (this._editSessions != null) {
        var fieldsDropped = {};
        
        for (var i in this._editSessions) {  
            if (this._editSessions[i] == null) continue;
            var vals = this._editSessions[i]._editValues,
            	// We want to hang onto primary key values, even if they're not shown in the
            	// ListGrid
                pks = (this.dataSource != null 
                    ? this.getDataSource().getPrimaryKeyFieldNames()
                    : [])
            ;

            for (var currentFieldName in vals) {
            	
                if (!this.fields.containsProperty(this.fieldIdProperty, currentFieldName) &&
                    !pks.contains(currentFieldName)) {
                	// track which fields were dropped so we can inform the user / developer
                    fieldsDropped[currentFieldName] = true;
                	// clearEditValue will clear the editValue and any validation errors for 
                	// the field
                	// Pass the additional 3rd parameter to avoid refreshing the updated cells
                    
                    this.clearEditValue(vals, currentFieldName, true);
                }
            }
        }
        
        fieldsDropped = isc.getKeys(fieldsDropped);
        if (fieldsDropped.length > 0) {
            this.logInfo("'setFields()' removed the following fields which had pending edit " +
                         "values for some row[s]: '" + fieldsDropped.join("', '") + "'.\n" +
                         "Edit values for these fields have been dropped.", "gridEdit");
        }
    }

	// re-show the editor, with the new set of fields
    if (editorShowing) {
    	// if the previous edit field is still showing, just re-show the editor
    	// and focus silently if appropriate.
        if (editFieldStillPresent) {
        	// 2nd and 3rd parameters are both false - even though this field may be in a
            // new position, it's the same logical field, so we don't want to fire
            // editorEnter()
            this.showInlineEditor(editRowNum, newEditColNum, false, false, true);
            if (hadFocus) editForm._focusInItemWithoutHandler(newEditFieldName);
        } else {
        	// If we've killed the previous edit field, but still want to be editing, use
        	// _startEditing() to start editing the new cell to ensure editValues get set up
            // if necessary
            this._startEditing(editRowNum, newEditColNum, !hadFocus);
        }
        
    	// Refocus without firing the handler if the old edit field is still around and had
    	// focus before the setFields.
        if (hadFocus) {
            var newEditFieldName = this.getEditFieldName();
            if (newEditFieldName == oldEditFieldName) {
                editForm._focusInItemWithoutHandler(newEditFieldName);
            } else {
                editForm.focusInItem(newEditFieldName)
            }
        }
    }
    
    // if we've already been sorted and we can sort, sort the data
    var sortFieldNum = this._getSortFieldNum();
	if (sortFieldNum != null && this._canSortData(sortFieldNum)) this.resort();

    if (this.summaryRow && this.showGridSummary) {
        this.summaryRow.setFields(this.completeFields.duplicate());
        this.summaryRow._recalculateSummaries();
    }
    
    // Re-run the logic to apply hilites (that method stores information on record and
    // field objects).
    
    if (this.hilites != null) this.applyHilites();
    
    // Remember the fieldState when setFields is first called.
    // If "setFieldState()" is called with a partial fieldState that doesn't include all fields, 
    // we'll reapply the default field state for
    // any fields that are not explicitly listed in the new state.
    // That is implemented in DBC.setFieldState
    // _suppressStoreFieldState flag used to ensure 'setFieldState()' doesn't also remember
    // that field state as the default
    
    if (!this._suppressStoreFieldState) this.defaultFieldState = this.getFieldState();
},

// override add/removeField to account for us having 'this.completeFields' which
// dataBoundComponent doesn't expect
addField : function (field, index) {
    return this.Super("addField", [field, index, this.completeFields], arguments);
},
removeField : function (field) {
    return this.Super("removeField", [field, this.completeFields], arguments);
},

// Helper method - should this grid show the special checkbox field for 
// selectionAppearance:"checkbox". TreeGrid's show the checkbox in the
// special tree field so we don't show this field.
_$checkbox:"checkbox",
shouldShowCheckboxField : function () {
    if (this.fieldSourceGrid) return this.fieldSourceGrid.shouldShowCheckboxField();
    return (this.selectionAppearance == this._$checkbox &&
        this.selectionType != this._$none &&
        !isc.isA.TreeGrid(this));
},

//> @method listGrid.focusInFilterEditor()
// If the filter editor (+link{listGrid.showFilterEditor}) is visible for this grid, 
// this method will explicitly put focus into the specified field in the filter editor.
// @group filterEditor
// @visibility external
// @param [fieldName] (String) Name of the field to put focus into. If unspecified focus will go
//                             to the first field in the editor
//<
focusInFilterEditor : function (fieldName) {
    if (this.filterEditor == null) return;
    var fieldNum = fieldName != null ? this.getColNum(fieldName) : null;
    this.filterEditor.startEditing(0, fieldNum);
},

//> @method listGrid.filterByEditor()
// If the filter editor (+link{listGrid.showFilterEditor}) is visible for this grid, 
// this method will perform a filter based on the current values in the editor.
// @group filterEditor
// @visibility external
//<
filterByEditor : function () {
    if (this.filterEditor != null) this.filterEditor.performAction();
},


// Override bindToDataSource. Unlike other data-bound widgets if this.showDetailFields is true
// we want to default our detailFields to be hidden (but accessable via the headerContextMenu)
// Note: We do this at init (setFields) time by setting the showIf property on the field,
// rather than overriding fieldShouldBeVisible() to return false for detail fields, so that
// when showField() is called on the field, that method can return true and allow the field
// to show.
bindToDataSource : function (fields, componentIsDetail, a,b,c,d) {
    var numericFieldName = false;

    var completeFields = this.invokeSuper(isc.ListGrid, "bindToDataSource", 
                                          fields, componentIsDetail, a,b,c,d);
    if (this.showDetailFields && completeFields != null) {
        for (var i = 0; i < completeFields.length; i++) {
            var field = completeFields[i];
            if (field.showIf == null && field.detail == true) {
                field.showIf = this._$false;
            }
            
            if (isc.isA.Number(parseInt(field.name)) && 
                parseInt(field.name).toString() == field.name) 
            {
                numericFieldName = true;
            }
            field._isFieldObject = true;
        }
    }
    this._noNumericFields = !numericFieldName;
    return completeFields;
},

// Field State
// --------------------------------------------------------------------------------------------
// The fieldState is an object capturing presentation information about the fields - 
// expected to be used as a way for developers to save the current presentation (EG in cookies)
// and re-load that presentation when the page is reloaded.
// fieldState is an opaque format. 


  
//> @type ListGridFieldState  
// An object containing the stored presentation information for the fields of a listGrid.
// Information contained in a <code>ListGridFieldState</code> object includes the 
// visibility and widths of the listGrid's fields.<br>
// Note that this object is a JavaScript string, and may be stored (for example) as a blob 
// on the server for state persistence across sessions.
// 
// @group viewState
// @visibility external
//<

//>	@method	listGrid.getFieldState() 
// Returns a snapshot of the current presentation of this listGrid's fields as 
// a +link{type:ListGridFieldState} object.
// <P>
// This object can later be passed to +link{listGrid.setFieldState()} to reset this grid's
// fields to the current state.
// <P>
// Note that the information stored includes the current width and visibility of each of this 
// grid's fields, as well as any +link{listGrid.canAddFormulaFields,formula} or
// +link{listGrid.canAddSummaryFields,summary fields} added by the user.
//
// @group viewState
// @see listGrid.setFieldState();
// @visibility external
// @return (ListGridFieldState) current state of this grid's fields.
//<
// LG.getFieldState() moved up to DBC

//>	@method	listGrid.setFieldState() 
// Sets some presentation properties (visibility, width, userFormula and userSummary) of the 
// listGrid fields based on the +link{type:ListGridFieldState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to +link{listGrid.getFieldState()}.
//
// @group viewState
// @param fieldState (ListGridFieldState) state to apply to the listGrid's fields.
// @visibility external
// @see listGrid.getFieldState()
//<
setFieldState : function (fieldState) {
    //!OBFUSCATEOK
    if (this.completeFields == null) this.setFields(this.fields);
    
    if (fieldState == null && this.fieldState != null) {
        if (isc.isA.String(this.fieldState)) { 
            fieldState = this.evalViewState(this.fieldState, "fieldState")
        } else {
            fieldState = this.fieldState;
        }
        this.completeFields = this._setFieldState(fieldState);
        // fieldState is init-only property, so null after use
        this.fieldState = null;
        return;
    }

    fieldState = this.evalViewState(fieldState, "fieldState")
    if (fieldState) {
        this.completeFields = this._setFieldState(fieldState);
        this.refreshFields();
    }
},


handleFieldStateChanged : function (preserveSelection) {
    
    this.fieldStateChanged();
    this.handleViewStateChanged();
    if (this.canSelectCells && !preserveSelection) {
        this.selection.deselectAll();
        if (this.fields && this.fields.length != this.selection.numCols) {
            this.selection.numCols = this.fields.length;
        }
    }
},

// fieldStateChanged implemented in DBC

// Override setHilites to fire the viewStateChanged notification
setHilites : function () {
    this.Super("setHilites", arguments);
    this.handleViewStateChanged();
},

// Override _handleHilitesChanged to fire the hilitesChanged notification.
_handleHilitesChanged : function (hilites) {
    this.Super("_handleHilitesChanged", arguments);
    if (isc.isA.Function(this.hilitesChanged)) this.hilitesChanged();
},

// Override redrawHilites to redraw the body rather than the entire grid.
redrawHilites : function () {
    this._markBodyForRedraw("redrawHilites");
},


// ---------------------------------------------------------------------------------------

//> @type SelectionAppearance
// How data selection should be presented to the user.
// @value "rowStyle" selected rows should be shown with different appearance - see
//                   +link{ListGrid.getCellStyle()} and optionally
//                   +link{ListGrid.selectionCanvas}.
// @value "checkbox" an extra, non-data column should be automatically added to the ListGrid,
//                   showing checkboxes that can be toggled to select rows.  
//                   See +link{listGrid.getCheckboxField()}.
// @visibility external
//<

// defaults for checkbox field
checkboxFieldDefaults: {
    name: "_checkboxField",
    excludeFromState:true,
    canEdit: false,
    shouldPrint:false,
    // Note we could probably allow filtering by this field for client-side grids - which would
    // allow you to view all selected / unselected rows easily, (nice feature) but it wouldn't
    // work for databound grids
    canFilter:false,
    canGroupBy: false,
    canSort: false,
    canExport: false,
    canHide: false,
    canReorder: false,
    canDragResize: false,
    // make this special field canHilite false so we don't see it in HiliteEditors by default
    canHilite: false,
    _isCheckboxField: true,
    type:"boolean",
    showDefaultContextMenu: false,
    hoverHTML: "return null;", // supress hover at row level, only hovering for header
    autoFreeze: true,
    showGridSummary: false,
    showGroupSummary: false,
    summaryValue: "&nbsp;"
},

//> @method listGrid.getCheckboxField()
// Returns the specially generated checkbox field used when +link{selectionAppearance} is
// "checkbox".
// <P>
// Called during +link{setFields()}, this method can be overridden to add advanced dynamic
// defaults to the checkbox field (call Super, modify the default field returned by Super,
// return the modified field).  Normal customization can be handled by just setting
// +link{AutoChild} properties, as mentioned under the docs for +link{listGrid.checkboxField}.
//
// @return (ListGridField)
// @group checkboxField
// @visibility external
//<
getCheckboxField : function () {
    var grid = this,
        cbField = {
        // default the width to the width of the icon plus an arbitrary buffer
            width:this.checkboxFieldHSpace + this._getCheckboxFieldImageWidth(),
            getAutoFreezePosition: function () { return grid.getCheckboxFieldPosition() }
        }
    ;
    isc.addProperties(cbField, this.checkboxFieldDefaults, this.checkboxFieldProperties);

    cbField.title = (this.canSelectAll == false || this.selectionType == "single" ? isc.nbsp :
        this._getCheckboxValueIconHTML(false, true, false, cbField));

    return cbField;
},

//> @attr listGrid.checkboxFieldHSpace (int : 15 : IR)
// How much horizontal space should the +link{getCheckboxField(),checkbox field} leave
// around the checkbox icon when +link{listGrid.selectionAppearance} is set to
// <code>"checkbox"</code>?
// <P>
// The automatically generated checkbox field will be sized to the width of the
// checkbox icon (specified via +link{listGrid.checkboxFieldImageWidth} or
// +link{listGrid.booleanImageWidth}) plus this value.
// @group checkboxField
// @visibility external
//<
checkboxFieldHSpace:15,

getCurrentCheckboxField : function () {
    var fields = this.completeFields || this.fields;
    if (!fields) return null;
    var cbFields = fields.find(this.fieldIdProperty, "_checkboxField");
    return !cbFields ? null : isc.isAn.Array(cbFields) ? cbFields[0] : cbFields;
},

_getCheckboxFieldImageWidth : function () {
    return this.checkboxFieldImageWidth || this.booleanImageWidth ||
            (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null);
},
_getCheckboxFieldImageHeight : function () {
    return this.checkboxFieldImageHeight || this.booleanImageHeight ||
            (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null);
},

//> @method listGrid.isCheckboxField()
// Identifies whether the passed-in field is the specially generated
// +link{listGrid.checkboxField,checkboxField} used when +link{selectionAppearance} is
// "checkbox".  Use this method in your custom event handlers to avoid inappropriately
// performing actions when the checkboxField is clicked on.
// 
// @param field (ListGridField) field to test
// @return (Boolean) whether the provided field is the checkbox field
// @group checkboxField
// @visibility external
//<
isCheckboxField : function (field) {
    if (!field || !field._isCheckboxField) return false;
    else return true;
},

// helper function to get the checkbox field position
// This is a special field which appears "stuck to" the left edge of the grid.
// We have a few other special fields:
// - rowNumbers column
// - groupTitle column when showing group summaries in header
// - record expansion icon column
getCheckboxFieldPosition : function () {
    if (this.fieldSourceGrid) return this.fieldSourceGrid.getCheckboxFieldPosition();
    if (this.selectionAppearance != "checkbox" || isc.isA.TreeGrid(this)) return -1;
    
    var pos = 0;
    if (this.shouldShowRowNumberField()) pos++;
    return pos;
},

//> @type ListGridSelectedState  
// An object containing the stored selection information for a listGrid.
// Note that this object is not intended to be interrogated directly, but may be stored 
// (for example) as a blob on the server for state persistence across sessions.
// 
// @group viewState
// @visibility external
//<
// ListGridSelectedState object is implemented as an array of primaryKeys indicating the 
// selected set of records.

//>	@method	listGrid.getSelectedState() 
// Returns a snapshot of the current selection within this listGrid as 
// a +link{type:ListGridSelectedState} object.<br>
// This object can be passed to +link{listGrid.setSelectedState()} to reset this grid's selection
// the current state (assuming the same data is present in the grid).<br>
// @group viewState
// @see listGrid.setSelectedState();
// @visibility external
// @return (ListGridSelectedState) current state of this grid's selection
//<
getSelectedState : function (supressWarnings) {
    if (!this.selection) return null;
    if (!this.dataSource || 
        isc.isAn.emptyObject(this.getDataSource().getPrimaryKeyFields())) 
    {
        if (!supressWarnings) {
            
            this.logWarn("can't getSelectedState without a DataSource " +
                         "with a primary key field set");
        }
        return null;
    }

    var selectedState = [],
        selection;

    if (this.canSelectCells) {
        selection = this.selection.getSelectedCells();

        // store primary keys and cell-range.  Works only with a DataSource
        for (var i = 0; i < selection.length; i++) {
            var cell = selection[i],
                PKs = this.getPrimaryKeys(cell[0]),
                entry = null
            ;
            for (var k=0; k<selectedState.length; k++) {
                if (this.objectsAreEqual(PKs, selectedState[k].PKs)) {
                    entry = selectedState[k];
                    break;
                }
            }
            if (!entry) {
                selectedState.add({ PKs: PKs, cells: [] });
                entry = selectedState[selectedState.length-1];
            }
            entry.cells.add(cell[1])
        }
    } else {
        selection = this.selection.getSelection() || [];

        // store primary keys only.  Works only with a DataSource
        for (var i = 0; i < selection.length; i++) {
            selectedState[i] = this.getPrimaryKeys(selection[i]);
        }
    }

    return isc.Comm.serialize(selectedState,false);
},


//> @method listGrid.getCellSelection()
// When +link{canSelectCells} is active, returns the +link{CellSelection} object that tracks
// and manages the current selection.  Returns null if +link{canSelectCells} is false.
// @return (CellSelection) current cellSelection
// @visibility external
//<
getCellSelection : function () {
    if (!this.canSelectCells) return null;
    return this.getSelectionObject();
},

//>	@method	listGrid.setSelectedState() 
// Reset this grid's selection to match the +link{type:ListGridSelectedState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to 
// +link{listGrid.getSelectedState()}.
//
// @group viewState
// @param selectedState (ListGridSelectedState) Object describing the desired selection state of
//                                              the grid
// @see listGrid.getSelectedState()
// @visibility external
//<
setSelectedState : function (selectedState) {
    
    selectedState = this.evalViewState(selectedState, "selectedState")
    if (!selectedState || selectedState.length == 0) {
        if (this.selection) this.selection.deselectAll();
        return;
    }
    
    if (selectedState[0].PKs && !this.canSelectCells) {
        if (this.selection) this.selection.deselectAll();
        this.logWarn("Cannot restore CellSelection state to a " + 
            "ListGrid with canSelectCells: false.");
        return;
    } else if (!selectedState[0].PKs && this.canSelectCells) {
        if (this.selection) this.selection.deselectAll();
        this.logWarn("Cannot restore Selection state to a " + 
            "ListGrid with canSelectCells: true.");
        return;
    }

    var selection = this.selection, 
        data = this.originalData || this.data;

    // ignore calls at an invalid time (data still loading or cache just invalidated)
    if (isc.ResultSet && isc.isA.ResultSet(data) && !data.lengthIsKnown()) return;

    if (data && selection) {
        selection.deselectAll();
        
        var records = [];

        for (var i = 0; i < selectedState.length; i++) {
            var item = selectedState[i]
            // resultSet.indexOf() looks up by matching PK values
            var PKs = selectedState[i];

            if (isc.isAn.Object(item) && item.cells != null) {
                // with cell-selection, we store an object with aPKs object an a cells array
                PKs = selectedState[i].PKs;
            }
            var index = data.findByKeys(PKs, this.getDataSource());
            // record may have been removed
            if (index != -1) {
                if (this.canSelectCells) {
                    // build an array of cell-reference arrays in the format [rowNum,colNum]
                    var cells = item.cells;
                    for (var k=0; k<cells.length; k++) {
                        records.add([index, cells[k]]);
                    }
                } else {
                    records.add(data.get(index));
                }
            }
        }
        if (this.canSelectCells) {
            this.selection.selectCellList(records);
        } else {
            this.selection.selectList(records);
        }
    }
},

//> @type ListGridSortState  
// An object containing the stored sort information for a listGrid.
// Note that this object is not intended to be interrogated directly, but may be stored 
// (for example) as a blob on the server for state persistence across sessions.
// 
// @group viewState
// @visibility external
//<
// ListGridSortState object is implemented as a simple JS object containing fieldName and sortDir
// attributes - also now supports the multiSorting subsystem by storing the grid's list of 
// sortSpecifiers


//>	@method	listGrid.getSortState() 
// Returns a snapshot of the current sort state within this listGrid as 
// a +link{type:ListGridSortState} object.<br>
// This object can be passed to +link{listGrid.setSortState()} to reset this grid's sort to
// the current state (assuming the same fields are present in the grid).<br>
// @group viewState
// @see listGrid.setSortState();
// @visibility external
// @return (ListGridSortState) current sort state for the grid.
//<
getSortState : function () {

    if (this.logIsInfoEnabled("sorting")) {
        this.logInfo("\n"+
            "grid.sortFieldNum is: "+this.sortFieldNum+"\n"+
            "grid.sortField is: "+this.sortField+"\n"+
            "grid.getField(grid.sortFieldNum) is:\n"+
                isc.echoAll(this.getField(this.sortFieldNum))+"\n"+
            "-----------------------------------------\n"+
            "grid._getSortFieldNum() is: "+this._getSortFieldNum()+"\n"+
            "grid.getField(grid._getSortFieldNum()) is:\n"+
                isc.echoAll(this.getField(this._getSortFieldNum()))+"\n"+
            "", "sorting"
        )
    }

    var sortFieldNum = this._getSortFieldNum(),
        sortField = (sortFieldNum != null ? this.getField(sortFieldNum) : null),
        sortFieldName = sortField != null ? this.getFieldName(sortField) : null,
        sortDir = this._getFieldSortDirection(sortField),
        sortState = {fieldName:sortFieldName, sortDir:sortDir}
    ;

    if (this._sortSpecifiers && this._sortSpecifiers.length > 0) {
        var specifiers = isc.shallowClone(this._sortSpecifiers);
        // remove some props added during sorting - keeps the output neat and they'll be 
        // re-applied on setSort()
        specifiers.clearProperty("primarySort");
        specifiers.clearProperty("sortIndex");
        specifiers.clearProperty("normalizer");
        specifiers.clearProperty("context");
        sortState.sortSpecifiers = specifiers;
    }

    // eval() of a string containing object literal text will js error - enclose in "(" ... ")" to 
    // avoid this.
    return "(" + isc.Comm.serialize(sortState,false) + ")";
},


//>	@method	listGrid.setSortState() 
// Reset this grid's sort state (sort field and direction or list of 
// +link{sortSpecifier}s) to match the 
// +link{type:ListGridSortState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to 
// +link{listGrid.getSortState()}.
//
// @param sortState (ListGridSortState) Object describing the desired sort state for the grid.
// @group viewState
// @see listGrid.getSortState()
// @visibility external
//<
setSortState : function (state) {
    state = this.evalViewState(state, "sortState")
    if (!state) {
        this.clearSort();
        return;
    }

    if (state.sortSpecifiers) {
        // multisort
        this.setSort(isc.shallowClone(state.sortSpecifiers));
    } else if (state.fieldName == null) {
        this.clearSort();
    } else {
        // will only get here now if the user has legacy state or creates their own state 
        // object which doesn't include sortSpecifiers - a sortSpecifier will be created 
        // anyway and further calls to getSortState() will include it
        var fieldNum = this.getFieldNum(state.fieldName)
        if (fieldNum != -1) this.sort(fieldNum, state.sortDir);
    }
},

//> @type ListGridViewState  
// An object containing the "view state" information for a listGrid.<br>
// This object contains state information reflecting<br>
// - +link{type:ListGridFieldState}<br>
// - +link{type:ListGridSortState}<br>
// - +link{type:ListGridSelectedState}<br>
// for the grid.<br>
// Note that this object is not intended to be interrogated directly, but may be stored 
// (for example) as a blob on the server for view state persistence across sessions.
// 
// @group viewState
// @visibility external
//<
// ListGridViewState object is implemented as a simple JS object containing the following 
// fields:
// - selected [a ListGridSelectedState object]
// - field [a ListGridFieldState object]
// - sort [a ListGridSortState object]

//>	@method	listGrid.getViewState() 
// Returns a snapshot of the current view state of this ListGrid.<br>
// This includes the field, sort, hilite, group, and selected state state of the grid, returned as a
// +link{type:ListGridViewState} object.<br>
// This object can be passed to +link{listGrid.setViewState()} to reset this grid's vew state
// to the current state (assuming the same data / fields are present in the grid).<br>
// @group viewState
// @see type:ListGridViewState
// @see listGrid.setViewState();
// @visibility external
// @return (ListGridViewState) current view state for the grid.
//<    
getViewState : function (returnObject) {

    var state = {
        selected:this.getSelectedState(true),
        field:this.getFieldState(),
        sort:this.getSortState(),
        hilite:this.getHiliteState(),
        group: this.getGroupState()
    };

    // Available so TG can call Super() and get an object back
    if (returnObject) return state;
    return "(" + isc.Comm.serialize(state,false) + ")";
},

//> @type ListGridGroupState  
// An object containing the stored grouping information for a listGrid.
// Note that this object is not intended to be interrogated directly, but may be stored 
// (for example) as a blob on the server for state persistence across sessions.
// 
// @group viewState
// @visibility external
//<
// ListGridGroupState object is implemented as a serialized array of objects specifying
// fieldName, groupingMode, granularity.
// We also support a simple comma separated string (the older format) for backcompat

//>	@method	listGrid.getGroupState() 
// Returns a snapshot of the current grouping state of this ListGrid.<br>
// This object can be passed to +link{listGrid.setGroupState()} to reset this grid's grouping
// to the current state (assuming the same data / fields are present in the grid).<br>
// @group viewState
// @see type:ListGridGroupState
// @see listGrid.setGroupState();
// @visibility external
// @return (ListGridGroupState) current view state for the grid.
//<
getGroupState : function () {
    var i, field, states, groupFields = this.getGroupByFields();
    if (groupFields == null) {
       return "";
    } else {
        states = [];
        for (i = 0; i < groupFields.length; i++) {
            var allFields = this.completeFields || this.fields || [];
            field = allFields.find("name", groupFields[i]);
            if (field) {
                states.add({
                    fieldName : groupFields[i],
                    groupingMode : field.groupingMode,
                    groupGranularity : field.groupGranularity,
                    groupPrecision : field.groupPrecision
                });
            }
        }
    }
    // eval() of a string containing object literal text will js error - enclose in "(" ... ")" to 
    // avoid this.
    return "(" + isc.Comm.serialize(states,false) + ")";

},

//>	@method	listGrid.setGroupState() 
// Reset this grid's grouping to match the +link{type:ListGridGroupState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to 
// +link{listGrid.getGroupState()}.
//
// @group viewState
// @param selectedState (ListGridGroupState) Object describing the desired selection state of
//                                              the grid
// @see listGrid.getGroupState()
// @visibility external
//<
setGroupState : function (state) {
    var i, field, fields;

    if (state) {
        if (state.startsWith("(")) {
            state = this.evalViewState(state, "groupState")
            fields = [];
            for (i = 0; i < state.length; i++) {
                field = this.completeFields.find("name", state[i].fieldName);
                if (field) {
                    fields.push(state[i].fieldName);
                    field.groupingMode = state[i].groupingMode;
                    field.groupGranularity = state[i].groupGranularity;
                    field.groupPrecision = state[i].groupPrecision;
                }

            }

            this.groupBy(fields);
        // Backcompat - handle the state being stored as a comma separated string of fieldNames
        } else {
            this.groupBy(state.split(","));
        }
    } else {
        this.ungroup();
    }
},

//>	@method	listGrid.setViewState() 
// Reset this grid's view state to match the +link{type:ListGridViewState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to 
// +link{listGrid.getViewState()}.
//
// @param viewState (ListGridViewState) Object describing the desired view state for the grid
// @group viewState
// @see listGrid.getViewState()
// @visibility external
//<
setViewState : function (state) {
    state = this.evalViewState(state, "viewState")
    if (!state) return;

    // Order is somewhat important - for example show fields before potentially sorting 
    // by them, etc
    if (state.field) this.setFieldState(state.field);
    this.setSortState(state.sort);
    this.setGroupState(state.group);
    this.setHiliteState(state.hilite);
    this.setSelectedState(state.selected);
},

// Group state changed. Fire both "groupStateChanged" and "viewStateChanged".
// These methods documented under stringMethods section.
handleGroupStateChanged : function () {
    this.groupStateChanged();
    this.handleViewStateChanged();
},
groupStateChanged : function () {},

// Fire the viewStateChanged notification. This is documented in registerStringMethods()

handleViewStateChanged : function () {
    this.fireOnPause("viewStateChangeNotification", {target:this,methodName:"viewStateChanged"}, 0);
},
getViewStateChangedFunction : function () {
    if (this._viewStateChangedFunction == null) {
        var grid = this;
        this._viewStateChangedFunction = function () {
            if (grid.destroyed) return;
            grid.viewStateChanged();
        };
    }
    return this._viewStateChangedFunction;
},
viewStateChanged : function() {
},

//> @method listGrid.setDataSource()
// @include dataBoundComponent.setDataSource()
// @visibility external
//<
// Override setDataSource() - we need to reset the stored filter criteria in case we are
// showing the filterEditor.
setDataSource : function (dataSource, fields) {
    // If we have a specified groupByField, ungroup when changing from one DataSource to another.
    // *Don't ungroup if we didn't have a dataSource before this method ran - this is likely to
    // happen on init when a developer can validly set groupBy before setting dataSource.
    
    var currentDS = this.getDataSource();
    if (currentDS != null && currentDS != dataSource && currentDS.ID != dataSource) {
        var groupByFields = this.getGroupByFields();
        if (groupByFields != null && groupByFields.length > 0 && groupByFields[0] != null &&
            this.originalData != null)
        {
            this.ungroup();
        }
    }

    // discard edits as they don't apply to the new DataSource records
    
    
    this.discardAllEdits();
    // The edit row form has fields as defined in the previous datasource. Drop the form
    // so it will be recreated on the next edit attempt.
    if (this._editRowForm) {
        this._editRowForm.destroy();
        delete this._editRowForm;
    }

    this.Super("setDataSource", arguments);
    this.clearFilterValues();
},


// determine which fields should be shown, and add them to the visible fields array.
// (Used as an internal helper - developers should call 'refreshFields' instead)
deriveVisibleFields : function () {
	// NOTE: we use setArray() so that this.fields remains the same array instance.
    this.fields.setArray(this.getVisibleFields(this.completeFields));
    delete this.fields._appliedInitialAutoFitWidth

    this.deriveFrozenFields();
    this.refreshMasterIndex();
},


//> @method ListGrid.refreshFields
// Re-evaluates +link{ListGridField.showIf} for each field, dynamically showing and 
// hiding the appropriate set of fields
// @visibility external
//<
refreshFields : function () {
    // Just fall through to 'setFields()' for now
	
	this._suppressStoreFieldState = true;
    this.setFields(this.completeFields);
    delete this._suppressStoreFieldState;
},

getFieldWidths : function (reason) {
    // o appropriate time to get field widths:
	// LG w/ header:
	// - initially: right after we draw the header
	// - after setFields() header rebuilt: right after we rebuild the header
	// LG w/o header:
	// - initially: anytime before the body is drawn
	// - after setFields(): anytime before the body is redrawn

	var sizes = this._getCalculatedFieldWidths();
	// When autoFitFieldWidths is true, we may need to tweak these values to either
	// fill the available space, or clip certain fields if we're overflowing the
	// available space.
    if (this.autoFitFieldWidths && !this._calculatingAutoFitFieldWidths) {
        this._calculatingAutoFitFieldWidths = true;
        var unfrozenWidths = sizes.duplicate(),
            frozenWidths = null;
            
        if (this.frozenFields != null) {
            var left = this.freezeStart();
            if (left) {
                frozenWidths = unfrozenWidths.slice(0, this.frozenFields.length);
                unfrozenWidths = unfrozenWidths.slice(this.frozenFields.length);
            } else {
                frozenWidths = unfrozenWidths.slice(-this.frozenFields.length);
                unfrozenWidths = unfrozenWidths.slice(0, -this.frozenFields.length);
            }
        }
        
        var availableSpace = this.getAvailableFieldWidth(true),
            totalSize = unfrozenWidths.sum();

        var unfrozenSpace = availableSpace;
        if (frozenWidths != null) unfrozenSpace -= frozenWidths.sum();

                
        
        // Case 1: the fields don't fill the available space.
        // Expand the autoFitExpandField to fill the available space in the body.
        // Note: We don't auto-expand frozen fields - that would require resizing the
        // frozen body as well. The getAutoFitExpandField() method already handles not
        // returning frozen fields.
        // NOTE: If we're fitting to data, but data is currently loading, don't expand a field
        // now - wait until we redraw with loaded data. Otherwise we don't really know the
        // rendered sizes of all fields, so we won't know how much to expand the expansion field
        // by. Then when data arrives and the other fields all resize, we end up rendering the
        // expansion field potentially too wide since the other fields may now overflow available
        // space.
        
        var validData = true;
        if (this.autoFitWidthApproach != "title") {        
            var dA = this.getDrawArea();
            // drawArea may be null if the body hasn't yet been created.
            if (!this.data || dA == null || Array.isLoading(this.data.get(dA[0]))) {
                validData = false;
            }
        }
        if (totalSize < unfrozenSpace && validData) {
            var expandField = this.getAutoFitExpandField();
            if (expandField) {
                // we want to update the sizes array (includes both frozen and
                // unfrozen fields) so get the global fieldNum for the expand field
                // and update that value.
                var expandFieldNum = this.getFieldNum(expandField);
                var diff = unfrozenSpace - totalSize;
                sizes[expandFieldNum] += diff;
                
                // If we're showing a header for the field we have to resize
                // the button too so it stays in synch with the sizes array we
                // return (which will be applied to the body)
                var button = this.getFieldHeaderButton(expandFieldNum);
                if (button && button.isDrawn()) {
                    button.setWidth(sizes[expandFieldNum]);
                }
            }
        // case 2: the auto-fit fields are overflowing the available space, clip them
        // if appropriate.
        // Note: we don't clip frozen fields - that would require resizing the actual bodies.
        } else if (totalSize > unfrozenSpace && this.autoFitClipFields != null) {
            // If any autoFitFields are marked as clippable, and we're now overflowing
            // horizontally, we want to re-run stretchResize logic ignoring 
            // calculated autoFit for those fields -- just let them take up the available
            // space.
            // If we're not showing a header this is easy
            // If we are (and autoFitWidthApproach includes the header title), we need to
            // have the header re-run it's logic so it takes account of other (non clippable)
            // fields' title-widths
            // This means reset it's policy to fill, resize our fields to "*", then reflow,
            // reset the policy back to whatever it was before (probably "none")
            var policy;
            if (this.header && this.header.isDrawn()) {
                 policy = this.header.hPolicy;
                 this.header.hPolicy = "fill";
            }
            var clipFields = this.autoFitClipFields;
            for (var i = 0; i < clipFields.length; i++) {
                var field = this.getField(clipFields[i]),
                    fieldNum = this.getFieldNum(field);
                if (field == null || fieldNum < 0) continue;
                
                // Don't attempt to clip frozen fields since that would require resizing
               // the frozen body.
                if (!this._suppressedFrozenFields && field.frozen) {
                    this.logInfo("auto-fitting field:" + field.name + 
                        " is present in the autoFitClipFields array for this grid, but is" +
                        " currently frozen. This is not supported - the field will not be clipped.",
                        "frozenFields");
                    continue;
                }

                
                // deleting the calculated autoFitWidth ensures that when 
                // _getCalcualtedFieldWidths runs stretchResizePolicy will simply resize
                // the row to fit if possible, or if a header is showing, the header reflow
                //.will achieve the same result.
                delete field._calculatedAutoFitWidth;
                var header = this.getFieldHeader(fieldNum);
                if (header && header.isDrawn()) {
                    button = header.getMember(this.getLocalFieldNum(fieldNum));
                    
                    button.setWidth(field.width || "*");
                    
                    button.setOverflow("hidden");
                }
            }
            if (this.header && this.header.isDrawn()) {
                var reflowReason = this._$gettingFieldWidths;
                if (reason != null) reflowReason += reason;
                this.header.reflowNow(reflowReason);
                this.header.hPolicy = policy;
                if (this.frozenHeader) {
                    this.frozenHeader.hPolicy = "fill";
                    this.frozenHeader.reflowNow(reflowReason);
                    this.frozenHeader.hPolicy = policy;
                }
            }
//            this.logWarn("after reflow..." + this.header.getMemberSizes());
            
            // rerun the method to get the stretch-resize calculated widths
            // (will run the appropriate logic based on whether or not we're showing
            // headers
            sizes = this._getCalculatedFieldWidths();
        }
        this._calculatingAutoFitFieldWidths = false;
    }
    
//     this.logWarn("getFieldWidths() ultimately gave sizes:" + sizes);
    
    return sizes;
    
},
_$gettingFieldWidths:"Getting listGrid fieldWidths. ",

// helper for getFieldWidths() - returns the stretch-resize calculated widths
// (based on the header if appropriate).
_getCalculatedFieldWidths : function () {
    
	
    

    var header = this.header;
    if (isc.isA.Layout(header) && header.isDrawn()) {
//          this.logWarn("using header-based field widths");
        
        // Force an immediate redraw of any dirty buttons.
        // This is required to ensure sizes are correct -- if redrawOnResize is true for
        // the button or label with overflow:"visible", and setWidth() is called on it
        // the redraw isn't immediate - we need to force a redraw now if it hasn't occurred
        // so getVisibleWidth() returns the new size.
        var buttons = header.members;
        for (var i = 0; i < buttons.length; i++) {
            if (buttons[i].isDirty()) buttons[i].redraw();
            if (buttons[i].label != null && buttons[i].label.isDirty()) buttons[i].label.redraw();
        }

    	// derive field widths from header sizes
        var sizes = header.getMemberSizes();
        if (sizes.length > 0) {
            if (this.allowMismatchedHeaderBodyBorder) {
                
                var leftHeaderBorder = header.getLeftBorderSize() + header.getLeftMargin(),
                    rightHeaderBorder = header.getRightBorderSize() + header.getRightMargin();
                if (leftHeaderBorder != 0) {
                    sizes[0] += leftHeaderBorder;
                }
            }

            var totalSize = sizes.sum(),
                availableSpace = header.getInnerWidth();
           
            if (this.allowMismatchedHeaderBodyBorder) {
                var overflowAmount = totalSize - availableSpace;
                if (overflowAmount > 0) {
                    sizes[sizes.length-1] += Math.min(overflowAmount, rightHeaderBorder);
                }
            }
        }
        if (this.frozenFields) {
            var frozenWidths = this.frozenHeader.getMemberSizes();
            sizes.addListAt(frozenWidths, this.freezeStart() ? 0 : sizes.length);
        }
        
        
    } else {
//         this.logWarn("using stretchResize-based field widths");
        var sizes = this.getStretchResizeWidths();
     
    }
    return sizes;
    
},

getStretchResizeWidths : function () {
    if (this.fields == null) return [];
    
    var widths = this.fields.getProperty("width"),
        autoFitWidths = this.fields.getProperty("_calculatedAutoFitWidth"),
        hasDynamicDefaults = [];
        
    for (var i = 0; i < widths.length; i++) {
        
        if (autoFitWidths[i] != null) {
            
            if (isc.isA.String(widths[i])) {
                hasDynamicDefaults.add(i);
            } else if ((widths[i] == null) || (widths[i] < autoFitWidths[i])) {
                widths[i] = autoFitWidths[i];
            }
        }
    }

    // no header: derive field widths via stretch resize policy on widths set in fields
    var innerWidth = (this.innerWidth != null ? this.innerWidth : this.getAvailableFieldWidth()),
        calculatedWidths = isc.Canvas.applyStretchResizePolicy(
                widths,
                innerWidth,
                this.minFieldSize
                );
    if (hasDynamicDefaults.length > 0) {
        var fieldOverflowed = false;
        for (var i = 0; i < hasDynamicDefaults.length; i++) {
            var j = hasDynamicDefaults[i];
            if (calculatedWidths[j] < autoFitWidths[j]) {
                fieldOverflowed = true;
                widths[j] = autoFitWidths[j];
            }
        }
        if (fieldOverflowed) {
            calculatedWidths = isc.Canvas.applyStretchResizePolicy(
                widths,
                innerWidth,
                this.minFieldSize
            );
        }
    }
    return calculatedWidths;
},

getAvailableFieldWidth : function (specifiedWidth) {
    if (specifiedWidth == null) {
        specifiedWidth = this.autoFitData != "both" && this.autoFitData != "horizontal";
    }

    var width = (!specifiedWidth
                    ? this.getVisibleWidth() : this.getWidth()) - this.getHMarginBorderPad();
	// leave a gap for the scrollbar if vertical scrolling is on or if we're configured to
	// always leave a gap
    var leaveGap = this._shouldLeaveScrollbarGap();
    
    if (leaveGap) {
        width -= this.body ? this.body.getScrollbarSize() : this.getScrollbarSize();
    }
    return Math.max(1, width);
},



//> @method ListGrid.getFieldWidth()
// Returns a numeric value for the width of some field within this listGrid.
// @param fieldNum (Number | String) Index or fieldName of field for which width is to be determined.
// @return (Number) width of the field in px, or null if the width can't be determined
// @visibility external
//<
getFieldWidth : function (fieldNum) {
    fieldNum = this.getFieldNum(fieldNum);
    if (fieldNum == -1 || !this.fields || fieldNum >= this.fields.length) return null;
    var width;
    if (this.body != null) width = this.getColumnWidth(fieldNum);
    if (width == null || isNaN(width)) width = this.getFieldWidths()[fieldNum];
    return width;
},

_adjustFieldSizesForBodyStyling : function (sizes, vertical) {
    if (vertical == null) vertical = false;
    
    if (sizes == null || sizes.length == 0) return sizes;
    
    
    if (!this.body) {
        
        return sizes;
    }
    
    // Adjust sizes of first and last field to account for styling on the body.
    
    sizes[sizes.length -1] = 
        this._adjustLastFieldForBodyStyling(sizes[sizes.length-1], sizes.sum(), vertical);
    sizes[0] = this._adjustFirstFieldForBodyStyling(sizes[0], vertical);
    
    return sizes;
},

_adjustFirstFieldForBodyStyling : function (size, vertical) {
    if (!this.body) return size;
    
    // always knock the left (or top) border off the first field, otherwise everything starts
    // shifted too far right (or down)
    var bodyStartAdjust = (vertical ? this.body.getTopBorderSize() + this.body.getTopMargin()
                                    : this.body.getLeftBorderSize() + this.body.getLeftMargin());
    if (bodyStartAdjust != 0) size -= bodyStartAdjust;
    
    return Math.max(0, size);
},

_adjustLastFieldForBodyStyling : function (size, totalFieldsWidth, vertical) {
    if (!this.body) return size;

    // Figure out whether all the sizes will expand as far or further than the bottom or right
    // edge. If so we want to knock off the end border from the last col or row.
    var bodyEndAdjust = (vertical ? this.body.getBottomBorderSize() + this.body.getBottomMargin()
                                  : this.body.getRightBorderSize() + this.body.getRightMargin());
    
    if (bodyEndAdjust != 0) {
        
        var overflowAmount = totalFieldsWidth - 
            (vertical ? this.body.getInnerHeight() : this.body.getInnerWidth());
        if (overflowAmount > 0) {
            size -= Math.min(overflowAmount, bodyEndAdjust);
        }
    }
    return Math.max(size,1);
},

setBodyFieldWidths : function (sizes) {
	// set the _fieldWidths array to the list passed in
    this._fieldWidths = sizes;
    
    // adjust the first / last column width for any left/right border or margin on the body.
    
    var origSizes = sizes;
    if (this.allowMismatchedHeaderBodyBorder) {
        if (isc.isAn.Array(sizes)) {
            sizes = this._adjustFieldSizesForBodyStyling(sizes.duplicate());
        }
    }

    // if we have frozenFields, separate out the sizes for those columns
    var frozenFields = this.frozenFields;
    if (frozenFields) {
        var frozenWidths = this.getFrozenSlots(sizes);
        sizes = this.getUnfrozenSlots(sizes);
        //this.logWarn("frozen widths: " + frozenWidths + ", remaining widths: " + sizes);
        this.frozenBody.setColumnWidths(frozenWidths);

        var freezeWidth = frozenWidths.sum();
        // this will automatically cause the main body to size to fill remaining space
        this.frozenBody.setWidth(freezeWidth);
         
        this.frozenBody._userWidth = freezeWidth;
        if (this.frozenHeader) this.frozenHeader.setWidth(freezeWidth);
    }

	// give the GridRenderer new fieldWidths
	// will mark the body for redraw.
    if (this.body != null) this.body.setColumnWidths(sizes);
	// If we're showing the edit form, update the widths of the form items
	
    if (this._editorShowing) {
        var items = this._editRowForm.getItems(),
            record = this.getRecord(this.getEditRow()),

        	// This method gives the widths of form items for every column - since
        	// we only create a subset of form items, we won't use every one.
            
            completeFormFieldWidths = this.getEditFormItemFieldWidths(record);

        for (var i = 0; i < items.length; i++) {
            var colNum = items[i].colNum;

            if (items[i].width != completeFormFieldWidths[colNum]) 
                items[i].setWidth(completeFormFieldWidths[colNum]);
        }
    }
},

//>	@type ListGridComponent
// Standard component-type displayed within a ListGrid, as contained by +link{listGrid.gridComponents}.
// @group appearance
// @value "filterEditor" The standard filter-component displayed when +link{listGrid.showFilterEditor}
//   is true
// @value "header" The header-component displayed when +link{listGrid.showHeader} is true.
// @value "body" The body component for the grid.
// @value "summaryRow" The summary-row component displayed when +link{listGrid.showGridSummary} is 
//   true.
// @visibility external
//<

//> @attr listGrid.gridComponents (Array of ListGridComponent : (see below) : IR)
// Array of components that make up this grid. This array controls which standard and/or custom
// parts will be displayed within this ListGrid.
// <P>
// ListGrid is a subclass of +link{VLayout} and consists of a number of member components. The
// standard set of members are automatically generated by the grid, and include (for example)
// the header (a Toolbar of buttons for each field) and the body
// (a GridRenderer displaying the actual data contained in the grid).<br>
// The default value of <code>gridComponents</code> is an Array of +link{ListGridComponent}s listing
// the standard components in their default order:
// <var class="SmartClient">
// <pre>
//    gridComponents : ["filterEditor", "header", 
//                      "body", "summaryRow"]
// </pre>
// </var>
// <var class="SmartGwt">
// <pre>
//    [ListGridComponent.FILTER_EDITOR, ListGridComponent.HEADER, ListGridComponent.BODY, 
//     ListGridComponent.SUMMARY_ROW]
// </pre>
// </var>
//
// You can override <code>gridComponents</code> to change the order of standard components.
// You can also omit standard components this way, although it more efficient to
// use the related "show" property if available (eg +link{showFilterEditor}). Note that
// this array must contain an entry for the <code>"body"</code> - listGrids with no body showing
// are unsupported.<br>
// <i>Advanced note:</i> The live components generated for each of these standard 
// +link{ListGridComponent} types may differ across different listGrids. For example if this
// grid has any +link{listGridField.frozen,frozen fields}, the "body" entry will actually
// be created as an HLayout containing two GridRenderers (one for frozen fields, 
// and one for unfrozen fields). This is really an implementation detail - the "body" entry
// in the gridComponents array simply specifies where the UI for the body should render within
// the ListGrid layout. 
// <P>
// By embedding a Canvas directly in this list you can add arbitrary additional components to the
// listGrid as members, and have them be displayed alongside the standard automatically generated
// parts of the ListGrid. 
// <P>
// Note that having added controls to gridComponents, you can still call APIs directly on
// those controls to change their appearance, and you can also show() and hide() them if
// they should not be shown in some circumstances.
// <P>
// Tip: custom controls need to set layoutAlign:"center" to appear vertically centered.
//
// @visibility external
//<

gridComponents:[
    "filterEditor",
    "header",
    "body",
    "summaryRow"
],

getGridMembers : function () {
    var gridComponents = this.gridComponents,
        members = [],
        bodyShown;
        
    for (var i = 0; i < gridComponents.length; i++) {
        var component = gridComponents[i],
            liveComponent = null;
            
        // allow arbitrary canvii to be shoehorned into the grid.
        if (isc.isA.Canvas(component)) {
            liveComponent = component;

        } else if (isc.isA.String(component)) {
        
            if (!this.shouldShowGridComponent(component)) continue;
            
            // This isn't quite as simple as auto-children -- for example if we have
            // frozen fields we use a Layout to show the 2 bodies - otherwise we simply
            // add the body directly to ourselves as a child.
            switch (component) {
                case "filterEditor" :
                    if (this.filterEditor == null) this.makeFilterEditor();
                    liveComponent = this.filterEditor;
                    break;
                    
                case "header" :
                    if (this.header == null) {
                        this.makeHeader();
                    }
                    liveComponent = this.headerLayout || this.header;
                    break;
                    
                case "body" :
                    bodyShown = true;
                    
                    this.createBodies();
                    liveComponent = this.bodyLayout || this.body;
                    break;
                    
                case "summaryRow" :
                    liveComponent = this.getSummaryRow();
                    break;
                    
//                 default :
//                     this.logWarn("getGridComponents() - Hit default - unrecognized component " + component);
            }
        }
        // Handle being passed anything you could pass to "addChild" (EG "autoChild:foo") by
        // explicitly calling 'createCanvas'.
        if (component != null && liveComponent == null) {
            liveComponent = this.createCanvas(component);
        }
        members.add(liveComponent);
    }
    if (!bodyShown) {
        this.logWarn("ListGrid specified with gridComponents:" + gridComponents + 
            ".  This does not include a \"body\" entry. ListGrids with no body are unsupported," +
            " displaying the body as the last member in the grid.");
        members[members.length] = this.createBodies();
    }
    return members;
},

showComponentPropertyMap:{
    header:"showHeader", 
    filterEditor:"showFilterEditor",
    summaryRow:"showGridSummary"
},
_$body:"body", _$header:"header",
shouldShowGridComponent : function (component) {
    if (component == this._$body) return true;
    if (component == this._$header && this.headerHeight == 0) {
        return false;
    }
    
    var property = this.showComponentPropertyMap[component];
    if (property == null) {
        this.showComponentPropertyMap[component] = property = 
                "show" + component.substring(0,1).toUpperCase + component.substring(1);
    }
    return this[property];
},

// createChildren - builds (or retrieves) our standard set of grid components and adds them 
// to self as members
// Called from prepareToDraw, also from other cases where we need to refresh UI (EG showing/hiding
// filterEditor)

createChildren : function () {
    
    this.updateGridComponents();
    
    
    if (isc.Browser.isMoz && isc.Browser.geckoVersion >= 20051111) {
        if (this.header) {
            this.body.mozOutlineOffset = "0px";
            if (this.body.mozOutlineColor == null) 
                this.body.mozOutlineColor = this.mozBodyOutlineColor;
        } else {
            if (this.body.mozOutlineColor == null) 
                this.body.mozOutlineColor = this.mozBodyNoHeaderOutlineColor;
            this.body.mozOutlineOffset = "-1px";
        }
    }
    
    
    this._useNativeTabIndex = false;
},

// actually creates the standard 'gridComponent' widgets and adds them as members.
// If components are already created they're just retrieved (not clobbered)
// If components are already members, the Layout 'setMembers' code should no-op.

updateGridComponents : function () {
    this.setMembers(this.getGridMembers());
},

// layoutChildren at the Layout level handles positioning and sizing members.
// We have to do a certain amount of tweaking to this, mainly to ensure we calculate field
// widths correctly, and, if we have frozen fields, assign sizes to the frozen header and body
// correctly.

layoutChildren : function (reason,b,c) {
    // If we've created our children, go ahead and call updateFieldWidths() to figure out
    // column sizes, and perform various member-management tasks.
    if (this.body != null) {
        // Note - If the grid as a whole was resized, our fields will potentially reflow. This could
        // change the required header heights in autoFitHeaderHeights mode, requiring
        // us to resize the header bar.
        // We already run the _updateHeaderHeight logic within updateFieldWidths so no
        // need to explicitly do that again here.
        this._updateFieldWidths(reason, b,c);
    }
    
    
    isc.VLayout._instancePrototype.layoutChildren.call(this,reason,b,c);

    if (this.body != null) {
        // Every time we layout children (may be due to body resize or body scroll change), if
        // we're showing the frozen body, ensure it's the correct (explicit) height
        if (this.frozenBody) {
            var adjustForHScroll = this.shrinkForFreeze && this.body.hscrollOn;
            var height = this.bodyLayout.getInnerHeight();
            if (adjustForHScroll) height -= this.body.getScrollbarSize();
            this.frozenBody.setHeight(height);
            if (this.frozenBody._needRowRedraw()) this.frozenBody.markForRedraw("height changed");
            // this will avoid the layout from scrapping this specified height on resize etc
            this.frozenBody._userHeight = height;
        }
    }
},


// override updateMemberTabIndex. If an explicit T.I. is set on the grid just apply it to all
// members

updateMemberTabIndex : function (newMember) {
    // If a developer specifies an explicit tab-index for a Layout as a whole, how to handle this?
    // For now just assign all members the same tab index
    if (this.tabIndex != null && !this._autoTabIndex && this.tabIndex != -1 &&
        // Skip this if we have a CanvasItem
        
        (this.canvasItem == null)) 
    {
        newMember.setTabIndex(this.tabIndex);
        return;
    }
    return this.Super("updateMemberTabIndex", arguments);
},

// If we're auto-fitting to our fields, we may have specified or calculated
// field widths that exceed the specified width for the widget as a whole (availableFieldWidth)
// In this case, use the greater value to size the header.
_updateHeaderWidth : function (fieldWidths, headerWidth, headerLayout) {

    

    // if autoFitMaxColumns is set, then limit the sum to the initially visible
    // columns; this prevents an infinite resizing loop where we try to resize
    // larger and other logic forces us back to a narrower width
    var maxColumns = this.autoFitMaxColumns;
    if (maxColumns != null) {
        if (this.frozenFields == null || this.freezeStart()) {
            fieldWidths = fieldWidths.slice(0, maxColumns);
        } else {
            var nFrozenFields = this.frozenFields.length;
            if (maxColumns < nFrozenFields) {
                fieldWidths = fieldWidths.slice(-nFrozenFields, -nFrozenFields + maxColumns);
            } else {
                var unfrozenFields = fieldWidths.slice(0, maxColumns - nFrozenFields);
                fieldWidths = fieldWidths.slice(-nFrozenFields);
                fieldWidths.addList(unfrozenFields);
            }
        }
    }
    var sum = fieldWidths.sum();

    
    if (this._showSortButton() && !this.body.vscrollOn) sum -= this.getScrollbarSize();

    // if autoFitMaxWidth is set, clip the sum based on its value
    if (this.autoFitMaxWidth != null && sum > this.autoFitMaxWidth) {
        sum = this.autoFitMaxWidth;
    }

    // if width has increased, update header width or resize header layout
    if (sum > headerWidth) {
        if (headerLayout) headerLayout.resizeTo(sum);
        else headerWidth = sum;
    }

    return headerWidth;
},


_updateFieldWidths : function (reason, b,c) {
	// don't do anything until we've created our children (eg resized before draw())
    if (this.body == null) return;
    // don't allow this method to fire recursively
    
    if (this._updatingFieldWidths) return;
    this._updatingFieldWidths = true;

	

    // wipe out fieldWidths on resize so they'll be recalculated.  
	
    this._fieldWidths = null;
    // If any fields has autoFitWidth set to true, calculate the auto-fit size for the
    // column apply it to the field
    // Note that we only care about the cases where we're fitting to the body content - if
    // we're fitting to the header this is handled by simply setting overflow to "visible"
    // on the header button.

    
    if (this.fields && !this.skipAutoFitWidths) {
    
        // The _appliedInitialAutoFitWidth flag allows us to avoid resizing
        // cols to fit content when this is not required. Should be cleared out
        // lazily by cases where we know content has changed and we should
        // reflow to accomodate it.
        if (!this.fields._appliedInitialAutoFitWidth) {
            var autoFitFieldWidths = this.getAutoFitValueWidths(null, true);
            if (autoFitFieldWidths == null) {
                this.fields.setProperty("_calculatedAutoFitWidth", null);
            } else {
                for (var i = 0; i < this.fields.length; i++) {
                    var field = this.fields[i];
                    if (autoFitFieldWidths[i] == null) {
                        field._calculatedAutoFitWidth = null;
                        continue;
                    }
                    var minWidth = this.getMinFieldWidth(field);
                    var headerButton = this.getFieldHeaderButton(i); 
                    if (headerButton != null) {
                       
                        if (minWidth < autoFitFieldWidths[i]) {
                        
                            // update the header if there is one
                            // Note: If autoFitWidthApproach is "both", the header title can
                            // still overflow this new specified size (giving us the desired behavior
                            // of fitting to the larger of the 2 drawn sizes)
                            headerButton.setWidth(autoFitFieldWidths[i]);
                            headerButton.parentElement.reflow();
                        } else {
                            headerButton.setWidth(minWidth);
                            headerButton.parentElement.reflow();
                        }
                        
                        if (headerButton.isDirty()) headerButton.redraw();
                        if (headerButton.label && headerButton.label.isDirty()) {
                            headerButton.label.redraw();
                        }
                        // If the autoFitWidth approach is "both" the header button
                        // may render wider than the specified width (overflow:"visible")
                        // In this case
                        // - we want the calculated autoFitWidth to be wider
                        // - we need to apply that wider size actually to the header button
                        //   (specicified size is used directly in calculations 
                        //   such as getFieldWidths()
                        if (this.getAutoFitWidthApproach(field) == "both") {
                            var headerDrawnWidth = headerButton.getVisibleWidth(),
                                expectedWidth = Math.max(minWidth, autoFitFieldWidths[i]);
                            if (headerDrawnWidth > expectedWidth) {
                                autoFitFieldWidths[i] = headerDrawnWidth;
                                headerButton.setWidth(headerDrawnWidth);
                            }
                        }
                    }
                    
                    field._calculatedAutoFitWidth = autoFitFieldWidths[i];
                    
                    if (minWidth < autoFitFieldWidths[i]) {
                    
                        // update the header if there is one
                        // Note: If autoFitWidthApproach is "both", the header title can
                        // still overflow this new specified size (giving us the desired behavior
                        // of fitting to the larger of the 2 drawn sizes)
                        var headerButton = this.getFieldHeaderButton(i);
                        
                        if (headerButton != null) {
                            headerButton.setWidth(autoFitFieldWidths[i]);
                            headerButton.parentElement.reflow();
                        }
                    } else {
                        var headerButton = this.getFieldHeaderButton(i);
                        if (headerButton != null) {
                            headerButton.setWidth(minWidth);
                            headerButton.parentElement.reflow();
                        }
                    }   
                    
                }
                // Hang a flag on the array to avoid re-calculating the width every time we
                // run stretchResizePolicy, etc
                this.fields._appliedInitialAutoFitWidth = true;
                                
                // Remember how big the grid as a whole is - if it resizes we may need to
                // resize the field that expands to fill available space.
                this._initialAutoFitGridWidth = this.getWidth(); 
                
            }
            
        } else if (this._initialAutoFitGridWidth != this.getWidth()) {
            var expandField = this.getAutoFitExpandField();
            if (expandField) {
                var expandFieldNum = this.getFieldNum(expandField);
                
                delete expandField._calculatedAutoFitWidth;

                // Pick up the autoFitValueWidth for the field in question
                
                var autoFitFieldWidths = this.getAutoFitValueWidths(null, true);
                if (autoFitFieldWidths) {
                    var autoFitValueWidth = autoFitFieldWidths[expandFieldNum];
                    if (isc.isA.Number(autoFitValueWidth)) {
                    
                        var minWidth = this.getMinFieldWidth(expandField);

                        expandField._calculatedAutoFitWidth = autoFitValueWidth;

                        // Note: If autoFitWidthApproach is "both", the header title can
                        // still overflow this new specified size (giving us the desired behavior
                        // of fitting to the larger of the 2 drawn sizes)
                        var headerButton = this.getFieldHeaderButton(expandFieldNum);
                        if (headerButton != null) {
                            headerButton.setWidth(Math.max(minWidth, autoFitValueWidth));
                            headerButton.parentElement.reflow();
                        }
                        this._initialAutoFitGridWidth = this.getWidth();
                    }
                }
            }
        }
    }
    
    var innerWidth = this.getAvailableFieldWidth(true),
        innerWidthChanged = (innerWidth != this.innerWidth);
    this.innerWidth = innerWidth;

//      this.logWarn("total columns width: " + innerWidth + 
//                   (this.body ? ", vscrollon: " + this.body.vscrollOn : ""));
    
    var header = this.header,
        headerHeight = (this.showHeader ? this.getHeaderHeight() : 0);
    if (header != null) {
        // place the header
        // in RTL, scrollbar is on left
        var left = (this.isRTL() && this._shouldLeaveScrollbarGap() ? this.body.getScrollbarSize() : 0);            

    	
        header.hPolicy = "fill";
        if (this.frozenHeader) this.frozenHeader.hPolicy = "fill";
        var autoFitHorizontal = this.autoFitData == "horizontal" || this.autoFitData == "both",
            headerWidth = this.innerWidth,
            fieldWidths;

        
        if (autoFitHorizontal && this._dragResizingField) {
            fieldWidths = this.getFieldWidths(reason + " [sizing horizontal auto-fit header]");
            headerWidth = this._updateHeaderWidth(fieldWidths, headerWidth);
        }
        
        if (!this.leaveScrollbarGap && header.isDrawn() && 
            headerWidth != header.getWidth() && reason == "body scroll changed") {
        	
        	//this.logWarn("header changing size" + this.getStackTrace());
            if (this._settingBodyFieldWidths || !this.resizeFieldsForScrollbar) {    
                header.hPolicy = "none";
            }
        }
        var headerLayout = this.headerLayout || header;
        headerLayout.resizeTo(headerWidth, headerHeight);
        
        headerLayout._userHeight = headerHeight;

        
        if (autoFitHorizontal && !this._dragResizingField) {
            fieldWidths = this.getFieldWidths(reason + " [sizing horizontal auto-fit header]");
            this._updateHeaderWidth(fieldWidths, headerWidth, headerLayout);
        }

    	// if we're in the middle of the initial drawing process, draw the header now so we can
    	// get fieldWidths from it to give to the body before it draws.  Also draw the header
    	// if we're completely drawn and it's undrawn because it was just recreated.
    	
        var isInitialDraw = (reason == "initial draw");
        if (!header.isDrawn() && (isInitialDraw || this.isDrawn())) {
            if (!this.frozenFields) {
                if (isInitialDraw) this._moveOffscreen(header);
                header.draw();
            } else {
                // running getFieldWidths before the header is drawn
                // to determine how much space will be required for the frozen header
                // (see _getCalculatedFieldWidths for explanation)
                if (fieldWidths == null) {
                    fieldWidths = this.getFieldWidths(reason + " [sizing frozen fields]");
                }
                var frozenWidths = this.getFrozenSlots(fieldWidths);
                var frozenWidth = frozenWidths.sum();    
                this.frozenHeader.setWidth(frozenWidth);
                
                this.frozenHeader._userWidth = frozenWidth;
                if (isInitialDraw) this._moveOffscreen(this.headerLayout);
                this.headerLayout.draw()
            }
        }

        if (header.isDrawn()) header.hPolicy = "none";
        if (this.frozenHeader && this.frozenHeader.isDrawn()) this.frozenHeader.hPolicy = "none";

        // ensure the sorter is showing in the right place and visible / hidden as appropriate
        
        if (this.sorter) {
            this.updateSorter();
        }
        
        if (this.autoFitHeaderHeights) {
            var headerHeight = this.getHeaderHeight();
            this.dropCachedHeaderButtonHeights();
            var newHeaderHeight = this.getHeaderHeight();
            if (headerHeight != newHeaderHeight ||
                this.header.getHeight() != newHeaderHeight) 
            {
                this.header.setHeight(newHeaderHeight);      
                this._updateHeaderHeight();
            } else {
                if (this.headerSpans) {
                    this.header._adjustSpans(true);
                }
            }
        }

    }

	// if we haven't figure out field widths, this is the time to get them, since the header
	// has been drawn if we have one, and the body is yet to be drawn.  
	// NOTE: if we are clear()d and then redrawn, and we have a header, we don't want to get
	// sizes from the header while it's undrawn, as it will not run the layout policy while
	// undrawn.
    if ((!this._fieldWidths || innerWidthChanged) &&
        ((header && header.isDrawn()) || headerHeight == 0)) 
    {
        var fieldWidths = this.getFieldWidths(reason);
        this.setBodyFieldWidths(fieldWidths);

        // update the filter editor field widths too [only need to worry about the body]
        if (this.filterEditor && this.filterEditor.body != null) {
            // Duplicate the fieldWidths - we don't want the modifications to this item's array
            // to directly impact the record editor's field widths array
            this.filterEditor.setBodyFieldWidths(fieldWidths.duplicate());
        }
        
        // update the summaryRow (if visible)
        if (this.summaryRow && this.showGridSummary && this.summaryRow.body != null) {
            this.summaryRow.setBodyFieldWidths(fieldWidths.duplicate());
        }
        
       	if (this.logIsDebugEnabled("layout")) {
            this.logDebug("new field widths: " + this._fieldWidths, "layout"); 
        }
    }
    
    // Normally we can rely on the standard layout behavior to size the filter editor.
    // however, if autoFitData is horizontal our content can exceed our specified size, and
    // the layout doesn't automatically handle expanding content to fit an overflowed breadth
    // Therefore in this case explicitly match the filter editor to the body width
    if (autoFitHorizontal && this.filterEditor) {
        this.filterEditor.setWidth(this.body.getVisibleWidth());
    }

    
    this._updatingFieldWidths = false;
},

//> @attr listGrid.autoFitHeaderHeights (boolean : null : IR)
// If this property is set to true, header buttons for either 
// +link{listGrid.fields,fields} or +link{listGrid.headerSpans,header spans} will
// automatically expand to accomodate their titles vertically.
// This means if you have a "tall" title - typically a long string where
// +link{listGridField.wrap} is set to true such that you end up with several lines of
// text - the button will render large enough to accomodate it.
// If necessary this will cause the header for the grid as a whole to expand beyond the
// specified +link{listGrid.headerHeight}.
// @visibility external
//<



// get the "expected" height for field or span header-buttons based on
// this.headerHeight, span.height, etc.
getHeaderButtonDefaultHeight : function (span, recalculate) {
    if (recalculate || span._defaultHeight == null) {
        this.calculateDefaultSpanHeights();
    }
    return span._defaultHeight;
},

// Perform a one-time iteration through all header buttons determining 
// "expected" heights based on header height etc, and cache these values.
calculateDefaultSpanHeights : function () {
    
    if (this.headerSpans == null) {
        // Shouldn't be called if headerSpans is null
        return;
    }
    
    if (this.headerSpans != null) {
        
        var fields = this.getFields(), 
            fieldNames = fields.getProperty("name");
        for (var i = 0; i < this.headerSpans.length; i++) {
            var availableSpace = this.headerHeight,
                span = this.headerSpans[i];
            
            this._calculateDefaultSpanHeight(span, availableSpace, fieldNames);
        }
        for (var i = 0; i < fields.length; i++) {
            if (this.spanMap[fields[i].name] == null) {
                fields[i]._defaultHeight = this.headerHeight;
            }
        }
    }
},

// helper for calculateDefaultSpanHeights()
_calculateDefaultSpanHeight : function (span, availableSpace, fieldNames, isField) {
    if (!isField) {
        // Skip anything that isn't visible
        if (!this._headerSpanVisible(fieldNames, span)) return;
    
        var defaultHeight;
        if (!span._spanAutoSizeHeight) {
            defaultHeight = span.height || this.headerSpanHeight;
        }
        if (defaultHeight == null) {
            var spanDepth = this.getNestedSpanDepth(span);
            // add one to account for field at the bottom
            spanDepth += 1;
            defaultHeight = Math.floor(availableSpace / spanDepth);
        }
        span._defaultHeight = defaultHeight;
        availableSpace -= defaultHeight;
        
        if (span.spans) {
            for (var i = 0; i < span.spans.length; i++) {
                this._calculateDefaultSpanHeight(span.spans[i], availableSpace, fieldNames);
            }
        } else {
            for (var i = 0; i < span.fields.length; i++) {
                // skip hidden / missing fields
                var field = this.getField(span.fields[i]);
                if (field) {
                    this._calculateDefaultSpanHeight(field, availableSpace, fieldNames, 
                        true);
                }
            }
        }
    } else {
        // not visible? Just bail
        if (!fieldNames.contains(span.name)) return;
        
        span._defaultHeight = availableSpace;
    }
},

getHeaderButtonMinHeight : function (field, recalculate) {
    if (!recalculate && field._calculatedMinHeight != null) {
        return field._calculatedMinHeight;
    }
    return this.getHeaderButtonMinHeights([field], recalculate)[0];
},

// Use an offscreen tester to figure out how much space is required for 
// a field, or span header button based on its specified title.
// Caches the value for re-use
getHeaderButtonMinHeights :function (fields, recalculate) {
    var mustCalculate = [];
    var heights = [];
    var testHTML = "";
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i];
        if (!recalculate && field._calculatedMinHeight != null) {
            heights[i] = field._calculatedMinHeight;
        } else {
            var isSpan = !this.fields.contains(field);
            var width;

            if (isSpan) {
                var spannedFields = this.getSpannedFields(field),
                    width = 0;
                //this.logWarn("fields:" + spannedFields);
                for (var ii = 0; ii < spannedFields.length; ii++) {
                    var spannedField = this.getField(spannedFields[ii]);
                    if (spannedField == null) {
                        continue;
                    }
                    width += this.getFieldWidth(spannedFields[ii]);
                }

            } else {
                width = this.getFieldWidth(field);
            }

            // XXX This basically matches framework behavior if we
            // don't see a useful numerical width. May want to revisit.
            if (isNaN(width)) {
                heights[i] = 1;
            } else {
                
                var baseStyle = field.headerBaseStyle,
                    buttonProperties = this.getHeaderButtonProperties();
                if (baseStyle == null) {
                    if (field.frozen && buttonProperties.frozenBaseStyle) {
                        baseStyle = buttonProperties.frozenBaseStyle;
                    } else if (buttonProperties.baseStyle) {
                        baseStyle = buttonProperties.baseStyle;
                    }
                }
                var titleStyle = field.headerTitleStyle;
                if (titleStyle == null) {
                    if (field.frozen && buttonProperties.frozenTitleStyle) {
                        titleStyle = buttonProperties.frozenTitleStyle;
                    } else if (buttonProperties.titleStyle) {
                        titleStyle = buttonProperties.titleStyle;
                    }
                }
                
                if (baseStyle == null || titleStyle == null) {
                    var buttonConstructor = this.headerButtonConstructor ||
                                this.header ? this.header.buttonConstructor 
                                            : isc.Toolbar.getPrototype().buttonConstructor;
                    if (isc.isA.String(buttonConstructor)) {
                        buttonConstructor = isc[buttonConstructor];
                    }
                    if (baseStyle == null)  {
                        baseStyle = buttonConstructor.getPrototype().baseStyle;
                    }
                    if (titleStyle == null) {
                        titleStyle = buttonConstructor.getPrototype().titleStyle;
                    }
                }

                var title = isSpan ? field.headerTitle || field.title
                                 : this.getHeaderButtonTitle(field);
                var oldConfig = field._calculatedHeightConfig;

                if (oldConfig &&
                    oldConfig.baseStyle == baseStyle &&
                    oldConfig.titleStyle == titleStyle &&
                    oldConfig.width == width &&
                    oldConfig.wrap == field.wrap &&
                    oldConfig.title == title)
                {
                    field._calculatedMinHeight = oldConfig.height;
                    heights[i] = oldConfig.height;
                }
            }
        }
        if (heights[i] != null) continue;

        // At this point we want to actually measure the
        // height (and cache the result).

        var config = {};
        config.width = width;
        config.baseStyle = baseStyle;
        config.titleStyle = titleStyle;
        config.wrap = field.wrap;
        config.title = title

        mustCalculate.add({
            index:i,
            config:config
        });

        testHTML += "<div style='position:absolute;width:" + width + "px;'><div class=" +
             (config.titleStyle || config.baseStyle) +
             (config.wrap ? ">" : " style='white-space:nowrap;'>") +
             title +
             "</div></div>"
    }
    if (mustCalculate.length == 0) return heights;

    if (isc.ListGrid.headerHeightTester == null) {
        isc.ListGrid.headerHeightTester = isc.Canvas.create({
            overflow:"hidden",
            top:-1000,
            autoDraw:true,
            contents:testHTML
        });
    } else {
        isc.ListGrid.headerHeightTester.setContents(testHTML);
    }
    // immediate redraw (setContents does a delayed redraw)
    if (isc.ListGrid.headerHeightTester.isDirty()) isc.ListGrid.headerHeightTester.redraw();
    
    var handle = isc.ListGrid.headerHeightTester.getHandle(),

    childNodes = handle.childNodes;
    if (childNodes.length != mustCalculate.length) {
        
    }
    for (var i = 0; i < childNodes.length; i++) {
        var div = childNodes[i];

        var config = mustCalculate[i].config,
            index = mustCalculate[i].index,
            field = fields[index];
            
        config.height = div.offsetHeight;
        field._calculatedHeightConfig = config;
        field._calculatedMinHeight = config.height;

        heights[index] = config.height;
    }
    return heights;
},

// Drops the cached expected (default) and min-heights for all fields / header spans
dropCachedHeaderButtonHeights : function () {
    for (var i = 0; i < this.fields.length; i++) {
        this.fields[i]._defaultHeight = null;
        this.fields[i]._calculatedMinHeight = null;
        if (this.spanMap) {
            var span = this.spanMap[this.fields[i].name];
            while (span != null) {
                span._defaultHeight = null;
                span._calculatedMinHeight = null;
                span = span.parentSpan;
            }
        }
    }
},

// getHeaderHeight() - if autoFitting headers vertically, this method will calculate
// the required height for the header-bar as a whole and return it
// (Otherwise returns specified header height).
getHeaderHeight : function () {
    if (this.headerHeight == 0 || !this.showHeader) return 0;
    if (!this.autoFitHeaderHeights) return this.headerHeight;
    
    var minHeight = 0;
    if (this.headerSpans == null) {
        minHeight = this.headerHeight;
        var fields = this.getFields();        

        var fieldHeights = this.getHeaderButtonMinHeights(fields);
        minHeight = Math.max(minHeight, fieldHeights.max());

    } else {

        var fields = this.getFields(),
            fieldColHeight = 0;
        
        // We've found that redraws are a significant bottleneck and that
        // redrawing the offscreen header height tester for every field and every
        // span is expensive.
        // Cut this down by passing all fields into getMinHeights... then all
        // spans (two redraws total rather than one per button)
        var fieldMinHeights = this.getHeaderButtonMinHeights(fields),
            spans = [];
            
        for (var i = 0; i < fields.length; i++) {
            if (this.spanMap[fields[i].name]) {
                var span = this.spanMap[fields[i].name];
                while (span) {
                    if (!spans.contains(span)) {
                        spans.add(span);
                    }
                    span = span.parentSpan;
                }
            }
        }
        
        
        var spanMinHeights = this.getHeaderButtonMinHeights(spans);
        for (var i = 0; i < fields.length; i++) {
            fieldColHeight = Math.max(
                                this.getHeaderButtonDefaultHeight(fields[i]),
                                this.getHeaderButtonMinHeight(fields[i])
                             );
            if (this.spanMap[fields[i].name]) {
                var span = this.spanMap[fields[i].name];
                while (span) {

                    fieldColHeight +=  Math.max(
                                        this.getHeaderButtonDefaultHeight(span),
                                        this.getHeaderButtonMinHeight(span)
                                       );
                    span = span.parentSpan;
                }
            }

            minHeight = Math.max(minHeight, fieldColHeight);
        }

    }
    return Math.max(minHeight, this.headerHeight);
},

// getAutoFitValueWidths: Calculates the auto-size width based on rendered field values
// Returns a sparse array of field sizes indexed by fieldNum in this.fields.
// By default expects no params and returns only auto-size width of the fields marked with
// autoFitWidth:true.
// If fields are passed in directly we'll calculate sizes for those fields only (regardless of
// any per-field autoFitWidth settings)
getAutoFitValueWidths : function (fields, checkApproach) {
    if (this.data == null) return;
    if (isc.isA.ResultSet(this.data)) {
        var drawArea = this.body.getDrawArea();
        if (!this.data.rangeIsLoaded(drawArea[0], drawArea[1])) {
            // If we're autofitting all fields, and data is loading, mark for resize
            // again - will occur on redraw from dataChanged/dataArrived
            if (fields == null) {
                this.updateFieldWidthsForAutoFitValue("Delayed resize pending data load");
                
            }
            return;
        }
    }
    
    var noAutoFit = (fields == null);
    if (fields == null) {
        fields = [];
        for (var i = 0; i < this.fields.length; i++) {
            var field = this.fields[i];
            if (this.shouldAutoFitField(field)) {
                // checkApproach - if passed don't collect valueWidths unless the field's
                // autoFitWidthApproach is 'value' or 'both'
                if (!checkApproach || this.getAutoFitWidthApproach(field) != "title") {
                    fields.add(field);
                    noAutoFit = false;
                }
            }
        }
    }
    if (noAutoFit) return [];
    
    // widths will be a sparse array containing just the calculated field widths
    var widths = [],
        checkAutoFitWidths = false,
        frozenAutoFitFields = [], frozenAutoFitColNums = [],
        autoFitFields = [], autoFitColNums = [];
    
    for (var i = 0; i < this.fields.length; i++) {
        if (fields.contains(this.fields[i])) { 
            var field = this.fields[i];
            // call getDefaultFieldWidth per field with the special flag to
            // suppress checking autoFitWidths
            // If we get back any nulls (indicating we hit this) pass these to
            // the getColumnAutoSize() method directly.
            // It's more efficient to pass a number of cols to that method 
            // at once, but we also want a custom "getDefaultFieldWidth" implementation
            // to behave as expected so it's a viable override point for customizing this
            // behavior.
            var defaultWidth = this.getDefaultFieldWidth(this.fields[i], true);
            if (defaultWidth == null) {
                checkAutoFitWidths = true;
                if (field.frozen) {
                    frozenAutoFitFields.add(this.getLocalFieldNum(i));
                    frozenAutoFitColNums.add(i);
                } else {
                    
                    autoFitFields.add(this.getLocalFieldNum(i));
                    autoFitColNums.add(i);
                }
            } else {
                widths[i] = defaultWidth;
            }
        }
    }
    if (checkAutoFitWidths) {

        if (frozenAutoFitFields.length > 0) {
            var bodyWidths = this._getBodyColumnAutoSize(true, frozenAutoFitFields);
            
            
            if (bodyWidths != null) {          
                for (var i = 0; i < bodyWidths.length; i++) {
                    widths[frozenAutoFitColNums[i]] = bodyWidths[i];
                }
            }
        }

        if (autoFitFields.length > 0) {
            var bodyWidths = this._getBodyColumnAutoSize(false, autoFitFields);
            if (bodyWidths != null) {          
                for (var i = 0; i < bodyWidths.length; i++) {
                    widths[autoFitColNums[i]] = bodyWidths[i];
                }
            }
        }
    }
    return widths;
},

// Helper to call gridRenderer logic to get auto-fit widths for a number of fields.
_getBodyColumnAutoSize : function (frozen, localFieldNums) {
    var body = frozen ? this.frozenBody : this.body;
    if (body == null) return null;
    var bodyWidths = body.getColumnAutoSize(localFieldNums);
    if (this.includeGridSummaryInAutoFitWidth && 
        this.showGridSummary && this.summaryRow != null) 
    {
        var summaryBody = (frozen ? this.summaryRow.frozenBody : this.summaryRow.body);
        if (summaryBody != null) {
            var summaryWidths = summaryBody.getColumnAutoSize(localFieldNums);
            if (summaryWidths != null) {
                if (bodyWidths == null) bodyWidths = [];
                for (var i = 0; i < summaryWidths.length; i ++) {
                    if (bodyWidths[i] == null || 
                        summaryWidths[i] > bodyWidths[i]) bodyWidths[i] = summaryWidths[i];
                }
            }
        }
    }
    return bodyWidths;
},

//> @method listGrid.getDefaultFieldWidth()
// Method to calculate and return the default width of a field. This method is called
// to calculate the size of each field's content as part of the 
// +link{listGrid.autoFitFieldWidths,field auto fit} behavior.
// <P>
// The default implementation varies by +link{ListGridFieldType,field type}. 
// For fields of type <code>"icon"</code>, or fields which show only a 
// +link{ListGridField.valueIcons,valueIcon} as a value, and for boolean fields which
// show a checkbox value, the width will be calculated based on the icon size and
// +link{listGrid.iconFieldPadding}. For other fields the width will be calculated
// based on the rendered width of content. Note that for 
// <code>"image"</code> type fields, this method will rely on the +link{listGridField.imageWidth}
// being specified.
// <P>
// Note that this width is the default width of "content" - it does not take into account
// the rendered size of the field title.
//
// @param field (ListGridField) Field for which the size should be determined
// @return (int) default size required for the field's content.
//
// @visibility external
//<
 
iconPadding:2,
getDefaultFieldWidth : function (field, suppressAutoFitWidths) {

    // special cases where we can avoid writing out and measuring content    
    if (field.type == "icon" && (field.iconWidth != null || field.iconSize != null)) {
        return (field.iconWidth || field.iconSize) + 2*this.cellPadding + 2*this.iconPadding;
    } else if (this.showValueIconOnly(field)) {
        return this.getValueIconWidth(field) + (2* this.cellPadding) +
                              this.getValueIconRightPadding(field) + 
                              this.getValueIconLeftPadding(field);
    }
    
    if (suppressAutoFitWidths) return null;

    
    var fieldNum = this.getFieldNum(field),
        localFieldNum = this.getLocalFieldNum(fieldNum);
    var widths = this._getBodyColumnAutoSize(field.frozen, [localFieldNum]),
        width = widths ? widths[0] : null;
    if (width == null) width = field.width;
    return width;
},

getAutoFitWidthApproach : function (field) {
    if (field.autoFitWidthApproach != null) return field.autoFitWidthApproach;
    return this.autoFitWidthApproach;
},
    
draw : function (a,b,c,d) {
    if (isc._traceMarkers) arguments.__this = this;
	if (!this.readyToDraw()) return this;

    // create children and set up fields if not already set up
    this.prepareForDraw();

	// call the superclass draw routine to do the actual drawing
	this.invokeSuper(isc.ListGrid, "draw", a,b,c,d);

	// get the actual rendered sizes of the columns
	this.body.getColumnSizes();

    
    for (var i = 0; i < this.bodies.length; i++) {
        var body = this.bodies[i];
        if (body._embeddedComponents) this.markForRedraw();
    }

	// if the sortFieldNum is specified, tell the header about it
    var sortFieldNum = this._getSortFieldNum();
	if (this.header && sortFieldNum != null) {
        var header = this.getFieldHeader(sortFieldNum),
            sortButton = this.getFieldHeaderButton(sortFieldNum);
        header.selectButton(sortButton);
    }

	// scroll the header along with the list (necessary for textDirection == RTL)
    this.bodyScrolled(this.body.getScrollLeft(), this.body.getScrollTop());
    
    
    if (this.sorter) this.updateSorter();
    
    // If we have embedded components ensure that our initial auto-fit widths take
    // their size into account.
    if (this.body._embeddedComponents || 
        (this.frozenBody && this.frozenBody._embeddedComponents)) 
    {
        this.updateFieldWidthsForAutoFitValue("Initial draw with embeddedComponents");
    }
    return this;
},

// should we show inactive editors for every record - returns true if we have any data and
// alwaysShowEditors is true at the grid or field level [can check specific fields]
_alwaysShowEditors : function (field, ignoreTotalRows) {
    // ignoreTotalRows -- useful to test whether we would ever show editors in every cell
    // as opposed to whether we currently do.
    
    // if we have no data don't show any edit cells
    if (!ignoreTotalRows && this.getTotalRows() == 0) return false;
    
    if (this.alwaysShowEditors) return true;
    var fields = field ? [field] : this.fields;
    if (fields) {
        for (var i = 0; i < fields.length; i++) {
            if (fields[i].alwaysShowEditors) {
                return true;
            }
        }
    }
    return false;
},

prepareForDraw : function () {
	

	// call setFields() for the first time, if it hasn't already been done 
    if (this.completeFields == null) this.setFields(this.fields);

    // if alwaysShowEditors is true, ensure we are editing
    if (this.getEditRow() == null) {
        var forceEditing = this._alwaysShowEditors();
        if (forceEditing) this.startEditing(null,null,true,null,true);
    }

    // if a grouping is already set, apply it with groupBy
    if (this.groupByField) {
        var fields;
        if (isc.isA.Array(this.groupByField)) {
            fields = this.groupByField;
        } else {
            fields = [ this.groupByField ];
        }            
        this.groupByField = null;

        // if we have no fields don't call groupBy yet - should occur on setFields
        //if (this.fields != null) 
        this.groupBy(fields);
    }

	// create the header and body. 
    this.createChildren();
},

//> @method listGrid.getGroupByFields()
// Get the current grouping of this listGrid as an array of fieldNames.
// <P>
// This method returns an arry containing the names of the field(s) by which this
// grid is grouped (either from +link{listGrid.groupByField} having been explicitly set or
// from a call to +link{listGrid.groupBy()}). If this grid is not currently grouped, this method
// will return null.
// @return (Array of String) Current grouping for this grid. If grouped by a single field an array
//  with a single element will be returned.
// @visibility external
//<

getGroupByFields : function () {
    var fields = this.groupByField;
    if (fields != null && !isc.isAn.Array(fields)) {
        fields = [fields];
    }
    return fields;
},

destroy : function (indirectDestroy) {
    if (this._dragLine) {
        this._dragLine.destroy();
        this._dragLine = null;
    }
    if (this._cornerMenu) this._cornerMenu.destroy();
    if (this._spanContextMenu) this._spanContextMenu.destroy();
    if (this.cellContextMenu) this.cellContextMenu.destroy();
    if (this._editRowForm) {
        this._editRowForm.destroy();
        delete this._editRowForm;
        delete this._editorShowing;
    }

	if (this.data){
        // if the data was autoCreated, destroy it to clean up RS<->DS links
        if (this.data._autoCreated && isc.isA.Function(this.data.destroy)) {
            this.data.destroy();
        } else {
            // ignore so we don't leak memory from the observation references
            this._ignoreData(this.data);
            delete this.data;
        }
    }
    // If we are grouped our ResultSet may be stored under this.originalData
    // Clean this up too (otherwise we can end up with the DataSource attempting to
    // notify a destroyed ListGrid via it's orphaned resultSet)
	if (this.originalData){
        if (this.originalData._autoCreated && isc.isA.Function(this.originalData.destroy)) {
            this.originalData.destroy();
        } else {
            this._ignoreData(this.originalData);
            delete this.originalData;
        }
    }
    
    if (this.selection) {
        this.destroySelectionModel();
    }
    if (this.selectionCanvas) this.selectionCanvas.destroy();
    if (this.selectionUnderCanvas) this.selectionUnderCanvas.destroy();
    if (this.rollOverCanvas) this.rollOverCanvas.destroy();
    if (this.rollUnderCanvas) this.rollUnderCanvas.destroy();
    
    this._dropODSData();
    
    // RecordComponents:
    // - if pooling mode is recycled, destroy any created recordComponents sitting in our pool.
    // - if pooling mode is data, destroy any created recordComponents that got scrolled out
    //   of view and never destroyed.
    
    var recordComponents = this.getRecordComponentPool();
    if (recordComponents) {
        for (var i = 0; i < recordComponents.length; i++) {
            var comp = recordComponents[i];
            if (comp && comp.destroy && !comp.destroyed) {
                if (!comp.dontAutoDestroy) comp.destroy();
                else comp.deparent();
            }
        }
    }
    if (this._orphanedRecordComponents != null) {
        for (var ID in this._orphanedRecordComponents) {
            var comp = this._orphanedRecordComponents[ID];
            if (!comp || comp.destroyed || comp.destroying 
                || comp._pendingDestroy) 
            {
                continue;
            }
            if (!comp.dontAutoDestroy) comp.destroy();
            else comp.deparent();
        }
    }

    this.Super("destroy", arguments);
},

//>	@method	listGrid.redrawHeader()	(A)
//			redraw the header
//		@group	drawing
//<
redrawHeader : function () {
	if (this.header) this.header.markForRedraw();
},

//>	@method	listGrid.getBaseStyle()
// Return the base stylename for this cell.  Has the following implementation by default:
// <ul>
// <li>If +link{listGrid.editFailedBaseStyle, this.editFailedBaseStyle} is defined, and the
//     cell is displaying a validation error return this value.</li>
// <li>If +link{listGrid.editPendingBaseStyle, this.editFailedPendingStyle} is defined, and
//     the cell is displaying an edit value that has not yet been saved (see 
//     +link{ListGrid.autoSaveEdits}) return this value.</li>
// <li>Otherwise return +link{ListGrid.recordBaseStyleProperty, record[listGrid.recordBaseStyleProperty]},
//     if defined, otherwise +link{ListGridField.baseStyle, field.baseStyle}.</li>
// </ul>
// If no custom style is found for the cell as described above, the default baseStyle will be
// returned. If +link{listGrid.baseStyle} is specified this will be used. Otherwise for
// grids showing fixed height rows which match +link{listGrid.normalCellHeight} 
// +link{listGrid.normalBaseStyle} will be used. For grids with variable, or modified
// cell heights, +link{listGrid.tallBaseStyle} will be used. 
// <P>
// Note also that
// enabling +link{listGrid.fastCellUpdates} will cause the <code>tallBaseStyle</code> to be
// used rather than +link{listGrid.normalBaseStyle}.
//
// @see getCellStyle()
//
// @param	record  (ListGridRecord)	Record associated with this cell. May be <code>null</code>
//                               for a new edit row at the end of this grid's data set.
// @param	rowNum	(number)	row number for the cell
// @param	colNum	(number)	column number of the cell
// @return	(CSSStyleName)	CSS class for this cell
// @visibility external
// @example replaceStyle
//<
getBaseStyle : function (record, rowNum, colNum) {

    
    if (this.canEdit == true && !this.isPrinting) {
        if (this.editFailedBaseStyle && this.cellHasErrors(rowNum, colNum))
            return this.editFailedBaseStyle;
    
        if (this.editPendingBaseStyle && this.cellHasChanges(rowNum, colNum, false))
            return this.editPendingBaseStyle;
    }

    if (this.originBaseStyle && this.selection._isCellSelectionOrigin(rowNum, colNum)) {
        return this.originBaseStyle;       
    }

    if (record && this.recordBaseStyleProperty && record[this.recordBaseStyleProperty]) 
        return record[this.recordBaseStyleProperty];
    
    var field = this.getField(colNum);
    if (field && field.baseStyle) return field.baseStyle;
    
    if (field && field.type == "summary" && this.recordSummaryBaseStyle) 
        return this.recordSummaryBaseStyle;
    
    if (field && field.frozen && !this._suppressedFrozenFields && this.frozenBaseStyle) {
        return this.frozenBaseStyle;
    }
    
    if (this.isPrinting && (this.printBaseStyle != null)) return this.printBaseStyle;
    
    var baseStyle = this.baseStyle;
    if (baseStyle == null) {
        if (this.cellHeight != this.normalCellHeight || 
            
            this.fastCellUpdates || 
            this.allowRowSpanning || 
            !this.shouldFixRowHeight(record, rowNum) ||
            
            (record != null && this._hasEmbeddedComponents(record)))
        {
            baseStyle = this.tallBaseStyle;
        } else {
            baseStyle = this.normalBaseStyle;
        }
    }
    return baseStyle;
},

getCellCSSText : function (record, rowNum, colNum) {
    // per-record cssText
    if (record) {
        var cssText = record[this.recordCSSTextProperty];
        if (cssText != null) return cssText;
    }

    var cssText;
    
    if (this.recordMarkedAsRemoved(rowNum) && this.removedCSSText) {
        cssText = this.removedCSSText;
    } else {
    
        if (this.isEditable()) {
            if (this.editFailedBaseStyle == null && this.editFailedCSSText && 
                this.cellHasErrors(rowNum, colNum)) 
            {
                cssText = this.editFailedCSSText;
            } else if (this.editPendingBaseStyle == null && this.editPendingCSSText && 
                this.cellHasChanges(rowNum, colNum, false)) 
            {
                cssText = this.editPendingCSSText;
            }
        } 
    }
    
    
    cssText = this.getRecordHiliteCSSText(record, cssText, this.getField(colNum), true);
    return cssText;

},

//>	@method	listGrid.getRawCellValue()
//		@group	data
//			return the raw data for one particular cell in the list
//
//		@param	record		(object)	a record in the data
//		@param	recordNum	(number)	number of that record (in case it's important for the output)
//		@param	fieldNum	(number)	number of the field to display
//
//		@return	(string)	raw value for this cell
//<

getRawCellValue : function (record, recordNum, fieldNum, isFieldName) {
    var field,
        fieldName,
        dataPath;

    if (isFieldName) {
        fieldName = fieldNum
        // when fieldName specified, search completeFields if available
        field = this.completeFields ? isc.Class.getArrayItem(fieldName, this.completeFields, this.fieldIdProperty) : this.getField(fieldName);
    } else {
        field = this.fields[fieldNum];
        fieldName = field ? field[this.fieldIdProperty] : fieldName;
    }
     
    dataPath = field ? this.getTrimmedFieldDataPath(field) : null;

    if (fieldName == null && dataPath == null) return this.emptyCellValue;
    // Note even if fieldName is set, field is not guaranteed to be present - this could
    // be a displayField which is being rendered in a different field's cell

    var editValue, value, undef;
    if (this.rowEditNotComplete(recordNum)) {
        editValue = this._getEditValue(recordNum, fieldNum);
    }
//this.logWarn("GRCV is running - row/col:" + [recordNum, fieldNum], editVal:" + editValue);    

    if (editValue !== undef) {
        // This ensures that the value for the cell as it is currently stored for local editing
        // is displayed, rather than the saved value of the cell.
        
        return editValue;
    // record can be null if there's no record for this cell
    } else if (record == null) {
        return this.emptyCellValue;

    // show the result of getRecordSummary for summary fields
    // Note that if shouldApplyRecordSummaryToRecord is true it's already stored on the record
    // object so we've already picked it up
    
    } else if (this.shouldShowRecordSummary(field,record) && 
                !this.shouldApplyRecordSummaryToRecord(field)) 
    {
        value = this.getRecordSummary(recordNum, field);
    
    
    // Default - just look at the record.
    } else {

        // let the dataset return a value if it supports doing so
        
        if (this.data.getFieldValue && field) {
            value = this.data.getFieldValue(record, dataPath, field, this);
           
        } else if (record.ownerDocument && field) {
            // if working with XML, fetch values via XPath selectors if so configured.  (Needed
            // for viewing XML elements when not dataBound, but could be moved to be a built-in
            // feature of both List and native Array)
            
            value = isc.xml.getFieldValue(record, fieldName, field);
        } else {
            value = isc.Canvas._getFieldValue(dataPath, field, record, this, true);    
            if (value == null) value = record[fieldName];
        }
    }
    
    // if the field has a 'getRawCellValue' attribute
    
    if (field && field.getRawCellValue) {
        // CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value"
        // Convert a string callback to a function
        isc.Func.replaceWithMethod(field, "getRawCellValue", "viewer,record,recordNum,field,fieldNum,value");

        // call it as a function (returns string | null)
        value = field.getRawCellValue(this, record, recordNum, field, fieldNum, value);
	}


    // handle user-formula and user-summary fields
    
    if (field) {
            
        if (field.userFormula && this.shouldShowUserFormula(field, record) &&
            !this.shouldApplyUserFormulaToRecord(field) )
        {
            value = this.getFormulaFieldValue(field, record);
        }
        // note: actual value computed later (after formatters have been applied
        // to raw value) - see getSummaryFieldValue() call in getFormattedCellValue
        if (field.userSummary) this.getSummaryFunction(field);
    }
    return value;
},

// shouldShowRecordSummary - returns true if this is a summary field unless
// this is a summary row and there's an explicit summaryFunction (which takes precedence over
// any record summary function).

_$summary:"summary",
shouldShowRecordSummary : function (field, record) {
    if (field && field.type == this._$summary) {
        if (record[this.groupSummaryRecordProperty]) {
            return (field.summaryFunction == null && field.getGroupSummary == null);
        
        } else if (record[this.gridSummaryRecordProperty]) {
            return (field.summaryFunction == null && field.getGridSummary == null);
        }
        return true;
    }
    return false;
},

// shouldShowUserFormula - As with record summaries, if you have a formula field
// you likely do not want the field-level record-formula to show in a summary row
// - instead the grid or group summary should be displayed as the field value.
shouldShowUserFormula : function (field, record) {
    var isSummary = this._isSummaryRow ||
                     (record && record[this.groupSummaryRecordProperty]);
    if (isSummary && !this._applyFieldFormulaAfterSummary(field)) {
        return false;
    }
    return true;
},

//> @attr listGrid.applyFormulaAfterSummary (Boolean : false : IRW)
// If +link{listGrid.field.userFormula} is set for some field, and this grid is showing
// +link{listGrid.showGroupSummary,group summaries} or a
// +link{listGrid.showGridSummary,grid summary}, this property determines what field value
// should be present in those summary rows. Should the field apply the user-formula to the
// calculated summary row, or should it apply a standard grid or group summary to the
// user-formula values displayed in the grid?
// <P>
// May be overridden at the field level via +link{listGridField.applyAfterSummary}
// @visibility external
//<

applyFormulaAfterSummary:false,

// If we have a formulaField, how should the cell value in a summary row be calculated?
// if applyFormulaAfterSummary is true we'llcalculate summary row values and apply
// the field-formula to that calculated row. Otherwise we will display "normal" summary
// values for the field in the row (for example summing all the calculated values in the
// formula field).
_applyFieldFormulaAfterSummary : function (field) {
    var applyAfterSummary = field && field.applyAfterSummary;
    if (applyAfterSummary == null) applyAfterSummary = this.applyFormulaAfterSummary;
    if (applyAfterSummary == null) applyAfterSummary = false;
    //this.logWarn("For field:" + field.name + ", applyFormulaAfterSummary:" + applyAfterSummary);
    return applyAfterSummary;
},

_$blank: "blank",
_getCheckboxValueIconHTML : function (isSel, canSelect, disabled, field) {
    var icon, iconStyle, extraExtraStuff;
    if (!canSelect) {
        // record cannot be selected but we want the space allocated for the checkbox anyway.
        icon = this._$blank;
    } else {
        // checked if selected, otherwise unchecked
        if (isSel) {
            icon = this.checkboxFieldTrueImage || this.booleanTrueImage;
            iconStyle = this.booleanBaseStyle == null ? null : this.booleanBaseStyle + "True";
        } else {
            icon = this.checkboxFieldFalseImage || this.booleanFalseImage;
            if (icon == null) icon = this._$blank;
            iconStyle = this.booleanBaseStyle == null ? null : this.booleanBaseStyle + "False";
        }
    }
    // if the record is disabled, make the checkbox image disabled as well
    if (disabled) {
        if (icon != this._$blank) icon = isc.Img.urlForState(icon, false, false, isc.StatefulCanvas.STATE_DISABLED);
        if (iconStyle != null) iconStyle += isc.StatefulCanvas.STATE_DISABLED;
    }
    if (icon == this._$blank) icon = isc.Canvas._blankImgURL;
    if (iconStyle != null) {
        extraExtraStuff = "class='" + iconStyle + this._$singleQuote;
    }

    return this.getValueIconHTML(icon, field, extraExtraStuff);
},

//>	@method	listGrid.getCellValue()   ([A])
//          Obtains the display value for a specific cell according to the given input 
//          parameters.<br>
//          To format the value displayed in the cell, make use of the 
//          <code>formatCellValue</code> methods rather than overriding this method directly.
//      @visibility external
//		@group	data
//
//		@param	record		(object)	the current record object
//		@param	recordNum	(number)	number of the record in the current set of displayed 
//                                      record (e.g. 0 for the first displayed record)
//		@param	fieldNum	(number)	number of the field in the listGrid.fields array
//
//      @see    method:ListGrid.formatCellValue
//		@return	(string)	display value for this cell
//<
_$HR:"<HR>",
getCellValue : function (record, recordNum, fieldNum, gridBody) {

    if (gridBody == null) gridBody = this.getFieldBody(fieldNum);

	// If we're handed an empty record, we may be looking at a new edit row - if so get
	// the edit values instead
    if (record == null) {
        if (this.showNewRecordRow && this._isNewRecordRow(recordNum)) {
            return this.getNewRecordRowCellValue();
        }
        record = this._getEditValues(recordNum, fieldNum);
    } else {
        // Special cases:
        // if it's a separator row, return a horizontal rule
        
        if (record[this.isSeparatorProperty]) return this._$HR;
    
        // group controller node - write out the groupNodeHTML
        if (record._isGroup && fieldNum != null) {
            
            var currentField = this.fields[fieldNum],
                groupTitleField = this.getGroupTitleField(),
                isGroupTitleField;
            if (groupTitleField == null) {
                isGroupTitleField = currentField._isGroupTitleColumn;
            } else {
                isGroupTitleField = currentField.name == groupTitleField;
            }
            
            // If we're showing multiple fields and we're not showing a summary in the
            // header, explicitly empty all fields except the groupTitleField
            if (this.singleCellGroupHeaders() || isGroupTitleField) {
                return this.getGroupNodeHTML(record, gridBody);
            } else if (!this.showGroupSummaryInHeader || this.isCheckboxField(currentField)) {
                return isc.nbsp;
            }
        }
    }

	// get the actual data record
	var field = this.fields[fieldNum],
		value = null;

	// if field is null, we've shrunk the number of columns, so forget it
    if (field == null) return isc.emptyString;

    // if this is a summary row and the field has a summaryValue specified, return that - all
    // the special fields (rowNumber, checkbox, expansion, remove) set this value to "&nbsp;"
    if (this._isSummaryRow && field.summaryValue) {
        return field.summaryValue;
    }

    // If this is the checkboxField, write out the checkbox HTML
    
    if (this.isCheckboxField(field)) {
        return this._getCheckboxValueIconHTML(!!this.selection.isSelected(record),
                                              this.body.canSelectRecord(record),
                                              record && record[this.recordEnabledProperty] == false,
                                              field);
    }

    // Determine whether we should be showing JUST a valueIcon
    // Do this before we figure out other HTML to avoid calculating stuff we're not going to 
    // actually use.
    var icon,
        iconStyle,
        iconOnly = this.showValueIconOnly(field),
        isEditCell;
	if (record != null) {
        
        
		if (record[this.singleCellValueProperty] != null) {
            return record[this.singleCellValueProperty];
        }
        
		if (Array.isLoading(record)) {
            
            if (!isc.Browser.isSafari || fieldNum == 0) {
                return this.loadingMessage;
            }
            return isc.nbsp;
        }
        
    	// If we are currently editing there are three possibilities for what we should write
    	// out for each cell:
    	// - This cell is being edited -- write out the edit form's element HTML
    	// - This cell has pending edit values, but is not currently being edited.
    	//   (this can happen if we're editing another cell in this row, or if an edit was not
    	//   successfully saved for this cell) -- display the edit values.
    	// - This cell should display the value for the record (default behavior)
    	// We catch the first case here, and write out the edit-form item into the cell if
    	// appropriate.
    	// The other two cases will be handled by 'getRawCellValue()', which will check for
    	// the presence of editValues on a cell.
    	

        // if rowSpanning is enabled, and the user starts editing a row that
        // is spanned by some cell, behavior is governed by rowSpanEditMode:
        // - if rowSpanEditMode is "first", canEditCell will return false for the
        //   logical cell so we won't show an editor in the cell.
        // - otherwise, we show an editor in the cell and populate it with values
        //   based on the edit row.
        //   This means the value displayed in the editor will come from the
        //   edit-row, whereas the value displayed while not in edit mode comes from
        //   the first spanned record (by default).
        //   In this case the edit-value displayed may be different to the static
        //   value displayed in the spanning cell. 
        //   The developer might use this mode if they have some advanced behavior like
        //   custom formatters that render a value for every spanned record, etc
    	var editRowNum = this._editRowNum,
    	    editStartRow = editRowNum;
    	if (editStartRow != null && this.allowRowSpanning) {
    	    editStartRow = this.getCellStartRow(editStartRow, fieldNum);
    	}
        isEditCell = (this._editorShowing && editStartRow == recordNum && 
                      (!this.editByCell || this._editColNum == fieldNum) &&
                      
                      this.canEditCell(editRowNum, fieldNum));
        if (isEditCell) {
            var editRecord = record;
            if (editRowNum != recordNum) {
                editRecord = this.getCellRecord(editRowNum, fieldNum);
            }
            // Avoid writing out live form items if we're just getting HTML for the
            // GR auto-sizer canvas.
            
            if (gridBody._gettingAutoSizeHTML) {
                value =this.getInactiveEditorCellValue(editRecord, editRowNum, fieldNum);
            } else {
                value = this.getEditItemCellValue(editRecord, editRowNum, fieldNum);
            }
        // Checking for this._editorShowing would mean that when changing edit cell, 
        // hideInlineEditor would render the row with no inactive editors, and they
        // wouldn't get shown again on showEditForm for the new cell without another
        // row-refresh
        } else if (this._showInactiveEditor(fieldNum) && this.canEditCell(recordNum,fieldNum)) {

            value = this.getInactiveEditorCellValue(record, recordNum, fieldNum);
            // set isEditCell -- this will suppress the standard 'valueIcon' stuff
            isEditCell = true;
        } else {
        	// get the value according to the field specification
            var field = this.fields[fieldNum],
                displayFieldValue,
                pickUpDisplayFieldValue;
            // If a displayField was specified on the field, and no separate optionDataSource,
            // we should display the value for the record from the displayField rather than the
            // data field.
            if (field.displayField != null) {
                pickUpDisplayFieldValue = !field.valueMap && 
                                          !field.getCellValue && 
                                          // this method actually checks for field.valueMap
                                          // being non null
                                          this._useDisplayFieldValue(field);
                                          
                if (pickUpDisplayFieldValue) {
                    var displayField = field.displayField;
                    displayFieldValue = this.getRawCellValue(record, recordNum, displayField, true);
                }
                // Note: we still need to proceed through the rest of the logic, since the
                // underlying data value drives the valueIcon
            }
            value = this.getRawCellValue(record, recordNum, fieldNum);
     		// if the field has a 'getCellValue' attribute
        	// NOTE: this is deprecated - overriding 'getCellValue()' at the Grid level is very
        	// advanced, so for simple formatting (which is likely to be done at either the Field
        	// or Grid level) we provide 'formatCellValue()' methods instead.)
        	// Leave this code in place for Back-compat only.
    
        	//>!BackCompat 2005.1.12
    		if (field.getCellValue) {
    			// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value"
    			// Convert a string callback to a function
    			isc.Func.replaceWithMethod(field, "getCellValue",
                                                 "viewer,record,recordNum,field,fieldNum,value");
    			value = field.getCellValue(this, record, recordNum, field, fieldNum, value);
    		}
        	//<!BackCompat

            
            icon = this.getValueIcon(field, value, record, recordNum);
            iconStyle = this.getValueIconStyle(field, value, record, recordNum);

            // if we're ONLY showing an icon we don't really need to figure out the text from
            // the valueMap.
            if (!iconOnly) {

                // if the field has an 'valueMap' parameter, treat the value as a key in the map
        		var valueMap = field.valueMap;
                if (valueMap) {
                    // NOTE: this can be really expensive, since we may eval the function that
                    // returns the value map for each record!!!
                    if (isc.isA.String(valueMap)) valueMap = this.getGlobalReference(valueMap);
                    // NOTE: don't look up displayed values in Array valueMaps, which just list legal
                    // values, and don't provide stored->displayed value mappings.
                    if (!isc.isAn.Array(valueMap)) {
                        if (isc.isAn.Array(value)) {
                            var displayValue = [];
                            for (var i = 0; i < value.length; i++) {
                                var dataVal = isc.getValueForKey(value[i], valueMap, value[i]);   
                                displayValue[displayValue.length] = dataVal;
                            }
                            value = displayValue;
                        } else {
                            value = isc.getValueForKey(value, valueMap);
                        }
                    }
                }
            }
            
            if (pickUpDisplayFieldValue) value = displayFieldValue;
        }
	}
    
    if (!isEditCell) {
    
        // If this is a summary row, avoid calling formatCellValue at all if we're not showing
        // a summary for the field.
        if (record && 
            ((record[this.groupSummaryRecordProperty] && !this.shouldShowGroupSummary(field)) ||
             (this._isSummaryRow && 
                ((record[this.gridSummaryRecordProperty] && !this.shouldShowGridSummary(field))
                 || (field.showGridSummary == false))
             )
            )
           )
        {
            value = this.emptyCellValue;
        } else {
    
            // If this cell isn't showing an editor we may need to write out an icon
            
            var iconHTML = null;
            
            if (icon != null) {
                var iconExtraExtraStuff = iconStyle == null ? null : "class='" + iconStyle + this._$singleQuote;
                iconHTML = this.getValueIconHTML(icon, field, iconExtraExtraStuff);
            }
    
            if (iconOnly) {
                // If there's no icon write out the empty cell value. This avoids us having
                // un-styled cells.
                if (!iconHTML || isc.isAn.emptyString(iconHTML)) iconHTML = this.emptyCellValue;
                value = iconHTML;
            } else {
                // use formatCellValue() to perform any additional formatting
                value = this._formatCellValue(value, record, field, recordNum, fieldNum);
                // apply hilites to capture htmlBefore/after
                var hilites = this.getFieldHilites(record, field);
                // Note hilites can == 0, so if(hilites) doesn't work
                if (hilites != null) value = this.applyHiliteHTML(hilites, value);
                
                if (iconHTML) {
                    if (field.valueIconOrientation != isc.Canvas.RIGHT)
                        value = iconHTML + value;
                    else
                        value = value + iconHTML;
                }
            }
            
            // Only show error icon HTML if we're not showing an editor for the cell - otherwise
            // we'd get doubled error icons.
            if (this.isEditable() && this.showErrorIcons && this.cellHasErrors(recordNum, fieldNum)) {
                value = this.getErrorIconHTML(recordNum, fieldNum) + value;
            }        
    
            // Add the hiliteIcons last
            var field = this.fields[fieldNum];
            var hilites = this.getFieldHilites(record, field);
            // Note that hilites can == 0, so if(hilites) won't work
            if (hilites != null) {
                value = this.applyHiliteIcon(hilites, field, value);
            }
        }
    }
    // if current field is the groupTitleField, add indentation to the cell value
    var gtf = this.isGrouped ? this.getGroupTitleField() : null;
    if (gtf && this.fields[fieldNum].name == gtf) {
        var groupIndent = 
            isc.Canvas.spacerHTML(this.groupIndentSize + this.groupLeadingIndent, 1);
        value = groupIndent + value;    
    }
    return value;
},

// Formula/summary -related overrides from DBC
getTitleFieldValue : function (record) {
    var titleField = this.getDataSource().getTitleField(),
        title = this.getCellValue(record, this.getRecordIndex(record), 
            this.getFieldNum(titleField), this.body
    );

    if (!title || title == "") {
    	title = this.getRawCellValue(record, this.getRecordIndex(record), titleField, true);
	}
	return title;
},

getRawValue : function (record, fieldName) {
    var recordIndex = this.getRecordIndex(record);
    
    return this.getRawCellValue(record, recordIndex, fieldName, true);
},

getFormattedValue : function (record, fieldName, value) {
    var recordIndex = this.getRecordIndex(record),
        field = this.getSpecifiedField(fieldName),
        fieldNum = this.getFieldNum(fieldName);

    if (this.fieldIsVisible(field) && value == null) {
        return this.getCellValue(record, recordIndex, fieldNum, this.body);
    }
	if (value == null) value = this.getRawValue(record, fieldName);
	// if the field has an 'valueMap' parameter, treat the value as a key in the map
    var valueMap = field ? field.valueMap : null;
    if (valueMap) {
        // NOTE: this can be really expensive, since we may eval the function that
        // returns the value map for each record!!!
        if (isc.isA.String(valueMap)) valueMap = this.getGlobalReference(valueMap);
        // NOTE: don't look up displayed values in Array valueMaps, which just list legal
        // values, and don't provide stored->displayed value mappings.
        if (!isc.isAn.Array(valueMap)) {
            if (isc.isAn.Array(value)) {
                var displayValue = [];
                for (var i = 0; i < value.length; i++) {
                    var dataVal = isc.getValueForKey(value[i], valueMap, value[i]);   
                    displayValue[displayValue.length] = dataVal;
                }
                value = displayValue;
            } else {
                value = isc.getValueForKey(value, valueMap);
            }
        }
    }

    var ret = this._formatCellValue(value, record, field, recordIndex, fieldNum);
    return ret;

},


// get the width of the specified field or null
getSpecifiedFieldWidth : function (fieldName) {

    var field = this.getField(fieldName);
    if (field == null) {
        // field may just be hidden - if so check for an explicit width on the
        // field object in our completeFields array and return that
        if (isc.isAn.Object(fieldName)) fieldName = fieldName[this.fieldIdProperty];
        field = this.getSpecifiedField(fieldName);
    }
    if (field == null) return;

    var fieldName = field[this.fieldIdProperty],
        members = this.header ? this.header.members : null,
        frozenMembers = this.frozenFields && this.frozenHeader ? this.frozenHeader.members : null,
        width;

    if (members || frozenMembers) {
        var member;
        
        if (members) member = members.find(this.fieldIdProperty, fieldName);
        if (!member && frozenMembers) {
            member = frozenMembers.find(this.fieldIdProperty, fieldName);
        }
        if (member) {
            if (member._userWidth && isc.isA.Number(member._userWidth)) {
                width = member._userWidth;
            }
        // If we don't have a member for the field, look at the specified width and
        // use it if possible.
        // This can happen if a field is currently hidden
        } else {
            if (field.width != null && isc.isA.Number(field.width)) {
                width = field.width;
            }
        }
    }
    return width;
},

getValueIconHTML : function (icon, field, extraExtraStuff) {
    
    var prefix = field.imageURLPrefix || field.baseURL || field.imgDir,
        suffix = field.imageURLSuffix,
        width = this.getValueIconWidth(field),
        height = this.getValueIconHeight(field),
        leftPad = this.getValueIconLeftPadding(field),
        rightPad = this.getValueIconRightPadding(field);
    
    if (suffix != null) icon += suffix;
                    
    var iconHTML = isc.Canvas._getValueIconHTML(icon, prefix, width, height,
                                      leftPad, rightPad,
                                      // no need for an ID
                                      null,
                                      // pass in the LG as an instance - required
                                      // for generating the path of the valueIcon src
                                      this,
                                      extraExtraStuff);
    return iconHTML;
},

//>	@method	listGrid.getCellAlign()
// Return the horizontal alignment for cell contents. Default implementation returns 
// +link{listGridField.cellAlign} if specified, otherwise +link{listGridField.align}.
//
// @see getCellStyle()
//
// @param   record (listGridRecord) this cell's record
// @param	rowNum	(number)	row number for the cell
// @param	colNum	(number)	column number of the cell
// @return	(Alignment)     Horizontal alignment of cell contents: 'right', 'center', or 'left'	
// @visibility external
//<
getCellAlign : function (record, rowNum, colNum) {
    var isRTL = this.isRTL();
    // single cells are aligned according to RTL - skip checkboxField (which needs center align)
    if (record && record[this.singleCellValueProperty] != null &&
        (!this.showSingleCellCheckboxField(record) || 
         !this.isCheckboxField(this.getField(colNum)))
       )
    {
        return isRTL ? isc.Canvas.RIGHT : isc.Canvas.LEFT;
    }
    
    return this.getFieldCellAlign(colNum, isRTL);
},

getFieldCellAlign : function (fieldNum, isRTL) {
    
    var field = this.getField(fieldNum);
    
    if (!field) return isRTL ? isc.Canvas.RIGHT : isc.Canvas.LEFT;
    
    
    if (field.cellAlign == null && field.userFormula) {
        return isRTL ? isc.Canvas.LEFT : isc.Canvas.RIGHT;
    }
    // This is set up in applyFieldDefaults if not explicitly specified on the field config
    var align = field.cellAlign || field.align;
    // If 'reverseRTLAlign' is true, flip the explicitly specified alignment.
    // This is true by default and makes sense for common cases where you really want
    // specified 'left' / 'right' to have the meaning of 'start'/'end'
    if (isRTL && this.reverseRTLAlign) {
        if (align == isc.Canvas.RIGHT) align = isc.Canvas.LEFT;
        else if (align == isc.Canvas.LEFT) align = isc.Canvas.RIGHT;
    }
    return align;
},

// Helper method
// If we're showing a singleCellValue for a record, should we show the checkboxField to the
// left of the single value
showSingleCellCheckboxField : function (record) {
    return (this.getCurrentCheckboxField() != null) && record &&
            !record._isGroup && !record[this.isSeparatorProperty];
},

//>	@method	listGrid.getCellVAlign()
// Return the vertical alignment for cell contents.
// Expected values are: 'top', 'center', or 'bottom'
//
// @see getCellStyle()
//
// @param   record (listgridRecord) this cell's record
// @param	rowNum	(number)	row number for the cell
// @param	colNum	(number)	column number of the cell
// @return	(Alignment)     Vertical alignment of cell contents: 'right', 'center', or 'left'	
// @visibility external
//<
// Unset by default


// Helper method: 
// If a displayField was specified for a field, should we pick up the 'displayField' value for
// this record and display it here?
// Basically this makes sense if no optionDataSource was specified, unless overridden by a valueMap
_useDisplayFieldValue : function (field) {
    // no display field, or we have an explicit valueMap
    if (!field || field.valueMap || field.displayField == null || 
        (field.displayField == field.name)) 
    {
        return false;
    }
    
    // displayField and no explicit optionDataSource, pick up the value from the record
    // unless explicitly suppressed by the developer
    if (field.optionDataSource == null) {
        if (field.displayValueFromRecord == false) return false;
        return true;
    }
    
    // field has an optionDataSource and autoFetchDisplayMap is true - in this case we'll
    // fetch against the ODS and build a valueMap from the data we get back - so even if we don't
    // have a valueMap yet, we soon will!
    var fetchDisplayMap = field.autoFetchDisplayMap;
    if (fetchDisplayMap == null) fetchDisplayMap = this.autoFetchDisplayMap;
    if (fetchDisplayMap) return false;
    
    // At this point, if optionDataSource and valueField matches we'll use the 
    // value from the record - however allow the developer to override this to either
    // a) force the display field value to be picked up (as if there was no ODS specified - somewhat
    // probable if autoFetchDisplayMap has been disabled, since the ODS has really no other meaning
    // to the grid) or
    // b) suppress the display field value even if the dataSources do match (for example may want
    // to show display-field values in edit mode only.
    if (field.displayValueFromRecord  != null) return field.displayValueFromRecord;
    
    // same dataSource / same valueField - pick up the value from the record.
    return (field.valueField == null || field.valueField == field.name) && 
            (isc.DS.get(field.optionDataSource) == this.getDataSource());
},         
                                   
// Row Spanning
// ---------------------------------------------------------------------------------------

//> @attr listGrid.allowRowSpanning (Boolean : false : IR)
// Should cells in this grid be allowed to span multiple rows?
// If set to <code>true</code>, the +link{listGrid.getRowSpan()} method will be
// called for every cell when rendering out the listGrid to determine how many
// rows the cell should span.
// <P>
// See +link{listGrid.getRowSpan()} for more details
// @visibility external
//<
allowRowSpanning:false,

//> @method listGrid.getRowSpan()
// Return how many rows this cell should span.  Default is 1. 
// <P>
// This method will only be called if +link{listGrid.allowRowSpanning} is set to
// <code>true</code>
// <P>
// When using row spanning, consider setting +link{useRowSpanStyling} to enable
// row-span-sensitive styling behaviors.
// <P>
// Note that the standard implementation assumes that the number of rows spanned by cells
// decreases or stays the same, starting with the first (leftmost) column in the grid and
// moving rightwards.
// <P>
// When using row spanning:
// <ul>
// <li> APIs that allow modifying the contents of cells (such as +link{getCellStyle()} or
// +link{listGridField.formatCellValue()}) will be called only once per row-spanning cell
// <li> when using +link{listGrid.canSelectCells,cell-based selection}, only the
// spanning cell is considered selected, and not any of the cells spanned through.  For
// example, if the cell at row 2 column 0 spans 2 cells, +link{CellSelection.isSelected()}
// will be true for 2,0 but false for 3,0.
// <li> if using incremental rendering (either horizontal or vertical),
// <code>getRowSpan()</code> may be called for a rowNum <b>in the middle of a spanning
// cell</b>, and should return the remaining span from that rowNum onward.
// <li> cell-level events such as +link{listGrid.recordClick()} will report the logical
// rowNum for spanned cells. In other words if a cell spans two rows, a different 
// rowNum parameter will be passed to the recordClick handler depending on whether the user
// clicks at the top of the spanning cell or the bottom. Developers can normalize
// this to the starting cell via the +link{listGrid.getCellStartRow()} API.
// <li> for cells that span multiple records, editing behavior may be controlled by the
// +link{listGrid.rowSpanEditMode} attribute.
// <li> rowSpanning can be used in conjunction with
//  +link{showRecordComponents(),recordComponents}. If record
// component are enabled on a grid with row-spanning cells the behavior is as follows:
//   <ul>
//    <li>Having +link{recordComponentPosition} set to "expand" is not currently supported
//        for grids that render out spanning cells.
//    <li>The method to retrieve / create record components will not be run for cells that
//        are "spanned". In other words if the first row in a grid spans 2 rows for some
//        field, the second logical row is "spanned" for that field - that cell doesn't 
//        render any content and won't attempt to create a recordComponent.
//    <li>If +link{showRecordComponentsByCell} is false, the method to create record
//        components will be called for every non-spanned cell in the first column
//        of the grid.
//    <li>Percentage sizing of record components spanning multiple cells will be
//        calculated relative to the set of spanned cells.
//   </ul>
// </ul>
// <P>
// More generally, the ListGrid has a data model of one +link{Record} per row, and spanning cells
// doesn't fit well with this model, meaning that many ListGrid features are incompatible
// with rowSpanning.  
//
// @include gridRenderer.getRowSpan()
// @visibility external
//< 
// Note the limitation of row spanning never increasing going left to right comes up 
// in a couple of spots:
// - the default selection model basically assumes the number of rows spanned is reducing 
//   and performs a selection of all spanned cells in the row, moving rightwards
// - the getRowHeight implementation makes this assumption to ensure frozen body row
//   heights are correct (we have an undocumented flag to remove this limitation, but
//   this introduces some extra calculation that would have to run, making this undesirable
//   for many-columned grids).

//> @method listGrid.getCellRowSpan()
// @include gridRenderer.getCellRowSpan()
// @visibility external
//<

//> @method listGrid.getCellStartRow()
// @include gridRenderer.getCellStartRow()
// @visibility external
//<

//> @attr listGrid.useRowSpanStyling (boolean : null : IR)
// Enables various styling behaviors that potentially make sense when +link{getRowSpan()} has
// been overriden to introduce spanning cells, and spanning is largest on the left and
// smaller as cells go to the right.  Specifically:
// <ul>
// <li> computes +link{alternateRecordStyles,banded styling} based on the span of the
//      cell in the left-most column
// <li> enables +link{listGrid.canSelectCells,cell-level selection}, including
//      +link{useCellRollOvers,cell-level rollover} styling
// <li> enables row-span-sensitive cell selection.  See also +link{rowSpanSelectionMode}
//      for available behaviors
// </ul>
// <P>
// Because this setting enables +link{canSelectCells}, it is incompatible with any APIs
// that expect a record-oriented data model.
// <P>
// Because this setting only makes sense when row spanning decreases from the first column
// to the last, it has unspecified behavior with +link{listGrid.canReorderFields}.
//
// @visibility external
//<

//> @type RowSpanSelectionMode
// Behavior of selection when row spanning is active.  See +link{listGrid.useRowSpanStyling}.
// @value "forward" when a cell is clicked on, select any cells in subsequent columns which 
//                  are at least partially spanned by the clicked cell
// @value "both"    when a cell is clicked on, selects any cells in any other columns which
//                  are at least partially spanned by the clicked cell
// @value "outerSpan" behaves like "forward", except as though the cell in the first column
//                    was clicked instead.  If the largest row spans are in the first
//                    column and all cells in subsequent columns do not extend out of the
//                    first cell's span, this creates a row-like selection model where the
//                    span of the left-most cell defines the "row" of cells being selected.
// @visibility external
//<

//> @attr listGrid.rowSpanSelectionMode (RowSpanSelectionMode : "forward" : IR)
// Chooses the selection mode when +link{useRowSpanStyling} is enabled.  See
// +link{RowSpanSelectionMode}.
//
// @visibility external
//<
rowSpanSelectionMode:"forward",

//> @attr rowSpanEditMode (RowSpanEditMode : "first" : IRWA)
// If +link{listGrid.allowRowSpanning} is enabled, this property may be used to specify
// editing behavior for cells that span multiple rows.
// @visibility external
//<
rowSpanEditMode:"first",

//> @type RowSpanEditMode
// When +link{listGrid.allowRowSpanning} is enabled, certain cells may span multiple
// rows. In this case, the cell displays the value from the record in the first row.
// If the grid is +link{listGrid.canEdit,editable} (and the 
// +link{listGridField.canEdit,field is also editable}), these settings allow the user
// to specify what happens to the data when the user edits this cell.
// <P>
// Note that in this scenario, a user may begin an edit on the row-spanning cell directly
// (via double-click for example), or on a cell in another column in any of the rows 
// spanned by the cell. The appropriate behavior with respect to user-experience and how
// the data is manipulated will depend on the application in question. Developers may
// of course entirely disable editing for the field via +link{listGridField.canEdit} or
// +link{listGridField.canEditCell()}.
// <P>
// See also: +link{listGrid.useRowSpanStyling}
//
// @value "first" This setting assumes that only the field-value for the first record
//  spanned by this cell is significant. In this case the editor will only show for this
//  cell if the user is editing the first spanned record. If the user initialized the edit
//  on another spanned row, the editor will not show for this field.
// @value "each" This setting assumes that each row's values are logically separate, so if
//  a cell spans multiple rows, and a user initializes an edit on some cell in the
//  second spanned row, the spanning cell will show an editor containing the value for
//  the second spanned row. This may differ from the value displayed when not
//  in edit mode (which is derived from the first spanned row by default). This setting may
//  be useful for developers who which to implement their own logic on how to handle
//  spanning cell display values and/or edit values (for example by customizing
//  +link{listGridField.formatCellValue} and applying custom logic to handle editing on
//  +link{listGridField.editorEnter} and +link{listGridField.editorExit}).
//
// @visibility external
//<


// Value Icons
// ---------------------------------------------------------------------------------------

_valueIconStyleTemplate:[
    "style='margin-left:",  // [0]
    ,                       // [1] - icon padding
    "px;margin-right:",     // [2]
    ,                       // [3] - icon padding
    "px;'"
],

//> @method ListGrid.getValueIcon()
// Returns the appropriate valueIcon for a cell based on the field and the data value for the 
// cell. Default implementation returns null if +link{ListGridField.suppressValueIcon} is true
// otherwise looks at +link{ListGridField.valueIcons}.
// @param field (ListGridField) field associated with the cell
// @param value (any) data value for the cell's record in this field.
// @param record (ListGridRecord) record associated with this cell
// @return (SCImgURL) url for the icon
// @group imageColumns
// @visibility external
//<
// We need the rowNum for checking whether the cell is editable
// calculate this from the record object via findRowNum(), but also support being passed an
// explicit rowNum for critical path code 
getValueIcon : function (field, value, record, rowNum) {
    if (!field.valueIcons || field.suppressValueIcon) {
        if (this._formatBooleanFieldAsImages(field)) {
            var img = (value ? this.booleanTrueImage : this.booleanFalseImage);
            // if the cell can't be edited and can be toggled, make sure it displays the 
            // disabled checkbox icon
            rowNum = (rowNum != null) ? rowNum : this.findRowNum(record);
            var colNum = field.masterIndex;
            if (!this.canEditCell(rowNum, colNum) && field.canToggle && img != this._$blank) {
                img = isc.Img.urlForState(img, false, false, isc.StatefulCanvas.STATE_DISABLED);
            }
            // If no image was specified, still write out a blank gif into the slot - this
            // allows us to recognize events over the (invisible) icon for canToggle behavior
            if (img == null || img == this._$blank) img = isc.Canvas.getImgURL(isc.Canvas._blankImgURL);
            return img;
        }
        return null;
    }
    var icon = field.valueIcons[value];
    return icon;
},

//> @method listGrid.getValueIconStyle() [A]
// Returns the CSS style for a cell based on the field and the data value for the cell.
// @param field (ListGridField) field associated with the cell
// @param value (any) data value for the cell's record in this field.
// @param record (ListGridRecord) record associated with this cell
// @return (CSSStyleName) CSS style to use, or null if no style should be used
//<
getValueIconStyle : function (field, value, record, rowNum) {
    if (this.booleanBaseStyle && field && (!field.valueIcons || field.suppressValueIcon)) {
        if (this._formatBooleanFieldAsImages(field)) {
            var imgStyle = this.booleanBaseStyle + (value ? "True" : "False");
            // if the cell can't be edited and can be toggled, make sure it displays the
            // disabled checkbox style
            rowNum = (rowNum != null) ? rowNum : this.findRowNum(record);
            var colNum = field.masterIndex;
            if (!this.canEditCell(rowNum, colNum) && field.canToggle) {
                imgStyle += isc.StatefulCanvas.STATE_DISABLED;
            }
            return imgStyle;
        }
    }
    return null;
},

// Helpers to get padding on each side of a field's valueIcon
getValueIconLeftPadding : function (field) {
    return (field && field.valueIconLeftPadding != null ? field.valueIconLeftPadding 
            : this.valueIconLeftPadding || 0);
},

getValueIconRightPadding : function (field) {
    return (field && field.valueIconRightPadding != null ? field.valueIconRightPadding 
            : this.valueIconRightPadding || 0);
},

// showValueIconOnly - returns true if the valueIcon should be displayed without any 
// text value for some field.
_$boolean:"boolean",
showValueIconOnly : function (field) {
    if (field.showValueIconOnly != null) return field.showValueIconOnly;
    // See discussion near showValueIconOnly docs.
    // If we have a valueIcons map, with no vmap, return text AND icon
    // if we have both valueIcons and a vmap, return just the icon
    // if we have no icon map, obviously return text (and icon, which is null)
    if (field.valueIcons != null && field.valueMap != null) return true;
    
    // If we are looking at a boolean field for which we want to show just the checkbox images
    // return true - otherwise false
    return this._formatBooleanFieldAsImages(field);

},

//> @method ListGrid.getValueIconWidth()
// If some field is showing valueIcons, this method returns the width those items should render
// Default implementation derives this from the first specified of 
// +link{ListGridField.valueIconWidth}, +link{ListGridField.valueIconSize}, 
// +link{ListGrid.valueIconWidth}, or +link{ListGrid.valueIconSize}
// @param field (object) field for which we're retrieving the valueIcon width
// @return (number) width for the icon
// @group imageColumns
// @visibility internal
//<
getValueIconWidth : function (field) {
    
    if (this.isCheckboxField(field)) return this._getCheckboxFieldImageWidth();
    if (this._formatBooleanFieldAsImages(field)) return this.booleanImageWidth;

    return (field.valueIconWidth != null ? field.valueIconWidth  :
                (field.valueIconSize != null ? field.valueIconSize : 
                    (this.valueIconWidth != null ? this.valueIconWidth : this.valueIconSize)));
},

//> @method ListGrid.getValueIconHeight()
// If some field is showing valueIcons, this method returns the height those items should render
// Default implementation derives this from the first specified of 
// +link{ListGridField.valueIconHeight}, +link{ListGridField.valueIconSize}, 
// +link{ListGrid.valueIconHeight}, or +link{ListGrid.valueIconSize}
// @param field (object) field for which we're retrieving the valueIcon height
// @return (number) height for the icon
// @group imageColumns
// @visibility internal
//<
getValueIconHeight : function (field) {
    if (this.isCheckboxField(field)) return this._getCheckboxFieldImageHeight();
    if (this.isExpansionField(field)) return this._getExpansionFieldImageHeight();
    if (this._formatBooleanFieldAsImages(field)) return this.booleanImageHeight;
    
    return (field.valueIconHeight != null ? field.valueIconHeight  :
                (field.valueIconSize != null ? field.valueIconSize : 
                    (this.valueIconHeight != null ? this.valueIconHeight : this.valueIconSize)));
},

// New record row: optional special row added to encourage new record entry
// ---------------------------------------------------------------------------------------

_isNewRecordRow : function (recordNum) {
    return this.showNewRecordRow && (recordNum == this.getTotalRows()-1);
},

//> @method ListGrid.getNewRecordCellValue()
//  Returns the contents to display in the new record row.
//  Note that this row displays a single cell spanning every column.
//  @return (string)    Value to display in new record row. Default 
//                      implementation returns <code>"-- Add New Row --"</code>
//<
getNewRecordRowCellValue : function () {
    return '<div align="center">' + this.newRecordRowMessage + '</div>';
},

// ---------------------------------------------------------------------------------------
getErrorIconHTML : function (rowNum, colNum) {
    var errors = this.getCellErrors(rowNum, colNum);
    if (errors == null) return isc._emptyString;

    var HTML = this.imgHTML(this.errorIconSrc, this.errorIconWidth, this.errorIconHeight, null, 
                            " isErrorIcon='true'");

    // Since we're writing the icon out to the left of our content, write a spacer out to the
    // right of the image to give us some padding between the image and the cell content
    
    if (isc.Browser.isIE && this._editorShowing && this.getEditRow() == rowNum && 
        (!this.editByCell || this.getEditCol() == colNum)) {
        HTML += " ";
    } else {
        
        HTML += isc.Canvas.spacerHTML(this.cellPadding, "auto");
    }
    return HTML;

},

// Handle events over the error icon by showing a hover containing the
// error text
_handleErrorIconOver : function (rowNum, colNum) {
    this._overErrorIcon = [rowNum,colNum];
    
    // This will fire the method on a delay unless the hover is already up
    // in which case it'll update it immediately.
    isc.Hover.setAction(this, this._handleErrorIconHover);
    

},
_handleErrorIconHover : function () {
    if (this._overErrorIcon == null) return;

    var rowNum = this._overErrorIcon[0],
        colNum = this._overErrorIcon[1];
        
    var errors = this.getCellErrors(rowNum, colNum);
    if (errors == null) return;
    
    var promptString = isc.FormItem.getErrorPromptString(errors);
    isc.Hover.show(promptString, this._getHoverProperties());
},

_handleErrorIconOut : function () {
    delete this._overErrorIcon;
    
    if (isc.Hover.isActive) {
        isc.Hover.clear();
    }
},

// _formatCellValue: Helper method to format the static cell value using developer defined
// formatCellValue() methods.
_$text:"text",
_formatCellValue : function (value, record, field, rowNum, colNum) {
    if (field && field.userSummary) {
        value = this.getSummaryFieldValue(field, record);
    
    } else {
        // If this is a summary row (group or grid level) we support custom formatting.
        // If custom formatters (formatGroupSummary / formatGridSummary) are present, apply them
        // and don't then run through additional formatting logic.
        var customFormatApplied = false;
        if (record && record[this.groupSummaryRecordProperty]) {
            if (field && field.formatGroupSummary) {
                if (!isc.isA.Function(field.formatGroupSummary)) {
                    field.formatGroupSummary =
                        isc.Func.expressionToFunction("value", field.formatGroupSummary);
                }
                if (isc.isA.Function(field.formatGroupSummary)) {                
                    value = field.formatGroupSummary(value);
                    customFormatApplied = true;
                }
            }
        }
        
        
        if (record && this._isSummaryRow) {
            if (field && field.formatGridSummary) {
                if (!isc.isA.Function(field.formatGridSummary)) {
                    field.formatGridSummary =
                        isc.Func.expressionToFunction("value", field.formatGridSummary);
                }
                if (isc.isA.Function(field.formatGridSummary)) {
                    value =  field.formatGridSummary(value);
                    customFormatApplied = true;
                }
            }
        }
        
        // If we already ran through group/grid summary formatters, don't run
        // standard cellFormatters on top of them!
        if (!customFormatApplied) {
            
            if (field && field.formatCellValue != null) {
                value = field.formatCellValue(value,record,rowNum,colNum,this);
            } else if (field && field.cellValueTemplate) {
                // NOTE: 
                // - probably don't need grid.cellValueTemplate, as this would be rare
                // - not exposed publicly yet
                // - might want XSLT option
                value = field.cellValueTemplate.evalDynamicString(this, {
                            value:value, record:record, field:field});
            // listGrid-wide formatter
            } else if (this.formatCellValue != null) {
                value = this.formatCellValue(value, record, rowNum, colNum);
                
            
            } else if (field && !field._suppressTypeFormatting) {
                
                value = this.applyCellTypeFormatters(value, record, field, rowNum, colNum);
            }
        }
    }
    // formatValueAsString() performs final conversions such as catching the case
    // where the value is null or "" and mapping to "&nbsp;" in the cell
    // We already do this as part of "getDefaultFormattedValue()".
    // A custom formatCellValue implementation may call 'getDefaultFormattedValue()' directly
    // in this case 'formatValueAsString()' will actually run twice, but should have
    // no ill effects since since the string won't be further modified by a second pass
    // through this method.
    value = this.formatValueAsString(value, record, field, rowNum, colNum);

    // hook for final processing of the display value that is applied to the actual display
    // value as derived by the various other hooks
    
    if (this.formatDisplayValue) {
        value = this.formatDisplayValue(value, record, rowNum, colNum);
    }
    
    // support escapeHTML flag per field
    if (field && field.escapeHTML) {
        if (isc.isA.String(value) && value != this._$nbsp) {
            value = value.asHTML();
        }
    }

	return value;
},

//> @method listGrid.getDefaultFormattedValue() [A]
// Get the value for some cell with default formatters applied.
// <P>
// This method is useful for cases where a developer wishes to
// conditionally customize a cell's formatting, but needs to see what
// the default formatted value would be.
// <P>
// For example - a developer might wish to apply a custom 
// +link{ListGridField.formatCellValue(),formatter} to some
// <code>link</code> type field, and be able to return the default 
// active link HTML in some cases. In this case a formatter could
// check for the conditions in which custom formatting should be applied
// and run appropriate custom logic to generate a value for display - otherwise
// return the result of this method to leave the standard formatted-value intact.
//
// @param record (Record) the cell's record object
// @param rowNum (int) rowNum for the cell
// @param colNum (int) colNum for the cell
// @return (string) Cell value with default formatters applied
// @see ListGridField.formatCellValue
// @visibility external
//<

getDefaultFormattedValue : function (record, rowNum, colNum) {
    var field = this.getField(colNum);
    var value = this.applyCellTypeFormatters(
                    this.getRawCellValue(record, rowNum, colNum), 
                    record, field, rowNum, colNum
                );
    return this.formatValueAsString(value, record, field, rowNum, colNum);
},

//> @method listGrid.getDefaultFormattedFieldValue() [A]
// Get a field value for some record with default field formatters applied.
// <P>
// This method differs from +link{getDefaultFormattedValue()} in a couple of ways. 
// Firstly, this method does not rely on the rowNum and colNum parameters to find the
// record and field in the grid.<br>
// Secondly, unlike +link{getDefaultFormattedValue()} this method <i>will</i> call
// any explicit cell formatter specified on the field passed in (though it will not
// call a +link{listGrid.formatCellValue(),component level formatter} if one exists).
// <P>
// This is useful for cases where a developer wishes to
// display a formatted value for some record and field combination which does not
// necessarily map to a cell displayed in the ListGrid.
// <P>
// If <code>rowNum</code> and <code>colNum</code> parameters are passed through to
// the field level cell formatter if one exists. If not explicitly provided these are
// defaulted to -1.
//
// @param record (Record) the record object
// @param field (ListGridField) the field object
// @param [rowNum] (int) rowNum (passed to any field level cell formatter)
// @param [colNum] (int) colNum (passed to any field level cell formatter)
// @return (string) Default formatted value
// @see ListGridField.formatCellValue
// @visibility external
//<

getDefaultFormattedFieldValue : function (record, field, rowNum, colNum) {
    var rawValue = isc.DataSource.getFieldValue(field, record);
    
    if (rowNum == null) rowNum = -1;
    if (colNum == null) colNum = -1;
    
    var value;
    if (field && field.formatCellValue != null) {
        value = field.formatCellValue(rawValue,record,rowNum,colNum,this);
    } else {
         value = this.applyCellTypeFormatters(rawValue, record, field, rowNum, colNum);
    }
    return this.formatValueAsString(value, record, field, rowNum, colNum);
},

// Apply type-formatters to a cell value.
applyCellTypeFormatters : function (value, record, field, rowNum, colNum, isMultipleElement) {
    
    if (!isMultipleElement && field && field.multiple && isc.isA.Array(value)) {
        var values = [];
        for (var i = 0; i < value.length; i++) {
            values[i] = this.applyCellTypeFormatters(value[i], record, field, 
                                                    rowNum, colNum, true);
        }
        // multipleValueSeparator documented at the DataSource level
        return values.join(field.multipleValueSeparator || ", ");
    }
    
    // check for formatter defined on a SimpleType definition
    if (field && field._shortDisplayFormatter != null) {
        value = field._simpleType.shortDisplayFormatter(value, field, this, record, rowNum, colNum);
    // We apply some standard (default) formatters to fields with particular data types.
    // NOTE: these should be moved to the built-in SimpleType definitions
    } else if (field && field._typeFormatter != null) {
        value = field._typeFormatter(value, field, this, record, rowNum, colNum);
    }
    
    // Note: this method does *not* run if there is a custom formatter
    // applied to a field (including a custom grid/group formatter for grid/group summary rows).
    //
    // apply the plural title if the special flag was set on the record
    if (value != null && !isc.isA.emptyString(value) && 
        record && field && record[this.recordApplyPluralTitleProperty]) 
    {
        var fieldNames = record[this.recordApplyPluralTitleProperty];
        if (fieldNames.contains(field.name)) {
            var title = field.pluralTitle;
            if (title == null) title = field.title;
            if (title != null) value += " " + title;
        }
    }
    // For summary rows where the summary method returned null (indicating unable to calculate a
    // summary), return the invalidSummaryValue if specified.
    if (value == null && 
        record && field &&
        this.invalidSummaryValue != null &&
        ((record[this.groupSummaryRecordProperty] && this.shouldShowGroupSummary(field)) ||
         (record[this.gridSummaryRecordProperty] && this.shouldShowGridSummary(field)))
       )
    {
        value = this.invalidSummaryValue;
    }
    
    return value;
},

// formatValueAsString: This takes a value to be displayed in a cell and ensures it displays
// correctly.
// This is run after any custom formatters / type-based formatters have been applied to
// the value.
formatValueAsString : function (value, record, field, rowNum, colNum) {

	// For "empty" values, write out the emptyCellValue
    
	if (value == null || isc.is.emptyString(value)) {
		// if the field specifies a 'emptyCellValue' 
		if (field.emptyCellValue != null) {
			// return the field-specific value
			value = field.emptyCellValue;
		} else if (field.type == "summary") {
		    value = this.invalidSummaryValue;
   
		} else {
			// otherwise return the emptyCellValue for the entire list
			value = this.emptyCellValue;
		}
        
    // In IE, an element containing only whitespace characters (space or enter) will not show css
    // styling properly.
    
    } else if (this._emptyCellValues[value] == true) {
        value = this._$nbsp;

	// convert the value to a string if it's not already        
	} else if (!isc.isA.String(value)) {
        // For date type values we want to respect dateFormatter / timeFormatter specified at the
        // field or component level.
        // If the specified fieldType is "date", "datetime" or "time" this is already handled by the
        // _typeFormatter function (which calls formatDateCellValue / formatTimeCellValue for 
        // these fields)
        
        if (isc.isA.Date(value)) {
        
            // We already know the "type" of this field is not "time" (as formatTimeCellValue)
            // would have run and converted it to a string already.
            // So we'll actually only format as time if the field has ane explicit timeFormatter
            // and no explicit dateFormatter.
            if (this._formatAsTime(field)) {
                var formatter = this._getTimeFormatter(field);
                var isLogicalTime = isc.SimpleType.inheritsFrom(field.type, "time");
                value = isc.Time.toTime(value, formatter, isLogicalTime);
            } else {
    
                var isDatetime = field && isc.SimpleType.inheritsFrom(field.type, "datetime"),
                    isLogicalDate = !isDatetime && field && isc.SimpleType.inheritsFrom(field.type, "date"),
                    formatter = this._getDateFormatter(field);
                if (isDatetime) value = value.toShortDateTime(formatter, true);
                else value = value.toShortDate(formatter, !isLogicalDate);
            }
        } else {
            value = isc.iscToLocaleString(value);
        }
    }
    return value;
},

// these Strings can be considered to be "empty" cells, causing bad styling.  Replace with
// '&nbsp;'

_emptyCellValues:{" ":true, "\n":true, "\r":true, "\r\n":true},
_$nbsp:"&nbsp;",

//>	@method listGrid.getEditItemCellValue()   ([IA])
//          Returns the HTML for a cell within a row that is being edited (as a result of a call
//          to 'editRow')<br>
//          Will <i>not</i> call 'updateEditRow()' to update the values displayed in the edit
//          row - this must be handled by calling methods, once we know the form element has
//          been written into the DOM.
//      @visibility internal
//		@group	editing
//
//      @param  record      (object)    the current record object
//      @param  rowNum      (number)    index of the record containing this cell
//      @param  colNum      (number)    index of the field containing this cell
//
//		@return	(string)	display value for this cell
//<
getEditItemCellValue : function (record, rowNum, colNum) {
    var itemName = this.getEditorName(rowNum, colNum);
    
    // If the editRowForm or item haven't yet been created, no op - implies this
    // is being called at an invalid time
    
    if (!this._editRowForm || !this._editRowForm.getItem(itemName)) {
        return "&nbsp;"
    }

	// Write a form item out into this cell.
	// We have already created the '_editRowForm' - a dynamic form with an appropriate set of
	// form items (see 'editRow').
	// Make use of the appropriate item's getStandaloneHTML() to write out the form element
    
    // If we have any errors for the field, set them on the form item too so the error icon
    // gets rendered out
    var errors = this.getCellErrors(rowNum, colNum);
    if (errors) {
        this._editRowForm.setFieldErrors(itemName, errors);
    }

	// get the HTML for the form item
	// Relies on the form item being present - this is fine as long as our logic to create
	// and remove edit form items for incremental rendering stays in synch with the set of 
	// cells being written out.    
    var item = this._editRowForm.getItem(itemName),    
        HTML = item.getStandaloneItemHTML(item.getValue(), false, true);

    // once we've retrieve the HTML, clear the errors so if we re-render the form for another item
    // etc, we don't end up with errors hanging around
    if (errors) {
        this._editRowForm.setFieldErrors(itemName, null);
    }
    
	
    var body = item.containerWidget;
    if (!body._drawnEditItems) body._drawnEditItems = [];   
    if (!body._drawnEditItems.contains(item)) {
        body._drawnEditItems.add(item);
    }
    
    return HTML;
},

// should we show inactive version of the edit form item for some cell?
_showInactiveEditor : function (fieldNum) {
    return this._alwaysShowEditors(fieldNum);
},


// getInactiveEditorCellValue()
// If this.alwaysShowEditors is true, we write inactive versions of our edit form items in every
// row (with appropriate value displayed for the record in question)
getInactiveEditorCellValue : function (record, rowNum, colNum) {
    var itemName = this.getEditorName(rowNum, colNum);

	// get the HTML for the form item
	// Relies on the form item being present - this is fine as long as our logic to create
	// and remove edit form items for incremental rendering stays in synch with the set of 
	// cells being written out.    
    var item = this._editRowForm.getItem(itemName);
    // Don't crash if the form doesn't actually contain an item for this field.
    
    if (item == null) {
        return this.getFormattedValue(record, rowNum, colNum);
    }
    var editorType = isc.DynamicForm.getEditorType(item,this._editRowForm),
        value = this.getRawCellValue(record, rowNum, colNum), 
        HTML;
        
    if (this.shouldShowEditorPlaceholder(this.getField(colNum),editorType)) {
        HTML = this.getEditorPlaceholderHTML(editorType, value, record,rowNum,colNum);
    } else {
    
        var context = {grid:this.getID(), record:record, rowNum:rowNum, colNum:colNum}
        
        var errors = this.getCellErrors(rowNum, colNum);
        if (errors) {
            this._editRowForm.setFieldErrors(itemName, errors);
        }
        HTML = item.getInactiveEditorHTML(value, false, true, context);
        if (errors) {
            this._editRowForm.setFieldErrors(itemName, null);
        }

        // the inactiveEditorContext ID gets stored directly on the cell
        // When we refresh the cell we'll throw away this inactive context (and potentially create
        // another one)
        // We want to be able to map from row+colNum to contextID so we don't have
        // to iterate through all the inactiveContexts on all the form items
        
        if (!this._inactiveEditorContextMap) this._inactiveEditorContextMap = {};
        var row_col = "_" + rowNum + "_" + colNum;
        if (this._inactiveEditorContextMap[row_col]) {
            this.logWarn("creating new inactiveEditor context for a cell without having cleared " +
                "the previous one." + this.getStackTrace(), "inactiveEditorHTML");
        }
        this._inactiveEditorContextMap[row_col] = context;
    }
    
	
    return HTML;
},


// if we're showing editors for every record, if this method returns true, suppress showing
// inactive editorHTML and show simplified placeholder HTML instead for the field in question.
// This allows us to not have to ensure that every editorType supports inactiveHTML in an efficient
// manner.
shouldShowEditorPlaceholder : function (field, editorType) {

    if (field.showEditorPlaceholder != null) return field.showEditorPlaceholder;
    
    return !this._supportsInactiveEditorHTML[editorType];
},
// list of editorTypes that support inactiveEditorHTML for alwaysShowEditors:true

_supportsInactiveEditorHTML:{
    text:true, TextItem:true,
    select:true, SelectItem:true,
    combobox:true, ComboBoxItem:true, comboBox:true,
    checkbox:true, CheckboxItem:true,
    date:true, DateItem:true,
    spinner:true, SpinnerItem:true,
    popUpTextArea:true, PopUpTextAreaItem:true
},

// getEditorPlaceholderHTML()
// If this.alwaysShowEditors is true, return the placeholder HTML to show in place of
// inactiveEditorHTML for cells where shouldShowEditorPlaceholder returns true.
getEditorPlaceholderHTML : function (editorType, value, record, rowNum, colNum) {
    // for now just return the value.
    return value;
},

// _editItemsDrawingNotification - function to notify the edit row form items when they are
// drawn, cleared or redrawn.

_editItemsDrawingNotification : function (item, fireMoved, gr) {
    var items;
    if (item) items = [item];
    else {
        items = [];
        var allItems = this._editRowForm.getItems();
        for (var i = 0; i < allItems.length; i++) {
            if (allItems[i].containerWidget == gr) items.add(allItems[i]);
        }
    }   
    var newlyDrawnItems = gr._drawnEditItems;

    for (var i = 0; i < items.length; i++) {
        var currentItem = items[i],
            wasDrawn = currentItem.isDrawn(),
            isDrawn = newlyDrawnItems ? newlyDrawnItems.contains(currentItem) : false,
            isCanvasItem = isc.CanvasItem && isc.isA.CanvasItem(currentItem);

        if (wasDrawn) {
            if (isDrawn) {
                currentItem.redrawn();
                // Redraw of the body will frequently result in repositioning the edit cells.
                // Fire the 'moved' handler on any visible form items that were present before
                // the redraw to notify them of being moved.
                
                if (fireMoved) currentItem.moved();
            } else {
                currentItem.cleared();
            }
            
        } else if (isDrawn) {
            // newly drawn
            if (isCanvasItem) currentItem.placeCanvas();
            currentItem.drawn();
        }
    }
    
    if (this._editRowForm) {
        this._editRowForm.destroyOrphanedItems("Grid edit-items removed");
    }
    
    
            
    // Get rid of the _drawnEditItems. Next time this method is run we only want to catch 
    // items which have subsequently been drawn
    delete gr._drawnEditItems;
},

// clearingInactiveEditorHTML
// Called when we're about to clear the generated inactiveEditorHTML due to cell refresh
// (so clearing a single row), or redraw (clearing all).
// Use this to clean up the generated inactive context objects so we don't fill up lots of memory on
// every redraw etc.
_clearingInactiveEditorHTML : function (rowNum,colNum) {

    // Don't crash if we have no inactiveEditorContextMap set.
    
    if (this._inactiveEditorContextMap == null) return;
    
    // have cells be rowNum/colNum pairs allowing us to clear just a single row, or
    // if we find we need more intelligent incremental update handling, single col, etc
    if (!this._editRowForm ||
        !this._alwaysShowEditors((colNum != null ? this.getField(colNum) : null),true)) return;
    
    if (rowNum != null) {
        var row_col = "_" + rowNum + "_" + colNum,
            context = this._inactiveEditorContextMap[row_col];
        if (context) {
            context.formItem.clearInactiveEditorContext(context);
            delete this._inactiveEditorContextMap[row_col];
        }
    } else {
        var editForm = this.getEditForm(),
            items = editForm.getItems();

        for (var i = 0; i < items.length; i++) {
            items[i].clearAllInactiveEditorContexts();
        }
        
        delete this._inactiveEditorContextMap;
    }
},

// setRecordValues()
// Method to update client-side data in place
// This is called directly by DynamicForms when saving values if this is acting as the selection
// component for a form.
setRecordValues : function (pks, values) {
    if (!this.shouldSaveLocally()) {
        this.logWarn("setRecordValues() not supported for grids where saveLocally is false");
        return;
        
    }
    if (!this.data) return;
    
    var rowNum = this.data.indexOf(pks),
        record = this.data.get(rowNum);
        
    // Incoming values will be in a structured form matching the dataPath, which is no good to
    // us because the records inside this component only contain fields from a specific 
    // sub-hierarchy.  For example, if this grid is bound to dataPath "/items" it will have
    // flat-named fields called "a" and "b" that are actually at dataPaths "/items/a" and 
    // "/items/b".  New values will come into this method in that complete form, so unless 
    // we do something, our record will up looking like this:
    // {
    //   a: "something",
    //   b: 10,
    //   items: {
    //     a: "the new value",
    //     b: 100
    //   }
    // }
    //
    // So, we just create a new values object by pulling out whatever is at this component's 
    // dataPath in the passed-in values (it will be the same as the form's dataPath, because 
    // the only way we get here is for selectionComponents, and a selectionComponent implicitly
    // shares a dataPath with the forms for which it is a selectionComponent)
    
    var newValues = isc.Canvas._getFieldValue(this.dataPath, null, values, this, true);
    
    
    if (!newValues) {
        newValues = values;
    }
        
    this.combineObjects(record, newValues);

    this.calculateRecordSummaries([record]);
    if (this.useCellRecords) {
        rowNum = this.findRowNum(record);
        var colNum = this.findColNum(record);
        this.refreshCell(rowNum,colNum);
    } else {
        this.refreshRow(rowNum);
    }
    
    // Reapply sort and filter, in case this change affects them
    this.resort();
    this.filterByEditor();
    
    if (this.valuesManager != null) {
        // _updateMultipleMemberValue handles combining a data-index into a path
        // 2nd param unnecessary since we're modifying the whole record, not a field within it.
        this.valuesManager._updateMultipleMemberValue(rowNum, null, record, this);
    }
    
},



combineObjects : function (destination, source) {
    return isc.combineObjects(destination, source);
},

//>	@method	listGrid.setRawCellValue()
//		@group	data
//			Set the raw data for one particular cell in the list.
//
//		@param	record		(object)	record in question
//		@param	recordNum	(number)	number of that record
//		@param	fieldNum	(number)	number of the field to display
//		@param	newValue	(any)		new value
//
//<
// Overridden at the cubeGrid level to handle being passed an entire record rather than a single
// field value for the cell
setRawCellValue : function (record, recordNum, fieldNum, newValue) {

	var field = this.fields[fieldNum];
	// if record or field is null, we're dealing with an invalid column, so forget it
	if (!record || !field) return;
	// if the field has a 'setRawCellValue' attribute
	// We'll assume this DIRECTLY sets the value on the LG's data object, so we're basically done 
	// at this point (other than firing dataChanged below to ensure everything gets updated 
	// visually)
	if (field.setRawCellValue) {
		// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value"
		// Convert a string callback to a function
		isc.Func.replaceWithMethod(field, "setRawCellValue", "viewer,record,recordNum,field,fieldNum,value");

		// call it as a function (returns string | null)
		field.setRawCellValue(this, record, recordNum, field, fieldNum, newValue);
        
	// otherwise just use the cells value in the normal way
	} else {
        isc.Canvas._saveFieldValue(null, field, newValue, record, this, true);
    }    

	// HACK: fire dataChanged() by hand.  Really, we need an interface to update fields on
	// objects in the List so the List can recognize the change and fire dataChanged() itself
    this.getOriginalData().dataChanged();
},

//>	@method	listGrid.getCellBooleanProperty()	(A)
//  Given a property name, and a cell, check for the value of that property (assumed to be a 
//  boolean) on the ListGrid, and the cell's field, returning false if the value is false at 
//  either level.
//  If true at the ListGrid and Field level, check the value of the the second "recordProperty"
//  on the record object for the cell.
//  (If recordProperty is not passed, the record object will not be examined).
//
//	@param	property	(string)	Name of the property to look for.
//	@param	rowNum	(number)	Row number of the cell.
//	@param	colNum	(string)	Field number for the cell.
//  @param  [recordProperty]    (string)    Name of the equivalent property to check on the
//                                          record object 
//
//	@return	(boolean)	true == at least one is true and none are false
//
//<

_$false:"false", _$true:"true",
getCellBooleanProperty : function (property, recordNum, fieldNum, recordProperty) {
	var trueFound = false,
		listValue = this[property]
	;
	if (listValue == false || listValue == this._$false) return false;

	var fieldValue = this.fields[fieldNum][property];
	if (fieldValue == false || fieldValue == this._$false) return false;
    
    if (recordProperty != null) {

    	var record = this.getRecord(recordNum, fieldNum),
	    	recordValue = (record != null ? record[recordProperty] : false)
    	;
    	if (recordValue == false || recordValue == this._$false) return false;
        if (recordValue == true || recordValue == this._$true) return true;
    }
	
	// At this point we know none of the values was an explicit false - but we only want to 
	// return true if the value was specified as true (rather than undefined) at some level.
	// We've already checked at the record level (if necessary)
	return (listValue == true) || (fieldValue == true) ||
            (listValue == this._$true) || (fieldValue == this._$true);
},


// ---------------------------------------------------------------------------------------

//> @method listGrid.setShowRecordComponents()
// Setter for the +link{listGrid.showRecordComponents} attribute
// @param showRecordComponents (boolean) new value for <code>this.showRecordComponents</code>
// @visibility external
//<
setShowRecordComponents : function (showRC) {
    if (this.showRecordComponents == showRC) return;
    
    
    if (showRC) {
        if (this.animateFolders) {
            this._animateFoldersWithoutRC = true
            this.animateFolders = false;
        }
    } else {
        if (this._animateFoldersWithoutRC) {
            this.animateFolders = true;
            delete this._animateFoldersWithoutRC;
        }
    }
    
    this.showRecordComponents = showRC;
    
    // Update virutalScrolling if necessary.
    // We'll also update virtual scrolling in createBodies() - this handles the case where
    // we're already showing recordComponents and a grid is frozen at runtime.
    this._updateVirtualScrollingForRecordComponents();
    
    // suppress 'drawAllMaxCells' type behavior - we don't want to render out potentially
    // hundreds of canvases.
    if (showRC) {
        this._oldDrawAllMaxCells = this.drawAllMaxCells;
        this.drawAllMaxCells = 0;
        if (this.body != null) this.body.drawAllMaxCells = 0;
      
    } else {
        if (this._oldDrawAllMaxCells != null) {
            this.drawAllMaxCells = this._oldDrawAllMaxCells;
            if (this.body != null) this.body.drawAllMaxCells = this._oldDrawAllMaxCells;
            delete this._oldDrawAllMaxCells;
        }
        
    }
    this.invalidateRecordComponents();
    
    // For "within" record components, if autoFitting to values, we may need to
    // resize smaller once the components have been removed.
    if (!showRC && 
        (this.recordComponentPosition == this._$within ||
         (this.recordComponentPosition == null && this.showRecordComponentsByCell))) 
    {
        this.updateFieldWidthsForAutoFitValue("showRecordComponents changed");
        this._updateFieldWidths();
    }

},

_updateVirtualScrollingForRecordComponents : function () {
    if (!this.showRecordComponents) {
        if (this._rcVScroll) {
            delete this.virtualScrolling;
            delete this._rcVScroll;
        }
    
    // Virtual scrolling:
    // Embedded componenents can make row heights unpredictable
    // (may not show for every row, may be 'position:"expand"', or exceed this.cellHeight 
    // etc)
    // Because of this we typically have to enable virtual scrolling for record components.
    // However we don't currently support virtual scrolling with frozen fields, so
    // don't enable it if we have frozen fields.
    // This *may* lead to unpredictable behavior. Cases where it's ok:
    // - if recordComonentPosition is 'within' and 
    //   the recordComponentPosition < this.cellHeight
    // - if recordComponentHeight is set (and truly is not exceeded by embedded components)
    
    } else {
        if (this.virtualScrolling == null || this._rcVScroll) {
            if (this.frozenFields == null) {
                this.virtualScrolling = true;
                this._rcVScroll = true;
                
            } else {

                if (this.recordComponentHeight == null) {
                    this.logWarn("This grid has frozen fields and is showing " +
                        "recordComponents. This may lead to unpredictable row heights which " +
                        "are not supported with frozen fields. Setting " +
                        "listGrid.recordComponentHeight will avoid this issue.",
                        "recordComponents"); 
                }
                if (this._rcVScroll) {
                    delete this.virtualScrolling;
                    delete this._rcVScroll;
                }
            }
        }
    }
    
    if (this.body && this.virtualScrolling != this.body.virtualScrolling) {
        this.body.virtualScrolling = this.virtualScrolling;
        if (this.frozenBody) {
            this.frozenBody.virtualScrolling = this.virtualScrolling;
        }
    }
    // No need to call 'redraw' on the body -- calling code is expected to handle this.
},



//> @method listGrid.getDrawArea() (A)
// Returns the extents of the rows and columns currently visible in this grid's viewport.
// <P>
// Note: if there are any +link{listGridField.frozen,frozen fields}, they are not included
// in the draw area range returned by this method. Frozen fields are assumed to never be 
// scrolled out of view.  The column coordinates returned by this method will only include
// unfrozen columns.
//
// @return	(Array of Number)	The row/col co-ordinates currently visible in the viewport as
//    [startRow, endRow, startCol, endCol].
// @visibility external
//<

getDrawArea : function () {
    if (this.body) {
        var drawArea = this.body.getDrawArea();
        if (this.frozenFields && this.freezeStart()) {
            drawArea[2] += this.frozenFields.length;
            drawArea[3] += this.frozenFields.length;
        }
        return drawArea;
    }
    
    return null;
},

// _drawAreaChanged() - notification fired on GridRenderer.redraw() when the
// previous draw area doesn't match the new draw area

_drawAreaChanged : function (oldStartRow, oldEndRow, oldStartCol, oldEndCol, body) {
    if (this.frozenFields && this.freezeStart()) {
        oldStartCol += this.frozenFields.length;
        oldEndCol += this.frozenFields.length;
    }
    var oldDrawArea = [oldStartRow, oldEndRow, oldStartCol, oldEndCol];
    if (oldDrawArea.equals(this.getDrawArea())) return;
    this.drawAreaChanged(oldStartRow,oldEndRow,oldStartCol,oldEndCol);
},

// documented in registerStringMethods
drawAreaChanged:function () {},

// if any records were programatically expanded before the grid was drawn,
// expand them properly now
updateExpansionComponents : function () {
    if (!this._recordsToExpand || this._recordsToExpand.length == 0) return;
    this._updatingExpansionComponents = true;
    for (var i=this._recordsToExpand.length-1; i>=0; i--) {
        this.expandRecord(this._recordsToExpand[i]);
    }
    delete this._updatingExpansionComponents;
    delete this._recordsToExpand;
},

// updateRecordComponents() - fired from redraw on grid body (or frozen body).
// This method essentially iterates through our current draw area and ensures that if
// showRecordComponents is true, we're showing recordComponents for each row (or cell),
// calling 'createRecordComponent()' or 'updateRecordComponent()' as necessary to create
// new components, and discarding (or pooling) record components that we previously created
// which are no longer visible.
// This method should not need to be called by developers directly.
// To force an explicit invalidation and refresh of recordComponents, use 
// invalidateRecordComponents()
// *Note: This method also handles updating backgroundComponents if specified

updateRecordComponents : function () {
    // Sanity check to avoid infinite loops if adding embedded components trips redraw of
    // body for example
    var debugLog = this.logIsDebugEnabled("recordComponents");
    if (this._updatingRecordComponents) {
        if (debugLog) {
            this.logDebug("updateRecordComponents called recursively - returning",
                "recordComponents");
        }
        return;
    }
    
     // If we're performing a show/hide row height animation, bail.
    // In this case the HTML in the body won't match the set of records in our data set
    // so we can't update / place embedded components properly
    var body = this.body,
        frozenBody = this.frozenBody;
    if (body == null) return;
    
    if (body._animatedShowStartRow !=  null) {
        return;
    }
    
    // This method is fired from various places including gridBody.draw.
    // In this case we may have one body drawn and the other not yet drawn. Wait for both
    // to be drawn before continuing - addEmbeddedComponent (used below) requires the
    // GR be drawn.
    // Also if _fieldWidths isn't set, we can't
    // yet size/position our embedded components. This can occur when we rebuildForFreeze.
    // Catch this case and return.
    if ((!body.isDrawn() || body._fieldWidths == null || 
        (body._fieldWidths.length == 0 && body.fields.length > 0)) ||
        (frozenBody && 
            (!frozenBody.isDrawn() || frozenBody._fieldWidths == null || 
            (frozenBody._fieldWidths.length == 0 && frozenBody.fields.length > 0)))) 
    {
        return;
    }

    this._updatingRecordComponents = true;
    
    // Implementation overview: The concept here is that if showRecordComponents is true,
    // we call a method 'createRecordComponent()' [or potentially 'updateRecordComponent']
    // for every visible row, or if showing by cell, every visible cell, lazily as its rendered
    // out.
    // When new cells are rendered out we want to
    // - call createRecordComponent() [or updateRC] for newly rendered cells
    // - *not* call createRC for cells that were visible and still are (regardless of whether
    //   createRecordComponent returned an actual component or just null)
    // - for cells that are no longer visible, clear up the created record components, 
    //   clearing them, destroying them or adding them to our 'recordComponentPool' depending on
    //   the recordComponentPoolingMode.
    // Rather than trying to achieve this by tracking viewports (which has the major disadvantage
    // of being fragile on data change or field config change), we take this approach:
    // We store all generated record components in 2 places:
    // - on the records themselves, indexed by fieldName, under record._recordComponents
    // - on the ListGrid in both an array and an object mapping componentIDs to true
    // If the createRecordComponent method returned null for any cell, we store a special
    // nullMarker object on the record._recordComponents object as well.
    //
    // When this method runs we can then iterate through all visible records / fields
    // - determine if we have visible record components already present, or null markers,
    //   in which case we leave the component alone
    // - otherwise run the method to create a new record component / get one from the pool and
    //   apply it to the cell.
    // Once we've gone through all visible cells we iterate through all the recordComponents
    // we previously created and wipe out (either clear, destroy or recycle) any that weren't
    // noted as being attached to a visible cell in the previous step.
    
    // _liveRecordComponents / _liveRecordComponentsObj is the full set of recordComponents
    // generated last time this method was run.
    var oldRecordCompArr = this._liveRecordComponents || [],
        oldRecordCompObj = this._liveRecordComponentsObj || {};

    if (debugLog) {
        this.logDebug("updateRecordComponents - old record components before refreshing:" + 
            this.echo(oldRecordCompArr),
            "recordComponents");
    }
        
    this._liveRecordComponentsObj = {};
    this._liveRecordComponents = [];
    
    // If showRecordComponents is false we can skip all logic to create
    // new recordComponents. If we had any previously created recordComponents we'll clear
    // them below. This will handle the showRecordComponents setting being changed dynamically.
    if (this.showRecordComponents || this.showBackgroundComponents) {
        
        // Determine what our current draw area is - set of drawn fields and rows.
        // This method is being called as part of redraw, before the render has occurred, so
        // we can't just look at body._firstDrawnRow / _lastDrawnRow etc - we need
        // to call the getDrawArea() APIs on the body to actually calculate the new values
        
        var drawArea = this.body.getDrawArea(),
            cellComponents = this.showRecordComponentsByCell,
            bodyID = this.body.getID(),
            frozenBodyID = this.frozenBody ? this.frozenBody.getID() : null;
        if (debugLog) {
            this.logDebug("updating to potentially show recordComponents for drawArea:" 
                + drawArea, "recordComponents");
        }
        for (var rowNum = drawArea[0]; rowNum <= drawArea[1]; rowNum++) {
            var record = this.getRecord(rowNum);
            
            if (record == null || Array.isLoading(record)) continue;
            
            if (this.showRecordComponents) {
                
                // If we don't have cell components we will add components to the (unfrozen) body,
                // one per row.
                if (!cellComponents) {
                
                    var shouldShowRecordComponent = null;
                    // Skip spanned cells.
                    
                    if (this.allowRowSpanning && this.getRowSpan) {
                        var isSpanned = this.getCellStartRow(rowNum, 0) != rowNum;
                        if (isSpanned) shouldShowRecordComponent = false;
                    }
                    
                    if (shouldShowRecordComponent == null) {
                         
                         shouldShowRecordComponent = this.shouldShowRecordComponent(record);
                    }
                    var liveComp = null;
                    if (shouldShowRecordComponent) {
                        // getLiveRecordComponent() will pick up the record component we've already
                        // applied to the record/field.
                        // NOTE: If createRecordComponent ran and returned null we store a special
                        // null-marker object which we'll get back here as well. This means we
                        // don't re-run createRecordComponent() unless we actually want to.
                        liveComp = this._getLiveRecordComponent(record, null, bodyID);
                        if (liveComp != null) {
                            if (liveComp.isNullMarker) {
                                liveComp = null;
                            } else {
                                var ID = liveComp.getID();
                                oldRecordCompObj[ID] = null;
                            }
                        } else {
                            liveComp = this._applyNewRecordComponent(record, null, this.body, rowNum);
                        }
                    }
                    
                    // Store pointers to both the component and its ID
                    
                    if (liveComp != null) {
                        var ID = liveComp.getID();
                        this._liveRecordComponentsObj[ID] = true;
                        this._liveRecordComponents[this._liveRecordComponents.length] = liveComp;
                    }
                    
                // same logic as above, but applied per cell to both the frozen and
                // unfrozen body.
                } else {
                    if (this.frozenBody != null) {
                        for (var fieldNum = 0; fieldNum < this.frozenBody.fields.length; fieldNum++) {
                            var field = this.frozenBody.fields[fieldNum],
                                fieldName = field.name;
                            var shouldShowRecordComponent = null;
                            // Skip spanned cells.
                            if (this.allowRowSpanning && this.getRowSpan) {
                                var isSpanned = this.getCellStartRow(rowNum, field.masterIndex) != rowNum;
                                if (isSpanned) shouldShowRecordComponent = false;
                            }
                            if (shouldShowRecordComponent == null) {
                                shouldShowRecordComponent = this.shouldShowRecordComponent(record, field.masterIndex);
                            }
                            var liveComp = null;
                                
                            if (shouldShowRecordComponent) {
                                liveComp = this._getLiveRecordComponent(record, fieldName, frozenBodyID);
                                if (liveComp != null) {
                                    if (!liveComp.isNullMarker) {
                                        var ID = liveComp.getID();
                                        oldRecordCompObj[ID] = null;
                                    } else {
                                        liveComp = null;
                                    }
                                } else {
                                    liveComp = this._applyNewRecordComponent(record, fieldName, this.frozenBody,
                                                    rowNum, fieldNum);
                                }
                            }
                            if (liveComp != null) {
                                var ID = liveComp.getID();
                                this._liveRecordComponentsObj[ID] = true;
                                this._liveRecordComponents[this._liveRecordComponents.length] = liveComp;
                            }
                        }
                    }
                    for (var bodyCol = drawArea[2]; bodyCol <= drawArea[3]; bodyCol++) {
                        var field = this.body.fields[bodyCol],
                            fieldName = field.name;
                        var shouldShowRecordComponent = null;
                        // Skip spanned cells.
                        if (this.allowRowSpanning && this.getRowSpan) {
                            var isSpanned = this.getCellStartRow(rowNum, field.masterIndex) != rowNum;
                            if (isSpanned) shouldShowRecordComponent = false;
                        }
                        if (shouldShowRecordComponent == null) {
                            shouldShowRecordComponent = this.shouldShowRecordComponent(record, field.masterIndex);
                        }
                        var liveComp = null;
                        
                        if (shouldShowRecordComponent) {
                            var liveComp = this._getLiveRecordComponent(record, fieldName, bodyID);
                            if (liveComp != null) {
                                if (!liveComp.isNullMarker) {
                                     var ID = liveComp.getID();
                                     oldRecordCompObj[ID] = null;
                                } else {
                                    liveComp = null;
                                }
                            } else {
                                liveComp = this._applyNewRecordComponent(record, fieldName, this.body,
                                            rowNum, bodyCol);
                            }
                        }
                        
                        if (liveComp != null) {
                            var ID = liveComp.getID();
                            
                            this._liveRecordComponentsObj[ID] = true;
                            this._liveRecordComponents[this._liveRecordComponents.length] = liveComp;
                        }
                    }
                }
            }
            
            
            if (this.showBackgroundComponents) {
                if (record && record.backgroundComponent) {
                    var component = this._hasEmbeddedComponents(record) ? 
                            this._getEmbeddedComponents(record).find("isBackgroundComponent", true) : null;
        
                    if (!component) {
                        // should be showing a backgroundComponent but it's not present yet - add it now
                        if (isc.isA.Canvas(record.backgroundComponent)) {
                            // backgroundComponent is specified as a canvas
                            var comp = record.backgroundComponent.addProperties(
                                this.backgroundComponentProperties,
                                { isBackgroundComponent: true }
                            );
                        } else {
                            // backgroundComponent is specified as properties
                            var props = isc.addProperties({ isBackgroundComponent: true },
                                this.backgroundComponentProperties,
                                record.backgroundComponent);
                            var comp = this.createAutoChild("backgroundComponent", props);
                        }
        
                        var tableIndex = body.getTableZIndex();
                        comp.setZIndex(tableIndex - 49);
                        comp.setWidth("100%");
                        comp.setHeight("100%");
                        comp.setOverflow("hidden");
                        
                        comp.removeOnHideField = true;

                        
                        this.addEmbeddedComponent(record.backgroundComponent, record, rowNum, 
                            null, "within");
                        
                        // This should stick with the record until it's wiped due to data change
                        // or similar (EG remapEmbeddedComponents)
                        // At that point, if this method runs again it'll be cleared
                    }
                }
            }
        }
    }
    
    if (this.logIsInfoEnabled("recordComponents")) {
        this.logInfo("updateRecordComponents - new recordComponents:" + 
            this.echo(this._liveRecordComponentsObj) +
            ", old record components (will be cleaned up if value is 'true'):" + 
            this.echo(oldRecordCompObj), "recordComponents");
    }
    
    // At this point we've iterated through our draw area (or showRecordComponents is false,
    // in which case we want to drop all pre existant Record Components).
    // Any pre-existant recordComponents that are still visible have been removed from
    // the 'oldRecordCompObj'.
    // Iterate through pre-existant record components that are left and clear them up
    // (remove from DOM if necessary, destroy / pool if necessary)
    for (var i = 0; i < oldRecordCompArr.length; i++) {
        // if it's been cleared from the oldRecordCompObj we know its still visible / being used
        var ID = oldRecordCompArr[i].getID();
        if (oldRecordCompObj[ID] != true) {
            continue;
        }
        if (debugLog) {
            this.logDebug("cleaning up RecordComponent:" + oldRecordCompArr[i], "recordComponents");
        }
        this._cleanUpRecordComponent(oldRecordCompArr[i]);
        oldRecordCompObj[ID] = null;
    }
    delete this._updatingRecordComponents;
},

// _applyNewRecordComponent()
// This method will run 'createRecordComponent()' or 'updateRecordComponent()' to
// get the recordComponent for some record or cell.
_applyNewRecordComponent : function (record, fieldName, body, rowNum, bodyCol) {
    
    if (this.logIsDebugEnabled("recordComponents")) {
        this.logDebug("getting record component for row/field:" + [rowNum,fieldName],
            "recordComponents");
    }

    var pool = this.recordComponentPoolingMode == "recycle",
        component,
        // same row variable - only used if we're picking up a pooled component
        sameRow,
        colNum = fieldName == null ? null : this.getColNum(fieldName)
    ;

    if (pool) {
        var compConfig = this.getFromRecordComponentPool(record, fieldName);
        component = compConfig ? compConfig[0] : null;
        sameRow = compConfig ? compConfig[1] : null;
    }

    if (!component) {
        if (this.createRecordComponent && isc.isA.Function(this.createRecordComponent)) {
            component = this.createRecordComponent(record, this.getColNum(fieldName));
            if (component != null) component.isRecordComponent = true;
            this.logDebug("created new record component:" + component, "recordComponents");
        }
    } else {
        if (this.updateRecordComponent && isc.isA.Function(this.updateRecordComponent)) {
            var origComponent = component;

            component = this.updateRecordComponent(record, colNum, component, !sameRow);

            // component may well be null - this is valid and allows the developer to
            // suppress components in cells even in pooling mode.
            if (component == null) {
                if (this.logIsInfoEnabled("recordComponents")) {
                    this.logInfo("showRecordComponents: updateRecordComponent() method " +
                        "failed to return an updated component.", "recordComponents");
                }
                // stick it back into the pool for future use
                this.addToRecordComponentPool(origComponent);
            }
            this.logDebug("updated record component from pool:" + component, "recordComponents");
        }
    }
    return this._finishApplyNewRecordComponent(record, fieldName, body, rowNum, bodyCol, component);
},

_finishApplyNewRecordComponent : function (record, fieldName, body, rowNum, bodyCol, component) {
    var pool = this.recordComponentPoolingMode == "recycle",
        addNullMarker = component == null;
    if (addNullMarker) {
        component = {
            isNullMarker:true,
            _embedBody:body.getID(),
            _recordComponentBatch:this._recordComponentSequence
        }
    }
    if (fieldName == null) fieldName = this._$noFieldString;
    
    this._addRecordComponent(record, fieldName, component);
    // We're applying a "currentFieldName" / "currentRecord" flag in addition
    // to the _currentFieldName applied by the embeddedComponents code. This is
    // intentional - we use these flags in pooling mode to pick up the component that
    // matched the previous record (if possible) and previous field if
    // poolComponentsPerColumn is true. Don't want to rely on the flags that are set up and
    // potentially cleared by the standard embeddedComponent subsystem.
    if (pool && !addNullMarker) {
        component.currentFieldName = fieldName;
        component.currentRecord = record;
    }
    if (!addNullMarker) {
        // have the recordComponent removed if the field it's in is hidden
        component.removeOnHideField = true;

        //this.logWarn("created component:" + component + ", adding to:" + [rowNum,fieldName]);
        return body.addEmbeddedComponent(component, record, rowNum, bodyCol, this.getRecordComponentPosition());
    }
},

//> @method listGrid.setDontAutoDestroyComponent()
// If +link{listGrid.showRecordComponents} is true, by default any created record components
// are destroyed once they are no longer in use (for example in the ListGrid as a whole is destroyed).
// This method may be used to suppress this behavior for some component. Typical usage might call
// this method as part of +link{listGrid.createRecordComponent()} to suppress this behavior.
// @param component (Canvas) component in question.
// @param dontAutoDestroy (boolean) If true, the component will not be destroyed automatically when
//   the grid is destroyed
// @visibility external
//<

setDontAutoDestroyComponent : function (component, dontAutoDestroy) {
    component.dontAutoDestroy = dontAutoDestroy;
},

// fired when a recordComponent's cell is no longer visible. Behavior depends on 
// recordComponentPoolingMode.
_cleanUpRecordComponent : function (component, forceDestroy) {
    if (this.logIsDebugEnabled("recordComponents")) {
        this.logDebug("cleaning up recordComponent:" + component,
            "recordComponents");
    }

    var poolingMode = this.recordComponentPoolingMode;
    // If passed the forceDestroy parameter, behave in 'viewport' mode regardless of
    // the actual pooling mode - this means we'll destroy the component passed in.
    if (forceDestroy) poolingMode = "viewport";
    
    if (poolingMode == "data") {
        // No need to attempt to pool, etc here:
        // If the record is still around, placeEmbeddedComponent() will have cleared it and
        // it'll simply re-render when scrolled back into view, etc.
        //
        // If the record is no longer present in the data array, remapEmbeddedComponents()
        // will have already marked it for destruction, and updateRecordComponents() has
        // already rebuilt the array of _liveRecordComponents so we won't be hanging onto
        // a pointer to it anymore.
        //
        // However - when the listGrid as a whole gets destroyed, we should ensure the
        // recordComponent gets destroyed too.
        // Handle this by storing a separate reference to it which we can look at on
        // destroy(). Use "pointersToThis" to ensure that if the component gets destroyed separately
        // (while this component is still intact) we don't hang onto this reference.
        if (!component.destroyed && !component.destroying && !component._pendingDestroy) {
            if (this._orphanedRecordComponents == null) {
                this._orphanedRecordComponents = {};
            }
            var ID = component.ID;
            if (this._orphanedRecordComponents[ID] != component) {
                this._orphanedRecordComponents[component.ID] = component;
                // set up 'pointersToThis' 
                if (component.pointersToThis == null) component.pointersToThis = [];
                component.pointersToThis.add({ 
                    object: this._orphanedRecordComponents, property: component.ID 
                });
            }
        }

    } else {
        var ID = component.ID,
            body = isc.Canvas.getById(component._embedBody),
            record = component.embeddedRecord,
            fieldName = component._currentFieldName;
        
        // wipe the component from the components-cache on the record [IF it hasn't been updated
        // to point at a new component]
        
        
        if (this._hasRecordComponents(record)) {
            this._deleteRecordComponent(record, fieldName);
        }
        
        // this component may have already been removed from the body, for example by
        // _remapEmbeddedComponentColumns(). In this case _embedBody will have been null
        // so we can detect this by the body var being unset here.
        if (body != null) {
            body.removeEmbeddedComponent(component.embeddedRecord, component);
        }
        
        if (poolingMode == "viewport") {
            if (!component.dontAutoDestroy) component.markForDestroy();
        } else {
            
            if (component.destroying || component.destroyed || component._pendingDestroy) return;
            this.addToRecordComponentPool(component);
        }
    }
},

// Helper method - look at a record and see if we currently have a recordComponent for it.

_$noFieldString:"_noField",
_recordComponentSequence:0,

getLiveRecordComponent : function (record, fieldName, bodyID) {
    if (!record) return null;
    if (isc.isA.Number(record)) record = this.getRecord(record);
    if (!bodyID) bodyID = this.body.getID();
    var comp = this._getLiveRecordComponent(record, fieldName, bodyID);
    return comp;
},
_getLiveRecordComponent : function (record, fieldName, bodyID) {
    if (fieldName == null) fieldName = this._$noFieldString;
    
    if (!this._hasRecordComponents(record)) return null;
    var recordComponents = this._getRecordComponents(record);
    
    if (recordComponents[fieldName] == null) return null;
    
    var component = recordComponents[fieldName];
    
    if (component._embedBody != bodyID) {
        return null;
    }
    if (component.isNullMarker && component._recordComponentBatch != this._recordComponentSequence) {
        return null;
    }
    
    // We should never see this but if a component gets destroyed without first being
    // cleared out of the record._recordComponents block, wipe it out now. This will force
    // creation of a new recordComponent in calling code.
    if (component.destroyed || component.destroying || component._pendingDestroy) {
        this.logWarn("Destroyed or Destroying record component:" + component + 
            " present on record. Ignoring", "recordComponents");
        recordComponents[fieldName] = null;
        return null;
    }
    return component;
},

//> @method listGrid.invalidateRecordComponents()
// Invalidates the currently visible set of +link{listGrid.showRecordComponents,recordComponents}
// and gets fresh ones for the visible rows in the grid according to the 
// +link{listGrid.recordComponentPoolingMode}
// <P>
// See also +link{listGrid.refreshRecordComponent()} which allows you to refresh a specific
// recordComponent
//
// @visibility external
//<
invalidateRecordComponents : function () {
    
    // force destruction of the visible recordComponents - otherwise this
    // method would have no visible effect in 'data' pooling mode.
    this.dropRecordComponents(true);
    
    if (this.showRecordComponents && this.isDrawn()) {
        this.updateRecordComponents();
    }
},

dropRecordComponents : function (forceDestroy) {
    
    // up the recordComponentSequence count. This is used to identify our special null markers
    // and essentially invalidates them, meaning we'll re-run the createRecordComponent logic
    // for records with null markers we've already set at this point.
    
    this._recordComponentSequence++;
    
    var oldRecordCompArr = this._liveRecordComponents || [];
        
    delete this._liveRecordComponents;
    delete this._liveRecordComponentsObj;
    
    for (var i = 0; i < oldRecordCompArr.length; i++) {
        this._cleanUpRecordComponent(oldRecordCompArr[i], forceDestroy);
    }
},

//> @method listGrid.refreshRecordComponent()
// Discards any currently visible set of +link{listGrid.showRecordComponents,recordComponent}
// and gets a fresh one for the specified record (or cell) according to the 
// +link{listGrid.recordComponentPoolingMode}
// <P>
// See also +link{listGrid.invalidateRecordComponents()} which allows you to refresh all record
// components that are currently visible in the grid.
//
// @param rowNum (int) Row to refresh
// @param [colNum] (Integer) Column to refresh. This parameter should be passed 
//      if +link{showRecordComponentsByCell} is true.
// @visibility external
//<
refreshRecordComponent : function (rowNum, colNum) {
    if (!this.showRecordComponents || rowNum == null || this.body == null) return;
    
    if (this.showRecordComponentsByCell && colNum == null) {
        this.logWarn("refreshRecordComponent() called with no colNum. This parameter is required when " +
            "showRecordComponentsByCell is true. Taking no action.");
        return;
    }
    
    var record = this.getRecord(rowNum);
    if (record == null || Array.isLoading(record)) return;
    
    var body = this.body,
        bodyColNum = null,
        fieldName = null;
        
    if (this.showRecordComponentsByCell) {
        body = this.getFieldBody(colNum);
        bodyColNum = this.getLocalFieldNum(colNum);
        fieldName = this.getFieldName(colNum);
    }

    var prevComp = this._getLiveRecordComponent(record, fieldName, body.getID());
    if (prevComp != null && prevComp.isNullMarker) prevComp = null;

    var shouldShowComponent = this.shouldShowRecordComponent(record, colNum);

    var liveComp;
    if (prevComp != null) {
        
        if (shouldShowComponent && isc.isA.Function(this.updateRecordComponent)) {
            liveComp = this.updateRecordComponent(record, colNum, prevComp, false);
            if (liveComp != prevComp) {
                this._cleanUpRecordComponent(prevComp, (this.recordComponentPoolingMode != "recycle"));
                this.addToRecordComponentPool(prevComp);

                liveComp = this._finishApplyNewRecordComponent(record, fieldName, body, rowNum, bodyColNum, liveComp);
                if (liveComp && liveComp.isNullMarker) liveComp = null;
            }
            shouldShowComponent = false;
        } else {
            // This will destroy the component, or add to the pool.
            this._cleanUpRecordComponent(prevComp, (this.recordComponentPoolingMode != "recycle"));
        }
    }

    if (shouldShowComponent) {
        liveComp = this._applyNewRecordComponent(record, fieldName, body, rowNum, bodyColNum);
        if (liveComp && liveComp.isNullMarker) liveComp = null;
    }

    // Update this._liveRecordComponents and this._liveRecordComponentsObj
    var index = this._liveRecordComponents.length;
    if (prevComp) {
        var ID = prevComp.getID();
        this._liveRecordComponentsObj[prevComp.getID()] = null;
        // Null the slot in the live recordComponents array and reuse it if possible for
        // efficiency.
        if (liveComp != null) {
            index = this._liveRecordComponents.indexOf(prevComp);
            this._liveRecordComponents[index] = null;
        } else {
            this._liveRecordComponents.remove(prevComp);
        }
    }
        
    if (liveComp != null) {
        var ID = liveComp.getID();
        this._liveRecordComponentsObj[ID] = true;
        this._liveRecordComponents[index] = liveComp;   
    }
},

getRecordComponentPosition : function () {
    if (this.recordComponentPosition != null) return this.recordComponentPosition;
    return (this.showRecordComponentsByCell ? "within" : "expand");
},

getRecordComponentPool : function () {
    if (!this._recordComponentPool) this._recordComponentPool = [];
    return this._recordComponentPool;
},

// we want to indicate whether the record changed. Handle this by returning a 2 element array
// - the component and a boolean.
getFromRecordComponentPool : function (record, fieldName) {
    var components = this.getRecordComponentPool(),
        subList = [],
        component;
    
    if (!components || components.length == 0) return null;
    
    if (this.showRecordComponentsByCell && this.poolComponentsPerColumn == true) {
        subList = components.findAll("currentFieldName", fieldName);
    } else {
        subList = components;
    }
    if (!subList || subList.length == 0) return null;
    
    for (var i = 0; i < subList.length; i++) {
        component = subList[i];
        var prevRecord = component.currentRecord;
        if (this.comparePrimaryKeys(prevRecord, record)) {
            components.remove(component);
            return [component,true];
        }
    }
    // we didn't find a component that previously sat in this record, just return an arbitrary
    // one - the last one on the list.
    components.length -= 1;
    return [component,false];
},

addToRecordComponentPool : function (component) {
    var components = this.getRecordComponentPool();

    components.add(component);
},

// Should we show a recordComponent for this record/col?
// checks for various records we want to skip, like the separator rows, and
// fires the public 'showRecordComponent()' method to allow custom suppression of RCs for
// certain rows or cells.
shouldShowRecordComponent : function (record, colNum) {
    
    if (record == null || record._isGroup || record[this.isSeparatorProperty]
        || Array.isLoading(record))
    {
        return false
    }
    return this.showRecordComponent(record,colNum);
},

// Override point documented in registerStringMethods().

showRecordComponent : function () {
    return true;
},

// notification from each body when getInnerHTML is called.
bodyDrawing : function (body) {

    if (isc._traceMarkers) arguments.__this = this;

    // don't fetch valueMap in response to draw on both the frozen and liquid body - just
    // on the normal (liquid) one
    if (body != this.body) return;

    // when the primary body draws, kick of a queue of requests for the visible data and any
    // valueMaps we'll need for fields with optionDataSource set
    var startedQueue;
    if (this._fetchValueMap) {
        // fetch valueMaps for fields where optionDataSource is set
        startedQueue = !isc.RPCManager.startQueue();
        this._fetchValueMapData();
    }
    
    this.requestVisibleRows();
    
    if (startedQueue) isc.RPCManager.sendQueue();
    this._fetchValueMap = null;
},


//> @attr listGrid.recordComponentHeight (Integer : null : IRWA)
// If +link{listGrid.showRecordComponents} is true, this attribute may be used to
// specify a standard height for record components.
// If specified every row in the grid will be sized tall enough to accommodate a recordComponent
// of this size.
// <P>
// Note that if this property is unset, the grid will not be able to know row heights in
// advance, and +link{listGridField.frozen,freezing of columns} is not currently supported in
// this case.
// @visibility external
//<


//> @method listGrid.setRecordComponentHeight()
// Setter for the +link{listGrid.recordComponentHeight}
// @param height (integer) recordComponent height
// @visibility external
//<
setRecordComponentHeight : function (height) {
    this.recordComponentHeight = height;
    if (this.isDrawn()) this.markForRedraw();
},

// Override 'getAvgRowHeight()' - if recordComponentHeight is specified and we're showing
// recordComponents, make use of it.
getAvgRowHeight : function (body) {
    
    if (this.showRecordComponents && this.recordComponentHeight != null) {
        return this.getRecordComponentRowHeight();
    }
    // standard behavior
    return body.fixedRowHeights ? body.cellHeight : Math.max(body.cellHeight,body.avgRowHeight);
    
},

_$expand:"expand",
getRecordComponentRowHeight : function () {
    if (this.recordComponentHeight == null) return null;
    var pos = this.getRecordComponentPosition();
    if (pos == this._$expand) return this.cellHeight + this.recordComponentHeight;
    else return Math.max(this.recordComponentHeight, this.cellHeight);
},

// ListGridField.optionDataSource handling
// ---------------------------------------------------------------------------------------

// This logic handles fields where an optionDataSource acts as a server-side valueMap
//
// For fields with a specified optionDataSource separate from this grid's dataSource, with
// displayField and valueField specified, we need to perform a fetch against the server to
// get a map from data values to display values

// _setOptionDataSources() (called from setFields())
// iterate through the list of fields and take note of optionDataSource property to look up the
// display map later when the LG is drawn or redrawn (see bodyDrawing)
_setOptionDataSources : function () {

    
    // _fetchValueMap - one time flag to kick off a fetch in bodyDrawing
    this._fetchValueMap = null;
    
    var oldODSs = this._optionDataSources;
    this._optionDataSources = [];
    var gridDS = this.getDataSource();
    
    for (var i = 0; i < this.completeFields.length; i++) {
        var field = this.completeFields[i];
        
        if (field == null) {
            this.logWarn("Fields array contains an empty entry");
            continue;
        }
        
        var displayField = field.displayField || field[this.fieldIdProperty],
            valueField = field.valueField || field[this.fieldIdProperty];
            
        // autoFetchDisplayMap can be set at the field or LG level
        if (field.optionDataSource == null ||
            // If the display field matches the valueField don't bother to do a fetch since
            // we don't need to map the static raw cell value to a display value.
            // If the field is editable this means that we'll rely on the editor to perform a fetch
            // based on the optionDataSource rather than being passed an explicit valueMap.
            field.displayField == null || (field.displayField == field.valueField) ||
            (field.valueField == null && (field.displayField == field[this.fieldIdProperty])) ||
            (field.autoFetchDisplayMap == false) ||
            (this.autoFetchDisplayMap == false && field.autoFetchDisplayMap == null))
        {
            continue;
        }
        
        var optionDS = isc.DS.get(field.optionDataSource);
        if (optionDS == null) {
            this.logWarn(field.optionDataSource + " dataSource not found, check value of " +
                         "optionDataSource property on the " + field[this.fieldIdProperty] +
                         " field");
            continue;
        
        
        }
        
        var optionDSID = optionDS.ID,
            fieldName = field[this.fieldIdProperty],
            optionCriteria = field.optionCriteria;
            
        // have we already set up an optionDataSource config object from our previous set of fields?
        // which matches this ODS / criteria?

        // If so re-use it - we may not even need to re-fetch data!
        var addedToConfig = false;
        // Never try to fold fields with explicitly specified optionFetchContext into 
        // a single responses
        if (oldODSs && !field.optionFilterContext) {
            for (var ii = 0; ii < oldODSs.length; ii++) {
                var ODSConfig = oldODSs[ii];
                if (ODSConfig == null) continue;
                if (this._fieldMatchesODSConfig(field, ODSConfig)) {
                    ODSConfig._fields = [field];
                    this._optionDataSources.add(ODSConfig);
                    oldODSs[ii] = null;
                    addedToConfig = true;
                    /*
                    this.logWarn("setOptionDataSources() field:"+ field.name + 
                        " bound to dataSource ID:"+ optionDSID + 
                        ", [crit:"+ this.echo(optionCriteria) +
                        "] matched existing optionDataSource config block from previous fields " +
                        "so added field to config.");
                    */
                    break;
                } 
            }
        }
        
        if (!addedToConfig && !field.optionFilterContext) {
            for (var ii = 0; ii < this._optionDataSources.length; ii++) {
                var ODSConfig = this._optionDataSources[ii];
                if (this._fieldMatchesODSConfig(field, ODSConfig)) {
                    ODSConfig._fields.add(field);
                    addedToConfig = true;
                    /*
                    this.logWarn("setOptionDataSources() field:"+ field.name + 
                        " bound to dataSource ID:"+ optionDSID + 
                        ", [crit:"+ this.echo(optionCriteria) +
                        "] matched existing optionDataSource config block so added to field.");
                    */
                    break;
                }
            }
        }
        
        if (!addedToConfig) {
            this._optionDataSources.add({
                _dsID:optionDSID,
                _fields:[field],
                _textMatchStyle:field.optionTextMatchStyle,
                _criteria:optionCriteria,
                _optionContext:field.optionFilterContext,
                _optionOperationId:field.optionOperationId
            });
            
            // -- is this the only case where we need to fetch?
            //    Assumption is that in other cases we either already kicked off a fetch
            //    or picked up the field from our 'old ds's', and so already have a resultSet
            
            // keep going for a bit...
            
            
            // in this case we need to kick off a fetch for this dataSource
            // Set the flag to handle this when the body gets redrawn to reflect the new set of 
            // fields
            /*
            this.logWarn("setOptionDataSources() field:"+ field.name + 
                " bound to dataSource ID:"+ optionDSID + 
                ", [crit:"+ this.echo(optionCriteria) +
                "] doesn't match fetch conditions for any other field with an option dataSource, " +
                "so performing a new fetch for it.");
            */
            this._fetchValueMap = true;
        }
    }
    
    // update the valueMaps for new fields bound to optionDataSources for which we already have
    // a resultSet
    for (var i = 0; i < this._optionDataSources.length; i++) {
        if (this._optionDataSources[i]._data != null) {
            this._updateValueMapFromODS(this._optionDataSources[i]);
        }
    }
    
    // If we had any previous optionDataSource config type options which are no longer required,
    // clean them up now by calling 'detroy()' on the resultSet (data object) and letting everything
    // go out of scope at the end of the method.
    if (oldODSs != null) {
        for (var i = 0; i < oldODSs.length; i++) {
            if (oldODSs[i] && oldODSs[i]._data) oldODSs[i]._data.destroy();
        }
    }
},

// little helper method - does a field with an optionDataSource specified match an existing
// optionDataSource config object.

_fieldMatchesODSConfig : function (field, config) {
        
    return (
        // same dataSource
        config._dsID == field.optionDataSource &&
        // optionTextMatchStyle should match
        
        (field.optionTextMatchStyle == config._textMatchStyle) &&
        // if optionOperationId is specified on a field it must also match
        (field.optionOperationId == config._optionOperationId) && 
        // criteria unset or match
        (isc.DataSource.getDataSource(config._dsID).compareCriteria(
            field.optionCriteria || {}, config._criteria || {}) == 0)
    );
    
},

// Given a resultSet bound to a dataSource, update the valueMap for any fields bound to this
// optionDataSource.
_updateValueMapFromODS : function (ODSConfig) {
    var odsID = ODSConfig._dsID,
        fields = ODSConfig._fields;
    
    if (fields == null || fields.length == 0) {
        this.logWarn("_updateValueMapFromODS fired for dataSource:"+ odsID + 
                    " which no longer applies to any fields in this ListGrid");
        return;
    }

    var resultSet = ODSConfig._data;
 
    // avoid trying to get a valueMap if we are called when the cache has just been invalidated
    // or before the ResultSet has been created
    if (resultSet == null || 
        (isc.isA.ResultSet(resultSet) && !resultSet.lengthIsKnown())) return;   

    var groupBy = this.getGroupByFields() || [],
        fieldNames = fields.getProperty(this.fieldIdProperty),
        needsRegroup = false
    ;

    if (!isc.isAn.Array(groupBy)) groupBy = [groupBy];

    for (var i = 0; i < fields.length; i++) {
        var field = fields[i],
            valueField = (field.valueField || field[this.fieldIdProperty]),
            map = resultSet.getValueMap(valueField, field.displayField)
        ;

        // If we haven't yet integrated the field into our fields array, store the
        // valueMap on it directly rather than calling this.setValueMap() -- this way it'll get
        // picked up when setFields() completes
        if (this.getField(field[this.fieldIdProperty]) == null) {
            field.valueMap = map;
        } else {
            this.setValueMap(field[this.fieldIdProperty], map);
        }
        
        if (groupBy.contains(field.valueField) || groupBy.contains(field.displayField)) {
            // if we've just loaded the ODS for one of the groupFields, we need to regroup to
            // re-evaluate the groupTitles
            this._needsRegroup = true;
        }

    }

    if (this._needsRegroup) {
        // we need a regroup - check to see if all ODSs have been loaded and only regroup if
        // they have
        var allConfigLoaded = true;
        for (var i = 0; i < this._optionDataSources.length; i++) {
            var ODSConfig = this._optionDataSources[i];
            if (!ODSConfig._data) {
                allConfigLoaded = false;
                break;
            }
        }

        if (allConfigLoaded) {
            this.regroup(true);
            delete this._needsRegroup;
        }
    }
},


// _fetchValueMapData()
// for fields with an optionDataSource, kick off a fetch to get a valueMap for the field
_fetchValueMapData : function () {
    
    var shouldSendQueue;
    
    // For each field with an optionDataSource, kick off a fetch if we have no data, otherwise
    // update the valueMap with the existing data
    for (var i = 0; i < this._optionDataSources.length; i++) {
        var ODSConfig = this._optionDataSources[i];
        if (ODSConfig._data == null) {
            if (shouldSendQueue == null) {
                shouldSendQueue = !isc.RPCManager.startQueue();
            }

            var optionDSID = ODSConfig._dsID;
            
            // add component context to request properties for rpc history tree in dev console
            // as a comma delimited list of fields bound to the ODS in question
            var compContext,
                optFields = ODSConfig._fields;
            if (optFields && optFields.length > 0) {
                compContext = "ODS:";
                for (var ii = 0; ii < optFields.length; ii++) {
                    compContext += optFields[ii].name;
                    if (ii < optFields.length - 1) compContext += ",";
                }
            }
            
            var context = ODSConfig._optionContext || {};
            isc.addProperties(
                context,
                {showPrompt:false, clientContext:{ODSConfig:ODSConfig},
                 componentId:this.getID(),
                 componentContext: compContext,
                 textMatchStyle:ODSConfig._textMatchStyle
                }
            );
            if (ODSConfig._optionOperationId != null) {
                context.operationId = ODSConfig._optionOperationId;
            }
            isc.DataSource.getDataSource(optionDSID).fetchData(
                ODSConfig._criteria, 
                {target:this, methodName:"_fetchValueMapCallback"},
                context
            );
        }
    }

    if (shouldSendQueue) isc.RPCManager.sendQueue();
},


// _fetchValueMapCallback()
// Callback from request to get all data from some field's optionDS.
// Creates a valueMap so we show the displayField value in the field
_fetchValueMapCallback : function (dsResponse, data, dsRequest) {

    var optionDataSourceConfig = dsRequest.clientContext.ODSConfig;
    // if the fields were changed before the fetch returned
    // OR we kicked off 2 fetches against the same dataSource and have already 
    // set up our resultSet data, bail
    if (!optionDataSourceConfig || (optionDataSourceConfig._data != null) ||
        !this._optionDataSources || !this._optionDataSources.contains(optionDataSourceConfig)) 
    {        
        return;
    }
    
    optionDataSourceConfig._data = isc.ResultSet.create({
        dataSource:optionDataSourceConfig._dsID,
        ODSConfig:optionDataSourceConfig,
        targetGrid:this,
        dataChanged:"this.targetGrid._updateValueMapFromODS(this.ODSConfig)",
        initialData:data
    })
    
    // Force an update now
    this._updateValueMapFromODS(optionDataSourceConfig);

    return true;
},


// Helper to destroy() the resultSet(s) we create to handle valueMaps from the server
_dropODSData : function () {
    if (this._optionDataSources == null) return;
    for (var i = 0; i < this._optionDataSources.length; i++) {
        var data = this._optionDataSources[i]._data;
        if (data) data.destroy();
    }
    delete this._optionDataSources;
},

requestVisibleRows : function () {
	// ask for all the rows we're about to render.  This enables better predictive fetching
    // relative to asking for data one row at a time while we render.
	
    if (isc.ResultSet && isc.isA.ResultSet(this.data)) {

        if (this.body == null || this.body._reused) {
            return this.data.getRange(0, this.dataPageSize);
        }
        // if the data set is empty no need to fetch
        if (this.data.lengthIsKnown() && this.data.getLength() == 0) return;

        
        if (!this.data.lengthIsKnown()) {
            this.body.showAllRows = false;
        } else {
            // NOTE: this check is necessary because the body itself forces showAllRows to true
            // on init if overflow:visible.  It would probably be more robust to pass a
            // one-time flag to getTableHTML() so that we don't clobber other changes to
            // showAllRows
            this.body.showAllRows = (this.body.overflow == isc.Canvas.VISIBLE ? 
                                     true : this.showAllRecords);
        }

        var drawRect = this.body.getDrawArea();
        
        if (this._scrollCell && isc.isAn.Array(this._scrollCell)) {
            // if scrolling was applied before draw(), move the drawRect to the requested row
            var diff = drawRect[1]-drawRect[0];
            drawRect[0] = this._scrollCell[0];
            drawRect[1] = drawRect[0]+diff;
        }

        // force all rows to be grabbed if we're grouping. (We'll need them anyway.)
        if (this.isGrouped) {
            return this.data.getRange(0, this.groupByMaxRecords+1);
        } else {            
            // getRange() is non-inclusive at the end, but getDrawArea() is inclusive
            // at the end so we need to increment drawRect[1] by 1.
            return this.data.getRange(drawRect[0], drawRect[1]+1);
        }
    }
    return null;
},

// Printing
// --------------------------------------------------------------------------------------------
                  
//>	@attr listGrid.printAutoFit (Boolean : true : IRW)
// Whether cell contents should wrap during printing.  Equivalent to +link{autoFit}, but
// specific to printed output.
// @group printing
// @visibility external
//<
printAutoFit:true,

//>	@attr listGrid.printWrapCells (Boolean : true : IRW)
// Whether cell contents should wrap during printing.  Equivalent to +link{wrapCells}, but
// specific to printed output.
// @group printing
// @visibility external
//<
printWrapCells:true,

//>	@attr listGrid.printHeaderStyle (CSSStyleName : "printHeader" : IRW)
// Style for header cells in printed output.  Defaults to +link{headerBaseStyle} if null.
// @group printing
// @visibility external
//<
printHeaderStyle:"printHeader",

//>	@attr listGrid.printBaseStyle (CSSStyleName : null : IRW)
// Style for non-header cells in printed output.  Defaults to +link{baseStyle} if null.
// @group printing
// @visibility external
//<
getPrintHeaders : function (startCol, endCol) {
    
    var defaultAlign = (this.isRTL() ? isc.Canvas.LEFT : isc.Canvas.RIGHT),
        printHeaderStyle = this.printHeaderStyle || this.headerBaseStyle,
        HTML;

    
    // We support arbitrarily nested, asymmetrical header-spans - these require
    // some slightly tricky logic so use a conditional to avoid this if not required.
    if (this.headerSpans) {

        // Step 1: We'll build an array of "logical columns" in this format:
        // [field1], [innerHeader1], [topHeader]
        // [field2], [innerHeader2], [topHeader]
        // [field3], [topHeader2]
        // Each array contains an entry for each row we'll write out (each header
        // span the field is embedded in, plus the field).
        // Note that the top row of HTML will be the last entry in each sub-array and
        // the bottom row will be the first entry (the field should appear below 
        // all its headers).
        // Also note we have repeats in here - we'll handle this by applying colSpans
        // to the generated HTML - and that the column arrays will be different lengths
        // due to different depth of nesting of header spans - we'll handle this by 
        // applying rowSpans.
        var logicalColumns = [],
            numRows = 1;
        
        for (var i = startCol; i < endCol; i++) {
            var field = this.getField(i);
            logicalColumns[i] = [field];
            
            var span = this.spanMap[field.name];
            
            // build a logical column from the fieldName up to the top span
            // (Note that we will have the same span in multiple cols, which is ok)
            while (span != null) {
                logicalColumns[i].add(span);
                span = span.parentSpan;
            }
            // Remember how deep the deepest nested column is - this is required to
            // allow us to apply numRows.
            numRows = Math.max(logicalColumns[i].length, numRows);
        }
        
        
        // Step 2: Iterate through the column arrays starting at the last entry
        // (outermost header)
        HTML = [];

        for (var i = numRows-1; i >= 0; i--) {
            HTML[HTML.length] = "<TR>";
            
            var lastEntry = null,
                colSpanSlot = null;
            for (var ii = startCol; ii < endCol; ii++) {
                var rowSpan = 1, colSpan = 1;
                // When we reach the first entry in the array we'll be looking at a field
                var isField = (i == 0);

                var entry = logicalColumns[ii][i];
                
                
                // If we hit a spanned marker we've already written out a row and/or 
                // colSpanning cell that covers this entry - short-circuit the logic
                // below in this case.
                if (entry == "spanned") {
                    continue;
                }
                // our logical column arrays will be different lengths as
                // we can have unequal depth header spans (including potentially 
                // just fields).
                // To handle this we'll have to apply rowSpans.
                // Sub-headers can have unequal depths - for example a header
                // span could contain a child span with fields and another child span
                // with its own child-span before their are fields.
                // This means we cant "take up all the slack" on the outer headers, we
                // have to apply rowSpans at potentially several levels.
                // Strategy:
                // - if this is a null entry there are fewer rows in this column than
                //   the total, so we need a rowSpan > 1 somewhere.
                //   Reach up the column until we find an entry - this will be the
                //   header we'll write out, and remember how deep we had to go 
                //   (that will be the rowSpan)
                // - The header may span multiple columns - repeat this process for
                //   each subsequent column until we find a header span that differs
                //   from the one we were looking at.
                // At this point we'll know how many cols the header spans, and how deep
                // we had to go to find it in each col. 
                // ColSpan will be the number of cols spanned.
                // RowSpan will be the minimum depth we had to go.
                // Now modify the logical column arrays to ensure we don't write out 
                // duplicate cells in the generated HTML:
                // - replace every cell we span with a "spanned" marker so we skip it
                // - if the entry we are using as the source for the header doesn't
                //   fall in this range, clear its entry from the logical column array.
                //   Then on the iteration for the row where we would have it it we'll
                //   hit another null entry, and that will cause us to look at the 
                //   next entry down instead and render it out with appropriate rowSpan
                var minDepth,
                    spanningColNum = ii,
                    spannedColOffsets = [];
                    
                // set colSpan to zero. We'll increment in the loop below
                colSpan = 0;
                
                while (spanningColNum < endCol) {
                    var entryToTest = null,
                        foundMismatch = false;
                    for (var offset = 0; (i-offset) >= 0; offset++) {
                        entryToTest = logicalColumns[spanningColNum][i-offset];
                        

                        if (entryToTest != null) {
                            // If we originally hit a null entry, pick up the first
                            // non null entry so we have something to actually write out.
                            if (entry == null) {
                                entry = entryToTest;
                                minDepth = offset;
                                if (i-offset == 0) {
                                    isField = true;
                                }
                            }
                            if (entry == entryToTest) {
                                spannedColOffsets[colSpan] = offset;
                                minDepth = Math.min(offset, minDepth);
                            } else {
                                foundMismatch = true;
                            }
                            break;                                
                        } 
                    }
                    if (foundMismatch) {
                        break;
                    }
                    spanningColNum ++;
                    
                    colSpan++;
                }
                
                // set rowSpan for the cell based on how deep we had to
                // go to find a real entry (shift from zero to 1-based)
                if (minDepth != null) {
                    rowSpan = minDepth+1;
                }
                
                
                   
                // For each column this entry spans, add markers indicating that
                // we're handling this via TD with rowSpan and colSpan set (and
                // clear out duplicate entries).
                for (var spannedCols = 0; spannedCols < spannedColOffsets.length; 
                    spannedCols++) 
                {
                
                    var logicalColArray = logicalColumns[spannedCols + ii],
                        offset = spannedColOffsets[spannedCols];
                        
                    for (var spannedRows = 0; spannedRows <= offset; spannedRows++) {
                        
                        if (spannedCols == 0 && spannedRows == 0) {
                            logicalColArray[i-spannedRows] = entry;
                        } else if (spannedRows <= minDepth) {
                            logicalColArray[i - spannedRows] = "spanned";
                        } else {
                            logicalColArray[i - spannedRows] = null;
                        }
                    }
                }
                
                
                
                // We don't expect to ever end up with a null entry - not sure
                // how this could happen but log a warning
                if (entry == null) {
                    this.logWarn("Error in getPrintHeaders() - unable to generate " +
                        "print header HTML from this component's specified headerSpans");
                }
            
                var align = "center",
                    cellValue;
                
                if (isField) {
                    align = entry.align || defaultAlign;
                    cellValue = this.getHeaderButtonTitle(entry.masterIndex);
                } else {
                    cellValue = entry.title;
                }

                var cellStart = HTML.length;
                
                HTML[HTML.length] = "<TD class='";
                HTML[HTML.length] = printHeaderStyle;
                HTML[HTML.length] = "' align='";
                HTML[HTML.length] = align;
                HTML[HTML.length] = "' rowSpan='";
                HTML[HTML.length] = rowSpan;
                HTML[HTML.length] = "' colSpan='";
                HTML[HTML.length] = colSpan;
                HTML[HTML.length] = "'>";
//                    HTML[HTML.length] = "' style='border:1px solid black'>";
                HTML[HTML.length] = cellValue;
                HTML[HTML.length] = "</TD>";
                
                
                
            }
            HTML[HTML.length] = "</TR>"
        }
//         this.logWarn("\n\nGenerated print header HTML (including spans):" + HTML.join(""));
        
    } else {
    
        var HTML = ["<TR>"];
            
        var cellStartHTML = ["<TD CLASS=", printHeaderStyle,
                             " ALIGN="].join("");
                             
        // Just iterate through the fields once, then assemble the HTML and return it.
        for (var colNum = startCol; colNum < endCol; colNum++) {
            var field = this.body.fields[colNum];
            var align = field.align || defaultAlign;
            HTML.addList([cellStartHTML, align, ">",
                                this.getHeaderButtonTitle(field.masterIndex), "</TD>"]);
        }
        
        // Output the standard header row
        HTML[HTML.length] = "</TR>";
    }
    return HTML.join(isc.emptyString);
},

// Write out arbitrary "footers" into the print HTML table
// This is used to write out the "summaryRow" rows if necessary

getPrintFooters : function (startCol, endCol) {
    if (!(this.summaryRow) || !(this.summaryRow.body) || !(this.showGridSummary)) {
        // return emty string if no data in the grid
        return "";
    }

    var chunkContext = {
        startRow:0,
        endRow:this.summaryRow.getTotalRows(),
        maxRows:this.summaryRow.printMaxRows,
        printProps:this.printProperties || {},
        html:[]
    }
    
    
    var printWidths = isc.Canvas.applyStretchResizePolicy(this.fields.getProperty("width"), 
                                                          chunkContext.printProps.width || isc.Page.getWidth());
    chunkContext.printWidths = printWidths;
    
    this.summaryRow.body.printChunkOnly = true;
    
    var HTML = this.summaryRow.body.getPrintHTMLChunk(chunkContext, true);
    delete this.summaryRow.body.printChunkOnly;
    // this.logWarn("footer HTML:" + HTML);
    return HTML; 
},

getPrintHTML : function (printProperties, callback) {

    

    var body = this.body;
    // we may have getPrintHTML called while we're undrawn - if so, we'll need to set up our 
    // children here
    if (body == null) {
        this.createChildren();
        body = this.body;
    }
    
    // if the body is pending a redraw, force it now
    if (this.isDirty() || body.isDirty()) {
        this.redraw("updating HTML for printing");
    }
    
    var printProps = isc.addProperties({}, printProperties, this.printProperties);

    // with a partial cache, print only the contiguous cache around the currently visible rows
    
    var startRow, endRow;
    if (isc.isA.ResultSet(this.data) && !this.data.allMatchingRowsCached()) {            
        var visRows = this.body.getVisibleRows(),
            firstVisibleRow = visRows ? visRows[0] : null,
            cachedRange = this.data.getCachedRange(firstVisibleRow);            
        if (cachedRange != null) {
            startRow = cachedRange[0];
            endRow = cachedRange[1];
        }        
    }
    
    // Printing of the body can "go asynchronous" in 2 ways
    // If we have embedded components, retriving their print HTML may be asynchronous
    // If we're attempting to print more than <this.printMaxRows> rows at a time we force
    // the HTML creation to be asynchronous to avoid any script-running-slowly notifications.
    
    
    
    var printWidths = isc.Canvas.applyStretchResizePolicy(this.fields.getProperty("width"), 
                                                printProps.width || isc.Page.getWidth());
    
    // getTablePrintHTML() - implemented at the GridBody level.
    // If it goes asynchronous it'll fire the callback and return null - otherwise it'll
    // return print HTML
    return body.getTablePrintHTML({startRow:startRow, endRow:endRow, callback:callback,
                                     printWidths:printWidths, printProps:printProps});

},

// This is run before getting the body tableHTML for printing
// If printing HTML in chunks it'll be run repeatedly for each chunk!
_prepareForPrinting : function (printWidths, printProperties) {
    this.isPrinting = this.body.isPrinting = true;
    this.currentPrintProperties = printProperties;
    
    var body = this.body,
        // don't print editors?
        oldEditorShowing = this._editorShowing;
    
    // properties to store off and restore
    var origProps = isc.getProperties(body, ["autoFit", "wrapCells", "showAllRows",
                                "showAllColumns", "fixedRowHeights", "_fieldWidths", "fields"
                                
                                
                    ]);                                             

    body.showAllRows = true;
    body.showAllColumns = true;
        
    this._editorShowing = false; // never show editors
    body.autoFit = this.printAutoFit;
    body.wrapCells = this.printWrapCells;
    body.fixedRowHeights = !this.printWrapCells;

    var fields = this.fields.duplicate();
    // suppress rendering out the shouldPrint fields
    fields.removeAll(fields.findAll("shouldPrint", false));
    body.fields = fields;
    
    // mark ourselves as unfrozen so we avoid logic to (for example) offset body col-num
    // with frozen col num)
    var oldFrozenFields = this.frozenFields;
    delete this.frozenFields;
    
    body._fieldWidths = printWidths;
    return {oldEditorShowing:oldEditorShowing, oldFrozenFields:oldFrozenFields,
            origProps:origProps};                                                

},

_donePrinting : function (context) {    
    var body = this.body,
        origProps = context.origProps,
        oldEditorShowing = context.oldEditorShowing;
        
    isc.addProperties(body, origProps);
    if (origProps.showAllRows == null) body.showAllRows = null;
    this._editorShowing = oldEditorShowing;
    this.frozenFields = context.oldFrozenFields;
    
    delete this.currentPrintProperties;
    
    this.isPrinting = this.body.isPrinting = false;
    
    this.body.markForRedraw("finished printing");
},

//> @attr ListGrid.printMaxRows (int : 100 : IRWA)
// Advanced property - when generating printHTML for a large ListGrid, rows are printed in
// batches in order to avoid triggering a native "script is running slowly" browser dialog.
// <P>
// For grids with exceptional numbers of columns or complex formatting logic, this number
// might need to be adjusted downward.
//
// @group printing
// @visibility external
//<
// Note that this means getPrintHTML() is frequently asynchronous for ListGrids
printMaxRows:100,


// Event Handling
// --------------------------------------------------------------------------------------------

//>	@method	listGrid.rowClick()	(A)
//
// Event handler for when rows in the body are clicked upon. The default implementation handles
// firing +link{ListGrid.startEditing()} if appropriate, and fires
// +link{ListGridField.recordClick()} and/or +link{ListGrid.recordClick()} if set. Developers
// should typically implement recordClick rather than overriding this method.
// <P>
// Note that this method fires in addition to any specified +link{listGrid.cellClick}
// handler (even if that method cancels the event as a whole by returning <code>false</code>).
//
//      @param  record      (ListGridRecord)    record object returned from getCellRecord()
//		@param	recordNum   (int)	index of the row where the click occurred
//		@param	fieldNum	(int)	index of the col where the click occurred
//      @param  [keyboardGenerated]   (boolean) indicates whether this was a synthesized record
//                                              click in response to a keyboard event
//		@group	events
//      @see    recordClick()
//
//      @group  events
//		@return	(Boolean)	
//      @visibility external
//<
rowClick : function (record, recordNum, fieldNum, keyboardGenerated) {

	// record the last record clicked (used for keyboard navigation)
    this._lastRecordClicked = recordNum;
 
	var record = this.getCellRecord(recordNum, fieldNum),
		field = this.fields[fieldNum];

    // don't fire recordClick on loading rows
    if (Array.isLoading(record)) return;

	// if the field has a 'recordClick' method, call that
	var value = this.getCellValue(record, recordNum, fieldNum);
	var rawValue = this.getRawCellValue(record, recordNum, fieldNum);

	// if the record is a group header, expand/collapse the group
	if (record != null && record._isGroup) {
	    if (this.canCollapseGroup == false) return;
	    // if the user navigates through the grid using arrow up / down keys, don't toggle
	    // the group when the user hits the group header (Except on explicit space keypress which is
	    // more of a deliberate 'pseudo-click' type interaction).
	    
	    if (keyboardGenerated) {
	        var key = isc.EH.getKey();
	        if (key != "Space" && key != "Enter") return;
	    }

        
        var mythis = this, myrecord=record;
        if (this.getEditRow() != null) {
            if (this.autoSaveEdits) {
                this.saveAllEdits(null, function () {
                    mythis.toggleFolder(myrecord);
                });
                // we'll toggle folder in the callback so no need to carry on here.
                return;
            } else {
                this.storeUpdatedEditorValue();
            }
        }
        this.toggleFolder(record);
		return;
	}

	if (field.recordClick && !(keyboardGenerated && field.keyboardFiresRecordClick == false)) {
		// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value,rawValue"
		// Convert a string callback to a function
		isc.Func.replaceWithMethod(field, "recordClick", 
                                         "viewer,record,recordNum,field,fieldNum,value,rawValue");
        var returnVal = field.recordClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return false;
	}

	// Note - there is also a (legacy) recordClick handler on the GridRenderer class, with a
	// different signature.
	// 'recordClick()' is not one of the GridAPIs - so won't be fired automatically from the
	// GridRenderer click handling code.  If the method was defined on both the ListGrid and
	// the Body, both methods would be fired.
	// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value,rawValue"
	if (this.recordClick) {
        var returnVal = this.recordClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return false;
    }
    
	// if the cell is editable on click, edit it now
	// We also start editing if editOnFocus is true.
    var editOnClick = this.isEditable() && (this.editEvent == isc.EH.CLICK || this.editOnFocus);

    // one-click toggling of boolean/valueMapped fields.
    // Note: also allows entering of editing if editEvent is click.
    if (field.canToggle && this.canEditCell(recordNum, fieldNum) && 
        this.shouldToggle(field, keyboardGenerated)) 
    {
        var valueMap = this.getEditorValueMap(field, this.getEditedRecord(recordNum,fieldNum));
        // autocreate a valueMap for boolean
        if (valueMap == null && isc.SimpleType.getBaseType(field.type) == this._$boolean) {
            valueMap = [true,false];
        }
        if (valueMap != null) {
            if (!isc.isAn.Array(valueMap)) valueMap = isc.getKeys(valueMap);
            if (valueMap.length > 1) {

                var fieldName = this.getFieldName(fieldNum),
                    editValue = this.getEditedCell(recordNum, fieldNum),
                    index = valueMap.indexOf(editValue);
                index += 1;
                if (index >= valueMap.length) index = 0;
                var oldValue = editValue;
                
                editValue = valueMap[index];
                
                var cancelChange;
                
                if (field.change != null) {
                    this.logInfo("canToggle firing specified field.change() event directly", "gridEdit");
                    cancelChange = this.fireCallback(
                            field.change, 
                            "form,item,value,oldValue",
                            [null,null,editValue,oldValue]
                    ) == false;
                }
                
                if (!cancelChange) {
    
                    // autoSaveEdits: Usually if canToggle is true we instantly commit
                    // however if we're also jumping into edit mode it makes more sense to
                    // just toggle the edit value and save when the user dismisses the editor
                    // as usual.
                    if (!editOnClick && this.autoSaveEdits) {
                        this.setEditValue(recordNum, fieldNum, editValue, true, false);
                            
                    } else {
                        this.setEditValue(recordNum, fieldNum, editValue);
                    }
                    if (field.changed != null) {
                        this.logInfo("canToggle firing specified field.changed() event directly", "gridEdit");
                        this.fireCallback(
                            field.changed,
                            "form,item,value",
                            [null,null,editValue]
                        );
                    }
                    
                    if (this.autoSaveEdits) this.saveEdits(null, null, recordNum, fieldNum);
                }
            }
        }
    }

	if (editOnClick) {
        if (this.handleEditCellEvent(recordNum, fieldNum, isc.ListGrid.CLICK) == true) {
            return true;    
        }
    	// If this was a keyboard event, and the keyboard click field is not editable, iterate 
    	// through the other fields, and edit the first editable one we find
        if (keyboardGenerated) {
            for (var i = 0; i< this.fields.length; i++) {
                if (i == fieldNum) continue;
                if (this.handleEditCellEvent(recordNum, i, isc.ListGrid.CLICK) == true) {
                    return true;
                }
            }
        }
    } 
},

// By default we only toggle if the event occurred over a field's value-icon
// Exception: If _formatBooleanFieldAsImage returns false we didn't write out an icon for 
// the cell
shouldToggle : function (field, keyboardGenerated) {
    if (keyboardGenerated) {
        var key = isc.EH.getKey();
	    if (key == this._$Space || key == this._$Enter) return true;
    }
    // Note: no need to check 'canToggle' - this method is only called for fields where
    // canToggle is true.
    if (!this._formatBooleanFieldAsImages(field)) return true;
    var part = this.getEventPart();
    return (part && part.part == "valueicon");
},

//>	@method	listGrid.rowDoubleClick()	(A)
// Event handler for when a body record is double-clicked.
// <P>
// Default implementation fires 'editCell' if appropriate, and handles firing
// 'recordDoubleClick' stringMethod if defined at the field or LG level (That method has a
// different signature from this one)
//
//      @param  record      (ListGridRecord)    record object returned from getCellRecord()
//		@param	recordNum   (number)	index of the row where the click occurred
//		@param	fieldNum	(number)	index of the col where the click occurred
//      @param  [keyboardGenerated]   (boolean) indicates whether this was a synthesized record
//                                              doubleclick in response to a keyboard event
//
//      @see    recordDoubleClick()
//		@group	events
//		@return	(boolean)	false if first click not on same record; true otherwise
//      @visibility external
//<
rowDoubleClick : function (record, recordNum, fieldNum, keyboardGenerated) {

	var field = this.fields[fieldNum], 
        value = this.getCellValue(record, recordNum, fieldNum),
        rawValue = this.getRawCellValue(record, recordNum, fieldNum);
    // suppress user-defined handlers on the group header node
    if (record != null && record._isGroup) return;

	if (field.recordDoubleClick) {
		// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value,rawValue"
		// Convert a string callback to a function
		isc.Func.replaceWithMethod(field, "recordDoubleClick", 
                                         "viewer,record,recordNum,field,fieldNum,value,rawValue");
        var returnVal = field.recordDoubleClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return returnVal;
	}
    if (this.recordDoubleClick != null) {
		// CALLBACK API:  available variables: "viewer,record,recordNum,field,fieldNum,value,rawValue"
        var returnVal = this.recordDoubleClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return returnVal;
    }

	// if the cell is editable, edit it now
	//	(editCell will return true if we've brought up the cell editor)
	if (this.isEditable() && this.editEvent == isc.EH.DOUBLE_CLICK) {
        if (this.handleEditCellEvent(recordNum, fieldNum, isc.ListGrid.DOUBLE_CLICK) == true) return true;    
    	// If this was a keyboard event, and the keyboard click field is not editable, iterate 
    	// through the other fields, and edit the first editable one we find
        if (keyboardGenerated) {
            for (var i = 0; i< this.fields.length; i++) {
                if (i == fieldNum) continue;
                if (this.handleEditCellEvent(recordNum, i, isc.ListGrid.DOUBLE_CLICK) == true) return true;
            }
        }
    }

},

// Body Context Menu
// --------------------------------------------------------------------------------------------
// This will fire 'ListGrid.cellContextClick' if defined.
// Otherwise implements default behavior of showing context menu for the cell.
// enhancement: check for contextMenu,cellContextClick on cell, row, field?
_cellContextClick : function (record, rowNum, colNum) {

	// clear any previous context menu items
	this.cellContextItems = null;

	// Call handler if defined; return false to cancel context menu if handler returns false.  The
	// handler can call lv.makeCellContextItems(record,recordNum,fieldNum), modify the items, and
	// set lv.cellContextItems to customize the context menu.
	if (this.cellContextClick) {
        var record = this.getCellRecord(rowNum, colNum);
		if (this.cellContextClick(record, rowNum, colNum) == false) return false;
    }

	// show cell context menus, or generic context menu?
	if (this.showCellContextMenus) {
	
		// create the cellContextMenu if necessary
		if (!this.cellContextMenu) this.cellContextMenu = this.getMenuConstructor().create(this.contextMenuProperties);

		// get standard menu items if the handler above did not set custom items
		if (!this.cellContextItems) {
			this.cellContextItems = this.makeCellContextItems(
                this.getCellRecord(rowNum, colNum), rowNum, colNum);
		}
		// if there are any menu items, set and show the menu
		if (isc.isAn.Array(this.cellContextItems) && this.cellContextItems.length > 0) {
			this.cellContextMenu.setData(this.cellContextItems);
			this.cellContextMenu.showContextMenu(this);
		}
		
		// return false to kill the standard context menu
		return false;
				
	} else {
	    // do normal Canvas context menu handling. Will fall through to this.showContextMenu.
    	return true;
	}
},

//> @method ListGrid.getShowChildDataSourceContextMenuItemTitle() [A]
// If +link{ListGrid.canOpenRecordDetailGrid} is true and +link{ListGrid.showCellContextMenus}
// is true, we will show menu items to drill into the child datasources in this grid's context
// menu. This method returns the title for that menu item. Override for localization.
// @param ds (DataSource) child datasource to be drilled into
// @return (string) By default returns <code>"Show " + ds.getPluralTitle()</code>
// @group i18nMessages
// @visibility nestedGrid
//<
getShowChildDataSourceContextMenuItemTitle : function (ds) {
    return "Show " + ds.getPluralTitle();
},

// generate standard cell context menu items
makeCellContextItems : function (record, rowNum, colNum) {
    if (this.dataSource != null) {
        var menuItems = [];

    	// menu items to drill into a child DataSource via a nested grid
        if (this.canOpenRecordDetailGrid) {
            var recordDS = isc.DS.get(this.getRecordDataSource(record)),
                childDataSources = recordDS.getChildDataSources();
            if (childDataSources != null) {
                for (var i = 0; i < childDataSources.length; i++) {
                    var ds = childDataSources[i];
                    menuItems.add({
                        title : this.getShowChildDataSourceContextMenuItemTitle(ds),
                        record : record,
                        dataSource : ds,
                        click : "target.openRecordDetailGrid(item.record, item.dataSource)"
                    });
                }
            }
        }

    	// menu item to edit with an embedded form
        if (this.canOpenRecordEditor) {
            
            menuItems.add({
                title : this.openRecordEditorContextMenuItemTitle,
                record : record,
                click : "target.endEditing();target.openRecordEditor(item.record)"
            });
        }

    	// if we are currently showing anything inside the row offer to dismiss it
        if (record != null && this._openRecord == record) {
            menuItems.add({
                title : this.dismissEmbeddedComponentContextMenuItemTitle,
                click : "target.closeRecord()"
            });
        }

    	// menu item to delete a record
        menuItems.add({
            title : this.deleteRecordContextMenuItemTitle,
            click : "target.removeSelectedData()"
        });
        return (menuItems.length > 0 ? menuItems : null);
    }
	return null;
},

// Hover
// ---------------------------------------------------------------------------------------

// override getCanHover.
// If this.canHover is explicitly set to true or false, respect it,
// Otherwise - if any fields are going to show hovers, return true

getCanHover : function () {
    if (this.canHover != null) return this.canHover;
    var showClippedValuesOnHover = this._getShowClippedValuesOnHover();
    if (showClippedValuesOnHover) return showClippedValuesOnHover;
    var fields = this.getFields();
    if (fields != null) {
        for (var i = 0; i < fields.length; i++) {
            if (fields[i].showHover) return true;
        }
    }
    // Either null or false
    return this.canHover;
},

_$hoverHTML: "hoverHTML",
_$fieldHoverHTMLArgNames: "record,value,rowNum,colNum,grid",
_getFieldHoverHTMLCallback : function (rowNum, colNum) {
    var field = this.getField(colNum);
    if (field == null) return null;
    if (field.showHover == false) return null;
    if (field.showHover == null && !this.canHover) return null;
    if (field.hoverHTML) {
        isc.Func.replaceWithMethod(field, this._$hoverHTML, this._$fieldHoverHTMLArgNames);
        return {
            target: field,
            methodName: this._$hoverHTML
        };
    }
    return null;
},

_isCellHoverSuppressed : function (rowNum, colNum) {
    // If we're showing an editor in the cell suppress the standard cell hover.
    if (this.isEditingCell(rowNum, colNum)) {
        return true;
    }
    // Don't attempt to write inactive editor HTML into the hover prompt. This would be
    // odd user-experience and can lead to warnings about inactiveEditorContexts getting
    // unexpectedly dropped.
    
    if (this._showInactiveEditor(colNum) && this.canEditCell(rowNum, colNum)) {
        return true;
    }
    return false;
},

// NOTE: JSDoc imported from GR
cellHoverHTML : function (record, rowNum, colNum) {
    var fieldHoverHTMLCallback = this._getFieldHoverHTMLCallback(rowNum, colNum),
        value = this.getCellValue(record, rowNum, colNum);
    if (fieldHoverHTMLCallback) {
        return isc.Class.fireCallback(fieldHoverHTMLCallback,
                                      this._$fieldHoverHTMLArgNames,
                                      [ record, value, rowNum, colNum, this ]);
    }

    if (value != null && !isc.isAn.emptyString(value) && value != this.emptyCellValue) {
        return value;
    }
},

cellValueHoverHTML : function (record, rowNum, colNum, defaultHTML) {
    var field = this.getField(colNum);
    if (field.showHover == false) return null;
    if (field.showHover == null && this.canHover == false) return null;

    return defaultHTML;
},

//> @method listGridField.hoverHTML()
// StringMethod override point for returning HTML to be shown in hovers over cells in the
// column described by this field.
// <P>
// Called only when +link{listGrid.canHover,canHover} and +link{listGrid.showHover,showHover}
// are both true.
// <P>
// The value of "this" within the method will by the +link{ListGridField,field definition}.
//
// @param record (ListGridRecord) record being hovered over
// @param value  (any) value of the cell being hovered over
// @param rowNum (number) row number where hover occurred
// @param colNum (number) column number where hover occurred
// @param grid   (ListGrid) ListGrid this field is a part of
// @return (HTML) HTML to show in the hover
// @group hovers
// @visibility external
// @example valueHoverTips
//<


//> @attr listGrid.showHoverComponents (Boolean : false : IRWA)
// When set to true and canHover is also true, shows a widget hovering at the mouse point.
// <P>
// A number of builtin modes are provided - see +link{type:HoverMode}.
// @group hoverComponents
// @visibility external
//<


//> @type HoverMode
// When +link{ListGrid.canHover, canHover} and 
// +link{ListGrid.showHoverComponents, showHoverComponents} are both true, HoverMode
// dictates the type of UI to be displayed when a user hovers over a row or cell.
// <P>
// There are a number of builtin HoverModes and you can override 
// +link{listGrid.getCellHoverComponent, getCellHoverComponent()} to create your own
// hover behaviors.
//  
//  @value  "detailField"  Show a single field's value in an +link{class:HtmlFlow}. Field 
//      to use is +link{listGrid.detailField}.
//  @value  "details"   Show a +link{class:DetailViewer} displaying those fields from the 
//      record which are not already displayed in the grid.
//  @value  "related"    Show a separate +link{class:ListGrid} containing related-records.
//      See +link{ListGridRecord.detailDS} and +link{ListGrid.recordDetailDSProperty} for 
//      more information.
//  @value  "detailRelated"    Show a +link{class:DetailViewer} displaying those fields
//      from the record not already displayed in the grid, together with a separate
//      +link{class:ListGrid} containing related-records.
// @group hoverComponents
// @visibility external
//<

//> @attr listGrid.hoverMode (HoverMode : null : IRWA)
// When +link{ListGrid.showHoverComponents, showHoverComponents} is true, the builtin mode
// to use when automatically creating a hover component for rows in this grid.
// <P>
// A number of builtin modes are provided - see +link{type:HoverMode}.  You can also override
// +link{ListGrid.getCellHoverComponent, getCellHoverComponent()} to provide a custom hover
// widget - in that case, this attribute is ignored.
// @group hoverComponents
// @visibility external
//<

_getCellHoverComponent : function (record, rowNum, colNum) {
    if (!this.showHoverComponents) return null;
    // If we're showing an editor in the cell suppress the standard cell hover.
    if (this._editorShowing && this.getEditRow() == rowNum && 
        (!this.editByCell || this.getEditCol() == colNum)) return null;
    var field = this.getField(colNum);
    if (field.showHover == false) return null;
    if (field.showHover == null && !this.canHover) return null;

    if (this.getCellHoverComponent && isc.isA.Function(this.getCellHoverComponent)) {
        return this.getCellHoverComponent(record, rowNum, colNum);
    } else return null;
},

//> @method listGrid.getCellHoverComponent()
// When +link{showHoverComponents} is set, this method is called to get the component to show
// as a hover for the current cell.
// <P>
// By default, this method returns one of a set of builtin components, according to the 
// value of +link{type:HoverMode, listGrid.hoverMode}.  You can override this method 
// to return any component you wish to provide as a hoverComponent, or invoke the superclass
// method to have the default hover component generated, then further customize it.
// <P>
// By default, components returned by <code>getCellHoverComponent()</code> will be
// automatically destroyed when the hover is hidden.  To prevent this, set
// +link{canvas.hoverAutoDestroy} to false on the returned component.
//
// @param record (ListGridRecord) record to get the hoverComponent for
// @param rowNum (int) row number for the cell
// @param colNum (int) column number of the cell
// @return (Canvas | Canvas Properties) the component to show as a hover
// @group hoverComponents
// @visibility external
//<
defaultCellHoverComponentWidth: 100,
defaultCellHoverComponentHeight: 1,
getCellHoverComponent : function (record, rowNum, colNum) {
    return this._getStockEmbeddedComponent(record, false, true, rowNum, colNum);
},

_getStockEmbeddedComponent : function (record, isExpansion, isHover, rowNum, colNum) {
    var gridFields = this.getFields(),
        dsFields = this.dataSource ? isc.getValues(this.getDataSource().getFields()) : 
            gridFields,
        defWidth = (!isHover ? null :
            this.hoverWidth || this.hoverSize || this.defaultCellHoverComponentWidth),
        defHeight = (!isHover ? null :
            this.hoverHeight || this.hoverSize || this.defaultCellHoverComponentHeight),
        remainingFields = [],
        component
    ;

    for (var i=0; i<dsFields.length; i++) {
        var field = dsFields.get(i);
        if (this.dataSource) {
            if (!gridFields.find("name", field.name)) {
                remainingFields.add(field);
            }
        } else {
            // show all fields when not databound
            if (!this.isExpansionField(field)) {
                remainingFields.add(field);
            }
        }
    } 

    var mode = (isHover ? this.hoverMode : (isExpansion ? this.expansionMode : null));

    var props;
    
    // create an appropriate subcomponent and bind it
    if (mode == "detailField") {
        if (!isExpansion && !record[this.detailField]) return null;
        component = this.createAutoChild("expansionDetailField", {
            styleName: this.hoverStyle,
            contents: record[this.detailField]
        });
        props = {
            width: (isHover ? defWidth : "100%"),
            height: (isHover ? defHeight : "100%"),
            members: [component]
        };
        
        if (isHover) {
            props = isc.addProperties(props, {
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }

        component = isc.VLayout.create(props);
    } else if (mode == "details") {
        
        remainingFields = remainingFields.map(function (field) {
            if (field.showIf == null) return field;
            return isc.addProperties({}, field, {showIf: "true"});
        });
        props = { dataSource: this.dataSource, fields: remainingFields };
        
        if (isHover) {
            props = isc.addProperties(props, {
                width: defWidth,
                height: defHeight,
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }
        component = this.createAutoChild("expansionDetails", props);
        
        if (isc.isA.ResultSet(record)) {
            component.setData(record);    
        } else {
            // create ResultSet wrapper on the record, it is possible to observe changes in the datasource
            var recordResultSet = isc.ResultSet.create({
                dataSource : this.dataSource,
                initialData : [record]
            });
            component.setData(recordResultSet);
        }
    } else if (mode == "related") {
        props = { dataSource: this.getRelatedDataSource(record) };
    
        if (isHover) {
            props = isc.addProperties(props, {
                canEdit: false,
                width: defWidth,
                height: defHeight,
                dataProperties: { context: { showPrompt: false } },
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }

        if (isExpansion) {
            // only propagate to autochild if childExpansionMode has been set
            if (this.childExpansionMode != null) props = isc.addProperties(props, {
                canExpandRecords: this.childExpansionMode ? true : false,
                expansionMode: this.childExpansionMode
            });
            // only propagate to autochild if expansionCanEdit has been set
            if (this.expansionCanEdit != null) props = isc.addProperties(props, {
                canEdit: this.expansionCanEdit
            });
        }

        component = this.createAutoChild("expansionRelated", props);

        // if editing is allowed in the sub-grid, set autoSaveEdits: true
        if (this.expansionCanEdit) component.autoSaveEdits = true;
        component.delayCall("fetchRelatedData", [record, this.dataSource, 
            function (dsResponse, data) { 
                if (data == null || data.length == 0) component.setBodyOverflow("visible");
        }]);

    } else if (mode == "detailRelated") {
        props = { dataSource: this.dataSource, fields: remainingFields };
        
        if (isHover) {
            props = isc.addProperties(props, {
                dataProperties: { context: { showPrompt: false } },
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }
        
        var detail = this.createAutoChild("expansionDetails", props)        
        detail.setData(record);

        props = { dataSource: this.getRelatedDataSource(record), height: "100%" };
        
        if (isHover) {
            props = isc.addProperties(props, {
                dataProperties: { context: { showPrompt: false } },
                hoverAutoDestroy: this.hoverAutoDestroy,
                canEdit: false
            });
        }
        if (isExpansion) {
            // only propagate to autochild if childExpansionMode has been set
            if (this.childExpansionMode != null) props = isc.addProperties(props, {
                canExpandRecords: this.childExpansionMode ? true : false,
                expansionMode: this.childExpansionMode
            });
            // only propagate to autochild if expansionCanEdit has been set
            if (this.expansionCanEdit != null) props = isc.addProperties(props, {
                canEdit: this.expansionCanEdit
            });
        }
        
        var related = this.createAutoChild("expansionRelated", props);

        props = { members:[detail, related] };
        if (isHover) {
            props = isc.addProperties(props, {
                width: defWidth,
                height: defHeight,
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }

        component = this.createAutoChild("expansionDetailRelated", props);

    } else if (mode == "editor") {
        component = isc.VLayout.create({
            autoDraw: false,
            width: "100%",
            height: "100%",
            creator: this
        });

        // create an editor form, edit the record in question and validate it - this causes an
        // editor to show validation errors when autoSaveEdits is false and saveAllEdits is 
        // called.
        var editorForm = this.createAutoChild("expansionEditor", {
            dataSource: this.dataSource,
            fields: remainingFields,
            extraSpace: 5,
            itemChanged : function () {
                this.saveButton.setDisabled(false);
            }
        });
        editorForm.editRecord(this.getEditedRecord(rowNum) || record);
        editorForm.validate();

        // create a save button for this expansionEditor - add refs to the editor-form and the
        // record being edited for later use.
        var saveButton = this.createAutoChild("expansionEditorSaveButton", {
            title: this.expansionEditorSaveButtonTitle,
            grid: this,
            form: editorForm,
            record: record,
            disabled: true,
            extraSpace: 5
        });

        // add a ref to the save-button onto the editor-form, so we can enable/disable the 
        // button according to whether changes have been made
        editorForm.saveButton = saveButton;

        // add a ref to the editorForm to the actual expansion-component - this is used when 
        // getting hold of the editor-form to check for changes when a record is collapsed
        component.formMember = editorForm;
        component.addMembers([editorForm, saveButton]);
    }

    if (component) component.expandedRecord = record;

    return component;
},

//> @attr listGridField.showAlternateStyle (boolean : null : IRWA)
// When set to false, don't apply alternate-row styling to this field.
// @visibility external
//<

// Selection
// --------------------------------------------------------------------------------------------

// Simple helper methods to avoid having to refer directly to this.selection
// Genericized up to DataBoundComponent, July 2008


//> @method listGrid.selectRecord()
// Select/deselect a +link{Record} passed in explicitly, or by index.
// <P>
// Note that this method selects records unconditionally, allowing multiple selected 
// records, even when +link{listGrid.selectionType} is "single".  To enforce mutually-exclusive 
// record-selection, use +link{listGrid.selectSingleRecord}.
// @include dataBoundComponent.selectRecord()
//<

//> @method listGrid.deselectRecord()
// @include dataBoundComponent.deselectRecord()
//<

//> @method listGrid.selectRecords()
// @include dataBoundComponent.selectRecords()
//<

//> @method listGrid.deselectRecords()
// @include dataBoundComponent.deselectRecords()
//<

//> @method listGrid.selectAllRecords()
// Select all records.
// <P>
// Note that this method will select records even if +link{listGrid.canSelectRecord()} returns
// false for the record in question. See also +link{listGrid.userSelectAllRecords()}
// @visibility external
//<
selectAllRecords : function () {
    this._dontRefreshSelection = true;
    this.selection.selectAll();
    this._dontRefreshSelection = null;
    this._markBodyForRedraw("select all");
    if (this.getCurrentCheckboxField() != null) {
        // don't check the "all records selected header box if we can't select all records
        if (!isc.ResultSet || !isc.isA.ResultSet(this.data) ||
                this.data.allMatchingRowsCached()) 
        {
            this._setCheckboxHeaderState(true);
        }
    }
    this.fireSelectionUpdated();
},

//> @method listGrid.deselectAllRecords()
// @include dataBoundComponent.deselectAllRecords()
//<
deselectAllRecords : function () {
    this._dontRefreshSelection = true;
    this.selection.deselectAll();
    this._dontRefreshSelection = null;
    this._markBodyForRedraw("deselect all");
    if (this.getCurrentCheckboxField() != null) this._setCheckboxHeaderState(false);
    this.fireSelectionUpdated();
},

//> @method listGrid.userSelectAllRecords()
// Selects every user-selectable record in the grid. Unlike +link{listGrid.selectAllRecords()},
// if a record is +link{listGrid.canSelectRecord(),unselectable}, this method will not
// attempt to select it.
// @visibility external
//<
userSelectAllRecords : function () {
    var data = this.data,
        total = this.data.getLength();
    if (isc.ResultSet && isc.isA.ResultSet(data) && !data.rangeIsLoaded(0, total)) {
        this.selection.warnSelectionRangeNotLoaded();
        return;
    }
    var records = data.getRange(0, total),
        finalRecords = [];
    for (var i = 0; i < records.length; i++) {
        if (!this.canSelectRecord(records[i])) continue;
        finalRecords[finalRecords.length] = records[i];
    }
    this.selection.selectList(finalRecords);
},

//> @method listGrid.anySelected()
// @include dataBoundComponent.anySelected()
//<

//> @method listGrid.selectSingleRecord()
// @include dataBoundComponent.selectSingleRecord
// @visibility external
//<

//> @method listGrid.canSelectCell() (A)
// If +link{listGrid.canSelectCells} is set to <code>true</code>, this method
// will be called for each cell the user attempts to select. If it returns false, the
// cell will not be selected.
//
// @param rowNum (int) rowNum being selected
// @param colNum (int) colNum being selected
// @return (boolean) return false to disallow selection
//
// @visibility external
//<
canSelectCell : function(rowNum, colNum) {
	return true;
},

//> @method listGrid.canSelectRecord()
// If +link{listGrid.selectionType} is not set to <code>"none"</code>, 
// This method will be called for each record the user attempts to select. If it returns false, the
// record will not be selected.
// <P>
// Note this method will not be called at all if +link{canSelectCells} is true.
//
// @param record (ListGridRecord) record being selected
// @return (boolean) return false to disallow selection
//
// @visibility external
//<
canSelectRecord : function(record) {
    return true;
},
// Keyboard Navigation
// --------------------------------------------------------------------------------------------

//> @method listGrid.keyPress()
// Handle a keyPress event on the ListGrid as a whole.  
// <P>
// Note that the majority of keyboard handling for a ListGrid is performed by
// +link{bodyKeyPress()} and most overrides are better performed there.
//
// @return (boolean) return false to cancel
// @visibility external
//<

//> @method listGrid.bodyKeyPress()
// Handle a keyPress event on the body.  
// <P>
// Default implementation handles navigating between records with arrow keys, and activating
// records with space and enter.
//
// @return (boolean) return false to cancel
//
// @visibility external
//<

_$ArrowUp:"Arrow_Up", _$ArrowDown:"Arrow_Down",
_$ArrowLeft:"Arrow_Left", _$ArrowRight:"Arrow_Right",
_$Space:"Space", _$Enter:"Enter",
_$f2:"f2",
bodyKeyPress : function (event, eventInfo) {
    
    // Easy to wrap entryPoint for SGWT.
    if (this.onBodyKeyPress(event, eventInfo) == false) return false;

    
    if (this._editorShowing) {
        var target = event.keyTarget,
            canvasItem;
        while (canvasItem == null && target != this && target != null) {
            canvasItem = target.canvasItem;
            target = target.parentElement;
        }
        if (canvasItem != null && canvasItem.form == this.getEditForm()) {
            var returnVal = this.editorKeyPress(canvasItem, isc.EH.getKey(),
                                isc.EH.getKeyEventCharacterValue());
            return (returnVal == null ? isc.EH.STOP_BUBBLING : returnVal);
        }
    }
    if (this.data.getLength() > 0) {

        // if we start editing on keypress, return false to kill the event (avoiding
        // page navigation on backspace keypress, etc)
        var EH = isc.EventHandler,
            keyName = event.keyName;  

    	// for arrow keys, navigate to the appropriate record
        var editOnKeyPress = this.editOnKeyPress && this.isEditable();
        if (editOnKeyPress && this._editOnKeyPress(event, eventInfo)) return false;

        switch (keyName) {
            case this._$ArrowUp:    return this._navigateToNextCell(-1,  0);
            case this._$ArrowDown:  return this._navigateToNextCell( 1,  0);
            case this._$ArrowLeft:  return this._navigateToNextCell( 0, -1);
            case this._$ArrowRight: return this._navigateToNextCell( 0,  1);
        }

    	// Generate a click on the current focus record when the user hits Space            
        if (keyName == this._$Space) {
            
            if (this.generateClickOnSpace) 
                if (this._generateFocusRecordClick() == false) return false;
                
            if (this.generateDoubleClickOnSpace)
                return this._generateFocusRecordDoubleClick();

    	// Generate a doubleClick on the current focus record when the user hits Enter    
        } else if (keyName == this._$Enter) {
            if (this.generateClickOnEnter)  
                if (this._generateFocusRecordClick() == false) return false;

            if (this.generateDoubleClickOnEnter) 
                return this._generateFocusRecordDoubleClick();

        // Invoke one of the copy/paste shortcuts controlled by useCopyPasteShortcuts
        } else if (EH.modifierKeyDown() && 
                   (keyName == "D" || keyName == "R" || keyName == "C" || keyName == "V")) {
            return this._invokeKeyboardCopyPasteShortcut(keyName);

        // support selecting all cells using Ctrl-A shortcut
        } else if (EH.modifierKeyDown() && keyName == "A" &&
                   this.canSelectCells && this.canSelectAll) {
            this.selection.selectAll();
            return false;

        // Start editing on f2 keypress if editOnF2Keypress is true.
        } else if (keyName == this._$f2 && this.editOnF2Keypress &&
                    this.isEditable() && this.editEvent != "none") 
        {
            var rowNum, colNum;
            if (this.canSelectCells && this.editByCell) {
                var cell = this.getFocusCell();
                if (cell[0] >= 0) rowNum = cell[0];
                if (cell[1] >= 0) colNum = cell[1];
            } else {
                rowNum = this.getFocusRow();
                if (rowNum < 0) rowNum = null;
            }
            this.startEditing(rowNum, colNum);
            return false;
        }
    }
    return true;
},

//> @method ListGrid.onBodyKeyPress()
// @include ListGrid.bodyKeyPress
// @return (boolean) return false to cancel default drop handling
// @visibility sgwt
//<

onBodyKeyPress : function () {
},


_navigateToNextCell : function (rowStep, colStep) {

    if (!this.canSelectCells) return this._navigateToNextRecord(rowStep);
    else if (this.selectionType == isc.Selection.NONE) return true;

    var navStyle = this.getArrowKeyAction();
    if (navStyle == this._$none) return true;

    var shiftedSelection = isc.EH.shiftKeyDown();

    // To match Excel, re-establish origin if it has been lost
    if (navStyle == this._$select && shiftedSelection) {
        this._ensureValidSelectionOrigin();
    }

    var hiliteCell = this.getFocusCell(shiftedSelection);
    var newRow = hiliteCell[0], newCol = hiliteCell[1];

    if (newRow != null && newCol != null) {
        newRow += rowStep;
        newCol += colStep;
    } else {
        newRow = newCol = 0;
    }

    // For Ctrl+Shift+Arrow Key, We don't want to end up on a disabled cell, so we 
    // still need to use the skipping logic below, but reverse the search direction
    var selectToEnd = navStyle == this._$select && isc.EH.modifierKeyDown();
    if (selectToEnd) {
        if      (rowStep > 0) newRow = this.getTotalRows() -1;
        else if (rowStep < 0) newRow = 0;
        if      (colStep > 0) newCol = this.getTotalCols() -1;
        else if (colStep < 0) newCol = 0;
        rowStep = -rowStep;
        colStep = -colStep;
    }

    // At this point we have a pointer to the cell we'd like to hilite / click.
    // If the cell is not enabled, find the next one that is
    while (!this.recordIsEnabled(newRow, newCol)) {
        if (rowStep != 0) newRow += rowStep;
        if (colStep != 0) newCol += colStep;

        // bail if reverse search returns to original cell
        if (newRow == hiliteCell[0] && newCol == hiliteCell[1]) return true;

        // if we are trying to navigate past the ends of the list, bail
        if (newRow < 0 || newCol < 0 || newRow >= this.getTotalRows() 
                                     || newCol >= this.getTotalCols()) return true;
    }

    // Note - we already returned if the navigation style is null, so it is fair to assume that
    // we're either going to simulate a click on a cell, or hilite one.
    // 
    // Clear out the last hilite. This will both clear out the keyboard specific properties 
    // (_lastKeyboardHiliteRow and col), and reset the css style of the last over cell
    this.clearLastHilite(); 

    // Remember which row and col we're interested in
    this._lastKeyboardHiliteRow = newRow;
    this._lastKeyboardHiliteCol = newCol;
    this._lastKeyboardHiliteBody = this.getFieldBody(newCol);
    
    if (navStyle == this._$select) this._generateCellClick(newRow, newCol);
    if (navStyle == this._$focus)  this._hiliteCell(newRow, newCol);

    this._handlingKeyboardNavigation = true;
    this.scrollToCell(newRow, newCol);
    this._handlingKeyboardNavigation = false;    

    return false; // stop event propagation        
},

_hiliteCell : function (row, col) {
    // if passed in a cell, resolve it to row / col
    if (isc.isAn.Object(row)) {
        col = row._colNum;
        row = row._rowNum;
    }

    // bail if the coordinates passed in don't match a valid row
    if (row == null || col == null || 
        row < 0     || col < 0     ||
        row >= this.getTotalRows() || col >= this.getTotalCols()) return;

    var body = this._lastKeyboardHiliteBody,
        colOffset = body.selection.firstCol || 0;
            
    body.lastOverRow = row;
    body.lastOverCol = col - colOffset;
    body.setRowStyle(row, null, col - colOffset);
},

_getLastMouseOverBody : function () {
    if (!this.frozenBody) return this.body;
    if (this.frozenBody.lastMouseOverRow == null && 
        this.frozenBody.lastMouseOverCol == null) return this.body;
    return this.frozenBody;
},

_ensureValidSelectionOrigin : function () {
     if (!this.selection._validateSelectionOrigin(this)) {
         var lastBody = this._getLastMouseOverBody(),
             colOffset = lastBody.selection.firstCol || 0;
         var row = lastBody.lastMouseOverRow || 0,
             col = lastBody.lastMouseOverCol || 0;
         if (this.recordIsEnabled(row, col + colOffset)) {
             lastBody.selectOnMouseDown(this.getCellRecord(row, col + colOffset), row, col);
         }
     }
 },

// editOnKeyPress behavior
// - modelled on spreadsheet style application editing
// - normal record selection and navigation occurs on click / arrow keypress when not currently
//   editing the grid
// - on character keypress, start editing (respecting the character typed)

_$f2:"f2",
_$Escape:"Escape",
_$Backspace:"Backspace",
_$Delete:"Delete",
_$keyPress:"keyPress",
_editOnKeyPress : function (event, eventInfo) {
    var keyName = eventInfo.keyName,
        charVal = isc.EH.getKeyEventCharacter(event);  
    // We don't want to start editing if the user hit a non character key, such as a function key
    // or escape, etc   
        
    if (keyName != this._$f2 && keyName != this._$Delete && keyName != this._$Backspace && 
        (keyName == this._$Escape || 
            isc.EH._nonCharacterKeyMap[keyName] || charVal == null || charVal == isc.emptyString)) 
    {
        return false;
    }

    var cell = this.getFocusCell(),
        row = cell[0] || 0,
        col = cell[1] || 0;
    // If we're already showing an editor just bail
    if (this._editorShowing || row < 0 || col < 0) return false;
    
    // on Enter / f2 keypress don't modify the value in the cell
    var undef;
    if (keyName == this._$Enter || keyName == this._$f2) charVal = null;

    var editVal;
    if (charVal != null) {
        if (keyName == this._$Delete || keyName == this._$Backspace) {
            editVal = null;
        } else if (this.autoSelectEditors) {
            editVal = charVal;
        } else {
            editVal = this.getEditedCell(row,col) + charVal;
        }
        // this flag ensures that when we focus in the item we put selection at the end, rather
        // than selecting the entire value.
        this._editorCursorAtEnd = true;
    }
    return this.handleEditCellEvent(cell[0], cell[1], this._$keyPress, editVal);
    
},

// getArrowKeyAction() - used by _navigateToNextRecord() to determine how the record
// should be hilighted.

// Strings used in navigation styles
_$none:"none", _$focus:"focus", _$select:"select", _$activate:"activate",
getArrowKeyAction : function () {

    var action = this.arrowKeyAction;

    // No action at all trumps everything
    if (action == this._$none) return this._$none;

    // if ctrl key is down but not shift, always just hilite / focus
    var doNotFocus = this.canSelectCells && isc.EH.shiftKeyDown();
    if (isc.EH.modifierKeyDown() && !doNotFocus) return this._$focus;

    // if an explicit arrowKeyAction has been set, respect it
    if (action != null) return action;

    // default to FOCUS if selectionAppearance is "checkbox"; SELECT otherwise
    return this.selectionAppearance == "checkbox" ? this._$focus : this._$select;
},


// In screenReader mode when putting focus onto an actual row element in the body,
// should we show normal keyboard hilite styling etc?
hiliteOnNativeRowFocus:true,

// _navigateToNextRecord()
// Called from this.bodyKeyPress() on arrow keys, to handle navigating around the listGrid.
// If step == +1, we want to navigate to the next record in the list, if step is -1, we want to
// navigate to the previous record.
// Determines which record to navigate to, and falls through to _generateCellClick() or 
// _hiliteRecord() depending on the result of this.getArrowKeyAction()
_navigateToNextRecord : function (step) {
	// Are we going to simulate a click on the next record or just hilight it?
    var navStyle = this.getArrowKeyAction();
    
	// If keyboard navigation is disabled return true to allow processing to continue.
    if (navStyle == this._$none || step === 0) return true;

	// Note: we are either going forward or backward one record - assume jumping over multiple 
	// records is not supported
	//
	// Default to selecting the next record
    if (step == null) step = 1;
    
	// Determine which record was last hilighted or clicked
    var newSelectionIndex;

	// By default we want the last row that had keyboard focus
    newSelectionIndex = this.getFocusRow(step > 0);
    
    // Otherwise, get the last record clicked
    if (newSelectionIndex == null) newSelectionIndex = this._lastRecordClicked;
    
    var originalSelection = isc.isA.Number(newSelectionIndex) ? newSelectionIndex : 0;
    
    if (isc.isA.Number(newSelectionIndex)) newSelectionIndex += step;
	// Default to the first record
    else newSelectionIndex = 0;
    
    var lastRow = this.getTotalRows() -1;
	// if we are trying to navigate past the ends just ensure the focus row is selected
    if (newSelectionIndex < 0 || newSelectionIndex > lastRow) {
        // bail if there were no records
        if (lastRow < 0) return true;
        
        // Ensure the original record is selected / focused
        
        newSelectionIndex = originalSelection;
        
    }
	// At this point we are sure that newSelectionIndex is a number.
	// If the number is beyond the end of the list in either direction, or 
	// the record is not enabled, recordIsEnabled() will return false.
	// Try the next record in the step direction, and so on until we find an enabled record or
	// hit the end of the list.

	// If the record is disabled, find the first non-disabled record (in the appropriate
	// direction)
    while (!this.recordIsEnabled(newSelectionIndex, 0)) {
        newSelectionIndex += step;  
    	// if we are trying to navigate past the ends of the list, bail
        if (newSelectionIndex < 0 || newSelectionIndex > lastRow) {
            newSelectionIndex = originalSelection;
            break;
        }
    }
    // move native focus to the selected row so that screen readers will read it
    if (isc.screenReader) {
        this.body._putNativeFocusInRow(newSelectionIndex);
    }
    
    //this.logWarn("navStyle: " + navStyle + ", target index: " + newSelectionIndex);

    if (navStyle == this._$focus) this._hiliteRecord(newSelectionIndex);
    else {
        // if the user hit up arrow on the first row or down arrow on the last row, don't
        // actually force a click handler to fire on the row.
        // This leads to odd interactions with ListGrid editing on click as arrow down on the
        // last row will hide the editor, then arrow down again will show it on the same
        // row but the first editable col.
        // If the user does want to force a click on the current row via the keyboard
        // they can always hit space or enter.
        
        if (newSelectionIndex == originalSelection) {
            var colNum = this._getKeyboardClickNum();
            if (colNum >= 0) {
                if (this.body.selectionEnabled() &&
                    this.recordIsEnabled(newSelectionIndex, colNum)) 
                {
                    this.selection.selectOnMouseDown(this, newSelectionIndex, colNum);
                    this.selection.selectOnMouseUp(this, newSelectionIndex, colNum);
                    
                    // Explicitly fire the selectionUpdated notification
                    this.fireSelectionUpdated();                    
                }
            }
        }
        else if (navStyle == this._$select)   this._generateCellClick(newSelectionIndex);
        else if (navStyle == this._$activate) this._generateRecordDoubleClick(newSelectionIndex);
    }
    this._handlingKeyboardNavigation = true;
    this.scrollRecordIntoView(newSelectionIndex)
    this._handlingKeyboardNavigation = false;

	// Don't allow the keypress event handling to continue here.
    return false;
},

_getKeyboardClickNum : function () {

	// If this.keyboardClickField was specified, return the appropriate colNum
	// Note - can be specified as a field number or field name...
    var kcf = this.keyboardClickField;
    if (kcf != null) {
        
        var kcCol;
        if (isc.isA.Number(kcf) && kcf > 0 && kcf < this.fields.length) {
            kcCol = this.fields[kcf];
        } else {
            kcCol = this.fields.find(this.fieldIdProperty, kcf);
        }
        if (kcCol && kcCol.ignoreKeyboardClicks) {
            this.logWarn("Explicitly specified keyboardClickField:" + this.keyboardClickField +
                " refers to a field which disallows keyboard click events.");
            kcCol = null;
        }
        if (kcCol != null) {
            return this.fields.indexOf(kcCol);
        }
    }
    if (this.getCurrentCheckboxField() != null) {
        return this.getCheckboxFieldPosition();
    }
    
    for (var i = 0; i < this.fields.length; i++) {
        if (this.fields[i].ignoreKeyboardClicks != true) return i;
    }
    // Unable to find a valid field - return -1. Calling code will have to handle this    
    return -1;
},

_generateCellClick : function (rowNum, colNum, focus) {    

    if (!this.canSelectCells) this.clearLastHilite();

	// if passed a record, resolve it to an index!
    if (isc.isAn.Object(rowNum)) rowNum = this.getRecordIndex(rowNum);

	// Make sure we're not trying to select a record beyond the ends of the list.
    if (!isc.isA.Number(rowNum) || rowNum < 0) rowNum = 0;
    if (rowNum >= this.data.getLength()) rowNum = this.data.getLength() -1;

    var body;
    if (this.canSelectCells) {
        if (colNum == null || colNum < 0 || colNum >= this.getTotalCols()) return;
        body = focus ? this.getFieldBody(colNum) : this._lastKeyboardHiliteBody;
    } else {
        body = this.body;
	    // remember we artificially selected this record from a keyboard event
        body._lastHiliteRow = rowNum;
    
        colNum = this._getKeyboardClickNum();
        if (colNum == -1) return;
    }
    var colOffset = body.selection.firstCol || 0;

	// Trigger the methods to perform the selection (selection.selectOnMouseDown AND 
	// selection.selectOnMouseUp)
	
    var performSelection = 
        (body.selectionEnabled() && this.recordIsEnabled(rowNum, colNum)) ;
	
    if (performSelection) body.selectOnMouseDown(this, rowNum, colNum - colOffset);
    
	// explicitly fire this 'rowClick' method, passing in the additional method flagging this
	// as a keyboard generated click
    this.rowClick(this.getCellRecord(rowNum, colNum), rowNum, colNum, true);
    if (performSelection) body.selectOnMouseUp(this, rowNum, colNum - colOffset);
    
	// Stop event propagation
    return false;

},

//> @method listGrid.getFocusRow() [A]
// Get the row that currently has keyboard focus.  Arrow key navigation moves relative to this
// row.
// 
// @return (Number) rowNum of the current focus row
// @visibility external
//<
// @param last (boolean) if multiple rows are selected, should we return the last row in the
//  selection (rather than the first?
getFocusRow : function (last) {
    
	// We want the last record hilighted by the keyboard.
	// Note: If the last keyboard hilite type event was a generated record click, the
	// lastHiliteRow will match the lastRecordclicked property for this widget.
	// If the last keyboard hilite type event was a hilite (rollover style) event, the 
	// lastHiliteRow will match the lastOverRow for the body.
	// If neither of these are true, we can assume a subsequent mouse event has occurred over
	// a different row, effectively invalidating the _lastHiliteRow property, so should be
    // ignored and deleted
    if (this.body._lastHiliteRow != null && 
        ((this.body._lastHiliteRow == this.body.lastOverRow) || 
         (this.body._lastHiliteRow == this._lastRecordClicked)) )
    {
        return this.body._lastHiliteRow;
    }
  
    delete this.body._lastHiliteRow;
    // If there is no valid keyboard hilite row, return a record from the end of the selection
    // We use 'direction' param to indicate whether it's more appropriate to return the
    // first or last selected record of a multiple selection
	var selection = this.getSelection();    
    if (selection.length == 0) return null;
    selection = selection[(last ? selection.length -1 : 0)]
    return this.getRecordIndex(selection);

},

// Helper method to get the last keyboard hilite cell position.
getFocusCell : function (querySelectionObject) {

    if (!this.canSelectCells) return [this.getFocusRow(), this._getKeyboardClickNum()]

    var row, col;
    
    // this._lastKeyboardHiliteRow and col are only valid if they match the last row clicked, or
    // the last row hilited - otherwise some non-keyboard manipulation has occurred since the values
    // were set.
    
    if (this._lastKeyboardHiliteRow != null && this._lastKeyboardHiliteCol != null) {
        var body = this._lastKeyboardHiliteBody;
        if (body != null) {
            var colOffset = body.selection.firstCol || 0;
            if ((this._lastKeyboardHiliteRow == body.lastOverRow &&
                 this._lastKeyboardHiliteCol == body.lastOverCol + colOffset) || 
                (this._lastSelectedBody      == body &&
                 this._lastKeyboardHiliteRow == body._lastSelectedRow &&
                 this._lastKeyboardHiliteCol == body._lastSelectedCol + colOffset))
            {
                row = this._lastKeyboardHiliteRow;
                col = this._lastKeyboardHiliteCol;
            }
        }
    }

    // If we didn't find a keyboard cell, grab the last selected cell    
    if (row == null || col == null) {
        if (querySelectionObject) {
            row = this.selection.lastRow;
            col = this.selection.lastCol;
        }
        var body = this._lastSelectedBody;
        if (body) {
            var colOffset = body.selection.firstCol || 0;
            if (row == null || col == null) {
                row = body._lastSelectedRow;
                col = body._lastSelectedCol + colOffset;
            }
        }
        if (row == null || col == null) {
            if (this.getSelection().getLength > 0) {
                var selection = this.getSelection();
                row = selection[0]._rowNum;
                col = selection[0]._colNum;
            }
        }
    }
    return [row, col];
},

_generateFocusRecordClick : function () {
    var cell = this.getFocusCell(), 
        row = cell[0], 
        col = cell[1];

    if (row != null && col != null) {
        this._generateCellClick(row, col, true);
        return false;
    }
	// allow event processing to continue..
    return true;
},

_generateRecordDoubleClick : function (rowNum) {

    // determine the appropriate col
    var colNum = this._getKeyboardClickNum();
    if (colNum == -1) return;
    // generate a double click, on the appropriate record (and field), passing in the
	// parameter flagging this as a keyboard synthesized click event.
	//this._handleRecordDoubleClick(currentRecord, colNum);
    this.rowDoubleClick(this.getCellRecord(rowNum, colNum), rowNum, colNum, true);
},

_generateFocusRecordDoubleClick : function () {

    if (this.canSelectCells) return this._generateFocusRecordClick();

    // determine the appropriate col
    var colNum = this._getKeyboardClickNum(),
        currentRecord = this.getFocusRow();
    
    if (colNum != -1 && currentRecord != null) {
        
        this._generateRecordDoubleClick(currentRecord);
        return false;
    }
    
	// allow event processing to continue..
    return true;

},

// Scrolling
// --------------------------------------------------------------------------------------------

//>	@method	listGrid.scrollRecordToTop()	(A)
// Scroll the listGrid body such that the specified row is visible at the top of the viewport.
//		@group	scrolling
//		@param	rowNum  (number)    Index of the row to scroll into view
//<
scrollRecordToTop : function (rowNum) { return this.scrollRecordIntoView(rowNum, false); },

//>	@method	listGrid.scrollRecordIntoView()	(A)
// Scroll the listGrid body such that the specified row is visible close to the
// center of the viewport.
//		@group	scrolling
//		@param	rowNum  (number)    Index of the row to scroll into view
//<

scrollRecordIntoView : function (rowNum, center) {
    return this.scrollToCell(rowNum, null, center);
},

//>	@method	listGrid.scrollToColumn()
// Scroll the grid to specified column such that the row appears near the center of the 
// viewport.
// <P>
// See +link{listGrid.scrollToCell()} for a full description of how
// this method interacts with incremental loading and rendering of data.
// @group	scrolling
// @param	colNum  (number)    Index of the column to scroll into view
// @visibility external
//<
scrollToColumn : function (colNum, center) {
    return this.scrollToCell(null, colNum, center);
},

//> @method listGrid.scrollToRow()
// Scroll the grid to specified row such that the row appears near the center of the 
// viewport, loading data if necessary.
// <P>
// See +link{listGrid.scrollToCell()} for a full description of how
// this method interacts with incremental loading and rendering of data.
//
// @group scrolling
// @param	rowNum  (number)    Row index of the cell to scroll into view
// @visibility external
//<
scrollToRow : function (rowNum) {
    this.scrollToCell(rowNum, 0);    
    // allow for chaining other function calls after scrollToRow()
    return this;
},

// helper callback to weed out calls to scrollToRow that should be ignored. 
// we only want to perform the last call to scrollToRow() that occurs before data arrives.
scrollCellCallback : function (rowNum, colNum, center, alwaysCenter, stamp) {    
    if (stamp == this._currentScrollCall) {
        // reset counter
        this._currentScrollCall = null;
        this.scrollToCell(rowNum, colNum, center, alwaysCenter);
    }
},

//>	@method	listGrid.scrollToCell()	(A)
// Will scroll the listGrid body such that the specified cell is visible close to the
// centre of the viewport.
// <P>
// This method has no effect if the cell is already visible in the viewport.
// <P>
// When scrolling vertically, this will cause data to be automatically loaded
// if +link{dataFetchMode,paging is active} and you scroll into an area of
// the data that isn't loaded.  Only rows around the target row will be
// loaded, not all intervening rows.  See also +link{ResultSet}.
// <P>
// Scrolling into an undrawn area will cause the body area of the grid to
// redraw, but this won't happen synchronously unless you explicitly call
// redraw().  Scrolling into an area of the data that is not yet loaded
// will never synchronously draw new rows, even if you call redraw() -
// wait for +link{dataArrived} to be notified when new rows have been
// loaded.
// <P>
// Calling this method with a row index larger than the
// current dataset will clamp to the end of the dataset (similarly horizontal 
// scrolling will clamp to the last column).
// <P>
// If a call to this method is made while data is still loading, such
// that the last row of the dataset is not yet known the grid will attempt to compensate
// by scrolling the record into view when data arrives, if it is valid.
// For better control over scrolling, developers should consider
// calling <code>scrollToRow()</code> or <code>scrollToCell</code>
// from +link{dataArrived()} if data is still loading.
// <P>
// With mixed-height rows it will only reliably work if virtualScrolling
// is enabled.
//
// @group	scrolling
// @param	rowNum  (int)    Row index of the cell to scroll into view
// @param	colNum  (int)    Column index of the cell to scroll into view
// @visibility external
//<
scrollToCell : function(rowNum, colNum, center, alwaysCenter) {
    return this.scrollCellIntoView(rowNum, colNum, center, alwaysCenter);
},
// NOTE: as of 2006.11.28, the 'scrollIntoView' method this eventually falls through to supports
// individual x and y positions of top/left, center, bottom/right.
// Therefore easy to modify this method to take x and y positions rather than a single boolean 
// "center" for either centered or displayed at top and left.
// alwaysCenter: scroll even if the cell is already in view (center it or place at left/top
//               according to "center" parameter
defaultScrollCenter: true,
scrollCellIntoView : function (rowNum, colNum, center, alwaysCenter) {
    if ((isc.isAn.Array(this.data) && this.data.length == 0 && this.dataSource)
        || (isc.ResultSet && isc.isA.ResultSet(this.data) && !this.data.lengthIsKnown())) {
        // keep track of conscective calls to scrollToRow() so we can only perform the most
        // recent one, before data arrives.        
        if (!this._currentScrollCall) this._currentScrollCall = 1;
        else this._currentScrollCall += 1;
        var stamp = this._currentScrollCall;
        isc.Page.waitFor(this, "dataArrived", 
            {method: this.scrollCellCallback, args: [rowNum, colNum, center, alwaysCenter, stamp],
            target:this}
        );
                
        return;
    }
     
	// if the body isn't drawn, we can't scroll the cell into view - set a flag to scroll the
	// body when it gets drawn
    if (!this.body || !this.body.isDrawn()) {
        this.logInfo("scrollCellIntoView() called before the body has been drawn.  Cell " +
                     rowNum + "," + colNum + " will scrolled into view on draw().");
        this._scrollCell = [rowNum, colNum];
        return;

    }
    
    // Force an immediate adjustOverflow before we calculate row / viewport size etc
    
    if (this.body._deferredOverflow || this.body._overflowQueued) {
        this.body.adjustOverflow("ScrollCellIntoView requesting size");
    }
    
    if (center == null) center = this.defaultScrollCenter;

    var x, y, width, height, body = this.body;
    if (rowNum != null) {
        if (!body._isVirtualScrolling) {
            y = body.getRowTop(rowNum);
            height = body.getRowSize(rowNum);
        } else {
            // If the row is already in the viewport, don't call scrollToTargetRow()
            // as this will shift it about which can be confusing if the user is 
            // navigating through rows with the keyboard.
            var undrawn = body._firstDrawnRow == null || rowNum < body._firstDrawnRow || 
                rowNum > body._lastDrawnRow,
                inViewport = !undrawn;
            if (inViewport) {
                var rowTop = body.getRowTop(rowNum),
                    rowHeight = body.getRowHeight ? 
                                    body.getRowHeight(this.getCellRecord(rowNum), rowNum, 0) :
                                    body.cellHeight,
                    scrollTop = body.getScrollTop();
                if (scrollTop > rowTop || 
                    ((body.getViewportHeight() + scrollTop) < (rowTop + rowHeight))) 
                {
                    inViewport = false;
                }
            }
            if (!inViewport) {
                // scrolling to a particular coordinate would be meaningless with unknown row
                // heights
                body._targetRow = rowNum;
                if (center) {
                    body._rowOffset = -1 * ((body.getViewportHeight() / 2) - body.cellHeight);
                } else {
                    body._rowOffset = 0;
                }
                body._scrollToTargetRow();
            }
            return;
        }
    }
    if (colNum != null) {
        if (this.frozenFields != null && colNum < this.frozenFields.length) {
            colNum = null;
        } else {
        
            x = this.getColumnLeft(colNum);
            width = this.getColumnWidth(colNum);
        }
    }
    
    // Catch the case where we're dirty, and the row being scrolled into view is not yet present
    // in our HTML (so we can't scroll into view until the redraw occurs)
    if (this.isDirty() || this.body.isDirty()) {
        var mustRedraw;
        if (rowNum != null) {
            var scrollHeight = body.getScrollHeight();
            if (y+height > scrollHeight) mustRedraw = true;
        }
        if (!mustRedraw && colNum != null) {
            var scrollWidth = body.getScrollWidth();
            if (x+width > scrollWidth) mustRedraw = true;
        }
        if (mustRedraw) this.redraw("scrollIntoView");
    }
    
	//this.logWarn("ScrollIntoView passed: " + [rowNum, colNum] + 
	//             ", calculated target cell position:" + [x,y] + ", size:" + [width,height]);

    body.scrollIntoView(x,y,width,height, (center ? "center" : "left"), (center ? "center" : "top"), 
                        null, null, alwaysCenter)

},

// Header/Body Scroll Sync
// --------------------------------------------------------------------------------------------
// Note - we keep the body / header's horizontal scroll position in synch by firing 
// syncHeaderScrolling when the body is scrolled, and syncBodyScrolling where the header is
// scrolled.
// We have to have these no-op if the header / body are already at the same place to avoid an
// infinite loop.
bodyScrolled : function (left, top, isFrozen) {
    // Assertion
    // the frozen body should never be clipping horizontally so if this was a
    // scroll from the frozen body, just synch up the vertical scroll position of the
    // unfrozen body and we're done.
    // NOTE: There's no obvious way for the user to scroll just the frozen body but this
    // could probably happen from interactions like keyboard events
    
    
    i