/*
 * Decompiled with CFR 0.152.
 */
package org.exoplatform.wallet.blockchain.service;

import java.io.IOException;
import java.math.BigInteger;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.commons.api.settings.SettingService;
import org.exoplatform.commons.api.settings.SettingValue;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.container.ExoContainer;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.container.PortalContainer;
import org.exoplatform.container.component.RequestLifeCycle;
import org.exoplatform.services.listener.ListenerService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.wallet.blockchain.service.EthereumClientConnector;
import org.exoplatform.wallet.contract.MeedsToken;
import org.exoplatform.wallet.model.ContractDetail;
import org.exoplatform.wallet.model.Wallet;
import org.exoplatform.wallet.model.transaction.TransactionDetail;
import org.exoplatform.wallet.service.BlockchainTransactionService;
import org.exoplatform.wallet.service.WalletAccountService;
import org.exoplatform.wallet.service.WalletTransactionService;
import org.exoplatform.wallet.utils.WalletUtils;
import org.picocontainer.Startable;
import org.web3j.abi.EventEncoder;
import org.web3j.abi.EventValues;
import org.web3j.abi.datatypes.Event;
import org.web3j.abi.datatypes.Type;
import org.web3j.protocol.core.DefaultBlockParameterName;
import org.web3j.protocol.core.Response;
import org.web3j.protocol.core.methods.response.EthSendTransaction;
import org.web3j.protocol.core.methods.response.Transaction;
import org.web3j.protocol.core.methods.response.TransactionReceipt;
import org.web3j.tx.Contract;

public class EthereumBlockchainTransactionService
implements BlockchainTransactionService,
Startable {
    private static final Log LOG = ExoLogger.getLogger(EthereumBlockchainTransactionService.class);
    private static final String TRANSFER_SIG = EventEncoder.encode((Event)MeedsToken.TRANSFER_EVENT);
    private static final String APPROVAL_SIG = EventEncoder.encode((Event)MeedsToken.APPROVAL_EVENT);
    private static final String TRANSFER_OWNERSHIP_SIG = EventEncoder.encode((Event)MeedsToken.OWNERSHIPTRANSFERRED_EVENT);
    private static final Map<String, String> CONTRACT_METHODS_BY_SIG = new HashMap<String, String>();
    private EthereumClientConnector ethereumClientConnector;
    private WalletAccountService accountService;
    private WalletTransactionService transactionService;
    private SettingService settingService;
    private ListenerService listenerService;
    private long networkId;

    public EthereumBlockchainTransactionService(SettingService settingService, EthereumClientConnector ethereumClientConnector, WalletTransactionService transactionService, WalletAccountService accountService) {
        this.settingService = settingService;
        this.ethereumClientConnector = ethereumClientConnector;
        this.transactionService = transactionService;
        this.accountService = accountService;
    }

    public void start() {
        try {
            this.networkId = WalletUtils.getNetworkId();
            if (this.ethereumClientConnector.isPermanentlyScanBlockchain()) {
                this.startWatchingBlockchain(this.getLastWatchedBlockNumber());
            }
        }
        catch (Exception e) {
            LOG.error("Error while getting latest block number from blockchain with network id: {}", new Object[]{this.networkId, e});
        }
    }

    public void stop() {
    }

    public void checkPendingTransactions() {
        List pendingTransactions = this.transactionService.getPendingTransactions();
        if (pendingTransactions != null && !pendingTransactions.isEmpty()) {
            LOG.debug("Checking on blockchain the status of {} transactions marked as pending in database", new Object[]{pendingTransactions.size()});
            this.startWatchingBlockchain(this.getLastWatchedBlockNumber());
            for (TransactionDetail pendingTransactionDetail : pendingTransactions) {
                try {
                    this.checkTransactionStatusOnBlockchain(pendingTransactionDetail.getHash(), true);
                }
                catch (Exception e) {
                    LOG.warn("Error treating pending transaction: {}", new Object[]{pendingTransactionDetail, e});
                }
            }
        } else if (!this.ethereumClientConnector.isPermanentlyScanBlockchain()) {
            this.stopWatchingBlockchain();
        }
    }

    public void scanNewerBlocks() throws IOException {
        long lastWatchedBlockNumber;
        long lastEthereumBlockNumber = this.ethereumClientConnector.getLastestBlockNumber();
        if (lastEthereumBlockNumber <= (lastWatchedBlockNumber = this.getLastWatchedBlockNumber())) {
            LOG.debug("No new blocks to verify. last watched = {}. last blockchain block = {}", new Object[]{lastWatchedBlockNumber, lastEthereumBlockNumber});
            return;
        }
        String contractAddress = WalletUtils.getContractAddress();
        Set<String> transactionHashes = this.ethereumClientConnector.getContractTransactions(contractAddress, lastWatchedBlockNumber, lastEthereumBlockNumber);
        LOG.debug("{} transactions has been found on contract {} between block {} and {}", new Object[]{transactionHashes.size(), contractAddress, lastWatchedBlockNumber, lastEthereumBlockNumber});
        boolean processed = true;
        for (String transactionHash : transactionHashes) {
            TransactionDetail transaction = this.transactionService.getTransactionByHash(transactionHash);
            if (transaction != null && transaction.isSucceeded()) continue;
            try {
                this.checkTransactionStatusOnBlockchain(transactionHash, false);
            }
            catch (Exception e) {
                LOG.warn("Error processing transaction with hash: {}", new Object[]{transactionHash, e});
                processed = false;
            }
        }
        if (processed) {
            this.saveLastWatchedBlockNumber(lastEthereumBlockNumber);
        }
    }

    public void sendRawTransactions() {
        List transactions = this.transactionService.getTransactionsToSend();
        HashSet<String> walletAddressesWithTransactionsSent = new HashSet<String>();
        for (TransactionDetail transactionDetail : transactions) {
            String from = transactionDetail.getFrom();
            long sendingAttemptCount = transactionDetail.getSendingAttemptCount();
            if (sendingAttemptCount > this.transactionService.getMaxAttemptsToSend()) {
                transactionDetail.setPending(false);
                transactionDetail.setSucceeded(false);
                this.transactionService.saveTransactionDetail(transactionDetail, true);
                continue;
            }
            if (!transactionDetail.isBoost() && (walletAddressesWithTransactionsSent.contains(from) || !this.transactionService.canSendTransactionToBlockchain(from))) continue;
            walletAddressesWithTransactionsSent.add(from);
            try {
                CompletableFuture<EthSendTransaction> sendTransactionToBlockchain = this.ethereumClientConnector.sendTransactionToBlockchain(transactionDetail);
                sendTransactionToBlockchain.thenAcceptAsync(ethSendTransaction -> {
                    PortalContainer container = PortalContainer.getInstance();
                    ExoContainerContext.setCurrentContainer((ExoContainer)container);
                    RequestLifeCycle.begin((ExoContainer)container);
                    try {
                        this.handleTransactionSendingRequest(transactionDetail, (EthSendTransaction)ethSendTransaction, null);
                    }
                    finally {
                        RequestLifeCycle.end();
                    }
                });
                sendTransactionToBlockchain.exceptionally(throwable -> {
                    PortalContainer container = PortalContainer.getInstance();
                    ExoContainerContext.setCurrentContainer((ExoContainer)container);
                    RequestLifeCycle.begin((ExoContainer)container);
                    try {
                        this.handleTransactionSendingRequest(transactionDetail, null, (Throwable)throwable);
                    }
                    finally {
                        RequestLifeCycle.end();
                    }
                    return null;
                });
                transactionDetail.setSendingAttemptCount(sendingAttemptCount + 1L);
                transactionDetail.setSentTimestamp(System.currentTimeMillis());
                this.transactionService.saveTransactionDetail(transactionDetail, false);
                this.getListenerService().broadcast("exo.wallet.transaction.sent", (Object)transactionDetail, (Object)transactionDetail);
            }
            catch (Throwable e) {
                if (this.isIOException(e)) {
                    LOG.error("IO Error while sending transaction {} to blockchain", new Object[]{transactionDetail, e});
                    continue;
                }
                LOG.error("Error while sending transaction {} to blockchain", new Object[]{transactionDetail, e});
                transactionDetail.setSendingAttemptCount(sendingAttemptCount + 1L);
                this.transactionService.saveTransactionDetail(transactionDetail, false);
            }
        }
    }

    public void checkPendingTransactionValidity(TransactionDetail transactionDetail) {
        boolean transactionSendingTimedOut;
        if (transactionDetail == null || !transactionDetail.isPending()) {
            return;
        }
        long pendingTransactionMaxDays = this.transactionService.getPendingTransactionMaxDays();
        long maxAttemptsToSend = this.transactionService.getMaxAttemptsToSend();
        String transactionHash = transactionDetail.getHash();
        long sentTimestamp = transactionDetail.getRawTransaction() == null || transactionDetail.getSendingAttemptCount() == 0L ? transactionDetail.getTimestamp() : transactionDetail.getSentTimestamp();
        Duration duration = Duration.ofMillis(System.currentTimeMillis() - sentTimestamp);
        boolean bl = transactionSendingTimedOut = pendingTransactionMaxDays > 0L && duration.toDays() >= pendingTransactionMaxDays;
        if (transactionSendingTimedOut) {
            Transaction transaction = this.ethereumClientConnector.getTransaction(transactionHash);
            if (transaction == null) {
                if (transactionDetail.getRawTransaction() == null) {
                    transactionDetail.setPending(false);
                    transactionDetail.setSucceeded(false);
                    transactionDetail.setNonce(0L);
                    LOG.info("Transaction '{}' was NOT FOUND on blockchain for more than '{}' days, so mark it as failed", new Object[]{transactionHash, pendingTransactionMaxDays});
                    this.transactionService.saveTransactionDetail(transactionDetail, true);
                    return;
                }
                if (transactionDetail.getSendingAttemptCount() <= maxAttemptsToSend) {
                    LOG.info("Transaction '{}' was NOT FOUND on blockchain for more than '{}' days, it will be resent again", new Object[]{transactionHash, duration.toDays()});
                    transactionDetail.setSentTimestamp(0L);
                    this.transactionService.saveTransactionDetail(transactionDetail, false);
                    return;
                }
                if (transactionDetail.getSendingAttemptCount() > maxAttemptsToSend) {
                    transactionDetail.setPending(false);
                    transactionDetail.setSucceeded(false);
                    this.transactionService.saveTransactionDetail(transactionDetail, false);
                }
            } else {
                LOG.info("Transaction '{}' was FOUND on blockchain for more than '{}' days, so avoid marking it as failed", new Object[]{transactionHash, pendingTransactionMaxDays});
                this.checkTransactionStatusOnBlockchain(transactionHash, transaction, true);
                return;
            }
        }
        if (transactionDetail.isPending() && transactionDetail.getSentTimestamp() == 0L && transactionDetail.getSendingAttemptCount() > maxAttemptsToSend) {
            transactionDetail.setPending(false);
            transactionDetail.setSucceeded(false);
            transactionDetail.setNonce(0L);
            LOG.info("Transaction '{}' was NOT FOUND on blockchain after attempting sending it for '{}' times, so mark it as failed", new Object[]{transactionHash, pendingTransactionMaxDays});
            this.transactionService.saveTransactionDetail(transactionDetail, true);
            return;
        }
        if (transactionDetail.isPending()) {
            long nonce = transactionDetail.getNonce();
            BigInteger lastMinedTransactionNonce = null;
            try {
                lastMinedTransactionNonce = this.ethereumClientConnector.getNonce(transactionDetail.getFrom(), DefaultBlockParameterName.LATEST);
            }
            catch (IOException e) {
                LOG.warn("Error retrieving last nonce of {}", new Object[]{transactionDetail.getFrom(), e});
                return;
            }
            if (lastMinedTransactionNonce != null && nonce < lastMinedTransactionNonce.longValue()) {
                transactionDetail.setPending(false);
                LOG.info("Transaction '{}' was sent with same nonce as previous mined transaction, so mark it as not pending anymore", new Object[]{transactionHash});
                this.transactionService.saveTransactionDetail(transactionDetail, true);
            }
        }
    }

    public void checkTransactionStatusOnBlockchain(String transactionHash, boolean pendingTransactionFromDatabase) {
        Transaction transaction = this.ethereumClientConnector.getTransaction(transactionHash);
        this.checkTransactionStatusOnBlockchain(transactionHash, transaction, pendingTransactionFromDatabase);
    }

    public TransactionDetail refreshTransactionFromBlockchain(String hash) {
        TransactionDetail transactionDetail = this.transactionService.getTransactionByHash(hash);
        if (transactionDetail == null) {
            return null;
        }
        this.checkTransactionStatusOnBlockchain(hash, true);
        return this.transactionService.getTransactionByHash(hash);
    }

    public long refreshBlockchainGasPrice() throws IOException {
        return this.ethereumClientConnector.getGasPrice().longValue();
    }

    private void stopWatchingBlockchain() {
        this.ethereumClientConnector.stopListeningToBlockchain();
    }

    private void startWatchingBlockchain(long lastWatchedBlockNumber) {
        try {
            if (!this.ethereumClientConnector.isListeningToBlockchain()) {
                if (lastWatchedBlockNumber <= 0L) {
                    lastWatchedBlockNumber = this.ethereumClientConnector.getLastestBlockNumber();
                    this.saveLastWatchedBlockNumber(lastWatchedBlockNumber);
                }
                this.ethereumClientConnector.renewBlockSubscription(lastWatchedBlockNumber + 1L);
            }
        }
        catch (Exception e) {
            throw new IllegalStateException("Error watching blockchain starting from block " + lastWatchedBlockNumber, e);
        }
    }

    private void checkTransactionStatusOnBlockchain(String transactionHash, Transaction transaction, boolean pendingTransactionFromDatabase) {
        boolean broadcastMinedTransaction;
        TransactionReceipt transactionReceipt;
        if (transaction == null) {
            TransactionDetail transactionDetail;
            if (pendingTransactionFromDatabase) {
                transactionDetail = this.transactionService.getPendingTransactionByHash(transactionHash);
                if (transactionDetail == null) {
                    throw new IllegalStateException("Transaction with hash " + transactionHash + " wasn't found in internal database while it should have been retrieved from it.");
                }
            } else {
                throw new IllegalStateException("Transaction with hash " + transactionHash + " is not marked as pending but the transaction wasn't found on blockchain");
            }
            LOG.debug("Transaction {} is marked as pending in database and is not yet found on blockchain", new Object[]{transactionHash});
            this.checkPendingTransactionValidity(transactionDetail);
            return;
        }
        String blockHash = transaction.getBlockHash();
        if (StringUtils.isBlank((CharSequence)blockHash) || StringUtils.equalsIgnoreCase((CharSequence)"0x0000000000000000000000000000000000000000000000000000000000000000", (CharSequence)blockHash) || transaction.getBlockNumber() == null) {
            if (!pendingTransactionFromDatabase) {
                throw new IllegalStateException("Transaction " + transactionHash + " is marked as pending in blockchain while it's not marked as pending in database");
            }
            LOG.debug("Transaction {} is marked as pending in database and is always pending on blockchain", new Object[]{transactionHash});
            return;
        }
        ContractDetail contractDetail = WalletUtils.getContractDetail();
        if (contractDetail == null) {
            throw new IllegalStateException("Principal contract detail wasn't found in database");
        }
        TransactionDetail transactionDetail = this.transactionService.getTransactionByHash(transactionHash);
        if (transactionDetail == null) {
            if (pendingTransactionFromDatabase) {
                throw new IllegalStateException("Transaction with hash " + transactionHash + " wasn't found in internal database while it should have been retrieved from it.");
            }
            String contractAddress = transaction.getTo();
            if (!StringUtils.equalsIgnoreCase((CharSequence)contractDetail.getAddress(), (CharSequence)contractAddress)) {
                LOG.debug("Transaction '{}' is not a contract transaction, thus it will not be added into database", new Object[]{transactionHash});
                return;
            }
            transactionDetail = new TransactionDetail();
            transactionDetail.setNetworkId(this.networkId);
            transactionDetail.setHash(transactionHash);
            transactionDetail.setPending(true);
        }
        if ((transactionReceipt = this.ethereumClientConnector.getTransactionReceipt(transactionHash)) == null) {
            throw new IllegalStateException("Couldn't find transaction receipt with hash '" + transactionHash + "' on blockchain");
        }
        boolean bl = broadcastMinedTransaction = transactionDetail.isPending() || transactionDetail.isSucceeded() != transactionReceipt.isStatusOK();
        if (pendingTransactionFromDatabase && !transactionDetail.isPending()) {
            long transactionsWithSameNonce = this.transactionService.countTransactionsByNonce(transactionDetail);
            if (transactionsWithSameNonce > 1L) {
                this.transactionService.cancelTransactionsWithSameNonce(transactionDetail);
            } else {
                LOG.debug("Transaction '{}' seems to be already marked as not pending, skip processing it", new Object[]{transactionHash});
                return;
            }
        }
        this.computeTransactionDetail(transactionDetail, contractDetail, transaction, transactionReceipt);
        if (pendingTransactionFromDatabase) {
            this.transactionService.saveTransactionDetail(transactionDetail, broadcastMinedTransaction);
        } else {
            if (StringUtils.isNotBlank((CharSequence)transactionDetail.getFrom()) && WalletUtils.isWalletEmpty((Wallet)transactionDetail.getFromWallet())) {
                transactionDetail.setFromWallet(this.accountService.getWalletByAddress(transactionDetail.getFrom()));
            }
            if (StringUtils.isNotBlank((CharSequence)transactionDetail.getTo()) && WalletUtils.isWalletEmpty((Wallet)transactionDetail.getToWallet())) {
                transactionDetail.setToWallet(this.accountService.getWalletByAddress(transactionDetail.getTo()));
            }
            if (StringUtils.isNotBlank((CharSequence)transactionDetail.getBy()) && WalletUtils.isWalletEmpty((Wallet)transactionDetail.getByWallet())) {
                transactionDetail.setByWallet(this.accountService.getWalletByAddress(transactionDetail.getBy()));
            }
            if (WalletUtils.hasKnownWalletInTransaction((TransactionDetail)transactionDetail)) {
                this.transactionService.saveTransactionDetail(transactionDetail, broadcastMinedTransaction);
            } else if (this.transactionService.isLogAllTransaction()) {
                WalletUtils.logStatistics((TransactionDetail)transactionDetail);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean handleTransactionSendingRequest(TransactionDetail transactionDetail, EthSendTransaction transaction, Throwable exception) {
        boolean transactionModified = false;
        String transactionHash = transactionDetail.getHash();
        LOG.debug("Handle transaction sending return: tx hash = {} , eth tx = {}, exception = {}", new Object[]{transactionHash, transaction, exception});
        if (this.isIOException(exception)) {
            LOG.warn("IO Error when sending transaction {}", new Object[]{transactionHash, exception});
            return transactionModified;
        }
        try {
            boolean hasError;
            if (transaction != null && transaction.getTransactionHash() != null && !StringUtils.equalsIgnoreCase((CharSequence)transaction.getTransactionHash(), (CharSequence)transactionHash)) {
                LOG.debug("Transaction with hash {} has been confirmed with different hash {}", new Object[]{transactionHash, transaction.getTransactionHash()});
                transactionHash = transaction.getTransactionHash();
                transactionDetail.setHash(transactionHash);
                transactionModified = true;
            }
            Response.Error transactionError = transaction == null ? null : transaction.getError();
            boolean bl = hasError = exception != null || transaction != null && (transactionError != null || StringUtils.isBlank((CharSequence)((CharSequence)transaction.getResult())));
            if (hasError) {
                String exceptionMessage = exception == null ? "No result returned" : exception.getMessage();
                String errorMessage = transaction == null || transactionError == null ? exceptionMessage : this.getTransactionErrorMessage(transactionError);
                LOG.warn("Error when sending transaction {} - {}.", new Object[]{transactionDetail, errorMessage, exception});
                Transaction transactionFromBlockchain = this.ethereumClientConnector.getTransaction(transactionHash);
                if (transactionFromBlockchain == null) {
                    TransactionReceipt receipt = this.ethereumClientConnector.getTransactionReceipt(transactionHash);
                    transactionDetail.setPending(receipt == null && transactionError == null);
                    transactionDetail.setSucceeded(receipt != null && receipt.isStatusOK());
                    transactionModified = true;
                }
            }
        }
        finally {
            if (transactionModified) {
                this.transactionService.saveTransactionDetail(transactionDetail, false);
            }
        }
        return transactionModified;
    }

    private void computeTransactionDetail(TransactionDetail transactionDetail, ContractDetail contractDetail, Transaction transaction, TransactionReceipt transactionReceipt) {
        String receiverAddress;
        String contractAddress;
        boolean isContractTransaction;
        transactionDetail.setFrom(transaction.getFrom());
        transactionDetail.setSucceeded(transactionReceipt.isStatusOK());
        transactionDetail.setGasUsed(transactionReceipt.getGasUsed().intValue());
        transactionDetail.setGasPrice(transaction.getGasPrice().doubleValue());
        transactionDetail.setPending(false);
        transactionDetail.setNonce(transaction.getNonce().longValue());
        if (transactionDetail.getTimestamp() <= 0L) {
            transactionDetail.setTimestamp(System.currentTimeMillis());
        }
        if (transaction.getValue().compareTo(BigInteger.ZERO) >= 0) {
            BigInteger weiAmount = transaction.getValue();
            transactionDetail.setValueDecimal(weiAmount, 18);
        }
        if (!(isContractTransaction = StringUtils.equalsIgnoreCase((CharSequence)(contractAddress = contractDetail.getAddress()), (CharSequence)(receiverAddress = transaction.getTo())))) {
            transactionDetail.setTo(receiverAddress);
            transactionDetail.setTokenFee(0.0);
            transactionDetail.setEtherFee(0.0);
            transactionDetail.setContractAddress(null);
            transactionDetail.setContractMethodName(null);
            transactionDetail.setContractAmount(0.0);
            transactionDetail.setNoContractFunds(false);
            return;
        }
        transactionDetail.setContractAddress(contractAddress);
        if (!transactionReceipt.isStatusOK()) {
            return;
        }
        String hash = transactionDetail.getHash();
        List logs = transactionReceipt.getLogs();
        if (logs != null && !logs.isEmpty()) {
            Integer contractDecimals = contractDetail.getDecimals();
            int logsSize = logs.size();
            LOG.debug("Retrieving information from blockchain for transaction {} with {} LOGS", new Object[]{hash, logsSize});
            int i = 0;
            boolean transactionLogTreated = false;
            while (i < logsSize) {
                BigInteger amount;
                EventValues parameters;
                String methodName;
                org.web3j.protocol.core.methods.response.Log log;
                List topics;
                if ((topics = (log = (org.web3j.protocol.core.methods.response.Log)logs.get(i++)).getTopics()) == null || topics.isEmpty()) {
                    LOG.warn("Transaction {} has NO topics", new Object[]{hash});
                    transactionDetail.setSucceeded(false);
                    continue;
                }
                String topic = (String)topics.get(0);
                LOG.debug("Treating transaction log {} with {} topics", new Object[]{hash, topics.size()});
                if (transactionLogTreated || StringUtils.isBlank((CharSequence)(methodName = CONTRACT_METHODS_BY_SIG.get(topic)))) continue;
                transactionDetail.setContractMethodName(methodName);
                if (StringUtils.equals((CharSequence)methodName, (CharSequence)"transfer")) {
                    parameters = Contract.staticExtractEventParameters((Event)MeedsToken.TRANSFER_EVENT, (org.web3j.protocol.core.methods.response.Log)log);
                    if (parameters == null) continue;
                    transactionDetail.setFrom(((Type)parameters.getIndexedValues().get(0)).getValue().toString());
                    transactionDetail.setTo(((Type)parameters.getIndexedValues().get(1)).getValue().toString());
                    amount = (BigInteger)((Type)parameters.getNonIndexedValues().get(0)).getValue();
                    transactionDetail.setContractAmountDecimal(amount, contractDecimals.intValue());
                    if (!StringUtils.equals((CharSequence)transactionReceipt.getFrom(), (CharSequence)transactionDetail.getFrom())) {
                        transactionDetail.setBy(transactionReceipt.getFrom());
                        transactionDetail.setContractMethodName("transferFrom");
                    }
                    transactionDetail.setAdminOperation(false);
                    continue;
                }
                if (StringUtils.equals((CharSequence)methodName, (CharSequence)"approve")) {
                    transactionLogTreated = true;
                    parameters = Contract.staticExtractEventParameters((Event)MeedsToken.APPROVAL_EVENT, (org.web3j.protocol.core.methods.response.Log)log);
                    if (parameters == null) continue;
                    transactionDetail.setFrom(((Type)parameters.getIndexedValues().get(0)).getValue().toString());
                    transactionDetail.setTo(((Type)parameters.getIndexedValues().get(1)).getValue().toString());
                    amount = (BigInteger)((Type)parameters.getNonIndexedValues().get(0)).getValue();
                    transactionDetail.setContractAmountDecimal(amount, contractDecimals.intValue());
                    transactionDetail.setAdminOperation(false);
                    continue;
                }
                if (StringUtils.equals((CharSequence)methodName, (CharSequence)"transferOwnership")) {
                    transactionLogTreated = true;
                    parameters = Contract.staticExtractEventParameters((Event)MeedsToken.OWNERSHIPTRANSFERRED_EVENT, (org.web3j.protocol.core.methods.response.Log)log);
                    if (parameters == null) continue;
                    transactionDetail.setTo(((Type)parameters.getNonIndexedValues().get(0)).getValue().toString());
                    transactionDetail.setAdminOperation(true);
                    continue;
                }
                if (transactionLogTreated || i + 1 != logsSize) continue;
                LOG.warn("Can't find contract method name of transaction {}", new Object[]{transactionDetail});
            }
        }
    }

    private long getLastWatchedBlockNumber() {
        SettingValue lastBlockNumberValue = this.settingService.get(WalletUtils.WALLET_CONTEXT, WalletUtils.WALLET_SCOPE, "ADDONS_ETHEREUM_LAST_BLOCK_NUMBER" + this.networkId);
        if (lastBlockNumberValue != null && lastBlockNumberValue.getValue() != null) {
            return Long.parseLong(lastBlockNumberValue.getValue().toString());
        }
        return 0L;
    }

    private void saveLastWatchedBlockNumber(long lastWatchedBlockNumber) {
        LOG.debug("Save watched block number {} on network {}", new Object[]{lastWatchedBlockNumber, this.networkId});
        this.settingService.set(WalletUtils.WALLET_CONTEXT, WalletUtils.WALLET_SCOPE, "ADDONS_ETHEREUM_LAST_BLOCK_NUMBER" + this.networkId, SettingValue.create((Long)lastWatchedBlockNumber));
    }

    private String getTransactionErrorMessage(Response.Error transactionError) {
        if (transactionError == null) {
            return null;
        }
        return "Code: " + transactionError.getCode() + ", Message: " + transactionError.getMessage() + ", Data: " + transactionError.getData();
    }

    private ListenerService getListenerService() {
        if (this.listenerService == null) {
            this.listenerService = (ListenerService)CommonsUtils.getService(ListenerService.class);
        }
        return this.listenerService;
    }

    private boolean isIOException(Throwable exception) {
        if (exception == null) {
            return false;
        }
        if (exception instanceof IOException) {
            return true;
        }
        if (exception.getCause() == exception) {
            return false;
        }
        return this.isIOException(exception.getCause());
    }

    static {
        CONTRACT_METHODS_BY_SIG.put(TRANSFER_SIG, "transfer");
        CONTRACT_METHODS_BY_SIG.put(APPROVAL_SIG, "approve");
        CONTRACT_METHODS_BY_SIG.put(TRANSFER_OWNERSHIP_SIG, "transferOwnership");
    }
}

