/*
 * Decompiled with CFR 0.152.
 */
package org.exoplatform.wallet.reward.service;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.wallet.model.ContractDetail;
import org.exoplatform.wallet.model.Wallet;
import org.exoplatform.wallet.model.WalletType;
import org.exoplatform.wallet.model.reward.RewardBudgetType;
import org.exoplatform.wallet.model.reward.RewardPeriod;
import org.exoplatform.wallet.model.reward.RewardPeriodType;
import org.exoplatform.wallet.model.reward.RewardPluginSettings;
import org.exoplatform.wallet.model.reward.RewardReport;
import org.exoplatform.wallet.model.reward.RewardSettings;
import org.exoplatform.wallet.model.reward.RewardStatus;
import org.exoplatform.wallet.model.reward.RewardTeam;
import org.exoplatform.wallet.model.reward.RewardTeamMember;
import org.exoplatform.wallet.model.reward.WalletPluginReward;
import org.exoplatform.wallet.model.reward.WalletReward;
import org.exoplatform.wallet.model.transaction.TransactionDetail;
import org.exoplatform.wallet.reward.api.RewardPlugin;
import org.exoplatform.wallet.reward.service.RewardReportService;
import org.exoplatform.wallet.reward.service.RewardSettingsService;
import org.exoplatform.wallet.reward.service.RewardTeamService;
import org.exoplatform.wallet.reward.storage.RewardReportStorage;
import org.exoplatform.wallet.service.WalletAccountService;
import org.exoplatform.wallet.service.WalletTokenAdminService;
import org.exoplatform.wallet.utils.RewardUtils;
import org.exoplatform.wallet.utils.WalletUtils;

public class WalletRewardReportService
implements RewardReportService {
    private static final Log LOG = ExoLogger.getLogger(WalletRewardReportService.class);
    private final WalletAccountService walletAccountService;
    private WalletTokenAdminService walletTokenAdminService;
    private final RewardSettingsService rewardSettingsService;
    private final RewardTeamService rewardTeamService;
    private final RewardReportStorage rewardReportStorage;

    public WalletRewardReportService(WalletAccountService walletAccountService, RewardSettingsService rewardSettingsService, RewardTeamService rewardTeamService, RewardReportStorage rewardReportStorage) {
        this.walletAccountService = walletAccountService;
        this.rewardSettingsService = rewardSettingsService;
        this.rewardTeamService = rewardTeamService;
        this.rewardReportStorage = rewardReportStorage;
    }

    public void sendRewards(long periodDateInSeconds, String username) throws Exception {
        if (!WalletUtils.isUserRewardingAdmin((String)username)) {
            throw new IllegalAccessException("User " + username + " is not allowed to send rewards");
        }
        RewardReport rewardReport = this.computeRewards(periodDateInSeconds);
        if (rewardReport.getPeriod().getEndDateInSeconds() > System.currentTimeMillis() / 1000L) {
            throw new IllegalStateException("Can't send rewards for current period");
        }
        if (rewardReport == null || rewardReport.getRewards() == null || rewardReport.getRewards().isEmpty()) {
            return;
        }
        if (rewardReport.getPendingTransactionCount() > 0L) {
            String startDateFormatted = rewardReport.getPeriod().getStartDateFormatted(Locale.getDefault().getLanguage());
            String endDateFormatted = rewardReport.getPeriod().getEndDateFormatted(Locale.getDefault().getLanguage());
            throw new IllegalStateException("There are some pending transactions for rewards of period between " + startDateFormatted + " and " + endDateFormatted + ", thus no reward sending is allowed until the transactions finishes");
        }
        String adminWalletAddress = this.getTokenAdminService().getAdminWalletAddress();
        if (StringUtils.isBlank((String)adminWalletAddress)) {
            throw new IllegalStateException("No admin wallet is configured");
        }
        HashSet rewards = new HashSet(rewardReport.getRewards());
        Iterator rewardedWalletsIterator = rewards.iterator();
        while (rewardedWalletsIterator.hasNext()) {
            WalletReward walletReward = (WalletReward)rewardedWalletsIterator.next();
            if (walletReward == null || !walletReward.isEnabled() || walletReward.getRewards() == null || walletReward.getTokensToSend() == 0.0 || walletReward.getTransaction() != null && (walletReward.getTransaction().isPending() || walletReward.getTransaction().isSucceeded())) {
                rewardedWalletsIterator.remove();
                continue;
            }
            if (walletReward.getTokensToSend() < 0.0) {
                throw new IllegalStateException("Can't send reward transaction for wallet of " + walletReward.getWallet().getType() + " " + walletReward.getWallet().getId() + " with a negative amount" + walletReward.getTokensToSend());
            }
            if (!StringUtils.equals((String)walletReward.getStatus(), (String)"success") && !StringUtils.equals((String)walletReward.getStatus(), (String)"pending")) continue;
            rewardedWalletsIterator.remove();
        }
        if (rewards.isEmpty()) {
            throw new IllegalStateException("No rewards to send for selected period");
        }
        ContractDetail contractDetail = WalletUtils.getContractDetail();
        BigInteger adminTokenBalance = this.getTokenAdminService().balanceOf(adminWalletAddress);
        double adminBalance = WalletUtils.convertFromDecimals((BigInteger)adminTokenBalance, (int)contractDetail.getDecimals());
        double rewardsAmount = rewardReport.getRemainingTokensToSend();
        if (rewardsAmount > adminBalance) {
            throw new IllegalStateException("Admin doesn't have enough funds to send rewards");
        }
        RewardPeriod rewardPeriod = rewardReport.getPeriod();
        for (WalletReward walletReward : rewards) {
            try {
                TransactionDetail transactionDetail = new TransactionDetail();
                transactionDetail.setFrom(adminWalletAddress);
                transactionDetail.setTo(walletReward.getWallet().getAddress());
                transactionDetail.setContractAmount(walletReward.getTokensToSend());
                transactionDetail.setValue(walletReward.getTokensToSend());
                String transactionLabel = this.getTransactionLabel(walletReward, contractDetail, rewardPeriod);
                transactionDetail.setLabel(transactionLabel);
                String transactionMessage = this.getTransactionMessage(walletReward, contractDetail, rewardPeriod);
                transactionDetail.setMessage(transactionMessage);
                transactionDetail = this.getTokenAdminService().reward(transactionDetail, username);
                walletReward.setTransaction(transactionDetail);
            }
            catch (Exception e) {
                LOG.warn("Error while sending reward transaction for user '{}'", new Object[]{walletReward.getWallet().getName(), e});
            }
        }
        this.rewardReportStorage.saveRewardReport(rewardReport);
    }

    public RewardReport computeRewards(long periodDateInSeconds) {
        if (periodDateInSeconds == 0L) {
            throw new IllegalArgumentException("periodDate is mandatory");
        }
        RewardReport rewardReport = this.getRewardReport(periodDateInSeconds);
        if (rewardReport == null) {
            rewardReport = new RewardReport();
            rewardReport.setPeriod(this.getRewardPeriod(periodDateInSeconds));
        }
        Set<Wallet> wallets = this.walletAccountService.listWallets().stream().filter(wallet -> WalletType.isUser((String)wallet.getType())).collect(Collectors.toSet());
        this.computeRewardDetails(rewardReport, wallets);
        return rewardReport;
    }

    public RewardReport getRewardReport(long periodTimeInSeconds) {
        RewardSettings rewardSettings = this.rewardSettingsService.getSettings();
        if (rewardSettings == null) {
            throw new IllegalStateException("Error computing rewards using empty settings");
        }
        if (rewardSettings.getPeriodType() == null) {
            throw new IllegalStateException("Error computing rewards using empty period type");
        }
        return this.rewardReportStorage.getRewardReport(rewardSettings.getPeriodType(), periodTimeInSeconds);
    }

    public void saveRewardReport(RewardReport rewardReport) {
        if (rewardReport == null) {
            throw new IllegalArgumentException("Reward report to save is null");
        }
        this.rewardReportStorage.saveRewardReport(rewardReport);
    }

    public List<RewardPeriod> getRewardPeriodsInProgress() {
        return this.rewardReportStorage.findRewardPeriodsByStatus(RewardStatus.PENDING);
    }

    public List<RewardPeriod> getRewardPeriodsNotSent() {
        return this.rewardReportStorage.findRewardPeriodsByStatus(RewardStatus.ESTIMATION);
    }

    public List<WalletReward> listRewards(String currentUser, int limit) {
        Identity identity = WalletUtils.getIdentityByTypeAndId((WalletType)WalletType.USER, (String)currentUser);
        if (identity == null) {
            return Collections.emptyList();
        }
        return this.rewardReportStorage.listRewards(Long.parseLong(identity.getId()), limit);
    }

    public double countRewards(String currentUser) {
        Identity identity = WalletUtils.getIdentityByTypeAndId((WalletType)WalletType.USER, (String)currentUser);
        if (identity == null) {
            throw new IllegalArgumentException("can't get user");
        }
        return this.rewardReportStorage.countRewards(Long.parseLong(identity.getId()));
    }

    public void replaceRewardTransactions(String oldHash, String newHash) {
        this.rewardReportStorage.replaceRewardTransactions(oldHash, newHash);
    }

    private RewardPeriod getRewardPeriod(long periodDateInSeconds) {
        RewardSettings rewardSettings = this.rewardSettingsService.getSettings();
        RewardPeriodType periodType = rewardSettings.getPeriodType();
        return periodType.getPeriodOfTime(RewardUtils.timeFromSeconds((long)periodDateInSeconds));
    }

    private void computeRewardDetails(RewardReport rewardReport, Set<Wallet> wallets) {
        Map<RewardPlugin, RewardPluginSettings> rewardPlugins = this.getEnabledRewardPlugins();
        RewardPeriod period = rewardReport.getPeriod();
        Set<WalletReward> walletRewards = this.retrieveWalletRewards(rewardReport, wallets);
        Set enabledRewards = walletRewards.stream().filter(WalletReward::isEnabled).collect(Collectors.toSet());
        Set<WalletReward> enabledTeamRewards = enabledRewards.stream().filter(wr -> wr.getTeam() == null || !wr.getTeam().isDisabled()).collect(Collectors.toSet());
        HashSet<WalletPluginReward> walletRewardsByPlugin = new HashSet<WalletPluginReward>();
        for (Map.Entry<RewardPlugin, RewardPluginSettings> pluginEntry : rewardPlugins.entrySet()) {
            RewardPlugin plugin = pluginEntry.getKey();
            RewardPluginSettings pluginSettings = pluginEntry.getValue();
            Set identityIds = walletRewards.stream().map(wr -> wr.getIdentityId()).collect(Collectors.toSet());
            Map earnedPoints = plugin.getEarnedPoints(identityIds, period.getStartDateInSeconds(), period.getEndDateInSeconds());
            Set<WalletReward> validWalletRewards = pluginSettings.isUsePools() ? enabledTeamRewards : enabledRewards;
            this.computeReward(pluginSettings, earnedPoints, validWalletRewards, walletRewardsByPlugin);
        }
        for (WalletReward walletReward : walletRewards) {
            Set rewardDetails = walletRewardsByPlugin.stream().filter(rewardByPlugin -> rewardByPlugin.getIdentityId() == walletReward.getIdentityId()).collect(Collectors.toSet());
            walletReward.setRewards(rewardDetails);
        }
    }

    private Set<WalletReward> retrieveWalletRewards(RewardReport rewardReport, Set<Wallet> wallets) {
        HashSet<WalletReward> walletRewards = rewardReport.getRewards();
        if (walletRewards == null) {
            walletRewards = new HashSet<WalletReward>();
            rewardReport.setRewards(walletRewards);
        }
        boolean completelyProceeded = rewardReport.isCompletelyProceeded();
        for (Wallet wallet : wallets) {
            WalletReward walletReward = walletRewards.stream().filter(wr -> wallet != null && wr.getWallet() != null && wr.getIdentityId() == wallet.getTechnicalId()).findFirst().orElse(null);
            if (walletReward == null) {
                walletReward = new WalletReward();
                walletRewards.add(walletReward);
            }
            walletReward.setWallet(wallet);
            if (completelyProceeded) continue;
            List rewardTeams = this.rewardTeamService.findTeamsByMemberId(walletReward.getIdentityId());
            walletReward.setTeams(rewardTeams);
        }
        return walletRewards;
    }

    private Map<RewardPlugin, RewardPluginSettings> getEnabledRewardPlugins() {
        RewardSettings rewardSettings = this.rewardSettingsService.getSettings();
        Set pluginSettings = rewardSettings.getPluginSettings().stream().filter(RewardPluginSettings::isEnabled).collect(Collectors.toSet());
        Collection rewardPlugins = this.rewardSettingsService.getRewardPlugins();
        return rewardPlugins.stream().filter(RewardPlugin::isEnabled).filter(rewardPlugin -> this.getPluginSetting(pluginSettings, rewardPlugin.getPluginId()) != null).collect(Collectors.toMap(Function.identity(), rewardPlugin -> this.getPluginSetting(pluginSettings, rewardPlugin.getPluginId())));
    }

    private RewardPluginSettings getPluginSetting(Set<RewardPluginSettings> pluginSettings, String pluginId) {
        for (RewardPluginSettings rewardPluginSettings : pluginSettings) {
            if (!StringUtils.equals((String)pluginId, (String)rewardPluginSettings.getPluginId())) continue;
            return rewardPluginSettings;
        }
        return null;
    }

    private void computeReward(RewardPluginSettings rewardPluginSettings, Map<Long, Double> earnedPoints, Set<WalletReward> enabledRewards, Set<WalletPluginReward> rewardMemberDetails) {
        RewardBudgetType budgetType = rewardPluginSettings.getBudgetType();
        if (budgetType == null) {
            LOG.warn("Budget type of reward plugin {} is empty, thus no computing is possible", new Object[]{rewardPluginSettings.getPluginId()});
            return;
        }
        String pluginId = rewardPluginSettings.getPluginId();
        double configuredPluginAmount = rewardPluginSettings.getAmount();
        if (configuredPluginAmount < 0.0) {
            throw new IllegalStateException("Plugin " + pluginId + " has a configured negative reward amount (" + configuredPluginAmount + ")");
        }
        Set<Long> validIdentityIdsToUse = enabledRewards.stream().map(WalletReward::getIdentityId).collect(Collectors.toSet());
        this.filterElligibleMembers(earnedPoints.entrySet(), validIdentityIdsToUse, rewardPluginSettings, rewardMemberDetails);
        double amountPerPoint = 0.0;
        double totalFixedBudget = 0.0;
        switch (budgetType) {
            case FIXED_PER_POINT: {
                amountPerPoint = configuredPluginAmount;
                this.addRewardsSwitchPointAmount(rewardMemberDetails, earnedPoints.entrySet(), pluginId, amountPerPoint);
                break;
            }
            case FIXED: {
                totalFixedBudget = configuredPluginAmount;
                this.addTeamMembersReward(rewardPluginSettings, earnedPoints, totalFixedBudget, rewardMemberDetails);
                break;
            }
            case FIXED_PER_MEMBER: {
                double budgetPerMember = configuredPluginAmount;
                int totalElligibleMembersCount = earnedPoints.size();
                totalFixedBudget = budgetPerMember * (double)totalElligibleMembersCount;
                this.addTeamMembersReward(rewardPluginSettings, earnedPoints, totalFixedBudget, rewardMemberDetails);
                break;
            }
            default: {
                throw new IllegalStateException("Budget type is not recognized in plugin settings: " + pluginId + ", budget type = " + budgetType);
            }
        }
    }

    private void addTeamMembersReward(RewardPluginSettings rewardPluginSettings, Map<Long, Double> earnedPoints, double totalFixedBudget, Set<WalletPluginReward> rewardMemberDetails) {
        if (totalFixedBudget <= 0.0) {
            return;
        }
        if (rewardPluginSettings.isUsePools()) {
            List teams = this.rewardTeamService.getTeams();
            Set<Long> identityIds = this.filterEligibleMembersAndTeams(teams, earnedPoints);
            this.buildNoPoolUsers(earnedPoints, teams, identityIds);
            this.computeTeamsMembersBudget(rewardPluginSettings.getPluginId(), teams, totalFixedBudget, rewardMemberDetails, earnedPoints);
        } else {
            double totalPoints = earnedPoints.entrySet().stream().collect(Collectors.summingDouble(entry -> (Double)entry.getValue()));
            if (totalPoints <= 0.0 || totalFixedBudget <= 0.0) {
                return;
            }
            double amountPerPoint = totalFixedBudget / totalPoints;
            this.addRewardsSwitchPointAmount(rewardMemberDetails, earnedPoints.entrySet(), rewardPluginSettings.getPluginId(), amountPerPoint);
        }
    }

    private void addRewardsSwitchPointAmount(Set<WalletPluginReward> rewardMemberDetails, Set<Map.Entry<Long, Double>> identitiesPointsEntries, String pluginId, double amountPerPoint) {
        for (Map.Entry<Long, Double> identitiyPointsEntry : identitiesPointsEntries) {
            Long identityId = identitiyPointsEntry.getKey();
            Double points = identitiyPointsEntry.getValue();
            double amount = points * amountPerPoint;
            WalletPluginReward rewardMemberDetail = new WalletPluginReward();
            rewardMemberDetail.setIdentityId(identityId.longValue());
            rewardMemberDetail.setPluginId(pluginId);
            rewardMemberDetail.setPoints(points.doubleValue());
            rewardMemberDetail.setAmount(amount);
            rewardMemberDetail.setPoolsUsed(false);
            rewardMemberDetails.add(rewardMemberDetail);
        }
    }

    private void filterElligibleMembers(Set<Map.Entry<Long, Double>> identitiesPointsEntries, Set<Long> validIdentityIdsToUse, RewardPluginSettings rewardPluginSettings, Set<WalletPluginReward> rewardMemberDetails) {
        String pluginId = rewardPluginSettings.getPluginId();
        double threshold = rewardPluginSettings.getThreshold();
        Iterator<Map.Entry<Long, Double>> identitiesPointsIterator = identitiesPointsEntries.iterator();
        while (identitiesPointsIterator.hasNext()) {
            Map.Entry<Long, Double> entry = identitiesPointsIterator.next();
            Long identityId = entry.getKey();
            Double points = entry.getValue();
            points = points == null ? 0.0 : points;
            if (points < 0.0) {
                throw new IllegalStateException("Plugin with id " + pluginId + " has assigned a negative points (" + points + ") to user with id " + identityId);
            }
            if (!(points < threshold) && points != 0.0 && validIdentityIdsToUse.contains(identityId)) continue;
            identitiesPointsIterator.remove();
            if (!(points > 0.0)) continue;
            WalletPluginReward rewardMemberDetail = new WalletPluginReward();
            rewardMemberDetail.setIdentityId(identityId.longValue());
            rewardMemberDetail.setPluginId(pluginId);
            rewardMemberDetail.setPoints(points.doubleValue());
            rewardMemberDetail.setAmount(0.0);
            rewardMemberDetail.setPoolsUsed(rewardPluginSettings.isUsePools());
            rewardMemberDetails.add(rewardMemberDetail);
        }
    }

    private void computeTeamsMembersBudget(String pluginId, List<RewardTeam> teams, double totalTeamsBudget, Set<WalletPluginReward> rewardMemberDetails, Map<Long, Double> earnedPoints) {
        double totalFixedTeamsBudget = 0.0;
        double computedRecipientsCount = 0.0;
        ArrayList<RewardTeam> computedBudgetTeams = new ArrayList<RewardTeam>();
        HashMap<Long, Double> totalPointsPerTeam = new HashMap<Long, Double>();
        for (RewardTeam rewardTeam2 : teams) {
            double totalTeamBudget;
            RewardBudgetType teamBudgetType = rewardTeam2.getRewardType();
            if (rewardTeam2.getMembers() == null || rewardTeam2.getMembers().isEmpty()) continue;
            double totalTeamPoints = rewardTeam2.getMembers().stream().collect(Collectors.summingDouble(member -> (Double)earnedPoints.get(member.getIdentityId())));
            if (teamBudgetType == RewardBudgetType.COMPUTED) {
                computedRecipientsCount += (double)rewardTeam2.getMembers().size();
                computedBudgetTeams.add(rewardTeam2);
                totalPointsPerTeam.put(rewardTeam2.getId(), totalTeamPoints);
                continue;
            }
            if (teamBudgetType == RewardBudgetType.FIXED_PER_MEMBER) {
                totalTeamBudget = rewardTeam2.getBudget() * (double)rewardTeam2.getMembers().size();
                this.addTeamRewardRepartition(rewardTeam2, totalTeamBudget, totalTeamPoints, pluginId, earnedPoints, rewardMemberDetails);
                totalFixedTeamsBudget += totalTeamBudget;
                continue;
            }
            if (teamBudgetType != RewardBudgetType.FIXED) continue;
            totalTeamBudget = rewardTeam2.getBudget();
            this.addTeamRewardRepartition(rewardTeam2, totalTeamBudget, totalTeamPoints, pluginId, earnedPoints, rewardMemberDetails);
            totalFixedTeamsBudget += rewardTeam2.getBudget().doubleValue();
        }
        if (totalFixedTeamsBudget >= totalTeamsBudget) {
            throw new IllegalStateException("Total fixed teams budget is higher than fixed budget for all users");
        }
        if (computedRecipientsCount > 0.0 && !computedBudgetTeams.isEmpty()) {
            double remaingBudgetForComputedTeams = totalTeamsBudget - totalFixedTeamsBudget;
            double budgetPerTeamMember = remaingBudgetForComputedTeams / computedRecipientsCount;
            computedBudgetTeams.forEach(rewardTeam -> {
                if (rewardTeam.getMembers() != null && !rewardTeam.getMembers().isEmpty()) {
                    double totalTeamBudget = budgetPerTeamMember * (double)rewardTeam.getMembers().size();
                    Double totalTeamPoints = (Double)totalPointsPerTeam.get(rewardTeam.getId());
                    this.addTeamRewardRepartition((RewardTeam)rewardTeam, totalTeamBudget, totalTeamPoints, pluginId, earnedPoints, rewardMemberDetails);
                }
            });
        }
    }

    private void buildNoPoolUsers(Map<Long, Double> earnedPoints, List<RewardTeam> teams, Set<Long> identityIds) {
        ArrayList<Long> noPoolsIdentityIds = new ArrayList<Long>(earnedPoints.keySet());
        noPoolsIdentityIds.removeAll(identityIds);
        if (!noPoolsIdentityIds.isEmpty()) {
            RewardTeam noPoolRewardTeam = new RewardTeam();
            noPoolRewardTeam.setDisabled(false);
            List noPoolRewardTeamList = noPoolsIdentityIds.stream().map(identityId -> {
                RewardTeamMember rewardTeamMember = new RewardTeamMember();
                rewardTeamMember.setIdentityId(identityId);
                return rewardTeamMember;
            }).collect(Collectors.toList());
            noPoolRewardTeam.setMembers(noPoolRewardTeamList);
            noPoolRewardTeam.setId(Long.valueOf(0L));
            noPoolRewardTeam.setRewardType(RewardBudgetType.COMPUTED);
            teams.add(noPoolRewardTeam);
        }
    }

    private Set<Long> filterEligibleMembersAndTeams(List<RewardTeam> teams, Map<Long, Double> earnedPoints) {
        HashSet<Long> identityIds = new HashSet<Long>();
        Iterator<RewardTeam> teamsIterator = teams.iterator();
        while (teamsIterator.hasNext()) {
            RewardTeam rewardTeam = teamsIterator.next();
            List members = rewardTeam.getMembers();
            if (members == null || members.isEmpty()) {
                teamsIterator.remove();
                continue;
            }
            Iterator membersIterator = members.iterator();
            while (membersIterator.hasNext()) {
                RewardTeamMember member = (RewardTeamMember)membersIterator.next();
                Long identityId = member.getIdentityId();
                identityIds.add(identityId);
                if (earnedPoints.containsKey(identityId)) continue;
                membersIterator.remove();
            }
        }
        return identityIds;
    }

    private String getTransactionLabel(WalletReward walletReward, ContractDetail contractDetail, RewardPeriod periodOfTime) {
        Wallet wallet = walletReward.getWallet();
        Locale locale = WalletUtils.getLocale((Wallet)wallet);
        String label = WalletUtils.getResourceBundleKey((Locale)locale, (String)"exoplatform.wallet.label.rewardTransactionLabel");
        if (StringUtils.isBlank((String)label)) {
            return "";
        }
        return label.replace("{0}", wallet.getName()).replace("{1}", WalletUtils.formatNumber((Object)walletReward.getTokensToSend(), (String)locale.getLanguage())).replace("{2}", contractDetail.getSymbol()).replace("{3}", RewardUtils.formatTime((Object)periodOfTime.getStartDateInSeconds(), (String)locale.getLanguage())).replace("{4}", RewardUtils.formatTime((Object)(periodOfTime.getEndDateInSeconds() - 1L), (String)locale.getLanguage()));
    }

    private String getTransactionMessage(WalletReward walletReward, ContractDetail contractDetail, RewardPeriod periodOfTime) {
        StringBuilder transactionMessage = new StringBuilder();
        Set walletRewardsByPlugin = walletReward.getRewards();
        Locale locale = WalletUtils.getLocale((Wallet)walletReward.getWallet());
        for (WalletPluginReward walletPluginReward : walletRewardsByPlugin) {
            String label;
            String transactionMessagePart = null;
            if (walletPluginReward.isPoolsUsed() && StringUtils.isNotBlank((String)walletReward.getPoolName())) {
                label = WalletUtils.getResourceBundleKey((Locale)locale, (String)"exoplatform.wallet.label.rewardTransactionMessageWithPool");
                if (StringUtils.isBlank((String)label)) continue;
                transactionMessagePart = label.replace("{0}", WalletUtils.formatNumber((Object)walletPluginReward.getAmount(), (String)locale.getLanguage())).replace("{1}", contractDetail.getSymbol()).replace("{2}", WalletUtils.formatNumber((Object)walletPluginReward.getPoints(), (String)locale.getLanguage())).replace("{3}", walletPluginReward.getPluginId()).replace("{4}", walletReward.getPoolName()).replace("{5}", RewardUtils.formatTime((Object)periodOfTime.getStartDateInSeconds(), (String)locale.getLanguage())).replace("{6}", RewardUtils.formatTime((Object)(periodOfTime.getEndDateInSeconds() - 1L), (String)locale.getLanguage()));
            } else {
                label = WalletUtils.getResourceBundleKey((Locale)locale, (String)"exoplatform.wallet.label.rewardTransactionMessageNoPool");
                if (StringUtils.isBlank((String)label)) continue;
                transactionMessagePart = label.replace("{0}", WalletUtils.formatNumber((Object)walletPluginReward.getAmount(), (String)locale.getLanguage())).replace("{1}", contractDetail.getSymbol()).replace("{2}", WalletUtils.formatNumber((Object)walletPluginReward.getPoints(), (String)locale.getLanguage())).replace("{3}", walletPluginReward.getPluginId()).replace("{4}", RewardUtils.formatTime((Object)periodOfTime.getStartDateInSeconds(), (String)locale.getLanguage())).replace("{5}", RewardUtils.formatTime((Object)(periodOfTime.getEndDateInSeconds() - 1L), (String)locale.getLanguage()));
            }
            transactionMessage.append(transactionMessagePart);
            transactionMessage.append("\r\n");
        }
        return transactionMessage.toString();
    }

    private void addTeamRewardRepartition(RewardTeam rewardTeam, double totalTeamBudget, double totalTeamPoints, String pluginId, Map<Long, Double> earnedPoints, Set<WalletPluginReward> rewardMemberDetails) {
        if (rewardTeam.getMembers() == null || rewardTeam.getMembers().isEmpty() || totalTeamBudget <= 0.0 || totalTeamPoints <= 0.0) {
            return;
        }
        double amountPerPoint = totalTeamBudget / totalTeamPoints;
        rewardTeam.getMembers().forEach(member -> {
            Long identityId = member.getIdentityId();
            Double points = (Double)earnedPoints.get(identityId);
            WalletPluginReward rewardMemberDetail = new WalletPluginReward();
            rewardMemberDetail.setIdentityId(identityId.longValue());
            rewardMemberDetail.setPluginId(pluginId);
            rewardMemberDetail.setPoints(points.doubleValue());
            rewardMemberDetail.setAmount(points * amountPerPoint);
            rewardMemberDetail.setPoolsUsed(true);
            rewardMemberDetails.add(rewardMemberDetail);
        });
    }

    private WalletTokenAdminService getTokenAdminService() {
        if (this.walletTokenAdminService == null) {
            this.walletTokenAdminService = (WalletTokenAdminService)CommonsUtils.getService(WalletTokenAdminService.class);
        }
        return this.walletTokenAdminService;
    }
}

