/*
 * Copyright (C) 2003-2010 eXo Platform SAS.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see<http://www.gnu.org/licenses/>.
 */
package org.exoplatform.swf.sonar.plugin.rules;

import org.sonar.api.server.rule.RulesDefinition;
import org.sonar.check.Priority;
import org.sonar.check.Rule;
import org.sonar.plugins.java.api.JavaCheck;
import org.sonar.plugins.java.api.JavaFileScanner;
import org.sonar.plugins.java.api.JavaFileScannerContext;
import org.sonar.plugins.java.api.tree.*;
import org.sonar.squidbridge.annotations.ActivatedByDefault;
import org.sonar.squidbridge.annotations.SqaleConstantRemediation;
import org.sonar.squidbridge.annotations.SqaleSubCharacteristic;

@Rule(key = "WebUiCsrfCheck",
        name = "WebUI CSRF protection check",
        description = "This rule checks that all the WebUI action listeners have CSRF protection activated",
        tags = {"security"},
        priority = Priority.BLOCKER)
@SqaleSubCharacteristic(RulesDefinition.SubCharacteristics.SECURITY_FEATURES)
@SqaleConstantRemediation("10min")
@ActivatedByDefault
public class WebUiCsrfCheck extends BaseTreeVisitor implements JavaFileScanner {
    private static final String EVENT_CONFIG    = "EventConfig";
    private static final String ISSUE_MESSAGE   = "WebUI CSRF protection must be activated";
    private static final String PHASE           = "phase";
    private static final String CSRF_CHECK      = "csrfCheck";
    private static final String DECODE          = "DECODE";

    private JavaFileScannerContext context;

    @Override
    public void scanFile(JavaFileScannerContext context) {
        this.context = context;
        scan(context.getTree());
    }

    /**
     *  Checks that, if phase is not Phase.DECODE,
     *  annotation must have argument csrfCheck=true
     */
    @Override
    public void visitAnnotation(AnnotationTree annotationTree) {
        if (EVENT_CONFIG.equals(getAnnotationName(annotationTree))) {
            if (isProcessWebUIPhase(annotationTree) && !isCsrfTrue(annotationTree)) {
                context.reportIssue(this, annotationTree, ISSUE_MESSAGE);
            }
        }
        super.visitAnnotation(annotationTree);
    }

    private boolean isCsrfTrue(AnnotationTree annotationTree) {
        for (ExpressionTree argument : annotationTree.arguments()) {
            if (argument.is(Tree.Kind.ASSIGNMENT)) {
                AssignmentExpressionTree assExpTree = (AssignmentExpressionTree) argument;

                //Check annotation variable is CSRF
                if (assExpTree.variable().is(Tree.Kind.IDENTIFIER)) {
                    IdentifierTree variable = (IdentifierTree) assExpTree.variable();
                    if (CSRF_CHECK.equalsIgnoreCase(variable.name())) {

                        //Check CSRF value is true
                        if (assExpTree.expression().is(Tree.Kind.BOOLEAN_LITERAL)) {
                            LiteralTree expression = (LiteralTree) assExpTree.expression();
                            if (Boolean.valueOf(expression.value())) {
                                return true;
                            }
                        }
                    }
                }
            }
        }
        return false;
    }

    private boolean isProcessWebUIPhase(AnnotationTree annotationTree) {
        for (ExpressionTree argument : annotationTree.arguments()) {
            if (argument.is(Tree.Kind.ASSIGNMENT)) {
                AssignmentExpressionTree assExpTree = (AssignmentExpressionTree) argument;

                //Check annotation variable is PHASE
                if (assExpTree.variable().is(Tree.Kind.IDENTIFIER)) {
                    IdentifierTree variable = (IdentifierTree) assExpTree.variable();
                    if (PHASE.equalsIgnoreCase(variable.name())) {

                        //Check value of Phase argument
                        String phaseValue = ((MemberSelectExpressionTree)assExpTree.expression()).identifier().name();
                        return !DECODE.equalsIgnoreCase(phaseValue);
                    }
                }
            }
        }
        //Process is the default Phase
        //=> return true if no Phase argument is specified
        return true;
    }

    private String getAnnotationName(ExpressionTree initializer) {
        String result = "";
        if (initializer.is(Tree.Kind.ANNOTATION)) {
            Tree annotationType = ((AnnotationTree) initializer).annotationType();
            if (annotationType.is(Tree.Kind.IDENTIFIER)) {
                result = ((IdentifierTree) annotationType).name();
            } else if (annotationType.is(Tree.Kind.MEMBER_SELECT)) {
                result = fullName((MemberSelectExpressionTree) annotationType);
            }
        }
        return result;
    }

    private String fullName(MemberSelectExpressionTree tree) {
        if (tree.expression().is(Tree.Kind.IDENTIFIER)) {
            return ((IdentifierTree) tree.expression()).name() + "." + tree.identifier().name();
        }
        return fullName((MemberSelectExpressionTree) tree.expression()) + "." + tree.identifier().name();
    }
}
