package org.exoplatform.tcm;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;

import org.jopendocument.dom.OOUtils;
import org.jopendocument.dom.spreadsheet.Sheet;
import org.jopendocument.dom.spreadsheet.SpreadSheet;

public class TestCaseHelper {

	private static final String INPUT_VALUE_SEPARATOR = ",";
	private static final String TESTCAMPAIGN_PROPERTIES_FILENAME = "testcampaign.properties";
	private static final String TEST_DEFINITIONS_DIR = "TestDefinitions";
	private static final String TESTRUN_PLAN_DIR = "TestPlan";
	private static final String TESTRUN_PERFORMED_DIR = "TestRun";
	private static final String COLMUN_NAME_PATH = "Path";
	private static final String TEST_PLAN_SUFFIX = "_TestPlan.ods";
	private static final String TEST_RESULT_SUFFIX = "_TestResults.ods";
	private static final String TEST_CASES_DEFINITION_FILES_SUFFIX = "_TestDefinition.ods";
	private static final String TEST_CASES_RUN_FILES_PREFIX = "TestCaseRun_";
	private static final String KEY_TEST_CAMPAIGN_TESTERS = "test.campaign.testers";
	private static final String KEY_TEST_CAMPAIGN_BROWSERS = "test.campaign.browsers";
	private static final int COLUMN_RELATED_FUNCTION = 1;
	private static final int COLUMN_CASE_ID = 2;
	private static final int COLUMN_PATH = 3;
	private static final int COLUMN_STEP_DESC = 5;
	private static final int COLUMN_PRIORITY = 8;
	public static final String TEST_PLAN_TEMPLATE = "/TestPlanTemplate.ods";
	public static final String TEST_RESULT_TEMPLATE = "/TestResultTemplate.ods";

	public static void generateTestPlan(String testCampaignDirName) throws IOException {

		Map<String, TestCaseFunctionInfo> theFunctionsByPaths = new HashMap<String, TestCaseFunctionInfo>();

		// New TestPlan sheet
		Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);
		String testersValue = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_TESTERS, "tester1,tester2");
		if (testersValue.length()==0)
			throw new RuntimeException("Value missing for " + KEY_TEST_CAMPAIGN_TESTERS + " in " + testCampaignDirName + "/" + TESTCAMPAIGN_PROPERTIES_FILENAME);
		// String campaignName =
		// testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_NAME, "");
		String testCampaignBrwosers = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_BROWSERS, "ie,ff");
		String browsers[] = testCampaignBrwosers.split(INPUT_VALUE_SEPARATOR);
		if (browsers.length == 0)
			throw new RuntimeException("Value missing for " + KEY_TEST_CAMPAIGN_BROWSERS + " in " + testCampaignDirName + "/" + TESTCAMPAIGN_PROPERTIES_FILENAME);

		File dirname = new File(testCampaignDirName, TEST_DEFINITIONS_DIR);
		System.out.println("Searching files prefixed by " + TEST_CASES_DEFINITION_FILES_SUFFIX + " in "
		      + dirname.getPath());
		File[] listFiles = getTestFiles(dirname);
		for (int i = 0; i < listFiles.length; i++) {
			File file = listFiles[i];
			if (file.getName().endsWith(TEST_CASES_DEFINITION_FILES_SUFFIX)) {
				fillTestCaseFonctionsMap(file, theFunctionsByPaths, browsers, 1);
			}
		}

		String[] testers = testersValue.split(INPUT_VALUE_SEPARATOR);

		int columnCount = testers.length + 3;
		String[] columns = new String[columnCount];
		for (int i = 0; i < testers.length; i++) {
			columns[i + 2] = testers[i];
			System.out.println("Tester:" + testers[i]);
		}
		columns[columnCount - 1] = "Description";

		List<String> functionIds = new ArrayList<String>(theFunctionsByPaths.keySet());
		Collections.sort(functionIds);

		// TODO sort the paths
		// Save the data to an ODS file and open it.
		InputStream template = TestCaseHelper.class.getResourceAsStream(TEST_PLAN_TEMPLATE);
		SpreadSheet testPlanSpreadSheet = SpreadSheet.createFromStream(template);
		final Sheet sheet = testPlanSpreadSheet.getSheet(0);

		// How many columns are already present
		int colShift = -1;
		Object valueAt = null;
		while (!"".equals(valueAt)) {
			colShift++;
			valueAt = sheet.getValueAt(colShift, 0);
			// TODO Map column position<=>name
		}

		// Add some columns
		int expectedColumnCount = colShift + testers.length;
		sheet.ensureColumnCount(expectedColumnCount + 1);
		for (int i = 0; i < testers.length; i++) {
			sheet.setValueAt(testers[i], colShift + i, 0);
		}
		sheet.setValueAt(COLMUN_NAME_PATH, expectedColumnCount, 0);

		// Add paths
		int row = 1;
		sheet.ensureRowCount(functionIds.size() + row);
		for (String path : functionIds) {
			TestCaseFunctionInfo functionInfo = theFunctionsByPaths.get(path);
			sheet.setValueAt(functionInfo.getId(), 0, row);
			sheet.setValueAt(functionInfo.getRelatedFunction(), 1, row);
			sheet.setValueAt(functionInfo.getTestCaseCount(), 2, row);
			sheet.setValueAt(functionInfo.getPath(), expectedColumnCount, row);
			row++;
		}

		TestCaseHelper.saveAsFile(testPlanSpreadSheet, testCampaignDirName + "/", testCampaignDirName + TEST_PLAN_SUFFIX,
		      false);
	}

	private static Properties readTestCampaignInfo(String testCampaignDirName) throws IOException, FileNotFoundException {
		Properties properties = new Properties();

		File file = new File(testCampaignDirName, TESTCAMPAIGN_PROPERTIES_FILENAME);
		System.out.println("Loading test campaign params in " + file.getPath());
		properties.load(new FileInputStream(file));
		return properties;
	}

	private static File[] getTestFiles(File testDefDir) {
		if (!testDefDir.isDirectory()) {
			throw new RuntimeException("Dir expected for " + testDefDir);
		}
		return testDefDir.listFiles();
	}

	public static boolean isTestCaseSheet(String sheetName) {
		return !("Summary_Report".equals(sheetName) || "Notes".equals(sheetName) || "Cover".equals(sheetName)
		      || "Check_List".equals(sheetName) || sheetName.contains("#"));
	}

	private static void fillTestCaseFonctionsMap(File testDefinitionFile,
	      Map<String, TestCaseFunctionInfo> theFunctionsById, String[] browsers, int functionIndex) throws IOException {
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testDefinitionFile);
		String fileName = testDefinitionFile.getName();
		String filePath = testDefinitionFile.getPath();
		System.out.println("Parsing file:" + filePath);
		String functionGroup = fileName.split("_")[functionIndex];

		int sheetCount = spreadSheet.getSheetCount();
		Sheet summarySheet = null;
		for (int i = 0; i < sheetCount; i++) {
			final Sheet sheet = spreadSheet.getSheet(i);
			int rowCount = sheet.getRowCount();
			int columnCount = sheet.getColumnCount();
			String sheetName = spreadSheet.getSheetName(sheet);

			// System.out.println("* Sheet: + " + sheetName);
			if ("Summary_Report".equals(sheetName))
				summarySheet = sheet;

			if (!isTestCaseSheet(sheetName))
				continue;

			int headerRow = 5;
			int columnTester = -1;
			for (int c = 0; c < columnCount - 1 && sheet.getRowCount() > headerRow; c++) {
				String columnName = "" + sheet.getValueAt(c, headerRow);
				if ("Tester".equals(columnName))
					columnTester = c;
				if ("".equals(columnName))
					break;
			}

			String previousPath = null;
			TestCaseFunctionInfo functionInfo = null;
			Float currentCaseId = new Float(0);

			for (int row = headerRow + 1; row > 2 && row < rowCount; row++) {
				String newPath = "" + sheet.getValueAt(COLUMN_PATH, row);
				Object caseId = sheet.getValueAt(COLUMN_CASE_ID, row);
				String relatedFunction = "" + sheet.getValueAt(COLUMN_RELATED_FUNCTION, row);
				String priority = "" + sheet.getValueAt(COLUMN_PRIORITY, row);
				String stepDesc = "" + sheet.getValueAt(COLUMN_STEP_DESC, row);

				// Stop on first empty path
				String message = "row " + row + " in sheet " + sheetName + " of " + filePath;
				if ("".equals(newPath)) {
					if (!("".equals(relatedFunction)) || !("".equals(stepDesc))) {
						System.err.println("[ERROR] Empty cell in " + message);
					}
				} else {
					System.out.println("Related function : " + relatedFunction);
					if ("".equals(relatedFunction))
						System.err.println("[ERROR] Empty related function : " + message);
					if (!newPath.equals(previousPath)) {
						functionInfo = theFunctionsById.get(relatedFunction);
						if (functionInfo != null && !functionInfo.getPath().equals(newPath))
							System.err.println("[ERROR] Related Function/Path not identical for " + relatedFunction
							      + " had  [" + functionInfo.getPath() + "] expected [" + newPath + "] " + message);
						else if (functionInfo == null) {
							functionInfo = new TestCaseFunctionInfo(newPath, relatedFunction, priority);
						}
						theFunctionsById.put(relatedFunction, functionInfo);
						currentCaseId = new Float(0);
					}
					previousPath = newPath;
					if (!(caseId instanceof Float))
						System.err.println("[ERROR] Wrong CaseID [" + caseId + "] " + message);
					else {
						if (caseId.equals(0))
							System.err.println("[ERROR] Wrong CaseID value [" + caseId + "] " + message);
						else if (!currentCaseId.equals(caseId)) {
							functionInfo.incrementTestCaseCount("" + caseId);
							currentCaseId = (Float) caseId;
						}
					}
				}

				if (columnTester > 0) {
					int colShiftBrowser = 1;
					String tester = "" + sheet.getValueAt(columnTester, row);
					for (String browser : browsers) {
						int column = columnTester + colShiftBrowser++;
						String browserResult = "" + sheet.getValueAt(column, row);
						boolean result = functionInfo.incrementTestResult(browser, tester, "" + caseId, browserResult);
						if (!result)
							System.err.println("[ERROR] Wrong result value : [" + browserResult + "] column " + column + " "
							      + message);
					}
					// Read Comment
					String comment = "" + sheet.getValueAt(columnTester + colShiftBrowser, row);
					if (comment.trim().length() > 0)
						functionInfo.addComment(tester + " - " + comment);
				}

			}
		}

		// Fill in the Descriptions
		int rowCount = summarySheet.getRowCount();
		int columnCount = summarySheet.getRowCount();
		int rowHeader = -1;
		int functionaliTyColumn = -1;
		int idColumn = 0;
		// Detect row header, and functionality index
		for (int row = 0; row < rowCount; row++) {
			String columnName = "" + summarySheet.getValueAt(idColumn, row);
			if ("Id".equalsIgnoreCase(columnName)) {
				rowHeader = row;
				for (int col = rowHeader; col < columnCount; col++) {
					columnName = "" + summarySheet.getValueAt(col, row);
					if ("Functionality".equals(columnName)) {
						functionaliTyColumn = col;
						break;
					}
				}
				break;
			}
		}
		if (functionaliTyColumn > 0) {
			for (int row = rowHeader + 1; row < rowCount; row++) {
				String id = "" + summarySheet.getValueAt(idColumn, row);
				String functionality = "" + summarySheet.getValueAt(functionaliTyColumn, row);
				TestCaseFunctionInfo testCaseFunctionInfo = theFunctionsById.get(id);
				if (testCaseFunctionInfo != null) {
					testCaseFunctionInfo.setRelatedFunction(functionGroup + " - " + functionality);
				}
			}
		} else {
			System.err.println("[ERROR] Functionality column not found for " + filePath);
		}

	}

	public static void generateTestCaseRun(String testCampaignDirName, String testPlanFileName) throws IOException {
		// For each tester collect its Paths and generate a file

		File file = new File(testCampaignDirName, testPlanFileName);
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(file);
		final Sheet sheet = spreadSheet.getSheet(0);
		int columnCount = sheet.getColumnCount();

		//
		Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);

		// First tester column
		int testerColumnStart = 3;
		int testFunctinIdColumn = 0;
		int testCaseCountColumn = 2;

		for (int column = testerColumnStart; column < columnCount; column++) {
			String testerName = "" + sheet.getValueAt(column, 0);
			if (testerName.length() == 0 || testerName.equals(COLMUN_NAME_PATH))
				break;
			int rowCount = sheet.getRowCount();
			int tcCount = 0;
			Set<String> functionIdsForTester = new HashSet<String>();
			for (int row = 1; row < rowCount; row++) {
				String functionId = "" + sheet.getValueAt(testFunctinIdColumn, row);
				String affectedToTester = "" + sheet.getValueAt(column, row);
				if (functionId.length() == 0)
					break;
				if (affectedToTester.length() > 0) {
					tcCount += Float.valueOf("" + sheet.getValueAt(testCaseCountColumn, row)).intValue();
					functionIdsForTester.add(functionId);
				}
			}

			System.out.println(testerName + ": " + tcCount + " TestCases");
			File[] listFiles = getTestFiles(new File(testCampaignDirName, TEST_DEFINITIONS_DIR));
			for (int i = 0; i < listFiles.length; i++) {
				File tcFile = listFiles[i];
				String testCaseDefinitionFileName = tcFile.getAbsolutePath();
				if (testCaseDefinitionFileName.endsWith(TEST_CASES_DEFINITION_FILES_SUFFIX)) {
					generateSignleTestCaseRun(testCaseDefinitionFileName, testerName, functionIdsForTester,
					      testCampaignInfo, testCampaignDirName);
				}
			}
		}

	}

	private static void generateSignleTestCaseRun(String testCaseDefinitionFileName, String testerName,
	      Set<String> testerFunctions, Properties testCampaignInfo, String testCampaignDirName) throws IOException {

		// Parameters
		// String testCampaignName =
		// testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_NAME, "");
		String testCampaignBrwosers = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_BROWSERS, "");
		String browsers[] = testCampaignBrwosers.split(INPUT_VALUE_SEPARATOR);
		// String testEnvironments =
		// testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_TARGETS, "");

		// Load the file.
		File testCaseDefinitionFile = new File(testCaseDefinitionFileName);
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testCaseDefinitionFile);
		int sheetCount = spreadSheet.getSheetCount();

		int assigned = 0;
		Set<String> functions = new TreeSet<String>();
		// Iterate on the sheets
		for (int i = 1; i < sheetCount; i++) {
			final Sheet sheet = spreadSheet.getSheet(i);
			String sheetName = spreadSheet.getSheetName(sheet);
			if (!isTestCaseSheet(sheetName))
				continue;

			int columnCount = sheet.getColumnCount();
			int headerRow = 5;

			for (int j = 1; j < columnCount; j++) {
				Object valueAt = sheet.getValueAt(j, headerRow);
				if ("".equals(valueAt)) {
					// Create the headers
					sheet.setValueAt("Tester", j, headerRow);
					int colShift = 1;
					for (String browser : browsers) {
						sheet.setValueAt(browser, j + colShift, headerRow);
						colShift++;
					}
					sheet.setValueAt("Comment/Bug", j + colShift, headerRow);

					int rowCount = sheet.getRowCount();
					// int oldStep = 0;
					// int caseValue = 1;
					for (int r = 6; r < rowCount; r++) {
						Object valueAtRow = sheet.getValueAt(COLUMN_RELATED_FUNCTION, r);
						String functionIdValue = "" + sheet.getValueAt(COLUMN_RELATED_FUNCTION, r);
						if ("".equals(valueAtRow) && "".equals(functionIdValue)) {
							break;
						}
						if (testerFunctions.contains(functionIdValue)) {
							// Values
							sheet.setValueAt(testerName, j, r);
							functions.add(functionIdValue);
							String step = "" + sheet.getValueAt(COLUMN_STEP_DESC, r);
							if (step.startsWith("Step 1"))
								assigned++;
							else if (!step.startsWith("Step "))
								System.err.println("[ERROR] Wrong step content [" + step + "]. row " + r + " of sheet "
								      + sheetName + " of file " + testCaseDefinitionFile.getName());
						}
					}
					break;
				}
			}

		}
		String newFileName = TEST_CASES_RUN_FILES_PREFIX + testerName + "_" + assigned + "_"
		      + testCaseDefinitionFile.getName().replaceAll(TEST_CASES_DEFINITION_FILES_SUFFIX, ".ods");

		TestCaseHelper.saveAsFile(spreadSheet, testCampaignDirName + "/" + TESTRUN_PLAN_DIR, newFileName, false);
		System.out.println("* " + newFileName + " " + assigned + " cases. " + functions);
		//
		File resultDir = new File(testCampaignDirName + "/" + TESTRUN_PERFORMED_DIR);
		resultDir.mkdirs();
		System.out.println("* Created " + resultDir.getPath() + " for results.");
	}

	public static void saveAsFile(final SpreadSheet spreadSheet, String targetDir, String fileName, boolean open)
	      throws IOException, FileNotFoundException {
		// Save to file and open it.
		File outputDir = new File(targetDir);
		outputDir.mkdirs();
		File outputFile = new File(outputDir, fileName);
		File fileSaved = spreadSheet.saveAs(outputFile);
		System.out.println("Generated file:" + fileSaved.getPath());
		if (open)
			OOUtils.open(fileSaved);
	}

	public static void generateTestPlanResult(String testCampaignDirName) throws IOException {

		Map<String, TestCaseFunctionInfo> theFunctionsByPaths = new HashMap<String, TestCaseFunctionInfo>();

		// New TestPlan sheet
		Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);
		String testersValue = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_TESTERS, "");
		// String campaignName =
		// testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_NAME, "");
		String testCampaignBrowsers = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_BROWSERS, "");
		String browsers[] = testCampaignBrowsers.split(INPUT_VALUE_SEPARATOR);

		File[] listFiles = getTestFiles(new File(testCampaignDirName, TESTRUN_PERFORMED_DIR));
		for (int i = 0; i < listFiles.length; i++) {
			File file = listFiles[i];
			if (file.getName().startsWith(TEST_CASES_RUN_FILES_PREFIX)) {
				fillTestCaseFonctionsMap(file, theFunctionsByPaths, browsers, 4);
			}
		}

		String[] testers = testersValue.split(INPUT_VALUE_SEPARATOR);

		int columnCount = testers.length + 3;
		String[] columns = new String[columnCount];
		for (int i = 0; i < testers.length; i++) {
			columns[i + 2] = testers[i];
			System.out.println("Tester:" + testers[i]);
		}
		columns[columnCount - 1] = "Description";

		List<String> functionIds = new ArrayList<String>(theFunctionsByPaths.keySet());
		Collections.sort(functionIds);

		// TODO sort the paths
		// Save the data to an ODS file and open it.
		InputStream template = TestCaseHelper.class.getResourceAsStream(TEST_RESULT_TEMPLATE);
		SpreadSheet testPlanSpreadSheet = SpreadSheet.createFromStream(template);
		final Sheet sheet = testPlanSpreadSheet.getSheet(0);
		final Sheet sheetFeedbacks = testPlanSpreadSheet.getSheet(1);

		// How many columns are already present
		int colShift = 0;
		sheet.setValueAt("Id", colShift++, 6);
		sheet.setValueAt("Function", colShift++, 6);
		sheet.setValueAt("Priority", colShift++, 6);
		sheet.setValueAt("TC Count", colShift++, 6);

		// Add some columns
		int expectedColumnCount = colShift + browsers.length * 4;
		int row = 7;
		sheet.ensureRowCount(functionIds.size() + row + 1);
		sheet.ensureColumnCount(expectedColumnCount + 1);
		int colShiftBrowsers = 0;
		for (String browser : browsers) {
			sheet.setValueAt(browser, colShift + colShiftBrowsers, 0);
			sheet.setValueAt("PASSED", colShift + colShiftBrowsers++, 6);
			sheet.setValueAt("FAILED", colShift + colShiftBrowsers++, 6);
			sheet.setValueAt("BLOCKED", colShift + colShiftBrowsers++, 6);
		}

		// Add paths
		for (String path : functionIds) {
			TestCaseFunctionInfo functionInfo = theFunctionsByPaths.get(path);
			sheet.setValueAt(functionInfo.getId(), 0, row);
			sheet.setValueAt(functionInfo.getRelatedFunction(), 1, row);
			sheet.setValueAt(functionInfo.getPriority(), 2, row);
			sheet.setValueAt(functionInfo.getTestCaseCount(), 3, row);
			colShiftBrowsers = 0;
			for (String browser : browsers) {
				// browser + "-NR"
				// sheet.setValueAt(""+functionInfo.getTestNotRunCount(browser),
				// colShift + colShiftBrowsers++, row);
				// browser + "-P"
				sheet.setValueAt(functionInfo.getTestPassedCount(browser), colShift + colShiftBrowsers++, row);
				// browser + "-F"
				sheet.setValueAt(functionInfo.getTestFailedCount(browser), colShift + colShiftBrowsers++, row);
				// browser + "-B"
				sheet.setValueAt(functionInfo.getTestBlockedCount(browser), colShift + colShiftBrowsers++, row);
			}
			row++;
		}

		sheetFeedbacks.setValueAt("Id", 0, 6);
		sheetFeedbacks.setValueAt("Function", 1, 6);
		sheetFeedbacks.setValueAt("Comment", 2, 6);
		row = 7;
		sheetFeedbacks.ensureRowCount(functionIds.size());
		for (String path : functionIds) {
			TestCaseFunctionInfo functionInfo = theFunctionsByPaths.get(path);
			Set<String> comments = functionInfo.getComments();
			sheetFeedbacks.ensureRowCount(row + comments.size());
			for (String comment : comments) {
				sheetFeedbacks.setValueAt(functionInfo.getId(), 0, row);
				sheetFeedbacks.setValueAt(functionInfo.getRelatedFunction(), 1, row);
				sheetFeedbacks.setValueAt(comment, 2, row);
				row++;
			}
		}
		TestCaseHelper.saveAsFile(testPlanSpreadSheet, testCampaignDirName + "/", testCampaignDirName
		      + TEST_RESULT_SUFFIX, false);
	}
}
