package org.exoplatform.tcm;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jopendocument.dom.OOUtils;
import org.jopendocument.dom.spreadsheet.Sheet;
import org.jopendocument.dom.spreadsheet.SpreadSheet;
/**
 * @author Dimitri BAELI, CanhBX
 *
 */

public class TestCaseHelper {

    private static Logger logger = Logger.getLogger(TestCaseManager.class.getName());

	private final String INPUT_VALUE_SEPARATOR = ",";	
	
	private final String TEST_DEFINITIONS_DIR = "TestDefinitions";
	private final String TESTRUN_PLAN_DIR = "TestPlan";
	private final String TESTRUN_PERFORMED_DIR = "TestRun";
	private final String COLUMN_NAME_PATH = "Path";
	
	private final String TEST_PLAN_SUFFIX = "_TestPlan.ods";
	private final String TEST_RESULT_SUFFIX = "_TestResults.ods";
	private final String TEST_CASES_DEFINITION_FILES_SUFFIX = "_TestDefinition.ods";
	private final String TEST_CASES_RUN_FILES_PREFIX = "TestCaseRun_";
	
	private final String TESTCAMPAIGN_PROPERTIES_FILENAME = "testcampaign.properties";
    private final String KEY_TEST_CAMPAIGN_TESTERS = "test.campaign.testers";
    private final String KEY_TEST_CAMPAIGN_NAME = "test.campaign.name";
	private final String KEY_TEST_CAMPAIGN_BROWSERS = "test.campaign.browsers";
	
	private final int COLUMN_CASE_NO_INDEX = 0;
	private final int COLUMN_RELATED_FUNCTION_INDEX = 1;
	private final int COLUMN_CASE_ID_INDEX = 2;
	private final int COLUMN_PATH_INDEX = 3;
	private final int COLUMN_STEP_DESC_INDEX = 5;
	private final int COLUMN_PRIORITY_INDEX = 8;
	
	private final String PRIORITY_HIGH = "High";
	private final String PRIORITY_MEDIUM = "Medium";
	private final String PRIORITY_LOW = "Low";
	
	private final String TOTAL_HIGH = "Total High";
	private final String TOTAL_MEDIUM = "Total Medium";
	private final String TOTAL_LOW = "Total Low";
	
	public final String TEST_PLAN_TEMPLATE = "TestPlanTemplate.ods";
	public final String TEST_RESULT_TEMPLATE = "TestResultTemplate.ods";
	
	public TestCaseHelper(){
		super();
	}
	
	/**
	 * Function to generate auto number for all TestDefinitions files
	 * ==> P0: TestDefinition number generator (Structure first, then may be auto assignment)
	 * 			input: 	all testDefinitions files in the folder <test-campaign-dirname>/TestDefinitions
	 * 
	 * 			output:	all testDefinitions files in the folder <test-campaign-dirname>/TestDefinitions which are numbered for CaseNo & CaseID
	 * 
	 */
	public void generateAutoNumber(String testCampaignDirName) throws IOException {
		//Searching for test definition files ....
		File dirname = new File(testCampaignDirName, TEST_DEFINITIONS_DIR);
		logger.log(Level.FINE, "Searching files prefixed by " + TEST_CASES_DEFINITION_FILES_SUFFIX + " in "
		      + dirname.getPath());
		File[] listFiles = getTestFiles(dirname);
		for (int i = 0; i < listFiles.length; i++) {
			File file = listFiles[i];
			if (file.getName().endsWith(TEST_CASES_DEFINITION_FILES_SUFFIX)) {
				generateAutoNumberForSingleFile(file, testCampaignDirName);
			}
		}		
	}
	
	/**
	 * Function to generate auto number for a TestDefinitions file
	 * @throws IOException 
	 * */
	private void generateAutoNumberForSingleFile(File testDefinitionFile,String testCampaignDirName) throws IOException{		
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testDefinitionFile);
		String filePath = testDefinitionFile.getPath();
		logger.log(Level.FINE, " Parsing file:" + filePath + " for generate auto number for CaseNo & CaseId columns");

		int sheetCount = spreadSheet.getSheetCount();
		for (int i = 0; i < sheetCount; i++) {
			Integer caseNoCounter = 0;
			Integer caseIdCounter = 0;
			final Sheet sheet = spreadSheet.getSheet(i);
			int rowCount = sheet.getRowCount();
			String sheetName = getSheetName(sheet);
			//String sheetName = sheet.getName();;
			
			if (!isTestCaseSheet(sheetName))
				continue;
			
			String previousPath = null;
			int headerRow = 5;
			for (int row = headerRow + 1; row > 2 && row < rowCount; row++) {
				String message = "row " + (row+1) + " in sheet " + sheetName + " of " + filePath;
				String newPath = "" + sheet.getValueAt(COLUMN_PATH_INDEX, row);
				String step = "" + sheet.getValueAt(COLUMN_STEP_DESC_INDEX, row);
				if(!"".equals(newPath)){
					if (!newPath.equals(previousPath)) {//not the same path
						caseIdCounter = 0;//reset counter
						previousPath = newPath;
					}
					if (!"".equals(step)){
						if (step.startsWith("Step 1:")){
							caseNoCounter++;
							caseIdCounter++;
							sheet.setValueAt(caseNoCounter, COLUMN_CASE_NO_INDEX, row);
							sheet.setValueAt(caseIdCounter, COLUMN_CASE_ID_INDEX, row);
						}else{
							if (step.startsWith("Step ") && caseNoCounter>0){//avoid error in first testcase row							
								sheet.setValueAt("", COLUMN_CASE_NO_INDEX, row);
								sheet.setValueAt(caseIdCounter, COLUMN_CASE_ID_INDEX, row);
							}else{
								System.err.println("[ERROR] Step description missing at "+message);
							}
						}
					}
				}else{//ignore empty rows
					break;
				}
			}
			testDefinitionFile = sheet.getSpreadSheet().saveAs(testDefinitionFile);
		}
	}
	
	/**
	 * Function to verify Test Definition Formats  
	 *   - Numbering
     *   - Priorities are filled for all test cases
     *   - For a test case, all steps have same priority
	 * @throws IOException 
	 */
	private boolean validateTestDefinitionFormat(File testDefinitionFile) throws IOException{
		return(validateNumbering(testDefinitionFile)&& validatePriority(testDefinitionFile) && validateSamePriority(testDefinitionFile));
	}
	
	/**
	 * Function to check that the numbering in CaseNo and CaseID of test definition file is correct
	 * @return  true: the numbering is ok
	 * 			false: the numbering has error
	 * @throws IOException 
	 * */
	private boolean validateNumbering(File testDefinitionFile) throws IOException{
		boolean isOk = true;
		int errorCounter =0;
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testDefinitionFile);
		String filePath = testDefinitionFile.getPath();
		logger.log(Level.FINE, " Parsing file:" + filePath + " for verifying Numbering");

		int sheetCount = spreadSheet.getSheetCount();
		for (int i = 0; i < sheetCount; i++) {
			final Sheet sheet = spreadSheet.getSheet(i);
			int rowCount = sheet.getRowCount();
			String sheetName = getSheetName(sheet);
			//String sheetName = sheet.getName();

			if (!isTestCaseSheet(sheetName))
				continue;

			int headerRow = 5;
			int counter = 0;
			for (int row = headerRow + 1; row > 2 && row < rowCount; row++) {
				String message = "row " + (row+1) + " in sheet " + sheetName + " of " + filePath;
				String step = "" + sheet.getValueAt(COLUMN_STEP_DESC_INDEX, row);
				if (!"".equals(step)){//get the priority of the first step of this testcase
					if (step.startsWith("Step 1:")){
						counter = 1;
					}else{
						if (!step.startsWith("Step ")){
							isOk = false;
							if(counter<1){	
								errorCounter++;
								System.err.println("[ERROR] Numbering error at " + message + "\nPlease re-check your steps declaration at this row to make sure it follows this format (case sensitive): Step "+counter+":");
							}else{
								errorCounter++;
								System.err.println("[ERROR] Numbering error at " + message + "\nPlease re-check your steps declaration at this row to make sure it follows these formats (case sensitive): Step "+(counter+1) + ": or Step 1:");
							}
						}else{
							counter++;
						}
					}
				}else{//ignore empty rows
					break;
				}
			}
		}
		if(errorCounter>0){
			logger.log(Level.FINE, " Total " + errorCounter + " errors are detected for this step, please correct them before continue to next step!");
		}else{
			logger.log(Level.FINE, " Validation for this step finish successfully, move to next step...");
		}
		
		return isOk;
	}
	
	/**
	 * Function to check that all test cases are filled priority
	 * @return	true: the priorities are filled
	 * 			false: the priorities are not full filled
	 * @throws IOException 
	 * */
	private boolean validatePriority(File testDefinitionFile) throws IOException{
		boolean isOk =true;
		int errorCounter =0;
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testDefinitionFile);
		String filePath = testDefinitionFile.getPath();
		logger.log(Level.FINE, " Parsing file:" + filePath + " for verifying all priority are full filled");
		
		int sheetCount = spreadSheet.getSheetCount();
		for (int i = 0; i < sheetCount; i++) {
			final Sheet sheet = spreadSheet.getSheet(i);
			int rowCount = sheet.getRowCount();
			String sheetName = getSheetName(sheet);
			//String sheetName = sheet.getName();
			
			if (!isTestCaseSheet(sheetName))
				continue;
			
			int headerRow = 5;
			for (int row = headerRow + 1; row > 2 && row < rowCount; row++) {
				String message = "row " + (row+1) + " in sheet " + sheetName + " of " + filePath;
				String priority = "" + sheet.getValueAt(COLUMN_PRIORITY_INDEX, row);
				String relatedFunction = "" + sheet.getValueAt(COLUMN_RELATED_FUNCTION_INDEX, row);
				if(!"".equals(relatedFunction)){
					if("".equals(priority)){
						isOk = false;
						System.err.println("[ERROR] Empty data in Priority column at "+message);
						errorCounter++;
					}else if(!priority.trim().equalsIgnoreCase(PRIORITY_HIGH) && !priority.trim().equalsIgnoreCase(PRIORITY_MEDIUM) && !priority.trim().equalsIgnoreCase(PRIORITY_LOW)){
						isOk = false;
						System.err.println("[ERROR] Priority type undefined for type : " + priority + " at "+ message);
						errorCounter++;
					}
				}else{//ignore empty rows
					break;
				}
			}
		}
		if(errorCounter>0){
			logger.info(" Total " + errorCounter + " errors are detected for this step, please correct them before continue to next step!");
		}else{
			logger.log(Level.FINE, " Validation for this step finish successfully, move to next step...");
		}
		
		return isOk;
	}
	
	/**
	 * Function to validate that all steps of a testcase have the same priority
	 * @return  true: all steps in any testcase have the same priority
	 * 			false: exist at least one testcase that has not same priority in all steps
	 * @throws IOException 
	 * */
	private boolean validateSamePriority(File testDefinitionFile) throws IOException{
		boolean isOk = true;
		int errorCounter = 0;
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testDefinitionFile);
		String filePath = testDefinitionFile.getPath();
		logger.log(Level.FINE, " Parsing file:" + filePath + " for verifying all steps in any test case have the same priority");

		int sheetCount = spreadSheet.getSheetCount();
		for (int i = 0; i < sheetCount; i++) {
			final Sheet sheet = spreadSheet.getSheet(i);
			int rowCount = sheet.getRowCount();
			String sheetName = getSheetName(sheet);
			//String sheetName = sheet.getName();

			if (!isTestCaseSheet(sheetName))
				continue;

			int headerRow = 5;

			Float previousCaseId = new Float(0);
			String previousPriority = null;
			for (int row = headerRow + 1; row > 2 && row < rowCount; row++) {
				String message = "row " + (row+1) + " in sheet " + sheetName + " of " + filePath;
				Object newCaseId = sheet.getValueAt(COLUMN_CASE_ID_INDEX, row);
				String newPriority = "" + sheet.getValueAt(COLUMN_PRIORITY_INDEX, row);
				String step = "" + sheet.getValueAt(COLUMN_STEP_DESC_INDEX, row);
				if (!"".equals(step)){//get the priority of the first step of this testcase
					if(step.startsWith("Step 1:")){
						previousPriority = newPriority;
						if(newCaseId instanceof Float){
							previousCaseId = (Float) newCaseId;
						}
					}else{
						if(!"".equals(newPriority)){
							if(!newPriority.equalsIgnoreCase(previousPriority)){
								if(newCaseId instanceof Float){
									if (newCaseId.equals((Float)previousCaseId)) {//that's not ok because the priorities are not the same in all steps of this test case
										isOk = false;
										errorCounter++;
										System.err.println("[ERROR] Not the same priority for the same testcase at " +message);
									}
								}
							}
						}
					}
				}else{//ignore empty rows
					break;
				}
			}
		}
		
		if(errorCounter>0){
			logger.info(" Total " + errorCounter + " errors are detected for this step, please correct them before continue to generate TestPlan process!");
		}else{
			logger.log(Level.FINE, " All validation steps for this file : \""+testDefinitionFile.getName()+"\" finish successfully, move to next step...");
		}
		
		return isOk;
	}
	
	/**
	 * Function to generate TestPlan
	 * According to the wiki page : http://wiki-int.exoplatform.org/display/exoTesting/Ideas+for+TestCampaignManager+2.0
	 * ==> P1: TestPlan generator (Structure first, then may be auto assignment)
	 * 			input: 	1) testcampaign.properties file : <test-campaign-dirname>/testcampaign.properties
	 * 					2) the test definition file: <test-campaign-dirname>/TestDefinitions/xxx_TestDefinition.ods
	 * 
	 * 			output:	the testplan file:  <test-campaign-dirname>/<test-campaign-dirname>_TestPlan.ods
	 */
	public void generateTestPlan(String testCampaignDirName) throws IOException {
		//Call function to generate auto number for CaseNo & CaseID columns
		generateAutoNumber(testCampaignDirName);
		
		Set<Boolean> resultSet = new HashSet<Boolean>();
		Boolean isOk;
		Map<String, TestCaseFunctionInfo> theFunctionsByPaths = new HashMap<String, TestCaseFunctionInfo>();

		//Retrieve data(testers,browsers) from testcampaign.properties files ....
		Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);
		
		//retrieve browsers data....
		String testCampaignBrowsers = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_BROWSERS, "ie,ff");
		String browsers[] = testCampaignBrowsers.split(INPUT_VALUE_SEPARATOR);

		//Searching for test definition files ....
		File dirname = new File(testCampaignDirName, TEST_DEFINITIONS_DIR);
		logger.log(Level.FINE, " Searching files prefixed by " + TEST_CASES_DEFINITION_FILES_SUFFIX + " in "
		      + dirname.getPath());
		File[] listFiles = getTestFiles(dirname);
		for (int i = 0; i < listFiles.length; i++) {
			File file = listFiles[i];
			if (file.getName().endsWith(TEST_CASES_DEFINITION_FILES_SUFFIX)) {
				isOk = validateTestDefinitionFormat(file);//validate the definition file
				resultSet.add(isOk);
				if(isOk){//if the data of this definition file is ok, fill retrieved data to map
					fillTestCaseFunctionsMap(file, theFunctionsByPaths, browsers, 1);
				}
			}
		}
		
		if(resultSet.contains(Boolean.TRUE)){//Check that at least one definition file worked well
			//retrieve testers data ...
			String testersValue = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_TESTERS, "tester1,tester2");
			String[] testers = testersValue.split(INPUT_VALUE_SEPARATOR);		
	
			List<String> functionIds = new ArrayList<String>(theFunctionsByPaths.keySet());
			Collections.sort(functionIds);
	
			//Get the template content ...
			File file = new File(TEST_PLAN_TEMPLATE);
			InputStream template = TestCaseHelper.class.getResourceAsStream("/"+TEST_PLAN_TEMPLATE);
			convertInputStreamToFile(file, template);		
			
			//Apply this template to the new spreadSheet ...
			SpreadSheet testPlanSpreadSheet = SpreadSheet.createFromFile(file);
			
			//Create a new sheet in this spreadsheet ...
			final Sheet sheet = testPlanSpreadSheet.getSheet(0);			
			//fix bug TCM-1 : Change the name of the sheet
			sheet.getElement().setAttribute("name", "TestPlan for "+testCampaignDirName, testPlanSpreadSheet.getNS().getTABLE());
			
			final Sheet sheetColorPicker = testPlanSpreadSheet.getSheet("ColorPicker");//Color picker sheet
			Map<String,String> colorMap = getColorPickerList(sheetColorPicker);//colorMap
			
			//Count the number of columns are already presented ...
			int headerRow = 3;
			int presentedColumnCount = -1;
			Object valueAt = null;
			while (!"".equals(valueAt)) {	
				presentedColumnCount++;
				valueAt = sheet.getValueAt(presentedColumnCount, headerRow);			
			}
	
			// Add some columns
			int currentColumnCount = presentedColumnCount + testers.length;
			sheet.ensureColumnCount(currentColumnCount + 1);
			for (int i = 0; i < testers.length; i++) {
				sheet.setValueAt(testers[i], presentedColumnCount + i, headerRow);
			}
			sheet.setValueAt(COLUMN_NAME_PATH, currentColumnCount, headerRow);

			int startDataRow = 4;
			int rowIndex = startDataRow;//start index of row
			
			//Multiple by 3 for displaying 3 priority types information for each testcase
			sheet.ensureRowCount(functionIds.size()*3 + rowIndex);
			
			for (String path : functionIds) {
				TestCaseFunctionInfo functionInfo = theFunctionsByPaths.get(path);
				//Set high priority row
				sheet.setValueAt(functionInfo.getId(), 0, rowIndex);					//set testcase ID
				sheet.setValueAt(functionInfo.getRelatedFunction(), 1, rowIndex);		//set function
				sheet.setValueAt(PRIORITY_HIGH, 2, rowIndex);
				sheet.setValueAt(functionInfo.getHighPriorityNumber(), 3, rowIndex);	//set number of testcase count for this priority
				sheet.setValueAt(functionInfo.getPath(), currentColumnCount, rowIndex);
				rowIndex++;
				
				//Set medium priority row
				sheet.setValueAt(functionInfo.getId(), 0, rowIndex);					//set testcase ID
				sheet.setValueAt(functionInfo.getRelatedFunction(), 1, rowIndex);		//set function
				sheet.setValueAt(PRIORITY_MEDIUM, 2, rowIndex);
				sheet.setValueAt(functionInfo.getMediumPriorityNumber(), 3, rowIndex);	//set number of testcase count for this priority
				sheet.setValueAt(functionInfo.getPath(), currentColumnCount, rowIndex);
				rowIndex++;			
				
				//Set low priority row
				sheet.setValueAt(functionInfo.getId(), 0, rowIndex);					//set testcase ID
				sheet.setValueAt(functionInfo.getRelatedFunction(), 1, rowIndex);		//set function
				sheet.setValueAt(PRIORITY_LOW, 2, rowIndex);
				sheet.setValueAt(functionInfo.getLowPriorityNumber(), 3, rowIndex);		//set number of testcase count for this priority
				sheet.setValueAt(functionInfo.getPath(), currentColumnCount, rowIndex);
				rowIndex++;
			}
	
			//A formula at the end of page gives number of Tests per tester (with "x" as the trigger) processing ....
			int endDataRow = rowIndex;
			int priorityColumnPosition = 2;//Defaul position in TestPlan file		
			
			
			/**
			 * Note for using
			 * Print total testcase with specific priority 
			 * E column here is the first column of testers columns list
			 * Example:
			 * Use this formula:  =SUMIF(C2:C196;"High";D2:D196)
			 * Print total assigned testcase for each tester
			 * Use this formula:  =SUMPRODUCT((C2:C196="High")*(E2:E196="x")*D2:D196)
			 */
			//Print total testcase with priority high
			rowIndex = 0;//Reset rowIndex
			sheet.setValueAt(TOTAL_HIGH, priorityColumnPosition, rowIndex);		
			sheet.getCellAt(priorityColumnPosition, rowIndex).setStyleName(colorMap.get("lightGreen"));
			sheet.setValueAt("=SUMIF(C"+(startDataRow+1)+":C"+endDataRow+";\""+PRIORITY_HIGH+"\";D"+(startDataRow+1)+":D"+endDataRow+")", priorityColumnPosition + 1, rowIndex);
			rowIndex++;
			
			//Print total testcase with priority medium
			sheet.setValueAt(TOTAL_MEDIUM, priorityColumnPosition, rowIndex);
			sheet.getCellAt(priorityColumnPosition, rowIndex).setStyleName(colorMap.get("yellow"));
			sheet.setValueAt("=SUMIF(C"+(startDataRow+1)+":C"+endDataRow+";\""+PRIORITY_MEDIUM+"\";D"+(startDataRow+1)+":D"+endDataRow+")", priorityColumnPosition + 1, rowIndex);
			rowIndex++;
			
			//Print total testcase with priority low
			sheet.setValueAt(TOTAL_LOW, priorityColumnPosition, rowIndex);
			sheet.getCellAt(priorityColumnPosition, rowIndex).setStyleName(colorMap.get("blue"));
			sheet.setValueAt("=SUMIF(C"+(startDataRow+1)+":C"+endDataRow+";\""+PRIORITY_LOW+"\";D"+(startDataRow+1)+":D"+endDataRow+")", priorityColumnPosition + 1, rowIndex);
			rowIndex++;		
			
			char startTesterColumnChar = 'E';
			int startTesterColumnCodePointAt = (int) startTesterColumnChar;
			
			rowIndex-=3;//go back 3 lines for adding more statistic data		
			//Add total high priority for each tester
			for(int i=0;i<testers.length;i++){			
				sheet.setValueAt("=SUMPRODUCT((C"+(startDataRow+1)+":C"+endDataRow+"=\""+PRIORITY_HIGH+"\")*("+((char)(startTesterColumnCodePointAt+i))+(startDataRow+1)+":"+((char)(startTesterColumnCodePointAt+i))+endDataRow+"=\"x\")*D"+(startDataRow+1)+":D"+endDataRow+")", priorityColumnPosition + 2 + i, rowIndex);
			}
			rowIndex++;
			
			//Add total medium priority for each tester
			for(int i=0;i<testers.length;i++){			
				sheet.setValueAt("=SUMPRODUCT((C"+(startDataRow+1)+":C"+endDataRow+"=\""+PRIORITY_MEDIUM+"\")*("+((char)(startTesterColumnCodePointAt+i))+(startDataRow+1)+":"+((char)(startTesterColumnCodePointAt+i))+endDataRow+"=\"x\")*D"+(startDataRow+1)+":D"+endDataRow+")", priorityColumnPosition + 2 + i, rowIndex);
			}
			rowIndex++;
			
			//Add total low priority for each tester
			for(int i=0;i<testers.length;i++){			
				sheet.setValueAt("=SUMPRODUCT((C"+(startDataRow+1)+":C"+endDataRow+"=\""+PRIORITY_LOW+"\")*("+((char)(startTesterColumnCodePointAt+i))+(startDataRow+1)+":"+((char)(startTesterColumnCodePointAt+i))+endDataRow+"=\"x\")*D"+(startDataRow+1)+":D"+endDataRow+")", priorityColumnPosition + 2 + i, rowIndex);
			}
			rowIndex++;
			
			final int COLUMN_TC_COUNT_INDEX = 3;
			//Reset style to default style for rows after header row
			for(int i = 4; i< sheet.getRowCount(); i++){
				for(int j = 0;j<sheet.getColumnCount();j++){
					if(((i-1)/3)%2==0){
						sheet.getCellAt(j, i).setStyleName("Default");
					}else{
						if(j==COLUMN_TC_COUNT_INDEX){
							sheet.getCellAt(j, i).setStyleName(colorMap.get("grayRight"));
						}else{
							sheet.getCellAt(j, i).setStyleName(colorMap.get("gray"));
						}
					}
				}
			}
			
			saveAsFile(testPlanSpreadSheet, testCampaignDirName + "/", testCampaignDirName + TEST_PLAN_SUFFIX,
			      false);
		}
	}
	
	/**
	 * Function to add content to file from InputStream 
	 * */
	private void convertInputStreamToFile(File file, InputStream inputStream){
		OutputStream out;
		try {
			out = new FileOutputStream(file);
			byte buf[]=new byte[1024];
		    int len;
		    while((len=inputStream.read(buf))>0)
		    out.write(buf,0,len);
		    out.close();
		    inputStream.close();
		}catch (IOException e){}	    
	}
	
	/**
	 * Function to read information from testcampaign.properties file 
	 * */
	private Properties readTestCampaignInfo(String testCampaignDirName) throws IOException, FileNotFoundException {
		Properties properties = new Properties();

		File file = new File(testCampaignDirName, TESTCAMPAIGN_PROPERTIES_FILENAME);
		logger.log(Level.FINE, " Loading test campaign params in " + file.getPath());
		properties.load(new FileInputStream(file));
		return properties;
	}

	/**
	 * Function to get all file in the specific directory
	 * */
	private File[] getTestFiles(File testDefDir) {
		if (!testDefDir.isDirectory()) {
			throw new RuntimeException("[ERROR] Dir expected for " + testDefDir);
		}
		return testDefDir.listFiles();
	}

	/**
	 * Function check to see that this sheet is testcaseSheet or not
	 * */
	public boolean isTestCaseSheet(String sheetName) {
		return !("Summary_Report".equals(sheetName) || "Notes".equals(sheetName) || "Cover".equals(sheetName)
		      || "Check_List".equals(sheetName) || sheetName.contains("#"));
	}

	/**
	 * Function to fill data to the testcaseFunction map
	 * */
	private void fillTestCaseFunctionsMap(File testDefinitionFile,
	      Map<String, TestCaseFunctionInfo> theFunctionsById, String[] browsers, int functionIndex) throws IOException {
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testDefinitionFile);
		String fileName = testDefinitionFile.getName();
		String filePath = testDefinitionFile.getPath();
		logger.log(Level.FINE, " Parsing file:" + filePath);
		String functionGroup = fileName.split("_")[functionIndex];

		int sheetCount = spreadSheet.getSheetCount();
		Sheet summarySheet = null;
		
		for (int i = 0; i < sheetCount; i++) {
			final Sheet sheet = spreadSheet.getSheet(i);
			int rowCount = sheet.getRowCount();
			int columnCount = sheet.getColumnCount();
			String sheetName = getSheetName(sheet);
			//String sheetName = sheet.getName();
			
			if ("Summary_Report".equals(sheetName))
				summarySheet = sheet;

			if (!isTestCaseSheet(sheetName))
				continue;

			//Ignore empty or missing data sheet
			if (columnCount < 9 || rowCount < 7){
				System.err.println("[ERROR] Empty data in sheet "+ sheetName);
				continue;
			}
			
			int headerRow = 5;
			int columnTester = -1;
			for (int c = 0; c < columnCount - 1 && rowCount > headerRow; c++) {
				String columnName = "" + sheet.getValueAt(c, headerRow);
				if ("Tester".equals(columnName))
					columnTester = c;
				if ("".equals(columnName))
					break;
			}

			String previousPath = null;
			TestCaseFunctionInfo functionInfo = null;
			Float currentCaseId = new Float(0);
			int autoCaseNo = 1;
			String comments = "";
			for (int row = headerRow + 1; row > 2 && row < rowCount; row++) {
				String relatedFunction = "" + sheet.getValueAt(COLUMN_RELATED_FUNCTION_INDEX, row);
				Object caseId = sheet.getValueAt(COLUMN_CASE_ID_INDEX, row);
				String newPath = "" + sheet.getValueAt(COLUMN_PATH_INDEX, row);
				String stepDesc = "" + sheet.getValueAt(COLUMN_STEP_DESC_INDEX, row);				
				String priority = "" + sheet.getValueAt(COLUMN_PRIORITY_INDEX, row);

				// Stop on first empty path
				String message = "row " + (row+1) + " in sheet " + sheetName + " of " + filePath;
				if ("".equals(newPath)) {
					if (!("".equals(relatedFunction)) || !("".equals(stepDesc))) {
						System.err.println("[ERROR] Empty cell in " + message);
					}
				} else {
					if ("".equals(relatedFunction))
						System.err.println("[ERROR] Empty related function : " + message);
					boolean isDifference = false;
					if (!newPath.equals(previousPath)) {
						comments = "";
						isDifference = true;
						functionInfo = theFunctionsById.get(relatedFunction);
						if (functionInfo != null && !functionInfo.getPath().equals(newPath))
							System.err.println("[ERROR] Related Function/Path not identical for " + relatedFunction
							      + " had  [" + functionInfo.getPath() + "] expected [" + newPath + "] " + message);
						else if (functionInfo == null) {
							functionInfo = new TestCaseFunctionInfo(newPath, relatedFunction);//only one priority
						}						
						theFunctionsById.put(relatedFunction, functionInfo);
						currentCaseId = new Float(0);
						previousPath = newPath;
					}
					
					//Add priority to priorityList for further processing...
					if(functionInfo!=null){
						functionInfo.getPriorityList().add(priority.trim().toUpperCase());
					}
					
					if (!(caseId instanceof Float))
						System.err.println("[ERROR] Wrong CaseID [" + caseId + "] " + message);
					else {
						if (((Float)caseId).intValue() == 0)
							System.err.println("[ERROR] Wrong CaseID value [" + caseId + "] " + message);
						else {
							if (!currentCaseId.equals((Float)caseId)) {
								currentCaseId = (Float) caseId;
								//Priority types processing ...
								if(priority.trim().equalsIgnoreCase(PRIORITY_HIGH)){
									functionInfo.increaseHighPriorityNumber();
								}else if(priority.trim().equalsIgnoreCase(PRIORITY_MEDIUM)){
									functionInfo.increaseMediumPriorityNumber();
								}else if(priority.trim().equalsIgnoreCase(PRIORITY_LOW)){
									functionInfo.increaseLowPriorityNumber();
								}							
							}else{
								if(!isDifference){
									if(priority.trim().equalsIgnoreCase(PRIORITY_HIGH)){
										functionInfo.increaseHighPriorityNumber();
									}else if(priority.trim().equalsIgnoreCase(PRIORITY_MEDIUM)){
										functionInfo.increaseMediumPriorityNumber();
									}else if(priority.trim().equalsIgnoreCase(PRIORITY_LOW)){
										functionInfo.increaseLowPriorityNumber();
									}
								}
							}
						}
					}
					
					//TestCaseRun files processing...
					if (columnTester > 0) {
						int colShiftBrowser = 1;
						String tester = "" + sheet.getValueAt(columnTester, row);
						for (String browser : browsers) {
							int column = columnTester + colShiftBrowser++;							
							String browserResult = "" + sheet.getValueAt(column, row);							
							boolean result =false;
							if(priority.trim().equalsIgnoreCase(PRIORITY_HIGH)){
								result = functionInfo.increaseTestResult(browser, "" + (autoCaseNo++), browserResult, PRIORITY_HIGH, functionInfo, message);
							}
							if(priority.trim().equalsIgnoreCase(PRIORITY_MEDIUM)){
								result = functionInfo.increaseTestResult(browser, "" + (autoCaseNo++), browserResult, PRIORITY_MEDIUM, functionInfo, message);
							}
							if(priority.trim().equalsIgnoreCase(PRIORITY_LOW)){
								result = functionInfo.increaseTestResult(browser, "" + (autoCaseNo++), browserResult, PRIORITY_LOW, functionInfo, message);
							}
							if (!result)
								System.err.println("[ERROR] Wrong result value : [" + browserResult + "] column " + column + " "
								      + message);
						}						
						// Read Comment
						String comment = "" + sheet.getValueAt(columnTester + colShiftBrowser, row);
						if (comment.trim().length() > 0){
							//highCaseNo = 1;
							//mediumCaseNo = 1;
							//lowCaseNo = 1;
							//For browser result sheets							
							//CaseId format
							if(Float.valueOf(caseId.toString()).intValue()>99){
								comments = tester+ " commented at CaseId : "+ Float.valueOf(caseId.toString()).intValue() +"\n\t"+comment+"\n";
							}else if (Float.valueOf(caseId.toString()).intValue()>9){
								comments = tester+ " commented at CaseId : 0"+ Float.valueOf(caseId.toString()).intValue() +"\n\t"+comment+"\n";
							}else{
								comments = tester+ " commented at CaseId : 00"+ Float.valueOf(caseId.toString()).intValue() +"\n\t"+comment+"\n";
							}
							functionInfo.setComments(functionInfo.getComments()+comments);
							
							//For feedbacks sheet
							//comments = tester+ " commented : " +"\n\t"+comment+"\n";
							functionInfo.addComment(Float.valueOf(caseId.toString()).intValue(), comment);
						}
					}
				}
			}
		}

		// Fill in the Descriptions
		int rowCount = summarySheet.getRowCount();
		int columnCount = summarySheet.getRowCount();
		int rowHeader = -1;
		int functionaliTyColumn = -1;
		int idColumn = 0;
		// Detect row header, and functionality index
		for (int row = 0; row < rowCount; row++) {
			String columnName = "" + summarySheet.getValueAt(idColumn, row);
			if ("Id".equalsIgnoreCase(columnName)) {
				rowHeader = row;
				for (int col = rowHeader; col < columnCount; col++) {
					columnName = "" + summarySheet.getValueAt(col, row);
					if ("Functionality".equals(columnName)) {
						functionaliTyColumn = col;
						break;
					}
				}
				break;
			}
		}
		if (functionaliTyColumn > 0) {
			for (int row = rowHeader + 1; row < rowCount; row++) {
				String id = "" + summarySheet.getValueAt(idColumn, row);
				String functionality = "" + summarySheet.getValueAt(functionaliTyColumn, row);
				TestCaseFunctionInfo testCaseFunctionInfo = theFunctionsById.get(id);
				if (testCaseFunctionInfo != null) {
					testCaseFunctionInfo.setRelatedFunction(functionGroup + " - " + functionality);
				}
			}
		} else {
			System.err.println("[ERROR] Functionality column not found for " + filePath);
		}

	}

	/**
	 * Function to Verify that number of Planned = Assigned 
	 * @return 	true: 	all testcases already been assigned 
	 * 			false: 	still exists at least one testcase not been assigned
	 * @throws IOException 
	 * */
	private boolean verifyPlannedAssigned(String testCampaignDirName, String testPlanFileName) throws IOException{
		boolean isOk = true;
		File file = new File(testCampaignDirName, testPlanFileName);
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(file);
		final Sheet sheet = spreadSheet.getSheet(0);
		
		String sheetName = getSheetName(sheet);
		String filePath = file.getPath();
		int rowCount = sheet.getRowCount();
		
		//Read testCampaignInfo
		Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);
		//retrieve testers data ...
		String testersValue = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_TESTERS, "tester1,tester2");
		String[] testers = testersValue.split(INPUT_VALUE_SEPARATOR);
		
		// First tester column
		int testerColumnStart = 4;
		int testerColumnEnd = testerColumnStart + testers.length;
		int testCaseCountColumn = 3;
		
		int headerRow=3;	
		int idColumn = 0;
		
		int errorCounter = 0;
		int fatalErrorCounter = 0;
		logger.log(Level.FINE, " Start verifying assigned status for testcases");
		for (int row = headerRow + 1; row < rowCount; row++) {
			String message = "row " + (row+1) + " in sheet \"" + sheetName + "\" of " + filePath;
			String id = "" + sheet.getValueAt(idColumn, row);
			Object testCaseCountValue = sheet.getValueAt(testCaseCountColumn, row);
			if(!"".equals(id)){
				boolean isAssigned = false;
				for(int i = testerColumnStart; i < testerColumnEnd; i++){
					String assignedSymbol = "" + sheet.getValueAt(i, row);
					if(!"".equals(assignedSymbol)&&assignedSymbol.equals("x")){//these testcases are assigned						
						isAssigned = true;						
					}else{
						if(testCaseCountValue instanceof Float){
							Float value = (Float) testCaseCountValue;
							if(value.intValue() == 0){//No need to check assigned with tcCount = 0
								isAssigned = true;
							}
						}else{
							errorCounter++;
							fatalErrorCounter++;
							System.err.println("[ERROR] Invalid data at column \"TC Count\" " + message);
						}
					}
					if(i == (testerColumnEnd-1) && isAssigned){
						//logger.log(Level.FINE, " TestCases at row " + (row+1) +" assigned");
					}
				}
				if(!isAssigned){
					isOk = false;
					errorCounter++;
					//System.err.println("[ERROR] TestCases at " + message + " are not assigned! Please re-check & assign them");
				}
			}else{
				break;
			}
		}
		logger.log(Level.FINE, " Finish verifying number Planned = Assigned\n The result is:");
		if(errorCounter>0){
			//logger.log(Level.FINE, " Total "+errorCounter+ " errors are detected!");
		}else{
			//logger.log(Level.FINE, " All planned testcases had been assigned!");
		}
		
		//Do this because of the requirement changing from Arthur
		if(fatalErrorCounter>0){
			System.err.println("[ERROR] Total "+fatalErrorCounter+ " fatal errors are detected, please correct them before generate TestCaseRun files!");
			isOk = false;
		}else{
			logger.log(Level.FINE, " All planned testcases had been assigned!");
			isOk = true;
		}
		
		
		return isOk;
	}
	
	/**
	 * Function to generate TestCaseRun
	 * According to the wiki page: http://wiki-int.exoplatform.org/display/exoTesting/Ideas+for+TestCampaignManager+2.0
	 * ==> P2 :TestCaseRun generator from TestPlan (1 TestCaseRun per tester) 
	 * 			input: TestPlan, TestCaseDefinition files
	 * 			output: One TestCaseRun file per tester: <test-campaign-dirname>/TestPlan/file_name.ods
	 * */
	public void generateTestCaseRun(String testCampaignDirName, String testPlanFileName) throws IOException {
		//Verify that number of Planned = Assigned 
		boolean isOk = verifyPlannedAssigned(testCampaignDirName, testPlanFileName);
		
		//Unblock generate TestCaseRun file process when all testcase are not assigned 
		if(isOk){
			// For each tester collect its Paths and generate a file
			File file = new File(testCampaignDirName, testPlanFileName);
			SpreadSheet spreadSheet = SpreadSheet.createFromFile(file);
			final Sheet sheet = spreadSheet.getSheet(0);
			int columnCount = sheet.getColumnCount();
	
			//Read testCampaignInfo
			Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);
	
			// First tester column
			int testerColumnStart = 4;
			int testFunctionIdColumn = 0;
			int testCaseCountColumn = 3;
			int priorityColumn = 2;
			int headerRow = 3;
			for (int column = testerColumnStart; column < columnCount; column++) {
				String testerColumnName = "" + sheet.getValueAt(column, headerRow);
				if (testerColumnName.length() == 0 || testerColumnName.equals(COLUMN_NAME_PATH))
					break;
				int rowCount = sheet.getRowCount();
				int tcCount = 0;
				String priorityValue = "";
				Set<TestCaseFunctionIdVsPriority> functionIdsForTester = new HashSet<TestCaseFunctionIdVsPriority>();
				
				//row start from 4 because we ignore 3 statistic lines + 1 header line
				for (int row = headerRow+1; row < rowCount;row++) {
					String functionId = "" + sheet.getValueAt(testFunctionIdColumn, row);
					String affectedToTester = "" + sheet.getValueAt(column, row);
					TestCaseFunctionIdVsPriority obj = new TestCaseFunctionIdVsPriority(); 
					if (functionId.length() == 0)
						break;
					if (affectedToTester.length() > 0) {
						tcCount += Float.valueOf("" + sheet.getValueAt(testCaseCountColumn, row)).intValue();
						priorityValue = "" + sheet.getValueAt(priorityColumn, row);
						obj.setFunctionId(functionId);
						obj.setPriority(priorityValue);
						functionIdsForTester.add(obj);
					}
				}
	
				logger.log(Level.FINE, " " + testerColumnName + ": " + tcCount + " TestCases");
				File[] listFiles = getTestFiles(new File(testCampaignDirName, TEST_DEFINITIONS_DIR));
				for (int i = 0; i < listFiles.length; i++) {
					File tcFile = listFiles[i];
					String testCaseDefinitionFileName = tcFile.getAbsolutePath();
					if (testCaseDefinitionFileName.endsWith(TEST_CASES_DEFINITION_FILES_SUFFIX)) {
						generateSingleTestCaseRun(testCaseDefinitionFileName, testerColumnName, functionIdsForTester,
						      testCampaignInfo, testCampaignDirName);
					}
				}
			}
		}
	}

	/**
	 * Function to generate singleTestCaseRun
	 * */
	private void generateSingleTestCaseRun(String testCaseDefinitionFileName, String testerName,
	      Set<TestCaseFunctionIdVsPriority> testerFunctions, Properties testCampaignInfo, String testCampaignDirName) throws IOException {

		// Parameters
		String testCampaignBrowsers = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_BROWSERS, "");
		String browsers[] = testCampaignBrowsers.split(INPUT_VALUE_SEPARATOR);

		// Load the file.
		File testCaseDefinitionFile = new File(testCaseDefinitionFileName);
		SpreadSheet spreadSheet = SpreadSheet.createFromFile(testCaseDefinitionFile);
		int sheetCount = spreadSheet.getSheetCount();

		int assigned = 0;
		Set<String> functions = new TreeSet<String>();
		// Iterate on the sheets
		for (int i = 1; i < sheetCount; i++) {
			final Sheet sheet = spreadSheet.getSheet(i);
			String sheetName = getSheetName(sheet);
			//String sheetName = sheet.getName();
			if (!isTestCaseSheet(sheetName))
				continue;

			int columnCount = sheet.getColumnCount();
			int headerRow = 5;

			for (int j = 1; j < columnCount; j++) {
				Object valueAt = sheet.getValueAt(j, headerRow);
				if ("".equals(valueAt)) {
					// Create the headers
					sheet.setValueAt("Tester", j, headerRow);
					int colShift = 1;
					for (String browser : browsers) {
						sheet.setValueAt(browser, j + colShift, headerRow);
						colShift++;
					}
					sheet.setValueAt("Comment/Bug", j + colShift, headerRow);

					int rowCount = sheet.getRowCount();
					for (int r = 6; r < rowCount; r++) {
						Object valueAtRow = sheet.getValueAt(COLUMN_RELATED_FUNCTION_INDEX, r);
						String functionIdValue = "" + sheet.getValueAt(COLUMN_RELATED_FUNCTION_INDEX, r);
						String priorityValue = "" + sheet.getValueAt(COLUMN_PRIORITY_INDEX, r);
						if ("".equals(valueAtRow) && "".equals(functionIdValue)) {
							break;
						}
						for(TestCaseFunctionIdVsPriority obj : testerFunctions){
							if(obj.getFunctionId().equals(functionIdValue) && obj.getPriority().equalsIgnoreCase(priorityValue)){
								// Values
								sheet.setValueAt(testerName, j, r);
								functions.add(functionIdValue);
								String step = "" + sheet.getValueAt(COLUMN_STEP_DESC_INDEX, r);
								if (!step.startsWith("Step ")){
									System.err.println("[ERROR] Wrong step content [" + step + "]. row " + (r+1) + " of sheet "
									      + sheetName + " of file " + testCaseDefinitionFile.getName());
								}else{
									assigned++;
								}
							}
						}
					}
					break;
				}
			}

		}
		String newFileName = TEST_CASES_RUN_FILES_PREFIX + testerName + "_" + assigned + "_"
		      + testCaseDefinitionFile.getName().replaceAll(TEST_CASES_DEFINITION_FILES_SUFFIX, ".ods");

		saveAsFile(spreadSheet, testCampaignDirName + "/" + TESTRUN_PLAN_DIR, newFileName, false);
		System.out.println("*[INFO] " + newFileName + " " + assigned + " cases. " + functions);

		File resultDir = new File(testCampaignDirName + "/" + TESTRUN_PERFORMED_DIR);
		resultDir.mkdirs();
		System.out.println("*[INFO] Created " + resultDir.getPath() + " for results.");
	}

    private static String getSheetName(Sheet sheet) {
        return sheet.getElement().getAttributeValue("name", sheet.getSpreadSheet().getNS().getTABLE());
    }

    /**
	 * Function to save spreadSheet content to file
	 * */
	public   void saveAsFile(final SpreadSheet spreadSheet, String targetDir, String fileName, boolean open)
	      throws IOException, FileNotFoundException {
		// Save to file and open it.
		File outputDir = new File(targetDir);
		outputDir.mkdirs();
		File outputFile = new File(outputDir, fileName);
		File fileSaved = spreadSheet.saveAs(outputFile);
		logger.log(Level.FINE, " Generated file:" + fileSaved.getPath());
		if (open)
			OOUtils.open(fileSaved);
	}

    public void listFailures(String testCampaignDirName, String testCampaignName) throws IOException {

        Map<String, TestCaseFunctionInfo> theFunctionsByIds = new HashMap<String, TestCaseFunctionInfo>();

        Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);
        String testersValue = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_TESTERS, "");
        String testCampaignBrowsers = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_BROWSERS, "");
        String browsers[] = testCampaignBrowsers.split(INPUT_VALUE_SEPARATOR);
        int browserLength = browsers.length;

        logger.log(Level.FINE, " Gathering TestCaseRun information.....");
        File[] listFiles = getTestFiles(new File(testCampaignDirName, TESTRUN_PERFORMED_DIR));
        for (int i = 0; i < listFiles.length; i++) {
            File file = listFiles[i];
            if (file.getName().startsWith(TEST_CASES_RUN_FILES_PREFIX)) {
                fillTestCaseFunctionsMap(file, theFunctionsByIds, browsers, 4);
            }
        }        

        List<String> functionIds = new ArrayList<String>(theFunctionsByIds.keySet());
        Collections.sort(functionIds);
        for (String id : functionIds) {
            TestCaseFunctionInfo functionInfo = theFunctionsByIds.get(id);
            int failedCount = 0;
            int blockedCount = 0;

            for (String browser : browsers) {
                failedCount += functionInfo.getTestFailedCount(browser, null);
                blockedCount += functionInfo.getTestBlockedCount(browser, null);
            }

            StringBuffer aggComments = new StringBuffer();
            Map<Integer, String> comments = functionInfo.getCommentMap();
            for (String comment : comments.values()) {
                String oneLine = comment.trim().replaceAll("\n","");
                aggComments.append(oneLine);
                if (oneLine.length()>0) {
                     aggComments.append(" ");
                }
            }
            
            if (failedCount + blockedCount > 0 || aggComments.toString().trim().length() > 0){
                System.out.println("| " + testCampaignName + "| " + id + " | " + functionInfo.getRelatedFunction() +" | " + (failedCount>0?"FAILED":(blockedCount>0?"BLOCKED":"PASSED")) + " | " + aggComments.toString()+ " |");
            }
        }
    }

	/**
	 * Function to generate TestPlanResult
	 * According to wiki page: http://wiki-int.exoplatform.org/display/exoTesting/Ideas+for+TestCampaignManager+2.0
	 * ==>P3: TestPlanResult generator from TestCaseRun files
	 * 			input: 	All the TestCaseRun, the TestPlan, Test Types performed
	 * 			output:	1. Overview template (wiki format compatible with wikibook)
	 * 					2. A focus on Functional tests (see existing)
	 * */
	public void generateTestPlanResult(String testCampaignDirName, String testCampaignName) throws IOException {

		Map<String, TestCaseFunctionInfo> theFunctionsByPaths = new HashMap<String, TestCaseFunctionInfo>();

		Properties testCampaignInfo = readTestCampaignInfo(testCampaignDirName);
		String testersValue = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_TESTERS, "");
		String testCampaignBrowsers = testCampaignInfo.getProperty(KEY_TEST_CAMPAIGN_BROWSERS, "");
		String browsers[] = testCampaignBrowsers.split(INPUT_VALUE_SEPARATOR);
		int browserLength = browsers.length;
		
		logger.log(Level.FINE, " Gathering TestCaseRun information.....");
		File[] listFiles = getTestFiles(new File(testCampaignDirName, TESTRUN_PERFORMED_DIR));
		for (int i = 0; i < listFiles.length; i++) {
			File file = listFiles[i];
			if (file.getName().startsWith(TEST_CASES_RUN_FILES_PREFIX)) {				
				fillTestCaseFunctionsMap(file, theFunctionsByPaths, browsers, 4);
			}
		}
		logger.log(Level.FINE, " Gathering TestCaseRun information successful.....");

		String[] testers = testersValue.split(INPUT_VALUE_SEPARATOR);

		int columnCount = testers.length + 3;
		String[] columns = new String[columnCount];
		for (int i = 0; i < testers.length; i++) {
			columns[i + 2] = testers[i];
			logger.log(Level.FINE, " Tester:" + testers[i]);
		}
		columns[columnCount - 1] = "Description";

		List<String> functionIds = new ArrayList<String>(theFunctionsByPaths.keySet());
		Collections.sort(functionIds);

		File file = new File(TEST_RESULT_TEMPLATE);
		InputStream template = TestCaseHelper.class.getResourceAsStream("/"+TEST_RESULT_TEMPLATE);
		convertInputStreamToFile(file, template);
		
		SpreadSheet testPlanResultSpreadSheet = SpreadSheet.createFromFile(file);
		
		final Sheet testResultSheet = testPlanResultSpreadSheet.getSheet(0);
		final Sheet sheetFeedbacks = testPlanResultSpreadSheet.getSheet(1);
		final Sheet sheetColorPicker = testPlanResultSpreadSheet.getSheet("ColorPicker");//Color picker sheet
		Map<String,String> colorMap = getColorPickerList(sheetColorPicker);//colorMap
		
		logger.log(Level.FINE, " Printing \"TestResults\" sheet....");
		
		// How many columns are already present
		int colShift = 0;
		testResultSheet.setValueAt("Id", colShift++, 8);
		testResultSheet.setValueAt("Function", colShift++, 8);
		testResultSheet.setValueAt("Priority", colShift++, 8);
		testResultSheet.setValueAt("TC Count", colShift++, 8);

		// Add some columns
		int expectedColumnCount = colShift + browsers.length * 5;
		int row = 9;
		testResultSheet.ensureRowCount(functionIds.size()*3 + row + 1);
		testResultSheet.ensureColumnCount(expectedColumnCount + 1);
		int colShiftBrowsers = 0;
		for (String browser : browsers) {
			testResultSheet.setValueAt(browser, colShift + colShiftBrowsers, 0);
			testResultSheet.setValueAt("PASSED", colShift + colShiftBrowsers++, 8);
			testResultSheet.setValueAt("FAILED", colShift + colShiftBrowsers++, 8);
			testResultSheet.setValueAt("BLOCKED", colShift + colShiftBrowsers++, 8);
			testResultSheet.setValueAt("N/A", colShift + colShiftBrowsers++, 8);
			testResultSheet.setValueAt("NOT RUN", colShift + colShiftBrowsers++, 8);
		}

        SummaryAggregator summaryAggregator = new SummaryAggregator(testCampaignName);
        //Print HEADER
        //System.out.println(SummaryAggregator.getSummaryHeader());
        
		for (String id : functionIds) {			
			TestCaseFunctionInfo functionInfo = theFunctionsByPaths.get(id);
			Set<String> priorityList = functionInfo.getPriorityList();			
			
			//Priority High
			if(priorityList.contains(PRIORITY_HIGH.toUpperCase())){
				colShiftBrowsers = 0;
				for (String browser : browsers) {			
					colShiftBrowsers = fillDataToResultSheet(testResultSheet, functionInfo, browser, browserLength, row, colShift, colShiftBrowsers, PRIORITY_HIGH, summaryAggregator);
				}
				row++;	
			}
			
			//Priority Medium
			if(priorityList.contains(PRIORITY_MEDIUM.toUpperCase())){
				colShiftBrowsers = 0;
				for (String browser : browsers) {			
					colShiftBrowsers = fillDataToResultSheet(testResultSheet, functionInfo, browser, browserLength, row, colShift, colShiftBrowsers, PRIORITY_MEDIUM, summaryAggregator);
				}
				row++;	
			}
			
			//Priority Low
			if(priorityList.contains(PRIORITY_LOW.toUpperCase())){
				colShiftBrowsers = 0;
				for (String browser : browsers) {			
					colShiftBrowsers = fillDataToResultSheet(testResultSheet, functionInfo, browser, browserLength, row, colShift, colShiftBrowsers, PRIORITY_LOW, summaryAggregator);
				}
				row++;	
			}
		}	

        System.out.println(summaryAggregator.getSummary());
		//Eat a small piece of time for user better looking :D... 
//        try {
//            Thread.sleep(250);
//        } catch (InterruptedException e) {
//            e.printStackTrace();
//        }
//		logger.log(Level.FINE, " Printing \"TestResults\" sheet successful....");
		
		//Set result for each browser, one sheet per browser
		int sheetBrowserIndex = 2;
		Sheet browserSheet;		
		int browserSheetRow = 1;
		
		for(String browser : browsers){
			logger.log(Level.FINE, " Printing \""+browser+"\" sheet...");
			browserSheet = testPlanResultSpreadSheet.getSheet(sheetBrowserIndex);	
			browserSheet.getElement().setAttribute("name", ""+browser, testPlanResultSpreadSheet.getNS().getTABLE());
			browserSheet.ensureRowCount(functionIds.size()*5*3 + 1);
			browserSheetRow = 1;
			for (String id : functionIds){				
				TestCaseFunctionInfo functionInfo = theFunctionsByPaths.get(id);				
				Set<String> priorityList = functionInfo.getPriorityList();
				
				//High priority
				if(priorityList.contains(PRIORITY_HIGH.toUpperCase())){
					browserSheetRow = fillDataToBrowserSheet(browserSheet, functionInfo, browser, browserLength, browserSheetRow, PRIORITY_HIGH, colorMap);
				}
				
				//Medium priority
				if(priorityList.contains(PRIORITY_MEDIUM.toUpperCase())){
					browserSheetRow = fillDataToBrowserSheet(browserSheet, functionInfo, browser, browserLength, browserSheetRow, PRIORITY_MEDIUM, colorMap);
				}
				
				//Low priority
				if(priorityList.contains(PRIORITY_LOW.toUpperCase())){
					browserSheetRow = fillDataToBrowserSheet(browserSheet, functionInfo, browser, browserLength, browserSheetRow, PRIORITY_LOW, colorMap);
				}
			}
			
			//Eat a small piece of time for user better looking :D... 
//            try {
//                Thread.sleep(250);
//            } catch (InterruptedException e) {
//                e.printStackTrace();
//            }
//			logger.log(Level.FINE, " Printing \""+browser+"\" sheet successful...");
			
			sheetBrowserIndex++;
		}
		
		
		logger.log(Level.FINE, " Printing \"Feedbacks\" sheet...");
		sheetFeedbacks.setValueAt("Id", 0, 6);
		sheetFeedbacks.setValueAt("Function", 1, 6);
		sheetFeedbacks.setValueAt("Comment", 2, 6);
		row = 7;
		sheetFeedbacks.ensureRowCount(functionIds.size());
		for (String path : functionIds) {
			TestCaseFunctionInfo functionInfo = theFunctionsByPaths.get(path);
			//String comments = functionInfo.getComments();
			Map<Integer, String> commentMap = functionInfo.getCommentMap();
			Set<Integer> caseNoSet = commentMap.keySet();
			TreeSet<Integer> treeSet = new TreeSet<Integer>(caseNoSet);
			
			for(Integer caseId : treeSet){
				sheetFeedbacks.ensureRowCount(row + 1);
				if(caseId>99){
					sheetFeedbacks.setValueAt(functionInfo.getId()+"."+caseId, 0, row);
				}else if(caseId>9){
					sheetFeedbacks.setValueAt(functionInfo.getId()+".0"+caseId, 0, row);
				}else{
					sheetFeedbacks.setValueAt(functionInfo.getId()+".00"+caseId, 0, row);
				}
				sheetFeedbacks.setValueAt(functionInfo.getRelatedFunction(), 1, row);
				sheetFeedbacks.setValueAt(commentMap.get(caseId), 2, row);
				row++;
			}
			
		}
		
		//Eat a small piece of time for user better looking :D... 
//        try {
//            Thread.sleep(250);
//        } catch (InterruptedException e) {
//            e.printStackTrace();
//        }
		logger.log(Level.FINE, " Printing \"Feedbacks\" sheet successful...");
		
		saveAsFile(testPlanResultSpreadSheet, testCampaignDirName + "/", testCampaignDirName
		      + TEST_RESULT_SUFFIX, false);
	}
	
	/**
	 * Function to fill data to TestResults sheet
	 * */
	private int fillDataToResultSheet(Sheet testResultSheet, TestCaseFunctionInfo functionInfo,String browser, int browserLength, int row, int colShift, int colShiftBrowsers, String priorityType, SummaryAggregator summaryAggregator){
		int returnedColShiftBrowsers = colShiftBrowsers;
		
		//Set data for TestResult Sheet	
		testResultSheet.setValueAt(functionInfo.getId(), 0, row);
		testResultSheet.setValueAt(functionInfo.getRelatedFunction(), 1, row);
        if (functionInfo.getRelatedFunction().length()==0) {
            System.err.println("[ERROR] Missing Related Function for " + functionInfo.getId() + " (Please fix Definition file in summary).");
        }
		testResultSheet.setValueAt(priorityType, 2, row);				
		
		if(priorityType.equalsIgnoreCase(PRIORITY_HIGH)){
			testResultSheet.setValueAt(functionInfo.getHighPriorityNumber()/browserLength, 3, row);
		}else if(priorityType.equalsIgnoreCase(PRIORITY_MEDIUM)){
			testResultSheet.setValueAt(functionInfo.getMediumPriorityNumber()/browserLength, 3, row);
		}else if(priorityType.equalsIgnoreCase(PRIORITY_LOW)){
			testResultSheet.setValueAt(functionInfo.getLowPriorityNumber()/browserLength, 3, row);
		}

        // Count Defined TestCases
		summaryAggregator.addTestcase(functionInfo.getTestCount(browser,priorityType));

		// browser + "-P"
		testResultSheet.setValueAt(functionInfo.getTestPassedCount(browser,priorityType), colShift + returnedColShiftBrowsers++, row);
        summaryAggregator.addPassed(functionInfo.getTestPassedCount(browser,priorityType));

		// browser + "-F"
		testResultSheet.setValueAt(functionInfo.getTestFailedCount(browser,priorityType), colShift + returnedColShiftBrowsers++, row);
        summaryAggregator.addFailed(functionInfo.getTestFailedCount(browser,priorityType));

		// browser + "-B"
		testResultSheet.setValueAt(functionInfo.getTestBlockedCount(browser,priorityType), colShift + returnedColShiftBrowsers++, row);
        summaryAggregator.addBlocked(functionInfo.getTestBlockedCount(browser,priorityType));

		// browser + "-NA"
		testResultSheet.setValueAt(functionInfo.getTestNACount(browser,priorityType), colShift + returnedColShiftBrowsers++, row);
        summaryAggregator.addNA(functionInfo.getTestNACount(browser,priorityType));

		// browser + "-NR"
		testResultSheet.setValueAt(functionInfo.getTestNotRunCount(browser,priorityType), colShift + returnedColShiftBrowsers++, row);				
        summaryAggregator.addNotrun(functionInfo.getTestNotRunCount(browser,priorityType));

		return returnedColShiftBrowsers;
	}
	
	/**
	 * Function to fill functionInfo data to browser result sheet
	 * */
	private int fillDataToBrowserSheet(Sheet browserSheet, TestCaseFunctionInfo functionInfo,String browser, int browserLength, int browserSheetRow, String priorityType, Map<String,String> colorMap){
		int startIndex =3;
		int colShiftIndex;
		int returnedBrowserSheetRow = browserSheetRow;
		
		//Set data for browser Sheet				
		browserSheet.setValueAt(functionInfo.getId(), 0, returnedBrowserSheetRow);
		//browserSheet.getCellAt(0, returnedBrowserSheetRow).setStyleName(colorMap.get("gray"));//Set format color
		browserSheet.setValueAt(priorityType, 1, returnedBrowserSheetRow);
		//browserSheet.getCellAt(1, returnedBrowserSheetRow).setStyleName(colorMap.get("gray"));//Set format color
		browserSheet.setValueAt(functionInfo.getComments(), 6, returnedBrowserSheetRow);
		
		int totalNumber = -1;
		if(priorityType.equalsIgnoreCase(PRIORITY_HIGH)){
			totalNumber = functionInfo.getHighPriorityNumber()/browserLength;
			browserSheet.setValueAt(functionInfo.getHighPriorityNumber()/browserLength, 2, returnedBrowserSheetRow);			
		}else if(priorityType.equalsIgnoreCase(PRIORITY_MEDIUM)){
			totalNumber = functionInfo.getMediumPriorityNumber()/browserLength;
			browserSheet.setValueAt(functionInfo.getMediumPriorityNumber()/browserLength, 2, returnedBrowserSheetRow);
		}else if(priorityType.equalsIgnoreCase(PRIORITY_LOW)){
			totalNumber = functionInfo.getLowPriorityNumber()/browserLength;
			browserSheet.setValueAt(functionInfo.getLowPriorityNumber()/browserLength, 2, returnedBrowserSheetRow);
		}				
		//Set format color
		//browserSheet.getCellAt(2, returnedBrowserSheetRow).setStyleName(colorMap.get("grayRight"));
		
		// browser + "-P"
		colShiftIndex = 0;
		browserSheet.setValueAt("Passed", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(functionInfo.getTestPassedCount(browser,priorityType), startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(round((float)functionInfo.getTestPassedCount(browser,priorityType)*100/totalNumber,2)+"%", startIndex + colShiftIndex++, returnedBrowserSheetRow);	
		returnedBrowserSheetRow++;
		
		// browser + "-F"
		startIndex = 3;
		colShiftIndex = 0;
		browserSheet.setValueAt("Failed", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		
		//Add function cell to this row
		browserSheet.setValueAt(functionInfo.getRelatedFunction(), 0, returnedBrowserSheetRow);
		browserSheet.setValueAt(functionInfo.getTestFailedCount(browser,priorityType), startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(round((float)functionInfo.getTestFailedCount(browser,priorityType)*100/totalNumber,2)+"%", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		returnedBrowserSheetRow++;
		
		// browser + "-B"
		colShiftIndex = 0;
		browserSheet.setValueAt("Blocked", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(functionInfo.getTestBlockedCount(browser,priorityType), startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(round((float)functionInfo.getTestBlockedCount(browser,priorityType)*100/totalNumber,2)+"%", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		returnedBrowserSheetRow++;
		
		// browser + "-NA"
		colShiftIndex = 0;
		browserSheet.setValueAt("N/A", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(functionInfo.getTestNACount(browser,priorityType), startIndex + colShiftIndex++, returnedBrowserSheetRow);
        browserSheet.setValueAt(round((float)functionInfo.getTestNACount(browser,priorityType)*100/totalNumber,2)+"%", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		returnedBrowserSheetRow++;
		
		// browser + "-NR"
		colShiftIndex = 0;
		browserSheet.setValueAt("Not Run", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(functionInfo.getTestNotRunCount(browser,priorityType), startIndex + colShiftIndex++, returnedBrowserSheetRow);
		browserSheet.setValueAt(round((float)functionInfo.getTestNotRunCount(browser,priorityType)*100/totalNumber,2)+"%", startIndex + colShiftIndex++, returnedBrowserSheetRow);
		returnedBrowserSheetRow++;
		
		return returnedBrowserSheetRow;
	}
	
	/**
	 * Function to get color style Name from colorPickerSheet
	 * */
	private Map<String,String> getColorPickerList(Sheet colorPickerSheet){		
		Map<String,String> colorStyleMap = new HashMap<String, String>();
		
		//Row 1
		int colorIndex = 0;		
		colorStyleMap.put("red", colorPickerSheet.getCellAt(colorIndex++,0).getStyleName());//Red color
		colorStyleMap.put("yellow", colorPickerSheet.getCellAt(colorIndex++,0).getStyleName());//Yellow color
		colorStyleMap.put("green", colorPickerSheet.getCellAt(colorIndex++,0).getStyleName());//Green color
		colorStyleMap.put("lightGreen", colorPickerSheet.getCellAt(colorIndex++,0).getStyleName());//Light green color
		colorStyleMap.put("oceanBlue", colorPickerSheet.getCellAt(colorIndex++,0).getStyleName());//Ocean blue color
		colorStyleMap.put("blue", colorPickerSheet.getCellAt(colorIndex++,0).getStyleName());//Blue color		
		colorStyleMap.put("gray", colorPickerSheet.getCellAt(colorIndex++,0).getStyleName());//Gray color
		
		//Row 2
		colorIndex = 0;
		colorStyleMap.put("redRight", colorPickerSheet.getCellAt(colorIndex++,1).getStyleName());//Red color with right alignment
		colorStyleMap.put("yellowRight", colorPickerSheet.getCellAt(colorIndex++,1).getStyleName());//Yellow color with right alignment
		colorStyleMap.put("greenRight", colorPickerSheet.getCellAt(colorIndex++,1).getStyleName());//Green color with right alignment
		colorStyleMap.put("lightGreenRight", colorPickerSheet.getCellAt(colorIndex++,1).getStyleName());//Light green color with right alignment
		colorStyleMap.put("oceanBlueRight", colorPickerSheet.getCellAt(colorIndex++,1).getStyleName());//Ocean blue color with right alignment
		colorStyleMap.put("blueRight", colorPickerSheet.getCellAt(colorIndex++,1).getStyleName());//Blue color with right alignment		
		colorStyleMap.put("grayRight", colorPickerSheet.getCellAt(colorIndex++,1).getStyleName());//Gray color with right alignment
		
		return colorStyleMap;
	}
	
	/**
     * Round a double value to a specified number of decimal 
     * places.
     *
     * @param val the value to be rounded.
     * @param places the number of decimal places to round to.
     * @return val rounded to places decimal places.
     */
    private double round(double val, int places) {
		long factor = (long)Math.pow(10,places);
	
		// Shift the decimal the correct number of places
		// to the right.
		val = val * factor;
	
		// Round to the nearest integer.
		long tmp = Math.round(val);
	
		// Shift the decimal the correct number of places
		// back to the left.
		return (double)tmp / factor;
    }

    /**
     * Round a float value to a specified number of decimal 
     * places.
     *
     * @param val the value to be rounded.
     * @param places the number of decimal places to round to.
     * @return val rounded to places decimal places.
     */
    private float round(float val, int places) {
    	return (float)round((double)val, places);
    }
}
