#!/bin/bash
#
# $Id: cargosh-plf-setup.sh 8959 2010-10-22 11:10:53Z hgomez $
#
# Copyright (C) 2003-2010 eXo Platform SAS.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see<http://www.gnu.org/licenses/>.#
#
#
# Purpose : Setup a Tomcat PLF single or clustering environnement for QA teams
#

echo CARGOSH-PLF-SETUP cargosh-binaries 0.3

#
# Default vars
#
NUM_INSTANCES=1

ARTIFACT="exo.platform.packaging.delivery"
VERSION="3.0.2"

CLEARDATA=false
CLUSTERING=false
MYSQL=false

DB_HOST="localhost"
DB_LOGIN="root"
DB_PASSWORD=""

USER_LOGIN=
USER_PASSWORD=

MIN_IN_POOL=
MAX_IN_POOL=
MAX_WAIT_IN_POOL=
MAX_HTTPAJP_CONNS=

SETUP_NAME="plf"
ZIP_NAME="eXoPlatform"

#
# Decode command line parameters
#
usage()
{
cat << EOF
usage: $0 options

This script setup PLF for QA teams.

OPTIONS:
   -h      Show this message
   -a      artifact (default: $ARTIFACT)
   -v      version (default: $VERSION)
   -m      select mysql (default: keep package sql settings)
   -u      activate clustering (default: $CLUSTERING)
   -i      instances number (default: 1)
   -n      number of instances (default: 1)
   -c      clear gatein data directory (default: $CLEARDATA)
   -l      db login (default: $DB_LOGIN)
   -p      db password (default: $DB_PASSWORD)
   -d      db hostl (default: $DB_HOST)
   -s      setup name (default: $SETUP_NAME)
   -I      min idle entries in DataSources pool (default: keep distribution default)
   -A      max entries in DataSources pool (default: keep distribution default)
   -W      max wait in DataSources pool in millis (default: keep distribution default)
   -H      max HTTP/AJP connections (default: keep distribution default)
   -i      instance number
   -z      zip name (default: $ZIP_NAME)
EOF
}

#
# Decode command line parameters
#
init()
{
    cygwin=false;
    linux=false;
    darwin=false;
    case "`uname`" in
    CYGWIN*) cygwin=true;;
    Linux*) linux=true;;
    Darwin*) darwin=true;;
    esac

    CURR_DATE=`date +%Y%m%d%H%M%S`

    #
    # without parameters, provide help
    #
    if [ $# = 0 ]; then
      usage
      exit 1;
    fi

    while getopts "hmuca:v:s:n:i:l:p:d:I:A:W:H:z:" OPTION
    do
         case $OPTION in
             h)
                 usage
                 exit 1
                 ;;
             m)
                 MYSQL=true
                 ;;
             u)
                 CLUSTERING=true
                 ;;
             c)
                 CLEARDATA=true
                 ;;
             a)
                 ARTIFACT=$OPTARG
                 ;;
             v)
                 VERSION=$OPTARG
                 ;;
             s)
                 SETUP_NAME=$OPTARG
                 ;;
             n)
                 NUM_INSTANCES=$OPTARG
                 INSTANCE_NUMBER=""
                 ;;
             i)
                 INSTANCE_NUMBER=$OPTARG
                 NUM_INSTANCES=""
                 ;;
             l)
                 DB_LOGIN=$OPTARG
                 ;;
             p)
                 DB_PASSWORD=$OPTARG
                 ;;
             d)
                 DB_HOST=$OPTARG
                 ;;
             I)
                 MIN_IN_POOL=$OPTARG
                 ;;
             A)
                 MAX_IN_POOL=$OPTARG
                 ;;
             W)
                 MAX_WAIT_IN_POOL=$OPTARG
                 ;;
             H)
                 MAX_HTTPAJP_CONNS=$OPTARG
                 ;;
             z)
                 ZIP_NAME=$OPTARG
                 ;;
             ?)
                 usage
                 exit
                 ;;
         esac
    done
}


#
# Prepare dirs
#
prepare()
{
    INSTALL_DIR=`pwd`/target/install

    if [ ! -z "$HUDSON_HOME" ]; then

        if [ -d $HUDSON_HOME ]; then
            INSTALL_DIR=$HUDSON_HOME/$SETUP_NAME
        fi

    fi

    if [ ! -z "$EXO_INSTALL_HOME" ]; then

        if [ -d $EXO_INSTALL_HOME ]; then
            INSTALL_DIR=$EXO_INSTALL_HOME
        fi

    fi

    echo "using $INSTALL_DIR to setup Tomcat PLF"

    # directory where all jars/wars are stored (if not defined outside, set to ./addons)
    if [ -z "$REF_BINARIES_DIR" ]; then
        REF_BINARIES_DIR=`pwd`/target/binaries
    fi

    # data directory
    GATEIN_DATA_DIR=$INSTALL_DIR/datas
    echo "using $GATEIN_DATA_DIR to store Gatein datas"

    mkdir -p $INSTALL_DIR
    mkdir -p $REF_BINARIES_DIR
    mkdir -p $GATEIN_DATA_DIR

    # handle case of 3.0.2-20101206.171743-13 provided to get 3.0.2-SNAPSHOT
    REALVERSION=`echo $VERSION | cut -d - -f 1`
    # REALVERSION 3.0.2 -> 3.0.2-SNAPSHOT
    if [ -f $REF_BINARIES_DIR/$ARTIFACT-$REALVERSION-SNAPSHOT.zip ]; then
        REALVERSION=${REALVERSION}-SNAPSHOT
    fi
}

#
# setup common parts
#
setup_common()
{
    if $CLEARDATA; then

        if [ ! -z "$GATEIN_DATA_DIR" ]; then
            echo "clearing Gatein datas ($GATEIN_DATA_DIR)"
            rm -rf $GATEIN_DATA_DIR/*
        fi

        #
        # Handle OS/X - Linux mysql location
        #
        if [ -x /usr/local/mysql/bin/mysqladmin ]; then
            MYSQLADMIN=/usr/local/mysql/bin/mysqladmin
        fi

        if [ -x /Applications/MAMP/Library/bin/mysqladmin ]; then
            MYSQLADMIN=/Applications/MAMP/Library/bin/mysqladmin
        fi

        if [ -x /usr/bin/mysqladmin ]; then
            MYSQLADMIN=/usr/bin/mysqladmin
        fi

        if $MYSQL; then
            echo "clearing MySQL DataBases"

            $MYSQLADMIN drop $SETUP_NAME-exo-idm_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN drop $SETUP_NAME-exo-jcr_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD

            echo "creating MySQL Databases"

            $MYSQLADMIN create $SETUP_NAME-exo-idm_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN create $SETUP_NAME-exo-jcr_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD

         fi

     fi

    if [ ! -f $REF_BINARIES_DIR/$ARTIFACT-$REALVERSION.zip ]; then

        unzip -t $REF_BINARIES_DIR/$ARTIFACT-$REALVERSION.zip >/dev/null

        if [ $? != "0" ]; then
            echo "couldn't unzip $REF_BINARIES_DIR/$ARTIFACT-*.zip, aborting"
            exit -1
        fi

     fi

    echo "cleaning install directory"

    if [ ! -z "$INSTALL_DIR" ]; then
        echo "clearing install dir ($INSTALL_DIR)"
        rm -rf $INSTALL_DIR/*
    fi

    pushd $INSTALL_DIR
    echo "unziping package..."
    unzip $REF_BINARIES_DIR/$ARTIFACT-$REALVERSION.zip >>/dev/null

    if [ ! -d $INSTALL_DIR/$ZIP_NAME-$REALVERSION ]; then
            echo "couldn't find $ZIP_NAME-$REALVERSION directory, aborting"
            exit -1
    fi

    echo "PLF package unziped"
    popd
}

#
# check if same instance is allready running (and killing it)
#
check_running_instance() {
    INST_ID=$1
    TOMCAT_INST_BASE=$INSTALL_DIR/tomcat$INST_ID

    TOMCAT_LOG=$TOMCAT_INST_BASE/bin/tomcat6-bundle/logs/tomcat.log
    TOMCAT_PID=$TOMCAT_INST_BASE/bin/tomcat6-bundle/logs/tomcat.pid

    if [ -f $TOMCAT_PID ]; then

        read KPID <  $TOMCAT_PID

        if [ ! -z "$TOMCAT_PID" ]; then

            if [ "$(ps -p $KPID | grep -c $KPID)" -eq "1" ]; then
                echo "WARNING, running instance detected stoping it"

                # send kill to process
                kill $KPID

                count="0"
                # wait it to finish
                until [ "$(ps -p $KPID | grep -c $KPID)" -eq "0" ] || \
                        [ "$count" -gt "$SHUTDOWN_WAIT" ]; do
                    if [ "$SHUTDOWN_VERBOSE" = "true" ]; then
                        echo "waiting for processes $KPID to exit"
                    fi
                    sleep 1
                    let count="${count}+1"
                done

                # If process still active after SHUTDOWN_WAIT seconds, brutal kill ;(
                if [ "$count" -gt "$SHUTDOWN_WAIT" ]; then
                    if [ "$SHUTDOWN_VERBOSE" = "true" ]; then
                        echo "killing processes which didn't stop after $SHUTDOWN_WAIT seconds"
                    fi
                    kill -9 $KPID
                fi

                # remove process id
                rm -f  $TOMCAT_PID

            fi
        fi
    fi

}

setup_instance()
{

    INST_ID=$1

    check_running_instance $INST_ID

    TOMCAT_INST_BASE=$INSTALL_DIR/tomcat$INST_ID

    echo "clearing Tomcat instance base ($TOMCAT_INST_BASE)"
    rm -rf $TOMCAT_INST_BASE
    mkdir -p $TOMCAT_INST_BASE

    cp -rf $INSTALL_DIR/$ZIP_NAME-$REALVERSION/* $TOMCAT_INST_BASE

    GATEIN_CONF=$TOMCAT_INST_BASE/bin/tomcat6-bundle/gatein/conf/configuration.properties
    TOMCAT_CONF=$TOMCAT_INST_BASE/bin/tomcat6-bundle/conf/server.xml
    TOMCAT_LIB=$TOMCAT_INST_BASE/bin/tomcat6-bundle/lib

    # gatein.data.dir=../gatein/data -> gatein.data.dir=GATEIN_DATA_DIR
    sed -i -e "s|../gatein/data|$GATEIN_DATA_DIR|g" $GATEIN_CONF


    if $CLUSTERING; then

        # Set jcr configuration for clustering
        sed -i -e "s|gatein.jcr.config.type=local|gatein.jcr.config.type=cluster|g" $GATEIN_CONF
        # Update ChangeFilterClass
        sed -i -e "s|gatein.jcr.index.changefilterclass=org.exoplatform.services.jcr.impl.core.query.DefaultChangesFilter|gatein.jcr.index.changefilterclass=org.exoplatform.services.jcr.impl.core.query.jbosscache.JBossCacheIndexChangesFilter|g" $GATEIN_CONF

    fi

    # update Datasource settings if needed

    if [ ! -z "$MIN_IN_POOL" ]; then
        sed -i -e "s|maxIdle=\"10\"|maxIdle=\"$MIN_IN_POOL\"|g" $TOMCAT_CONF
    fi

    if [ ! -z "$MAX_IN_POOL" ]; then
        sed -i -e "s|maxActive=\"20\"|maxActive=\"$MAX_IN_POOL\"|g" $TOMCAT_CONF
    fi

    if [ ! -z "$MAX_WAIT_IN_POOL" ]; then
        sed -i -e "s|maxWait=\"10000\"|maxWait=\"$MAX_WAIT_IN_POOL\"|g" $TOMCAT_CONF
    fi


    if $MYSQL; then

        # change log/pwd
        sed -i -e "s|username=\"sa\"|username=\"$DB_LOGIN\"|g" $TOMCAT_CONF
        sed -i -e "s|password=\"\"|password=\"$DB_PASSWORD\"|g" $TOMCAT_CONF
        # change driver
        sed -i -e "s|org.hsqldb.jdbcDriver|com.mysql.jdbc.Driver|g" $TOMCAT_CONF
        # change urls (jcr/idm)
        sed -i -e "s|jdbc:hsqldb:file:../gatein/data/hsql/exo-jcr_portal|jdbc:mysql://$DB_HOST/$SETUP_NAME-exo-jcr_portal|g" $TOMCAT_CONF
        sed -i -e "s|jdbc:hsqldb:file:../gatein/data/hsql/exo-idm_portal|jdbc:mysql://$DB_HOST/$SETUP_NAME-exo-idm_portal|g" $TOMCAT_CONF

        rm -f $TOMCAT_LIB/mysql-connector*.jar
        cp $REF_BINARIES_DIR/mysql-connector-java-*.jar $TOMCAT_LIB

    fi

    # change HTTP/HTTPS/AJP/STOP ports -> 18xYY
    sed -i -e "s|8005|18${INST_ID}05|g" $TOMCAT_CONF
    sed -i -e "s|8080|18${INST_ID}80|g" $TOMCAT_CONF
    sed -i -e "s|8009|18${INST_ID}09|g" $TOMCAT_CONF
    sed -i -e "s|8443|18${INST_ID}43|g" $TOMCAT_CONF

    # Setup AJP/HTTP connections
    if [ ! -z "$MAX_HTTPAJP_CONNS" ]; then
        sed -i -e "s|maxThreads=\"...\"||g" $TOMCAT_CONF
        sed -i -e "s|redirectPort=|maxThreads=\"$MAX_HTTPAJP_CONNS\" redirectPort=|g" $TOMCAT_CONF
    fi

    # fix sed -e under OS/X
    rm -f $GATEIN_CONF-e
    rm -f $TOMCAT_CONF-e
}

#
# MAIN HERE
#

init $@
prepare
setup_common

# instance or instances ?
if [ ! -z "$INSTANCE_NUMBER" ]; then

    setup_instance $INSTANCE_NUMBER

else


    for GINST in `eval echo {1..$NUM_INSTANCES}`; do

        setup_instance $GINST

    done

fi
