/*
 * reset
 * https://github.com/tfoster/reset
 *
 * Copyright (c) 2013 Tyler Foster
 * Licensed under the MIT license.
 */

'use strict';
var loggerManager = require('./loggerManager.js'),
	TypeManager = require('./TypeManager.js'),
	HandlerFactory = require('./HandlerFactory.js'),
	_ = require('underscore'),
	url = require('url'),
	path = require('path'),
	fs = require('fs'),
	assert = require('assert');

// Get a reference to the active logger to avoid need to get it for each call.
var logger = loggerManager.get();

//Watch for updates to the logger and update our reference when it changes.
loggerManager.on('updated', function(lgr) {
	logger = lgr;
});

var Reset = function(server, context) {
	this.server = server.addListener('request', this.requestRouter.bind(this));
	this.context = context || '/';
	this.handlers = {};
	logger.info('Initializing REST server...');
	this.handlerFactory = new HandlerFactory(this.context);
	this.typeManager = new TypeManager(this.context);

	this.typeManager.on('registered', (function(descriptor) {
		this.handlers[descriptor.name] = this.handlerFactory.create(descriptor);
	}).bind(this));
};

Reset.prototype.requestRouter = function(req, res) {
	try {
		// We need to ensure the request is within the current instances context.
		if (req.url.length < this.context ||
			req.url.substr(0, this.context.length) !== this.context) {
			return;
		}

		var parsedUrl = url.parse(req.url),
			pathSansContext = parsedUrl.path.substr(this.context.length),
			resourceName = path.dirname(pathSansContext);
		this.handlers[resourceName].handle(req, res);
	} catch (e) {
		logger.error(e);
	}
};

Reset.prototype.registerType = function(descriptor) {
	this.typeManager.registerType(descriptor);
};

/**
 * registerTypes allows you to register mulitiple types.
 *
 * Note: This function will block to avoid starting the server in
 * an incomplete state.
 *
 * @param descriptors A path to a directory containing type descriptors, an array 
 * type descriptor file names or an array of type descriptor instances.
 */
Reset.prototype.registerTypes = function(descriptors) {
	this.typeManager.registerTypes(descriptors);
}

Reset.setLogger =function(lgr) {
	loggerManager.set(lgr);
};

Reset.getLogger = function() {
	return loggerManager.get();
};

module.exports = Reset;