/**
 * UserAuth is a type describing the minimal requirements create a session from a user group key. Generate with {@link createUserAuth}.
 *
 * Optional token generated by {@link https://textileio.github.io/js-threads/classes/_textile_threads_client.client.html#gettoken | Client.getToken} or {@link https://textileio.github.io/js-threads/classes/_textile_threads_client.client.html#gettoken | Client.getTokenChallenge}.
 * @public
 * @example
 * Import
 * ```typescript
 * import {UserAuth} from '@textile/threads';
 * ```
 * @param {string} key - API key. Can be embedded/shared within an app.
 * @param {string} sig - The signature of the authentication message.
 * @param {string} msg - The authentication message.
 * @param {string} token - User verification token generated by {@link https://textileio.github.io/js-threads/classes/_textile_threads_client.client.html#gettoken | Client.getToken} or {@link https://textileio.github.io/js-threads/classes/_textile_threads_client.client.html#gettoken | Client.getTokenChallenge}.
 */
type UserAuth = {
    key: string;
    sig: string;
    msg: string;
    token?: string;
};
/**
 * KeyInfo is a type that contains the API Secret. It should never be shared in insecure environments.
 * @public
 * @example
 * Import
 * ```typescript
 * import {KeyInfo} from '@textile/threads';
 * ```
 * @param {string} key - API key. Can be embedded/shared within an app.
 * @param {string} secret - User group/account secret. Should not be embedded/shared publicly.
 */
type KeyInfo = {
    /**
     * API key. Can be embedded/shared within an app.
     */
    key: string;
    /**
     * User group/account secret. Should not be embedded/shared publicly.
     */
    secret?: string;
};
/**
 * Create an API signature for use in authenticated systems. Generate with {@link createAPISig}.
 * @public
 * @example
 * Import
 * ```typescript
 * import {APISig} from '@textile/threads';
 * ```
 * @param {string} sig - The signature of the authentication message.
 * @param {string} msg - The authentication message.
 */
type APISig = {
    sig: string;
    msg: string;
};
/**
 * createAPISig generates an authorization signature and message only.
 *
 * This function should NOT be used client-side, as it requires a key secret.
 * @public
 * @example
 * Basic usage
 * ```typescript
 * import {createAPISig, APISig} from '@textile/threads'
 *
 * async function sign (key: string) {
 *   const sig: APISig = await createAPISig(key)
 *   return sig
 * }
 * ```
 * @param {string} secret - The key secret to generate the signature. See KeyInfo for details.
 * @param {Date} date - An optional future Date to use as signature message. Once `date` has passed, this
 * authorization signature and message will expire. Defaults to one minute from `Date.now`.
 */
declare function createAPISig(secret: string, date?: Date): Promise<APISig>;
/**
 * Generate a UserAuth containing API key, signature, and message.
 *
 * The gRPC APIs will throw (or return an authorization error) if the message date has passed.
 * This function should NOT be used client-side, as it requires a key secret. The result does
 * not contain the secret and therefor CAN be used client side.
 * @public
 * @example
 * Create a new UserAuth
 * ```typescript
 * import {createUserAuth, KeyInfo, UserAuth} from '@textile/threads';
 *
 * async function auth (keyInfo: KeyInfo) {
 *   // Create an expiration and create a signature. 60s or less is recommended.
 *   const expiration = new Date(Date.now() + 60 * 1000)
 *   // Generate a new UserAuth
 *   const userAuth: UserAuth = await createUserAuth(keyInfo.key, keyInfo.secret ?? '', expiration)
 *   return userAuth
 * }
 * ```
 *
 * @param {string} key - The API key secret to generate the signature. See KeyInfo for details.
 * @param {string} secret - The API key secret to generate the signature. See KeyInfo for details.
 * @param {Date} date - An optional future Date to use as signature message. Default 1 minute from now.
 * @param {string} token - An optional user API token.
 */
declare function createUserAuth(key: string, secret: string, date?: Date, token?: string): Promise<UserAuth>;
declare const errors: {
    invalidKeyError: Error;
};
declare const randomBytes: (byteLength: number) => Uint8Array;
/**
 * keyFromString returns a key by decoding a base32-encoded string.
 * @param k Input base32-encoded string.
 */
declare const keyFromString: (k: string) => Uint8Array;
/**
 * String returns the base32-encoded string representation of raw key bytes.
 * @param k Input key bytes.
 */
declare const keyToString: (k: Uint8Array) => string;
/**
 * Key is a thread encryption key with two components.
 * @param sk Network key is used to encrypt outer log record linkages.
 * @param rk Read key is used to encrypt inner record events.
 */
declare class ThreadKey {
    readonly service: Uint8Array;
    readonly read?: Uint8Array | undefined;
    constructor(service: Uint8Array, read?: Uint8Array | undefined);
    /**
     * Create a new set of keys.
     * @param withRead Whether to also include a random read key.
     */
    static fromRandom(withRead?: boolean): ThreadKey;
    /**
     * Create Key from bytes.
     * @param bytes Input bytes of (possibly both) key(s).
     */
    static fromBytes(bytes: Uint8Array): ThreadKey;
    /**
     * Create Key by decoding a base32-encoded string.
     * @param s The base32-encoded string.
     */
    static fromString(s: string): ThreadKey;
    isDefined(): boolean;
    canRead(): boolean;
    toBytes(): Uint8Array;
    /**
     * Return the base32-encoded string representation of raw key bytes.
     * For example:
     * Full: "brv7t5l2h55uklz5qwpntcat26csaasfchzof3emmdy6povabcd3a2to2qdkqdkto2prfhizerqqudqsdvwherbiy4nazqxjejgdr4oy"
     * Network: "bp2vvqody5zm6yqycsnazb4kpqvycbdosos352zvpsorxce5koh7q"
     */
    toString(): string;
}
export { UserAuth, KeyInfo, APISig, createAPISig, createUserAuth, errors, randomBytes, keyFromString, keyToString, ThreadKey };
