/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2000 - 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.infinispan.context;

import org.infinispan.CacheException;
import org.infinispan.context.impl.LocalTxInvocationContext;
import org.infinispan.context.impl.NonTxInvocationContext;
import org.infinispan.context.impl.RemoteTxInvocationContext;
import org.infinispan.factories.annotations.Inject;
import org.infinispan.transaction.xa.TransactionTable;
import org.infinispan.transaction.xa.TransactionXaAdapter;

import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

/**
 * Default implementation for {@link org.infinispan.context.InvocationContextContainer}.
 *
 * @author Manik Surtani (<a href="mailto:manik@jboss.org">manik@jboss.org</a>)
 * @author Mircea.Markus@jboss.com
 * @since 4.0
 */
public class InvocationContextContainerImpl implements InvocationContextContainer {

   private TransactionManager tm;
   private TransactionTable transactionTable;

   ThreadLocal<InvocationContext> icTl = new ThreadLocal<InvocationContext>();

   @Inject
   public void init(TransactionManager tm, TransactionTable transactionTable) {
      this.tm = tm;
      this.transactionTable = transactionTable;
   }

   public InvocationContext createInvocationContext() {
      Transaction tx = getRunningTx();
      InvocationContext existing = icTl.get();
      if (tx != null) {
         LocalTxInvocationContext localContext;
         if ((existing == null) || !(existing instanceof LocalTxInvocationContext)) {
            localContext = new LocalTxInvocationContext();
            icTl.set(localContext);
         } else {
            localContext = (LocalTxInvocationContext) existing;
         }
         TransactionXaAdapter xaAdapter = transactionTable.getXaCacheAdapter(tx);
         localContext.setXaCache(xaAdapter);
         return localContext;
      } else {
         NonTxInvocationContext nonTxContext;
         if ((existing == null) || !(existing instanceof NonTxInvocationContext)) {
            nonTxContext = new NonTxInvocationContext();
            icTl.set(nonTxContext);
         } else {
            nonTxContext = (NonTxInvocationContext) existing;
         }
         nonTxContext.setOriginLocal(true);
         return nonTxContext;
      }
   }

   public LocalTxInvocationContext createTxInvocationContext() {
      InvocationContext existing = icTl.get();
      if (existing != null && existing instanceof LocalTxInvocationContext) {
         return (LocalTxInvocationContext) existing;
      }
      LocalTxInvocationContext localTxContext = new LocalTxInvocationContext();
      icTl.set(localTxContext);
      return localTxContext;
   }

   public RemoteTxInvocationContext createRemoteTxInvocationContext() {
      InvocationContext existing = icTl.get();
      if (existing != null && existing instanceof RemoteTxInvocationContext) {
         return (RemoteTxInvocationContext) existing;
      }
      RemoteTxInvocationContext remoteTxContext = new RemoteTxInvocationContext();
      icTl.set(remoteTxContext);
      return remoteTxContext;
   }

   public NonTxInvocationContext createNonTxInvocationContext() {
      InvocationContext existing = icTl.get();
      if (existing != null && existing instanceof NonTxInvocationContext) {
         NonTxInvocationContext context = (NonTxInvocationContext) existing;
         context.setOriginLocal(true);
         return context;
      }
      NonTxInvocationContext remoteTxContext = new NonTxInvocationContext();
      icTl.set(remoteTxContext);
      return remoteTxContext;
   }

   public NonTxInvocationContext createRemoteInvocationContext() {
      InvocationContext existing = icTl.get();
      if (existing != null && existing instanceof NonTxInvocationContext) {
         NonTxInvocationContext context = (NonTxInvocationContext) existing;
         context.setOriginLocal(false);
         return context;
      }
      NonTxInvocationContext remoteNonTxContext = new NonTxInvocationContext();
      remoteNonTxContext.setOriginLocal(false);
      icTl.set(remoteNonTxContext);
      return remoteNonTxContext;
   }

   public InvocationContext getInvocationContext() {
      InvocationContext invocationContext = icTl.get();
      if (invocationContext == null)
         throw new IllegalStateException("This method can only be called after associating the current thread with a context");
      return invocationContext;
   }

   public InvocationContext suspend() {
      InvocationContext invocationContext = icTl.get();
      icTl.remove();
      return invocationContext;
   }

   public void resume(InvocationContext ctxt) {
      if (ctxt != null) icTl.set(ctxt);
   }

   private Transaction getRunningTx() {
      try {
         return tm == null ? null : tm.getTransaction();
      } catch (SystemException e) {
         throw new CacheException(e);
      }
   }
}