/*
 *  Copyright 2022-2023 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.instancio.internal.nodes;

import org.instancio.exception.InstancioException;
import org.instancio.internal.ApiValidator;
import org.instancio.internal.reflection.DeclaredAndInheritedFieldsCollector;
import org.instancio.internal.reflection.FieldCollector;
import org.instancio.internal.util.Format;
import org.instancio.internal.util.ObjectUtils;
import org.instancio.internal.util.TypeUtils;
import org.instancio.internal.util.Verify;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Class for creating a node hierarchy for a given {@link Type}.
 */
@SuppressWarnings("PMD.GodClass")
public final class NodeFactory {
    private static final Logger LOG = LoggerFactory.getLogger(NodeFactory.class);

    private final FieldCollector fieldCollector = new DeclaredAndInheritedFieldsCollector();
    private final NodeContext nodeContext;

    public NodeFactory(final NodeContext nodeContext) {
        this.nodeContext = nodeContext;
    }

    public Node createRootNode(final Type type) {
        return createNode(type, null, null);
    }

    private Node createNode(final Type type, @Nullable final Field field, @Nullable final Node parent) {
        Verify.notNull(type, "'type' is null");

        if (parent != null && parent.getDepth() >= nodeContext.getMaxDepth()) {
            LOG.trace("Maximum depth ({}) reached {}", nodeContext.getMaxDepth(), parent);
            return null;
        }

        if (LOG.isTraceEnabled()) {
            LOG.trace("Creating node for: {}", Format.withoutPackage(type));
        }

        final Node node;

        if (type instanceof Class) {
            node = fromClass((Class<?>) type, field, parent);
        } else if (type instanceof ParameterizedType) {
            node = fromParameterizedType((ParameterizedType) type, field, parent);
        } else if (type instanceof TypeVariable) {
            node = fromTypeVariable((TypeVariable<?>) type, field, parent);
        } else if (type instanceof WildcardType) {
            node = fromWildcardType((WildcardType) type, field, parent);
        } else if (type instanceof GenericArrayType) {
            node = fromGenericArrayNode((GenericArrayType) type, field, parent);
        } else {
            throw new InstancioException("Unsupported type: " + type.getClass());
        }

        LOG.trace("Created node: {}", node);
        return node;
    }

    private Node fromWildcardType(final WildcardType type, @Nullable final Field field, @Nullable final Node parent) {
        return createNode(type.getUpperBounds()[0], field, parent);
    }

    private Node fromTypeVariable(final TypeVariable<?> type, @Nullable final Field field, @Nullable final Node parent) {
        final Type resolvedType = resolveTypeVariable(type, parent);

        if (resolvedType == null) {
            LOG.warn("Unable to resolve type variable '{}'. Parent: {}", type, parent);
            return null;
        }

        return createNode(resolvedType, field, parent);
    }

    private Optional<Class<?>> resolveSubtype(final Node node) {
        final Optional<Class<?>> subtype = nodeContext.getSubtype(node);

        if (subtype.isPresent()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Resolved subtype: {} -> {}", node.getRawType().getName(), subtype.get().getName());
            }
            return subtype;
        }

        return Optional.ofNullable(resolveSubtypeFromAncestors(node));
    }

    private static Class<?> resolveSubtypeFromAncestors(final Node node) {
        Node next = node;
        while (next != null) {
            final Type actualType = next.getTypeMap().getActualType(node.getRawType());
            if (actualType != null) {
                return TypeUtils.getRawType(actualType);
            }
            next = next.getParent();
        }
        return null;
    }

    private Node createNodeWithSubtypeMapping(final Type type, @Nullable final Field field, @Nullable final Node parent) {
        final Class<?> rawType = TypeUtils.getRawType(type);

        Node node = Node.builder()
                .nodeContext(nodeContext)
                .type(type)
                .rawType(rawType)
                .targetClass(rawType)
                .field(field)
                .parent(parent)
                .nodeKind(getNodeKind(rawType))
                .build();

        final Class<?> targetClass = resolveSubtype(node).orElse(rawType);

        // Handle the case where: Child<T> extends Parent<T>
        // If the child node inherits a TypeVariable field declaration from
        // the parent, we need to map Parent.T -> Child.T to resolve the type variable
        final Map<Type, Type> genericSuperclassTypeMap = createSuperclassTypeMap(targetClass);

        if (!genericSuperclassTypeMap.isEmpty()) {
            node = node.toBuilder()
                    .additionalTypeMap(genericSuperclassTypeMap)
                    .build();
        }

        if (!rawType.isPrimitive() && rawType != targetClass && !targetClass.isEnum()) {
            ApiValidator.validateSubtype(rawType, targetClass);

            if (LOG.isDebugEnabled()) {
                LOG.debug("Subtype mapping '{}' to '{}'", Format.withoutPackage(rawType), Format.withoutPackage(targetClass));
            }

            // Re-evaluate node kind and type map
            return node.toBuilder()
                    .targetClass(targetClass)
                    .nodeKind(getNodeKind(targetClass))
                    .additionalTypeMap(createBridgeTypeMap(rawType, targetClass))
                    .build();
        }

        return node;
    }

    private Node fromClass(final Class<?> type, @Nullable final Field field, @Nullable final Node parent) {
        final Node node = createNodeWithSubtypeMapping(type, field, parent);
        if (node.hasAncestorEqualToSelf()) {
            return null;
        }

        final Class<?> targetClass = node.getTargetClass();

        if (isContainer(node)) {
            Type[] types = targetClass.isArray()
                    ? new Type[]{targetClass.getComponentType()}
                    : targetClass.getTypeParameters();

            // e.g. CustomMap extends HashMap<String, Long> - type parameters are empty
            // and need to be resolved from superclass
            if (types.length == 0) {
                types = TypeUtils.getGenericSuperclassTypeArguments(targetClass);
            }

            final List<Node> children = createContainerNodeChildren(node, types);
            node.setChildren(children);
        } else {
            final List<Node> children = createChildrenFromFields(targetClass, node);
            node.setChildren(children);
        }
        return node;
    }

    private NodeKind getNodeKind(final Class<?> rawType) {
        for (NodeKindResolver resolver : nodeContext.getNodeKindResolvers()) {
            Optional<NodeKind> resolve = resolver.resolve(rawType);
            if (resolve.isPresent()) {
                return resolve.get();
            }
        }
        return NodeKind.DEFAULT;
    }

    private Node fromParameterizedType(final ParameterizedType type, @Nullable final Field field, @Nullable final Node parent) {
        final Node node = createNodeWithSubtypeMapping(type, field, parent);
        if (node.hasAncestorEqualToSelf()) {
            return null;
        }

        final List<Node> children = isContainer(node)
                ? createContainerNodeChildren(node, type.getActualTypeArguments())
                : createChildrenFromFields(node.getTargetClass(), node);

        node.setChildren(children);
        return node;
    }

    private Node fromGenericArrayNode(final GenericArrayType type, @Nullable final Field field, @Nullable final Node parent) {
        Type gcType = type.getGenericComponentType();
        if (gcType instanceof TypeVariable) {
            gcType = resolveTypeVariable((TypeVariable<?>) gcType, parent);
        }

        final Node node = createArrayNodeWithSubtypeMapping(type, gcType, field, parent);
        final List<Node> children = createContainerNodeChildren(node, gcType);
        node.setChildren(children);
        return node;
    }

    private Node createArrayNodeWithSubtypeMapping(
            final Type arrayType,
            final Type genericComponentType,
            @Nullable final Field field,
            @Nullable final Node parent) {

        final Class<?> rawComponentType = TypeUtils.getRawType(genericComponentType);
        final Node node = Node.builder()
                .nodeContext(nodeContext)
                .type(arrayType)
                .rawType(TypeUtils.getArrayClass(rawComponentType))
                .targetClass(TypeUtils.getArrayClass(rawComponentType))
                .field(field)
                .parent(parent)
                .nodeKind(NodeKind.ARRAY)
                .build();

        final Class<?> targetClass = resolveSubtype(node).orElse(rawComponentType);
        final Class<?> targetClassComponentType = targetClass.getComponentType();

        if (!rawComponentType.isPrimitive()
                && targetClassComponentType != null
                && rawComponentType != targetClassComponentType) {

            ApiValidator.validateSubtype(rawComponentType, targetClassComponentType);

            if (LOG.isDebugEnabled()) {
                LOG.debug("Subtype mapping '{}' to '{}'",
                        Format.withoutPackage(rawComponentType),
                        Format.withoutPackage(targetClass));
            }

            final Map<Type, Type> typeMapForSubtype = createBridgeTypeMap(rawComponentType, targetClassComponentType);

            // Map component type to match array subtype
            typeMapForSubtype.put(rawComponentType, targetClassComponentType);

            return node.toBuilder()
                    .targetClass(targetClass)
                    .additionalTypeMap(typeMapForSubtype)
                    .build();
        }

        return node;
    }

    /**
     * Creates children for a "container" node (that is an array, collection, or map).
     * Children of a container node have no 'field' property since their values
     * are not assigned via fields, but added via {@link Collection#add(Object)},
     * {@link Map#put(Object, Object)}, etc.
     *
     * @param parent of the children
     * @param types  children's types
     * @return a list of children, or an empty list if no children were created (e.g. to avoid cycles)
     */
    private List<Node> createContainerNodeChildren(final Node parent, final Type... types) {
        final List<Node> results = new ArrayList<>(types.length);
        for (Type type : types) {
            final Node node = createNode(type, null, parent);
            if (node != null) {
                results.add(node);
            }
        }
        return results;
    }

    private List<Node> createChildrenFromFields(final Class<?> targetClass, final Node parent) {
        final List<Node> list = new ArrayList<>();
        for (Field f : fieldCollector.getFields(targetClass)) {
            Node node = createNode(ObjectUtils.defaultIfNull(f.getGenericType(), f.getType()), f, parent);
            if (node != null) {
                list.add(node);
            }
        }
        return list;
    }

    private static boolean isContainer(final Node node) {
        return node.is(NodeKind.COLLECTION)
                || node.is(NodeKind.MAP)
                || node.is(NodeKind.ARRAY)
                || node.is(NodeKind.CONTAINER);
    }

    private Type resolveTypeVariable(final TypeVariable<?> typeVar, @Nullable final Node parent) {
        Type mappedType = parent == null ? typeVar : parent.getTypeMap().getOrDefault(typeVar, typeVar);
        Node ancestor = parent;

        while ((mappedType == null || mappedType instanceof TypeVariable) && ancestor != null) {
            Type rootTypeMapping = nodeContext.getRootTypeMap().get(mappedType);
            if (rootTypeMapping != null) {
                return rootTypeMapping;
            }

            mappedType = ancestor.getTypeMap().getOrDefault(mappedType, mappedType);

            if (mappedType instanceof Class || mappedType instanceof ParameterizedType) {
                break;
            }

            ancestor = ancestor.getParent();
        }
        return mappedType == typeVar ? null : mappedType; // NOPMD
    }

    private static Map<Type, Type> createSuperclassTypeMap(final Class<?> targetClass) {
        Map<Type, Type> resultTypeMap = null;

        Type supertype = targetClass.getGenericSuperclass();

        while (supertype instanceof ParameterizedType) {
            if (resultTypeMap == null) {
                resultTypeMap = new HashMap<>();
            }

            addTypeParameters((ParameterizedType) supertype, resultTypeMap);

            final Class<?> rawSuper = TypeUtils.getRawType(supertype);
            supertype = rawSuper.getGenericSuperclass();
        }

        if (resultTypeMap == null) {
            return Collections.emptyMap();
        }

        LOG.trace("Created superclass type map: {}", resultTypeMap);
        return resultTypeMap;
    }

    /**
     * A "bridge type map" is required for performing type substitutions of parameterized types.
     * For example, a subtype may declare a type variable that maps to a type variable declared
     * by the super type. This method provides the "bridge" mapping that allows resolving the actual
     * type parameters.
     * <p>
     * For example, given the following classes:
     *
     * <pre>{@code
     *     interface Supertype<A> {}
     *     class Subtype<B> implements Supertype<B>
     * }</pre>
     * <p>
     * the method returns a map of {@code {B -> A}}
     * <p>
     * NOTE: in its current form, this method only handles the most basic use cases.
     *
     * @param source source type
     * @param target target type
     * @return additional type mappings that might help resolve type variables
     */
    private static Map<Type, Type> createBridgeTypeMap(final Class<?> source, final Class<?> target) {
        if (source.equals(target)) {
            return Collections.emptyMap();
        }

        final Map<Type, Type> typeMap = new HashMap<>();
        final TypeVariable<?>[] subtypeParams = target.getTypeParameters();
        final TypeVariable<?>[] supertypeParams = source.getTypeParameters();

        if (subtypeParams.length == supertypeParams.length) {
            for (int i = 0; i < subtypeParams.length; i++) {
                typeMap.put(subtypeParams[i], supertypeParams[i]);
            }
        }

        // If subtype has a generic superclass, add its type variables and type arguments to the type map
        final Type supertype = target.getGenericSuperclass();
        if (supertype instanceof ParameterizedType) {
            addTypeParameters((ParameterizedType) supertype, typeMap);
        }

        return typeMap;
    }

    private static void addTypeParameters(final ParameterizedType parameterizedType, final Map<Type, Type> typeMap) {
        final Class<?> rawSuperclassType = TypeUtils.getRawType(parameterizedType);
        final TypeVariable<?>[] typeVars = rawSuperclassType.getTypeParameters();
        final Type[] typeArgs = parameterizedType.getActualTypeArguments();

        if (typeVars.length == typeArgs.length) {
            for (int i = 0; i < typeVars.length; i++) {
                typeMap.put(typeVars[i], typeArgs[i]);
            }
        }
    }
}
