/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.i18n.translate;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

/**
 * Goal which reports missing message keys for a set of target languages.
 *
 * @goal check
 * 
 * @phase process-sources
 */
public class ReportMissingKeysMojo extends AbstractMojo {

    /**
     * @parameter expression="${languageKeys}"
     */
    private String[]                             languageKeys;

    /**
     * @parameter default-value="messages.properties"
     */
    private String                               mainMessagesFile;

    /**
     * Message file pattern
     *
     * @parameter default-value="messages_(\\w+)\\.properties$"
     */
    private String                               messageFilePattern;

    private MessageFileService                   messageFileService;

    /**
     * Messages directory location
     *
     * @parameter default-value="${basedir}/src/main/resources/properties/i18n/"
     */
    private String                               messagesDirectory;

    private final Map<LanguageFile, Set<String>> reportedPropertyFilesMap = new LinkedHashMap<LanguageFile, Set<String>>();

    /**
     * @parameter default-value="true"
     */
    private boolean                              warnOnly;

    @Override
    public void execute() throws MojoExecutionException {

        Set<String> mainKeys = null;
        Map<String, String> mainMap = null;

        try {

            this.messageFileService = new MessageFileService();
            this.getLog().info("Locating main messages file at " + this.getMainMessagesFile());
            final Resource resource = new FileSystemResource(this.getMainMessagesFile());
            mainMap = this.messageFileService.getMessageMapFromFile(resource);
            mainKeys = mainMap.keySet();

        } catch (final IOException ex) {
            this.getLog().error("Main messages file could not be located");
            throw new MojoExecutionException(ex.getMessage());
        }

        final File msgDir = new File(this.messagesDirectory);
        if (!msgDir.exists()) {
            throw new MojoExecutionException("Main messages directory " + this.messagesDirectory + " does not exist.");
        }

        final Pattern pattern = Pattern.compile(this.messageFilePattern, Pattern.MULTILINE | Pattern.CASE_INSENSITIVE);
        final Map<String, String> keys = new HashMap<String, String>();

        if (this.languageKeys.length == 0) {

            msgDir.listFiles(new FilenameFilter() {

                @Override
                public boolean accept(final File dir, final String name) {
                    boolean matches = false;
                    final Matcher matcher = pattern.matcher(name);

                    if (matcher.find()) {
                        final String langKey = matcher.group(1);
                        keys.put(langKey, name);
                        ReportMissingKeysMojo.this.getLog().info("Added language file " + name + " to the processing queue.");
                        matches = true;
                    }

                    return matches;
                }
            });
        } else {
            Arrays.sort(this.languageKeys);

            msgDir.listFiles(new FilenameFilter() {

                @Override
                public boolean accept(final File dir, final String name) {
                    boolean matches = false;

                    if (!name.equalsIgnoreCase(ReportMissingKeysMojo.this.mainMessagesFile)) {
                        final Matcher matcher = pattern.matcher(name);

                        if (matcher.find()) {
                            final String langKey = matcher.group(1);

                            final int idx = Arrays.binarySearch(ReportMissingKeysMojo.this.languageKeys, langKey);

                            if (idx >= 0) {
                                keys.put(langKey, name);
                                ReportMissingKeysMojo.this.getLog().info("Added language file " + name + " to the processing queue.");
                                matches = true;
                            }
                        }
                    }

                    return matches;
                }
            });

        }

        final Iterator<String> itKeys = keys.keySet().iterator();
        while (itKeys.hasNext()) {
            try {

                final String key = itKeys.next();
                final String fileName = keys.get(key);

                final LanguageFile file = new LanguageFile(this.messagesDirectory, fileName, key);

                if (file.exists() && file.canRead()) {

                    final Resource resource2 = new FileSystemResource(file);
                    final Map<String, String> map = this.messageFileService.getMessageMapFromFile(resource2);
                    final Set<String> targetKeys = map.keySet();

                    // create a set of any keys that are in the main message file,
                    // but not in the target file
                    final Set<String> missingKeys = new HashSet<String>();
                    missingKeys.addAll(mainKeys);
                    missingKeys.removeAll(targetKeys);

                    // report any missing keys for the current language
                    if (missingKeys.size() > 0) {
                        this.getLog().info("Locating messages file at " + file.getCanonicalPath());

                        this.getLog().warn(
                                "Found " + missingKeys.size() + " missing keys for " + key + " in file " + file.getName()
                                        + ". Missing keys are:");

                        final Iterator<String> it = missingKeys.iterator();
                        while (it.hasNext()) {
                            final String keyMissing = it.next();
                            this.getLog().warn("\t" + keyMissing + "=" + mainMap.get(keyMissing));
                        }

                        this.getReportedPropertyFiles().put(file, missingKeys);
                    }
                } else {

                    this.getLog().error("Messages file for language '" + key + "' (" + key + ") cannot be located");

                }

            } catch (final Exception ex) {
                throw new MojoExecutionException(ex.getMessage());
            }
        }

        this.reportFinalResults();
    }

    public String[] getLanguageKeys() {
        return this.languageKeys;
    }

    public Map<LanguageFile, Set<String>> getReportedPropertyFiles() {
        return this.reportedPropertyFilesMap;
    }

    public void setLanguageKeys(final String[] languageKeys) {
        this.languageKeys = languageKeys;
    }

    private void reportFinalResults() throws MissingKeysException {
        if (this.getReportedPropertyFiles().size() == 0) {
            if (this.warnOnly) {
                this.getLog().warn("No language files were found at the specified messages directory: " + this.messagesDirectory);
            } else {
                throw new MissingKeysException("No language files were found at the specified messages directory: "
                        + this.messagesDirectory);
            }
        } else if (this.getReportedPropertyFiles().size() > 0) {
            if (this.warnOnly) {
                this.getLog().warn("Found " + this.getReportedPropertyFiles().size() + " property file(s) with missing keys.");
            } else {
                throw new MissingKeysException("Found " + this.getReportedPropertyFiles().size() + " property file(s) with missing keys.");
            }
        } else {
            this.getLog().info("Verified all language files for keys: " + Arrays.toString(this.languageKeys));

        }
    }

    protected File getMainMessagesFile() {
        return new File(this.messagesDirectory, this.mainMessagesFile);
    }
}
