/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.jasig.portal.channels.jsp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Map.Entry;

import javax.servlet.ServletContext;

import org.jasig.portal.PortalException;
import org.jasig.portal.PortalSessionManager;

/**
 * This class represents a deployment specification file that holds information
 * on what CAR resources have been extracted from CARs and placed into a 
 * classloader accessible location so that they are accessible via
 * the web application's classloader. This is done so that JSPs included in CARs
 * can access the classes that were included in the CAR. The format of entries
 * is as follows:
 * 
 * <pre>
 * 
 *  carCount: the number of cars having deployed resources
 *   
 *  car_N: the path to the car file for the Nth car entry.
 *   
 *  car_N_deployed: the date of extraction for resources of the Nth car
 *   
 *  some.resource.path=car_N: indicates a resource extracted for the Nth car
 *   
 *  file_some.resource.path: fully qualified path to extracted file
 *  
 * </pre>
 * 
 * @author Mark Boyd
 * @deprecated All IChannel implementations should be migrated to portlets
 */
@Deprecated
public class DeploymentSpec
{
    private static final SimpleDateFormat cFormatter =
        new SimpleDateFormat("yyyy.MMMMM.dd HH:mm");
    private static final String PROP_CAR_ID_DEPLOYED_SUFFIX = "_deployed";
    private static final String PROP_CAR_ID_PREFIX = "car_";
    private static final String PROP_CAR_COUNT = "carCount";
        
    private static final String DEPLOYMENT_FILE_NAME = "jspCars.deployed";
    private static final String DEPLOYMENT_PATH = "/WEB-INF";
    
    private static boolean initialized = false;

    private Properties mDeployInfo = null;
    private File mDepFile = null;
    private String mDepInfoPath = null;
    private int mCarCount = 0;
    private Map mCarIdsByCarPath;
    
    private static final DeploymentSpec instance = new DeploymentSpec();
    
    public static DeploymentSpec getInstance() throws PortalException
    {
        if (! initialized)
            instance.loadDeploymentInfo();
        return instance;
    }
    
    private DeploymentSpec()
    {
    }
    
    private String getDeploymentSpecPath() throws PortalException
    {
        if (mDepInfoPath == null)
        {
            ServletContext ctx =
                PortalSessionManager.getInstance().getServletContext();
            mDepInfoPath = ctx.getRealPath(DEPLOYMENT_PATH);
            if (mDepInfoPath == null)
                throw new PortalException(
                    "Unable to locate WEB-INF directory.");
        }
        return mDepInfoPath;
    }

    /**
     * Store updated car deployment file. 
     */
    private void writeDeploymentSpec() throws PortalException
    {
        String filePath = getDeploymentSpecPath();
        String fullName = filePath + "/" + DEPLOYMENT_FILE_NAME;

        File dirPath = new File(filePath);
        if (! dirPath.exists())
            dirPath.mkdirs();
            
        try
        {
            FileOutputStream out = new FileOutputStream(fullName);
            mDeployInfo.store(
                out,
                "Deployment information for JSP Car file resources. " +
                "Generated by " + Deployer.class.getName() + ". Car paths " +
                "are relative to the CAR directory. Deployment paths for " +
                ".class and .properties files are relative to the value of " +
                "property " + Deployer.CLASSES_DEPLOY_DIR_PROP + 
                " while .jsp, .jspf, and .jsf files are relative to the " +
                "value of property " + Deployer.JSP_DEPLOY_DIR_PROP + 
                ". Both properties are specified in portal.properties and " +
                "should not be changed.");
            out.flush();
            out.close();
        }
        catch (Exception e)
        {
            throw new PortalException(
                "Unable to store deployment "
                    + "information in '"
                    + fullName
                    + "'.",
                e);
        }
    }

    /**
     * Obtains a new car entry ID for adding deployment information for a car
     * and its resources.
     * 
     * @return
     * @throws PortalException
     */
    private String getNewCarEntryId()
    {
        int newId = ++mCarCount;
        mDeployInfo.setProperty(PROP_CAR_COUNT, "" + newId);
        return PROP_CAR_ID_PREFIX + newId;
    }
    
    /**
     * Adds deployment information about the passed-in CAR and its resources
     * into the deployment specification file.
     * 
     * @param carFilePath
     * @param resources
     * @throws PortalException
     */
    public void addDeploymentFor(String carFilePath, List resources)
            throws PortalException
    {
        String carEntryId = (String) mCarIdsByCarPath.get(carFilePath);

        if (carEntryId == null) // no previous deployment
            carEntryId = getNewCarEntryId();

        mDeployInfo.put(carEntryId, carFilePath);
        mDeployInfo.put(carEntryId + "_deployed", cFormatter.format(new Date()));

        for (Iterator itr = resources.iterator(); itr.hasNext();)
        {
            String resource = (String) itr.next();
                mDeployInfo.put(resource, carEntryId);
        }
        writeDeploymentSpec();
    }

    /**
     * Removes path entries for resources extracted from the indicated CAR
     * from the car jsp deployment specification.
     *  
     * @param carFilePath
     * @return
     * @throws PortalException
     */
    public List removeEntriesFor(String carFilePath)
    {
        String carId = null;
        
        for(Iterator itr = mDeployInfo.entrySet().iterator(); itr.hasNext();)
        {
            Map.Entry entry = (Entry) itr.next();
            if (entry.getValue().equals(carFilePath))
            {
                carId = (String) entry.getKey();
                break;
            }
        }
        if (carId == null)
            return null; // no previous info to purge
        
        List entries = new ArrayList();
        for(Iterator itr = mDeployInfo.entrySet().iterator(); itr.hasNext();)
        {
            Map.Entry entry = (Entry) itr.next();
            if (entry.getValue().equals(carId))
            {
                String resPath = (String) entry.getKey();
                entries.add(resPath);
                itr.remove();
            }
        }
        return entries;
    }
    
    /**
     * Determines if an entry for a resource is
     * found in the deployment information file for jsp cars.
     * 
     * @param classFilePath
     * @return
     */
    public boolean isDeployInfoAvailableFor(String resource)
    {
        return mDeployInfo.containsKey(resource);
    }

    /**
     * Loads information about a previous deployments stored in property file
     * format. Deployment of resources from the CAR means placing them in 
     * WEB-INF/classes. This is necessary for both <code>.jsp</code> and 
     * <code>.class</code> files. For the former it is necessary so that they 
     * are in a location that will allow the server to compile them. For the
     * latter it is necessary so that the classes will be accessible to the
     * classloader used by the java server pages. The CAR classloader is known
     * only to uPortal code.
     * 
     * @param depUrl
     * @return
     */
    private void loadDeploymentInfo() throws PortalException
    {
        String path = getDeploymentSpecPath();
        mDepFile = new File(path + "/" + DEPLOYMENT_FILE_NAME);
        Properties newInfo = new Properties();
        Map carIdsByCarPath = new HashMap();
        int carCount = 0;
        
        if (mDepFile.exists())
        {
            try
            {
                FileInputStream fis = new FileInputStream(mDepFile);
                newInfo.load(fis);
                String sCarCount = newInfo.getProperty(PROP_CAR_COUNT);
                carCount = Integer.parseInt(sCarCount);
                fis.close();
            } catch (Exception e)
            {
                throw new PortalException(
                        "Unable to load deployment information from file '"
                                + mDepFile.getAbsolutePath() + "'.", e);
            }
            for (Iterator itr = newInfo.entrySet().iterator(); itr.hasNext();)
            {
                Map.Entry entry = (Entry) itr.next();
                String key = (String) entry.getKey();

                if (key.startsWith(PROP_CAR_ID_PREFIX)
                        && !key.endsWith(PROP_CAR_ID_DEPLOYED_SUFFIX))
                    carIdsByCarPath.put(entry.getValue(), key);
            }
        }
        mCarIdsByCarPath = carIdsByCarPath;
        mCarCount = carCount;
        mDeployInfo = newInfo;
        initialized = true;
    }

    /**
     * Returns the timestamp in milliseconds when the specified resource was
     * deployed or -1 if the resource is not included in the list of resources
     * deployed.
     * 
     * @param classFilePath
     * @return
     * @throws PortalException
     */
    public long getTimeOfDeploymentForResource(String resourcePath)
            throws PortalException
    {
        String carDate = null;
        String carId = null;
        long depDate = -1;
        try
        {
            carId = mDeployInfo.getProperty(resourcePath);
            
            if (carId != null)
            {
                carDate = mDeployInfo.getProperty(carId
                        + PROP_CAR_ID_DEPLOYED_SUFFIX);
                depDate = cFormatter.parse(carDate).getTime();
            }
        } catch (Exception e1)
        {
            throw new PortalException("Unable to parse date '" + carDate
                    + "' for entry '" + resourcePath + "' from file '"
                    + mDepFile.getAbsolutePath() + "'.");
        }
        return depDate;
    }
}
