/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.deployers;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.ejb3.common.deployers.spi.AttachmentNames;
import org.jboss.ejb3.deployers.metadata.processor.JBossMetaDataProcessorFactory;
import org.jboss.logging.Logger;
import org.jboss.metadata.ejb.jboss.JBossMetaData;
import org.jboss.metadata.process.chain.ProcessorChain;
import org.jboss.metadata.process.chain.ejb.jboss.JBossMetaDataProcessorChain;
import org.jboss.metadata.process.processor.JBossMetaDataProcessor;
import org.jboss.metadata.validation.chain.ValidatorChain;

import java.util.ArrayList;
import java.util.List;

/**
 * Ejb3MetadataProcessingDeployer
 *
 * Runs post-merge processing on EJB3 Metadata
 * to apply rules, perform validation, etc
 *
 * JBMETA-132
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class Ejb3MetadataProcessingDeployer extends AbstractDeployer
{

   // ------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------||
   // ------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(Ejb3MetadataProcessingDeployer.class);

   /**
    * Deployer Input, set to merged metadata
    */
   private static final String INPUT = AttachmentNames.MERGED_METADATA;

   /**
    * Deployer Output, the Processed metadata
    */
   public static final String OUTPUT = AttachmentNames.PROCESSED_METADATA;
   
   
   /**
    *  {@link JBossMetaDataProcessorFactory} processor factories which will be used
    *  to create processors {@link JBossMetaDataProcessor} to process the merged {@link JBossMetaData}
    *  
    */
   private List<JBossMetaDataProcessorFactory<JBossMetaData>> processorFactories = new ArrayList<JBossMetaDataProcessorFactory<JBossMetaData>>();
   
   /**
    * A metadata validator chain which will be used to validate the merged metadata,
    * after processing the metadata through all available {@link JBossMetaDataProcessor}s
    * 
    */
   private ValidatorChain<JBossMetaData> validatorChain;

   // ------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------||
   // ------------------------------------------------------------------------------||

   /**
    * Constructor; sets deployment stage and requisite input/output chain
    */
   public Ejb3MetadataProcessingDeployer()
   {
      // Set the Stage to post-CL
      this.setStage(DeploymentStages.POST_CLASSLOADER);

      // Input is the JBossMetaData post-merge
      this.addInput(INPUT);

      // Output is a flag upon which other deployers may rely
      this.addOutput(OUTPUT);
   }

   // ------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------||
   // ------------------------------------------------------------------------------||

   /**
    * For DeploymentUnits with a merged metadata attachment representing
    * EJB3 beans, will run a standard ProcessorChain. The processed
    * metadata is then run through a validator chain
    *
    * @see org.jboss.deployers.spi.deployer.Deployer#deploy(org.jboss.deployers.structure.spi.DeploymentUnit)
    */
   public void deploy(DeploymentUnit du) throws DeploymentException
   {
      // Obtain the Merged Metadata
      JBossMetaData md = du.getAttachment(INPUT, JBossMetaData.class);

      
      // If metadata's not present as an attachment, return
      if (md == null)
      {
         return;
      }

      // If this is not an EJB3 Deployment, return
      if (!md.isEJB3x())
      {
         return;
      }

      // Create the processor chain 
      ProcessorChain<JBossMetaData> chain = new JBossMetaDataProcessorChain<JBossMetaData>();
      for (JBossMetaDataProcessorFactory<JBossMetaData> processorFactory : this.processorFactories)
      {
            JBossMetaDataProcessor<JBossMetaData> processor = processorFactory.create(du);
            chain.addProcessor(processor);
      }
      
      // Create new processed metadata
      JBossMetaData processedMetadata = chain.process(md);
      
      // now validate the processed metadata
      if (this.validatorChain != null)
      {
         this.validatorChain.validate(processedMetadata);
      }

      // Set the processed metadata as the output
      du.addAttachment(OUTPUT, processedMetadata, JBossMetaData.class);
   }

   /**
    * Add a processor factory to the set of processor factories which will be
    * used to process the merged {@link JBossMetaData}
    * @param processorFactory
    */
   public void addProcessorFactory(JBossMetaDataProcessorFactory<JBossMetaData> processorFactory)
   {
      this.processorFactories.add(processorFactory);
   }
   /**
    * Removes a processor factory from the set of processor factories which are 
    * responsible for processing the merged {@link JBossMetaData}
    * 
    * @param processorFactory
    */
   public void removeProcessorFactory(JBossMetaDataProcessorFactory<JBossMetaData> processorFactory)
   {
      this.processorFactories.remove(processorFactory);
   }
   
   /**
    * Sets the validator chain which will be used to validate the metadata
    * after running it through the available {@link JBossMetaDataProcessor}s
    * 
    * @param validatorChain
    */
   public void setValidatorChain(ValidatorChain<JBossMetaData> validatorChain)
   {
      this.validatorChain = validatorChain;
   }
   
   /**
    * Returns the validator chain used to validate the processed metadata
    * 
    * @return
    */
   public ValidatorChain<JBossMetaData> getValidatorChain()
   {
      return this.validatorChain;
   }
   
}
