/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm;

import java.util.List;
import java.util.Map;


/** manages runtime process executions.
 * 
 * @author Tom Baeyens
 */
public interface ExecutionService {

  /** starts a new execution for the ProcessDefinition with the given processDefinitionDbid.
   * @param processDefinitionId the {@link ProcessDefinition#getId() unique id} of the process definition. */
  Execution startExecution(String processDefinitionId);
  
  /** starts a new execution for the ProcessDefinition with the given processDefinitionDbid. 
   * @param processDefinitionId the {@link ProcessDefinition#getId() unique id} of the process definition.
   * @param executionKey is a user provided reference for the new execution that must be unique over all 
   *    process versions with the same name. */
  Execution startExecution(String processDefinitionId, String executionKey);

  /** starts a new execution for the ProcessDefinition with the given processDefinitionDbid. 
   * @param processDefinitionId the {@link ProcessDefinition#getId() unique id} of the process definition.
   * @param variables are the initial values of the process variables that will be set before the execution starts. */
  Execution startExecution(String processDefinitionId, Map<String, Object> variables);

  /** starts a new execution for the ProcessDefinition with the given processDefinitionDbid.
   * @param processDefinitionId the {@link ProcessDefinition#getId() unique id} of the process definition.
   * @param variables are the initial values of the process variables that will be set before the execution starts.
   * @param executionKey is a user provided reference for the new execution that must be unique over all 
   *    process versions with the same name. */
  Execution startExecution(String processDefinitionId, Map<String, Object> variables, String executionKey);

  /** starts a new execution in the latest version of the given process definition.
  * @param processDefinitionName is the name of the process definition for which the latest version will be taken. */ 
  Execution startExecutionInLatest(String processDefinitionName);

  /** starts a new execution in the latest version of the given processDefinitionName.
   * @param processDefinitionName is the name of the process definition 
   *   for which the latest version will be taken. 
   * @param executionKey is a user provided reference for the new execution 
   *   that must be unique over all process versions with the same name. */
  Execution startExecutionInLatest(String processDefinitionName, String executionKey);

  /** starts a new execution in the latest version of the given processDefinitionName. 
   * @param processDefinitionName is the name of the process definition 
   *   for which the latest version will be taken. 
   * @param variables are the initial values of the process variables that 
   *   will be set before the execution starts (read: before the initial 
   *   node is executed). */
  Execution startExecutionInLatest(String processDefinitionName, Map<String, Object> variables);

  /** starts a new execution in the latest version of the given processDefinitionName. 
   * @param processDefinitionName is the name of the process definition for which the latest version will be taken. 
   * @param variables are the initial values of the process variables that will be set before the execution starts.
   * @param executionKey is a user provided reference for the new execution that must be unique over all 
   *    process versions with the same name. */
  Execution startExecutionInLatest(String processDefinitionName, Map<String, Object> variables, String executionKey);

  /** the execution that is uniquely defined by the process definition and the business key. */
  Execution findExecution(String processDefinitionName, String executionKey);

  /** the execution that is uniquely defined by the process definition and the business key. */
  Execution findExecution(String executionId);

  
  /** provides an external trigger to an execution. */
  Execution signalExecution(String executionId);
  
  /** provides a named external trigger to an execution. */
  Execution signalExecution(String executionId, String signalName);
   
  /** provides a named external trigger to an execution with parameters. */
  Execution signalExecution(String executionId, String signalName, Map<String, Object> parameters);

  /** provides a external trigger to an execution with parameters. */
  Execution signalExecution(String executionId, Map<String, Object> parameters);
  
  
  /** provides a named external trigger to an execution. */
  Execution signalExecutionByKey(String processDefinitionName, String executionKey);
  
  /** provides a named external trigger to an execution. */
  Execution signalExecutionByKey(String processDefinitionName, String executionKey, String signalName);
  
  /** provides a named external trigger to an execution with parameters. */
  Execution signalExecutionByKey(String processDefinitionName, String executionKey, Map<String, Object> parameters);

  /** provides a named external trigger to an execution with parameters. */
  Execution signalExecutionByKey(String processDefinitionName, String executionKey, String signalName, Map<String, Object> parameters);

  
  /** creates or overwrites a variable value on the referenced execution */ 
  Execution setVariable(String executionId, String name, Object value);
  
  /** creates or overwrites the variable values on the referenced execution */ 
  Execution setVariables(String executionId, Map<String, Object> variables);
  
  /** retrieves a variable */ 
  Object getVariable(String executionId, String variableName);
  
  /** retrieves a map of variables */ 
  Map<String, Object> getVariables(String executionId, List<String> variableNames);
}
