/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;

import org.jbpm.pvm.Execution;
import org.jbpm.pvm.PvmException;
import org.jbpm.pvm.activity.ActivityExecution;
import org.jbpm.pvm.client.ClientProcessDefinition;
import org.jbpm.pvm.client.ClientProcessInstance;
import org.jbpm.pvm.env.Environment;
import org.jbpm.pvm.internal.job.MessageImpl;
import org.jbpm.pvm.internal.job.TimerImpl;
import org.jbpm.pvm.internal.log.Log;
import org.jbpm.pvm.internal.model.op.AtomicOperation;
import org.jbpm.pvm.internal.model.op.ExecuteNode;
import org.jbpm.pvm.internal.model.op.MoveToChildNode;
import org.jbpm.pvm.internal.model.op.MoveToParentNode;
import org.jbpm.pvm.internal.model.op.ProceedToDestination;
import org.jbpm.pvm.internal.model.op.Signal;
import org.jbpm.pvm.internal.model.op.TakeTransition;
import org.jbpm.pvm.internal.type.Converter;
import org.jbpm.pvm.internal.type.Type;
import org.jbpm.pvm.internal.type.Variable;
import org.jbpm.pvm.internal.type.VariableTypeResolver;
import org.jbpm.pvm.internal.type.variable.NullVariable;
import org.jbpm.pvm.internal.type.variable.UnpersistableVariable;
import org.jbpm.pvm.internal.util.Clock;
import org.jbpm.pvm.internal.util.EqualsUtil;
import org.jbpm.pvm.internal.util.Priority;
import org.jbpm.pvm.job.Timer;
import org.jbpm.pvm.listener.EventListener;
import org.jbpm.pvm.listener.EventListenerExecution;
import org.jbpm.pvm.model.Comment;
import org.jbpm.pvm.model.Event;
import org.jbpm.pvm.model.IdGenerator;
import org.jbpm.pvm.model.Node;
import org.jbpm.pvm.model.ObservableElement;
import org.jbpm.pvm.model.OpenExecution;
import org.jbpm.pvm.model.Transition;
import org.jbpm.pvm.processlog.ProcessLog;
import org.jbpm.pvm.session.LogSession;
import org.jbpm.pvm.session.MessageSession;
import org.jbpm.pvm.session.TimerSession;

/**
 * @author Tom Baeyens
 */
public class ExecutionImpl implements ClientProcessInstance,
                                      ActivityExecution, 
                                      EventListenerExecution, 
                                      Serializable {

  private static final long serialVersionUID = 1L;

  private static final Log log = Log.getLog(Execution.class.getName());
  
  // atomic operations 
  public static final AtomicOperation EXECUTE_NODE = new ExecuteNode();
  public static final AtomicOperation PROCEED_TO_DESTINATION = new ProceedToDestination();
  public static final AtomicOperation TAKE_TRANSITION = new TakeTransition();
  public static final AtomicOperation PROPAGATE_TO_PARENT = new MoveToParentNode();
  
  protected long dbid;
  protected int dbversion;

  /** an optional name for this execution.  can be used to 
   * differentiate concurrent paths of execution like e.g. 
   * the 'shipping' and 'billing' paths. */
  protected String name;

  /** a key for this execution. typically this is an externally provided reference 
   * that is unique within the scope of the process definition.  */
  protected String key;

  /** a unique id for this execution. */
  protected String id;

  /** @see Execution */  
  protected String state;

  protected ProcessDefinitionImpl processDefinition;
  
  // current position /////////////////////////////////////////////////////////
  
  /** current node */
  protected NodeImpl node;
  
  /** transition is not to be made persistable by default */
  protected TransitionImpl transition;
  
  /** the node from which the transition was taken.  This can be different from 
   * the transition source in case a transition of an eclosing node was taken.
   * transitionOrigin is not to be made persistable by default */
  protected NodeImpl transitionOrigin;

  protected EventImpl event;
  protected ObservableElementImpl eventSource;
  
  /** are concurrent executions that related to this execution. */
  protected Collection<ExecutionImpl> executions;

  /** the parent child relation of executions is convenient for some forms of
   * concurrency. */
  protected ExecutionImpl parent = null;
  protected ExecutionImpl processInstance;
  
  boolean hasVariables;
  protected Map<String, Variable> variables;
  
  boolean hasTimers;
  protected Set<TimerImpl> timers;
  
  /** the super process link in case this is a sub process execution */  
  protected ExecutionImpl superProcessExecution;
  
  /** the sub process link in case of sub process execution */
  protected ExecutionImpl subProcessInstance;
 
  /** the free text comments users make on this execution */
  protected Set<CommentImpl> comments;

  protected int priority = Priority.NORMAL;

  /** maintains the index of the next log record.  That way, the logs don't
   * have to be loaded to add one.  Instead, for each log that is added to
   * this execution, the nextLogIndex is used and incremented. */
  protected int nextLogIndex;

  /** caches the child executions by execution name.  This member might be
   * null and is only created from the executions in case its needed.  Note
   * that not all executions are forced to have a name and duplicates are allowed.
   * In case the {@link #executions} change, the executionsMap can be nulled or
   * also updated (but a check needs to be added whether it exists). */
  protected transient Map<String, OpenExecution> executionsMap = null;

  // transient members

  /** the queue of atomic operations to be performed for this execution. */
  protected Queue<AtomicOperation> atomicOperations;

  public enum Propagation {
    UNSPECIFIED, WAIT, EXPLICIT
  }
  protected Propagation propagation = null;

  protected Node previousNode;
  protected Transition previousTransition;
  protected Exception exception;

  // It's important that this refers to a separate entity.  This 
  // execution must do nullpointercheck before accessing the 
  // process modifications.  That way, good performance is guaranteed 
  // for the most common scenario: a persistent execution without 
  // process modifications. 
  protected ProcessModificationsImpl processModifications;

  // construction /////////////////////////////////////////////////////////////
  
  public void initializeProcessInstance(ProcessDefinitionImpl processDefinition, String key) {
    this.processDefinition = processDefinition;
    this.node = (NodeImpl) processDefinition.getInitial();
    this.processInstance = this;
    this.state = STATE_CREATED;
    this.key = key;

    IdGenerator keyGenerator = Environment.getFromCurrent(IdGenerator.class, false);
    if (keyGenerator!=null) {
      this.id = keyGenerator.createId(processDefinition, null, this);
    }
  }

  // execution method : start /////////////////////////////////////////////////

  public void begin() {
    if (state!=STATE_CREATED) {
      throw new PvmException(toString()+" is already begun: "+state);
    }
    ExecutionImpl scopedExecution = initializeScopes();
    this.state = STATE_ACTIVE;
    fire(Event.PROCESS_BEGIN, processDefinition);
    if (node!=null) {
      scopedExecution.performAtomicOperation(EXECUTE_NODE);
    }
  }

  protected ExecutionImpl initializeScopes() {
    LinkedList<NodeImpl> enteredNodes = new LinkedList<NodeImpl>();

    NodeImpl initial = processDefinition.getInitial();
    ExecutionImpl scopedExecution = null;
    
    if (initial!=null) {
      enteredNodes.add(initial);
      NodeImpl parentNode = initial.getParentNode();
      while (parentNode!=null) {
        enteredNodes.addFirst(parentNode);
        parentNode = parentNode.getParentNode();
      }
      
      scopedExecution = this;

      initializeVariables(processDefinition, this);
      initializeTimers(processDefinition);
      
      for (NodeImpl enteredNode: enteredNodes) {
        if (enteredNode.isLocalScope()) {
          scopedExecution.setNode(enteredNode);
          scopedExecution = scopedExecution.createScope(enteredNode);
        }
      }
      
      scopedExecution.setNode(initial);
    }
    return scopedExecution;
  }

  public ExecutionImpl createScope(CompositeElementImpl scope) {
    ExecutionImpl child = createExecution(scope.getName());
    
    // copy the current state from the child execution to the parent execution
    child.setNode(getNode());
    child.setTransition(getTransition());
    child.setPropagation(getPropagation());
    child.setTransitionOrigin(getTransitionOrigin());
    child.setPreviousTransition(getPreviousTransition());
    child.setPreviousNode(getPreviousNode());
    
    child.initializeVariables(scope, this);
    child.initializeTimers(scope);
    
    return child;
  }
  
  public ExecutionImpl destroyScope(CompositeElementImpl scope) {
    destroyTimers(scope);
    destroyVariables(scope, parent);
    
    // copy the current state from the child execution to the parent execution
    getParent().setNode(getNode());
    getParent().setTransition(getTransition());
    getParent().setPropagation(getPropagation());
    getParent().setTransitionOrigin(getTransitionOrigin());
    getParent().setPreviousTransition(getPreviousTransition());
    getParent().setPreviousNode(getPreviousNode());
    
    end();
    parent.removeExecution(this);

    return parent;
  }
  
  // basic object methods /////////////////////////////////////////////////////

  public String toString() {
    if (name!=null) {
      return "execution["+name+"]";
    }
    if (parent==null) {
      return "process-instance";
    }
    return "execution";
  }
  
  // execution method : end ///////////////////////////////////////////////////

  public void end() {
    end(Execution.STATE_ENDED);
  }

  public void end(String state) {
    if (state==null) {
      throw new PvmException("state is null");
    }
    if (state.equals(STATE_ACTIVE)
        || state.equals(STATE_CREATED)
        || state.equals(STATE_INACTIVE)
        || state.equals(STATE_SUSPENDED)
        || state.equals(STATE_ASYNC)) {
      throw new PvmException("invalid end state: "+state);
    }
      
    if (log.isDebugEnabled()) {
      if (state==STATE_ENDED) {
        log.debug(toString()+" ends");
      } else {
        log.debug(toString()+" ends with state "+state);
      }
    }
    
    // end all child executions
    if (executions!=null) {
      for (ExecutionImpl child: executions) {
        child.end(state);
      }
    }
    
    lock(state);

    this.propagation = Propagation.EXPLICIT;
    if (parent==null) {
      fire(Event.PROCESS_END, processDefinition);
      if (superProcessExecution!=null) {
        log.trace(toString()+" signals super process execution");
        superProcessExecution.signal();
      }
    }
  }

  public void cancel() {
    end(Execution.STATE_CANCELLED);
  }

  // execution method : suspend and resume ////////////////////////////////////

  /** @see Execution#suspend() */
  public void suspend() {
    if (isSuspended()) {
      throw new PvmException(toString()+" is suspended");
    }
    lock(STATE_SUSPENDED);
  }

  /** @see Execution#resume() */
  public void resume() {
    if (! isSuspended()) {
      throw new PvmException(toString()+" is not suspended");
    }
    unlock();
  }

  // execution method : signal ////////////////////////////////////////////////

  public void signal() {
    signal(null, (Map)null);
  }

  public void signal(String signal) {
    signal(signal, (Map)null);
  }
  
  public void signal(Map<String, Object> parameters) {
    signal(null, parameters);
  }

  public void signal(String signal, Map<String, Object> parameters) {
    checkLock();
    if (node!=null) {
      performAtomicOperation(new Signal(signal, parameters, node));
    } else if (transition!=null) {
      performAtomicOperation(ExecutionImpl.PROCEED_TO_DESTINATION);
    } else {
      throw new PvmException("execution is not in a node or in a transition");
    }
  }
  
  public void signal(Execution execution) {
    ((ExecutionImpl)execution).signal(null, (Map)null);
  }

  public void signal(String signalName, Execution execution) {
    ((ExecutionImpl)execution).signal(signalName, (Map)null);
  }

  public void signal(Map<String, Object> parameters, Execution execution) {
    ((ExecutionImpl)execution).signal(null, parameters);
  }

  public void signal(String signalName, Map<String, Object> parameters, Execution execution) {
    ((ExecutionImpl)execution).signal(signalName, parameters);
  }

  // execution method : take ////////////////////////////////////////////////
  
  /** @see Execution#takeDefaultTransition() */
  public void takeDefaultTransition() {
    TransitionImpl defaultTransition = node.getDefaultTransition();
    if (defaultTransition==null) {
      throw new PvmException("there is no default transition in "+node);
    }
    take(defaultTransition);
  }

  /** @see Execution#take(String) */
  public void take(String transitionName) {
    if (node==null) {
      throw new PvmException(toString()+" is not positioned in node");
    }
    TransitionImpl transition = findTransition(transitionName);
    if (transition==null) {
      throw new PvmException("there is no transition "+transitionName+" in "+node);
    }
    take(transition);
  }

  /** @see Execution#takeDefaultTransition() */
  public void take(Transition transition) {
    checkLock();

    setPropagation(Propagation.EXPLICIT);
    setTransition((TransitionImpl) transition);
    // copy the current node as the transition origin.  the origin can be different from 
    // the transition source in case a transition is taken from an enclosing node
    setTransitionOrigin(getNode());
    setPreviousTransition(null);

    performAtomicOperation(TAKE_TRANSITION);
  }
  
  public void take(Transition transition, Execution execution) {
    ((ExecutionImpl)execution).take(transition);
  }

  // execution method : execute ///////////////////////////////////////////////

  /** @see Execution#execute(String) */
  public void execute(String nodeName) {
    if (node==null) {
      throw new PvmException("node is null");
    }
    Node nestedNode = node.getNode(nodeName);
    if (nestedNode==null) {
      throw new PvmException("node "+nodeName+" doesn't exist in "+node);
    }
    execute(nestedNode);
  }
  
  /** @see Execution#execute(Node) */
  public void execute(Node node) {
    if (node==null) {
      throw new PvmException("node is null");
    }
    checkLock();
    
    this.propagation = Propagation.EXPLICIT;
    performAtomicOperation(new MoveToChildNode((NodeImpl) node));
  }
  
  // execution method : waitForSignal /////////////////////////////////////////
  
  public void waitForSignal() {
    propagation = Propagation.WAIT;
  }
  
  // execution method : proceed ///////////////////////////////////////////////

  public void proceed() {
    checkLock();

    // in graph based processDefinition languages we assume that a
    // default transition is available
    TransitionImpl defaultTransition = findDefaultTransition();
    if (defaultTransition!=null) {
      take(defaultTransition);
      
    // in block structured processDefinition languages we assume that 
    // there is no default transition and that there is a 
    // parent node of the current node
    } else {
      NodeImpl parentNode = node.getParentNode();

      // if there is a parent node
      if (parentNode!=null) {
        // propagate to the parent
        performAtomicOperation(PROPAGATE_TO_PARENT);
        
      }  else {
        // When we don't know how to proceed, i don't know if it's best to 
        // throw new PvmException("don't know how to proceed");
        // or to end the execution.  Because of convenience for testing, 
        // I opted to end the execution.
        end();
      }
    }
  }

  public void move(Node destination, Execution execution) {
    ((ExecutionImpl)execution).move(destination);
  }

  public void move(Node destination) {
    checkLock();
    setNode((NodeImpl) destination);
  }

  // execution : internal methods /////////////////////////////////////////////

  public void moveTo(NodeImpl destination) {
    // if the parent node needs to know the previous node
    if (destination.isPreviousNeeded()) {
      setPreviousNode(getNode());
      setPreviousTransition(getTransition());
    } else {
      this.previousNode = null;
      this.previousTransition = null;
    }

    // move the execution to the destination
    node = destination;
    transition = null;
    transitionOrigin = null;
  }

  public ExecutionImpl beginNode(NodeImpl node) {
    ExecutionImpl propagatingExecution = this;
    if (node.isLocalScope()) {
      propagatingExecution = createScope(node);
    }
    fire(Event.NODE_BEGIN, node);
    return propagatingExecution;
  }

  public ExecutionImpl endNode(NodeImpl node) {
    ExecutionImpl propagatingExecution = this;
    fire(Event.NODE_END, node);
    if (node.isLocalScope()) {
      propagatingExecution = destroyScope(node);
    }
    return propagatingExecution;
  }

  public synchronized void performAtomicOperation(AtomicOperation operation) {
    if (operation.isAsync(this)) {
      sendContinuationMessage(operation);
    } else {
      performAtomicOperationSync(operation);
    }
  }
  
  // variables ////////////////////////////////////////////////////////////////

  protected void initializeVariables(CompositeElementImpl scope, ExecutionImpl outerExecution) {
    // loop over all variable definitions
    List<VariableDefinitionImpl> variableDefinitions = scope.getVariableDefinitions();
    if (!variableDefinitions.isEmpty()){
      if (log.isTraceEnabled()) {
        log.trace("initializing variables in scope "+scope);
      }
      variables = new HashMap<String, Variable>();
      for (VariableDefinitionImpl variableDefinition: variableDefinitions) {
        String key = variableDefinition.getKey();
        Object value = variableDefinition.getSourceValue(outerExecution);
        Type type = variableDefinition.getType();
        createVariable(key, value, type);
      }
    }
  }

  protected void destroyVariables(CompositeElementImpl scope, ExecutionImpl outerExecution) {
    // loop over all variable definitions
    List<VariableDefinitionImpl> variableDefinitions = scope.getVariableDefinitions();
    if (variableDefinitions!=null) {
      if (log.isTraceEnabled()) {
        log.trace("destroying var scope "+scope);
      }
      
      for (VariableDefinitionImpl variableDefinition: variableDefinitions) {
        String destination = variableDefinition.getDestination();
        if (destination!=null) {
          String key = variableDefinition.getKey();
          Object value = variableDefinition.getDestinationValue(this);
          outerExecution.setVariable(key, value);
        }
      }
    }
  }
  

  public void createVariable(String key, Object value) {
    createVariable(key, value, null, null);
  }

  public void createVariable(String key, Object value, String typeName) {
    createVariable(key, value, typeName, null);
  }

  public void createVariable(String key, Object value, Type type) {
    createVariable(key, value, null, type);
  }

  public void createVariable(String key, Object value, String typeName, Type type) {
    if (isFinished()) {
      throw new PvmException("can't create variable '"+key+"' on "+this+": "+state);
    }

    log.debug("create variable '"+key+"' in '"+this+"' with value '"+value+"'");
    
    if (type==null) {
      Environment environment = Environment.getCurrent();
      if (environment!=null) {
        VariableTypeResolver variableTypeResolver = environment.get(VariableTypeResolver.class);
        if (variableTypeResolver!=null) {
          if (typeName!=null) {
            type = variableTypeResolver.findTypeByName(typeName);
          }
          if (type==null) {
            type = variableTypeResolver.findTypeByMatch(key, value);
          }
        }
      }
    }
    
    Variable variable = null;

    if (type!=null) {
      Class<?> variableClass = type.getVariableClass();
      try {
        log.trace("creating new "+type+" variable "+key);
        variable = (Variable) variableClass.newInstance();
      } catch (Exception e) {
        throw new PvmException("couldn't instantiate variable instance class '"+variableClass.getName()+"'");
      }
      Converter converter = type.getConverter();
      variable.setConverter(converter);

    } else {
      if (value==null) {
        log.trace("creating null variable for "+key);
        variable = new NullVariable();
      } else {
        log.trace("creating new unpersistable variable for "+key);
        variable = new UnpersistableVariable();
      }
    }

    variable.setKey(key);
    variable.setValue(value);
    variable.setProcessInstance(processInstance);
    
    if (variables==null) {
      variables = new HashMap<String, Variable>();
    }
    variables.put(variable.getKey(), variable);
    hasVariables = true;

    // TODO add create-variable-log
  }

  public void setVariable(String key, Object value) {
    if (isFinished()) {
      throw new PvmException("can't update variable '"+key+"' on "+this+": "+state);
    }
    Variable variable = getVariableObject(key);
    // if there is already a variable instance and it doesn't support the current type...
    if ( (variable!=null) 
         && (!variable.supports(value))
       ) {
      // delete the old variable instance
      log.debug("variable type change. deleting '"+key+"' from '"+this+"'");
      removeVariable(key);
      variable = null;
    }

    if (variable!=null) {
      log.debug("updating variable '"+key+"' in '"+this+"' to value '"+value+"'");
      variable.setValue(value);

    } else if (parent==null) {
      createVariable(key, value, null, null);

    } else {
      parent.setVariable(key,value);
    }
  }
  
  public void setVariables(Map<String, Object> variables) {
    if (variables!=null) {
      for (String key: variables.keySet()) {
        Object value = variables.get(key);
        setVariable(key, value);
      }
    }
  }
  
  public Object getVariable(String key) {
    Object value = null;

    Variable variable = getVariableObject(key);
    if (variable!=null) {
      return variable.getValue();
    }
    
    if (parent!=null) {
      return parent.getVariable(key);
    }

    return null;
  }

  public Variable getVariableObject(String key) {
    return (hasVariables ? (Variable) variables.get(key) : null);
  }

  public boolean hasVariable(String key) {
    return ( (hasVariables && variables.containsKey(key))
             || (parent!=null && parent.hasVariable(key))
           );
  }

  public Set<String> getVariableKeys() {
    Set<String> variableKeys = null;
    if (parent!=null) {
      variableKeys = parent.getVariableKeys();
    } else {
      variableKeys = new HashSet<String>();
    }
    if (hasVariables) {
      variableKeys.addAll(variables.keySet());
    }
    return variableKeys;
  }

  public Map<String, Object> getVariables() {
    Map<String, Object> values = null;
    if (parent!=null) {
      values = parent.getVariables();
    } else {
      values = new HashMap<String, Object>();
    }
    if (hasVariables) {
      for (Map.Entry<String, Variable> entry: variables.entrySet()) {
        String name = (String) entry.getKey();
        Variable variable = entry.getValue();
        Object value = variable.getValue();
        values.put(name, value);
      }
    }
    return values;
  }
  
  public boolean hasVariables() {
    return ( hasVariables
             || (parent!=null && parent.hasVariables())
           );
  }

  public boolean removeVariable(String key) {
    if (isFinished()) {
      throw new PvmException("can't remove variable '"+key+"' on "+this+": "+state);
    }

    Variable variable = null;
    if (hasVariables) {
      variable = variables.remove(key);
      if (variables.isEmpty()) {
        hasVariables = false;
      }
      if (variable!=null) {
        return true;
      }
    }
    if (parent!=null) {
      return parent.removeVariable(key);
    }
    // the actual value is not returned to prevent that an object 
    // has to be fetched from the db for it to be deleted
    return false;
  }

  public void removeVariables() {
    if (hasVariables) {
      variables.clear();
    }
    hasVariables = false;
  }

  // timers ///////////////////////////////////////////////////////////////////

  protected void initializeTimers(CompositeElementImpl scope) {
    // initialize the timers
    Set<TimerDefinitionImpl> timerDefinitions = scope.getTimerDefinitions();
    if (!timerDefinitions.isEmpty()) {
      timers = new HashSet<TimerImpl>();
      for (TimerDefinitionImpl timerDefinition: timerDefinitions) {
        createTimer(
            timerDefinition.getEventName(),
            timerDefinition.getSignalName(),
            timerDefinition.getDueDateDescription(),
            timerDefinition.getDueDate(),
            timerDefinition.getRepeat(),
            timerDefinition.isExclusive(),
            timerDefinition.getRetries()
        );
      }
    }
  }

  protected void destroyTimers(CompositeElementImpl scope) {
    log.debug("destroying timers of "+toString());
    if (hasTimers && timers!=null && !timers.isEmpty()) {
      // get the TimerSession from the environment
      Environment environment = Environment.getCurrent();
      if (environment==null) {
        throw new PvmException("non environment for initializing timers");
      }
      TimerSession timerSession = environment.get(TimerSession.class);
      if (timerSession==null) {
        throw new PvmException("no TimerSession in environment for initializing timers");
      }
      for (Timer timer : timers) {
        timerSession.cancel(timer);
      }
      timers.clear();
      hasTimers = false;
    }
  }

  public void createTimer(String eventName, String signalName, String dueDateDescription) {
    createTimer(eventName, signalName, dueDateDescription, null, null, null, null);
  }

  public void createTimer(String eventName, String signalName, String dueDateDescription, String repeat) {
    createTimer(eventName, signalName, dueDateDescription, null, repeat, null, null);
  }

  public void createTimer(String eventName, String signalName, String dueDateDescription, Date dueDate, String repeat, Boolean isExclusive, Integer retries) {
    if ( (eventName==null)
         && (signalName==null)
       ) {
      throw new PvmException("no event or signal specified");
    }
    if (log.isDebugEnabled()) {
      log.debug("creating timer on "+this.toString());
    }
    
    // instantiate the timer
    TimerImpl timerImpl = instantiateTimer();
    // create the bidirectional reference
    timerImpl.setExecution(this);
    timers.add(timerImpl);
    hasTimers = true;
    // setInverseReference(timerImpl);
    
    // initialise all the timer properties
    timerImpl.setEventName(eventName);
    timerImpl.setSignalName(signalName);
    if (dueDate!=null) {
      timerImpl.setDueDate(dueDate);
    } else {
      timerImpl.setDueDateDescription(dueDateDescription);
    }
    
    // the if is added to keep the original default
    if (isExclusive!=null) {
      timerImpl.setExclusive(isExclusive);
    }
    
    // the if is added to keep the original default
    if (retries!=null) {
      timerImpl.setRetries(retries);
    }

    // the if is added to keep the original default
    if (repeat!=null) {
      timerImpl.setRepeat(repeat);
    }

    // get the TimerSession from the environment
    Environment environment = Environment.getCurrent();
    if (environment==null) {
      throw new PvmException("non environment for initializing timers");
    }
    TimerSession timerSession = environment.get(TimerSession.class);
    if (timerSession==null) {
      throw new PvmException("no TimerSession in environment for initializing timers");
    }
    
    // schedule the timer with the TimerSession
    timerSession.schedule(timerImpl);
  }

  public boolean hasTimers() {
    return hasTimers;
  }

  public Set<Timer> getTimers() {
    return (Set) timers;
  }
  
  protected TimerImpl instantiateTimer() {
    return new TimerImpl();
  }

  
  // state ////////////////////////////////////////////////////////////////////

  /** @see Execution#getState() */
  public String getState() {
    return state;
  }

  /** @see Execution#lock(String) */
  public void lock(String state) {
    if (state==null) {
      throw new PvmException("given state is null");
    }
    checkLock();
    log.trace("locking "+this);
    this.state = state;
  }
  
  /** @see Execution#unlock() */
  public void unlock() {
    if (STATE_ACTIVE.equals(state)) {
      throw new PvmException("state is already active");
    }
    log.trace("unlocking "+this);
    this.state = STATE_ACTIVE;
  }

  /** @see Execution#isActive() */
  public boolean isActive() {
    return STATE_ACTIVE.equals(state);
  }

  /** @see Execution#isLocked() */
  public boolean isLocked() {
    return ! isActive();
  }
  
  /** @see Execution#isSuspended() */
  public boolean isSuspended() {
    return STATE_SUSPENDED.equals(state);
  }

  /** @see Execution#isEnded() */
  public boolean isEnded() {
    return STATE_ENDED.equals(state);
  }

  /** @see Execution#isFinished() */
  public boolean isFinished() {
    return STATE_ENDED.equals(state)
           || STATE_CANCELLED.equals(state);
  }

  // state : internal methods /////////////////////////////////////////////////

  protected void checkLock() {
    if (!STATE_ACTIVE.equals(state)) {
      throw new PvmException(toString()+" is not active: "+state);
    }
  }

  // asynchronous continuations ////////////////////////////////////////////////  

  public void sendContinuationMessage(AtomicOperation operation) {
    Environment environment = Environment.getCurrent();
    MessageSession messageSession = environment.get(MessageSession.class);
    if (messageSession==null) {
      throw new PvmException("no message-session configured to send asynchronous continuation message");
    }
    MessageImpl<?> asyncMessage = operation.createAsyncMessage(this);
    lock("async continuation message "+asyncMessage);
    messageSession.send(asyncMessage);
  }

  public void performAtomicOperationSync(AtomicOperation operation) {
    if (atomicOperations==null) {
      
      // initialise the fifo queue of atomic operations
      atomicOperations = new LinkedList<AtomicOperation>();
      atomicOperations.offer(operation);
      
      try {
        while (! atomicOperations.isEmpty()) {
          AtomicOperation atomicOperation = atomicOperations.poll();
          atomicOperation.perform(this);
        }

      } catch (RuntimeException e ) {
        throw e;
      } finally {
        atomicOperations = null;
      }
    } else {
      atomicOperations.offer(operation);
    }
  }

  // events ///////////////////////////////////////////////////////////////////

  /** @see Execution#fire(String, ObservableElement) */
  public void fire(String eventName, ObservableElement eventSource) {
    fire(eventName, eventSource, (ObservableElementImpl) eventSource);
  }

  /** fires the event on the given *processElement* and then propagates the event 
   * up to the *processElement* parent chain. */
  void fire(String eventName, ObservableElement eventSource, ObservableElementImpl observableElement) {
    if (observableElement!=null) {
      EventImpl event = (EventImpl) observableElement.getEvent(eventName);
      if (event!=null) {
        if (log.isTraceEnabled()) {
          if (observableElement==eventSource) {
            log.trace("firing "+event+" on "+eventSource);
          } else {
            log.trace("firing "+event+" on "+observableElement+", propagated from source "+eventSource);
          }
        }
        fire(event, eventSource, observableElement);
      }
      propagateEvent(eventName, eventSource, observableElement);
    }
  }

  /** this method enables specific process languages to overwrite the event propagation behaviour */
  protected void propagateEvent(String eventName, ObservableElement eventSource, ObservableElementImpl observableElement) {
    fire(eventName, eventSource, observableElement.getParent());
  }

  /** fires the given event without propagation */
  void fire(EventImpl event, ObservableElement eventSource, ObservableElement observableElement) {
    try {
      this.event = event;
      this.eventSource = (ObservableElementImpl) eventSource;
      
      List<EventListenerReference> eventListenerReferences = event.getListenerReferences();
      
      if (eventListenerReferences!=null) {
        for (EventListenerReference eventListenerReference: eventListenerReferences) {
          
          if ( (observableElement.equals(eventSource)) // this event is not propagated
               || (eventListenerReference.isPropagationEnabled()) // propagation is allowed
             ) {
            EventListener eventListener = eventListenerReference.get();
            
            log.trace("executing "+eventListener+" for "+event);
            try {
              // TODO can/should this invocation be unified with the exception handler invocation of the event notification method?
              eventListener.notify(this);
            } catch (Exception e) {
              log.trace("exception during action: "+e);
              handleException((ObservableElementImpl) observableElement, event, eventListenerReference, e, "couldn't run action "+eventListener);
            }
          }
        }
      }

    } finally {
      this.eventSource = null;
      this.event = null;
    }
  }
  
  public void handleException(ObservableElementImpl observableElement,
                              EventImpl event,
                              EventListenerReference eventListenerReference,
                              Exception exception,
                              String rethrowMessage) {
    
    List<ProcessElementImpl> processElements = new ArrayList<ProcessElementImpl>();
    if (eventListenerReference!=null) {
      processElements.add(eventListenerReference);
    }
    if (event!=null) {
      processElements.add(event);
    }
    while (observableElement!=null) {
      processElements.add(observableElement);
      observableElement = observableElement.getParent();
    }
    
    for (ProcessElementImpl processElement: processElements) {
      List<ExceptionHandlerImpl> exceptionHandlers = processElement.getExceptionHandlers();
      if (exceptionHandlers!=null) {
        for (ExceptionHandlerImpl exceptionHandler: exceptionHandlers) {
          if (exceptionHandler.matches(exception)) {
            try {
              exceptionHandler.handle(this, exception);
              return;
            } catch (Exception rethrowException) {
              if (!exceptionHandler.isRethrowMasked()) {
                exception = rethrowException;
              }
            }
            break;
          }
        }
      }
    }

    log.trace("rethrowing exception cause no exception handler for "+exception);
    ExceptionHandlerImpl.rethrow(exception, rethrowMessage+": "+exception.getMessage());
  }
  

  /** searches for an event up the process element parent hierarchy starting 
   * from the given process element and returns an event or null if no such 
   * event exists. */ 
  EventImpl findEvent(String eventName, ObservableElementImpl observableElement) {
    EventImpl event = null;
    while ( (event==null)
            && (observableElement!=null)
          ) {
      event = (EventImpl) observableElement.getEvent(eventName);
      if (event==null) {
        observableElement = observableElement.getParent();
      }
    }
    return event;
  }

  // comments /////////////////////////////////////////////////////////////////
  
  public Comment createComment(String message) {
    if (message==null) {
      throw new PvmException("message is null");
    }
    CommentImpl comment = new CommentImpl(message);
    addComment(comment);
    return comment;
  }
  
  public void removeComment(Comment comment) {
    throw new UnsupportedOperationException("please implement me");
  }

  public void addComment(CommentImpl comment) {
    if (comment==null) {
      throw new PvmException("comment is null");
    }
    if (comments==null) {
      comments = new LinkedHashSet<CommentImpl>();
    }
    comments.add(comment);
  }

  // child executions /////////////////////////////////////////////////////////

  /** @see Execution#createProcessInstance() */
  public ExecutionImpl createExecution() {
    return createExecution((String)null);
  }

  public Execution createExecution(Execution parent) {
    return ((ExecutionImpl)parent).createExecution();
  }

  public Execution createExecution(String name, Execution parent) {
    return ((ExecutionImpl)parent).createExecution(name);
  }

  /** @see Execution#createProcessInstance(String) */
  public ExecutionImpl createExecution(String name) {
    // creating a child execution implies that this execution 
    // is not a leave any more and therefore, it is inactivated
    if (isActive()) {
      lock(STATE_INACTIVE);
      propagation = Propagation.EXPLICIT;
    }

    // create child execution
    ExecutionImpl childExecution = newChildExecution();
    childExecution.processDefinition = this.processDefinition;
    childExecution.processInstance = this.processInstance;
    childExecution.node = this.node;
    childExecution.state = STATE_ACTIVE;
    childExecution.name = name;
    log.debug("creating "+childExecution);
    // add it to this execution
    addExecution(childExecution);
    // invalidate the cached executionsMap
    executionsMap = null;
    
    IdGenerator keyGenerator = Environment.getFromCurrent(IdGenerator.class, false);
    if (keyGenerator!=null) {
      childExecution.id = keyGenerator.createId(processDefinition, this, childExecution);
    }
    
    return childExecution;
  }

  protected ExecutionImpl newChildExecution() {
    return new ExecutionImpl();
  }

  public void addExecution(Execution execution) {
    ExecutionImpl executionImpl = (ExecutionImpl) execution;
    executionImpl.parent = this;
    if (executions==null) {
      executions = new ArrayList<ExecutionImpl>();
    }
    executions.add(executionImpl);
  }

  /** @see Execution#getExecution(String) */
  public ExecutionImpl getExecution(String name) {
    Map<String, OpenExecution> executionsMap = getExecutionsMap();
    return (ExecutionImpl) (executionsMap!=null ? executionsMap.get(name) : null);
  }

  public void removeExecution(Execution child) {
    if (executions!=null) {
      if (executions.remove(child)) {
        if ( state.equals(STATE_INACTIVE) && 
            (executions.isEmpty())
          ) {
         if (log.isTraceEnabled()) {
           log.trace("last child execution was removed; unlocking");
         }
         state = STATE_ACTIVE;
        } else if (log.isTraceEnabled()) {
          log.trace("removed "+child+" from "+this);
        }
        // invalidate the executionsMap cache
        executionsMap = null;
      } else {
        throw new PvmException(child+" is not a child execution of "+this);
      }
    }
  }

  public void removeExecution(Execution child, Execution parent) {
    ((ExecutionImpl)parent).removeExecution(child);
  }

  public Map<String, OpenExecution> getExecutionsMap() {
    if ( (executionsMap==null)
         && (executions!=null)
        ) {
      // initialize executionsMap cache
      executionsMap = new HashMap<String, OpenExecution>();
      for(ExecutionImpl execution: executions) {
        String executionName = execution.getName();
        // the next test makes sure that the first execution wins
        // in case there are multiple executions with the same name
        if (! executionsMap.containsKey(executionName)) {
          executionsMap.put(executionName, execution);
        }
      }
    }
    return executionsMap;
  }
  
  public boolean hasExecution(String name) {
    return ( (getExecutionsMap()!=null)
             && executionsMap.containsKey(name)
           );
  }

  // sub process creation /////////////////////////////////////////////////////

  public ClientProcessInstance createSubProcessInstance(ClientProcessDefinition processDefinition) {
    return createSubProcessInstance(processDefinition, null);
  }
  
  public ClientProcessInstance createSubProcessInstance(ClientProcessDefinition processDefinition, String key) {
    if (subProcessInstance!=null) {
      throw new PvmException(toString()+" already has a sub process instance: "+subProcessInstance);
    }
    subProcessInstance = (ExecutionImpl) processDefinition.createProcessInstance(key);
    subProcessInstance.setSuperProcessExecution(this);
    return subProcessInstance;
  }
  
  public ClientProcessInstance beginSubProcessInstance(ClientProcessDefinition processDefinition) {
    return beginSubProcessInstance(processDefinition, null);
  }
  
  public ClientProcessInstance beginSubProcessInstance(ClientProcessDefinition processDefinition, String key) {
    createSubProcessInstance(processDefinition, key);
    subProcessInstance.begin();
    return subProcessInstance;
  }
  
  // node name ////////////////////////////////////////////////////////////////
  
  public String getNodeName() {
    if (node==null) {
      return null;
    }
    return node.getName();
  }
  
  ////////////////////////////////////////////////////////////////////////////////

  public void addLog(ProcessLog processLog) {
    Environment environment = Environment.getCurrent();
    if (environment!=null) {
      LogSession logSession = environment.get(LogSession.class);
      if (logSession!=null) {
        processLog.setExecution(this);
        processLog.setTime(Clock.getCurrentTime());
        logSession.add(processLog);
      }
    }
  }

  public int nextLogIndex() {
    return nextLogIndex++;
  }

  // overridable by process languages /////////////////////////////////////////
  
  /** by default this will use {@link NodeImpl#findOutgoingTransition(String)} to 
   * search for the outgoing transition, which includes a search over the parent chain 
   * of the current node.  This method allows process languages to overwrite this default 
   * implementation of the transition lookup by transitionName.*/
  protected TransitionImpl findTransition(String transitionName) {
    return node.findOutgoingTransition(transitionName);
  }

  protected TransitionImpl findDefaultTransition() {
    return node.findDefaultTransition();
  }

  
  // extensions ///////////////////////////////////////////////////////////////

  public <T> T getExtension(Class<T> extensionClass) {
    if (extensionClass==null) {
      throw new PvmException("extensionClass is null.  and this execution doesn't support extensions");
    }
    throw new PvmException("unsuppported extension "+extensionClass.getName());
  }

  // equals ///////////////////////////////////////////////////////////////////
  // hack to support comparing hibernate proxies against the real objects
  // since this always falls back to ==, we don't need to overwrite the hashcode
  public boolean equals(Object o) {
    return EqualsUtil.equals(this, o);
  }

  // getters and setters /////////////////////////////////////////////////////////

  public List<Comment> getComments() {
    if (comments==null) {
      return Collections.emptyList();
    }
    return new ArrayList<Comment>(comments);
  }

  public Event getEvent() {
    return event;
  }
  public ObservableElement getEventSource() {
    return eventSource;
  }
  public Collection<OpenExecution> getExecutions() {
    return (Collection) executions;
  }
  public String getName() {
    return name;
  }
  public ExecutionImpl getParent() {
    return parent;
  }
  public int getPriority() {
    return priority;
  }
  public ProcessDefinitionImpl getProcessDefinition() {
    return processDefinition;
  }
  public TransitionImpl getTransition() {
    return transition;
  }
  public void setEvent(EventImpl event) {
    this.event = event;
  }
  public void setEventSource(ObservableElementImpl eventSource) {
    this.eventSource = eventSource;
  }
  public void setPriority(int priority) {
    this.priority = priority;
  }
  public void setTransition(TransitionImpl transition) {
    this.transition = transition;
  }
  public Node getPreviousNode() {
    return previousNode;
  }
  public Transition getPreviousTransition() {
    return previousTransition;
  }
  public ExecutionImpl getProcessInstance() {
    return processInstance;
  }
  public void setProcessInstance(ExecutionImpl processInstance) {
    this.processInstance = processInstance;
  }
  public void setComments(Set<CommentImpl> comments) {
    this.comments = comments;
  }
  public NodeImpl getTransitionOrigin() {
    return transitionOrigin;
  }
  public void setTransitionOrigin(NodeImpl transitionOrigin) {
    this.transitionOrigin = transitionOrigin;
  }
  public Exception getException() {
    return exception;
  }
  public void setException(Exception exception) {
    this.exception = exception;
  }
  public ProcessModificationsImpl getProcessModifications() {
    return processModifications;
  }
  public void setProcessModifications(ProcessModificationsImpl processModifications) {
    this.processModifications = processModifications;
  }
  public String getKey() {
    return key;
  }
  public Propagation getPropagation() {
    return propagation;
  }
  public void setPropagation(Propagation propagation) {
    this.propagation = propagation;
  }
  public void setName(String name) {
    this.name = name;
  }
  public void setState(String state) {
    this.state = state;
  }
  public void setExecutions(Collection<ExecutionImpl> executions) {
    this.executions = executions;
  }
  public void setParent(ExecutionImpl parent) {
    this.parent = parent;
  }
  public void setPreviousNode(Node previousNode) {
    this.previousNode = previousNode;
  }
  public void setPreviousTransition(Transition previousTransition) {
    this.previousTransition = previousTransition;
  }
  public void setProcessDefinition(ProcessDefinitionImpl processDefinition) {
    this.processDefinition = processDefinition;
  }
  public ExecutionImpl getSuperProcessExecution() {
    return superProcessExecution;
  }
  public void setSuperProcessExecution(ExecutionImpl superProcessExecution) {
    this.superProcessExecution = superProcessExecution;
  }
  public ExecutionImpl getSubProcessInstance() {
    return subProcessInstance;
  }
  public void setSubProcessInstance(ExecutionImpl subProcessExecution) {
    this.subProcessInstance = subProcessExecution;
  }
  public NodeImpl getNode() {
    return node;
  }
  public void setNode(NodeImpl node) {
    this.node = node;
  }
  public long getDbid() {
    return dbid;
  }
  public void setKey(String key) {
    this.key = key;
  }
  public String getId() {
    return id;
  }
  public void setId(String id) {
    this.id = id;
  }
}
