/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.model.op;

import java.util.ArrayList;
import java.util.List;

import org.jbpm.pvm.internal.job.MessageImpl;
import org.jbpm.pvm.internal.log.Log;
import org.jbpm.pvm.internal.model.ExecutionImpl;
import org.jbpm.pvm.internal.model.NodeImpl;
import org.jbpm.pvm.internal.model.ObservableElementImpl;
import org.jbpm.pvm.internal.model.TransitionImpl;
import org.jbpm.pvm.model.Condition;
import org.jbpm.pvm.model.Event;

public class TakeTransition implements AtomicOperation {

  private static Log log = Log.getLog(TakeTransition.class.getName());
  
  public boolean isAsync(ExecutionImpl execution) {
    return execution.getNode().isLeaveAsync()
           || execution.getTransition().isTakeAsync();
  }

  public void perform(ExecutionImpl execution) {
    TransitionImpl transition = execution.getTransition();

    if (execution.getName()!=null) {
      log.debug(execution.toString()+" takes "+transition);
    } else {
      log.debug("taking "+transition);
    }
    
    List<NodeImpl> leftNodes = getNodesLeft(execution.getNode(), transition.getDestination());
    ExecutionImpl propagatingExecution = execution;
    for (NodeImpl leftNode : leftNodes) {
      propagatingExecution = propagatingExecution.endNode(leftNode);
    }
    
    propagatingExecution.setNode(null);
    propagatingExecution.fire(Event.TRANSITION_TAKE, transition);
    
    boolean wait = false;
    
    Condition waitCondition = transition.getWaitCondition();
    if (waitCondition!=null) {
      wait = waitCondition.evaluate(propagatingExecution);
    }
    
    if (!wait) {
      propagatingExecution.performAtomicOperation(ExecutionImpl.PROCEED_TO_DESTINATION);
    }
  }

  List<NodeImpl> getNodesLeft(NodeImpl source, NodeImpl destination) {
    List<NodeImpl> nodesLeft = new ArrayList<NodeImpl>();
    
    if (source.equals(destination)) {
      nodesLeft.add(source);
    } else {
      List<ObservableElementImpl> destinationChain = destination.getParentChain();
      
      if (!destinationChain.contains(source)) {
        NodeImpl sourceNode = source;
        while ( (sourceNode!=null)
                && (!destinationChain.contains(sourceNode))
              ) {
          nodesLeft.add(sourceNode);
          sourceNode = sourceNode.getParentNode(); 
        }
      }
    }
    
    return nodesLeft;
  }

  public String toString() {
    return "take-transition";
  }

  public MessageImpl<?> createAsyncMessage(ExecutionImpl execution) {
    return new TakeTransitionMessage(execution);
  }
}