/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.processlog;

import java.io.Serializable;
import java.util.Date;
import java.util.List;

import org.jbpm.pvm.Execution;
import org.jbpm.pvm.internal.model.ExecutionImpl;
import org.jbpm.pvm.session.LogSession;

/** base class for process logs.  Process logs are dispatched to the 
 * {@link LogSession} that is configured in the environment.
 * 
 * <p>ProcessLogs that are send to a LogSession is the probe mechanism 
 * provided to listen into the process execution progress and details.
 * It is the mechanism to collect process history and process statistics.
 * </p>
 * 
 * @author Tom Baeyens
 */
public abstract class ProcessLogImpl implements Serializable, ProcessLog {
  
  private static final long serialVersionUID = 1L;

  protected long dbid;
  protected int index = -1;
  protected Date time;
  protected Execution execution;
  protected Execution processInstance;
  
  public ProcessLogImpl() {
  }

  public void setExecution(ExecutionImpl execution) {
    this.execution = execution;
    this.processInstance = (execution!=null ? execution.getProcessInstance() : null);
  }

  /** text id of the type of this process log used for 
   * XML serialization. */
  public abstract String getType();

  /** xml representation of the process log properties
   * used for XML serialization.  Properties time, execution 
   * and processInstance should not be included. This method 
   * is made abstract to remind and encourage 
   * implementors to provide XML serialization capabilities to 
   * the fixed process log XML schema.  But it is allowed to 
   * return null if XML serialisation if you're sure that 
   * XML serialization will never be necessary. */
  public abstract List<ProcessLogProperty> getProperties();

  /** provides a text description for this update
   * which can be used e.g. in the admin web console.
   * This texts hould not contain time, execution
   * nor processInstance. 
   * The default implementation will include the 
   * {@link #getType() type} and then append all the top 
   * level properties. */
  public String toString() {
    StringBuffer text = new StringBuffer();
    text.append(getType());
    text.append('[');
    List<ProcessLogProperty> properties = getProperties();
    if (properties!=null) {
      for (int i=0; i<properties.size(); i++) {
        ProcessLogProperty property = properties.get(i);
        text.append(property.name);
        if (property.value!=null) {
          text.append('=');
          text.append(property.value);
        }
        if (i < (properties.size()-1)) {
          text.append('|');
        }
      }
    }
    text.append(']');
    return text.toString();
  }

  public long getDbid() {
    return dbid;
  }
  public Date getTime() {
    return time;
  }
  public void setTime(Date date) {
    this.time = date;
  }
  public Execution getExecution() {
    return execution;
  }
  public int getIndex() {
    return index;
  }
  public Execution getProcessInstance() {
    return processInstance;
  }
  public void setIndex(int index) {
    this.index = index;
  }
}
