/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.backend.wasm.ir2wasm

import com.intellij.util.containers.reverse
import org.jetbrains.kotlin.backend.common.compilationException
import org.jetbrains.kotlin.backend.common.serialization.cityHash64
import org.jetbrains.kotlin.backend.wasm.MultimoduleCompileOptions
import org.jetbrains.kotlin.backend.wasm.WasmBackendContext
import org.jetbrains.kotlin.backend.wasm.importedStringConstants
import org.jetbrains.kotlin.backend.wasm.ir2wasm.WasmCompiledModuleFragment.*
import org.jetbrains.kotlin.backend.wasm.utils.fitsLatin1
import org.jetbrains.kotlin.ir.backend.js.ic.IrICProgramFragment
import org.jetbrains.kotlin.ir.util.IdSignature
import org.jetbrains.kotlin.wasm.ir.*
import org.jetbrains.kotlin.wasm.ir.WasmFunction
import org.jetbrains.kotlin.wasm.ir.source.location.SourceLocation
import java.util.IdentityHashMap
import kotlin.collections.MutableMap
import kotlin.collections.mutableMapOf
import kotlin.collections.set

class BuiltinIdSignatures(
    val throwable: IdSignature?,
    val kotlinAny: IdSignature?,
    val tryGetAssociatedObject: IdSignature?,
    val jsToKotlinAnyAdapter: IdSignature?,
    val jsToKotlinStringAdapter: IdSignature?,
    val unitGetInstance: IdSignature?,
    val runRootSuites: IdSignature?,
    val createString: IdSignature?,
    val registerModuleDescriptor: IdSignature?,
)

class WasmCompiledFileFragment(
    val fragmentTag: String?,

    val definedFunctions: MutableMap<IdSignature, WasmFunction> = mutableMapOf(),
    val definedGlobalFields: MutableMap<IdSignature, WasmGlobal> = mutableMapOf(),
    val definedGlobalVTables: MutableMap<IdSignature, WasmGlobal> = mutableMapOf(),
    val definedGlobalClassITables: MutableMap<IdSignature, WasmGlobal> = mutableMapOf(),
    val definedRttiGlobal: MutableMap<IdSignature, WasmGlobal> = mutableMapOf(),
    val definedRttiSuperType: MutableMap<IdSignature, IdSignature?> = mutableMapOf(),
    val definedGcTypes: MutableMap<IdSignature, WasmTypeDeclaration> = mutableMapOf(),
    val definedVTableGcTypes: MutableMap<IdSignature, WasmStructDeclaration> = mutableMapOf(),
    val definedFunctionTypes: MutableMap<IdSignature, WasmFunctionType> = mutableMapOf(),

    val globalLiterals: MutableSet<LiteralGlobalSymbol> = mutableSetOf(),
    val globalLiteralsId: MutableMap<String, WasmSymbol<Int>> = mutableMapOf(),
    val stringLiteralId: MutableMap<String, WasmSymbol<Int>> = mutableMapOf(),

    val constantArrayDataSegmentId: MutableMap<Pair<List<Long>, WasmType>, WasmSymbol<Int>> = mutableMapOf(),
    val jsFuns: MutableMap<IdSignature, JsCodeSnippet> = mutableMapOf(),
    val jsModuleImports: MutableMap<IdSignature, String> = mutableMapOf(),
    val jsBuiltinsPolyfills: MutableMap<String, String> = mutableMapOf(),
    val exports: MutableList<WasmExport<*>> = mutableListOf(),
    val mainFunctionWrappers: MutableList<IdSignature> = mutableListOf(),
    var testFunctionDeclarators: MutableList<IdSignature> = mutableListOf(),
    val equivalentFunctions: MutableList<Pair<String, IdSignature>> = mutableListOf(),
    val jsModuleAndQualifierReferences: MutableSet<JsModuleAndQualifierReference> = mutableSetOf(),
    val classAssociatedObjectsInstanceGetters: MutableList<ClassAssociatedObjects> = mutableListOf(),
    var builtinIdSignatures: BuiltinIdSignatures? = null,
    val objectInstanceFieldInitializers: MutableList<IdSignature> = mutableListOf(),
    val nonConstantFieldInitializers: MutableList<IdSignature> = mutableListOf(),
) : IrICProgramFragment()

enum class ExceptionTagType { WASM_TAG, JS_TAG, TRAP }

class WasmCompiledModuleFragment(
    private val wasmCompiledFileFragments: List<WasmCompiledFileFragment>,
    private val isWasmJsTarget: Boolean
) {
    // Used during linking
    private val serviceCodeLocation = SourceLocation.NoLocation("Generated service code")

    private val stringDataSectionIndex = WasmImmediate.DataIdx(0)
    private val stringAddressesAndLengthsIndex = WasmImmediate.DataIdx(1)

    private inline fun tryFindBuiltInFunction(select: (BuiltinIdSignatures) -> IdSignature?): IdSignature? {
        for (fragment in wasmCompiledFileFragments) {
            val builtinSignatures = fragment.builtinIdSignatures ?: continue
            val signature = select(builtinSignatures) ?: continue
            return signature.takeIf { fragment.definedFunctions.containsKey(it) } // Can be removed by DCE
        }
        return null
    }

    private inline fun tryFindBuiltInType(select: (BuiltinIdSignatures) -> IdSignature?): IdSignature? {
        for (fragment in wasmCompiledFileFragments) {
            val builtinSignatures = fragment.builtinIdSignatures ?: continue
            val signature = select(builtinSignatures) ?: continue
            return signature.takeIf { fragment.definedGcTypes.containsKey(it) } // Can be removed by DCE
        }
        return null
    }

    class JsCodeSnippet(val importName: WasmSymbol<String>, val jsCode: String)

    private fun partitionDefinedAndImportedFunctions(definedDeclarations: DefinedDeclarationsResolver): Pair<MutableList<WasmFunction.Defined>, MutableList<WasmFunction.Imported>> {
        val definedFunctions = mutableListOf<WasmFunction.Defined>()
        val importedFunctions = mutableListOf<WasmFunction.Imported>()
        definedDeclarations.functions.values.distinct().forEach { function ->
            when (function) {
                is WasmFunction.Defined -> definedFunctions.add(function)
                is WasmFunction.Imported -> importedFunctions.add(function)
            }
        }
        return definedFunctions to importedFunctions
    }

    private fun createAndExportServiceFunctions(
        definedDeclarations: DefinedDeclarationsResolver,
        stringEntities: StringLiteralWasmEntities,
        stringPoolSize: Int,
        initializeUnit: Boolean,
        stringPoolSizeWithGlobals: Int,
        wasmElements: MutableList<WasmElement>,
        exports: MutableList<WasmExport<*>>,
    ) {
        stringAddressesAndLengthsField(
            definedDeclarations = definedDeclarations,
        )

        createFieldInitializerFunction(definedDeclarations = definedDeclarations, stringPoolSize = stringPoolSize)

        val registerAssociatedObjectGetter =
            createAssociatedObjectGetter(
                definedDeclarations = definedDeclarations,
                wasmElements = wasmElements,
            )

        createAndExportMasterInitFunction(
            definedDeclarations = definedDeclarations,
            exports = exports,
            registerAssociatedObjectGetter = registerAssociatedObjectGetter,
            initializeUnit = initializeUnit
        )

        createStringPoolField(definedDeclarations, stringPoolSizeWithGlobals)

        if (isWasmJsTarget) {
            createStringLiteralFunction(
                definedDeclarations = definedDeclarations,
                stringEntities = stringEntities,
                wasmElements = wasmElements,
                stringLiteralType = StringLiteralType.JsString,
            )
        }

        createStringLiteralFunction(
            definedDeclarations = definedDeclarations,
            stringEntities = stringEntities,
            wasmElements = wasmElements,
            stringLiteralType = StringLiteralType.Latin1,
        )

        createStringLiteralFunction(
            definedDeclarations = definedDeclarations,
            stringEntities = stringEntities,
            wasmElements = wasmElements,
            stringLiteralType = StringLiteralType.Utf16,
        )

        createStartUnitTestsFunction(definedDeclarations, exports)
    }

    private class StringLiteralWasmEntities(
        val createStringSignature: IdSignature,
        val kotlinStringType: WasmType,
        val wasmCharArrayType: WasmType,
        val wasmCharArrayDeclaration: IdSignature,
    )

    private fun getStringLiteralWasmEntities(
        definedDeclarations: DefinedDeclarationsResolver,
    ): StringLiteralWasmEntities {
        val createStringSignature = tryFindBuiltInFunction { it.createString }
            ?: compilationException("kotlin.createString is not file in fragments", null)
        val createStringFunction = definedDeclarations.functions[createStringSignature]
            ?: compilationException("kotlin.createString is not file in fragments", null)

        val createStringFunctionTypeSignature = (createStringFunction.type as FunctionHeapTypeSymbol).type
        val createStringFunctionType = definedDeclarations.functionTypes.getValue(createStringFunctionTypeSignature)
        val kotlinStringType = createStringFunctionType.resultTypes[0]
        val wasmCharArrayType = createStringFunctionType.parameterTypes[0]
        val wasmCharArrayDeclaration = (wasmCharArrayType.getHeapType() as GcHeapTypeSymbol).type
        val wasmStringArrayDeclaration =
            WasmArrayDeclaration("string_array", WasmStructFieldDeclaration("string", kotlinStringType, true))
        definedDeclarations.gcTypes[Synthetics.GcTypes.wasmStringArrayType.value] = wasmStringArrayDeclaration

        val newStringLiteralFunctionType = WasmFunctionType(listOf(WasmI32), listOf(kotlinStringType))
        definedDeclarations.functionTypes[Synthetics.FunctionHeapTypes.stringLiteralFunctionType.type] = newStringLiteralFunctionType

        val newStringLiteralJsFunctionType = WasmFunctionType(listOf(WasmI32, WasmRefType(WasmHeapType.Simple.Extern)), listOf(kotlinStringType))
        definedDeclarations.functionTypes[Synthetics.FunctionHeapTypes.jsStringLiteralFunctionType.type] = newStringLiteralJsFunctionType

        return StringLiteralWasmEntities(
            createStringSignature = createStringSignature,
            kotlinStringType = kotlinStringType,
            wasmCharArrayType = wasmCharArrayType,
            wasmCharArrayDeclaration = wasmCharArrayDeclaration,
        )
    }

    fun linkWasmCompiledFragments(
        multimoduleOptions: MultimoduleCompileOptions?,
        exceptionTagType: ExceptionTagType
    ): WasmModule {
        val definedDeclarations = getDefinedDeclarationsFromFragments()

        val data = mutableListOf<WasmData>()
        val stringPoolSize = bindStringPoolSymbolsAndGetSize(data)
        val stringPoolSizeWithGlobals = bindGlobalLiterals(definedDeclarations, stringPoolSize)

        bindConstantArrayDataSegmentIds(data)

        val exports = mutableListOf<WasmExport<*>>()
        wasmCompiledFileFragments.flatMapTo(exports) { it.exports }

        val memories = createAndExportMemory(exports, multimoduleOptions?.stdlibModuleNameForImport)
        val (importedMemories, definedMemories) = memories.partition { it.importPair != null }

        val parameterlessNoReturnFunctionType = WasmFunctionType(emptyList(), emptyList())
        definedDeclarations.functionTypes[Synthetics.FunctionHeapTypes.parameterlessNoReturnFunctionType.type] = parameterlessNoReturnFunctionType

        val stringEntities = getStringLiteralWasmEntities(definedDeclarations)

        createAndBindSpecialITableTypes(definedDeclarations)
        createAndBindRttiTypeDeclaration(definedDeclarations)

        val elements = mutableListOf<WasmElement>()
        createAndExportServiceFunctions(
            definedDeclarations = definedDeclarations,
            stringEntities = stringEntities,
            stringPoolSize = stringPoolSize,
            initializeUnit = multimoduleOptions?.initializeUnit ?: true,
            stringPoolSizeWithGlobals = stringPoolSizeWithGlobals,
            wasmElements = elements,
            exports = exports,
        )

        val globals = getGlobals(definedDeclarations)

        val tags = getTags(definedDeclarations, exceptionTagType)
        require(tags.size <= 1) { "Having more than 1 tag is not supported" }

        val (importedTags, definedTags) = tags.partition { it.importPair != null }

        val (importedGlobals, definedGlobals) = globals.partition { it.importPair != null }

        val (definedFunctions, importedFunctions) = partitionDefinedAndImportedFunctions(definedDeclarations)

        val importsInOrder = mutableListOf<WasmNamedModuleField>()
        importsInOrder.addAll(importedFunctions)
        importsInOrder.addAll(importedTags)
        importsInOrder.addAll(importedGlobals)
        importsInOrder.addAll(importedMemories)

        val recursiveTypeGroups = getTypes(definedDeclarations)

        return WasmModule(
            resolver = definedDeclarations,
            recGroups = recursiveTypeGroups,
            importsInOrder = importsInOrder,
            importedFunctions = importedFunctions,
            importedMemories = importedMemories,
            definedFunctions = definedFunctions,
            importedTags = importedTags,
            tables = emptyList(),
            memories = definedMemories,
            globals = definedGlobals,
            importedGlobals = importedGlobals,
            exports = exports,
            startFunction = null,  // Module is initialized via export call
            elements = elements,
            data = data,
            dataCount = true,
            tags = definedTags
        ).apply { calculateIds() }
    }

    private fun createAndBindSpecialITableTypes(
        definedDeclarations: DefinedDeclarationsResolver,
    ) {
        val wasmAnyArrayType = WasmArrayDeclaration(
            name = "AnyArray",
            field = WasmStructFieldDeclaration("", WasmRefNullType(WasmHeapType.Simple.Any), false)
        )
        definedDeclarations.gcTypes[Synthetics.GcTypes.wasmAnyArrayType.value] = wasmAnyArrayType

        val specialSlotITableTypeSlots = mutableListOf<WasmStructFieldDeclaration>()
        val wasmAnyRefStructField = WasmStructFieldDeclaration("", WasmAnyRef, false)
        repeat(WasmBackendContext.SPECIAL_INTERFACE_TABLE_SIZE) {
            specialSlotITableTypeSlots.add(wasmAnyRefStructField)
        }
        specialSlotITableTypeSlots.add(
            WasmStructFieldDeclaration(
                name = "",
                type = WasmRefNullType(Synthetics.HeapTypes.wasmAnyArrayType),
                isMutable = false
            )
        )
        val specialSlotITableType = WasmStructDeclaration(
            name = "SpecialITable",
            fields = specialSlotITableTypeSlots,
            superType = null,
            isFinal = true
        )
        definedDeclarations.gcTypes[Synthetics.GcTypes.specialSlotITableType.value] = specialSlotITableType
    }

    private fun getTags(definedDeclarations: DefinedDeclarationsResolver, exceptionTagType: ExceptionTagType): List<WasmTag> {
        val exceptionTag = when (exceptionTagType) {
            ExceptionTagType.TRAP -> null
            ExceptionTagType.JS_TAG -> {
                val jsExceptionTagFuncType = WasmFunctionType(
                    parameterTypes = listOf(WasmExternRef),
                    resultTypes = emptyList()
                )
                definedDeclarations.functionTypes[Synthetics.FunctionHeapTypes.jsExceptionTagFuncType.type] = jsExceptionTagFuncType
                WasmTag(Synthetics.FunctionHeapTypes.jsExceptionTagFuncType, WasmImportDescriptor("intrinsics", WasmSymbol("tag")))
            }
            ExceptionTagType.WASM_TAG -> {
                val throwableDeclaration = tryFindBuiltInType { it.throwable }
                    ?: compilationException("kotlin.Throwable is not found in fragments", null)

                val tagFuncType = WasmRefNullType(GcHeapTypeSymbol(throwableDeclaration))

                val throwableTagFuncType = WasmFunctionType(
                    parameterTypes = listOf(tagFuncType),
                    resultTypes = emptyList()
                )
                definedDeclarations.functionTypes[Synthetics.FunctionHeapTypes.jsExceptionTagFuncType.type] = throwableTagFuncType
                WasmTag(Synthetics.FunctionHeapTypes.jsExceptionTagFuncType)
            }
        }
        return listOfNotNull(exceptionTag)
    }

    private fun getTypes(definedDeclarations: DefinedDeclarationsResolver): List<RecursiveTypeGroup> {

        val allFunctionTypes = definedDeclarations.functionTypes
        val reversedFunctionTypeMap = allFunctionTypes.reverse()
        //Rebind all function types to canonical
        for (functionType in allFunctionTypes) {
            val canonicalSignature = reversedFunctionTypeMap.getValue(functionType.value)
            if (functionType.key != canonicalSignature) {
                val canonicalType = allFunctionTypes.getValue(canonicalSignature)
                allFunctionTypes[functionType.key] = canonicalType
            }
        }

        val heapTypeResolver: (WasmHeapType.Type) -> WasmTypeDeclaration = definedDeclarations::resolve

        val recursiveGroups = with(RecursiveGroupBuilder(heapTypeResolver)) {
            addTypes(definedDeclarations.gcTypes.values)
            addTypes(definedDeclarations.vTableGcTypes.values)
            addTypes(allFunctionTypes.values.toSet())
            build()
        }

        val gcTypesReversed = IdentityHashMap<WasmTypeDeclaration, IdSignature>()
        for (gcType in definedDeclarations.gcTypes) {
            gcTypesReversed[gcType.value] = gcType.key
        }

        recursiveGroups.forEach { group ->
            val single = group.singleOrNull()
            if (single != null && single !is WasmStructDeclaration) {
                return@forEach
            }

            if (group.size > 1) {
                val needStableSort = group.any { it is WasmStructDeclaration }
                canonicalSort(group, needStableSort, heapTypeResolver)
            }

            val firstGroupGcTypeSignature = group.firstNotNullOfOrNull {
                gcTypesReversed[it]
            }

            if (firstGroupGcTypeSignature != null) {
                val mixin64BitIndex = firstGroupGcTypeSignature.toString().cityHash64().toULong()

                val mixIn = WasmStructDeclaration(
                    name = "mixin_type",
                    fields = encodeIndex(mixin64BitIndex),
                    superType = null,
                    isFinal = true
                )
                group.add(mixIn)
            }
        }

        return recursiveGroups
    }

    private fun createAndBindRttiTypeDeclaration(
        definedDeclarations: DefinedDeclarationsResolver,
    ) {
        val wasmLongArray = WasmArrayDeclaration("LongArray", WasmStructFieldDeclaration("Long", WasmI64, false))
        definedDeclarations.gcTypes[Synthetics.GcTypes.wasmLongArray.value] = wasmLongArray

        val stringLiteralFunctionRef = WasmRefNullType(
            if (isWasmJsTarget)
                Synthetics.FunctionHeapTypes.jsStringLiteralFunctionType
            else
                Synthetics.FunctionHeapTypes.stringLiteralFunctionType
        )

        val fieldsList = mutableListOf(
            WasmStructFieldDeclaration("implementedIFaceIds", WasmRefNullType(Synthetics.HeapTypes.wasmLongArray), false),
            WasmStructFieldDeclaration("superClassRtti", WasmRefNullType(Synthetics.HeapTypes.rttiType), false),            WasmStructFieldDeclaration("packageNamePoolId", WasmI32, false),
            WasmStructFieldDeclaration("simpleNamePoolId", WasmI32, false),
            WasmStructFieldDeclaration("klassId", WasmI64, false),
            WasmStructFieldDeclaration("typeInfoFlag", WasmI32, false),
            WasmStructFieldDeclaration("qualifierStringLoader", stringLiteralFunctionRef, false),
            WasmStructFieldDeclaration("simpleNameStringLoader", stringLiteralFunctionRef, false),
        )
        if (isWasmJsTarget) {
            fieldsList.add(WasmStructFieldDeclaration("packageNameGlobal", WasmRefType(WasmHeapType.Simple.Extern), false))
            fieldsList.add(WasmStructFieldDeclaration("simpleNameGlobal", WasmRefType(WasmHeapType.Simple.Extern), false))
        }
        val rttiTypeDeclaration = WasmStructDeclaration(
            name = "RTTI",
            fields = fieldsList,
            superType = null,
            isFinal = true
        )

        definedDeclarations.gcTypes[Synthetics.GcTypes.rttiType.value] = rttiTypeDeclaration
    }

    private fun getGlobals(definedDeclarations: DefinedDeclarationsResolver) = mutableListOf<WasmGlobal>().apply {
        addAll(definedDeclarations.globalFields.values)
        addAll(definedDeclarations.globalVTables.values)
        addAll(definedDeclarations.globalClassITables.values)

        val rttiGlobals = mutableMapOf<IdSignature, WasmGlobal>()
        val rttiSuperTypes = mutableMapOf<IdSignature, IdSignature?>()
        wasmCompiledFileFragments.forEach { fragment ->
            rttiGlobals.putAll(fragment.definedRttiGlobal)
            rttiSuperTypes.putAll(fragment.definedRttiSuperType)
        }

        fun wasmRttiGlobalOrderKey(superType: IdSignature?): Int =
            superType?.let { wasmRttiGlobalOrderKey(rttiSuperTypes[it]) + 1 } ?: 0

        rttiGlobals.keys.sortedBy(::wasmRttiGlobalOrderKey).mapTo(this) { rttiGlobals[it]!! }

        addAll(definedDeclarations.globalLiteralGlobals.values)
    }

    private fun createAndExportMemory(exports: MutableList<WasmExport<*>>, stdlibModuleNameForImport: String?): List<WasmMemory> {
        val memorySizeInPages = 0
        val importPair = stdlibModuleNameForImport?.let { WasmImportDescriptor(it, WasmSymbol("memory")) }
        val memory = WasmMemory(WasmLimits(memorySizeInPages.toUInt(), null/* "unlimited" */), importPair)

        // Need to export the memory in order to pass complex objects to the host language.
        // Export name "memory" is a WASI ABI convention.
        val exportMemory = WasmExport.Memory("memory", memory)
        exports.add(exportMemory)
        return listOf(memory)
    }

    private fun createAndExportMasterInitFunction(
        definedDeclarations: DefinedDeclarationsResolver,
        exports: MutableList<WasmExport<*>>,
        registerAssociatedObjectGetter: Boolean,
        initializeUnit: Boolean,
    ) {
        val masterInitFunction = WasmFunction.Defined("_initialize", Synthetics.FunctionHeapTypes.parameterlessNoReturnFunctionType)
        with(WasmExpressionBuilder(masterInitFunction.instructions)) {
            if (initializeUnit) {
                val unitGetInstance = tryFindBuiltInFunction { it.unitGetInstance }
                    ?: compilationException("kotlin.Unit_getInstance is not file in fragments", null)
                buildCall(unitGetInstance, serviceCodeLocation)
            }

            buildCall(Synthetics.Functions.fieldInitializerFunction, serviceCodeLocation)

            if (registerAssociatedObjectGetter) {
                // we do not register descriptor while no need in it
                val registerModuleDescriptor = tryFindBuiltInFunction { it.registerModuleDescriptor }
                    ?: compilationException("kotlin.registerModuleDescriptor is not file in fragments", null)

                buildInstr(WasmOp.REF_FUNC, serviceCodeLocation, Synthetics.Functions.associatedObjectGetter)
                buildInstr(WasmOp.STRUCT_NEW, serviceCodeLocation, Synthetics.GcTypes.associatedObjectGetterWrapper)
                buildCall(registerModuleDescriptor, serviceCodeLocation)
            }

            wasmCompiledFileFragments.forEach { fragment ->
                fragment.mainFunctionWrappers.forEach { signature ->
                    buildCall(signature, serviceCodeLocation)
                }
            }
            buildInstr(WasmOp.RETURN, serviceCodeLocation)
        }

        exports.add(WasmExport.Function("_initialize", masterInitFunction))
        definedDeclarations.functions[Synthetics.Functions.masterInitFunction.value] = masterInitFunction
    }

    private fun createAssociatedObjectGetter(
        definedDeclarations: DefinedDeclarationsResolver,
        wasmElements: MutableList<WasmElement>,
    ): Boolean {
        // If AO accessor removed by DCE - we do not need it then
        if (tryFindBuiltInFunction { it.tryGetAssociatedObject } == null) return false

        val kotlinAny = tryFindBuiltInType { it.kotlinAny }
            ?: compilationException("kotlin.Any is not found in fragments", null)

        val nullableAnyWasmType = WasmRefNullType(GcHeapTypeSymbol(kotlinAny))
        val associatedObjectGetterType = WasmFunctionType(listOf(WasmI64, WasmI64), listOf(nullableAnyWasmType))
        definedDeclarations.functionTypes[Synthetics.FunctionHeapTypes.associatedObjectGetterType.type] = associatedObjectGetterType

        val classIdLocal = WasmLocal(0, "classId", WasmI64, true)
        val keyIdLocal = WasmLocal(1, "keyId", WasmI64, true)
        val associatedObjectGetter = WasmFunction.Defined(
            name = "_associatedObjectGetter",
            type = Synthetics.FunctionHeapTypes.associatedObjectGetterType,
            locals = mutableListOf(classIdLocal, keyIdLocal)
        )
        definedDeclarations.functions[Synthetics.Functions.associatedObjectGetter.value] = associatedObjectGetter

        // Make this function possible to func.ref
        wasmElements.add(
            WasmElement(
                type = WasmFuncRef,
                values = listOf(WasmTable.Value.Function(WasmSymbol(associatedObjectGetter))),
                mode = WasmElement.Mode.Declarative
            )
        )

        val jsToKotlinAnyAdapter by lazy {
            tryFindBuiltInFunction { it.jsToKotlinAnyAdapter }
                ?: compilationException("kotlin.jsToKotlinAnyAdapter is not found in fragments", null)
        }

        associatedObjectGetter.instructions.clear()
        with(WasmExpressionBuilder(associatedObjectGetter.instructions)) {
            wasmCompiledFileFragments.forEach { fragment ->
                for ((klassId, associatedObjectsInstanceGetters) in fragment.classAssociatedObjectsInstanceGetters) {
                    buildGetLocal(classIdLocal, serviceCodeLocation)
                    buildConstI64(klassId, serviceCodeLocation)
                    buildInstr(WasmOp.I64_EQ, serviceCodeLocation)
                    buildIf("Class matches")
                    associatedObjectsInstanceGetters.forEach { (keyId, getter, isExternal) ->
                        if (definedDeclarations.functions.containsKey(getter)) { //Could be deleted with DCE
                            buildGetLocal(keyIdLocal, serviceCodeLocation)
                            buildConstI64(keyId, serviceCodeLocation)
                            buildInstr(WasmOp.I64_EQ, serviceCodeLocation)
                            buildIf("Object matches")
                            buildCall(getter, serviceCodeLocation)
                            if (isExternal) {
                                buildCall(jsToKotlinAnyAdapter, serviceCodeLocation)
                            }
                            buildInstr(WasmOp.RETURN, serviceCodeLocation)
                            buildEnd()
                        }
                    }
                    buildEnd()
                }
            }
            buildRefNull(WasmHeapType.Simple.None, serviceCodeLocation)
            buildInstr(WasmOp.RETURN, serviceCodeLocation)
        }

        val associatedObjectGetterTypeRef =
            WasmRefType(Synthetics.HeapTypes.associatedObjectGetterType)

        val associatedObjectGetterWrapper = WasmStructDeclaration(
            name = "AssociatedObjectGetterWrapper",
            fields = listOf(WasmStructFieldDeclaration("getter", associatedObjectGetterTypeRef, false)),
            superType = null,
            isFinal = true
        )

        definedDeclarations.gcTypes[Synthetics.GcTypes.associatedObjectGetterWrapper.value] = associatedObjectGetterWrapper

        return true
    }

    private fun createStartUnitTestsFunction(definedDeclarations: DefinedDeclarationsResolver, exports: MutableList<WasmExport<*>>) {
        val runRootSuites = tryFindBuiltInFunction { it.runRootSuites } ?: return
        if (!definedDeclarations.functions.containsKey(runRootSuites)) return

        val startUnitTestsFunction = WasmFunction.Defined("startUnitTests", Synthetics.FunctionHeapTypes.parameterlessNoReturnFunctionType)
        with(WasmExpressionBuilder(startUnitTestsFunction.instructions)) {
            wasmCompiledFileFragments.forEach { fragment ->
                fragment.testFunctionDeclarators.forEach { declarator ->
                    buildCall(declarator, serviceCodeLocation)
                }
            }
            buildCall(runRootSuites, serviceCodeLocation)
        }
        exports.add(WasmExport.Function("startUnitTests", startUnitTestsFunction))
        definedDeclarations.functions[Synthetics.Functions.startUnitTestsFunction.value] = startUnitTestsFunction
    }

    private fun createFieldInitializerFunction(definedDeclarations: DefinedDeclarationsResolver, stringPoolSize: Int) {
        val fieldInitializerFunction = WasmFunction.Defined("_fieldInitialize", Synthetics.FunctionHeapTypes.parameterlessNoReturnFunctionType)
        with(WasmExpressionBuilder(fieldInitializerFunction.instructions)) {
            buildConstI32(0, serviceCodeLocation)
            buildConstI32(stringPoolSize, serviceCodeLocation)
            buildInstr(
                WasmOp.ARRAY_NEW_DATA,
                serviceCodeLocation,
                Synthetics.GcTypes.wasmLongArrayDeclaration,
                stringAddressesAndLengthsIndex,
            )
            buildSetGlobal(Synthetics.Globals.addressesAndLengthsGlobal, serviceCodeLocation)

            wasmCompiledFileFragments.forEach { fragment ->
                fragment.objectInstanceFieldInitializers.forEach { objectInitializer ->
                    buildCall(objectInitializer, serviceCodeLocation)
                }
            }

            wasmCompiledFileFragments.forEach { fragment ->
                fragment.nonConstantFieldInitializers.forEach { nonConstantInitializer ->
                    buildCall(nonConstantInitializer, serviceCodeLocation)
                }
            }
        }
        definedDeclarations.functions[Synthetics.Functions.fieldInitializerFunction.value] = fieldInitializerFunction
    }

    private fun stringAddressesAndLengthsField(
        definedDeclarations: DefinedDeclarationsResolver,
    ) {
        val wasmLongArrayDeclaration =
            WasmArrayDeclaration("long_array", WasmStructFieldDeclaration("long", WasmI64, false))
        definedDeclarations.gcTypes[Synthetics.GcTypes.wasmLongArrayDeclaration.value] = wasmLongArrayDeclaration

        val stringAddressesAndLengthsInitializer = listOf(
            wasmInstrWithoutLocation(
                operator = WasmOp.REF_NULL,
                immediate1 = WasmImmediate.HeapType(WasmRefNullrefType)
            ),
        )

        val refAddressesAndLengthsType =
            WasmRefNullType(Synthetics.HeapTypes.wasmLongArrayDeclaration)

        val global = WasmGlobal("_addressesAndLengths", refAddressesAndLengthsType, true, stringAddressesAndLengthsInitializer)
        definedDeclarations.globalFields[Synthetics.Globals.addressesAndLengthsGlobal.value] = global
    }

    private fun createStringPoolField(
        definedDeclarations: DefinedDeclarationsResolver,
        stringPoolSize: Int
    ) {
        val stringCacheFieldInitializer = listOf(
            wasmInstrWithoutLocation(
                operator = WasmOp.I32_CONST,
                immediate1 = WasmImmediate.ConstI32(stringPoolSize),
            ),
            wasmInstrWithoutLocation(
                operator = WasmOp.ARRAY_NEW_DEFAULT,
                immediate1 = Synthetics.GcTypes.wasmStringArrayType
            ),
        )

        val refToArrayOfNullableStringsType =
            WasmRefType(Synthetics.HeapTypes.wasmStringArrayType)

        val global = WasmGlobal("_stringPool", refToArrayOfNullableStringsType, false, stringCacheFieldInitializer)
        definedDeclarations.globalFields[Synthetics.Globals.stringPoolGlobal.value] = global
    }

    private enum class StringLiteralType {
        JsString,
        Latin1,
        Utf16
    }

    private fun createStringLiteralFunction(
        definedDeclarations: DefinedDeclarationsResolver,
        stringEntities: StringLiteralWasmEntities,
        wasmElements: MutableList<WasmElement>,
        stringLiteralType: StringLiteralType,
    ) {
        val isJsString = stringLiteralType == StringLiteralType.JsString
        val isLatin1 = stringLiteralType == StringLiteralType.Latin1

        val byteArray = WasmArrayDeclaration("byte_array", WasmStructFieldDeclaration("byte", WasmI8, false))
        definedDeclarations.gcTypes[Synthetics.GcTypes.byteArray.value] = byteArray

        var localIter = 0
        val poolIdLocal = WasmLocal(localIter++, "poolId", WasmI32, true)

        val jsString: WasmLocal?
        val startAddress: WasmLocal?
        val length: WasmLocal?
        val addressAndLength: WasmLocal?

        if (isJsString) {
            jsString = WasmLocal(localIter++, "jsString", WasmRefType(WasmHeapType.Simple.Extern), true)
            startAddress = null
            length = null
            addressAndLength = null
        } else {
            jsString = null
            startAddress = WasmLocal(localIter++, "startAddress", WasmI32, false)
            length = WasmLocal(localIter++, "length", WasmI32, false)
            addressAndLength = WasmLocal(localIter++, "addressAndLength", WasmI64, false)
        }

        val temporary = WasmLocal(localIter++, "temporary", stringEntities.kotlinStringType, false)

        val stringLiteralFunctionType =
            if (isJsString)
                Synthetics.FunctionHeapTypes.jsStringLiteralFunctionType
            else
                Synthetics.FunctionHeapTypes.stringLiteralFunctionType

        val stringLiteralFunction = WasmFunction.Defined(
            name = "_stringLiteral${stringLiteralType.name}",
            type = stringLiteralFunctionType,
            locals = listOfNotNull(poolIdLocal, jsString, startAddress, length, addressAndLength, temporary).toMutableList()
        )
        with(WasmExpressionBuilder(stringLiteralFunction.instructions)) {
            buildBlock("cache_check", stringEntities.kotlinStringType) { blockResult ->
                buildGetGlobal(Synthetics.Globals.stringPoolGlobal, serviceCodeLocation)
                buildGetLocal(poolIdLocal, serviceCodeLocation)
                buildInstr(
                    WasmOp.ARRAY_GET,
                    serviceCodeLocation,
                    Synthetics.GcTypes.wasmStringArrayType
                )
                buildBrInstr(WasmOp.BR_ON_NON_NULL, blockResult, serviceCodeLocation)

                // cache miss
                if (isJsString) {
                    buildGetLocal(jsString ?: error("jsString is not set"), serviceCodeLocation)
                    val jsToKotlinStringAdapter = tryFindBuiltInFunction { it.jsToKotlinStringAdapter }
                        ?: compilationException("jsToKotlinStringAdapter is not found", null)
                    buildCall(jsToKotlinStringAdapter, serviceCodeLocation)
                } else {
                    buildGetGlobal(Synthetics.Globals.addressesAndLengthsGlobal, serviceCodeLocation)
                    buildGetLocal(poolIdLocal, serviceCodeLocation)
                    buildInstr(
                        op = WasmOp.ARRAY_GET,
                        location = serviceCodeLocation,
                        Synthetics.GcTypes.wasmLongArrayDeclaration,
                    )
                    buildSetLocal(addressAndLength ?: error("addressAndLength is not set"), serviceCodeLocation)

                    //Get length
                    buildGetLocal(addressAndLength, serviceCodeLocation)
                    buildConstI64(32L, serviceCodeLocation)
                    buildInstr(
                        op = WasmOp.I64_SHR_S,
                        location = serviceCodeLocation,
                    )
                    buildInstr(
                        op = WasmOp.I32_WRAP_I64,
                        location = serviceCodeLocation,
                    )
                    buildSetLocal(length ?: error("length is not set"), serviceCodeLocation)

                    //Get startAddress
                    buildGetLocal(addressAndLength, serviceCodeLocation)
                    buildInstr(
                        op = WasmOp.I32_WRAP_I64,
                        location = serviceCodeLocation,
                    )
                    buildSetLocal(startAddress ?: error("startAddress is not set"), serviceCodeLocation)

                    // create new string
                    buildGetLocal(startAddress, serviceCodeLocation)
                    buildGetLocal(length, serviceCodeLocation)

                    if (!isLatin1) {
                        buildInstr(
                            op = WasmOp.ARRAY_NEW_DATA,
                            location = serviceCodeLocation,
                            GcTypeSymbol(stringEntities.wasmCharArrayDeclaration), stringDataSectionIndex
                        )
                    } else {
                        val iterator = WasmLocal(localIter++, "intIterator", WasmI32, false)
                        val wasmByteArray = WasmLocal(localIter++, "byteArray", WasmRefType(Synthetics.HeapTypes.byteArray), false)
                        val wasmCharArray = WasmLocal(localIter++, "charArray", stringEntities.wasmCharArrayType, false)
                        stringLiteralFunction.locals.addAll(listOf(iterator, wasmByteArray, wasmCharArray))

                        buildInstr(
                            op = WasmOp.ARRAY_NEW_DATA,
                            location = serviceCodeLocation,
                            Synthetics.GcTypes.byteArray, stringDataSectionIndex
                        )
                        buildSetLocal(wasmByteArray, serviceCodeLocation)

                        buildGetLocal(length, serviceCodeLocation)
                        buildInstr(
                            op = WasmOp.ARRAY_NEW_DEFAULT,
                            location = serviceCodeLocation,
                            GcTypeSymbol(stringEntities.wasmCharArrayDeclaration)
                        )
                        buildSetLocal(wasmCharArray, serviceCodeLocation)

                        buildBlock("loop_body") { loopExit ->
                            buildLoop("copy_loop") { loop ->
                                buildGetLocal(iterator, serviceCodeLocation)
                                buildGetLocal(length, serviceCodeLocation)
                                buildInstr(WasmOp.I32_EQ, serviceCodeLocation)
                                buildBrIf(loopExit, serviceCodeLocation)

                                // char array set
                                buildGetLocal(wasmCharArray, serviceCodeLocation)
                                buildGetLocal(iterator, serviceCodeLocation)

                                // byte array get
                                buildGetLocal(wasmByteArray, serviceCodeLocation)
                                buildGetLocal(iterator, serviceCodeLocation)
                                buildInstr(WasmOp.ARRAY_GET_U, serviceCodeLocation, Synthetics.GcTypes.byteArray)

                                buildInstr(WasmOp.ARRAY_SET, serviceCodeLocation, GcTypeSymbol(stringEntities.wasmCharArrayDeclaration))

                                buildGetLocal(iterator, serviceCodeLocation)
                                buildConstI32(1, serviceCodeLocation)
                                buildInstr(WasmOp.I32_ADD, serviceCodeLocation)
                                buildSetLocal(iterator, serviceCodeLocation)
                                buildBr(loop, serviceCodeLocation)
                            }
                        }
                        buildGetLocal(wasmCharArray, serviceCodeLocation)
                    }

                    buildCall(stringEntities.createStringSignature, serviceCodeLocation)
                }
                buildSetLocal(temporary, serviceCodeLocation)

                //remember and return string
                buildGetGlobal(Synthetics.Globals.stringPoolGlobal, serviceCodeLocation)
                buildGetLocal(poolIdLocal, serviceCodeLocation)
                buildGetLocal(temporary, serviceCodeLocation)
                buildInstr(
                    WasmOp.ARRAY_SET,
                    serviceCodeLocation,
                    Synthetics.GcTypes.wasmStringArrayType
                )
                buildGetLocal(temporary, serviceCodeLocation)
            }
            buildInstr(WasmOp.RETURN, serviceCodeLocation)
        }

        // Make this function possible to func.ref
        wasmElements.add(
            WasmElement(
                type = WasmFuncRef,
                values = listOf(WasmTable.Value.Function(WasmSymbol(stringLiteralFunction))),
                mode = WasmElement.Mode.Declarative
            )
        )

        val functionSignature = when (stringLiteralType) {
            StringLiteralType.JsString -> Synthetics.Functions.createStringLiteralJsString.value
            StringLiteralType.Latin1 -> Synthetics.Functions.createStringLiteralLatin1.value
            StringLiteralType.Utf16 -> Synthetics.Functions.createStringLiteralUtf16.value
        }
        definedDeclarations.functions[functionSignature] = stringLiteralFunction
    }

    private fun <T> putAllChecked(from: Map<IdSignature, T>, to: MutableMap<IdSignature, T>, info: String) {
        val oldFromSize = to.size
        to.putAll(from)
        if (oldFromSize + from.size != to.size) {
            compilationException("Declaration redefinition happened on $info.", null)
        }
    }

    private fun getDefinedDeclarationsFromFragments(): DefinedDeclarationsResolver {
        val singleFragment = wasmCompiledFileFragments.singleOrNull()
        val definedDeclarations = if (singleFragment != null) {
            DefinedDeclarationsResolver(
                functions = singleFragment.definedFunctions,
                globalFields = singleFragment.definedGlobalFields,
                globalVTables = singleFragment.definedGlobalVTables,
                globalClassITables = singleFragment.definedGlobalClassITables,
                globalRTTI = singleFragment.definedRttiGlobal,
                gcTypes = singleFragment.definedGcTypes,
                vTableGcTypes = singleFragment.definedVTableGcTypes,
                functionTypes = singleFragment.definedFunctionTypes,
            )
        } else {
            DefinedDeclarationsResolver().also { definedDeclarations ->
                wasmCompiledFileFragments.forEach { fragment ->
                    putAllChecked(fragment.definedFunctions, definedDeclarations.functions, "functions")
                    putAllChecked(fragment.definedGlobalFields, definedDeclarations.globalFields, "globalFields")
                    putAllChecked(fragment.definedGlobalVTables, definedDeclarations.globalVTables, "globalVTables")
                    putAllChecked(fragment.definedGlobalClassITables, definedDeclarations.globalClassITables, "globalClassITables")
                    putAllChecked(fragment.definedRttiGlobal, definedDeclarations.globalRTTI, "globalRTTI")
                    putAllChecked(fragment.definedGcTypes, definedDeclarations.gcTypes, "gcTypes")
                    putAllChecked(fragment.definedVTableGcTypes, definedDeclarations.vTableGcTypes, "vTableGcTypes")
                    putAllChecked(fragment.definedFunctionTypes, definedDeclarations.functionTypes, "functionTypes")
                }
            }
        }

        rebindEquivalentFunctions(definedDeclarations.functions)
        bindUniqueJsFunNames()

        return definedDeclarations
    }

    private fun bindGlobalLiterals(definedDeclarations: DefinedDeclarationsResolver, stringPoolSize: Int): Int {
        var literalCounter = stringPoolSize
        val literalGlobalIdMap = mutableMapOf<String, Int>()

        wasmCompiledFileFragments.forEach { fragment ->
            var globalCounter = 0
            for (symbol in fragment.globalLiterals) {
                definedDeclarations.globalLiteralGlobals.computeIfAbsent(symbol.value) { string ->
                    WasmGlobal(
                        name = "string_${globalCounter++}",
                        type = WasmRefType(WasmHeapType.Simple.Extern),
                        isMutable = false,
                        init = emptyList(),
                        importPair = WasmImportDescriptor(importedStringConstants, WasmSymbol(string))
                    )
                }
            }
            for ((stringValue, literalIdSymbol) in fragment.globalLiteralsId) {
                var stringId = literalGlobalIdMap[stringValue]
                if (stringId == null) {
                    stringId = literalCounter
                    literalGlobalIdMap[stringValue] = stringId
                    literalCounter++
                }
                literalIdSymbol.bind(stringId)
            }
        }
        return literalCounter
    }

    private fun bindStringPoolSymbolsAndGetSize(data: MutableList<WasmData>): Int {
        val stringDataSectionBytes = mutableListOf<Byte>()
        var stringDataSectionStart = 0
        val visitedStrings = mutableMapOf<String, Int>()
        val addressesAndLengths = mutableListOf<Long>()
        wasmCompiledFileFragments.forEach { fragment ->
            for ((string, literalIdSymbol) in fragment.stringLiteralId.entries) {
                val visitedStringId = visitedStrings[string]
                val stringId: Int
                if (visitedStringId == null) {
                    stringId = visitedStrings.size
                    visitedStrings[string] = stringId

                    addressesAndLengths.add(stringDataSectionStart.toLong() or (string.length.toLong() shl 32))
                    val constData = ConstantDataCharArray(string.toCharArray(), string.fitsLatin1)
                    stringDataSectionBytes += constData.toBytes().toList()
                    stringDataSectionStart += constData.sizeInBytes
                } else {
                    stringId = visitedStringId
                }
                literalIdSymbol.bind(stringId)
            }
        }

        data.add(WasmData(WasmDataMode.Passive, stringDataSectionBytes.toByteArray()))
        val constDataAddressesAndLengths = ConstantDataIntegerArray(addressesAndLengths, LONG_SIZE_BYTES)
        data.add(WasmData(WasmDataMode.Passive, constDataAddressesAndLengths.toBytes()))

        return visitedStrings.size
    }

    private fun bindConstantArrayDataSegmentIds(data: MutableList<WasmData>) {
        wasmCompiledFileFragments.forEach { fragment ->
            fragment.constantArrayDataSegmentId.entries.forEach { (constantArraySegment, symbol) ->
                symbol.bind(data.size)
                val integerSize = when (constantArraySegment.second) {
                    WasmI8 -> BYTE_SIZE_BYTES
                    WasmI16 -> SHORT_SIZE_BYTES
                    WasmI32 -> INT_SIZE_BYTES
                    WasmI64 -> LONG_SIZE_BYTES
                    else -> TODO("type ${constantArraySegment.second} is not implemented")
                }
                val constData = ConstantDataIntegerArray(constantArraySegment.first, integerSize)
                data.add(WasmData(WasmDataMode.Passive, constData.toBytes()))
            }
        }
    }

    private fun bindUniqueJsFunNames() {
        val jsCodeCounter = mutableMapOf<String, Int>()
        wasmCompiledFileFragments.forEach { fragment ->
            fragment.jsFuns.forEach { jsCodeSnippet ->
                val jsFunName = jsCodeSnippet.value.importName.owner
                val counterValue = jsCodeCounter.getOrPut(jsFunName, defaultValue = { 0 })
                jsCodeCounter[jsFunName] = counterValue + 1
                if (counterValue > 0) {
                    jsCodeSnippet.value.importName.bind("${jsFunName}_$counterValue")
                }
            }
        }
    }

    private fun rebindEquivalentFunctions(allDefinedFunctions: MutableMap<IdSignature, WasmFunction>) {
        val equivalentFunctions = mutableMapOf<String, WasmFunction>()
        wasmCompiledFileFragments.forEach { fragment ->
            for ((signatureString, idSignature) in fragment.equivalentFunctions) {
                val func = equivalentFunctions[signatureString]
                if (func == null) {
                    // First occurrence of the adapter, register it (if not removed by DCE).
                    equivalentFunctions[signatureString] = allDefinedFunctions[idSignature] ?: continue
                } else {
                    // Adapter already exists, remove this one and use the existing adapter.
                    allDefinedFunctions[idSignature]?.let { duplicate ->
                        fragment.exports.removeAll { it.field == duplicate }
                    }
                    fragment.jsFuns.remove(idSignature)
                    fragment.jsModuleImports.remove(idSignature)

                    // Rebind adapter function to the single instance
                    // There might not be any unbound references in case it's called only from JS side
                    allDefinedFunctions[idSignature] = func
                }
            }
        }
    }
}

data class ClassAssociatedObjects(
    val klass: Long,
    val objects: List<AssociatedObject>
)

data class AssociatedObject(
    val obj: Long,
    val getterFunc: IdSignature,
    val isExternal: Boolean,
)

private fun WasmExpressionBuilder.buildCall(symbol: IdSignature, location: SourceLocation) {
    buildInstr(WasmOp.CALL, location, FuncSymbol(symbol))
}