/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.protocol.oidc.utils;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.ws.rs.core.Response;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64Url;
import org.keycloak.common.util.SecretGenerator;
import org.keycloak.events.EventBuilder;
import org.keycloak.services.CorsErrorResponseException;
import org.keycloak.services.resources.Cors;

public class PkceUtils {
    private static final Logger logger = Logger.getLogger(PkceUtils.class);
    private static final Pattern VALID_CODE_VERIFIER_PATTERN = Pattern.compile("^[0-9a-zA-Z\\-\\.~_]+$");

    public static String generateCodeVerifier() {
        return Base64Url.encode((byte[])SecretGenerator.getInstance().randomBytes(64));
    }

    public static String encodeCodeChallenge(String codeVerifier, String codeChallengeMethod) {
        try {
            switch (codeChallengeMethod) {
                case "S256": {
                    return PkceUtils.generateS256CodeChallenge(codeVerifier);
                }
            }
            return codeVerifier;
        }
        catch (Exception ex) {
            return null;
        }
    }

    public static String generateS256CodeChallenge(String codeVerifier) throws Exception {
        MessageDigest md = MessageDigest.getInstance("SHA-256");
        md.update(codeVerifier.getBytes(StandardCharsets.ISO_8859_1));
        byte[] digestBytes = md.digest();
        return Base64Url.encode((byte[])digestBytes);
    }

    public static boolean validateCodeChallenge(String verifier, String codeChallenge, String codeChallengeMethod) {
        try {
            switch (codeChallengeMethod) {
                case "plain": {
                    return verifier.equals(codeChallenge);
                }
                case "S256": {
                    return PkceUtils.generateS256CodeChallenge(verifier).equals(codeChallenge);
                }
            }
            return false;
        }
        catch (Exception ex) {
            return false;
        }
    }

    public static void checkParamsForPkceEnforcedClient(String codeVerifier, String codeChallenge, String codeChallengeMethod, String authUserId, String authUsername, EventBuilder event, Cors cors) {
        if (codeVerifier == null) {
            logger.warnf("PKCE code verifier not specified, authUserId = %s, authUsername = %s", (Object)authUserId, (Object)authUsername);
            event.error("code_verifier_missing");
            throw new CorsErrorResponseException(cors, "invalid_grant", "PKCE code verifier not specified", Response.Status.BAD_REQUEST);
        }
        PkceUtils.verifyCodeVerifier(codeVerifier, codeChallenge, codeChallengeMethod, authUserId, authUsername, event, cors);
    }

    public static void checkParamsForPkceNotEnforcedClient(String codeVerifier, String codeChallenge, String codeChallengeMethod, String authUserId, String authUsername, EventBuilder event, Cors cors) {
        if (codeChallenge != null && codeVerifier == null) {
            logger.warnf("PKCE code verifier not specified, authUserId = %s, authUsername = %s", (Object)authUserId, (Object)authUsername);
            event.error("code_verifier_missing");
            throw new CorsErrorResponseException(cors, "invalid_grant", "PKCE code verifier not specified", Response.Status.BAD_REQUEST);
        }
        if (codeChallenge != null) {
            PkceUtils.verifyCodeVerifier(codeVerifier, codeChallenge, codeChallengeMethod, authUserId, authUsername, event, cors);
        }
    }

    public static void verifyCodeVerifier(String codeVerifier, String codeChallenge, String codeChallengeMethod, String authUserId, String authUsername, EventBuilder event, Cors cors) {
        if (!PkceUtils.isValidPkceCodeVerifier(codeVerifier)) {
            logger.infof("PKCE invalid code verifier", new Object[0]);
            event.error("invalid_code_verifier");
            throw new CorsErrorResponseException(cors, "invalid_grant", "PKCE invalid code verifier", Response.Status.BAD_REQUEST);
        }
        logger.debugf("PKCE supporting Client, codeVerifier = %s", (Object)codeVerifier);
        String codeVerifierEncoded = codeVerifier;
        try {
            if (codeChallengeMethod != null && codeChallengeMethod.equals("S256")) {
                logger.debugf("PKCE codeChallengeMethod = %s", (Object)codeChallengeMethod);
                codeVerifierEncoded = PkceUtils.generateS256CodeChallenge(codeVerifier);
            } else {
                logger.debug((Object)"PKCE codeChallengeMethod is plain");
                codeVerifierEncoded = codeVerifier;
            }
        }
        catch (Exception nae) {
            logger.infof("PKCE code verification failed, not supported algorithm specified", new Object[0]);
            event.error("pkce_verification_failed");
            throw new CorsErrorResponseException(cors, "invalid_grant", "PKCE code verification failed, not supported algorithm specified", Response.Status.BAD_REQUEST);
        }
        if (!codeChallenge.equals(codeVerifierEncoded)) {
            logger.warnf("PKCE verification failed. authUserId = %s, authUsername = %s", (Object)authUserId, (Object)authUsername);
            event.error("pkce_verification_failed");
            throw new CorsErrorResponseException(cors, "invalid_grant", "PKCE verification failed", Response.Status.BAD_REQUEST);
        }
        logger.debugf("PKCE verification success. codeVerifierEncoded = %s, codeChallenge = %s", (Object)codeVerifierEncoded, (Object)codeChallenge);
    }

    private static boolean isValidPkceCodeVerifier(String codeVerifier) {
        if (codeVerifier.length() < 43) {
            logger.infof(" Error: PKCE codeVerifier length under lower limit , codeVerifier = %s", (Object)codeVerifier);
            return false;
        }
        if (codeVerifier.length() > 128) {
            logger.infof(" Error: PKCE codeVerifier length over upper limit , codeVerifier = %s", (Object)codeVerifier);
            return false;
        }
        Matcher m = VALID_CODE_VERIFIER_PATTERN.matcher(codeVerifier);
        return m.matches();
    }
}

