/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.protocol.oidc.endpoints;

import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.core.Response;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jboss.logging.Logger;
import org.keycloak.common.Profile;
import org.keycloak.events.EventBuilder;
import org.keycloak.models.ClientModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.protocol.oidc.OIDCAdvancedConfigWrapper;
import org.keycloak.protocol.oidc.TokenManager;
import org.keycloak.protocol.oidc.endpoints.request.AuthorizationEndpointRequest;
import org.keycloak.protocol.oidc.endpoints.request.AuthorizationEndpointRequestParserProcessor;
import org.keycloak.protocol.oidc.endpoints.request.RequestUriType;
import org.keycloak.protocol.oidc.utils.OIDCResponseMode;
import org.keycloak.protocol.oidc.utils.OIDCResponseType;
import org.keycloak.protocol.oidc.utils.RedirectUtils;
import org.keycloak.services.CorsErrorResponseException;
import org.keycloak.services.ErrorPageException;
import org.keycloak.services.ServicesLogger;
import org.keycloak.services.resources.Cors;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.util.TokenUtil;
import org.keycloak.utils.StringUtil;

public class AuthorizationEndpointChecker {
    private EventBuilder event;
    private AuthorizationEndpointRequest request;
    private KeycloakSession session;
    private ClientModel client;
    private RealmModel realm;
    private String redirectUri;
    private OIDCResponseType parsedResponseType;
    private OIDCResponseMode parsedResponseMode;
    private MultivaluedMap<String, String> params;
    private static final Logger logger = Logger.getLogger(AuthorizationEndpointChecker.class);
    private static final Pattern VALID_CODE_CHALLENGE_PATTERN = Pattern.compile("^[0-9a-zA-Z\\-\\.~_]+$");

    public AuthorizationEndpointChecker event(EventBuilder event) {
        this.event = event;
        return this;
    }

    public AuthorizationEndpointChecker request(AuthorizationEndpointRequest request) {
        this.request = request;
        return this;
    }

    public AuthorizationEndpointChecker session(KeycloakSession session) {
        this.session = session;
        return this;
    }

    public AuthorizationEndpointChecker client(ClientModel client) {
        this.client = client;
        return this;
    }

    public AuthorizationEndpointChecker realm(RealmModel realm) {
        this.realm = realm;
        return this;
    }

    public AuthorizationEndpointChecker params(MultivaluedMap<String, String> params) {
        this.params = params;
        return this;
    }

    public String getRedirectUri() {
        return this.redirectUri;
    }

    public OIDCResponseType getParsedResponseType() {
        return this.parsedResponseType;
    }

    public OIDCResponseMode getParsedResponseMode() {
        return this.parsedResponseMode;
    }

    public void checkRedirectUri() throws AuthorizationCheckException {
        String redirectUriParam = this.request.getRedirectUriParam();
        boolean isOIDCRequest = TokenUtil.isOIDCRequest((String)this.request.getScope());
        this.event.detail("redirect_uri", redirectUriParam);
        this.redirectUri = RedirectUtils.verifyRedirectUri(this.session, redirectUriParam, this.client, isOIDCRequest);
        if (this.redirectUri == null) {
            this.event.error("invalid_redirect_uri");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalidParameterMessage", "redirect_uri");
        }
    }

    public void checkResponseType() throws AuthorizationCheckException {
        String responseType = this.request.getResponseType();
        if (responseType == null) {
            ServicesLogger.LOGGER.missingParameter("response_type");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Missing parameter: response_type");
        }
        this.event.detail("response_type", responseType);
        try {
            this.parsedResponseType = OIDCResponseType.parse(responseType);
        }
        catch (IllegalArgumentException iae) {
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "unsupported_response_type", null);
        }
        OIDCResponseMode parsedResponseMode = null;
        try {
            parsedResponseMode = OIDCResponseMode.parse(this.request.getResponseMode(), this.parsedResponseType);
        }
        catch (IllegalArgumentException iae) {
            ServicesLogger.LOGGER.invalidParameter("response_mode");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Invalid parameter: response_mode");
        }
        this.event.detail("response_mode", parsedResponseMode.toString().toLowerCase());
        if (this.parsedResponseType.isImplicitOrHybridFlow() && parsedResponseMode == OIDCResponseMode.QUERY) {
            ServicesLogger.LOGGER.responseModeQueryNotAllowed();
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Response_mode 'query' not allowed for implicit or hybrid flow");
        }
        this.parsedResponseMode = parsedResponseMode;
        if (!(!this.parsedResponseType.isImplicitOrHybridFlow() || parsedResponseMode != OIDCResponseMode.QUERY_JWT || StringUtil.isNotBlank((String)this.client.getAttribute("authorization.encrypted.response.alg")) && StringUtil.isNotBlank((String)this.client.getAttribute("authorization.encrypted.response.enc")))) {
            ServicesLogger.LOGGER.responseModeQueryJwtNotAllowed();
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Response_mode 'query.jwt' is allowed only when the authorization response token is encrypted");
        }
        if ((this.parsedResponseType.hasResponseType("code") || this.parsedResponseType.hasResponseType("none")) && !this.client.isStandardFlowEnabled()) {
            ServicesLogger.LOGGER.flowNotAllowed("Standard");
            this.event.error("not_allowed");
            throw new AuthorizationCheckException(Response.Status.UNAUTHORIZED, "unauthorized_client", "Client is not allowed to initiate browser login with given response_type. Standard flow is disabled for the client.");
        }
        if (this.parsedResponseType.isImplicitOrHybridFlow() && !this.client.isImplicitFlowEnabled()) {
            ServicesLogger.LOGGER.flowNotAllowed("Implicit");
            this.event.error("not_allowed");
            throw new AuthorizationCheckException(Response.Status.UNAUTHORIZED, "unauthorized_client", "Client is not allowed to initiate browser login with given response_type. Implicit flow is disabled for the client.");
        }
    }

    public boolean isInvalidResponseType(AuthorizationCheckException ex) {
        return "Missing parameter: response_type".equals(ex.getErrorDescription()) || "unsupported_response_type".equals(ex.getError());
    }

    public void checkInvalidRequestMessage() throws AuthorizationCheckException {
        if (this.request.getInvalidRequestMessage() != null) {
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", this.request.getInvalidRequestMessage());
        }
    }

    public void checkOIDCRequest() {
        if (!TokenUtil.isOIDCRequest((String)this.request.getScope())) {
            ServicesLogger.LOGGER.oidcScopeMissing();
        }
    }

    public void checkValidScope() throws AuthorizationCheckException {
        boolean validScopes = Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DYNAMIC_SCOPES) ? TokenManager.isValidScope(this.request.getScope(), this.request.getAuthorizationRequestContext(), this.client) : TokenManager.isValidScope(this.request.getScope(), this.client);
        if (!validScopes) {
            ServicesLogger.LOGGER.invalidParameter("scope");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_scope", "Invalid scopes: " + this.request.getScope());
        }
    }

    public void checkOIDCParams() throws AuthorizationCheckException {
        boolean isOIDCRequest = TokenUtil.isOIDCRequest((String)this.request.getScope());
        if (!isOIDCRequest && this.parsedResponseType.toString().equals("token")) {
            return;
        }
        if (this.parsedResponseType.hasResponseType("id_token") && this.request.getNonce() == null) {
            ServicesLogger.LOGGER.missingParameter("nonce");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Missing parameter: nonce");
        }
    }

    public void checkPKCEParams() throws AuthorizationCheckException {
        String codeChallenge = this.request.getCodeChallenge();
        String codeChallengeMethod = this.request.getCodeChallengeMethod();
        if (this.parsedResponseType != null && this.parsedResponseType.isImplicitFlow()) {
            return;
        }
        String pkceCodeChallengeMethod = OIDCAdvancedConfigWrapper.fromClientModel(this.client).getPkceCodeChallengeMethod();
        if (pkceCodeChallengeMethod != null && !pkceCodeChallengeMethod.isEmpty()) {
            this.checkParamsForPkceEnforcedClient(codeChallengeMethod, pkceCodeChallengeMethod, codeChallenge);
        } else {
            this.checkParamsForPkceNotEnforcedClient(codeChallengeMethod, pkceCodeChallengeMethod, codeChallenge);
        }
    }

    public void checkParRequired() throws AuthorizationCheckException {
        boolean isParRequired = this.realm.getParPolicy().isRequirePushedAuthorizationRequests(this.client);
        if (!isParRequired) {
            return;
        }
        String requestUriParam = (String)this.params.getFirst((Object)"request_uri");
        if (requestUriParam != null && AuthorizationEndpointRequestParserProcessor.getRequestUriType(requestUriParam) == RequestUriType.PAR) {
            return;
        }
        ServicesLogger.LOGGER.missingParameter("request_uri");
        this.event.error("invalid_request");
        throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Pushed Authorization Request is only allowed.");
    }

    private boolean isValidPkceCodeChallenge(String codeChallenge) {
        if (codeChallenge.length() < 43) {
            logger.debugf("PKCE codeChallenge length under lower limit , codeChallenge = %s", (Object)codeChallenge);
            return false;
        }
        if (codeChallenge.length() > 128) {
            logger.debugf("PKCE codeChallenge length over upper limit , codeChallenge = %s", (Object)codeChallenge);
            return false;
        }
        Matcher m = VALID_CODE_CHALLENGE_PATTERN.matcher(codeChallenge);
        return m.matches();
    }

    private void checkParamsForPkceEnforcedClient(String codeChallengeMethod, String pkceCodeChallengeMethod, String codeChallenge) throws AuthorizationCheckException {
        if (codeChallengeMethod == null) {
            logger.info((Object)"PKCE enforced Client without code challenge method.");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Missing parameter: code_challenge_method");
        }
        if (!codeChallengeMethod.equals(pkceCodeChallengeMethod)) {
            logger.info((Object)"PKCE enforced Client code challenge method is not configured one.");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Invalid parameter: code challenge method is not configured one");
        }
        if (codeChallenge == null) {
            logger.info((Object)"PKCE supporting Client without code challenge");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Missing parameter: code_challenge");
        }
        if (!this.isValidPkceCodeChallenge(codeChallenge)) {
            logger.infof("PKCE supporting Client with invalid code challenge specified in PKCE, codeChallenge = %s", (Object)codeChallenge);
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Invalid parameter: code_challenge");
        }
    }

    private void checkParamsForPkceNotEnforcedClient(String codeChallengeMethod, String pkceCodeChallengeMethod, String codeChallenge) throws AuthorizationCheckException {
        if (codeChallenge == null && codeChallengeMethod != null) {
            logger.info((Object)"PKCE supporting Client without code challenge");
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Missing parameter: code_challenge");
        }
        if (codeChallenge == null) {
            logger.debug((Object)"PKCE non-supporting Client");
            return;
        }
        if (codeChallengeMethod != null) {
            if (!codeChallengeMethod.equals("S256") && !codeChallengeMethod.equals("plain")) {
                logger.infof("PKCE supporting Client with invalid code challenge method not specified in PKCE, codeChallengeMethod = %s", (Object)codeChallengeMethod);
                this.event.error("invalid_request");
                throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Invalid parameter: code_challenge_method");
            }
        } else {
            codeChallengeMethod = "plain";
        }
        if (!this.isValidPkceCodeChallenge(codeChallenge)) {
            logger.infof("PKCE supporting Client with invalid code challenge specified in PKCE, codeChallenge = %s", (Object)codeChallenge);
            this.event.error("invalid_request");
            throw new AuthorizationCheckException(Response.Status.BAD_REQUEST, "invalid_request", "Invalid parameter: code_challenge");
        }
    }

    public class AuthorizationCheckException
    extends Exception {
        private final Response.Status status;
        private final String error;
        private final String errorDescription;

        public AuthorizationCheckException(Response.Status status, String error, String errorDescription) {
            this.status = status;
            this.error = error;
            this.errorDescription = errorDescription;
        }

        public void throwAsErrorPageException(AuthenticationSessionModel authenticationSession) {
            throw new ErrorPageException(AuthorizationEndpointChecker.this.session, authenticationSession, this.status, this.error, this.errorDescription);
        }

        public void throwAsCorsErrorResponseException(Cors cors) {
            AuthorizationEndpointChecker.this.event.detail("detail", this.errorDescription).error(this.error);
            throw new CorsErrorResponseException(cors, this.error, this.errorDescription, this.status);
        }

        public String getError() {
            return this.error;
        }

        public String getErrorDescription() {
            return this.errorDescription;
        }
    }
}

