/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.saml.metadata.resolver.impl;

import java.security.NoSuchAlgorithmException;

import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.codec.StringDigester;
import net.shibboleth.utilities.java.support.codec.StringDigester.OutputFormat;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

import com.google.common.base.Function;

//TODO reference to protocol spec pending in Javadoc.

/**
 * Function which digests an entity ID using SHA-1 and returns it in the format specified by
 * the Metadata Query Protocol specification.
 */
public class MetadataQueryProtocolSHA1Transformer implements Function<String, String> {
    
    /** String digester instance. */
    private StringDigester digester;
    
    /** Constructor. */
    public MetadataQueryProtocolSHA1Transformer() {
        try {
            digester = new StringDigester("SHA-1", OutputFormat.HEX_LOWER);
        } catch (final NoSuchAlgorithmException e) {
            // this can't really happen b/c SHA-1 is required to be supported on all JREs.
        }
    }

    /** {@inheritDoc} */
    @Override @Nullable public String apply(@Nullable final String input) {
        final String entityID = Constraint.isNotNull(StringSupport.trimOrNull(input), "Entity ID was null or empty");
        final String digested = digester.apply(entityID);
        if (digested == null) {
            return null;
        }
        return "{sha1}" + digested;
    }

}
