/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.api;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Collection;
import java.util.ArrayList;
import java.beans.Introspector;
import java.lang.annotation.Annotation;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class ClassIntrospector {

  /** . */
  private final ClassTypeInfo type;

  public ClassIntrospector(ClassTypeInfo type) {
    this.type = type;
  }

  public <A extends Annotation> Collection<MethodInfo> resolveMethods(Class<A> annotationClass) {
    ArrayList<MethodInfo> methods = new ArrayList<MethodInfo>();
    AnnotationIntrospector<A> introspector = new AnnotationIntrospector<A>(annotationClass);
    for (MethodInfo method : getMethods()) {
      if (introspector.resolve(method) != null) {
        methods.add(method);
      }
    }
    return methods;
  }

  public Map<String, MethodInfo> getGetterMap() {
    Map<String, MethodInfo> getterMap = new HashMap<String, MethodInfo>();
    for (MethodInfo getter : getGetters()) {
      String getterName = getter.getName();
      String name;
      if (getterName.startsWith("get")) {
        name = Introspector.decapitalize(getterName.substring(3));
      } else {
        name = Introspector.decapitalize(getterName.substring(2));
      }
      getterMap.put(name, getter);
    }
    return getterMap;
  }

  public Iterable<MethodInfo> getGetters() {
    final MethodContainer getters = new MethodContainer();
    HierarchyVisitor visitor = new HierarchyVisitor() {
      public boolean enter(ClassTypeInfo type) {
        return true;
      }
      public void leave(ClassTypeInfo type) {
        if (!type.getName().equals(Object.class.getName())) {
          for (MethodInfo method : type.getDeclaredMethods()) {
            String methodName = method.getName();
            if (
              methodName.startsWith("get") &&
              methodName.length() > 3 &&
              Character.isUpperCase(methodName.charAt(3)) &&
              method.getParameterTypes().size() == 0) {
              getters.add(method);
            }
          }
        }
      }
    };

    //
    type.accept(visitor);

    //
    return getters;
  }

  public Map<String, Set<MethodInfo>> getSetterMap() {
    Map<String, Set<MethodInfo>> setterMap = new HashMap<String, Set<MethodInfo>>();
    for (MethodInfo setter : getSetters()) {
      String name = Introspector.decapitalize(setter.getName().substring(3));
      Set<MethodInfo> setters = setterMap.get(name);
      if (setters == null) {
        setters = new HashSet<MethodInfo>();
        setterMap.put(name, setters);
      }
      setters.add(setter);
    }
    return setterMap;
  }

  public Iterable<MethodInfo> getSetters() {
    final MethodContainer setters = new MethodContainer();
    HierarchyVisitor visitor = new HierarchyVisitor() {
      public boolean enter(ClassTypeInfo type) {
        return true;
      }
      public void leave(ClassTypeInfo type) {
        if (!type.getName().equals(Object.class.getName())) {
          for (MethodInfo method : type.getDeclaredMethods()) {
            String methodName = method.getName();
            if (
              methodName.startsWith("set") &&
              methodName.length() > 3 &&
              Character.isUpperCase(methodName.charAt(3)) &&
              method.getParameterTypes().size() == 1) {
              setters.add(method);
            }
          }
        }
      }
    };

    //
    type.accept(visitor);

    //
    return setters;
  }

  public Iterable<MethodInfo> getMethods() {
    return getMethods(type);
  }

  private static MethodContainer getMethods(TypeInfo type) {
    if (type instanceof ClassTypeInfo) {
      return getMethods((ClassTypeInfo)type);
    } else if (type instanceof ParameterizedTypeInfo) {
      return getMethods(((ParameterizedTypeInfo)type).getRawType());
    } else {
      throw new UnsupportedOperationException("Cannot get methods from type " + type);
    }
  }

  private static MethodContainer getMethods(ClassTypeInfo clazz) {
    MethodContainer  methods;
    TypeInfo superType = clazz.getSuperType();
    if (superType == null || (superType instanceof ClassTypeInfo && ((ClassTypeInfo)superType).getName().equals(Object.class.getName()))) {
      methods = new MethodContainer();
    } else {
      methods = getMethods(superType);
    }

    //
    for (TypeInfo interfaceType : clazz.getInterfaces()) {
      methods.addAll(getMethods(interfaceType));
    }

    //
    for (MethodInfo declaredMethod : clazz.getDeclaredMethods()) {
      methods.add(declaredMethod);
    }

    //
    return methods;
  }

  private static class MethodContainer implements Iterable<MethodInfo>  {

    /** . */
    private final Map<MethodSignature, MethodInfo> map = new HashMap<MethodSignature, MethodInfo>();

    public void addAll(Iterable<MethodInfo> methods) {
      for (MethodInfo method : methods) {
        add(method);
      }
    }

    public boolean add(MethodInfo method) {
      MethodSignature key = method.getSignature();
      MethodInfo existing = map.get(key);
      if (existing != null) {
        if (method.getReturnType().isCovariantWith(existing.getReturnType())) {
          if (existing.getOwner().isAssignableFrom(method.getOwner())) {
            map.put(key, method);
            return true;
          }
        }
      } else {
        map.put(key, method);
        return true;
      }
      return false;
    }

    public Iterator<MethodInfo> iterator() {
      return map.values().iterator();
    }
  }
}
