/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.api;

import java.util.List;
import java.util.ArrayList;

/**
 * The signature of method, identifies a method.
 *
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class MethodSignature {

  /** . */
  private final String name;

  /** . */
  private final List<String> parameterRawNames;

  public MethodSignature(String name, List<TypeInfo> parameterTypes) {
    List<String> parameterRawNames = new ArrayList<String>();
    for (TypeInfo parameterType : parameterTypes) {
      String rawName = computeRawName(parameterType);
      parameterRawNames.add(rawName);
    }

    //
    this.name = name;
    this.parameterRawNames = parameterRawNames;
  }

  public String getName() {
    return name;
  }

  private String computeRawName(TypeInfo type) {
    if (type instanceof ClassTypeInfo) {
      return ((ClassTypeInfo)type).getName();
    } else if (type instanceof ParameterizedTypeInfo) {
      return computeRawName(((ParameterizedTypeInfo)type).getRawType());
    } else if (type instanceof TypeVariableInfo) {
      return computeRawName(((TypeVariableInfo)type).getBounds().get(0));
    } else if (type instanceof ArrayTypeInfo) {
      return "[L" + computeRawName(((ArrayTypeInfo)type).getComponentType()) + ";";
    } else {
      throw new AssertionError();
    }
  }

  @Override
  public int hashCode() {
    return name.hashCode() ^ parameterRawNames.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj instanceof MethodSignature) {
      MethodSignature that = (MethodSignature)obj;
      return name.equals(that.name) && parameterRawNames.equals(that.parameterRawNames);
    }
    return false;
  }

  @Override
  public String toString() {
    return "MethodSignature[name=" + name + ",parameterRawNames=" + parameterRawNames + "]";
  }
}
