/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.apt;

import org.reflext.spi.model.TypeModel;
import org.reflext.spi.model.TypeKind;
import org.reflext.spi.model.GenericDeclarationKind;
import org.reflext.api.LiteralType;
import org.reflext.api.ClassKind;

import javax.lang.model.type.TypeMirror;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeVariable;
import javax.lang.model.type.NoType;
import javax.lang.model.type.WildcardType;
import javax.lang.model.type.ArrayType;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.TypeParameterElement;
import javax.lang.model.element.Element;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.HashMap;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class JavaxLangTypeModel implements TypeModel<Object> {

  /** . */
  static final Map<String, LiteralType> primitiveWrapperMap;

  static {
    Map<String, LiteralType> map = new HashMap<String, LiteralType>();
    map.put(Boolean.class.getName(), LiteralType.BOOLEAN);
    map.put(Byte.class.getName(), LiteralType.BYTE);
    map.put(Short.class.getName(), LiteralType.SHORT);
    map.put(Integer.class.getName(), LiteralType.INT);
    map.put(Long.class.getName(), LiteralType.LONG);
    map.put(Float.class.getName(), LiteralType.FLOAT);
    map.put(Double.class.getName(), LiteralType.DOUBLE);
    primitiveWrapperMap = map;
  }

  public TypeKind getKind(Object type) {

    if (type == null) {
      throw new NullPointerException();
    }

    if (type instanceof TypeElement) {
      TypeElement elt = (TypeElement)type;
      String name = elt.getQualifiedName().toString();

      //
      if (primitiveWrapperMap.containsKey(name)) {
        return TypeKind.SIMPLE;
      } else {
        return TypeKind.CLASS;
      }
    } else {
      TypeMirror mirror = (TypeMirror)type;
      switch (mirror.getKind()) {
        case VOID:
          return TypeKind.VOID;
        case BOOLEAN:
        case BYTE:
        case SHORT:
        case INT:
        case LONG:
        case FLOAT:
        case DOUBLE:
          return TypeKind.SIMPLE;
        case DECLARED:
          DeclaredType declaredType = (DeclaredType)mirror;
          if (declaredType.getTypeArguments().isEmpty()) {
            String name = ((TypeElement)declaredType.asElement()).getQualifiedName().toString();
            if (primitiveWrapperMap.containsKey(name)) {
              return TypeKind.SIMPLE;
            } else {
              return TypeKind.CLASS;
            }
          } else {
            return TypeKind.PARAMETERIZED;
          }
        case TYPEVAR:
          return TypeKind.VARIABLE;
        case WILDCARD:
          return TypeKind.WILDCARD;
        case ARRAY:
          return TypeKind.ARRAY;
        default:
          throw new AssertionError("Cannot handle type " + type + " of kind " + mirror.getKind());
      }
    }

  }

  public String getClassName(Object classType) {
    if (classType instanceof TypeElement) {
      TypeElement typeElt = (TypeElement)classType;
      switch (typeElt.getNestingKind()) {
        case TOP_LEVEL:
          return typeElt.getQualifiedName().toString();
        case MEMBER:
          return ((TypeElement)typeElt.getEnclosingElement()).getQualifiedName() + "$" + typeElt.getSimpleName();
        case LOCAL:
        case ANONYMOUS:
          throw new UnsupportedOperationException("todo nesting kind");
      }
      return typeElt.getQualifiedName().toString();
    } else {
      TypeMirror typeMirror = (TypeMirror)classType;
      switch (typeMirror.getKind()) {
        case BOOLEAN:
          return boolean.class.getName();
        case BYTE:
          return byte.class.getName();
        case SHORT:
          return short.class.getName();
        case INT:
          return int.class.getName();
        case LONG:
          return long.class.getName();
        case FLOAT:
          return float.class.getName();
        case DOUBLE:
          return double.class.getName();
        case VOID:
          return Void.class.getName();
        case DECLARED:
          DeclaredType declaredType = (DeclaredType)classType;
          return getClassName(declaredType.asElement());
        default:
          throw new AssertionError("Cannot handle type " + typeMirror + " of kind  " + typeMirror.getKind() + " as class type");
      }
    }
  }

  public Iterable<Object> getInterfaces(Object classType) {
    if (classType instanceof TypeElement) {
      TypeElement typeElt = (TypeElement)classType;
      List<Object> interfaces = new ArrayList<Object>();
      for (TypeMirror itf : typeElt.getInterfaces()) {
        interfaces.add(itf);
      }
      return interfaces;
    } else {
      DeclaredType declaredType = (DeclaredType)classType;
      return getInterfaces(declaredType.asElement());
    }
  }

  public ClassKind getClassKind(Object classType) {
    if (classType instanceof TypeElement) {
      TypeElement typeElt = (TypeElement)classType;
      switch (typeElt.getKind()) {
        case ENUM:
          return ClassKind.ENUM;
        case ANNOTATION_TYPE:
          return ClassKind.ANNOTATION;
        case INTERFACE:
          return ClassKind.INTERFACE;
        case CLASS:
          return ClassKind.CLASS;
        default:
          throw new AssertionError("Cannot handle type " + typeElt + " of kind  " + typeElt.getKind() + " as class type");
      }
    } else {
      TypeMirror mirror = (TypeMirror)classType;
      switch (mirror.getKind()) {
        case BOOLEAN:
        case BYTE:
        case SHORT:
        case INT:
        case LONG:
        case FLOAT:
        case DOUBLE:
          return ClassKind.CLASS;
        case VOID:
          return ClassKind.CLASS;
        case DECLARED:
          DeclaredType declared = (DeclaredType)mirror;
          return getClassKind(declared.asElement());
        default:
          throw new AssertionError();
      }
    }
  }

  public Object getSuperClass(Object classType) {
    if (classType instanceof TypeElement) {
      TypeElement typeElt = (TypeElement)classType;
      TypeMirror superType = typeElt.getSuperclass();
      if (superType instanceof NoType) {
        return null;
      } else {
        return superType;
      }
    } else {
      DeclaredType declaredType = (DeclaredType)classType;
      return getSuperClass(declaredType.asElement());
    }
  }

  public Iterable<Object> getTypeParameters(Object classType) {
    if (classType instanceof TypeElement) {
      TypeElement typeElt = (TypeElement)classType;
      List<Object> typeParameters = new ArrayList<Object>();
      for (TypeParameterElement typeParameterElt : typeElt.getTypeParameters()) {
        typeParameters.add(typeParameterElt.asType());
      }
      return typeParameters;
    } else {
      DeclaredType declaredType = (DeclaredType)classType;
      return getTypeParameters(declaredType.asElement());
    }
  }

  public String getName(Object typeVariable) {
    TypeParameterElement tpe = (TypeParameterElement)((TypeVariable)typeVariable).asElement();
    return tpe.getSimpleName().toString();
  }

  public GenericDeclarationKind getGenericDeclarationKind(Object typeVariable) {
    TypeParameterElement tpe = (TypeParameterElement)((TypeVariable)typeVariable).asElement();
    Element genericElt =tpe.getGenericElement();
    switch (genericElt.getKind()) {
      case INTERFACE:
      case CLASS:
        return GenericDeclarationKind.TYPE;
      case METHOD:
        return GenericDeclarationKind.METHOD;
      default:
        throw new AssertionError();
    }
  }

  public LiteralType getLiteralType(Object simpleType) {
    if (simpleType instanceof TypeElement) {
      TypeElement elt = (TypeElement)simpleType;
      String name = elt.getQualifiedName().toString();
      LiteralType literalType = primitiveWrapperMap.get(name);
      if (literalType == null) {
        throw new AssertionError("Cannot handle type " + simpleType);
      } else {
        return literalType;
      }
    } else {
      TypeMirror mirror = (TypeMirror)simpleType;
      switch (mirror.getKind()) {
        case BOOLEAN:
          return LiteralType.BOOLEAN;
        case BYTE:
          return LiteralType.BYTE;
        case SHORT:
          return LiteralType.SHORT;
        case INT:
          return LiteralType.INT;
        case LONG:
          return LiteralType.LONG;
        case FLOAT:
          return LiteralType.FLOAT;
        case DOUBLE:
          return LiteralType.DOUBLE;
        case DECLARED:
          return getLiteralType(((DeclaredType)mirror).asElement());
        default:
          throw new AssertionError("Cannot handle type " + simpleType + " of kind " + mirror.getKind());
      }
    }
  }

  public Object getComponentType(Object arrayType) {
    return ((ArrayType)arrayType).getComponentType();
  }

  public boolean isPrimitive(Object simpleType) {
    if (simpleType instanceof TypeElement) {
      TypeElement elt = (TypeElement)simpleType;
      String name = elt.getQualifiedName().toString();
      if (primitiveWrapperMap.containsKey(name)) {
        return false;
      } else {
        throw new AssertionError("Cannot handle type " + simpleType);
      }
    } else {
      TypeMirror mirror = (TypeMirror)simpleType;
      switch (mirror.getKind()) {
        case BOOLEAN:
        case BYTE:
        case SHORT:
        case INT:
        case LONG:
        case FLOAT:
        case DOUBLE:
          return true;
        case DECLARED:
          return isPrimitive(((DeclaredType)mirror).asElement());
        default:
          throw new AssertionError("Cannot handle type " + simpleType + " of kind " + mirror.getKind());
      }
    }
  }

  public Object getGenericDeclaration(Object typeVariable) {
    return ((TypeParameterElement)((TypeVariable)typeVariable).asElement()).getGenericElement();
  }

  public Object getRawType(Object parameterizedType) {
    return ((DeclaredType)parameterizedType).asElement();
  }

  public Iterable<Object> getTypeArguments(Object parameterizedType) {
    List<Object> typeArguments = new ArrayList<Object>();
    for (TypeMirror typeArgument : ((DeclaredType)parameterizedType).getTypeArguments()) {
      typeArguments.add(typeArgument);
    }
    return typeArguments;
  }

  private List<Object> resolveBounds(TypeMirror bound) {
    if (bound == null) {
      return Collections.emptyList();
    } else {
      ArrayList<Object> bounds = new ArrayList<Object>();
      if (bound instanceof DeclaredType) {
        DeclaredType bilto = (DeclaredType)bound;
        TypeElement boundElt = (TypeElement)bilto.asElement();
        switch (boundElt.getNestingKind()) {
          case ANONYMOUS:

            // We need to rewrite it
            if (!(boundElt.getSuperclass() instanceof NoType)) {
              bounds.add(boundElt.getSuperclass());
            }
            for (TypeMirror itf : boundElt.getInterfaces()) {
              bounds.add(itf);
            }
            break;
          default:
            bounds.add(bound);
            break;
        }
      } else {
        bounds.add(bound);
      }
      return bounds;
    }
  }

  public Iterable<Object> getBounds(Object typeVariable) {
    TypeMirror upperBound = ((TypeVariable)typeVariable).getUpperBound();
    return resolveBounds(upperBound);
  }

  public Iterable<Object> getUpperBounds(Object wildcardType) {
    TypeMirror extendsBound = ((WildcardType)wildcardType).getExtendsBound();
    return resolveBounds(extendsBound);
  }

  public Iterable<Object> getLowerBounds(Object wildcardType) {
    TypeMirror superBounds = ((WildcardType)wildcardType).getSuperBound();
    return resolveBounds(superBounds);
  }
}
