/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.core;

import org.reflext.api.TypeResolver;
import org.reflext.spi.model.*;
import org.reflext.api.TypeInfo;

import java.util.HashMap;
import java.util.Map;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class TypeResolverImpl<T, M, A, P> implements TypeResolver<T> {

  public static <T, M, A, P> TypeResolver<T> create(ReflectionModel<T, M, A, P> model) {
    return new TypeResolverImpl<T,M,A,P>(model);
  }

  /** . */
  final TypeModel<T> typeModel;

  /** . */
  final MethodModel<T, M> methodModel;

  /** . */
  final AnnotationModel<T, T, A, P> typeAnnotationModel;

  /** . */
  final AnnotationModel<T, M, A, P> methodAnnotationModel;

  /** . */
  final Map<T, TypeInfo> cache;

  private TypeResolverImpl(ReflectionModel<T, M, A, P> model) {
    this.typeModel = model.getTypeModel();
    this.methodModel = model.getMethodModel();
    this.typeAnnotationModel = model.getTypeAnnotationModel();
    this.methodAnnotationModel = model.getMethodAnnotationModel();
    this.cache = new HashMap<T, TypeInfo>();
  }

  public TypeInfo resolve(T type) throws NullPointerException {
    if (type == null) {
      throw new NullPointerException("No null type accepted");
    }
    TypeInfo cached = cache.get(type);
    if (cached == null) {
      cached = _getType(type);
      if (cached != null) {
        cache.put(type, cached);
      }
    }
    return _getType(type);
  }

  AbstractTypeInfo<T, M, A, P> _getType(T type) {
    TypeKind kind = typeModel.getKind(type);
    switch (kind) {
      case SIMPLE:
        return new SimpleTypeInfoImpl<T, M, A, P>(this, type);
      case VOID:
        return new VoidTypeInfoImpl<T, M, A, P>(this, type);
      case CLASS:
        return new ClassTypeInfoImpl<T, M, A, P>(this, type);
      case PARAMETERIZED:
        return new ParameterizedTypeInfoImpl<T, M, A, P>(this, type);
      case VARIABLE:
        return new TypeVariableInfoImpl<T, M, A, P>(this, type);
      case WILDCARD:
        return new WildcardTypeInfoImpl<T, M, A, P>(this, type);
      case ARRAY:
        return new ArrayTypeInfoImpl<T, M, A, P>(this, type);
      default:
        throw new AssertionError("Cannot handle type " + type + " with kind " + kind);
    }
  }
}
