/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.jlr;

import org.reflext.spi.model.MethodModel;
import org.reflext.api.AccessScope;

import java.lang.reflect.Type;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.TypeVariable;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.ArrayList;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class JavaLangReflectMethodModel implements MethodModel<Type, Method> {

  public Iterable<Method> getDeclaredMethods(Type classType) {
    Class<?> clazz = (Class<?>)classType;
    MethodContainer methods = new MethodContainer();
    for (Method declaredMethod : clazz.getDeclaredMethods()) {
      int modifiers = declaredMethod.getModifiers();
      if (!Modifier.isStatic(modifiers) && !Modifier.isPrivate(modifiers)) {
        if (!declaredMethod.isBridge()) {
          methods.add(declaredMethod);
        }
      }
    }
    return methods;
  }

  public Iterable<Type> getParameterTypes(Method method) {
    return Arrays.asList(method.getGenericParameterTypes());
  }

  public String getName(Method method) {
    return method.getName();
  }

  public Type getReturnType(Method method) {
    return method.getGenericReturnType();
  }
  
  public AccessScope getAccess(Method method) {
    int modifiers = method.getModifiers();
    if (Modifier.isPrivate(modifiers)) {
      return AccessScope.PRIVATE;
    } else if (Modifier.isPublic(modifiers)) {
      return AccessScope.PUBLIC;
    } else if (Modifier.isProtected(modifiers)) {
      return AccessScope.PROTECTED;
    } else {
      return AccessScope.PACKAGE_PROTECTED;
    }
  }

  public boolean isAbstract(Method method) {
    return Modifier.isAbstract(method.getModifiers());
  }

  public boolean isStatic(Method method) {
    return Modifier.isStatic(method.getModifiers());
  }

  public boolean isNative(Method method) {
    return Modifier.isNative(method.getModifiers());
  }

  public boolean isFinal(Method method) {
    return Modifier.isFinal(method.getModifiers());
  }

  public Iterable<Type> getTypeParameters(Method method) {
    ArrayList<Type> typeParameters = new ArrayList<Type>();
    for (TypeVariable typeParameter : method.getTypeParameters()) {
      typeParameters.add(typeParameter);
    }
    return typeParameters;
  }

  public Iterable<String> getParameterNames(Method method) {
    return null;
  }

  public Type getOwner(Method method) {
    return method.getDeclaringClass();
  }

  public Method getGenericDeclaration(Type typeVariable) {
    return (Method)((TypeVariable)typeVariable).getGenericDeclaration();
  }

  public <A extends Annotation> A getDeclaredAnnotation(Method method, Class<A> annotationClass) {
    for (Annotation annotation : method.getAnnotations()) {
      if (annotationClass.isInstance(annotation)) {
        return annotationClass.cast(annotation);
      }
    }
    return null;
  }
}
