/*
 * Decompiled with CFR 0.152.
 */
package org.openqa.selenium.grid.sessionmap.redis;

import com.google.common.collect.ImmutableMap;
import io.lettuce.core.KeyValue;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.ImmutableCapabilities;
import org.openqa.selenium.NoSuchSessionException;
import org.openqa.selenium.events.EventBus;
import org.openqa.selenium.grid.config.Config;
import org.openqa.selenium.grid.data.NodeRemovedEvent;
import org.openqa.selenium.grid.data.NodeRestartedEvent;
import org.openqa.selenium.grid.data.Session;
import org.openqa.selenium.grid.data.SessionClosedEvent;
import org.openqa.selenium.grid.log.LoggingOptions;
import org.openqa.selenium.grid.server.EventBusOptions;
import org.openqa.selenium.grid.sessionmap.SessionMap;
import org.openqa.selenium.grid.sessionmap.config.SessionMapOptions;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.json.Json;
import org.openqa.selenium.redis.GridRedisClient;
import org.openqa.selenium.remote.RemoteTags;
import org.openqa.selenium.remote.SessionId;
import org.openqa.selenium.remote.tracing.AttributeKey;
import org.openqa.selenium.remote.tracing.AttributeMap;
import org.openqa.selenium.remote.tracing.Span;
import org.openqa.selenium.remote.tracing.Status;
import org.openqa.selenium.remote.tracing.Tags;
import org.openqa.selenium.remote.tracing.Tracer;

public class RedisBackedSessionMap
extends SessionMap {
    private static final Logger LOG = Logger.getLogger(RedisBackedSessionMap.class.getName());
    private static final Json JSON = new Json();
    private static final String REDIS_URI_KEY = "session.uri_key";
    private static final String REDIS_URI_VALUE = "session.uri_value";
    private static final String REDIS_CAPABILITIES_KEY = "session.capabilities_key";
    private static final String REDIS_CAPABILITIES_VALUE = "session.capabilities_value";
    private static final String REDIS_START_KEY = "session.start";
    private static final String REDIS_START_VALUE = "session.start_value";
    private static final String DATABASE_SYSTEM = AttributeKey.DATABASE_SYSTEM.getKey();
    private static final String DATABASE_OPERATION = AttributeKey.DATABASE_OPERATION.getKey();
    private final GridRedisClient connection;
    private final EventBus bus;
    private final URI serverUri;

    public RedisBackedSessionMap(Tracer tracer, URI serverUri, EventBus bus) {
        super(tracer);
        Require.nonNull((String)"Redis Server Uri", (Object)serverUri);
        this.bus = (EventBus)Require.nonNull((String)"Event bus", (Object)bus);
        this.connection = new GridRedisClient(serverUri);
        this.serverUri = serverUri;
        this.bus.addListener(SessionClosedEvent.listener(this::remove));
        this.bus.addListener(NodeRemovedEvent.listener(nodeStatus -> nodeStatus.getSlots().stream().filter(slot -> slot.getSession() != null).map(slot -> slot.getSession().getId()).forEach(this::remove)));
        bus.addListener(NodeRestartedEvent.listener(nodeStatus -> this.removeByUri(nodeStatus.getExternalUri())));
    }

    public static SessionMap create(Config config) {
        Tracer tracer = new LoggingOptions(config).getTracer();
        EventBus bus = new EventBusOptions(config).getEventBus();
        URI sessionMapUri = new SessionMapOptions(config).getSessionMapUri();
        return new RedisBackedSessionMap(tracer, sessionMapUri, bus);
    }

    public boolean add(Session session) {
        Require.nonNull((String)"Session to add", (Object)session);
        try (Span span = this.tracer.getCurrentContext().createSpan("MSET sessionUriKey <sessionUri> capabilitiesKey <capabilities> ");){
            AttributeMap attributeMap = this.tracer.createAttributeMap();
            RemoteTags.SESSION_ID.accept(span, session.getId());
            RemoteTags.SESSION_ID_EVENT.accept(attributeMap, session.getId());
            RemoteTags.CAPABILITIES.accept(span, session.getCapabilities());
            RemoteTags.CAPABILITIES_EVENT.accept(attributeMap, session.getCapabilities());
            this.setCommonSpanAttributes(span);
            this.setCommonEventAttributes(attributeMap);
            String uriKey = this.uriKey(session.getId());
            String uriValue = session.getUri().toString();
            String stereotypeKey = this.stereotypeKey(session.getId());
            String stereotypeJson = JSON.toJson((Object)session.getStereotype());
            String capabilitiesKey = this.capabilitiesKey(session.getId());
            String capabilitiesJson = JSON.toJson((Object)session.getCapabilities());
            String startKey = this.startKey(session.getId());
            String startValue = JSON.toJson((Object)session.getStartTime());
            span.setAttribute(REDIS_URI_KEY, uriKey);
            span.setAttribute(REDIS_URI_VALUE, uriValue);
            span.setAttribute(REDIS_CAPABILITIES_KEY, capabilitiesKey);
            span.setAttribute(REDIS_CAPABILITIES_VALUE, capabilitiesJson);
            span.setAttribute(DATABASE_OPERATION, "MSET");
            attributeMap.put(REDIS_URI_KEY, uriKey);
            attributeMap.put(REDIS_URI_VALUE, uriValue);
            attributeMap.put(REDIS_CAPABILITIES_KEY, capabilitiesKey);
            attributeMap.put(REDIS_CAPABILITIES_VALUE, capabilitiesJson);
            attributeMap.put(REDIS_START_KEY, startKey);
            attributeMap.put(REDIS_START_VALUE, startValue);
            attributeMap.put(DATABASE_OPERATION, "MSET");
            span.addEvent("Inserted into the database", attributeMap);
            this.connection.mset((Map<String, String>)ImmutableMap.of((Object)uriKey, (Object)uriValue, (Object)stereotypeKey, (Object)stereotypeJson, (Object)capabilitiesKey, (Object)capabilitiesJson, (Object)startKey, (Object)startValue));
            boolean bl = true;
            return bl;
        }
    }

    public Session get(SessionId id) throws NoSuchSessionException {
        Require.nonNull((String)"Session ID", (Object)id);
        try (Span span = this.tracer.getCurrentContext().createSpan("GET capabilitiesKey");){
            AttributeMap attributeMap = this.tracer.createAttributeMap();
            RemoteTags.SESSION_ID.accept(span, id);
            RemoteTags.SESSION_ID_EVENT.accept(attributeMap, id);
            this.setCommonSpanAttributes(span);
            this.setCommonEventAttributes(attributeMap);
            span.setAttribute(DATABASE_OPERATION, "GET");
            attributeMap.put(DATABASE_OPERATION, "GET");
            URI uri = this.getUri(id);
            attributeMap.put(REDIS_URI_KEY, this.uriKey(id));
            attributeMap.put(AttributeKey.SESSION_URI.getKey(), uri.toString());
            String capabilitiesKey = this.capabilitiesKey(id);
            String rawCapabilities = this.connection.get(capabilitiesKey);
            String stereotypeKey = this.stereotypeKey(id);
            String rawStereotype = this.connection.get(stereotypeKey);
            span.setAttribute(REDIS_CAPABILITIES_KEY, capabilitiesKey);
            attributeMap.put(REDIS_CAPABILITIES_KEY, capabilitiesKey);
            if (rawCapabilities != null) {
                span.setAttribute(REDIS_CAPABILITIES_VALUE, rawCapabilities);
            }
            ImmutableCapabilities caps = rawCapabilities == null ? new ImmutableCapabilities() : (Capabilities)JSON.toType(rawCapabilities, Capabilities.class);
            ImmutableCapabilities stereotype = rawStereotype == null ? new ImmutableCapabilities() : (Capabilities)JSON.toType(rawStereotype, Capabilities.class);
            String rawStart = this.connection.get(this.startKey(id));
            Instant start = (Instant)JSON.toType(rawStart, Instant.class);
            RemoteTags.CAPABILITIES.accept(span, caps);
            RemoteTags.CAPABILITIES_EVENT.accept(attributeMap, caps);
            span.addEvent("Retrieved session from the database", attributeMap);
            Session session = new Session(id, uri, (Capabilities)stereotype, (Capabilities)caps, start);
            return session;
        }
    }

    public URI getUri(SessionId id) throws NoSuchSessionException {
        Require.nonNull((String)"Session ID", (Object)id);
        try (Span span = this.tracer.getCurrentContext().createSpan("GET sessionURI");){
            AttributeMap attributeMap = this.tracer.createAttributeMap();
            RemoteTags.SESSION_ID.accept(span, id);
            RemoteTags.SESSION_ID_EVENT.accept(attributeMap, id);
            this.setCommonSpanAttributes(span);
            this.setCommonEventAttributes(attributeMap);
            span.setAttribute(DATABASE_OPERATION, "GET");
            attributeMap.put(DATABASE_OPERATION, "GET");
            String uriKey = this.uriKey(id);
            List<KeyValue<String, String>> rawValues = this.connection.mget(uriKey);
            String rawUri = (String)rawValues.get(0).getValueOrElse(null);
            span.setAttribute(REDIS_URI_KEY, uriKey);
            attributeMap.put(REDIS_URI_KEY, uriKey);
            if (rawUri == null) {
                NoSuchSessionException exception = new NoSuchSessionException("Unable to find session.");
                span.setAttribute("error", true);
                span.setStatus(Status.NOT_FOUND);
                Tags.EXCEPTION.accept(attributeMap, exception);
                attributeMap.put(AttributeKey.EXCEPTION_MESSAGE.getKey(), "Session URI does not exist in the database :" + exception.getMessage());
                span.addEvent(AttributeKey.EXCEPTION_EVENT.getKey(), attributeMap);
                throw exception;
            }
            span.setAttribute(REDIS_URI_VALUE, rawUri);
            attributeMap.put(REDIS_URI_KEY, uriKey);
            attributeMap.put(REDIS_URI_VALUE, rawUri);
            try {
                URI exception = new URI(rawUri);
                return exception;
            }
            catch (URISyntaxException e) {
                span.setAttribute("error", true);
                span.setStatus(Status.INTERNAL);
                Tags.EXCEPTION.accept(attributeMap, e);
                attributeMap.put(AttributeKey.SESSION_URI.getKey(), rawUri);
                attributeMap.put(AttributeKey.EXCEPTION_MESSAGE.getKey(), "Unable to convert session id to uri: " + e.getMessage());
                span.addEvent(AttributeKey.EXCEPTION_EVENT.getKey(), attributeMap);
                throw new NoSuchSessionException(String.format("Unable to convert session id (%s) to uri: %s", id, rawUri), (Throwable)e);
            }
        }
    }

    public void remove(SessionId id) {
        Require.nonNull((String)"Session ID", (Object)id);
        try (Span span = this.tracer.getCurrentContext().createSpan("DEL sessionUriKey capabilitiesKey");){
            AttributeMap attributeMap = this.tracer.createAttributeMap();
            RemoteTags.SESSION_ID.accept(span, id);
            RemoteTags.SESSION_ID_EVENT.accept(attributeMap, id);
            this.setCommonSpanAttributes(span);
            this.setCommonEventAttributes(attributeMap);
            span.setAttribute(DATABASE_OPERATION, "DEL");
            attributeMap.put(DATABASE_OPERATION, "DEL");
            String uriKey = this.uriKey(id);
            String capabilitiesKey = this.capabilitiesKey(id);
            String stereotypeKey = this.stereotypeKey(id);
            String startKey = this.startKey(id);
            span.setAttribute(REDIS_URI_KEY, uriKey);
            span.setAttribute(REDIS_CAPABILITIES_KEY, capabilitiesKey);
            span.setAttribute(REDIS_START_KEY, startKey);
            attributeMap.put(REDIS_URI_KEY, uriKey);
            attributeMap.put(REDIS_CAPABILITIES_KEY, capabilitiesKey);
            attributeMap.put(REDIS_START_KEY, startKey);
            span.addEvent("Deleted session from the database", attributeMap);
            this.connection.del(uriKey, capabilitiesKey, stereotypeKey, startKey);
        }
    }

    public void removeByUri(URI uri) {
        List<String> uriKeys = this.connection.getKeysByPattern("session:*:uri");
        if (uriKeys.isEmpty()) {
            return;
        }
        String[] keys = new String[uriKeys.size()];
        keys = uriKeys.toArray(keys);
        List<KeyValue<String, String>> keyValues = this.connection.mget(keys);
        keyValues.stream().filter(entry -> ((String)entry.getValue()).equals(uri.toString())).map(KeyValue::getKey).map(key -> {
            String[] sessionKey = key.split(":");
            return new SessionId(sessionKey[1]);
        }).forEach(this::remove);
    }

    public boolean isReady() {
        return this.connection.isOpen();
    }

    private String uriKey(SessionId id) {
        Require.nonNull((String)"Session ID", (Object)id);
        return "session:" + id.toString() + ":uri";
    }

    private String capabilitiesKey(SessionId id) {
        Require.nonNull((String)"Session ID", (Object)id);
        return "session:" + id.toString() + ":capabilities";
    }

    private String startKey(SessionId id) {
        Require.nonNull((String)"Session ID", (Object)id);
        return "session:" + id.toString() + ":start";
    }

    private String stereotypeKey(SessionId id) {
        Require.nonNull((String)"Session ID", (Object)id);
        return "session:" + id.toString() + ":stereotype";
    }

    private void setCommonSpanAttributes(Span span) {
        span.setAttribute("span.kind", Span.Kind.CLIENT.toString());
        span.setAttribute(DATABASE_SYSTEM, "redis");
    }

    private void setCommonEventAttributes(AttributeMap map) {
        map.put(DATABASE_SYSTEM, "redis");
        if (this.serverUri != null) {
            map.put(AttributeKey.DATABASE_CONNECTION_STRING.getKey(), this.serverUri.toString());
        }
    }

    public GridRedisClient getRedisClient() {
        return this.connection;
    }
}

