/*
 * Copyright 2018-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.amqp.rabbit.core;

import java.util.Map;

import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.event.AmqpEvent;
import org.springframework.util.Assert;

/**
 * Represents a broker event generated by the Event Exchange Plugin
 * (https://www.rabbitmq.com/event-exchange.html).
 *
 * @author Gary Russell
 * @since 2.1
 *
 */
@SuppressWarnings("serial")
public class BrokerEvent extends AmqpEvent {

	private final MessageProperties properties;

	/**
	 * Create an instance with the provided source and properties.
	 * @param source the source.
	 * @param properties the properties.
	 */
	public BrokerEvent(Object source, MessageProperties properties) {
		super(source);
		Assert.notNull(properties, "MessageProperties cannot be null");
		this.properties = properties;
	}

	/**
	 * The event type ({@link MessageProperties#getReceivedRoutingKey()}).
	 * @return the type.
	 */
	public String getEventType() {
		return this.properties.getReceivedRoutingKey();
	}

	/**
	 * Properties of the event {@link MessageProperties#getHeaders()}.
	 * @return the properties.
	 */
	public Map<String, Object> getEventProperties() {
		return this.properties.getHeaders();
	}

	/**
	 * The complete {@link MessageProperties} from the event representing the event.
	 * (The body is always blank).
	 * @return the message properties.
	 */
	public MessageProperties getMessageProperties() {
		return this.properties;
	}

	@Override
	public String toString() {
		return "BrokerEvent [eventType=" + this.getEventType() + ", properties=" + this.getEventProperties() + "]";
	}

}
