/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.hadoop.fs;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPInputStream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.ChecksumFileSystem;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.io.IOUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.data.hadoop.HadoopException;
import org.springframework.data.hadoop.fs.FsShellPermissions;
import org.springframework.data.hadoop.fs.PrettyPrintList;
import org.springframework.data.hadoop.fs.PrettyPrintMap;
import org.springframework.data.hadoop.fs.TextRecordInputStream;
import org.springframework.util.Assert;
import org.springframework.util.FileCopyUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

public class FsShell
implements Closeable,
DisposableBean {
    private boolean internalFs = false;
    private FileSystem fs;
    private final Configuration configuration;
    private Trash trash;

    public FsShell(Configuration configuration) {
        this(configuration, null);
    }

    public FsShell(Configuration configuration, FileSystem fs) {
        this.configuration = configuration;
        try {
            this.fs = fs != null ? fs : FileSystem.get((Configuration)configuration);
            this.internalFs = fs == null;
            this.trash = new Trash(configuration);
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot create shell " + ex.getMessage(), ex);
        }
    }

    public void destroy() throws Exception {
        this.close();
    }

    @Override
    public void close() throws IOException {
        if (this.internalFs && this.fs != null) {
            this.fs.close();
            this.fs = null;
        }
    }

    private String getContent(InputStream in) throws IOException {
        StringWriter writer = new StringWriter(in.available());
        InputStreamReader reader = new InputStreamReader(in, "UTF-8");
        FileCopyUtils.copy((Reader)reader, (Writer)writer);
        return writer.toString();
    }

    public Collection<Path> cat(String uri) {
        return this.cat(new String[]{uri});
    }

    public Collection<Path> cat(String ... uris) {
        PrettyPrintList<Path> results = new PrettyPrintList<Path>(new PrettyPrintList.ListPrinter<Path>(){

            @Override
            public String toString(Path e) throws IOException {
                try {
                    FileSystem srcFS = FsShell.this.getFS(e);
                    return FsShell.this.getContent((InputStream)srcFS.open(e));
                }
                catch (IOException ex) {
                    return "No such file or directory " + e.toUri();
                }
            }
        });
        try {
            if (!ObjectUtils.isEmpty((Object[])uris)) {
                for (String uri : uris) {
                    Path src = new Path(uri);
                    FileSystem srcFS = this.getFS(src);
                    results.addAll(Arrays.asList(FileUtil.stat2Paths((FileStatus[])srcFS.globStatus(src), (Path)src)));
                }
            }
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot execute command " + ex.getMessage(), ex);
        }
        return Collections.unmodifiableCollection(results);
    }

    public void chgrp(String group, String uri) {
        this.chgrp(group, uri);
    }

    public void chgrp(String group, String ... uris) {
        this.chgrp(false, group, uris);
    }

    public void chgrpr(String group, String uri) {
        this.chgrp(group, new String[]{uri});
    }

    public void chgrpr(String group, String ... uris) {
        this.chgrp(true, group, uris);
    }

    public void chgrp(boolean recursive, String group, String uri) {
        this.chgrp(recursive, group, new String[]{uri});
    }

    public void chgrp(boolean recursive, String group, String ... uris) {
        FsShellPermissions.changePermissions(this.fs, this.configuration, FsShellPermissions.Op.CHGRP, recursive, group, uris);
    }

    public void chmod(String mode, String uri) {
        this.chmod(mode, new String[]{uri});
    }

    public void chmod(String mode, String ... uris) {
        this.chmod(false, mode, uris);
    }

    public void chmodr(String mode, String uri) {
        this.chmodr(mode, new String[]{uri});
    }

    public void chmodr(String mode, String ... uris) {
        this.chmod(true, mode, uris);
    }

    public void chmodr(Integer mode, String uri) {
        this.chmodr(mode, new String[]{uri});
    }

    public void chmodr(Integer mode, String ... uris) {
        this.chmod(true, String.valueOf(mode), uris);
    }

    public void chmod(Integer mode, String uri) {
        this.chmod(mode, new String[]{uri});
    }

    public void chmod(Integer mode, String ... uris) {
        this.chmod(false, String.valueOf(mode), uris);
    }

    public void chmod(boolean recursive, Integer mode, String uri) {
        this.chmod(recursive, mode, new String[]{uri});
    }

    public void chmod(boolean recursive, Integer mode, String ... uris) {
        this.chmod(recursive, String.valueOf(mode), uris);
    }

    public void chmod(boolean recursive, String mode, String uri) {
        this.chmod(recursive, mode, new String[]{uri});
    }

    public void chmod(boolean recursive, String mode, String ... uris) {
        FsShellPermissions.changePermissions(this.fs, this.configuration, FsShellPermissions.Op.CHMOD, recursive, mode, uris);
    }

    public void chown(String mode, String uri) {
        this.chown(mode, new String[]{uri});
    }

    public void chown(String mode, String ... uris) {
        this.chown(false, mode, uris);
    }

    public void chownr(String mode, String uri) {
        this.chownr(mode, new String[]{uri});
    }

    public void chownr(String mode, String ... uris) {
        this.chown(true, mode, uris);
    }

    public void chown(boolean recursive, String owner, String uri) {
        this.chown(recursive, owner, new String[]{uri});
    }

    public void chown(boolean recursive, String owner, String ... uris) {
        FsShellPermissions.changePermissions(this.fs, this.configuration, FsShellPermissions.Op.CHOWN, recursive, owner, uris);
    }

    public void copyFromLocal(String src, String dst) {
        this.copyFromLocal(src, dst, null);
    }

    public void copyFromLocal(String src, String src2, String ... dst) {
        Object[] va = FsShell.parseVarargs(src, src2, dst);
        List srcs = (List)va[0];
        Path dstPath = (Path)va[1];
        try {
            FileSystem dstFs = this.getFS(dstPath);
            dstFs.copyFromLocalFile(false, false, srcs.toArray(new Path[srcs.size()]), dstPath);
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot copy resources " + ex.getMessage(), ex);
        }
    }

    public void copyToLocal(String src, String localdst) {
        this.copyToLocal(true, false, src, localdst);
    }

    public void copyToLocal(boolean ignorecrc, boolean crc, String src, String localdst) {
        File dst = new File(localdst);
        Path srcpath = new Path(src);
        try {
            FileSystem srcFs = this.getFS(srcpath);
            srcFs.setVerifyChecksum(ignorecrc);
            if (crc && !(srcFs instanceof ChecksumFileSystem)) {
                crc = false;
            }
            FileStatus[] srcs = srcFs.globStatus(srcpath);
            boolean dstIsDir = dst.isDirectory();
            if (srcs.length > 1 && !dstIsDir) {
                throw new IllegalArgumentException("When copying multiple files, destination should be a directory.");
            }
            for (FileStatus status : srcs) {
                Path p = status.getPath();
                File f = dstIsDir ? new File(dst, p.getName()) : dst;
                this.copyToLocal(srcFs, p, f, crc);
            }
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot copy resources " + ex.getMessage(), ex);
        }
    }

    private void copyToLocal(FileSystem srcFS, Path src, File dst, boolean copyCrc) throws IOException {
        String COPYTOLOCAL_PREFIX = "_copyToLocal_";
        if (!srcFS.getFileStatus(src).isDir()) {
            if (dst.exists()) {
                throw new IOException("Target " + dst + " already exists");
            }
            File tmp = FileUtil.createLocalTempFile((File)dst.getAbsoluteFile(), (String)"_copyToLocal_", (boolean)true);
            if (!FileUtil.copy((FileSystem)srcFS, (Path)src, (File)tmp, (boolean)false, (Configuration)srcFS.getConf())) {
                throw new IOException("Failed to copy " + src + " to " + dst);
            }
            if (!tmp.renameTo(dst)) {
                throw new IOException("Failed to rename tmp file " + tmp + " to local destination \"" + dst + "\".");
            }
            if (copyCrc) {
                if (!(srcFS instanceof ChecksumFileSystem)) {
                    throw new IOException("Source file system does not have crc files");
                }
                ChecksumFileSystem csfs = (ChecksumFileSystem)srcFS;
                File dstcs = FileSystem.getLocal((Configuration)srcFS.getConf()).pathToFile(csfs.getChecksumFile(new Path(dst.getCanonicalPath())));
                this.copyToLocal(csfs.getRawFileSystem(), csfs.getChecksumFile(src), dstcs, false);
            }
        } else {
            dst.mkdirs();
            for (FileStatus path : srcFS.listStatus(src)) {
                this.copyToLocal(srcFS, path.getPath(), new File(dst, path.getPath().getName()), copyCrc);
            }
        }
    }

    public Map<Path, ContentSummary> count(String uri) {
        return this.count(new String[]{uri});
    }

    public Map<Path, ContentSummary> count(String ... uris) {
        return this.count(false, uris);
    }

    public Map<Path, ContentSummary> count(boolean quota, String uri) {
        return this.count(quota, new String[]{uri});
    }

    public Map<Path, ContentSummary> count(final boolean quota, String ... uris) {
        PrettyPrintMap<Path, ContentSummary> results = new PrettyPrintMap<Path, ContentSummary>(uris.length, new PrettyPrintMap.MapPrinter<Path, ContentSummary>(){

            @Override
            public String toString(Path p, ContentSummary c) throws IOException {
                return c.toString(quota) + p;
            }
        });
        for (String src : uris) {
            try {
                Path srcPath = new Path(src);
                FileSystem fs = this.getFS(srcPath);
                Object[] statuses = fs.globStatus(srcPath);
                Assert.notEmpty((Object[])statuses, (String)("Can not find listing for " + src));
                for (Object s : statuses) {
                    Path p = s.getPath();
                    results.put(p, fs.getContentSummary(p));
                }
            }
            catch (IOException ex) {
                throw new HadoopException("Cannot find listing " + ex.getMessage(), ex);
            }
        }
        return Collections.unmodifiableMap(results);
    }

    public void cp(String src, String dst) {
        this.cp(src, dst, null);
    }

    public void cp(String src, String src2, String ... dst) {
        Object[] va = FsShell.parseVarargs(src, src2, dst);
        List srcs = (List)va[0];
        Path dstPath = (Path)va[1];
        try {
            boolean isDestDir;
            FileSystem dstFs = dstPath.getFileSystem(this.configuration);
            boolean bl = isDestDir = !dstFs.isFile(dstPath);
            if ((StringUtils.hasText((String)src2) || ObjectUtils.isEmpty((Object[])dst) && dst.length > 2) && !isDestDir) {
                throw new IllegalArgumentException("When copying multiple files, destination " + dstPath.toUri() + " should be a directory.");
            }
            for (Path path : srcs) {
                FileSystem srcFs = path.getFileSystem(this.configuration);
                Object[] from = FileUtil.stat2Paths((FileStatus[])srcFs.globStatus(path), (Path)path);
                if (!ObjectUtils.isEmpty((Object[])from) && from.length > 1 && !isDestDir) {
                    throw new IllegalArgumentException("When copying multiple files, destination should be a directory.");
                }
                for (Object fromPath : from) {
                    FileUtil.copy((FileSystem)srcFs, (Path)fromPath, (FileSystem)dstFs, (Path)dstPath, (boolean)false, (Configuration)this.configuration);
                }
            }
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot copy resources " + ex.getMessage(), ex);
        }
    }

    public Map<Path, Long> du(String uri) {
        return this.du(new String[]{uri});
    }

    public Map<Path, Long> du(String ... uris) {
        return this.du(false, uris);
    }

    public Map<Path, Long> du(boolean summary, String string) {
        return this.du(summary, new String[]{string});
    }

    public Map<Path, Long> du(final boolean summary, String ... strings) {
        if (ObjectUtils.isEmpty((Object[])strings)) {
            strings = new String[]{"."};
        }
        int BORDER = 2;
        PrettyPrintMap<Path, Long> results = new PrettyPrintMap<Path, Long>(strings.length, new PrettyPrintMap.MapPrinter<Path, Long>(){

            @Override
            public String toString(Path path, Long size) throws Exception {
                if (summary) {
                    return ("".equals(path) ? "." : path) + "\t" + size;
                }
                return String.format("%-12d", size) + path;
            }
        });
        try {
            for (String src : strings) {
                Path srcPath = new Path(src);
                FileSystem srcFs = this.getFS(srcPath);
                FileStatus[] fileStatus = srcFs.globStatus(srcPath);
                if (summary) {
                    for (FileStatus status : fileStatus) {
                        results.put(status.getPath(), srcFs.getContentSummary(status.getPath()).getLength());
                    }
                    continue;
                }
                Object[] items = srcFs.listStatus(FileUtil.stat2Paths((FileStatus[])fileStatus, (Path)srcPath));
                if (ObjectUtils.isEmpty((Object[])items) && !srcFs.exists(srcPath)) {
                    throw new HadoopException("Cannot access " + src + ": No such file or directory.");
                }
                for (Object status : items) {
                    Long size = status.isDir() ? srcFs.getContentSummary(status.getPath()).getLength() : status.getLen();
                    results.put(status.getPath(), size);
                }
            }
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot inspect resources " + ex.getMessage(), ex);
        }
        return Collections.unmodifiableMap(results);
    }

    public Map<Path, Long> dus(String string) {
        return this.dus(new String[]{string});
    }

    public Map<Path, Long> dus(String ... strings) {
        return this.du(true, strings);
    }

    public void expunge() {
        try {
            this.trash.expunge();
            this.trash.checkpoint();
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot expunge trash" + ex.getMessage(), ex);
        }
    }

    public void get(String src, String dst) {
        this.copyToLocal(src, dst);
    }

    public void get(boolean ignorecrc, boolean crc, String src, String dst) {
        this.copyToLocal(ignorecrc, crc, src, dst);
    }

    public void getmerge(String src, String localdst) {
        this.getmerge(src, localdst, false);
    }

    public void getmerge(String src, String localdst, boolean addnl) {
        Path srcPath = new Path(src);
        Path dst = new Path(localdst);
        try {
            FileSystem srcFs = this.getFS(srcPath);
            Path[] srcs = FileUtil.stat2Paths((FileStatus[])srcFs.globStatus(srcPath), (Path)srcPath);
            for (int i = 0; i < srcs.length; ++i) {
                if (addnl) {
                    FileUtil.copyMerge((FileSystem)srcFs, (Path)srcs[i], (FileSystem)FileSystem.getLocal((Configuration)this.configuration), (Path)dst, (boolean)false, (Configuration)this.configuration, (String)"\n");
                    continue;
                }
                FileUtil.copyMerge((FileSystem)srcFs, (Path)srcs[i], (FileSystem)FileSystem.getLocal((Configuration)this.configuration), (Path)dst, (boolean)false, (Configuration)this.configuration, null);
            }
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot getmerge " + ex.getMessage(), ex);
        }
    }

    public Collection<FileStatus> ls(String match) {
        return this.ls(false, new String[]{match});
    }

    public Collection<FileStatus> ls(String ... match) {
        return this.ls(false, match);
    }

    public Collection<FileStatus> ls(boolean recursive, String match) {
        return this.ls(recursive, new String[]{match});
    }

    public Collection<FileStatus> ls(boolean recursive, String ... match) {
        PrettyPrintList<FileStatus> results = new PrettyPrintList<FileStatus>(new PrettyPrintList.ListPrinter<FileStatus>(){

            @Override
            public String toString(FileStatus stat) throws Exception {
                SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm");
                int maxReplication = 3;
                int maxLen = 10;
                int maxOwner = 10;
                int maxGroup = 10;
                StringBuilder sb = new StringBuilder();
                sb.append((stat.isDir() ? "d" : "-") + stat.getPermission() + " ");
                sb.append(String.format("%" + maxReplication + "s ", !stat.isDir() ? Short.valueOf(stat.getReplication()) : "-"));
                sb.append(String.format("%-" + maxOwner + "s ", stat.getOwner()));
                sb.append(String.format("%-" + maxGroup + "s ", stat.getGroup()));
                sb.append(String.format("%" + maxLen + "d ", stat.getLen()));
                sb.append(df.format(new Date(stat.getModificationTime())) + " ");
                sb.append(stat.getPath().toUri().getPath());
                return sb.toString();
            }
        });
        try {
            for (String src : match) {
                Path srcPath = new Path(src);
                FileSystem srcFs = this.getFS(srcPath);
                Object[] srcs = srcFs.globStatus(srcPath);
                if (!ObjectUtils.isEmpty((Object[])srcs)) {
                    for (Object status : srcs) {
                        this.ls((FileStatus)status, srcFs, recursive, results);
                    }
                    continue;
                }
                throw new IllegalArgumentException("Cannot access " + srcPath + ": No such file or directory.");
            }
            return Collections.unmodifiableCollection(results);
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot list resources " + ex.getMessage(), ex);
        }
    }

    private void ls(FileStatus src, FileSystem srcFs, boolean recursive, Collection<FileStatus> results) throws IOException {
        Object[] items;
        results.add(src);
        if (src.isDir() && !ObjectUtils.isEmpty((Object[])(items = srcFs.listStatus(src.getPath())))) {
            for (Object stat : items) {
                if (recursive && stat.isDir()) {
                    this.ls((FileStatus)stat, srcFs, recursive, results);
                    continue;
                }
                results.add((FileStatus)stat);
            }
        }
    }

    public Collection<FileStatus> lsr(String match) {
        return this.ls(true, match);
    }

    public Collection<FileStatus> lsr(String ... match) {
        return this.ls(true, match);
    }

    public void mkdir(String uri) {
        this.mkdir(new String[]{uri});
    }

    public void mkdir(String ... uris) {
        for (String src : uris) {
            try {
                Path p = new Path(src);
                FileSystem srcFs = this.getFS(p);
                FileStatus fstatus = null;
                try {
                    fstatus = srcFs.getFileStatus(p);
                    if (fstatus.isDir()) {
                        throw new IllegalStateException("Cannot create directory " + src + ": File exists");
                    }
                    throw new IllegalStateException(src + " exists but is not a directory");
                }
                catch (FileNotFoundException e) {
                    if (srcFs.mkdirs(p)) continue;
                    throw new HadoopException("Failed to create " + src);
                }
            }
            catch (IOException ex) {
                throw new HadoopException("Cannot create directory " + ex.getMessage(), ex);
            }
        }
    }

    public void moveFromLocal(String localsrc, String dst) {
        this.moveFromLocal(localsrc, dst, null);
    }

    public void moveFromLocal(String localsrc, String localsrc2, String ... dst) {
        Object[] va = FsShell.parseVarargs(localsrc, localsrc2, dst);
        List srcs = (List)va[0];
        Path dstPath = (Path)va[1];
        try {
            FileSystem dstFs = dstPath.getFileSystem(this.configuration);
            dstFs.moveFromLocalFile(srcs.toArray(new Path[srcs.size()]), dstPath);
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot move resources " + ex.getMessage(), ex);
        }
    }

    public void moveToLocal(String src, String dst) {
        this.moveToLocal(false, src, dst);
    }

    public void moveToLocal(boolean crc, String src, String dst) {
        throw new UnsupportedOperationException("Option 'moveToLocal' is not implemented yet.");
    }

    public void mv(String src, String dst) {
        this.mv(src, dst, null);
    }

    public void mv(String src, String src2, String ... dst) {
        Object[] va = FsShell.parseVarargs(src, src2, dst);
        List sources = (List)va[0];
        Path dstPath = (Path)va[1];
        try {
            boolean isDstDir;
            FileSystem dstFs = this.getFS(dstPath);
            boolean bl = isDstDir = !dstFs.isFile(dstPath);
            if (sources.size() > 1 && !isDstDir) {
                throw new IllegalArgumentException("Destination must be a dir when moving multiple files");
            }
            for (Path srcPath : sources) {
                URI dstURI;
                FileSystem srcFs = this.getFS(srcPath);
                URI srcURI = srcFs.getUri();
                if (srcURI.compareTo(dstURI = dstFs.getUri()) != 0) {
                    throw new IllegalArgumentException("src and destination filesystems do not match.");
                }
                Path[] srcs = FileUtil.stat2Paths((FileStatus[])srcFs.globStatus(srcPath), (Path)srcPath);
                if (srcs.length > 1 && !isDstDir) {
                    throw new IllegalArgumentException("When moving multiple files, destination should be a directory.");
                }
                for (Path s : srcs) {
                    if (srcFs.rename(s, dstPath)) continue;
                    FileStatus srcFstatus = null;
                    FileStatus dstFstatus = null;
                    try {
                        srcFstatus = srcFs.getFileStatus(s);
                    }
                    catch (FileNotFoundException e) {
                        // empty catch block
                    }
                    try {
                        dstFstatus = dstFs.getFileStatus(dstPath);
                    }
                    catch (IOException e) {
                        // empty catch block
                    }
                    if (srcFstatus != null && dstFstatus != null && srcFstatus.isDir() && !dstFstatus.isDir()) {
                        throw new IllegalArgumentException("cannot overwrite non directory " + dstPath + " with directory " + s);
                    }
                    throw new HadoopException("Failed to rename " + s + " to " + dstPath);
                }
            }
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot rename resources " + ex.getMessage(), ex);
        }
    }

    public void put(String localsrc, String dst) {
        this.put(localsrc, dst, null);
    }

    public void put(String localsrc, String localsrc2, String ... dst) {
        this.copyFromLocal(localsrc, localsrc2, dst);
    }

    public void rm(String uri) {
        this.rm(new String[]{uri});
    }

    public void rm(String ... uris) {
        this.rm(false, false, uris);
    }

    public void rm(boolean recursive, String uri) {
        this.rm(recursive, new String[]{uri});
    }

    public void rm(boolean recursive, String ... uris) {
        this.rm(recursive, false, uris);
    }

    public void rm(boolean recursive, boolean skipTrash, String ... uris) {
        for (String uri : uris) {
            try {
                Path src = new Path(uri);
                FileSystem srcFs = this.getFS(src);
                for (Path p : FileUtil.stat2Paths((FileStatus[])srcFs.globStatus(src), (Path)src)) {
                    FileStatus status = srcFs.getFileStatus(p);
                    if (status.isDir() && !recursive) {
                        throw new IllegalStateException("Cannot remove directory \"" + src + "\", if recursive deletion was not specified");
                    }
                    if (!skipTrash) {
                        try {
                            Trash trashTmp = new Trash(srcFs, this.configuration);
                            trashTmp.moveToTrash(p);
                        }
                        catch (IOException ex) {
                            throw new HadoopException("Cannot move to Trash resource " + p, ex);
                        }
                    }
                    srcFs.delete(p, recursive);
                }
            }
            catch (IOException ex) {
                throw new HadoopException("Cannot delete (all) resources " + ex.getMessage(), ex);
            }
        }
    }

    public void rmr(String uri) {
        this.rmr(new String[]{uri});
    }

    public void rmr(String ... uris) {
        this.rm(true, false, uris);
    }

    public void rmr(boolean skipTrash, String uri) {
        this.rmr(skipTrash, new String[]{uri});
    }

    public void rmr(boolean skipTrash, String ... uris) {
        this.rm(true, skipTrash, uris);
    }

    public void setrep(short replication, String uri) {
        this.setrep(replication, new String[]{uri});
    }

    public void setrep(short replication, String ... uris) {
        this.setrep(false, replication, uris);
    }

    public void setrep(boolean recursive, short replication, String uri) {
        this.setrep(recursive, replication, new String[]{uri});
    }

    public void setrep(boolean recursive, short replication, String ... uris) {
        this.setrep(-1L, recursive, replication, uris);
    }

    public void setrepr(short replication, String ... uris) {
        this.setrep(-1L, true, replication, uris);
    }

    public void setrepr(short replication, String uri) {
        this.setrepr(replication, new String[]{uri});
    }

    public void setrepr(long secondsToWait, short replication, String uri) {
        this.setrepr(secondsToWait, replication, new String[]{uri});
    }

    public void setrepr(long secondsToWait, short replication, String ... uris) {
        this.setrep(secondsToWait, true, replication, uris);
    }

    public void setrep(long secondsToWait, boolean recursive, short replication, String uri) {
        this.setrep(secondsToWait, recursive, replication, new String[]{uri});
    }

    public void setrep(long secondsToWait, boolean recursive, short replication, String ... uris) {
        block9: {
            Assert.isTrue((replication >= 1 ? 1 : 0) != 0, (String)"Replication must be >=1");
            ArrayList<Path> waitList = secondsToWait >= 0L ? new ArrayList<Path>() : null;
            try {
                FileSystem srcFs;
                for (String uri : uris) {
                    Path[] srcs;
                    Path srcPath = new Path(uri);
                    srcFs = this.getFS(srcPath);
                    for (Path src : srcs = FileUtil.stat2Paths((FileStatus[])srcFs.globStatus(srcPath), (Path)srcPath)) {
                        this.setrep(replication, recursive, srcFs, src, waitList);
                    }
                }
                if (waitList == null) break block9;
                boolean waitUntilDone = secondsToWait == 0L;
                long timeLeft = TimeUnit.SECONDS.toMillis(secondsToWait);
                for (Path path : waitList) {
                    srcFs = this.getFS(path);
                    FileStatus status = srcFs.getFileStatus(path);
                    long len = status.getLen();
                    boolean done = false;
                    while (!done) {
                        int i;
                        BlockLocation[] locations = srcFs.getFileBlockLocations(status, 0L, len);
                        for (i = 0; i < locations.length && locations[i].getHosts().length == replication; ++i) {
                        }
                        done = i == locations.length;
                        if (done || !waitUntilDone && timeLeft <= 5000L) continue;
                        try {
                            Thread.sleep(10000L);
                        }
                        catch (InterruptedException e) {
                            return;
                        }
                        timeLeft = -1000L;
                    }
                }
            }
            catch (IOException ex) {
                throw new HadoopException("Cannot set replication " + ex.getMessage(), ex);
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void setrep(short replication, boolean recursive, FileSystem srcFs, Path src, List<Path> waitList) throws IOException {
        if (srcFs.isFile(src)) {
            if (!srcFs.setReplication(src, replication)) throw new HadoopException("Cannot set replication for " + src);
            if (waitList == null) return;
            waitList.add(src);
            return;
        } else {
            Object[] items;
            if (!recursive || ObjectUtils.isEmpty((Object[])(items = srcFs.listStatus(src)))) return;
            for (Object status : items) {
                this.setrep(replication, recursive, srcFs, status.getPath(), waitList);
            }
        }
    }

    public boolean test(String uri) {
        return this.test(true, false, false, uri);
    }

    public boolean test(boolean exists, boolean zero, boolean directory, String uri) {
        Path f = new Path(uri);
        boolean result = true;
        try {
            FileSystem srcFs = this.getFS(f);
            result = srcFs.exists(f);
            if (result && zero) {
                result &= srcFs.getFileStatus(f).getLen() == 0L;
            }
            if (result && directory) {
                result &= srcFs.getFileStatus(f).isDir();
            }
            return result;
        }
        catch (IOException ex) {
            throw new HadoopException("Cannot test resource " + uri + ";" + ex.getMessage(), ex);
        }
    }

    public Collection<String> text(String uri) {
        return this.text(new String[]{uri});
    }

    public Collection<String> text(String ... uris) {
        PrettyPrintList<String> texts = new PrettyPrintList<String>(new PrettyPrintList.ListPrinter<String>(){

            @Override
            public String toString(String e) throws Exception {
                return e + "\n";
            }
        });
        for (String uri : uris) {
            Object in = null;
            FSDataInputStream i = null;
            try {
                Path srcPat = new Path(uri);
                FileSystem srcFs = this.getFS(srcPat);
                for (Path src : FileUtil.stat2Paths((FileStatus[])srcFs.globStatus(srcPat), (Path)srcPat)) {
                    Assert.isTrue((boolean)srcFs.isFile(src), (String)"Source must be a file");
                    i = srcFs.open(src);
                    switch (i.readShort()) {
                        case 8075: {
                            i.seek(0L);
                            in = new GZIPInputStream((InputStream)i);
                            break;
                        }
                        case 21317: {
                            if (i.readByte() != 81) break;
                            i.close();
                            in = new TextRecordInputStream(src, srcFs, this.configuration);
                            break;
                        }
                        default: {
                            in = i;
                        }
                    }
                    i.seek(0L);
                    texts.add(this.getContent((InputStream)in));
                }
            }
            catch (IOException ex) {
                throw new HadoopException("Cannot read " + uri + ";" + ex.getMessage(), ex);
            }
            finally {
                IOUtils.closeStream(in);
                IOUtils.closeStream(i);
            }
        }
        return texts;
    }

    public void touchz(String uri) {
        this.touchz(new String[]{uri});
    }

    public void touchz(String ... uris) {
        for (String uri : uris) {
            try {
                Path src = new Path(uri);
                FileSystem srcFs = this.getFS(src);
                if (srcFs.exists(src)) {
                    FileStatus st = srcFs.getFileStatus(src);
                    if (st.isDir()) {
                        throw new IllegalArgumentException(src + " is a directory");
                    }
                    if (st.getLen() == 0L) continue;
                    throw new IllegalArgumentException(src + " must be a zero-length file");
                }
                IOUtils.closeStream((Closeable)srcFs.create(src));
            }
            catch (IOException ex) {
                throw new HadoopException("Cannot touchz " + uri + ";" + ex.getMessage(), ex);
            }
        }
    }

    private static Object[] parseVarargs(String src1, String src2, String ... dst) {
        Assert.hasText((String)src1, (String)"at least one valid source path needs to be specified");
        Path dstPath = null;
        ArrayList<Path> srcs = new ArrayList<Path>();
        srcs.add(new Path(src1));
        if (!ObjectUtils.isEmpty((Object[])dst)) {
            srcs.add(new Path(src2));
            for (int i = 0; i < dst.length - 1; ++i) {
                srcs.add(new Path(dst[i]));
            }
            dstPath = new Path(dst[dst.length - 1]);
        } else {
            dstPath = new Path(src2);
        }
        return new Object[]{srcs, dstPath};
    }

    private FileSystem getFS(Path path) throws IOException {
        if (StringUtils.hasText((String)path.toUri().getScheme())) {
            return path.getFileSystem(this.configuration);
        }
        return this.fs;
    }
}

