/*
 * Copyright 2002-2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.integration.jdbc;

import java.util.List;

import javax.sql.DataSource;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.integration.Message;
import org.springframework.integration.handler.AbstractReplyProducingMessageHandler;
import org.springframework.integration.support.MessageBuilder;
import org.springframework.jdbc.core.JdbcOperations;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

/**
 * @author Dave Syer
 * 
 * @since 2.0
 */
public class JdbcOutboundGateway extends AbstractReplyProducingMessageHandler implements InitializingBean {

	private final JdbcMessageHandler handler;

	private final JdbcPollingChannelAdapter poller;

	private volatile SqlParameterSourceFactory sqlParameterSourceFactory = new ExpressionEvaluatingSqlParameterSourceFactory();

	private volatile boolean keysGenerated;

	public JdbcOutboundGateway(DataSource dataSource, String updateQuery) {
		this(new JdbcTemplate(dataSource), updateQuery, null);
	}

	public JdbcOutboundGateway(DataSource dataSource, String updateQuery, String selectQuery) {
		this(new JdbcTemplate(dataSource), updateQuery, selectQuery);
	}

	public JdbcOutboundGateway(JdbcOperations jdbcOperations, String updateQuery) {
		this(jdbcOperations, updateQuery, null);
	}

	public JdbcOutboundGateway(JdbcOperations jdbcOperations, String updateQuery, String selectQuery) {
		if (selectQuery != null) {
			poller = new JdbcPollingChannelAdapter(jdbcOperations, selectQuery);
			poller.setMaxRowsPerPoll(1);
		}
		else {
			poller = null;
		}
		handler = new JdbcMessageHandler(jdbcOperations, updateQuery);
	}

	public void setMaxRowsPerPoll(int maxRows) {
		poller.setMaxRowsPerPoll(maxRows);
	}

	@Override
	protected void onInit() {
		handler.afterPropertiesSet();
	}

	@Override
	protected Object handleRequestMessage(Message<?> requestMessage) {
		List<?> list = handler.executeUpdateQuery(requestMessage, keysGenerated);
		if (poller != null) {
			SqlParameterSource sqlQueryParameterSource = sqlParameterSourceFactory
					.createParameterSource(requestMessage);
			if (keysGenerated) {
				if (!list.isEmpty()) {
					if (list.size() == 1) {
						sqlQueryParameterSource = sqlParameterSourceFactory.createParameterSource(list.get(0));
					}
					else {
						sqlQueryParameterSource = sqlParameterSourceFactory.createParameterSource(list);
					}
				}
			}
			list = poller.doPoll(sqlQueryParameterSource);
			if (list.isEmpty()) {
				return null;
			}
		}
		Object payload = list;
		if (list.isEmpty()) {
			return null;
		}
		if (list.size() == 1) {
			payload = list.get(0);
		}
		return MessageBuilder.withPayload(payload).copyHeaders(requestMessage.getHeaders()).build();
	}

	/**
	 * Flag to indicate that the update query is an insert with autogenerated keys, which will be logged at debug level.
	 * @param keysGenerated the flag value to set
	 */
	public void setKeysGenerated(boolean keysGenerated) {
		this.keysGenerated = keysGenerated;
	}

	public void setRequestSqlParameterSourceFactory(SqlParameterSourceFactory sqlParameterSourceFactory) {
		handler.setSqlParameterSourceFactory(sqlParameterSourceFactory);
	}

	public void setReplySqlParameterSourceFactory(SqlParameterSourceFactory sqlParameterSourceFactory) {
		this.sqlParameterSourceFactory = sqlParameterSourceFactory;
	}

	public void setRowMapper(RowMapper<?> rowMapper) {
		poller.setRowMapper(rowMapper);
	}

}
