/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('entityResourceDisplayer', ['jquery', 'resource'], function($, $resource) {
  'use strict';

  var displayFromBreadcrumb = function(resourceReference, breadcrumb) {
    breadcrumb.addClass('resource-hint');
    var label = breadcrumb.find('.active').remove().find('a').addClass('resource-label');
    var icon = $('<span class="resource-icon"></span>').addClass($resource.types[resourceReference.type].icon);
    var remove = $('<span class="glyphicon glyphicon-remove remove"></span>');
    // Remove the home icon from the breadcrumb because it distracts the user (from the resource icon) and because it
    // doesn't bring additional information to identify the resource. We all know that every path starts from home.
    breadcrumb.find('.wiki').first().remove();
    return $('<div></div>').append(icon).append(document.createTextNode(' ')).append(label)
      .append(document.createTextNode(' ')).append(remove).add(breadcrumb);
  };

  var maybeDisplayFromBreadcrumb = function(resourceReference, breadcrumb, deferred) {
    if (breadcrumb.hasClass('breadcrumb')) {
      deferred.resolve(displayFromBreadcrumb(resourceReference, breadcrumb));
    } else {
      deferred.reject();
    }
  };

  $resource.displayers.doc = function(resourceReference) {
    var deferred = $.Deferred();
    $.post(XWiki.currentDocument.getURL('get'), {
      language: $('html').attr('lang'),
      xpage: 'hierarchy_reference',
      reference: resourceReference.reference,
      limit: 5
    }).done(function(html) {
      maybeDisplayFromBreadcrumb(resourceReference, $(html), deferred);
    }).fail(function() {
      deferred.reject();
    });
    return deferred.promise();
  };

  $resource.displayers.attach = function(resourceReference) {
    var deferred = $.Deferred();
    var attachmentReference = $resource.convertResourceReferenceToEntityReference(resourceReference);
    $.post(XWiki.currentDocument.getURL('get'), {
      language: $('html').attr('lang'),
      xpage: 'hierarchy_reference',
      reference: XWiki.Model.serialize(attachmentReference.parent),
      limit: 4
    }).done(function(html) {
      var attachmentLink = $('<li class="attachment active"><a></a></li>');
      var attachmentURL = new XWiki.Document(attachmentReference.parent).getURL('download') + '/' +
        encodeURIComponent(attachmentReference.name);
      attachmentLink.find('a').attr('href', attachmentURL).text(attachmentReference.name);
      var breadcrumb = $(html);
      breadcrumb.find('.active').removeClass('active').after(attachmentLink);
      maybeDisplayFromBreadcrumb(resourceReference, breadcrumb, deferred);
    }).fail(function() {
      deferred.reject();
    });
    return deferred.promise();
  };
});

/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('entityResourcePickerTranslationKeys', [], [
  'select',
  'doc.title',
  'attach.title'
]);

define('entityResourcePicker', [
  'jquery', 'resource', 'modal', 'l10n!entityResourcePicker', 'tree'
], function($, $resource, $modal, translations) {
  'use strict';

  var createTreeElement = function(attributes) {
    return $(document.createElement('div')).attr($.extend({
      'class': 'ckeditor-tree jstree-no-links',
      'data-edges': true,
      'data-finder': true,
      'data-icons': true,
      'data-responsive': true
    }, attributes));
  };

  var createTreePicker = function(modal, handler) {
    var treeElement = modal.find('.ckeditor-tree');
    var selectButton = modal.find('.modal-footer .btn-primary');

    var validateSelection = function(tree) {
      // jshint camelcase:false
      var selectedNodes = tree.get_selected(true);
      for (var i = 0; i < selectedNodes.size(); i++) {
        if (!handler.canSelect(selectedNodes[i])) {
          return false;
        }
      }
      return selectedNodes.size() > 0;
    };

    modal.on('shown.bs.modal', function(event) {
      // Open to the specified node only once. Preserve the tree state otherwise.
      var openToNodeId = handler.openToNodeId;
      if (typeof openToNodeId === 'string' && openToNodeId !== modal.data('openTo')) {
        modal.data('openTo', openToNodeId);
      } else {
        openToNodeId = false;
      }
      var tree = $.jstree.reference(treeElement);
      if (!tree) {
        // Initialize the tree and hook the event listeners.
        tree = treeElement.xtree({
          core: {
            multiple: treeElement.data('multiple') === 'true'
          }
        }).one('ready.jstree', function(event, data) {
          if (openToNodeId) {
            data.instance.openTo(openToNodeId);
          }
        }).on('changed.jstree', function(event, data) {
          selectButton.prop('disabled', !validateSelection(data.instance));
        }).on('dblclick', '.jstree-anchor', function(event) {
          if (validateSelection($.jstree.reference(this))) {
            selectButton.click();
          }
        });
      } else if (openToNodeId) {
        // jshint camelcase:false
        tree.deselect_all();
        tree.close_all();
        tree.openTo(openToNodeId);
      }
    });

    selectButton.click(function() {
      modal.modal('hide');
      var tree = $.jstree.reference(treeElement);
      // jshint camelcase:false
      handler.select(tree.get_selected(true));
    });

    handler.open = function(openToNodeId) {
      this.openToNodeId = openToNodeId;
      modal.modal();
    };

    return handler;
  };

  var getEntityReference = function(node) {
    var separatorIndex = node.id.indexOf(':');
    var nodeType = node.id.substr(0, separatorIndex);
    var nodeStringReference = node.id.substr(separatorIndex + 1);
    return XWiki.Model.resolve(nodeStringReference, XWiki.EntityType.byName(nodeType));
  };

  var getEntity = function(node) {
    return {
      title: node.text,
      reference: getEntityReference(node)
    };
  };

  var getEntityNodeId = function(entityReference) {
    return XWiki.EntityType.getName(entityReference.type) + ':' + XWiki.Model.serialize(entityReference);
  };

  var resolveDocumentReference = function(entityReference) {
    var documentReference = entityReference.extractReference(XWiki.EntityType.DOCUMENT);
    if (!documentReference) {
      var spaceReference = entityReference.extractReference(XWiki.EntityType.SPACE);
      if (!spaceReference) {
        var wikiReference = entityReference.extractReference(XWiki.EntityType.WIKI) ||
          new XWiki.WikiReference(XWiki.currentWiki);
        spaceReference = new XWiki.EntityReference('Main', XWiki.EntityType.SPACE, wikiReference);
      }
      documentReference = new XWiki.EntityReference('WebHome', XWiki.EntityType.DOCUMENT, spaceReference);
    }
    return documentReference;
  };

  var getNodeToOpenTo = function(targetEntityType, entityReference) {
    var targetEntityReference = entityReference.extractReference(targetEntityType);
    if (targetEntityReference) {
      // The target entity is a parent of the specified entity.
      return getEntityNodeId(targetEntityReference);

    // The target entity might be a child of the specified entity.
    } else if (targetEntityType === XWiki.EntityType.DOCUMENT) {
      return getEntityNodeId(resolveDocumentReference(entityReference));
    } else if (targetEntityType === XWiki.EntityType.ATTACHMENT) {
      return 'attachments:' + XWiki.Model.serialize(resolveDocumentReference(entityReference));
    }

    // Otherwise just try to open to the specified entity.
    return getEntityNodeId(entityReference);
  };

  var createEntityTreePicker = function(modal, handler) {
    var treePickerHandler = createTreePicker(modal, {
      canSelect: function(node) {
        return (node.data || {}).type === handler.entityType;
      },
      select: function(nodes) {
        handler.select(nodes.map(getEntity));
      }
    });
    handler.open = function(entityReference) {
      treePickerHandler.open(getNodeToOpenTo(XWiki.EntityType.byName(handler.entityType), entityReference));
    };
    return handler;
  };

  var convertEntityToResource = function(entity) {
    return $.extend({}, entity, {
      reference: $resource.convertEntityReferenceToResourceReference(entity.reference),
      entityReference: entity.reference
    });
  };

  var createResourcePicker = function(modal, handler) {
    var entityTreePickerHandler = createEntityTreePicker(modal, {
      entityType: $resource.types[handler.resourceType].entityType,
      select: function(entities) {
        handler.select(entities.map(convertEntityToResource));
      }
    });
    handler.open = function(resourceReference) {
      entityTreePickerHandler.open($resource.convertResourceReferenceToEntityReference(resourceReference));
    };
    return handler;
  };

  var treeURL = {
    doc: new XWiki.Document('DocumentTree', 'XWiki').getURL('get', $.param({
      outputSyntax: 'plain',
      language: $('html').attr('lang'),
      showAttachments: false,
      showTranslations: false,
      showWikis: true
    })),
    attach: new XWiki.Document('DocumentTree', 'XWiki').getURL('get', $.param({
      outputSyntax: 'plain',
      language: $('html').attr('lang'),
      showTranslations: false,
      showWikis: true
    }))
  };

  var registerResourcePicker = function(resourceType, title) {
    var modal = $modal.createModal({
      'class': 'entity-resource-picker-modal',
      title: title,
      content: createTreeElement({
        'data-url': treeURL[resourceType]
      }),
      acceptLabel: translations.get('select')
    });

    var picker = createResourcePicker(modal, {resourceType: resourceType});
    $resource.pickers[resourceType] = function(resourceReference) {
      var deferred = $.Deferred();
      picker.select = function(resources) {
        // We assume that only one resource can be selected.
        deferred.resolve(resources[0]);
      };
      picker.open(resourceReference);
      return deferred.promise();
    };
  };

  if (typeof $.fn.xtree === 'function') {
    registerResourcePicker('doc', translations.get('doc.title'));
    registerResourcePicker('attach', translations.get('attach.title'));
  }
});

/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('entityResourceSuggesterTranslationKeys', [], [
  'doc.placeholder',
  'attach.placeholder'
]);

define('entityResourceSuggester', [
  'jquery', 'resource', 'l10n!entityResourceSuggester'
], function($, $resource, translations) {
  'use strict';

  var search = function(query, input, deferred, entityType) {  
    $.post(new XWiki.Document('SuggestSolrService', 'XWiki').getURL('get'), {
      outputSyntax: 'plain',
      language: $('html').attr('lang'),
      query: query.join('\n'),
      input: input,
      nb: 8
    }).done(function(response) {
      if (response.documentElement) {
        var results = response.getElementsByTagName('rs');
        var suggestions = [];
        for (var i = 0; i < results.length; i++) {
          suggestions.push(convertSearchResultToResource(results.item(i), entityType));
        }
        deferred.resolve(suggestions);
      } else {
        deferred.resolve([]);
      }
    }).fail(function() {
      deferred.resolve([]);
    });
  };

  var convertSearchResultToResource = function(result, expectedEntityType) {
    var entityReference, location, title = result.childNodes[0].nodeValue;
    var entityType = result.getAttribute('type');
    if (entityType) {
      var serializedEntityReference = result.getAttribute('id');
      entityReference = XWiki.Model.resolve(serializedEntityReference, XWiki.EntityType.byName(entityType));
      location = result.getAttribute('info');
    } else {
      // Before XWiki 7.2
      var serializedDocumentReference = result.getAttribute('info');
      entityReference = XWiki.Model.resolve(serializedDocumentReference, XWiki.EntityType.DOCUMENT);
      // We show the document reference as location for both document and attachment search results.
      location = getLocation(entityReference);
      if (expectedEntityType === XWiki.EntityType.ATTACHMENT) {
        entityReference = new XWiki.AttachmentReference(title, entityReference);
      }
    }
    return {
      reference: $resource.convertEntityReferenceToResourceReference(entityReference),
      entityReference: entityReference,
      title: title,
      location: location
    };
  };

  // Before XWiki 7.2
  var getLocation = function(entityReference) {
    var references = entityReference.getReversedReferenceChain();
    if (references[0].type === XWiki.EntityType.WIKI && references[0].name === XWiki.currentWiki) {
      // Show the local reference if possible.
      references.shift();
    }
    return references.map(function(reference) {
      return reference.name;
    }).join(' / ');
  };

  var display = function(resource) {
    var suggestion = $(
      '<span class="resource-label"><span class="resource-icon"></span> </span>' +
      '<span class="resource-hint"></span>'
    );
    suggestion.find('.resource-icon').addClass($resource.types[resource.reference.type].icon);
    suggestion.first().append(document.createTextNode(resource.title)).next().html(resource.location);
    // Remove the home icon from the resource location displayed as hint because it distracts the user and it is
    // redundant. The home icon is useful to navigate to the home page but the resource suggestion hint is not used for
    // navigation so the home icon is not needed. We know that every path starts from home.
    suggestion.last().text(function(index, text) {
      return text.substr(0, 3) === ' / ' ? text.substr(3) : text;
    });
    return suggestion;
  };

  var advancedSearchPattern = /[+\-!(){}\[\]\^"~*?:\\]+/;

  $resource.types.doc.placeholder = translations.get('doc.placeholder');
  $resource.suggesters.doc = {
    retrieve: function(resourceReference) {
      var deferred = $.Deferred();
      var query = [
        'q=__INPUT__',
        'fq=type:DOCUMENT'
      ];
      var input = resourceReference.reference.trim();
      if (input) {
        query.push('qf=title^2 name');
        if (!advancedSearchPattern.test(input)) {
          query[0] = 'q=(__INPUT__)^2 __INPUT__*';
        }
      } else {
        // Recently modified pages.
        input = '*:*';
        query.push('sort=date desc');
      }
      search(query, input, deferred, XWiki.EntityType.DOCUMENT);
      return deferred.promise();
    },
    display: display
  };

  $resource.types.attach.placeholder = translations.get('attach.placeholder');
  $resource.suggesters.attach = {
    retrieve: function(resourceReference) {
      var deferred = $.Deferred();
      var query = [
        'q=__INPUT__',
        'fq=type:ATTACHMENT'
      ];
      var input = resourceReference.reference.trim();
      if (input) {
        query.push('qf=filename');
        if (!advancedSearchPattern.test(input)) {
          query[0] = 'q=(__INPUT__)^2 __INPUT__*';
        }
      } else {
        // Recently modified attachments.
        input = '*:*';
        query.push('sort=attdate_sort desc');
      }
      search(query, input, deferred, XWiki.EntityType.ATTACHMENT);
      return deferred.promise();
    },
    display: display
  };
});

/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('resourceTranslationKeys', [], [
  'attach.label',
  'attach.placeholder',
  'data.label',
  'doc.label',
  'doc.placeholder',
  'icon.label',
  'mailto.label',
  'mailto.placeholder',
  'path.label',
  'path.placeholder',
  'unc.label',
  'unc.placeholder',
  'unknown.label',
  'url.label',
  'url.placeholder',
  'user.label',
  'user.placeholder'
]);

define('resource', ['l10n!resource'], function(translations) {
  'use strict';

  var types = {
    attach: {
      label: translations.get('attach.label'),
      icon: 'glyphicon glyphicon-paperclip',
      placeholder: translations.get('attach.placeholder'),
      entityType: 'attachment'
    },
    data: {
      label: translations.get('data.label'),
      icon: 'glyphicon glyphicon-briefcase',
      placeholder: 'image/png;base64,AAAAAElFTkSuQmCC'
    },
    doc: {
      label: translations.get('doc.label'),
      icon: 'glyphicon glyphicon-file',
      placeholder: translations.get('doc.placeholder'),
      allowEmptyReference: true,
      entityType: 'document'
    },
    icon: {
      label: translations.get('icon.label'),
      icon: 'glyphicon glyphicon-flag',
      placeholder: 'help'
    },
    mailto: {
      label: translations.get('mailto.label'),
      icon: 'glyphicon glyphicon-envelope',
      placeholder: translations.get('mailto.placeholder')
    },
    path: {
      label: translations.get('path.label'),
      icon: 'glyphicon glyphicon-road',
      placeholder: translations.get('path.placeholder')
    },
    unc: {
      label: translations.get('unc.label'),
      icon: 'glyphicon glyphicon-hdd',
      placeholder: translations.get('unc.placeholder')
    },
    unknown: {
      label: translations.get('unknown.label'),
      icon: 'glyphicon glyphicon-question-sign',
      placeholder: ''
    },
    url: {
      label: translations.get('url.label'),
      icon: 'glyphicon glyphicon-globe',
      placeholder: translations.get('url.placeholder')
    },
    user: {
      label: translations.get('user.label'),
      icon: 'glyphicon glyphicon-user',
      placeholder: translations.get('user.placeholder')
    }
  };

  var entityTypeToResourceType = ['wiki', 'space', 'doc', 'attach'];
  var convertEntityReferenceToResourceReference = function(entityReference, baseEntityReference) {
    baseEntityReference = baseEntityReference || XWiki.currentDocument.getDocumentReference();
    return {
      type: entityTypeToResourceType[entityReference.type],
      // We know the target entity precisely (no ambiguity).
      typed: true,
      reference: getSerializedRelativeReference(entityReference, baseEntityReference)
    };
  };

  var getSerializedRelativeReference = function(entityReference, baseReference) {
    var relativeReference = entityReference.relativeTo(baseReference);
    var relativeRootReference = getRelativeRootReference(entityReference, relativeReference);
    if (relativeRootReference.parent && relativeRootReference.parent.type === relativeRootReference.type) {
      // The root of the relative reference is not complete: there are more components with the same type in the
      // absolute reference. Check if the entity type supports partial relative reference.
      if (entityReference.type === XWiki.EntityType.DOCUMENT) {
        // We need to prefix the serialized relative reference with the separator that corresponds to the root entity
        // type (SPACE). TODO: Don't hard-code the space separator.
        return '.' + XWiki.Model.serialize(relativeReference);
      } else {
        // We need to add the remaining components of the same type.
        do {
          relativeRootReference = relativeRootReference.parent;
        } while (relativeRootReference.parent && relativeRootReference.parent.type === relativeRootReference.type);
        var parent = relativeRootReference.parent;
        delete relativeRootReference.parent;
        var serializedRelativeReference = XWiki.Model.serialize(entityReference);
        relativeRootReference.parent = parent;
        return serializedRelativeReference;
      }
    } else {
      return XWiki.Model.serialize(relativeReference);
    }
  };

  var getRelativeRootReference = function(absoluteReference, relativeReference) {
    while (relativeReference.parent) {
      absoluteReference = absoluteReference.parent;
      relativeReference = relativeReference.parent;
    }
    return absoluteReference;
  };

  var convertResourceReferenceToEntityReference = function(resourceReference, baseEntityReference) {
    baseEntityReference = baseEntityReference || XWiki.currentDocument.getDocumentReference();
    var entityType = XWiki.EntityType.byName(types[resourceReference.type].entityType);
    return XWiki.Model.resolve(resourceReference.reference, entityType, baseEntityReference);
  };

  return {
    types: types,
    pickers: {},
    suggesters: {},
    displayers: {},
    convertEntityReferenceToResourceReference: convertEntityReferenceToResourceReference,
    convertResourceReferenceToEntityReference: convertResourceReferenceToEntityReference
  };
});

/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('resourcePickerTranslationKeys', [], [
  'attach.hint',
  'doc.hint'
]);

define('resourcePicker', [
  'jquery', 'resource', 'l10n!resourcePicker', 'bootstrap3-typeahead'
], function($, $resource, translations) {
  'use strict';

  var resourcePickerTemplate =
    '<div class="resourcePicker">' +
      '<div class="input-group">' +
        '<input type="text" class="resourceReference" />' +
        '<div class="input-group-btn">'+
          '<button type="button" class="resourceType btn btn-default">' +
            '<span class="icon">' +
          '</button>' +
          '<button type="button" class="btn btn-default dropdown-toggle" data-toggle="dropdown">' +
            '<span class="caret"></span>' +
          '</button>' +
          '<ul class="resourceTypes dropdown-menu dropdown-menu-right"></ul>' +
        '</div>' +
      '</div>' +
      '<div class="resourceDisplay"></div>' +
    '</div>';

  var createResourcePicker = function(element, options) {
    options = options || {};

    var resourcePicker = $(resourcePickerTemplate);
    element.on('selectResource', onSelectResource).hide().after(resourcePicker);

    // Compute the list of supported resource types.
    var resourceTypes = $(element).attr('data-resourceTypes') || options.resourceTypes || [];
    if (typeof resourceTypes === 'string') {
      resourceTypes = resourceTypes.split(/\s*,\s*/);
    }

    // Initialize the layout.
    if (resourceTypes.length === 0) {
      resourcePicker.find('.input-group').attr('class', 'input-wrapper').children('.input-group-btn').hide();
    } else if (resourceTypes.length === 1) {
      var resourceTypeButton = resourcePicker.find('.resourceType');
      // Move the resource type button at the end in order to have round borders, and hide the resource type drop down.
      resourceTypeButton.appendTo(resourceTypeButton.parent()).prevAll().hide();
    }

    // Populate the Resource Type drop down.
    resourceTypes.forEach(addResourceType, resourcePicker);

    addResourcePickerBehaviour(resourcePicker);

    // Initialize the resource selection.
    element.trigger('selectResource');

    return resourcePicker;
  };

  var addResourceType = function(resourceTypeId) {
    var resourceType = $resource.types[resourceTypeId] || {
      label: resourceTypeId,
      icon: 'glyphicon glyphicon-question-sign'
    };
    // There is at least one resource type. Make sure it's visible.
    this.find('.input-wrapper').attr('class', 'input-group').children('.input-group-btn').show();
    var resourceTypeList = this.find('.resourceTypes');
    if (resourceTypeList.children().length === 1) {
      // There are multiple resource types. We need to show the resource type list.
      var resourceTypeButton = resourceTypeList.next('.resourceType');
      resourceTypeButton.prependTo(resourceTypeButton.parent()).nextAll().show();
    }
    var resourceTypeElement = $('<li><a href="#"><span class="icon"></span></a></li>');
    resourceTypeElement.appendTo(resourceTypeList).find('a').attr({
      'data-id': resourceTypeId,
      'data-placeholder': resourceType.placeholder
    }).children('.icon').addClass(resourceType.icon)
      .after(document.createTextNode(' ' + resourceType.label));
  };

  var addResourcePickerBehaviour = function(resourcePicker) {
    // Resource type behaviour.
    resourcePicker.on('click', '.resourceTypes a', maybeChangeResourceType)
      .on('changeResourceType', clearOrRestoreSelectedReference)
      .on('changeResourceType', updateResourceSuggester);

    // Dedicated resource pickers.
    var resourceTypeButton = resourcePicker.find('button.resourceType');
    resourceTypeButton.click(openPicker);

    var resourceDisplay = resourcePicker.find('.resourceDisplay');
    var resourceReferenceInput = resourcePicker.find('input.resourceReference');
    resourceDisplay.on('click', '.remove', function(event) {
      resourceDisplay.addClass('hidden').empty();
      resourceReferenceInput.change();
    });
    resourceDisplay.on('click', 'a', function(event) {
      // Prevent the users from leaving the edit mode by mistake.
      event.preventDefault();
    });
    resourceReferenceInput.change(function(event) {
      // Update the original resource reference input if there's no resource displayed.
      if (resourceDisplay.hasClass('hidden')) {
        // We don't fire the selectResource event because we don't need to update the resource picker display.
        resourcePicker.prev('input').val(resourceTypeButton.val() + ':' + resourceReferenceInput.val());
      }
    });
    // Note that we don't register the event listener directly on the text input because all the keydown event listeners
    // are being removed when we destroy the typeahead (suggest). The good part is that we don't have to test if the
    // list of suggestions is visible because there is another listener below that does this and stops the event
    // propagation.
    resourcePicker.on('keydown', 'input.resourceReference', function(event) {
      // Trigger the change event when pressing the Enter key in order to update the original resource reference input,
      // because the Enter key is often used as a shortcut for submitting the typed value.
      if (event.which === 13) {
        $(event.target).change();
      }
    });
    // Make sure the original resource reference input is updated before its value is retrieved.
    resourcePicker.prev('input').on('beforeGetValue', function() {
      resourceReferenceInput.change();
    });
  };

  var maybeChangeResourceType = function(event) {
    event.preventDefault();
    var selectedResourceType = $(this);
    var resourcePicker = selectedResourceType.closest('.resourcePicker');
    var resourceTypeButton = resourcePicker.find('button.resourceType');
    var oldValue = resourceTypeButton.val();
    var newValue = selectedResourceType.attr('data-id');
    if (newValue === oldValue) {
      return;
    }
    resourceTypeButton.val(newValue);
    var disabled = typeof $resource.pickers[newValue] !== 'function';
    resourceTypeButton.prop('disabled', disabled);
    if (disabled) {
      resourceTypeButton.attr('title', selectedResourceType.text().trim());
      // Just in case someone checks the disabled attribute instead of the property.
      resourceTypeButton.attr('disabled', 'disabled');
    } else {
      resourceTypeButton.attr('title', translations.get(newValue + '.hint'));
    }
    resourceTypeButton.find('.icon').attr('class', selectedResourceType.find('.icon').attr('class'));
    resourcePicker.find('.resourceReference').attr('placeholder', selectedResourceType.attr('data-placeholder'));
    resourcePicker.trigger('changeResourceType', {
      oldValue: oldValue,
      newValue: newValue
    });
  };

  var openPicker = function(event) {
    var resourceTypeButton = $(this);
    var resourcePicker = resourceTypeButton.closest('.resourcePicker');
    var resourceReference = getResourceReference(resourcePicker.prev('input'));
    // Use the selected resource if it matches the currently selected resource type, otherwise use the resource
    // reference typed in the resource picker input.
    if (resourceReference.type !== resourceTypeButton.val()) {
      resourceReference = {
        type: resourceTypeButton.val(),
        reference: resourcePicker.find('input.resourceReference').val()
      };
    }
    $resource.pickers[resourceReference.type](resourceReference).done($.proxy(selectResource, resourcePicker));
  };

  var selectResource = function(resource) {
    // Update the original resource reference input and also the resource picker display because the resource was
    // selected from outside (e.g. from a dedicated picker or from a list of suggestions).
    this.prev('input').val(resource.reference.type + ':' + resource.reference.reference)
      .trigger('selectResource', resource);
  };

  var onSelectResource = function(event, resource) {
    resource = resource || {
      reference: getResourceReference($(event.target))
    };
    var resourcePicker = $(event.target).nextAll('div.resourcePicker').first();
    selectResourceType(resourcePicker, resource.reference.type);
    selectResourceReference(resourcePicker, resource.reference);
  };

  var getResourceReference = function(input) {
    var resourceReference = input.val();
    var separatorIndex = resourceReference.indexOf(':');
    var resourceType = resourceReference.substr(0, separatorIndex);
    resourceReference = resourceReference.substr(separatorIndex + 1);
    if (resourceType.length === 0) {
      if (resourceReference.length === 0) {
        // Use the first resource type available.
        resourceType = input.next('.resourcePicker').find('.resourceTypes a').first().attr('data-id');
      }
      resourceType = resourceType || 'unknown';
    }
    return {
      type: resourceType,
      reference: resourceReference,
      // Don't display any reference if the text input is empty.
      isNew: separatorIndex < 0 && resourceReference.length === 0
    };
  };

  var selectResourceType = function(resourcePicker, resourceTypeId) {
    var resourceTypeButton = resourcePicker.find('.resourceType');
    if (resourceTypeId === resourceTypeButton.val()) {
      // The specified resource type is already selected.
      return;
    }
    var resourceTypeAnchor = resourcePicker.find('a').filter(function() {
      return $(this).attr('data-id') === resourceTypeId;
    });
    if (resourceTypeAnchor.length === 0) {
      // Unsupported resource type. We need to add it to the list before selecting it.
      addResourceType.call(resourcePicker, resourceTypeId);
      resourceTypeAnchor = resourcePicker.find('a').filter(function() {
        return $(this).attr('data-id') === resourceTypeId;
      });
    }
    resourceTypeAnchor.click();
  };

  var selectResourceReference = function(resourcePicker, resourceReference) {
    var resourceReferenceInput = resourcePicker.find('.resourceReference');
    var resourceDisplayContainer = resourcePicker.find('.resourceDisplay');
    var displayer = $resource.displayers[resourceReference.type];
    if (typeof displayer === 'function' && !resourceReference.isNew && (resourceReference.reference.length > 0 ||
        $resource.types[resourceReference.type].allowEmptyReference)) {
      resourceReferenceInput.val('');
      resourceDisplayContainer.empty().addClass('loading').removeClass('hidden');
      displayer(resourceReference).done(function(resourceDisplay) {
        // Empty the container before appending the resource display because we don't cancel the previous (unfinished)
        // display requests. The displayer could handle this itself but we would need to pass additional information
        // (something to identify the resource picker that made the display request).
        resourceDisplayContainer.empty().removeClass('loading').attr({
          'data-resourceType': resourceReference.type,
          'data-resourceReference': resourceReference.reference
        }).append(resourceDisplay).removeClass('hidden');
      }).fail(function() {
        resourceReferenceInput.val(resourceReference.reference);
        resourceDisplayContainer.addClass('hidden');
      });
    } else {
      resourceDisplayContainer.addClass('hidden').empty();
      resourceReferenceInput.val(resourceReference.reference);
    }
  };

  var clearOrRestoreSelectedReference = function(event, data) {
    var resourcePicker = $(this);
    var resourceDisplayContainer = resourcePicker.find('.resourceDisplay');
    if (resourceDisplayContainer.attr('data-resourceType') === data.newValue &&
        !resourceDisplayContainer.is(':empty')) {
      // Restore the selected resource.
      resourceDisplayContainer.removeClass('hidden');
      resourcePicker.prev('input').val(resourceDisplayContainer.attr('data-resourceType') + ':' +
        resourceDisplayContainer.attr('data-resourceReference'));
    } else {
      // Clear the selected resource.
      // Hide the resource display but keep its content because we want to restore it later.
      resourceDisplayContainer.addClass('hidden');
      // Update the hidden reference input based on what is available on the resource picker input.
      resourcePicker.find('input.resourceReference').change();
    }
  };

  var updateResourceSuggester = function(event, data) {
    var resourcePicker = $(this);
    var resourceReferenceInput = resourcePicker.find('input.resourceReference');
    resourceReferenceInput.typeahead('destroy');
    var suggester = $resource.suggesters[data.newValue];
    if (suggester) {
      resourceReferenceInput.keydown(stopPropagationIfShowingSuggestions).typeahead({
        afterSelect: $.proxy(selectResource, resourcePicker),
        delay: 500,
        displayText: function(resource) {
          // HACK: The string returned by this function is passed to the highlighter where we need to access all the
          // resource properties in order to be able to display the resource suggestion.
          // jshint -W053
          var reference = new String(resource.reference.reference);
          reference.__resource = resource;
          return reference;
        },
        highlighter: !suggester.display || function(resourceReference) {
          return suggester.display(resourceReference.__resource);
        },
        matcher: function(resource) {
          // By default, we assume the suggestions are already filtered.
          return true;
        },
        minLength: 0,
        sorter: function(resources) {
          // By default, we assume the suggestions are already sorted.
          return resources;
        },
        source: function(resourceReference, callback) {
          suggester.retrieve({
            type: data.newValue,
            reference: resourceReference
          }).done(callback);
        }
      });
    } else {
      resourceReferenceInput.off('keydown', stopPropagationIfShowingSuggestions);
    }
  };

  var stopPropagationIfShowingSuggestions = function(event) {
    if ((event.which === 27 /* ESC */ || event.which === 13 /* Enter */) &&
        $(this).next('.dropdown-menu').filter(':visible').length > 0) {
      // Stop the event propagation if the key is handled by the suggester.
      // Enter key is used to select a suggestion. Escape key is used to hide the suggestions.
      event.stopPropagation();
    }
  };

  $.fn.resourcePicker = function(options) {
    return this.each(function() {
      if (!$(this).next().is('div.resourcePicker')) {
        createResourcePicker($(this), options);
      }
    });
  };
});

define('resourcePicker.bundle', [
  'resourcePicker',
  'entityResourcePicker',
  'entityResourceSuggester',
  'entityResourceDisplayer'
], function() {});

