/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.rendering.wikimacro.macro.wikimacroparameter;

import java.io.StringReader;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;

import org.xwiki.component.annotation.Component;
import org.xwiki.context.Execution;
import org.xwiki.rendering.block.Block;
import org.xwiki.rendering.block.MetaDataBlock;
import org.xwiki.rendering.block.XDOM;
import org.xwiki.rendering.listener.MetaData;
import org.xwiki.rendering.macro.AbstractMacro;
import org.xwiki.rendering.macro.MacroContentParser;
import org.xwiki.rendering.macro.MacroExecutionException;
import org.xwiki.rendering.macro.descriptor.DefaultContentDescriptor;
import org.xwiki.rendering.macro.descriptor.MacroDescriptor;
import org.xwiki.rendering.macro.descriptor.ParameterDescriptor;
import org.xwiki.rendering.macro.wikibridge.WikiMacroParameters;
import org.xwiki.rendering.parser.ParseException;
import org.xwiki.rendering.parser.Parser;
import org.xwiki.rendering.transformation.MacroTransformationContext;
import org.xwiki.stability.Unstable;

import com.xpn.xwiki.XWikiContext;

/**
 * A macro that should only be used inside a WikiMacro to show the content of the wikimacro and make it inline editable.
 * @since 11.5RC1
 * @version $Id: 26d8e4ae3d36346372797ac94ebe4631a6311834 $
 */
@Component
@Named("wikimacroparameter")
@Singleton
@Unstable
public class WikiMacroParameterMacro extends AbstractMacro<WikiMacroParameterMacroParameters>
{
    /**
     * The description of the macro.
     */
    private static final String DESCRIPTION = "Display editable parameter of a wikimacro.";

    private static final String WIKIMACRO_PARAMETERS = "params";

    private static final String WIKIMACRO_DESCRIPTOR = "descriptor";

    /**
     * The {@link Execution} component used for accessing XWikiContext.
     */
    @Inject
    private Execution execution;

    @Inject
    private MacroContentParser contentParser;

    @Inject
    @Named("plain/1.0")
    private Parser plainTextParser;

    /**
     * Default constructor.
     */
    public WikiMacroParameterMacro()
    {
        super("WikiMacro Parameter", DESCRIPTION, new DefaultContentDescriptor(false),
            WikiMacroParameterMacroParameters.class);
        setDefaultCategory(DEFAULT_CATEGORY_DEVELOPMENT);
    }

    @Override
    public boolean supportsInlineMode()
    {
        return true;
    }

    /**
     * Utility method for accessing XWikiContext.
     *
     * @return the XWikiContext.
     */
    private XWikiContext getContext()
    {
        return (XWikiContext) this.execution.getContext().getProperty("xwikicontext");
    }

    private String extractMacroParameter(Map<String, Object> macroInfo, String parameterName)
    {
        String parameterContent = null;
        if (macroInfo != null && macroInfo.containsKey(WIKIMACRO_PARAMETERS)) {
            WikiMacroParameters parameters = (WikiMacroParameters) macroInfo.get(WIKIMACRO_PARAMETERS);
            if (parameters != null) {
                parameterContent = (String) parameters.get(parameterName);
            }
        }

        return parameterContent;
    }

    private MetaData getNonGeneratedParameterMetaData(Map<String, Object> macroInfo, String parameterName)
    {
        MetaData result;

        if (macroInfo != null && macroInfo.containsKey(WIKIMACRO_DESCRIPTOR)) {
            MacroDescriptor macroDescriptor = (MacroDescriptor) macroInfo.get(WIKIMACRO_DESCRIPTOR);
            result = AbstractMacro.getNonGeneratedContentMetaData(macroDescriptor.getParameterDescriptorMap(),
                parameterName);
        } else {
            result = AbstractMacro.getNonGeneratedContentMetaData(null);
        }

        return result;
    }

    private MacroDescriptor getMacroDescriptor(Map<String, Object> macroInfo)
    {
        MacroDescriptor result = null;
        if (macroInfo != null && macroInfo.containsKey(WIKIMACRO_DESCRIPTOR)) {
            result = (MacroDescriptor) macroInfo.get(WIKIMACRO_DESCRIPTOR);
        }
        return result;
    }

    private XDOM parseParameterValue(String macroParameterContent, String macroParameterName,
        Map<String, Object> macroInfo, MacroTransformationContext context)
        throws MacroExecutionException
    {
        MacroDescriptor macroDescriptor = getMacroDescriptor(macroInfo);
        ParameterDescriptor parameterDescriptor = null;
        if (macroDescriptor != null) {
            parameterDescriptor = macroDescriptor.getParameterDescriptorMap().get(macroParameterName);
        }

        if (parameterDescriptor == null || !parameterDescriptor.getParameterType().equals(Block.LIST_BLOCK_TYPE)) {
            try {
                return this.plainTextParser.parse(new StringReader(macroParameterContent));
            } catch (ParseException e) {
                throw new MacroExecutionException("Error while parsing the macro parameter content in plain.", e);
            }
        } else {
            return this.contentParser.parse(macroParameterContent, context, true, context.isInline());
        }
    }

    @Override
    public List<Block> execute(WikiMacroParameterMacroParameters parameters, String content,
        MacroTransformationContext context)
        throws MacroExecutionException
    {
        if (parameters == null) {
            throw new MacroExecutionException("The parameter is mandatory.");
        }
        List<Block> result = Collections.emptyList();
        Map<String, Object> macroInfo = (Map) getContext().get("macro");
        String parameterValue = extractMacroParameter(macroInfo, parameters.getName());
        if (parameterValue != null) {
            MetaData nonGeneratedContentMetaData = this.getNonGeneratedParameterMetaData(macroInfo,
                parameters.getName());
            nonGeneratedContentMetaData.addMetaData("wikimacrocontent", "true");
            XDOM parse = this.parseParameterValue(parameterValue, parameters.getName(), macroInfo, context);
            result = Collections.singletonList(new MetaDataBlock(parse.getChildren(),
                nonGeneratedContentMetaData));
        }

        return result;
    }
}
