/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.gwt.dom.client.internal.ie;

import org.xwiki.gwt.dom.client.DOMUtils;
import org.xwiki.gwt.dom.client.Document;
import org.xwiki.gwt.dom.client.Element;
import org.xwiki.gwt.dom.client.Style;

import com.google.gwt.core.client.JsArrayString;
import com.google.gwt.dom.client.Node;

/**
 * Contains methods from {@link DOMUtils} that require a different implementation in Internet Explorer.
 * 
 * @version $Id$
 */
public class IEDOMUtils extends DOMUtils
{
    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#getComputedStyleProperty(Element, String)
     */
    public native String getComputedStyleProperty(Element el, String propertyName)
    /*-{
        // We force it to be a string because we treat it as a string in the java code.
        return '' + el.currentStyle[propertyName];
    }-*/;

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#importNode(Document, Node, boolean)
     */
    public Node importNode(Document doc, Node externalNode, boolean deep)
    {
        switch (externalNode.getNodeType()) {
            case Node.TEXT_NODE:
                return doc.createTextNode(externalNode.getNodeValue());
            case Node.ELEMENT_NODE:
                Element externalElement = Element.as(externalNode);
                Element internalElement = doc.xCreateElement(externalElement.getTagName());
                JsArrayString attrNames = getAttributeNames(externalElement);
                for (int i = 0; i < attrNames.length(); i++) {
                    String attrName = attrNames.get(i);
                    internalElement.setAttribute(attrName, externalElement.getAttribute(attrName));
                }
                if (deep) {
                    // TODO
                }
                return internalElement;
            default:
                throw new IllegalArgumentException("Cannot import node of type " + externalNode.getNodeType() + "!");
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#getAttributeNames(Element)
     */
    public native JsArrayString getAttributeNames(Element element)
    /*-{
        var attrNames = [];
        for(var i = 0; i < element.attributes.length; i++){
            if(element.attributes[i].specified) {
                attrNames.push(element.attributes[i].nodeName);
            }
        }
        return attrNames;
    }-*/;

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#setInnerHTML(Element, String)
     * @see http://code.google.com/p/google-web-toolkit/issues/detail?id=3146
     */
    public void setInnerHTML(Element element, String html)
    {
        element.setInnerHTML("<span>iesucks</span>" + html);
        element.removeChild(element.getFirstChild());
    }

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#getAttribute(Element, String)
     */
    public native String getAttribute(Element element, String name)
    /*-{
        // it seems that IE cannot return the style attribute value with getAttributeNode("style").nodeValue
        // http://www.quirksmode.org/dom/w3c_core.html
        if (name == "style") {
            return element.style.cssText;
        }
        // the class, for example, is not returned on getAttribute("class") but getAttribute("className") or 
        // getAttributeNode("class").nodeValue so make this the same for all attributes 
        var attrNode = element.getAttributeNode(name);
        if (attrNode) {
            // make sure we don't print "undefined" and always return a String
            return (attrNode.nodeValue || '' ) + ''; 
        }
        return '';
    }-*/;

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#setAttribute(Element, String, String)
     */
    public void setAttribute(Element element, String name, String value)
    {
        // In IE we can't set the style attribute using the standard setAttribute method from the DOM API.
        if (Style.STYLE_ATTRIBUTE.equalsIgnoreCase(name)) {
            element.getStyle().setProperty("cssText", value);
        } else if ("class".equalsIgnoreCase(name)) {
            element.setClassName(value);
        } else {
            element.setAttribute(name, value);
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#getInnerText(Element)
     */
    public native String getInnerText(Element element)
    /*-{
        return element.innerText;
    }-*/;

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#hasAttribute(Element, String)
     */
    public native boolean hasAttribute(Element element, String attrName)
    /*-{
        var attrNode = element.getAttributeNode(attrName);
        return attrNode && attrNode.specified;
    }-*/;

    /**
     * {@inheritDoc}
     * 
     * @see DOMUtils#hasAttributes(Element)
     */
    public boolean hasAttributes(Element element)
    {
        return getAttributeNames(element).length() > 0;
    }
}
