/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.gwt.user.client.ui;

import org.xwiki.gwt.dom.client.DOMUtils;
import org.xwiki.gwt.dom.client.Element;

import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.AbstractImagePrototype.ImagePrototypeElement;

/**
 * Improves the default menu item widget provided by GWT:
 * <ul>
 * <li>Allows us to disable and re-enable it.</li>
 * </ul>
 * 
 * @version $Id: MenuItem.java 24888 2009-11-06 02:16:19Z sdumitriu $
 */
public class MenuItem extends com.google.gwt.user.client.ui.MenuItem implements SourcesMenuEvents
{
    /**
     * The style name suffix attached to disabled menu items.
     */
    private static final String DEPENDENT_STYLENAME_DISABLED_ITEM = "disabled";

    /**
     * The style name suffix attached to selected menu items.
     */
    private static final String DEPENDENT_STYLENAME_SELECTED_ITEM = "selected";

    /**
     * The command associated with a disabled menu item.
     */
    private static final Command DISABLED_COMMAND = new Command()
    {
        public void execute()
        {
            // do nothing
        }
    };

    /**
     * Flag indicating the enabled state of this menu item.
     */
    private boolean enabled = true;

    /**
     * The menu item icon.
     */
    private Element icon;

    /**
     * The menu item label.
     */
    private HTML label;

    /**
     * The list of registered listeners for menu events generated by this menu item.
     */
    private final MenuListenerCollection menuListeners = new MenuListenerCollection();

    /**
     * Constructs a new menu item that fires a command when it is selected.
     * 
     * @param text the item's text
     * @param cmd the command to be fired when it is selected
     */
    public MenuItem(String text, Command cmd)
    {
        super(text, cmd);
    }

    /**
     * Constructs a new menu item that fires a command when it is selected.
     * 
     * @param text the item's text
     * @param asHTML <code>true</code> to treat the specified text as html
     * @param cmd the command to be fired when it is selected
     */
    public MenuItem(String text, boolean asHTML, Command cmd)
    {
        super(text, asHTML, cmd);
    }

    /**
     * Constructs a new menu item that cascades to a sub-menu when it is selected.
     * 
     * @param text the item's text
     * @param subMenu the sub-menu to be displayed when it is selected
     */
    public MenuItem(String text, MenuBar subMenu)
    {
        super(text, subMenu);
    }

    /**
     * Constructs a new menu item that cascades to a sub-menu when it is selected.
     * 
     * @param text the item's text
     * @param asHTML <code>true</code> to treat the specified text as html
     * @param subMenu the sub-menu to be displayed when it is selected
     */
    public MenuItem(String text, boolean asHTML, MenuBar subMenu)
    {
        super(text, asHTML, subMenu);
    }

    /**
     * Disables or enables this menu item.
     * 
     * @param enabled {@code true} to enable this menu item, {@code false} to disable it
     */
    public void setEnabled(boolean enabled)
    {
        if (this.enabled != enabled) {
            this.enabled = enabled;
            if (enabled) {
                removeStyleDependentName(DEPENDENT_STYLENAME_DISABLED_ITEM);
            } else {
                addStyleDependentName(DEPENDENT_STYLENAME_DISABLED_ITEM);
            }
        }
    }

    /**
     * @return {@code true} if this menu item is enabled, {@code false} otherwise
     */
    public boolean isEnabled()
    {
        return enabled;
    }

    /**
     * Sets the icon of this menu item using the element created from an AbstractImagePrototype.
     * 
     * @param icon the icon to be displayed on the left of this menu item
     */
    public void setIcon(ImagePrototypeElement icon)
    {
        setIcon((Element) icon.cast());
    }

    /**
     * Sets the icon of this menu item.
     * 
     * @param icon the icon to be displayed on the left of this menu item
     */
    public void setIcon(Element icon)
    {
        if (this.icon != null) {
            DOMUtils.getInstance().detach(this.icon);
        }
        this.icon = icon;
        icon.setClassName("gwt-MenuItemIcon");
        DOMUtils.getInstance().insertAt(getElement(), icon, 0);
    }

    /**
     * @return the icon of this menu item
     */
    public Element getIcon()
    {
        return icon;
    }

    /**
     * Creates the label of this menu item.
     */
    private void createLabel()
    {
        label = new HTML();
        label.addStyleName("gwt-MenuItemLabel");
        getElement().appendChild(label.getElement());
    }

    /**
     * {@inheritDoc}
     * 
     * @see com.google.gwt.user.client.ui.MenuItem#setHTML(String)
     */
    public void setHTML(String html)
    {
        if (label == null) {
            createLabel();
        }
        label.setHTML(html);
    }

    /**
     * {@inheritDoc}
     * 
     * @see com.google.gwt.user.client.ui.MenuItem#getHTML()
     */
    public String getHTML()
    {
        if (label == null) {
            return "";
        } else {
            return label.getHTML();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see com.google.gwt.user.client.ui.MenuItem#setText(String)
     */
    public void setText(String text)
    {
        if (label == null) {
            createLabel();
        }
        label.setText(text);
    }

    /**
     * {@inheritDoc}
     * 
     * @see com.google.gwt.user.client.ui.MenuItem#getText()
     */
    public String getText()
    {
        if (label == null) {
            return "";
        } else {
            return label.getText();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see com.google.gwt.user.client.ui.MenuItem#getCommand()
     */
    public Command getCommand()
    {
        if (enabled) {
            return super.getCommand();
        } else {
            return DISABLED_COMMAND;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see com.google.gwt.user.client.ui.MenuItem#getSubMenu()
     */
    public com.google.gwt.user.client.ui.MenuBar getSubMenu()
    {
        if (enabled) {
            return super.getSubMenu();
        } else {
            return null;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see com.google.gwt.user.client.ui.MenuItem#addStyleDependentName(String)
     */
    public void addStyleDependentName(String styleSuffix)
    {
        super.addStyleDependentName(styleSuffix);
        // This is an ugly hack required because com.google.gwt.user.client.ui.MenuItem#setSelectionStyle(boolean) has
        // only package-level access.
        if (DEPENDENT_STYLENAME_SELECTED_ITEM.equals(styleSuffix)) {
            menuListeners.fireMenuItemSelected(this);
        }
    }

    /**
     * Visually marks this menu item as selected or unselected based on the given argument.<br/>
     * NOTE: This is just a hack required because #setSelectionStyle(boolean) is private.
     * 
     * @param selected {@code true} to select this menu item, {@code false} to remove the selection
     */
    public native void xSetSelectionStyle(boolean selected)
    /*-{
        this.@com.google.gwt.user.client.ui.MenuItem::setSelectionStyle(Z)(selected);
    }-*/;

    /**
     * {@inheritDoc}
     * 
     * @see SourcesMenuEvents#addMenuListener(MenuListener)
     */
    public void addMenuListener(MenuListener listener)
    {
        menuListeners.add(listener);
    }

    /**
     * {@inheritDoc}
     * 
     * @see SourcesMenuEvents#removeMenuListener(MenuListener)
     */
    public void removeMenuListener(MenuListener listener)
    {
        menuListeners.remove(listener);
    }
}
