/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.gwt.wysiwyg.client.widget.wizard.util;

import java.util.ArrayList;
import java.util.List;

import org.xwiki.gwt.user.client.StringUtils;
import org.xwiki.gwt.user.client.ui.wizard.WizardStep;
import org.xwiki.gwt.wysiwyg.client.Strings;
import org.xwiki.gwt.wysiwyg.client.wiki.EntityConfig;
import org.xwiki.gwt.wysiwyg.client.wiki.EntityReference;
import org.xwiki.gwt.wysiwyg.client.wiki.WikiServiceAsync;

/**
 * Generic wizard step that allows the user to select an attachment to link to by aggregating a current page selector
 * step and an "all pages" selector step.
 * 
 * @param <T> the type of entity link configuration managed by this wizard step
 * @version $Id: AttachmentSelectorAggregatorWizardStep.java 28844 2010-05-12 13:22:39Z mflorea $
 */
public class AttachmentSelectorAggregatorWizardStep<T extends EntityConfig> extends
    AbstractEntitySelectorAggregatorWizardStep<T>
{
    /**
     * Flag indicating if the selection is limited to the current page or not.
     */
    private final boolean selectionLimitedToCurrentPage;

    /**
     * This wizard step's title.
     */
    private String stepTitle;

    /**
     * The wizard step used to select attachments only from the current page.
     */
    private WizardStep currentPageSelector;

    /**
     * The wizard step used to select attachments from any page.
     */
    private WizardStep allPagesSelector;

    /**
     * Creates a new attachment selector wizard step.
     * 
     * @param selectionLimitedToCurrentPage {@code true} to limit the selection to the current page, {@code false}
     *            otherwise
     * @param wikiService the service to be used to access the attachments
     */
    public AttachmentSelectorAggregatorWizardStep(boolean selectionLimitedToCurrentPage, WikiServiceAsync wikiService)
    {
        super(wikiService);
        this.selectionLimitedToCurrentPage = selectionLimitedToCurrentPage;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String getRequiredStep()
    {
        EntityReference origin = getData().getOrigin();
        EntityReference destination = getData().getDestination();
        if (selectionLimitedToCurrentPage || StringUtils.isEmpty(getData().getData().getReference())
            || samePage(origin, destination)) {
            // Selection limited to current page, or no reference specified or the targeted attachment is attached to
            // the origin page.
            return Strings.INSTANCE.selectorSelectFromCurrentPage();
        } else {
            return Strings.INSTANCE.selectorSelectFromAllPages();
        }
    }

    /**
     * Utility method for checking if two entity references point to the same page.
     * 
     * @param origin the origin entity reference
     * @param destination the destination entity reference
     * @return {@code true} if the given entity references point to the same page, {@code false} otherwise
     */
    private boolean samePage(EntityReference origin, EntityReference destination)
    {
        return StringUtils.areEqual(origin.getPageName(), destination.getPageName())
            && StringUtils.areEqual(origin.getSpaceName(), destination.getSpaceName())
            && StringUtils.areEqual(origin.getWikiName(), destination.getWikiName());
    }

    /**
     * {@inheritDoc}
     * 
     * @see AbstractEntitySelectorAggregatorWizardStep#getStepInstance(String)
     */
    @Override
    protected WizardStep getStepInstance(String name)
    {
        if (name.equals(Strings.INSTANCE.selectorSelectFromCurrentPage())) {
            return currentPageSelector;
        } else if (name.equals(Strings.INSTANCE.selectorSelectFromAllPages())) {
            return allPagesSelector;
        }
        return null;
    }

    /**
     * Sets the wizard step to be used for selecting an attachment from the current page.
     * 
     * @param currentPageSelector the new current page attachment selector
     */
    public void setCurrentPageSelector(WizardStep currentPageSelector)
    {
        this.currentPageSelector = currentPageSelector;
    }

    /**
     * Sets the wizard step to be used for selecting an attachment from all pages.
     * 
     * @param allPagesSelector the new all pages attachment selector
     */
    public void setAllPagesSelector(WizardStep allPagesSelector)
    {
        this.allPagesSelector = allPagesSelector;
    }

    /**
     * {@inheritDoc}
     * 
     * @see AbstractEntitySelectorAggregatorWizardStep#getStepNames()
     */
    @Override
    protected List<String> getStepNames()
    {
        List<String> stepNames = new ArrayList<String>();
        stepNames.add(Strings.INSTANCE.selectorSelectFromCurrentPage());
        if (!selectionLimitedToCurrentPage) {
            stepNames.add(Strings.INSTANCE.selectorSelectFromAllPages());
        }
        return stepNames;
    }

    /**
     * {@inheritDoc}
     * 
     * @see AbstractEntitySelectorAggregatorWizardStep#getStepTitle()
     */
    public String getStepTitle()
    {
        return stepTitle;
    }

    /**
     * Sets the step title.
     * 
     * @param stepTitle the new step title
     */
    public void setStepTitle(String stepTitle)
    {
        this.stepTitle = stepTitle;
    }
}
