/**
 * <h1>AWS Cloud Development Kit Library</h1>
 * <p>
 * The AWS CDK construct library provides APIs to define your CDK application and add
 * CDK constructs to the application.
 * <p>
 * <h2>Usage</h2>
 * <p>
 * <h3>Upgrade from CDK 1.x</h3>
 * <p>
 * When upgrading from CDK 1.x, remove all dependencies to individual CDK packages
 * from your dependencies file and follow the rest of the sections.
 * <p>
 * <h3>Installation</h3>
 * <p>
 * To use this package, you need to declare this package and the <code>constructs</code> package as
 * dependencies.
 * <p>
 * According to the kind of project you are developing:
 * <p>
 * For projects that are CDK libraries in NPM, declare them both under the <code>devDependencies</code> <strong>and</strong> <code>peerDependencies</code> sections.
 * To make sure your library is compatible with the widest range of CDK versions: pick the minimum <code>aws-cdk-lib</code> version
 * that your library requires; declare a range dependency with a caret on that version in peerDependencies, and declare a
 * point version dependency on that version in devDependencies.
 * <p>
 * For example, let's say the minimum version your library needs is <code>2.38.0</code>. Your <code>package.json</code> should look like this:
 * <p>
 * <blockquote><pre>
 * {
 *   "peerDependencies": {
 *     "aws-cdk-lib": "^2.38.0",
 *     "constructs": "^10.0.0"
 *   },
 *   "devDependencies": {
 *     /* Install the oldest version for testing so we don't accidentally use features from a newer version than we declare *&#47;
 *     "aws-cdk-lib": "2.38.0"
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * For CDK apps, declare them under the <code>dependencies</code> section. Use a caret so you always get the latest version:
 * <p>
 * <blockquote><pre>
 * {
 *   "dependencies": {
 *     "aws-cdk-lib": "^2.38.0",
 *     "constructs": "^10.0.0"
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Use in your code</h3>
 * <p>
 * <h4>Classic import</h4>
 * <p>
 * You can use a classic import to get access to each service namespaces:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.App;
 * import software.amazon.awscdk.services.s3.*;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "TestStack");
 * 
 * new Bucket(stack, "TestBucket");
 * </pre></blockquote>
 * <p>
 * <h4>Barrel import</h4>
 * <p>
 * Alternatively, you can use "barrel" imports:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.App;
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.services.s3.Bucket;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "TestStack");
 * 
 * new Bucket(stack, "TestBucket");
 * </pre></blockquote>
 * <p>
 * <!--BEGIN CORE DOCUMENTATION-->
 * <p>
 * <h2>Stacks and Stages</h2>
 * <p>
 * A <code>Stack</code> is the smallest physical unit of deployment, and maps directly onto
 * a CloudFormation Stack. You define a Stack by defining a subclass of <code>Stack</code>
 * -- let's call it <code>MyStack</code> -- and instantiating the constructs that make up
 * your application in <code>MyStack</code>'s constructor. You then instantiate this stack
 * one or more times to define different instances of your application. For example,
 * you can instantiate it once using few and cheap EC2 instances for testing,
 * and once again using more and bigger EC2 instances for production.
 * <p>
 * When your application grows, you may decide that it makes more sense to split it
 * out across multiple <code>Stack</code> classes. This can happen for a number of reasons:
 * <p>
 * <ul>
 * <li>You could be starting to reach the maximum number of resources allowed in a single
 * stack (this is currently 500).</li>
 * <li>You could decide you want to separate out stateful resources and stateless resources
 * into separate stacks, so that it becomes easy to tear down and recreate the stacks
 * that don't have stateful resources.</li>
 * <li>There could be a single stack with resources (like a VPC) that are shared
 * between multiple instances of other stacks containing your applications.</li>
 * </ul>
 * <p>
 * As soon as your conceptual application starts to encompass multiple stacks,
 * it is convenient to wrap them in another construct that represents your
 * logical application. You can then treat that new unit the same way you used
 * to be able to treat a single stack: by instantiating it multiple times
 * for different instances of your application.
 * <p>
 * You can define a custom subclass of <code>Stage</code>, holding one or more
 * <code>Stack</code>s, to represent a single logical instance of your application.
 * <p>
 * As a final note: <code>Stack</code>s are not a unit of reuse. They describe physical
 * deployment layouts, and as such are best left to application builders to
 * organize their deployments with. If you want to vend a reusable construct,
 * define it as a subclasses of <code>Construct</code>: the consumers of your construct
 * will decide where to place it in their own stacks.
 * <p>
 * <h2>Stack Synthesizers</h2>
 * <p>
 * Each Stack has a <em>synthesizer</em>, an object that determines how and where
 * the Stack should be synthesized and deployed. The synthesizer controls
 * aspects like:
 * <p>
 * <ul>
 * <li>How does the stack reference assets? (Either through CloudFormation
 * parameters the CLI supplies, or because the Stack knows a predefined
 * location where assets will be uploaded).</li>
 * <li>What roles are used to deploy the stack? These can be bootstrapped
 * roles, roles created in some other way, or just the CLI's current
 * credentials.</li>
 * </ul>
 * <p>
 * The following synthesizers are available:
 * <p>
 * <ul>
 * <li><code>DefaultStackSynthesizer</code>: recommended. Uses predefined asset locations and
 * roles created by the modern bootstrap template. Access control is done by
 * controlling who can assume the deploy role. This is the default stack
 * synthesizer in CDKv2.</li>
 * <li><code>LegacyStackSynthesizer</code>: Uses CloudFormation parameters to communicate
 * asset locations, and the CLI's current permissions to deploy stacks. This
 * is the default stack synthesizer in CDKv1.</li>
 * <li><code>CliCredentialsStackSynthesizer</code>: Uses predefined asset locations, and the
 * CLI's current permissions.</li>
 * </ul>
 * <p>
 * Each of these synthesizers takes configuration arguments. To configure
 * a stack with a synthesizer, pass it as one of its properties:
 * <p>
 * <blockquote><pre>
 * MyStack.Builder.create(app, "MyStack")
 *         .synthesizer(DefaultStackSynthesizer.Builder.create()
 *                 .fileAssetsBucketName("my-orgs-asset-bucket")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more information on bootstrapping accounts and customizing synthesis,
 * see <a href="https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html">Bootstrapping in the CDK Developer Guide</a>.
 * <p>
 * <h2>Nested Stacks</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Nested stacks</a> are stacks created as part of other stacks. You create a nested stack within another stack by using the <code>NestedStack</code> construct.
 * <p>
 * As your infrastructure grows, common patterns can emerge in which you declare the same components in multiple templates. You can separate out these common components and create dedicated templates for them. Then use the resource in your template to reference other templates, creating nested stacks.
 * <p>
 * For example, assume that you have a load balancer configuration that you use for most of your stacks. Instead of copying and pasting the same configurations into your templates, you can create a dedicated template for the load balancer. Then, you just use the resource to reference that template from within other templates.
 * <p>
 * The following example will define a single top-level stack that contains two nested stacks: each one with a single Amazon S3 bucket:
 * <p>
 * <blockquote><pre>
 * public class MyNestedStack extends NestedStack {
 *     public MyNestedStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyNestedStack(Construct scope, String id, NestedStackProps props) {
 *         super(scope, id, props);
 * 
 *         new Bucket(this, "NestedBucket");
 *     }
 * }
 * 
 * public class MyParentStack extends Stack {
 *     public MyParentStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyParentStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         new MyNestedStack(this, "Nested1");
 *         new MyNestedStack(this, "Nested2");
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Resources references across nested/parent boundaries (even with multiple levels of nesting) will be wired by the AWS CDK
 * through CloudFormation parameters and outputs. When a resource from a parent stack is referenced by a nested stack,
 * a CloudFormation parameter will automatically be added to the nested stack and assigned from the parent; when a resource
 * from a nested stack is referenced by a parent stack, a CloudFormation output will be automatically be added to the
 * nested stack and referenced using <code>Fn::GetAtt "Outputs.Xxx"</code> from the parent.
 * <p>
 * Nested stacks also support the use of Docker image and file assets.
 * <p>
 * <h2>Accessing resources in a different stack</h2>
 * <p>
 * You can access resources in a different stack, as long as they are in the
 * same account and AWS Region (see <a href="#accessing-resources-in-a-different-stack-and-region">next section</a> for an exception).
 * The following example defines the stack <code>stack1</code>,
 * which defines an Amazon S3 bucket. Then it defines a second stack, <code>stack2</code>,
 * which takes the bucket from stack1 as a constructor property.
 * <p>
 * <blockquote><pre>
 * Map&lt;String, String&gt; prod = Map.of("account", "123456789012", "region", "us-east-1");
 * 
 * StackThatProvidesABucket stack1 = StackThatProvidesABucket.Builder.create(app, "Stack1").env(prod).build();
 * 
 * // stack2 will take a property { bucket: IBucket }
 * StackThatExpectsABucket stack2 = new StackThatExpectsABucket(app, "Stack2", new StackThatExpectsABucketProps()
 *         .bucket(stack1.getBucket())
 *         .env(prod)
 *         );
 * </pre></blockquote>
 * <p>
 * If the AWS CDK determines that the resource is in the same account and
 * Region, but in a different stack, it automatically synthesizes AWS
 * CloudFormation
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-exports.html">Exports</a>
 * in the producing stack and an
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-importvalue.html">Fn::ImportValue</a>
 * in the consuming stack to transfer that information from one stack to the
 * other.
 * <p>
 * <h2>Accessing resources in a different stack and region</h2>
 * <p>
 * <blockquote>
 * <p>
 * <strong>This feature is currently experimental</strong>
 * <p>
 * </blockquote>
 * <p>
 * You can enable the Stack property <code>crossRegionReferences</code>
 * in order to access resources in a different stack <em>and</em> region. With this feature flag
 * enabled it is possible to do something like creating a CloudFront distribution in <code>us-east-2</code> and
 * an ACM certificate in <code>us-east-1</code>.
 * <p>
 * <blockquote><pre>
 * Stack stack1 = Stack.Builder.create(app, "Stack1")
 *         .env(Environment.builder()
 *                 .region("us-east-1")
 *                 .build())
 *         .crossRegionReferences(true)
 *         .build();
 * Certificate cert = Certificate.Builder.create(stack1, "Cert")
 *         .domainName("*.example.com")
 *         .validation(CertificateValidation.fromDns(PublicHostedZone.fromHostedZoneId(stack1, "Zone", "Z0329774B51CGXTDQV3X")))
 *         .build();
 * 
 * Stack stack2 = Stack.Builder.create(app, "Stack2")
 *         .env(Environment.builder()
 *                 .region("us-east-2")
 *                 .build())
 *         .crossRegionReferences(true)
 *         .build();
 * Distribution.Builder.create(stack2, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new HttpOrigin("example.com"))
 *                 .build())
 *         .domainNames(List.of("dev.example.com"))
 *         .certificate(cert)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When the AWS CDK determines that the resource is in a different stack <em>and</em> is in a different
 * region, it will "export" the value by creating a custom resource in the producing stack which
 * creates SSM Parameters in the consuming region for each exported value. The parameters will be
 * created with the name '/cdk/exports/${consumingStackName}/${export-name}'.
 * In order to "import" the exports into the consuming stack a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html#dynamic-references-ssm">SSM Dynamic reference</a>
 * is used to reference the SSM parameter which was created.
 * <p>
 * In order to mimic strong references, a Custom Resource is also created in the consuming
 * stack which marks the SSM parameters as being "imported". When a parameter has been successfully
 * imported, the producing stack cannot update the value.
 * <p>
 * See the <a href="https://github.com/aws/aws-cdk/blob/main/packages/&#64;aws-cdk/core/adr/cross-region-stack-references">adr</a>
 * for more details on this feature.
 * <p>
 * <h3>Removing automatic cross-stack references</h3>
 * <p>
 * The automatic references created by CDK when you use resources across stacks
 * are convenient, but may block your deployments if you want to remove the
 * resources that are referenced in this way. You will see an error like:
 * <p>
 * <blockquote><pre>
 * Export Stack1:ExportsOutputFnGetAtt-****** cannot be deleted as it is in use by Stack1
 * </pre></blockquote>
 * <p>
 * Let's say there is a Bucket in the <code>stack1</code>, and the <code>stack2</code> references its
 * <code>bucket.bucketName</code>. You now want to remove the bucket and run into the error above.
 * <p>
 * It's not safe to remove <code>stack1.bucket</code> while <code>stack2</code> is still using it, so
 * unblocking yourself from this is a two-step process. This is how it works:
 * <p>
 * DEPLOYMENT 1: break the relationship
 * <p>
 * <ul>
 * <li>Make sure <code>stack2</code> no longer references <code>bucket.bucketName</code> (maybe the consumer
 * stack now uses its own bucket, or it writes to an AWS DynamoDB table, or maybe you just
 * remove the Lambda Function altogether).</li>
 * <li>In the <code>stack1</code> class, call <code>this.exportValue(this.bucket.bucketName)</code>. This
 * will make sure the CloudFormation Export continues to exist while the relationship
 * between the two stacks is being broken.</li>
 * <li>Deploy (this will effectively only change the <code>stack2</code>, but it's safe to deploy both).</li>
 * </ul>
 * <p>
 * DEPLOYMENT 2: remove the resource
 * <p>
 * <ul>
 * <li>You are now free to remove the <code>bucket</code> resource from <code>stack1</code>.</li>
 * <li>Don't forget to remove the <code>exportValue()</code> call as well.</li>
 * <li>Deploy again (this time only the <code>stack1</code> will be changed -- the bucket will be deleted).</li>
 * </ul>
 * <p>
 * <h2>Durations</h2>
 * <p>
 * To make specifications of time intervals unambiguous, a single class called
 * <code>Duration</code> is used throughout the AWS Construct Library by all constructs
 * that that take a time interval as a parameter (be it for a timeout, a
 * rate, or something else).
 * <p>
 * An instance of Duration is constructed by using one of the static factory
 * methods on it:
 * <p>
 * <blockquote><pre>
 * Duration.seconds(300); // 5 minutes
 * Duration.minutes(5); // 5 minutes
 * Duration.hours(1); // 1 hour
 * Duration.days(7); // 7 days
 * Duration.parse("PT5M");
 * </pre></blockquote>
 * <p>
 * Durations can be added or subtracted together:
 * <p>
 * <blockquote><pre>
 * Duration.minutes(1).plus(Duration.seconds(60)); // 2 minutes
 * Duration.minutes(5).minus(Duration.seconds(10));
 * </pre></blockquote>
 * <p>
 * <h2>Size (Digital Information Quantity)</h2>
 * <p>
 * To make specification of digital storage quantities unambiguous, a class called
 * <code>Size</code> is available.
 * <p>
 * An instance of <code>Size</code> is initialized through one of its static factory methods:
 * <p>
 * <blockquote><pre>
 * Size.kibibytes(200); // 200 KiB
 * Size.mebibytes(5); // 5 MiB
 * Size.gibibytes(40); // 40 GiB
 * Size.tebibytes(200); // 200 TiB
 * Size.pebibytes(3);
 * </pre></blockquote>
 * <p>
 * Instances of <code>Size</code> created with one of the units can be converted into others.
 * By default, conversion to a higher unit will fail if the conversion does not produce
 * a whole number. This can be overridden by unsetting <code>integral</code> property.
 * <p>
 * <blockquote><pre>
 * Size.mebibytes(2).toKibibytes(); // yields 2048
 * Size.kibibytes(2050).toMebibytes(SizeConversionOptions.builder().rounding(SizeRoundingBehavior.FLOOR).build());
 * </pre></blockquote>
 * <p>
 * <h2>Secrets</h2>
 * <p>
 * To help avoid accidental storage of secrets as plain text, we use the <code>SecretValue</code> type to
 * represent secrets. Any construct that takes a value that should be a secret (such as
 * a password or an access key) will take a parameter of type <code>SecretValue</code>.
 * <p>
 * The best practice is to store secrets in AWS Secrets Manager and reference them using <code>SecretValue.secretsManager</code>:
 * <p>
 * <blockquote><pre>
 * SecretValue secret = SecretValue.secretsManager("secretId", SecretsManagerSecretOptions.builder()
 *         .jsonField("password") // optional: key of a JSON field to retrieve (defaults to all content),
 *         .versionId("id") // optional: id of the version (default AWSCURRENT)
 *         .versionStage("stage")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Using AWS Secrets Manager is the recommended way to reference secrets in a CDK app.
 * <code>SecretValue</code> also supports the following secret sources:
 * <p>
 * <ul>
 * <li><code>SecretValue.unsafePlainText(secret)</code>: stores the secret as plain text in your app and the resulting template (not recommended).</li>
 * <li><code>SecretValue.secretsManager(secret)</code>: refers to a secret stored in Secrets Manager</li>
 * <li><code>SecretValue.ssmSecure(param, version)</code>: refers to a secret stored as a SecureString in the SSM
 * Parameter Store. If you don't specify the exact version, AWS CloudFormation uses the latest
 * version of the parameter.</li>
 * <li><code>SecretValue.cfnParameter(param)</code>: refers to a secret passed through a CloudFormation parameter (must have <code>NoEcho: true</code>).</li>
 * <li><code>SecretValue.cfnDynamicReference(dynref)</code>: refers to a secret described by a CloudFormation dynamic reference (used by <code>ssmSecure</code> and <code>secretsManager</code>).</li>
 * <li><code>SecretValue.resourceAttribute(attr)</code>: refers to a secret returned from a CloudFormation resource creation.</li>
 * </ul>
 * <p>
 * <code>SecretValue</code>s should only be passed to constructs that accept properties of type
 * <code>SecretValue</code>. These constructs are written to ensure your secrets will not be
 * exposed where they shouldn't be. If you try to use a <code>SecretValue</code> in a
 * different location, an error about unsafe secret usage will be thrown at
 * synthesis time.
 * <p>
 * If you rotate the secret's value in Secrets Manager, you must also change at
 * least one property on the resource where you are using the secret, to force
 * CloudFormation to re-read the secret.
 * <p>
 * <code>SecretValue.ssmSecure()</code> is only supported for a limited set of resources.
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html#template-parameters-dynamic-patterns-resources">Click here for a list of supported resources and properties</a>.
 * <p>
 * <h2>ARN manipulation</h2>
 * <p>
 * Sometimes you will need to put together or pick apart Amazon Resource Names
 * (ARNs). The functions <code>stack.formatArn()</code> and <code>stack.splitArn()</code> exist for
 * this purpose.
 * <p>
 * <code>formatArn()</code> can be used to build an ARN from components. It will automatically
 * use the region and account of the stack you're calling it on:
 * <p>
 * <blockquote><pre>
 * Stack stack;
 * 
 * 
 * // Builds "arn:&lt;PARTITION&gt;:lambda:&lt;REGION&gt;:&lt;ACCOUNT&gt;:function:MyFunction"
 * stack.formatArn(ArnComponents.builder()
 *         .service("lambda")
 *         .resource("function")
 *         .arnFormat(ArnFormat.COLON_RESOURCE_NAME)
 *         .resourceName("MyFunction")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <code>splitArn()</code> can be used to get a single component from an ARN. <code>splitArn()</code>
 * will correctly deal with both literal ARNs and deploy-time values (tokens),
 * but in case of a deploy-time value be aware that the result will be another
 * deploy-time value which cannot be inspected in the CDK application.
 * <p>
 * <blockquote><pre>
 * Stack stack;
 * 
 * 
 * // Extracts the function name out of an AWS Lambda Function ARN
 * ArnComponents arnComponents = stack.splitArn(arn, ArnFormat.COLON_RESOURCE_NAME);
 * String functionName = arnComponents.getResourceName();
 * </pre></blockquote>
 * <p>
 * Note that the format of the resource separator depends on the service and
 * may be any of the values supported by <code>ArnFormat</code>. When dealing with these
 * functions, it is important to know the format of the ARN you are dealing with.
 * <p>
 * For an exhaustive list of ARN formats used in AWS, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">AWS ARNs and
 * Namespaces</a>
 * in the AWS General Reference.
 * <p>
 * <h2>Dependencies</h2>
 * <p>
 * <h3>Construct Dependencies</h3>
 * <p>
 * Sometimes AWS resources depend on other resources, and the creation of one
 * resource must be completed before the next one can be started.
 * <p>
 * In general, CloudFormation will correctly infer the dependency relationship
 * between resources based on the property values that are used. In the cases where
 * it doesn't, the AWS Construct Library will add the dependency relationship for
 * you.
 * <p>
 * If you need to add an ordering dependency that is not automatically inferred,
 * you do so by adding a dependency relationship using
 * <code>constructA.node.addDependency(constructB)</code>. This will add a dependency
 * relationship between all resources in the scope of <code>constructA</code> and all
 * resources in the scope of <code>constructB</code>.
 * <p>
 * If you want a single object to represent a set of constructs that are not
 * necessarily in the same scope, you can use a <code>DependencyGroup</code>. The
 * following creates a single object that represents a dependency on two
 * constructs, <code>constructB</code> and <code>constructC</code>:
 * <p>
 * <blockquote><pre>
 * // Declare the dependable object
 * DependencyGroup bAndC = new DependencyGroup();
 * bAndC.add(constructB);
 * bAndC.add(constructC);
 * 
 * // Take the dependency
 * constructA.node.addDependency(bAndC);
 * </pre></blockquote>
 * <p>
 * <h3>Stack Dependencies</h3>
 * <p>
 * Two different stack instances can have a dependency on one another. This
 * happens when an resource from one stack is referenced in another stack. In
 * that case, CDK records the cross-stack referencing of resources,
 * automatically produces the right CloudFormation primitives, and adds a
 * dependency between the two stacks. You can also manually add a dependency
 * between two stacks by using the <code>stackA.addDependency(stackB)</code> method.
 * <p>
 * A stack dependency has the following implications:
 * <p>
 * <ul>
 * <li>Cyclic dependencies are not allowed, so if <code>stackA</code> is using resources from
 * <code>stackB</code>, the reverse is not possible anymore.</li>
 * <li>Stacks with dependencies between them are treated specially by the CDK
 * toolkit:
 * <p>
 * <ul>
 * <li>If <code>stackA</code> depends on <code>stackB</code>, running <code>cdk deploy stackA</code> will also
 * automatically deploy <code>stackB</code>.</li>
 * <li><code>stackB</code>'s deployment will be performed <em>before</em> <code>stackA</code>'s deployment.</li>
 * </ul></li>
 * </ul>
 * <p>
 * <h3>CfnResource Dependencies</h3>
 * <p>
 * To make declaring dependencies between <code>CfnResource</code> objects easier, you can declare dependencies from one <code>CfnResource</code> object on another by using the <code>cfnResource1.addDependency(cfnResource2)</code> method. This method will work for resources both within the same stack and across stacks as it detects the relative location of the two resources and adds the dependency either to the resource or between the relevant stacks, as appropriate. If more complex logic is in needed, you can similarly remove, replace, or view dependencies between <code>CfnResource</code> objects with the <code>CfnResource</code> <code>removeDependency</code>, <code>replaceDependency</code>, and <code>obtainDependencies</code> methods, respectively.
 * <p>
 * <h2>Custom Resources</h2>
 * <p>
 * Custom Resources are CloudFormation resources that are implemented by arbitrary
 * user code. They can do arbitrary lookups or modifications during a
 * CloudFormation deployment.
 * <p>
 * Custom resources are backed by <em>custom resource providers</em>. Commonly, these are
 * Lambda Functions that are deployed in the same deployment as the one that
 * defines the custom resource itself, but they can also be backed by Lambda
 * Functions deployed previously, or code responding to SNS Topic events running on
 * EC2 instances in a completely different account. For more information on custom
 * resource providers, see the next section.
 * <p>
 * Once you have a provider, each definition of a <code>CustomResource</code> construct
 * represents one invocation. A single provider can be used for the implementation
 * of arbitrarily many custom resource definitions. A single definition looks like
 * this:
 * <p>
 * <blockquote><pre>
 * CustomResource.Builder.create(this, "MyMagicalResource")
 *         .resourceType("Custom::MyCustomResource") // must start with 'Custom::'
 * 
 *         // the resource properties
 *         .properties(Map.of(
 *                 "Property1", "foo",
 *                 "Property2", "bar"))
 * 
 *         // the ARN of the provider (SNS/Lambda) which handles
 *         // CREATE, UPDATE or DELETE events for this resource type
 *         // see next section for details
 *         .serviceToken("ARN")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Custom Resource Providers</h3>
 * <p>
 * Custom resources are backed by a <strong>custom resource provider</strong> which can be
 * implemented in one of the following ways. The following table compares the
 * various provider types (ordered from low-level to high-level):
 * <p>
 * | Provider                                                             | Compute Type | Error Handling | Submit to CloudFormation |   Max Timeout   | Language | Footprint |
 * | -------------------------------------------------------------------- | :----------: | :------------: | :----------------------: | :-------------: | :------: | :-------: |
 * | <a href="#amazon-sns-topic">sns.Topic</a>                                       | Self-managed |     Manual     |          Manual          |    Unlimited    |   Any    |  Depends  |
 * | <a href="#aws-lambda-function">lambda.Function</a>                              |  AWS Lambda  |     Manual     |          Manual          |      15min      |   Any    |   Small   |
 * | <a href="#the-corecustomresourceprovider-class">core.CustomResourceProvider</a> |  AWS Lambda  |      Auto      |           Auto           |      15min      | Node.js  |   Small   |
 * | <a href="#the-custom-resource-provider-framework">custom-resources.Provider</a> |  AWS Lambda  |      Auto      |           Auto           | Unlimited Async |   Any    |   Large   |
 * <p>
 * Legend:
 * <p>
 * <ul>
 * <li><strong>Compute type</strong>: which type of compute can be used to execute the handler.</li>
 * <li><strong>Error Handling</strong>: whether errors thrown by handler code are automatically
 * trapped and a FAILED response is submitted to CloudFormation. If this is
 * "Manual", developers must take care of trapping errors. Otherwise, events
 * could cause stacks to hang.</li>
 * <li><strong>Submit to CloudFormation</strong>: whether the framework takes care of submitting
 * SUCCESS/FAILED responses to CloudFormation through the event's response URL.</li>
 * <li><strong>Max Timeout</strong>: maximum allows/possible timeout.</li>
 * <li><strong>Language</strong>: which programming languages can be used to implement handlers.</li>
 * <li><strong>Footprint</strong>: how many resources are used by the provider framework itself.</li>
 * </ul>
 * <p>
 * <strong>A NOTE ABOUT SINGLETONS</strong>
 * <p>
 * When defining resources for a custom resource provider, you will likely want to
 * define them as a <em>stack singleton</em> so that only a single instance of the
 * provider is created in your stack and which is used by all custom resources of
 * that type.
 * <p>
 * Here is a basic pattern for defining stack singletons in the CDK. The following
 * examples ensures that only a single SNS topic is defined:
 * <p>
 * <blockquote><pre>
 * public Topic getOrCreate(Construct scope) {
 *     Stack stack = Stack.of(scope);
 *     String uniqueid = "GloballyUniqueIdForSingleton"; // For example, a UUID from `uuidgen`
 *     IConstruct existing = stack.node.tryFindChild(uniqueid);
 *     if (existing) {
 *         return (Topic)existing;
 *     }
 *     return new Topic(stack, uniqueid);
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Amazon SNS Topic</h4>
 * <p>
 * Every time a resource event occurs (CREATE/UPDATE/DELETE), an SNS notification
 * is sent to the SNS topic. Users must process these notifications (e.g. through a
 * fleet of worker hosts) and submit success/failure responses to the
 * CloudFormation service.
 * <p>
 * <blockquote>
 * <p>
 * You only need to use this type of provider if your custom resource cannot run on AWS Lambda, for reasons other than the 15
 * minute timeout. If you are considering using this type of provider because you want to write a custom resource provider that may need
 * to wait for more than 15 minutes for the API calls to stabilize, have a look at the <a href="#the-custom-resource-provider-framework"><code>custom-resources</code></a> module first.
 * <p>
 * Refer to the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-custom-resources.html">CloudFormation Custom Resource documentation</a> for information on the contract your custom resource needs to adhere to.
 * <p>
 * </blockquote>
 * <p>
 * Set <code>serviceToken</code> to <code>topic.topicArn</code>  in order to use this provider:
 * <p>
 * <blockquote><pre>
 * Topic topic = new Topic(this, "MyProvider");
 * 
 * CustomResource.Builder.create(this, "MyResource")
 *         .serviceToken(topic.getTopicArn())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>AWS Lambda Function</h4>
 * <p>
 * An AWS lambda function is called <em>directly</em> by CloudFormation for all resource
 * events. The handler must take care of explicitly submitting a success/failure
 * response to the CloudFormation service and handle various error cases.
 * <p>
 * <blockquote>
 * <p>
 * <strong>We do not recommend you use this provider type.</strong> The CDK has wrappers around Lambda Functions that make them easier to work with.
 * <p>
 * If you do want to use this provider, refer to the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-custom-resources.html">CloudFormation Custom Resource documentation</a> for information on the contract your custom resource needs to adhere to.
 * <p>
 * </blockquote>
 * <p>
 * Set <code>serviceToken</code> to <code>lambda.functionArn</code> to use this provider:
 * <p>
 * <blockquote><pre>
 * SingletonFunction fn = new SingletonFunction(this, "MyProvider", functionProps);
 * 
 * CustomResource.Builder.create(this, "MyResource")
 *         .serviceToken(fn.getFunctionArn())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>The <code>core.CustomResourceProvider</code> class</h4>
 * <p>
 * The class <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_core.CustomResourceProvider.html"><code>&#64;aws-cdk/core.CustomResourceProvider</code></a> offers a basic low-level
 * framework designed to implement simple and slim custom resource providers. It
 * currently only supports Node.js-based user handlers, represents permissions as raw
 * JSON blobs instead of <code>iam.PolicyStatement</code> objects, and it does not have
 * support for asynchronous waiting (handler cannot exceed the 15min lambda
 * timeout). The <code>CustomResourceProviderRuntime</code> supports runtime <code>nodejs12.x</code>,
 * <code>nodejs14.x</code>, <code>nodejs16.x</code>, <code>nodejs18.x</code>.
 * <p>
 * <blockquote>
 * <p>
 * <strong>As an application builder, we do not recommend you use this provider type.</strong> This provider exists purely for custom resources that are part of the AWS Construct Library.
 * <p>
 * The <a href="#the-custom-resource-provider-framework"><code>custom-resources</code></a> provider is more convenient to work with and more fully-featured.
 * <p>
 * </blockquote>
 * <p>
 * The provider has a built-in singleton method which uses the resource type as a
 * stack-unique identifier and returns the service token:
 * <p>
 * <blockquote><pre>
 * String serviceToken = CustomResourceProvider.getOrCreate(this, "Custom::MyCustomResourceType", CustomResourceProviderProps.builder()
 *         .codeDirectory(String.format("%s/my-handler", __dirname))
 *         .runtime(CustomResourceProviderRuntime.NODEJS_18_X)
 *         .description("Lambda function created by the custom resource provider")
 *         .build());
 * 
 * CustomResource.Builder.create(this, "MyResource")
 *         .resourceType("Custom::MyCustomResourceType")
 *         .serviceToken(serviceToken)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The directory (<code>my-handler</code> in the above example) must include an <code>index.js</code> file. It cannot import
 * external dependencies or files outside this directory. It must export an async
 * function named <code>handler</code>. This function accepts the CloudFormation resource
 * event object and returns an object with the following structure:
 * <p>
 * <blockquote><pre>
 * exports.handler = async function(event) {
 *   const id = event.PhysicalResourceId; // only for "Update" and "Delete"
 *   const props = event.ResourceProperties;
 *   const oldProps = event.OldResourceProperties; // only for "Update"s
 * 
 *   switch (event.RequestType) {
 *     case "Create":
 *       // ...
 * 
 *     case "Update":
 *       // ...
 * 
 *       // if an error is thrown, a FAILED response will be submitted to CFN
 *       throw new Error('Failed!');
 * 
 *     case "Delete":
 *       // ...
 *   }
 * 
 *   return {
 *     // (optional) the value resolved from `resource.ref`
 *     // defaults to "event.PhysicalResourceId" or "event.RequestId"
 *     PhysicalResourceId: "REF",
 * 
 *     // (optional) calling `resource.getAtt("Att1")` on the custom resource in the CDK app
 *     // will return the value "BAR".
 *     Data: {
 *       Att1: "BAR",
 *       Att2: "BAZ"
 *     },
 * 
 *     // (optional) user-visible message
 *     Reason: "User-visible message",
 * 
 *     // (optional) hides values from the console
 *     NoEcho: true
 *   };
 * }
 * </pre></blockquote>
 * <p>
 * Here is an complete example of a custom resource that summarizes two numbers:
 * <p>
 * <code>sum-handler/index.js</code>:
 * <p>
 * <blockquote><pre>
 * exports.handler = async (e) =&gt; {
 *   return {
 *     Data: {
 *       Result: e.ResourceProperties.lhs + e.ResourceProperties.rhs,
 *     },
 *   };
 * };
 * </pre></blockquote>
 * <p>
 * <code>sum.ts</code>:
 * <p>
 * <blockquote><pre>
 * import software.constructs.Construct;
 * import software.amazon.awscdk.CustomResource;
 * import software.amazon.awscdk.CustomResourceProvider;
 * import software.amazon.awscdk.CustomResourceProviderRuntime;
 * import software.amazon.awscdk.Token;
 * 
 * public class SumProps {
 *     private Number lhs;
 *     public Number getLhs() {
 *         return this.lhs;
 *     }
 *     public SumProps lhs(Number lhs) {
 *         this.lhs = lhs;
 *         return this;
 *     }
 *     private Number rhs;
 *     public Number getRhs() {
 *         return this.rhs;
 *     }
 *     public SumProps rhs(Number rhs) {
 *         this.rhs = rhs;
 *         return this;
 *     }
 * }
 * 
 * public class Sum extends Construct {
 *     public final Number result;
 * 
 *     public Sum(Construct scope, String id, SumProps props) {
 *         super(scope, id);
 * 
 *         String resourceType = "Custom::Sum";
 *         String serviceToken = CustomResourceProvider.getOrCreate(this, resourceType, CustomResourceProviderProps.builder()
 *                 .codeDirectory(String.format("%s/sum-handler", __dirname))
 *                 .runtime(CustomResourceProviderRuntime.NODEJS_18_X)
 *                 .build());
 * 
 *         CustomResource resource = CustomResource.Builder.create(this, "Resource")
 *                 .resourceType(resourceType)
 *                 .serviceToken(serviceToken)
 *                 .properties(Map.of(
 *                         "lhs", props.getLhs(),
 *                         "rhs", props.getRhs()))
 *                 .build();
 * 
 *         this.result = Token.asNumber(resource.getAtt("Result"));
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Usage will look like this:
 * <p>
 * <blockquote><pre>
 * Sum sum = new Sum(this, "MySum", new SumProps().lhs(40).rhs(2));
 * CfnOutput.Builder.create(this, "Result").value(Token.asString(sum.getResult())).build();
 * </pre></blockquote>
 * <p>
 * To access the ARN of the provider's AWS Lambda function role, use the <code>getOrCreateProvider()</code>
 * built-in singleton method:
 * <p>
 * <blockquote><pre>
 * CustomResourceProvider provider = CustomResourceProvider.getOrCreateProvider(this, "Custom::MyCustomResourceType", CustomResourceProviderProps.builder()
 *         .codeDirectory(String.format("%s/my-handler", __dirname))
 *         .runtime(CustomResourceProviderRuntime.NODEJS_18_X)
 *         .build());
 * 
 * String roleArn = provider.getRoleArn();
 * </pre></blockquote>
 * <p>
 * This role ARN can then be used in resource-based IAM policies.
 * <p>
 * To add IAM policy statements to this role, use <code>addToRolePolicy()</code>:
 * <p>
 * <blockquote><pre>
 * CustomResourceProvider provider = CustomResourceProvider.getOrCreateProvider(this, "Custom::MyCustomResourceType", CustomResourceProviderProps.builder()
 *         .codeDirectory(String.format("%s/my-handler", __dirname))
 *         .runtime(CustomResourceProviderRuntime.NODEJS_18_X)
 *         .build());
 * provider.addToRolePolicy(Map.of(
 *         "Effect", "Allow",
 *         "Action", "s3:GetObject",
 *         "Resource", "*"));
 * </pre></blockquote>
 * <p>
 * Note that <code>addToRolePolicy()</code> uses direct IAM JSON policy blobs, <em>not</em> a
 * <code>iam.PolicyStatement</code> object like you will see in the rest of the CDK.
 * <p>
 * <h4>The Custom Resource Provider Framework</h4>
 * <p>
 * The <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html"><code>&#64;aws-cdk/custom-resources</code></a> module includes an advanced framework for
 * implementing custom resource providers.
 * <p>
 * Handlers are implemented as AWS Lambda functions, which means that they can be
 * implemented in any Lambda-supported runtime. Furthermore, this provider has an
 * asynchronous mode, which means that users can provide an <code>isComplete</code> lambda
 * function which is called periodically until the operation is complete. This
 * allows implementing providers that can take up to two hours to stabilize.
 * <p>
 * Set <code>serviceToken</code> to <code>provider.serviceToken</code> to use this type of provider:
 * <p>
 * <blockquote><pre>
 * Provider provider = Provider.Builder.create(this, "MyProvider")
 *         .onEventHandler(onEventHandler)
 *         .isCompleteHandler(isCompleteHandler)
 *         .build();
 * 
 * CustomResource.Builder.create(this, "MyResource")
 *         .serviceToken(provider.getServiceToken())
 *         .build();
 * </pre></blockquote>
 * <p>
 * See the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-cdk-lib.custom_resources-readme.html">documentation</a> for more details.
 * <p>
 * <h2>AWS CloudFormation features</h2>
 * <p>
 * A CDK stack synthesizes to an AWS CloudFormation Template. This section
 * explains how this module allows users to access low-level CloudFormation
 * features when needed.
 * <p>
 * <h3>Stack Outputs</h3>
 * <p>
 * CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html">stack outputs</a> and exports are created using
 * the <code>CfnOutput</code> class:
 * <p>
 * <blockquote><pre>
 * CfnOutput.Builder.create(this, "OutputName")
 *         .value(myBucket.getBucketName())
 *         .description("The name of an S3 bucket") // Optional
 *         .exportName("TheAwesomeBucket")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Parameters</h3>
 * <p>
 * CloudFormation templates support the use of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">Parameters</a> to
 * customize a template. They enable CloudFormation users to input custom values to
 * a template each time a stack is created or updated. While the CDK design
 * philosophy favors using build-time parameterization, users may need to use
 * CloudFormation in a number of cases (for example, when migrating an existing
 * stack to the AWS CDK).
 * <p>
 * Template parameters can be added to a stack by using the <code>CfnParameter</code> class:
 * <p>
 * <blockquote><pre>
 * CfnParameter.Builder.create(this, "MyParameter")
 *         .type("Number")
 *         .default(1337)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The value of parameters can then be obtained using one of the <code>value</code> methods.
 * As parameters are only resolved at deployment time, the values obtained are
 * placeholder tokens for the real value (<code>Token.isUnresolved()</code> would return <code>true</code>
 * for those):
 * <p>
 * <blockquote><pre>
 * CfnParameter param = CfnParameter.Builder.create(this, "ParameterName").build();
 * 
 * // If the parameter is a String
 * param.getValueAsString();
 * 
 * // If the parameter is a Number
 * param.getValueAsNumber();
 * 
 * // If the parameter is a List
 * param.getValueAsList();
 * </pre></blockquote>
 * <p>
 * <h3>Pseudo Parameters</h3>
 * <p>
 * CloudFormation supports a number of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/pseudo-parameter-reference.html">pseudo parameters</a>,
 * which resolve to useful values at deployment time. CloudFormation pseudo
 * parameters can be obtained from static members of the <code>Aws</code> class.
 * <p>
 * It is generally recommended to access pseudo parameters from the scope's <code>stack</code>
 * instead, which guarantees the values produced are qualifying the designated
 * stack, which is essential in cases where resources are shared cross-stack:
 * <p>
 * <blockquote><pre>
 * // "this" is the current construct
 * Stack stack = Stack.of(this);
 * 
 * stack.getAccount(); // Returns the AWS::AccountId for this stack (or the literal value if known)
 * stack.getRegion(); // Returns the AWS::Region for this stack (or the literal value if known)
 * stack.getPartition();
 * </pre></blockquote>
 * <p>
 * <h3>Resource Options</h3>
 * <p>
 * CloudFormation resources can also specify <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-product-attribute-reference.html">resource
 * attributes</a>. The <code>CfnResource</code> class allows
 * accessing those through the <code>cfnOptions</code> property:
 * <p>
 * <blockquote><pre>
 * CfnBucket rawBucket = CfnBucket.Builder.create(this, "Bucket").build();
 * // -or-
 * CfnBucket rawBucketAlt = (CfnBucket)myBucket.getNode().getDefaultChild();
 * 
 * // then
 * rawBucket.getCfnOptions().getCondition() = CfnCondition.Builder.create(this, "EnableBucket").build();
 * rawBucket.getCfnOptions().getMetadata() = Map.of(
 *         "metadataKey", "MetadataValue");
 * </pre></blockquote>
 * <p>
 * Resource dependencies (the <code>DependsOn</code> attribute) is modified using the
 * <code>cfnResource.addDependency</code> method:
 * <p>
 * <blockquote><pre>
 * CfnResource resourceA = new CfnResource(this, "ResourceA", resourceProps);
 * CfnResource resourceB = new CfnResource(this, "ResourceB", resourceProps);
 * 
 * resourceB.addDependency(resourceA);
 * </pre></blockquote>
 * <p>
 * <h4>CreationPolicy</h4>
 * <p>
 * Some resources support a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html">CreationPolicy</a> to be specified as a CfnOption.
 * <p>
 * The creation policy is invoked only when AWS CloudFormation creates the associated resource. Currently, the only AWS CloudFormation resources that support creation policies are <code>CfnAutoScalingGroup</code>, <code>CfnInstance</code>, <code>CfnWaitCondition</code> and <code>CfnFleet</code>.
 * <p>
 * The <code>CfnFleet</code> resource from the <code>aws-appstream</code> module supports specifying <code>startFleet</code> as
 * a property of the creationPolicy on the resource options. Setting it to true will make AWS CloudFormation wait until the fleet is started before continuing with the creation of
 * resources that depend on the fleet resource.
 * <p>
 * <blockquote><pre>
 * CfnFleet fleet = CfnFleet.Builder.create(this, "Fleet")
 *         .instanceType("stream.standard.small")
 *         .name("Fleet")
 *         .computeCapacity(ComputeCapacityProperty.builder()
 *                 .desiredInstances(1)
 *                 .build())
 *         .imageName("AppStream-AmazonLinux2-09-21-2022")
 *         .build();
 * fleet.getCfnOptions().getCreationPolicy() = CfnCreationPolicy.builder()
 *         .startFleet(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The properties passed to the level 2 constructs <code>AutoScalingGroup</code> and <code>Instance</code> from the
 * <code>aws-ec2</code> module abstract what is passed into the <code>CfnOption</code> properties <code>resourceSignal</code> and
 * <code>autoScalingCreationPolicy</code>, but when using level 1 constructs you can specify these yourself.
 * <p>
 * The CfnWaitCondition resource from the <code>aws-cloudformation</code> module suppports the <code>resourceSignal</code>.
 * The format of the timeout is <code>PT#H#M#S</code>. In the example below AWS Cloudformation will wait for
 * 3 success signals to occur within 15 minutes before the status of the resource will be set to
 * <code>CREATE_COMPLETE</code>.
 * <p>
 * <blockquote><pre>
 * CfnResource resource;
 * 
 * 
 * resource.getCfnOptions().getCreationPolicy() = CfnCreationPolicy.builder()
 *         .resourceSignal(CfnResourceSignal.builder()
 *                 .count(3)
 *                 .timeout("PR15M")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Intrinsic Functions and Condition Expressions</h3>
 * <p>
 * CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html">intrinsic functions</a>. These functions
 * can be accessed from the <code>Fn</code> class, which provides type-safe methods for each
 * intrinsic function as well as condition expressions:
 * <p>
 * <blockquote><pre>
 * Object myObjectOrArray;
 * Object myArray;
 * 
 * 
 * // To use Fn::Base64
 * Fn.base64("SGVsbG8gQ0RLIQo=");
 * 
 * // To compose condition expressions:
 * CfnParameter environmentParameter = new CfnParameter(this, "Environment");
 * Fn.conditionAnd(Fn.conditionEquals("Production", environmentParameter), Fn.conditionNot(Fn.conditionEquals("us-east-1", Aws.REGION)));
 * 
 * // To use Fn::ToJsonString
 * Fn.toJsonString(myObjectOrArray);
 * 
 * // To use Fn::Length
 * Fn.len(Fn.split(",", myArray));
 * </pre></blockquote>
 * <p>
 * When working with deploy-time values (those for which <code>Token.isUnresolved</code>
 * returns <code>true</code>), idiomatic conditionals from the programming language cannot be
 * used (the value will not be known until deployment time). When conditional logic
 * needs to be expressed with un-resolved values, it is necessary to use
 * CloudFormation conditions by means of the <code>CfnCondition</code> class:
 * <p>
 * <blockquote><pre>
 * CfnParameter environmentParameter = new CfnParameter(this, "Environment");
 * CfnCondition isProd = CfnCondition.Builder.create(this, "IsProduction")
 *         .expression(Fn.conditionEquals("Production", environmentParameter))
 *         .build();
 * 
 * // Configuration value that is a different string based on IsProduction
 * String stage = Fn.conditionIf(isProd.logicalId, "Beta", "Prod").toString();
 * 
 * // Make Bucket creation condition to IsProduction by accessing
 * // and overriding the CloudFormation resource
 * Bucket bucket = new Bucket(this, "Bucket");
 * CfnBucket cfnBucket = (CfnBucket)myBucket.getNode().getDefaultChild();
 * cfnBucket.getCfnOptions().getCondition() = isProd;
 * </pre></blockquote>
 * <p>
 * <h3>Mappings</h3>
 * <p>
 * CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html">mappings</a> are created and queried using the
 * <code>CfnMappings</code> class:
 * <p>
 * <blockquote><pre>
 * CfnMapping regionTable = CfnMapping.Builder.create(this, "RegionTable")
 *         .mapping(Map.of(
 *                 "us-east-1", Map.of(
 *                         "regionName", "US East (N. Virginia)"),
 *                 "us-east-2", Map.of(
 *                         "regionName", "US East (Ohio)")))
 *         .build();
 * 
 * regionTable.findInMap(Aws.REGION, "regionName");
 * </pre></blockquote>
 * <p>
 * This will yield the following template:
 * <p>
 * <blockquote><pre>
 * Mappings:
 *   RegionTable:
 *     us-east-1:
 *       regionName: US East (N. Virginia)
 *     us-east-2:
 *       regionName: US East (Ohio)
 * </pre></blockquote>
 * <p>
 * Mappings can also be synthesized "lazily"; lazy mappings will only render a "Mappings"
 * section in the synthesized CloudFormation template if some <code>findInMap</code> call is unable to
 * immediately return a concrete value due to one or both of the keys being unresolved tokens
 * (some value only available at deploy-time).
 * <p>
 * For example, the following code will not produce anything in the "Mappings" section. The
 * call to <code>findInMap</code> will be able to resolve the value during synthesis and simply return
 * <code>'US East (Ohio)'</code>.
 * <p>
 * <blockquote><pre>
 * CfnMapping regionTable = CfnMapping.Builder.create(this, "RegionTable")
 *         .mapping(Map.of(
 *                 "us-east-1", Map.of(
 *                         "regionName", "US East (N. Virginia)"),
 *                 "us-east-2", Map.of(
 *                         "regionName", "US East (Ohio)")))
 *         .lazy(true)
 *         .build();
 * 
 * regionTable.findInMap("us-east-2", "regionName");
 * </pre></blockquote>
 * <p>
 * On the other hand, the following code will produce the "Mappings" section shown above,
 * since the top-level key is an unresolved token. The call to <code>findInMap</code> will return a token that resolves to
 * <code>{ "Fn::FindInMap": [ "RegionTable", { "Ref": "AWS::Region" }, "regionName" ] }</code>.
 * <p>
 * <blockquote><pre>
 * CfnMapping regionTable;
 * 
 * 
 * regionTable.findInMap(Aws.REGION, "regionName");
 * </pre></blockquote>
 * <p>
 * An optional default value can also be passed to <code>findInMap</code>. If either key is not found in the map and the mapping is lazy, <code>findInMap</code> will return the default value and not render the mapping.
 * If the mapping is not lazy or either key is an unresolved token, the call to <code>findInMap</code> will return a token that resolves to
 * <code>{ "Fn::FindInMap": [ "MapName", "TopLevelKey", "SecondLevelKey", { "DefaultValue": "DefaultValue" } ] }</code>, and the mapping will be rendered.
 * Note that the <code>AWS::LanguageExtentions</code> transform is added to enable the default value functionality.
 * <p>
 * For example, the following code will again not produce anything in the "Mappings" section. The
 * call to <code>findInMap</code> will be able to resolve the value during synthesis and simply return
 * <code>'Region not found'</code>.
 * <p>
 * <blockquote><pre>
 * CfnMapping regionTable = CfnMapping.Builder.create(this, "RegionTable")
 *         .mapping(Map.of(
 *                 "us-east-1", Map.of(
 *                         "regionName", "US East (N. Virginia)"),
 *                 "us-east-2", Map.of(
 *                         "regionName", "US East (Ohio)")))
 *         .lazy(true)
 *         .build();
 * 
 * regionTable.findInMap("us-west-1", "regionName", "Region not found");
 * </pre></blockquote>
 * <p>
 * <h3>Dynamic References</h3>
 * <p>
 * CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">dynamically resolving</a> values
 * for SSM parameters (including secure strings) and Secrets Manager. Encoding such
 * references is done using the <code>CfnDynamicReference</code> class:
 * <p>
 * <blockquote><pre>
 * new CfnDynamicReference(CfnDynamicReferenceService.SECRETS_MANAGER, "secret-id:secret-string:json-key:version-stage:version-id");
 * </pre></blockquote>
 * <p>
 * <h3>Template Options &amp; Transform</h3>
 * <p>
 * CloudFormation templates support a number of options, including which Macros or
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html">Transforms</a> to use when deploying the stack. Those can be
 * configured using the <code>stack.templateOptions</code> property:
 * <p>
 * <blockquote><pre>
 * Stack stack = new Stack(app, "StackName");
 * 
 * stack.getTemplateOptions().getDescription() = "This will appear in the AWS console";
 * stack.getTemplateOptions().getTransforms() = List.of("AWS::Serverless-2016-10-31");
 * stack.getTemplateOptions().getMetadata() = Map.of(
 *         "metadataKey", "MetadataValue");
 * </pre></blockquote>
 * <p>
 * <h3>Emitting Raw Resources</h3>
 * <p>
 * The <code>CfnResource</code> class allows emitting arbitrary entries in the
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html">Resources</a> section of the CloudFormation template.
 * <p>
 * <blockquote><pre>
 * CfnResource.Builder.create(this, "ResourceId")
 *         .type("AWS::S3::Bucket")
 *         .properties(Map.of(
 *                 "BucketName", "bucket-name"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * As for any other resource, the logical ID in the CloudFormation template will be
 * generated by the AWS CDK, but the type and properties will be copied verbatim in
 * the synthesized template.
 * <p>
 * <h3>Including raw CloudFormation template fragments</h3>
 * <p>
 * When migrating a CloudFormation stack to the AWS CDK, it can be useful to
 * include fragments of an existing template verbatim in the synthesized template.
 * This can be achieved using the <code>CfnInclude</code> class.
 * <p>
 * <blockquote><pre>
 * new CfnInclude(this, "ID", new CfnIncludeProps()
 *         .template(Map.of(
 *                 "Resources", Map.of(
 *                         "Bucket", Map.of(
 *                                 "Type", "AWS::S3::Bucket",
 *                                 "Properties", Map.of(
 *                                         "BucketName", "my-shiny-bucket")))))
 *         );
 * </pre></blockquote>
 * <p>
 * <h3>Termination Protection</h3>
 * <p>
 * You can prevent a stack from being accidentally deleted by enabling termination
 * protection on the stack. If a user attempts to delete a stack with termination
 * protection enabled, the deletion fails and the stack--including its status--remains
 * unchanged. Enabling or disabling termination protection on a stack sets it for any
 * nested stacks belonging to that stack as well. You can enable termination protection
 * on a stack by setting the <code>terminationProtection</code> prop to <code>true</code>.
 * <p>
 * <blockquote><pre>
 * Stack stack = Stack.Builder.create(app, "StackName")
 *         .terminationProtection(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also set termination protection with the setter after you've instantiated the stack.
 * <p>
 * <blockquote><pre>
 * Stack stack = Stack.Builder.create(app, "StackName").build();
 * stack.getTerminationProtection() = true;
 * </pre></blockquote>
 * <p>
 * By default, termination protection is disabled.
 * <p>
 * <h3>Description</h3>
 * <p>
 * You can add a description of the stack in the same way as <code>StackProps</code>.
 * <p>
 * <blockquote><pre>
 * Stack stack = Stack.Builder.create(app, "StackName")
 *         .description("This is a description.")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>CfnJson</h3>
 * <p>
 * <code>CfnJson</code> allows you to postpone the resolution of a JSON blob from
 * deployment-time. This is useful in cases where the CloudFormation JSON template
 * cannot express a certain value.
 * <p>
 * A common example is to use <code>CfnJson</code> in order to render a JSON map which needs
 * to use intrinsic functions in keys. Since JSON map keys must be strings, it is
 * impossible to use intrinsics in keys and <code>CfnJson</code> can help.
 * <p>
 * The following example defines an IAM role which can only be assumed by
 * principals that are tagged with a specific tag.
 * <p>
 * <blockquote><pre>
 * CfnParameter tagParam = new CfnParameter(this, "TagName");
 * 
 * CfnJson stringEquals = CfnJson.Builder.create(this, "ConditionJson")
 *         .value(Map.of(
 *                 String.format("aws:PrincipalTag/%s", tagParam.getValueAsString()), true))
 *         .build();
 * 
 * PrincipalBase principal = new AccountRootPrincipal().withConditions(Map.of(
 *         "StringEquals", stringEquals));
 * 
 * Role.Builder.create(this, "MyRole").assumedBy(principal).build();
 * </pre></blockquote>
 * <p>
 * <strong>Explanation</strong>: since in this example we pass the tag name through a parameter, it
 * can only be resolved during deployment. The resolved value can be represented in
 * the template through a <code>{ "Ref": "TagName" }</code>. However, since we want to use
 * this value inside a <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag"><code>aws:PrincipalTag/TAG-NAME</code></a>
 * IAM operator, we need it in the <em>key</em> of a <code>StringEquals</code> condition. JSON keys
 * <em>must be</em> strings, so to circumvent this limitation, we use <code>CfnJson</code>
 * to "delay" the rendition of this template section to deploy-time. This means
 * that the value of <code>StringEquals</code> in the template will be <code>{ "Fn::GetAtt": [ "ConditionJson", "Value" ] }</code>, and will only "expand" to the operator we synthesized during deployment.
 * <p>
 * <h3>Stack Resource Limit</h3>
 * <p>
 * When deploying to AWS CloudFormation, it needs to keep in check the amount of resources being added inside a Stack. Currently it's possible to check the limits in the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html">AWS CloudFormation quotas</a> page.
 * <p>
 * It's possible to synthesize the project with more Resources than the allowed (or even reduce the number of Resources).
 * <p>
 * Set the context key <code>&#64;aws-cdk/core:stackResourceLimit</code> with the proper value, being 0 for disable the limit of resources.
 * <p>
 * <h3>Template Indentation</h3>
 * <p>
 * The AWS CloudFormation templates generated by CDK include indentation by default.
 * Indentation makes the templates more readable, but also increases their size,
 * and CloudFormation templates cannot exceed 1MB.
 * <p>
 * It's possible to reduce the size of your templates by suppressing indentation.
 * <p>
 * To do this for all templates, set the context key <code>&#64;aws-cdk/core:suppressTemplateIndentation</code> to <code>true</code>.
 * <p>
 * To do this for a specific stack, add a <code>suppressTemplateIndentation: true</code> property to the
 * stack's <code>StackProps</code> parameter. You can also set this property to <code>false</code> to override
 * the context key setting.
 * <p>
 * <h2>App Context</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/cdk/v2/guide/context.html">Context values</a> are key-value pairs that can be associated with an app, stack, or construct.
 * One common use case for context is to use it for enabling/disabling <a href="https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html">feature flags</a>. There are several places
 * where context can be specified. They are listed below in the order they are evaluated (items at the
 * top take precedence over those below).
 * <p>
 * <ul>
 * <li>The <code>node.setContext()</code> method</li>
 * <li>The <code>postCliContext</code> prop when you create an <code>App</code></li>
 * <li>The CLI via the <code>--context</code> CLI argument</li>
 * <li>The <code>cdk.json</code> file via the <code>context</code> key:</li>
 * <li>The <code>cdk.context.json</code> file:</li>
 * <li>The <code>~/.cdk.json</code> file via the <code>context</code> key:</li>
 * <li>The <code>context</code> prop when you create an <code>App</code></li>
 * </ul>
 * <p>
 * <h3>Examples of setting context</h3>
 * <p>
 * <blockquote><pre>
 * App.Builder.create()
 *         .context(Map.of(
 *                 "&#64;aws-cdk/core:newStyleStackSynthesis", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * App app = new App();
 * app.node.setContext("&#64;aws-cdk/core:newStyleStackSynthesis", true);
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * App.Builder.create()
 *         .postCliContext(Map.of(
 *                 "&#64;aws-cdk/core:newStyleStackSynthesis", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * cdk synth --context &#64;aws-cdk/core:newStyleStackSynthesis=true
 * </pre></blockquote>
 * <p>
 * <em>cdk.json</em>
 * <p>
 * <blockquote><pre>
 * {
 *   "context": {
 *     "&#64;aws-cdk/core:newStyleStackSynthesis": true
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <em>cdk.context.json</em>
 * <p>
 * <blockquote><pre>
 * {
 *   "&#64;aws-cdk/core:newStyleStackSynthesis": true
 * }
 * </pre></blockquote>
 * <p>
 * <em>~/.cdk.json</em>
 * <p>
 * <blockquote><pre>
 * {
 *   "context": {
 *     "&#64;aws-cdk/core:newStyleStackSynthesis": true
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h2>IAM Permissions Boundary</h2>
 * <p>
 * It is possible to apply an <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">IAM permissions boundary</a>
 * to all roles within a specific construct scope. The most common use case would
 * be to apply a permissions boundary at the <code>Stage</code> level.
 * <p>
 * <blockquote><pre>
 * Stage prodStage = Stage.Builder.create(app, "ProdStage")
 *         .permissionsBoundary(PermissionsBoundary.fromName("cdk-${Qualifier}-PermissionsBoundary"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Any IAM Roles or Users created within this Stage will have the default
 * permissions boundary attached.
 * <p>
 * For more details see the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_iam-readme.html#permissions-boundaries">Permissions Boundary</a> section in the IAM guide.
 * <p>
 * <h2>Policy Validation</h2>
 * <p>
 * If you or your organization use (or would like to use) any policy validation tool, such as
 * <a href="https://docs.aws.amazon.com/cfn-guard/latest/ug/what-is-guard.html">CloudFormation
 * Guard</a> or
 * <a href="https://www.openpolicyagent.org/">OPA</a>, to define constraints on your
 * CloudFormation template, you can incorporate them into the CDK application.
 * By using the appropriate plugin, you can make the CDK application check the
 * generated CloudFormation templates against your policies immediately after
 * synthesis. If there are any violations, the synthesis will fail and a report
 * will be printed to the console or to a file (see below).
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note</strong>
 * This feature is considered experimental, and both the plugin API and the
 * format of the validation report are subject to change in the future.
 * <p>
 * </blockquote>
 * <p>
 * <h3>For application developers</h3>
 * <p>
 * To use one or more validation plugins in your application, use the
 * <code>policyValidationBeta1</code> property of <code>Stage</code>:
 * <p>
 * <blockquote><pre>
 * // globally for the entire app (an app is a stage)
 * App app = App.Builder.create()
 *         .policyValidationBeta1(List.of(
 *             // These hypothetical classes implement IPolicyValidationPluginBeta1:
 *             new ThirdPartyPluginX(),
 *             new ThirdPartyPluginY()))
 *         .build();
 * 
 * // only apply to a particular stage
 * Stage prodStage = Stage.Builder.create(app, "ProdStage")
 *         .policyValidationBeta1(List.of(
 *             new ThirdPartyPluginX()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Immediately after synthesis, all plugins registered this way will be invoked to
 * validate all the templates generated in the scope you defined. In particular, if
 * you register the templates in the <code>App</code> object, all templates will be subject to
 * validation.
 * <p>
 * <blockquote>
 * <p>
 * <strong>Warning</strong>
 * Other than modifying the cloud assembly, plugins can do anything that your CDK
 * application can. They can read data from the filesystem, access the network
 * etc. It's your responsibility as the consumer of a plugin to verify that it is
 * secure to use.
 * <p>
 * </blockquote>
 * <p>
 * By default, the report will be printed in a human readable format. If you want a
 * report in JSON format, enable it using the <code>&#64;aws-cdk/core:validationReportJson</code>
 * context passing it directly to the application:
 * <p>
 * <blockquote><pre>
 * App app = App.Builder.create()
 *         .context(Map.of("&#64;aws-cdk/core:validationReportJson", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can set this context key-value pair using the <code>cdk.json</code> or
 * <code>cdk.context.json</code> files in your project directory (see
 * <a href="https://docs.aws.amazon.com/cdk/v2/guide/context.html">Runtime context</a>).
 * <p>
 * If you choose the JSON format, the CDK will print the policy validation report
 * to a file called <code>policy-validation-report.json</code> in the cloud assembly
 * directory. For the default, human-readable format, the report will be printed to
 * the standard output.
 * <p>
 * <h3>For plugin authors</h3>
 * <p>
 * The communication protocol between the CDK core module and your policy tool is
 * defined by the <code>IPolicyValidationPluginBeta1</code> interface. To create a new plugin you must
 * write a class that implements this interface. There are two things you need to
 * implement: the plugin name (by overriding the <code>name</code> property), and the
 * <code>validate()</code> method.
 * <p>
 * The framework will call <code>validate()</code>, passing an <code>IPolicyValidationContextBeta1</code> object.
 * The location of the templates to be validated is given by <code>templatePaths</code>. The
 * plugin should return an instance of <code>PolicyValidationPluginReportBeta1</code>. This object
 * represents the report that the user wil receive at the end of the synthesis.
 * <p>
 * <blockquote><pre>
 * public class MyPlugin implements IPolicyValidationPluginBeta1 {
 *     public final Object name;
 * 
 *     public PolicyValidationPluginReportBeta1 validate(IPolicyValidationContextBeta1 context) {
 *         // First read the templates using context.templatePaths...
 * 
 *         // ...then perform the validation, and then compose and return the report.
 *         // Using hard-coded values here for better clarity:
 *         return PolicyValidationPluginReportBeta1.builder()
 *                 .success(false)
 *                 .violations(List.of(PolicyViolationBeta1.builder()
 *                         .ruleName("CKV_AWS_117")
 *                         .description("Ensure that AWS Lambda function is configured inside a VPC")
 *                         .fix("https://docs.bridgecrew.io/docs/ensure-that-aws-lambda-function-is-configured-inside-a-vpc-1")
 *                         .violatingResources(List.of(PolicyViolatingResourceBeta1.builder()
 *                                 .resourceLogicalId("MyFunction3BAA72D1")
 *                                 .templatePath("/home/johndoe/myapp/cdk.out/MyService.template.json")
 *                                 .locations(List.of("Properties/VpcConfig"))
 *                                 .build()))
 *                         .build()))
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * In addition to the name, plugins may optionally report their version (<code>version</code>
 * property ) and a list of IDs of the rules they are going to evaluate (<code>ruleIds</code>
 * property).
 * <p>
 * Note that plugins are not allowed to modify anything in the cloud assembly. Any
 * attempt to do so will result in synthesis failure.
 * <p>
 * If your plugin depends on an external tool, keep in mind that some developers may
 * not have that tool installed in their workstations yet. To minimize friction, we
 * highly recommend that you provide some installation script along with your
 * plugin package, to automate the whole process. Better yet, run that script as
 * part of the installation of your package. With <code>npm</code>, for example, you can run
 * add it to the <code>postinstall</code>
 * <a href="https://docs.npmjs.com/cli/v9/using-npm/scripts">script</a> in the <code>package.json</code>
 * file.
 * <p>
 * <h2>Annotations</h2>
 * <p>
 * Construct authors can add annotations to constructs to report at three different
 * levels: <code>ERROR</code>, <code>WARN</code>, <code>INFO</code>.
 * <p>
 * Typically warnings are added for things that are important for the user to be
 * aware of, but will not cause deployment errors in all cases. Some common
 * scenarios are (non-exhaustive list):
 * <p>
 * <ul>
 * <li>Warn when the user needs to take a manual action, e.g. IAM policy should be
 * added to an referenced resource.</li>
 * <li>Warn if the user configuration might not follow best practices (but is still
 * valid)</li>
 * <li>Warn if the user is using a deprecated API</li>
 * </ul>
 * <p>
 * <h3>Acknowledging Warnings</h3>
 * <p>
 * If you would like to run with <code>--strict</code> mode enabled (warnings will throw
 * errors) it is possible to <code>acknowledge</code> warnings to make the warning go away.
 * <p>
 * For example, if &gt; 10 IAM managed policies are added to an IAM Group, a warning
 * will be created:
 * <p>
 * <blockquote><pre>
 * IAM:Group:MaxPoliciesExceeded: You added 11 to IAM Group my-group. The maximum number of managed policies attached to an IAM group is 10.
 * </pre></blockquote>
 * <p>
 * If you have requested a <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-quotas.html#reference_iam-quotas-entities">quota increase</a>
 * you may have the ability to add &gt; 10 managed policies which means that this
 * warning does not apply to you. You can acknowledge this by <code>acknowledging</code> the
 * warning by the <code>id</code>.
 * <p>
 * <blockquote><pre>
 * Annotations.of(this).acknowledgeWarning("IAM:Group:MaxPoliciesExceeded", "Account has quota increased to 20");
 * </pre></blockquote>
 * <p>
 * <!--END CORE DOCUMENTATION-->
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk;
