/*
 * Decompiled with CFR 0.152.
 */
package org.opends.quicksetup.util;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.opends.messages.Message;
import org.opends.messages.QuickSetupMessages;
import org.opends.quicksetup.Application;
import org.opends.quicksetup.ApplicationException;
import org.opends.quicksetup.ReturnCode;
import org.opends.quicksetup.util.Utils;

public class FileManager {
    private static final Logger LOG = Logger.getLogger(FileManager.class.getName());
    private Application application = null;

    public FileManager() {
    }

    public FileManager(Application app) {
        this.application = app;
    }

    public void synchronize(File source, File target) throws ApplicationException {
        String[] sourceFileNames;
        if (source != null && target != null && (sourceFileNames = source.list()) != null) {
            for (String sourceFileName : sourceFileNames) {
                File sourceFile = new File(source, sourceFileName);
                this.copyRecursively(sourceFile, target, null, false);
            }
        }
    }

    public void synchronize(File source, File target, FileFilter filter) throws ApplicationException {
        String[] sourceFileNames;
        if (source != null && target != null && (sourceFileNames = source.list()) != null) {
            for (String sourceFileName : sourceFileNames) {
                File sourceFile = new File(source, sourceFileName);
                this.copyRecursively(sourceFile, target, filter, false);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void rename(File fileToRename, File target) throws ApplicationException {
        if (fileToRename != null && target != null) {
            File file = target;
            synchronized (file) {
                if (target.exists() && !target.delete()) {
                    throw new ApplicationException(ReturnCode.FILE_SYSTEM_ACCESS_ERROR, QuickSetupMessages.INFO_ERROR_DELETING_FILE.get(Utils.getPath(target)), null);
                }
            }
            if (!fileToRename.renameTo(target)) {
                throw new ApplicationException(ReturnCode.FILE_SYSTEM_ACCESS_ERROR, QuickSetupMessages.INFO_ERROR_RENAMING_FILE.get(Utils.getPath(fileToRename), Utils.getPath(target)), null);
            }
        }
    }

    public void move(File object, File newParent) throws ApplicationException {
        this.move(object, newParent, null);
    }

    public void move(File object, File newParent, FileFilter filter) throws ApplicationException {
        if (filter == null || filter.accept(object)) {
            new MoveOperation(object, newParent).apply();
        }
    }

    public void delete(File object) throws ApplicationException {
        this.delete(object, null);
    }

    public void delete(File object, FileFilter filter) throws ApplicationException {
        if (filter == null || filter.accept(object)) {
            new DeleteOperation(object, DeletionPolicy.DELETE_IMMEDIATELY).apply();
        }
    }

    public void deleteChildren(File parentDir) throws ApplicationException {
        File[] children;
        if (parentDir != null && parentDir.exists() && parentDir.isDirectory() && (children = parentDir.listFiles()) != null) {
            for (File child : children) {
                this.delete(child);
            }
        }
    }

    public void deleteRecursively(File file) throws ApplicationException {
        this.deleteRecursively(file, null, DeletionPolicy.DELETE_IMMEDIATELY);
    }

    public void deleteRecursively(File file, FileFilter filter, DeletionPolicy deletePolicy) throws ApplicationException {
        this.operateRecursively(new DeleteOperation(file, deletePolicy), filter);
    }

    public File copy(File objectFile, File destDir) throws ApplicationException {
        CopyOperation co = new CopyOperation(objectFile, destDir, false);
        co.apply();
        return co.getDestination();
    }

    public File copy(File objectFile, File destDir, boolean overwrite) throws ApplicationException {
        CopyOperation co = new CopyOperation(objectFile, destDir, overwrite);
        co.apply();
        return co.getDestination();
    }

    public void copyRecursively(File objectFile, File destDir) throws ApplicationException {
        this.copyRecursively(objectFile, destDir, null);
    }

    public void copyRecursively(File objectFile, File destDir, FileFilter filter) throws ApplicationException {
        this.copyRecursively(objectFile, destDir, filter, false);
    }

    public void copyRecursively(File objectFile, File destDir, FileFilter filter, boolean overwrite) throws ApplicationException {
        this.operateRecursively(new CopyOperation(objectFile, destDir, overwrite), filter);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean filesDiffer(File f1, File f2) throws IOException {
        boolean differ = false;
        FileReader fr1 = new FileReader(f1);
        FileReader fr2 = new FileReader(f2);
        try {
            boolean done = false;
            while (!differ && !done) {
                int c2;
                int c1 = fr1.read();
                differ = c1 != (c2 = fr2.read());
                done = c1 == -1 || c2 == -1;
            }
        }
        finally {
            fr1.close();
            fr2.close();
        }
        return differ;
    }

    private void operateRecursively(FileOperation op, FileFilter filter) throws ApplicationException {
        File file = op.getObjectFile();
        if (file.exists()) {
            if (file.isFile()) {
                if (filter != null) {
                    if (filter.accept(file)) {
                        op.apply();
                    }
                } else {
                    op.apply();
                }
            } else {
                File[] children = file.listFiles();
                if (children != null) {
                    for (File aChildren : children) {
                        FileOperation newOp = op.copyForChild(aChildren);
                        this.operateRecursively(newOp, filter);
                    }
                }
                if (filter != null) {
                    if (filter.accept(file)) {
                        op.apply();
                    }
                } else {
                    op.apply();
                }
            }
        } else {
            if (this.application != null) {
                this.application.notifyListeners(this.application.getFormattedWarning(QuickSetupMessages.INFO_FILE_DOES_NOT_EXIST.get(String.valueOf(file))));
            }
            LOG.log(Level.INFO, "file '" + file.toString() + "' does not exist");
        }
    }

    private class MoveOperation
    extends FileOperation {
        File destination;

        public MoveOperation(File objectFile, File newParent) {
            super(objectFile);
            this.destination = null;
            this.destination = new File(newParent, objectFile.getName());
        }

        public FileOperation copyForChild(File child) {
            return new MoveOperation(child, this.destination);
        }

        public void apply() throws ApplicationException {
            File objectFile = this.getObjectFile();
            if (this.destination.exists()) {
                FileManager.this.deleteRecursively(this.destination);
            }
            if (!objectFile.renameTo(this.destination)) {
                throw ApplicationException.createFileSystemException(QuickSetupMessages.INFO_ERROR_FAILED_MOVING_FILE.get(Utils.getPath(objectFile), Utils.getPath(this.destination)), null);
            }
        }
    }

    private class DeleteOperation
    extends FileOperation {
        private DeletionPolicy deletionPolicy;

        public DeleteOperation(File objectFile, DeletionPolicy deletionPolicy) {
            super(objectFile);
            this.deletionPolicy = deletionPolicy;
        }

        public FileOperation copyForChild(File child) {
            return new DeleteOperation(child, this.deletionPolicy);
        }

        public void apply() throws ApplicationException {
            File file = this.getObjectFile();
            boolean isFile = file.isFile();
            if (FileManager.this.application != null && FileManager.this.application.isVerbose()) {
                if (isFile) {
                    FileManager.this.application.notifyListeners(FileManager.this.application.getFormattedWithPoints(QuickSetupMessages.INFO_PROGRESS_DELETING_FILE.get(file.getAbsolutePath())));
                } else {
                    FileManager.this.application.notifyListeners(FileManager.this.application.getFormattedWithPoints(QuickSetupMessages.INFO_PROGRESS_DELETING_DIRECTORY.get(file.getAbsolutePath())));
                }
            }
            LOG.log(Level.INFO, "deleting " + (isFile ? " file " : " directory ") + file.getAbsolutePath());
            boolean delete = false;
            int nTries = 5;
            for (int i = 0; i < nTries && !delete; ++i) {
                if (DeletionPolicy.DELETE_ON_EXIT.equals((Object)this.deletionPolicy)) {
                    file.deleteOnExit();
                    delete = true;
                } else {
                    delete = file.delete();
                    if (!delete && DeletionPolicy.DELETE_ON_EXIT_IF_UNSUCCESSFUL.equals((Object)this.deletionPolicy)) {
                        file.deleteOnExit();
                        delete = true;
                    }
                }
                if (delete) continue;
                try {
                    Thread.sleep(1000L);
                    continue;
                }
                catch (Exception ex) {
                    // empty catch block
                }
            }
            if (!delete) {
                Message errMsg = isFile ? QuickSetupMessages.INFO_ERROR_DELETING_FILE.get(file.getAbsolutePath()) : QuickSetupMessages.INFO_ERROR_DELETING_DIRECTORY.get(file.getAbsolutePath());
                throw new ApplicationException(ReturnCode.FILE_SYSTEM_ACCESS_ERROR, errMsg, null);
            }
            if (FileManager.this.application != null && FileManager.this.application.isVerbose()) {
                FileManager.this.application.notifyListeners(FileManager.this.application.getFormattedDoneWithLineBreak());
            }
        }
    }

    private class CopyOperation
    extends FileOperation {
        private File destination;
        private boolean overwrite;

        public CopyOperation(File objectFile, File destDir, boolean overwrite) {
            super(objectFile);
            this.destination = new File(destDir, objectFile.getName());
            this.overwrite = overwrite;
        }

        public FileOperation copyForChild(File child) {
            return new CopyOperation(child, this.destination, this.overwrite);
        }

        public File getDestination() {
            return this.destination;
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        public void apply() throws ApplicationException {
            File objectFile = this.getObjectFile();
            if (objectFile.isDirectory()) {
                if (this.destination.exists()) return;
                this.destination.mkdirs();
                return;
            }
            if (this.destination.exists() && this.overwrite) {
                FileManager.this.deleteRecursively(this.destination);
            }
            if (!this.destination.exists()) {
                if (Utils.insureParentsExist(this.destination)) {
                    if (FileManager.this.application != null && FileManager.this.application.isVerbose()) {
                        FileManager.this.application.notifyListeners(FileManager.this.application.getFormattedWithPoints(QuickSetupMessages.INFO_PROGRESS_COPYING_FILE.get(objectFile.getAbsolutePath(), this.destination.getAbsolutePath())));
                    }
                    LOG.log(Level.INFO, "copying file '" + objectFile.getAbsolutePath() + "' to '" + this.destination.getAbsolutePath() + "'");
                    FileInputStream fis = null;
                    FileOutputStream fos = null;
                    try {
                        int i;
                        fis = new FileInputStream(objectFile);
                        fos = new FileOutputStream(this.destination);
                        byte[] buf = new byte[1024];
                        while ((i = fis.read(buf)) != -1) {
                            fos.write(buf, 0, i);
                        }
                        if (this.destination.exists() && Utils.isUnix()) {
                            String permissions = Utils.getFileSystemPermissions(objectFile);
                            Utils.setPermissionsUnix(Utils.getPath(this.destination), permissions);
                        }
                        if (FileManager.this.application == null || !FileManager.this.application.isVerbose()) return;
                        FileManager.this.application.notifyListeners(FileManager.this.application.getFormattedDoneWithLineBreak());
                        return;
                    }
                    catch (Exception e) {
                        Message errMsg = QuickSetupMessages.INFO_ERROR_COPYING_FILE.get(objectFile.getAbsolutePath(), this.destination.getAbsolutePath());
                        throw new ApplicationException(ReturnCode.FILE_SYSTEM_ACCESS_ERROR, errMsg, null);
                    }
                    finally {
                        if (fis != null) {
                            try {
                                fis.close();
                            }
                            catch (IOException e) {}
                        }
                        if (fos != null) {
                            try {
                                fos.close();
                            }
                            catch (IOException e) {}
                        }
                    }
                }
                Message errMsg = QuickSetupMessages.INFO_ERROR_COPYING_FILE.get(objectFile.getAbsolutePath(), this.destination.getAbsolutePath());
                throw new ApplicationException(ReturnCode.FILE_SYSTEM_ACCESS_ERROR, errMsg, null);
            }
            LOG.log(Level.INFO, "Ignoring file '" + objectFile.getAbsolutePath() + "' since '" + this.destination.getAbsolutePath() + "' already exists");
            if (FileManager.this.application == null || !FileManager.this.application.isVerbose()) return;
            FileManager.this.application.notifyListeners(QuickSetupMessages.INFO_INFO_IGNORING_FILE.get(objectFile.getAbsolutePath(), this.destination.getAbsolutePath()));
            FileManager.this.application.notifyListeners(FileManager.this.application.getLineBreak());
        }
    }

    private abstract class FileOperation {
        private File objectFile = null;

        public FileOperation(File objectFile) {
            this.objectFile = objectFile;
        }

        protected File getObjectFile() {
            return this.objectFile;
        }

        public abstract FileOperation copyForChild(File var1);

        public abstract void apply() throws ApplicationException;
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum DeletionPolicy {
        DELETE_IMMEDIATELY,
        DELETE_ON_EXIT,
        DELETE_ON_EXIT_IF_UNSUCCESSFUL;

    }
}

