package tools.jackson.module.kotlin

import com.fasterxml.jackson.annotation.Nulls
import tools.jackson.databind.BeanDescription
import tools.jackson.databind.DeserializationConfig
import tools.jackson.databind.DeserializationContext
import tools.jackson.databind.JavaType
import tools.jackson.databind.ValueDeserializer
import tools.jackson.databind.deser.SettableBeanProperty
import tools.jackson.databind.deser.ValueInstantiator
import tools.jackson.databind.deser.ValueInstantiators
import tools.jackson.databind.deser.bean.PropertyValueBuffer
import tools.jackson.databind.deser.std.StdValueInstantiator
import tools.jackson.databind.exc.InvalidNullException
import java.lang.reflect.TypeVariable
import kotlin.reflect.KType
import kotlin.reflect.KTypeProjection
import kotlin.reflect.jvm.javaType

internal class KotlinValueInstantiator(
    src: StdValueInstantiator,
    private val cache: ReflectionCache,
    private val nullToEmptyCollection: Boolean,
    private val nullToEmptyMap: Boolean,
    private val nullIsSameAsDefault: Boolean
) : StdValueInstantiator(src) {
    private fun JavaType.requireEmptyValue() =
        (nullToEmptyCollection && this.isCollectionLikeType) || (nullToEmptyMap && this.isMapLikeType)

    private fun KType.isGenericTypeVar() = javaType is TypeVariable<*>

    // If the argument is a value class that wraps nullable and non-null,
    // and the input is explicit null, the value class is instantiated with null as input.
    private fun requireValueClassSpecialNullValue(
        isNullableParam: Boolean,
        valueDeserializer: ValueDeserializer<*>?
    ): Boolean = !isNullableParam &&
            valueDeserializer is WrapsNullableValueClassDeserializer<*> &&
            valueDeserializer.handledType().kotlin.wrapsNullable()

    private fun SettableBeanProperty.skipNulls(): Boolean =
        nullIsSameAsDefault || (metadata.valueNulls == Nulls.SKIP)

    override fun createFromObjectWith(
        ctxt: DeserializationContext,
        props: Array<out SettableBeanProperty>,
        buffer: PropertyValueBuffer
    ): Any? {
        val valueCreator: ValueCreator<*> = cache.valueCreatorFromJava(_withArgsCreator)
            ?: return super.createFromObjectWith(ctxt, props, buffer)

        val bucket = valueCreator.generateBucket()

        valueCreator.valueParameters.forEachIndexed { idx, paramDef ->
            val jsonProp = props[idx]
            val isMissing = !buffer.hasParameter(jsonProp)
            val valueDeserializer: ValueDeserializer<*>? by lazy { jsonProp.valueDeserializer }

            val paramType = paramDef.type
            var paramVal = if (!isMissing || jsonProp.hasInjectableValueId()) {
               buffer.getParameter(ctxt, jsonProp) ?: run {
                   // Deserializer.getNullValue could not be used because there is no way to get and parse parameters
                   // from the BeanDescription and using AnnotationIntrospector would override user customization.
                   if (requireValueClassSpecialNullValue(paramDef.type.isMarkedNullable, valueDeserializer)) {
                       (valueDeserializer as WrapsNullableValueClassDeserializer<*>).boxedNullValue?.let { return@run it }
                   }

                   if (jsonProp.skipNulls() && paramDef.isOptional) return@forEachIndexed

                   null
               }
            } else {
                when {
                    paramDef.isOptional || paramDef.isVararg -> return@forEachIndexed
                    // do not try to create any object if it is nullable and the value is missing
                    paramType.isMarkedNullable -> null
                    // Primitive types always try to get from a buffer, considering several settings
                    jsonProp.type.isPrimitive -> buffer.getParameter(ctxt, jsonProp)
                    // to get suitable "missing" value provided by nullValueProvider
                    else -> jsonProp.nullValueProvider?.getAbsentValue(ctxt)
                }
            }

            val propType = jsonProp.type

            if (paramVal == null) {
                if (propType.requireEmptyValue()) {
                    paramVal = valueDeserializer!!.getEmptyValue(ctxt)
                } else {
                    val pname = jsonProp.name
                    val isMissingAndRequired = isMissing && jsonProp.isRequired

                    // Since #310 reported that the calculation cost is high, isGenericTypeVar is determined last.
                    if (isMissingAndRequired || (!paramType.isMarkedNullable && !paramType.isGenericTypeVar())) {
                        throw KotlinInvalidNullException(
                            paramDef.name,
                            this.valueClass,
                            ctxt.parser,
                            "Instantiation of ${this.valueTypeDesc} value failed for JSON property $pname due to missing (therefore NULL) value for creator parameter ${paramDef.name} which is a non-nullable type",
                            jsonProp.fullName,
                        ).wrapWithPath(this.valueClass, pname)
                    }
                }
            }

            bucket[paramDef] = paramVal
        }

        valueCreator.checkAccessibility(ctxt)

        return valueCreator.callBy(bucket)
    }

    private fun SettableBeanProperty.hasInjectableValueId(): Boolean = injectableValueId != null
}

internal class KotlinInstantiators(
    private val cache: ReflectionCache,
    private val nullToEmptyCollection: Boolean,
    private val nullToEmptyMap: Boolean,
    private val nullIsSameAsDefault: Boolean,
) : ValueInstantiators.Base() {
    override fun modifyValueInstantiator(
        deserConfig: DeserializationConfig,
        beanDescriptorRef: BeanDescription.Supplier,
        defaultInstantiator: ValueInstantiator
    ): ValueInstantiator {
        return if (beanDescriptorRef.beanClass.isKotlinClass()) {
            if (defaultInstantiator::class == StdValueInstantiator::class) {
                KotlinValueInstantiator(
                    defaultInstantiator as StdValueInstantiator,
                    cache,
                    nullToEmptyCollection,
                    nullToEmptyMap,
                    nullIsSameAsDefault
                )
            } else {
                // TODO: return defaultInstantiator and let default method parameters and nullability go unused?  or die with exception:
                throw IllegalStateException("KotlinValueInstantiator requires that the default ValueInstantiator is StdValueInstantiator")
            }
        } else {
            defaultInstantiator
        }
    }
}
